/* gnome-db-data-proxy.c
 *
 * Copyright (C) 2004 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "gnome-db-data-proxy.h"
#include "gnome-db-data-model.h"
#include "gnome-db-parameter.h"
#include "gnome-db-data-set.h"
#include "marshal.h"

/* 
 * Main static functions 
 */
static void gnome_db_data_proxy_class_init (GnomeDbDataProxyClass * class);
static void gnome_db_data_proxy_init (GnomeDbDataProxy * srv);
static void gnome_db_data_proxy_dispose (GObject   * object);
static void gnome_db_data_proxy_finalize (GObject   * object);

static void gnome_db_data_proxy_set_property    (GObject              *object,
						guint                 param_id,
						const GValue         *value,
						GParamSpec           *pspec);
static void gnome_db_data_proxy_get_property    (GObject              *object,
						guint                 param_id,
						GValue               *value,
						GParamSpec           *pspec);
/*
 * GtkTreeModel interface
 */
static void              data_proxy_tree_model_init (GtkTreeModelIface *iface);
static GtkTreeModelFlags data_proxy_get_flags       (GtkTreeModel *tree_model);
static gint              data_proxy_get_n_columns   (GtkTreeModel *tree_model);
static GType             data_proxy_get_column_type (GtkTreeModel *tree_model, gint index);
static gboolean          data_proxy_get_iter        (GtkTreeModel *tree_model, GtkTreeIter *iter, GtkTreePath *path);
static GtkTreePath      *data_proxy_get_path        (GtkTreeModel *tree_model, GtkTreeIter *iter);
static void              data_proxy_get_value       (GtkTreeModel *tree_model, GtkTreeIter *iter, gint column, GValue *value);
static gboolean          data_proxy_iter_next       (GtkTreeModel *tree_model, GtkTreeIter *iter);
static gboolean          data_proxy_iter_children   (GtkTreeModel *tree_model, GtkTreeIter *iter, GtkTreeIter *parent);
static gboolean          data_proxy_iter_has_child  (GtkTreeModel *tree_model, GtkTreeIter *iter);
static gint              data_proxy_iter_n_children (GtkTreeModel *tree_model, GtkTreeIter *iter);
static gboolean          data_proxy_iter_nth_child  (GtkTreeModel *tree_model, GtkTreeIter *iter, GtkTreeIter *parent, gint n);
static gboolean          data_proxy_iter_parent     (GtkTreeModel *tree_model, GtkTreeIter *iter, GtkTreeIter *child);

static gint model_to_iter_numbers  (GnomeDbDataProxy *proxy, gint row);
static gint iter_to_model_numbers  (GnomeDbDataProxy *proxy, gint row);

static void nullified_object_cb (GnomeDbBase *obj, GnomeDbDataProxy *proxy);
#ifdef debug
static void gnome_db_data_proxy_dump (GnomeDbDataProxy *proxy, guint offset);
#endif

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	LAST_SIGNAL
};

static gint gnome_db_data_proxy_signals[LAST_SIGNAL] = {  };

/* properties */
enum
{
	PROP_0,
	PROP_AUTOCOMMIT,
	PROP_ADD_NULL_ENTRY
};

/*
 * Structure to hold the status and all the modifications made to a single
 * row. It is not possible to have ((model_row < 0) && !modif_values)
 */
typedef struct
{
	gint           model_row;    /* row index in the GnomeDbDataModel, -1 if new row */
	gboolean       to_be_deleted;/* TRUE if row is to be deleted */
	GSList        *modif_values; /* list of RowValue structures */
	GdaValue     **orig_values;  /* array of the original GdaValues, indexed on the column numbers */
	gint           orig_values_size;
} RowModif;
#define ROW_MODIF(x) ((RowModif *)(x))

/*
 * Structure to hold the modifications made to a signle value
 */
typedef struct
{
	RowModif      *row_modif;    /* RowModif in which this structure instance appears */
	gint           model_column; /* column index in the GnomeDbDataModel */
        GdaValue      *value;        /* can also be GDA_VALUE_TYPE_LIST for multiple values; values are owned here */
        guint          attributes;   /* #GnomeDbDataEntry attributes */
} RowValue;
#define ROW_VALUE(x) ((RowValue *)(x))

/*
 * NOTE about the row numbers:
 * 
 * There are 2 kinds of row numbers: the row numbers as identified in the GtkTreeModel interface by each GtkTreePath
 * and the row numbers in the GnomeDbDataModel.
 *
 * There may be more rows in the GtkTreeModel than in the GnomeDbDataModel if:
 *  - some rows have been added in the proxy
 *    and are not yet in the GnomeDbDataModel (RowModif->model_row = -1 in this case); or
 *  - there is a NULL row at the beginning
 *
 * The rule to go from one row numbering to the other is to use the model_to_iter_numbers() and iter_to_model_numbers()
 * functions.
 */
struct _GnomeDbDataProxyPrivate
{
	GnomeDbDataModel  *model;
	guint             *cols_non_modif_attrs; /* array of GnomeDbValueAttribute to proxy cols. attributes */
	GnomeDbDataSet    *info_data_set; /* used for columns attributes and initial values when a new row is created */

	gint               model_nb_cols; /* = gda_data_model_get_n_columns (model) */
	gboolean           autocommit;
	gint               stamp;       /* Random integer to check whether an iter belongs to our model */

	GSList            *all_modifs; /* list of RowModif structures, for memory management */
	GSList            *new_rows;   /* list of RowModif, no data allocated in this list */

	GHashTable        *modif_rows;  /* key = row number for the GnomeDbDataModel, value = RowModif, NOT for new rows */

	gint               internal_changes;
	gboolean           model_refresh_pending;

	gboolean           add_null_entry; /* artificially add a NULL entry at the beginning of the tree model */

	guint              idle_add_event_source; /* !=0 when data rows are being added */

	/* chunking: ONLY accounting for @model's rows, not the new rows */
	gint               sample_first_row;
	gint               sample_last_row;
	gint               sample_size;
	gint               current_nb_rows;
};

/*
 * The model_to_iter_numbers() and iter_to_model_numbers() convert row numbers between GnomeDbDataModel
 * and GtkTreeModel.
 *
 * The current implementation is that the first N rows are GnomeDbDataModel rows, and the next M ones
 * are new rows. It has the inconvenient of not being able to mix existing rows in the GnomeDbDataModel
 * with new rows.
 *
 * Also if there is a NULL row at the beginning, the whole schema is shifted by 1
 */
static gint
model_to_iter_numbers (GnomeDbDataProxy *proxy, gint row)
{
	if (proxy->priv->add_null_entry)
		return row - proxy->priv->sample_first_row + 1;
	else
		return row - proxy->priv->sample_first_row;
}

static gint
iter_to_model_numbers (GnomeDbDataProxy *proxy, gint row)
{
	if (proxy->priv->add_null_entry) {
		if (row == 0)
			return -1;
		
		if (row < proxy->priv->current_nb_rows + 1)
			return row + proxy->priv->sample_first_row - 1;
		else
			return -1;
	}
	else {
		if (row < proxy->priv->current_nb_rows)
			return row + proxy->priv->sample_first_row;
		else
			return -1;
	}
}

static RowModif *
find_row_modif_for_iter (GnomeDbDataProxy *proxy, GtkTreeIter *iter)
{
	RowModif *rm = NULL;
	gint model_row;

	model_row = iter_to_model_numbers (proxy, GPOINTER_TO_INT (iter->user_data));
	if (model_row >= 0)
		rm = g_hash_table_lookup (proxy->priv->modif_rows, GINT_TO_POINTER (model_row));
	else {
		gint offset = proxy->priv->current_nb_rows + (proxy->priv->add_null_entry ? 1 : 0);

		offset = GPOINTER_TO_INT (iter->user_data) - offset;
		rm = g_slist_nth_data (proxy->priv->new_rows, offset);
	}

	return rm;
}

/*
 * Free a RowModif structure
 *
 * Warning: only the allocated RowModif is freed, it's not removed from any list or hash table.
 */
static void
row_modifs_free (RowModif *rm)
{
	GSList *list;
	gint i;
		
	list = rm->modif_values;
	while (list) {
		if (ROW_VALUE (list->data)->value)
			gda_value_free (ROW_VALUE (list->data)->value);
		g_free (list->data);
		list = g_slist_next (list);
	}
	g_slist_free (rm->modif_values);
	
	if (rm->orig_values) {
		for (i=0; i < rm->orig_values_size; i++) {
			if (rm->orig_values [i])
				gda_value_free (rm->orig_values [i]);
		}
		g_free (rm->orig_values);
	}
	
	g_free (rm);
}

/*
 * Allocates a new RowModif
 *
 * WARNING: the new RowModif is not inserted in any list or hash table.
 */
static RowModif *
row_modifs_new (GnomeDbDataProxy *proxy, GtkTreeIter *iter)
{
	RowModif *rm;
	gint i;
	GdaValue *tmpval;
	
	rm = g_new0 (RowModif, 1);
	if (iter) {
		rm->orig_values = g_new0 (GdaValue *, proxy->priv->model_nb_cols);
		rm->orig_values_size = proxy->priv->model_nb_cols;
		for (i=0; i<proxy->priv->model_nb_cols; i++) {
			gtk_tree_model_get (GTK_TREE_MODEL (proxy), iter, i, &tmpval, -1);
			rm->orig_values [i] = gda_value_copy (tmpval);
		}
	}
	
	return rm;
}

/* module error */
GQuark gnome_db_data_proxy_error_quark (void)
{
        static GQuark quark;
        if (!quark)
                quark = g_quark_from_static_string ("gnome_db_data_proxy_error");
        return quark;
}

GType
gnome_db_data_proxy_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbDataProxyClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_data_proxy_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbDataProxy),
			0,
			(GInstanceInitFunc) gnome_db_data_proxy_init
		};

		static const GInterfaceInfo tree_model_info = {
			(GInterfaceInitFunc) data_proxy_tree_model_init,
			NULL,
			NULL
		};

		type = g_type_register_static (GNOME_DB_TYPE_BASE, "GnomeDbDataProxy", &info, 0);
		g_type_add_interface_static (type, GTK_TYPE_TREE_MODEL, &tree_model_info);
	}
	return type;
}

static void
data_proxy_tree_model_init (GtkTreeModelIface *iface)
{
	iface->get_flags       = data_proxy_get_flags;
	iface->get_n_columns   = data_proxy_get_n_columns;
	iface->get_column_type = data_proxy_get_column_type;
	iface->get_iter        = data_proxy_get_iter;
	iface->get_path        = data_proxy_get_path;
	iface->get_value       = data_proxy_get_value;
	iface->iter_next       = data_proxy_iter_next;
	iface->iter_children   = data_proxy_iter_children;
	iface->iter_has_child  = data_proxy_iter_has_child;
	iface->iter_n_children = data_proxy_iter_n_children;
	iface->iter_nth_child  = data_proxy_iter_nth_child;
	iface->iter_parent     = data_proxy_iter_parent;
}

static void
gnome_db_data_proxy_class_init (GnomeDbDataProxyClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);


	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_data_proxy_dump;
#endif

	object_class->dispose = gnome_db_data_proxy_dispose;
	object_class->finalize = gnome_db_data_proxy_finalize;

	/* Properties */
	object_class->set_property = gnome_db_data_proxy_set_property;
	object_class->get_property = gnome_db_data_proxy_get_property;

	g_object_class_install_property (object_class, PROP_AUTOCOMMIT,
					 g_param_spec_boolean ("autocommit", NULL, NULL, FALSE,
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (object_class, PROP_ADD_NULL_ENTRY,
					 g_param_spec_boolean ("prepend_null_entry", NULL, NULL, FALSE,
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}

static void
gnome_db_data_proxy_init (GnomeDbDataProxy *proxy)
{
	proxy->priv = g_new0 (GnomeDbDataProxyPrivate, 1);
	proxy->priv->modif_rows = g_hash_table_new (NULL, NULL);
	proxy->priv->stamp = g_random_int ();

	proxy->priv->add_null_entry = FALSE;
	proxy->priv->idle_add_event_source = 0;

	proxy->priv->sample_first_row = 0;
	proxy->priv->sample_last_row = 0;
	proxy->priv->sample_size = 300;
	proxy->priv->current_nb_rows = 0;
}

static void adjust_displayed_chunck (GnomeDbDataProxy *proxy);
static gboolean idle_add_model_rows (GnomeDbDataProxy *proxy);


static void model_data_refreshed_cb (GnomeDbDataModel *model, GnomeDbDataProxy *proxy);

/**
 * gnome_db_data_proxy_new
 * @dict: a #GnomeDbDict object
 * @type: the #GnomeDbServerDataType requested
 *
 * Creates a new proxy of type @type
 *
 * Returns: a new #GnomeDbDataProxy object
 */
GObject *
gnome_db_data_proxy_new (GnomeDbDataModel *model)
{
	GObject   *obj;
	GnomeDbDataProxy *proxy;
	gint col;

	g_return_val_if_fail (model && IS_GNOME_DB_DATA_MODEL (model), NULL);

	obj = g_object_new (GNOME_DB_TYPE_DATA_PROXY, "dict", gnome_db_base_get_dict (GNOME_DB_BASE (model)), NULL);
	proxy = GNOME_DB_DATA_PROXY (obj);

	g_object_ref (model);
	gnome_db_base_connect_nullify (GNOME_DB_BASE (model), G_CALLBACK (nullified_object_cb), obj);
	proxy->priv->model = model;

	if (gnome_db_data_model_get_status (model) & GNOME_DB_DATA_MODEL_NEEDS_INIT_REFRESH) 
		gnome_db_data_model_refresh (model, NULL);

	proxy->priv->model_nb_cols = gda_data_model_get_n_columns (GDA_DATA_MODEL (model));

	proxy->priv->info_data_set = gnome_db_data_model_get_new_data_set (model);
	proxy->priv->cols_non_modif_attrs = g_new0 (guint, proxy->priv->model_nb_cols);
	for (col = 0; col < proxy->priv->model_nb_cols; col++) {
		GnomeDbParameter *param;
		
		param = gnome_db_data_model_get_param_at_column (proxy->priv->model, proxy->priv->info_data_set, col);
		proxy->priv->cols_non_modif_attrs[col] = GNOME_DB_VALUE_IS_UNCHANGED;
		if (! gnome_db_parameter_get_not_null (param))
			proxy->priv->cols_non_modif_attrs[col] |= GNOME_DB_VALUE_CAN_BE_NULL;
		if (gnome_db_parameter_get_default_value (param))
			proxy->priv->cols_non_modif_attrs[col] |= GNOME_DB_VALUE_CAN_BE_DEFAULT;
	}

	g_signal_connect (G_OBJECT (model), "data-refreshed",
			  G_CALLBACK (model_data_refreshed_cb), proxy);

	adjust_displayed_chunck (proxy);

        return obj;
}


static void
nullified_object_cb (GnomeDbBase *obj, GnomeDbDataProxy *proxy)
{
	gnome_db_base_nullify (GNOME_DB_BASE (proxy));
}

static void
gnome_db_data_proxy_dispose (GObject *object)
{
	GnomeDbDataProxy *proxy;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_DATA_PROXY (object));

	proxy = GNOME_DB_DATA_PROXY (object);
	if (proxy->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		if (proxy->priv->idle_add_event_source) {
			g_idle_remove_by_data (proxy);
			proxy->priv->idle_add_event_source = 0;
		}

		if (proxy->priv->info_data_set) {
			g_object_unref (proxy->priv->info_data_set);
			proxy->priv->info_data_set = NULL;
		}

		if (proxy->priv->model) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (proxy->priv->model),
                                                              G_CALLBACK (nullified_object_cb), proxy);
			g_object_unref (proxy->priv->model);
			proxy->priv->model = NULL;
		}

		if (proxy->priv->cols_non_modif_attrs) {
			g_free (proxy->priv->cols_non_modif_attrs);
			proxy->priv->cols_non_modif_attrs = NULL;
		}
		
		if (proxy->priv->modif_rows) {
			gnome_db_data_proxy_reset_all (proxy);
			g_hash_table_destroy (proxy->priv->modif_rows);
			proxy->priv->modif_rows = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_data_proxy_finalize (GObject   * object)
{
	GnomeDbDataProxy *proxy;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_DATA_PROXY (object));

	proxy = GNOME_DB_DATA_PROXY (object);
	if (proxy->priv) {
		if (proxy->priv->new_rows) {
			g_slist_free (proxy->priv->new_rows);
			proxy->priv->new_rows = NULL;
		}
			
		g_free (proxy->priv);
		proxy->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_data_proxy_set_property (GObject              *object,
				  guint                 param_id,
				  const GValue         *value,
				  GParamSpec           *pspec)
{
	GnomeDbDataProxy *proxy;

	proxy = GNOME_DB_DATA_PROXY (object);
	if (proxy->priv) {
		switch (param_id) {
		case PROP_AUTOCOMMIT:
			proxy->priv->autocommit = g_value_get_boolean (value);
			break;
		case PROP_ADD_NULL_ENTRY:
			if (proxy->priv->add_null_entry != g_value_get_boolean (value)) {
				gint offset;
				GtkTreePath *path;
				GtkTreeIter iter;

				offset = proxy->priv->add_null_entry ? -1 : 1;

				/* GtkTreeModel interface management */
				if (offset == 1) {
					/* add a row in the GtkTreeModel */
					proxy->priv->add_null_entry = g_value_get_boolean (value);
					path = gtk_tree_path_new_from_indices (0, -1);
					data_proxy_get_iter (GTK_TREE_MODEL (proxy), &iter, path);
					gtk_tree_model_row_inserted (GTK_TREE_MODEL (proxy), path, &iter);
					gtk_tree_path_free (path);
				}
				else {
					/* remove a row in the GtkTreeModel */
					path = gtk_tree_path_new_from_indices (0, -1);
					gtk_tree_model_row_deleted (GTK_TREE_MODEL (proxy), path);
					gtk_tree_path_free (path);
					proxy->priv->add_null_entry = g_value_get_boolean (value);
				}
				proxy->priv->stamp = g_random_int ();				
			}
			break;
		}
	}
}

static void
gnome_db_data_proxy_get_property (GObject              *object,
				  guint                 param_id,
				  GValue               *value,
				  GParamSpec           *pspec)
{
	GnomeDbDataProxy *proxy;

	proxy = GNOME_DB_DATA_PROXY (object);
	if (proxy->priv) {
		switch (param_id) {
		case PROP_AUTOCOMMIT:
			g_value_set_boolean (value, proxy->priv->autocommit);
			break;
		case PROP_ADD_NULL_ENTRY:
			g_value_set_boolean (value, proxy->priv->add_null_entry);
			break;
		}
	}
}

/*
 * Callback called when the data model has refreshed its contents.
 * We try to re-map all the modifications to the new data, and the modifications which can't be
 * re-mapped are discarded
 */
static void
model_data_refreshed_cb (GnomeDbDataModel *model, GnomeDbDataProxy *proxy)
{
	if (proxy->priv->internal_changes) {
		proxy->priv->model_refresh_pending = TRUE;
	}
	else {
		/* NOTE: to do this correctly, we need to have the list of columns which represent
		 * a primary key, which we don't have right now.
		 *
		 * FIXME: the current implementation clears all the outstanding modifications without trying
		 * to merge them with the new data
		 */
		gint i, old_nb_rows;
		GtkTreePath *path;
		old_nb_rows = gnome_db_data_proxy_get_n_rows (proxy);

		proxy->priv->model_nb_cols = gda_data_model_get_n_columns (GDA_DATA_MODEL (model));

		/* Free memory for the modifications */
		while (proxy->priv->all_modifs) {
			gint model_row = ROW_MODIF (proxy->priv->all_modifs->data)->model_row;
			row_modifs_free (ROW_MODIF (proxy->priv->all_modifs->data));
			if (model_row >= 0)
				g_hash_table_remove (proxy->priv->modif_rows, GINT_TO_POINTER (model_row));
			proxy->priv->all_modifs = g_slist_delete_link (proxy->priv->all_modifs, proxy->priv->all_modifs);
		}
		if (proxy->priv->new_rows) {
			int j = g_slist_length (proxy->priv->new_rows);

			/* emit the GtkTreeModel::"row_deleted" signal for all the new rows 
			 * (using the same row number!) */
			path = gtk_tree_path_new_from_indices (proxy->priv->current_nb_rows + (proxy->priv->add_null_entry ? 1 : 0), -1);
			for (i = 0; i < j ; i++) 
				gtk_tree_model_row_deleted (GTK_TREE_MODEL (proxy), path);
			gtk_tree_path_free (path);
			proxy->priv->stamp = g_random_int ();
				
			g_slist_free (proxy->priv->new_rows);
			proxy->priv->new_rows = NULL;
		}

		proxy->priv->model_refresh_pending = FALSE;
		adjust_displayed_chunck (proxy);
	}
}

/**
 * gnome_db_data_proxy_get_n_rows
 * @proxy: a #GnomeDbDataProxy object
 *
 * Considering the data stored within @proxy, get the number of rows.
 *
 * Returns: the number of rows in the proxy's data
 */
gint
gnome_db_data_proxy_get_n_rows (GnomeDbDataProxy *proxy)
{
	gint nbrows;
	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), -1);
	g_return_val_if_fail (proxy->priv, -1);

	nbrows = proxy->priv->current_nb_rows;
	nbrows += g_slist_length (proxy->priv->new_rows);
	if (proxy->priv->add_null_entry)
		nbrows += 1;

	return nbrows;
}

/**
 * gnome_db_data_proxy_get_n_columns
 * @proxy: a #GnomeDbDataProxy object
 *
 * Considering the data stored within @proxy, get the number of columns.
 *
 * Returns: the number of columns in the data proxy
 */
gint
gnome_db_data_proxy_get_n_columns (GnomeDbDataProxy *proxy)
{
	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), -1);
	g_return_val_if_fail (proxy->priv, -1);

	return proxy->priv->model_nb_cols;
}

/**
 * gnome_db_data_proxy_is_read_only
 * @proxy: a #GnomeDbDataProxy object
 *
 * Returns: TRUE if the data in @proxy can't be modified
 */
gboolean
gnome_db_data_proxy_is_read_only (GnomeDbDataProxy *proxy)
{
	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), TRUE);
	g_return_val_if_fail (proxy->priv, TRUE);

	return ! gda_data_model_is_updatable (GDA_DATA_MODEL (proxy->priv->model));
}


RowModif *find_or_create_row_modif (GnomeDbDataProxy *proxy, GtkTreeIter *iter, gint col, RowValue **ret_rv);

/**
 * gnome_db_data_proxy_set_value
 * @proxy: a #GnomeDbDataProxy object
 * @iter:
 * @col:
 * @value: the value to store (gets copied)
 *
 * Stores a value in the proxy.
 *
 * Returns: TRUE on succes
 */
gboolean
gnome_db_data_proxy_set_value (GnomeDbDataProxy *proxy, GtkTreeIter *iter, 
			       gint col, const GdaValue *value)
{
	gint row;
	gboolean value_set = FALSE;
	GtkTreePath *path;

	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), FALSE);
	g_return_val_if_fail (proxy->priv, FALSE);
	g_return_val_if_fail (iter, FALSE);
	g_return_val_if_fail (iter->stamp == proxy->priv->stamp, FALSE);

	row = GPOINTER_TO_INT (iter->user_data);
	if ((row == 0) && proxy->priv->add_null_entry) {
		g_warning ("Trying to set read-only NULL row");
		return FALSE;
	}

	path = gtk_tree_path_new ();
	gtk_tree_path_append_index (path, row);

	if (col < 0) {
		switch (col) {
		case PROXY_COL_MODEL_N_COLUMNS:
		case PROXY_COL_MODEL_ROW:
		case PROXY_COL_MODEL_POINTER:
		case PROXY_COL_MODIFIED:
		case PROXY_COL_TO_DELETE:
			g_warning ("Trying to set read-only column: %d", col);
			value_set = TRUE;
			break;
		}
	}

	if ((col >= 0) && (col < proxy->priv->model_nb_cols)) {
		/* Storing a GdaValue value */
		RowModif *rm;
		RowValue *rv = NULL;
		GValue *cmp_value;

		/* compare with the current stored value */
		cmp_value = g_new0 (GValue, 1);
		data_proxy_get_value (GTK_TREE_MODEL (proxy), iter, col, cmp_value);
		if (! gda_value_compare_ext (value, g_value_get_pointer (cmp_value))) {
			/* nothing to do: values are equal */
			g_value_unset (cmp_value);
			g_free (cmp_value);
			gtk_tree_path_free (path);
			return TRUE;
		}
		g_value_unset (cmp_value);
		g_free (cmp_value);

		/* from now on we have a new value for the row */
		rm = find_or_create_row_modif (proxy, iter, col, &rv);

		if (rv) {
			/* compare with the original value (before modifications) and either delete the RowValue, 
			 * or alter it */
			/* g_print ("Use existing RV\n"); */

			if (rv->value) {
				gda_value_free (rv->value);
				rv->value = NULL;
			}

			if (rm->orig_values && (col < rm->orig_values_size) && 
			    ! gda_value_compare_ext (value, rm->orig_values [col])) {
				/* remove the RowValue */
				rm->modif_values = g_slist_remove (rm->modif_values, rv);
				g_free (rv);
				rv = NULL;
			}
			else {
				/* simply alter the RowValue */
				if (value) {
					rv->attributes &= ~GNOME_DB_VALUE_IS_NULL;
					rv->value = gda_value_copy (value);
				}
				else
					rv->attributes |= GNOME_DB_VALUE_IS_NULL;
			}
		}
		else {
			/* create a new RowValue */
			/*g_print ("New RV\n");*/
			rv = g_new0 (RowValue, 1);
			rv->row_modif = rm;
			rv->model_column = col;
			rv->attributes = proxy->priv->cols_non_modif_attrs [col];

			if (value && !gda_value_is_null (value)) 
				rv->value = gda_value_copy (value);
			else
				rv->attributes |= GNOME_DB_VALUE_IS_NULL;
			if (rm->model_row >= 0)
				rv->attributes |= GNOME_DB_VALUE_HAS_VALUE_ORIG;
			else
				rv->attributes &= ~GNOME_DB_VALUE_HAS_VALUE_ORIG;
			rm->modif_values = g_slist_prepend (rm->modif_values, rv);
		}
		
		if (rv) {
			rv->attributes &= ~GNOME_DB_VALUE_IS_UNCHANGED;
			rv->attributes &= ~GNOME_DB_VALUE_IS_DEFAULT;
		}

		if (!rm->to_be_deleted && !rm->modif_values && (rm->model_row >= 0)) {
			/* remove that RowModif, it's useless */
			/* g_print ("Removing RM\n"); */
			g_hash_table_remove (proxy->priv->modif_rows, GINT_TO_POINTER (rm->model_row));
			proxy->priv->all_modifs = g_slist_remove (proxy->priv->all_modifs, rm);
			row_modifs_free (rm);
		}

		gtk_tree_model_row_changed (GTK_TREE_MODEL (proxy), path, iter);
	}

	if ((col >= proxy->priv->model_nb_cols) && (col < 2*proxy->priv->model_nb_cols)) {
		/* Storing an action => apply that action */
		guint action;
		gint model_col = col - proxy->priv->model_nb_cols;
		
		if (gda_value_isa (value, GDA_VALUE_TYPE_UINTEGER)) {
			gboolean action_ok = FALSE;
			
			action = gda_value_get_uinteger (value);
			
			if ((action & GNOME_DB_VALUE_IS_NULL) ||
			    (action & GNOME_DB_VALUE_IS_DEFAULT) ||
			    (action & GNOME_DB_VALUE_IS_UNCHANGED)) 
				action_ok = TRUE;
			if (action_ok) {
				if (action & GNOME_DB_VALUE_IS_NULL) 
					gnome_db_data_proxy_set_value (proxy, iter, model_col, NULL);
				else {
					RowModif *rm;
					RowValue *rv = NULL;

					rm = find_or_create_row_modif (proxy, iter, model_col, &rv);
					g_assert (rm);

					if (action & GNOME_DB_VALUE_IS_DEFAULT) {
						if (!rv) {
							/* create a new RowValue */
							/*g_print ("New RV\n");*/
							rv = g_new0 (RowValue, 1);
							rv->row_modif = rm;
							rv->model_column = model_col;
							rv->attributes = proxy->priv->cols_non_modif_attrs [col];
							
							rv->value = NULL;
							rv->attributes &= ~GNOME_DB_VALUE_IS_UNCHANGED;
							if (rm->model_row >= 0)
								rv->attributes |= GNOME_DB_VALUE_HAS_VALUE_ORIG;
							else
								rv->attributes &= ~GNOME_DB_VALUE_HAS_VALUE_ORIG;
							
							rm->modif_values = g_slist_prepend (rm->modif_values, rv);	
						}

						rv->attributes |= GNOME_DB_VALUE_IS_DEFAULT;
						gtk_tree_model_row_changed (GTK_TREE_MODEL (proxy), path, iter);
					}
					if (action & GNOME_DB_VALUE_IS_UNCHANGED) {
						if (!rm->orig_values)
							g_warning ("Action = GNOME_DB_VALUE_IS_UNCHANGED, no RowValue!");
						else
							gnome_db_data_proxy_set_value (proxy, iter, model_col,
										       rm->orig_values [model_col]);
					}
				}
			}
			else
				g_warning ("Trying to set read-only column: %d", col);
		}
		else 
			g_warning ("Column %d needs a GDA_VALUE_TYPE_UINTEGER value", col);	
	}

	gtk_tree_path_free (path);

	if (value_set) {
		g_warning ("Unknown GnomeDbDataProxy column: %d", col);
		return FALSE;
	}

#ifdef debug_NO
	gnome_db_base_dump (proxy, 5);
#endif

	return TRUE;
}

/*
 * Stores the new RowValue in @rv
 */
RowModif *
find_or_create_row_modif (GnomeDbDataProxy *proxy, GtkTreeIter *iter, gint col, RowValue **ret_rv)
{
	RowModif *rm;
	RowValue *rv = NULL;
	gint model_row;

	model_row = iter_to_model_numbers (proxy, GPOINTER_TO_INT (iter->user_data));
	rm = find_row_modif_for_iter (proxy, iter);

	if (!rm) {
		/* create a new RowModif */
		g_assert (model_row >= 0);
		rm = row_modifs_new (proxy, iter);
		rm->model_row = model_row;
		if (model_row >= 0)
			g_hash_table_insert (proxy->priv->modif_rows, GINT_TO_POINTER (model_row), rm);
		proxy->priv->all_modifs = g_slist_prepend (proxy->priv->all_modifs, rm);
	}
	else {
		/* there are already some modifications to the row, try to catch the RowValue if available */
		GSList *list;
		
		list = rm->modif_values;
		while (list && !rv) {
			if (ROW_VALUE (list->data)->model_column == col)
				rv = ROW_VALUE (list->data);
			list = g_slist_next (list);
		}
	}

	if (ret_rv)
		*ret_rv = rv;
	return rm;
}

/**
 * gnome_db_data_proxy_get_value
 * @proxy: a #GnomeDbDataProxy object
 * @iter: a valid #GtkTreeIter
 * @col:
 *
 * Retreive the value stored at the row pointed by @iter and at the column @col.
 *
 * Returns: the #GdaValue, or %NULL on error
 */
const GdaValue *
gnome_db_data_proxy_get_value (GnomeDbDataProxy *proxy, GtkTreeIter *iter, gint col)
{
	GdaValue *retval = NULL;
	GValue gvalue;

	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), NULL);
	g_return_val_if_fail (proxy->priv, NULL);
	g_return_val_if_fail (iter, NULL);
	g_return_val_if_fail (iter->stamp == proxy->priv->stamp, NULL);

	memset (&gvalue, 0, sizeof (GValue));
	data_proxy_get_value (GTK_TREE_MODEL (proxy), iter, col, &gvalue);
	
	if (G_VALUE_HOLDS_POINTER (&gvalue))
		retval = g_value_get_pointer (&gvalue);
	g_value_unset (&gvalue);

	return retval;
}

/**
 * gnome_db_data_proxy_get_values
 * @proxy: a #GnomeDbDataProxy object
 * @iter: a valid #GtkTreeIter
 * @cols_index:
 * @n_cols:
 *
 * Retreive a whole list of values from the @proxy store. This function calls gnome_db_data_proxy_get_value()
 * for each column index specified in @cols_index, and generates a #GSlist on the way.
 *
 * Returns: a new list of values (the list must be freed, not the values), or %NULL if an error occured
 */
GSList *
gnome_db_data_proxy_get_values (GnomeDbDataProxy *proxy, GtkTreeIter *iter, gint *cols_index, gint n_cols)
{
	GSList *retval = NULL;
	gint i;
	const GdaValue *value;

	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), NULL);
	g_return_val_if_fail (proxy->priv, NULL);
	g_return_val_if_fail (iter, NULL);
	g_return_val_if_fail (iter->stamp == proxy->priv->stamp, NULL);

	for (i = 0; i < n_cols; i++) {
		value = gnome_db_data_proxy_get_value (proxy, iter, cols_index[i]);
		if (value)
			retval = g_slist_prepend (retval, value);
		else {
			g_slist_free (retval);
			return NULL;
		}
	}

	return g_slist_reverse (retval);
}


/**
 * gnome_db_data_proxy_delete
 * @proxy: a #GnomeDbDataProxy object
 * @iter:  A set #GtkTreeIter
 *
 * Marks the row pointerd by @iter to be deleted
 */
void
gnome_db_data_proxy_delete (GnomeDbDataProxy *proxy, GtkTreeIter *iter)
{
	RowModif *rm;
	gboolean do_signal = FALSE;
	gint model_row;

	g_return_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy));
	g_return_if_fail (proxy->priv);
	g_return_if_fail (iter);
	g_return_if_fail (iter->stamp == proxy->priv->stamp);

	model_row = iter_to_model_numbers (proxy, GPOINTER_TO_INT (iter->user_data));

	rm = find_row_modif_for_iter (proxy, iter);
	if (rm) {
		if (! rm->to_be_deleted) {
			if (rm->model_row < 0) {
				/* remove the row completely because it does not exist in the data model */
				GtkTreePath *path;
				
				proxy->priv->all_modifs = g_slist_remove (proxy->priv->all_modifs, rm);
				proxy->priv->new_rows = g_slist_remove (proxy->priv->new_rows, rm);
				row_modifs_free (rm);

				proxy->priv->stamp = g_random_int ();				
				path = gtk_tree_path_new ();
				gtk_tree_path_append_index (path, GPOINTER_TO_INT (iter->user_data));
				gtk_tree_model_row_deleted (GTK_TREE_MODEL (proxy), path);
				gtk_tree_path_free (path);
			}
			else {
				rm->to_be_deleted = TRUE;
				do_signal = TRUE;
			}
		}
	}
	else {
		/* the row is an existing row in the data model, create a new RowModif */
		rm = row_modifs_new (proxy, iter);
		rm->model_row = model_row;
		g_hash_table_insert (proxy->priv->modif_rows, GINT_TO_POINTER (model_row), rm);
		proxy->priv->all_modifs = g_slist_prepend (proxy->priv->all_modifs, rm);
		rm->to_be_deleted = TRUE;
		do_signal = TRUE;
	}

	if (do_signal) {
		GtkTreePath *path;
		
		path = gtk_tree_path_new ();
		gtk_tree_path_append_index (path, GPOINTER_TO_INT (iter->user_data));
		gtk_tree_model_row_changed (GTK_TREE_MODEL (proxy), path, iter);
		gtk_tree_path_free (path);
	}
}

/**
 * gnome_db_data_proxy_undelete
 * @proxy: a #GnomeDbDataProxy object
 * @iter:  A set #GtkTreeIter
 *
 * Remove the "to be deleted" mark the row pointerd by @iter, if it existed.
 */
void
gnome_db_data_proxy_undelete (GnomeDbDataProxy *proxy, GtkTreeIter *iter)
{
	RowModif *rm;
	gboolean do_signal = FALSE;
	gint model_row;

	g_return_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy));
	g_return_if_fail (proxy->priv);
	g_return_if_fail (iter);
	g_return_if_fail (iter->stamp == proxy->priv->stamp);

	model_row = iter_to_model_numbers (proxy, GPOINTER_TO_INT (iter->user_data));
	rm = find_row_modif_for_iter (proxy, iter);
	if (rm) {
		rm->to_be_deleted = FALSE;
		if (!rm->modif_values) {
			/* get rid of that RowModif */
			do_signal= TRUE;
			
			g_hash_table_remove (proxy->priv->modif_rows, GINT_TO_POINTER (model_row));
			proxy->priv->all_modifs = g_slist_remove (proxy->priv->all_modifs, rm);
			row_modifs_free (rm);
		}
		else
			do_signal= TRUE;
	}

	if (do_signal) {
		/* signal a change in the model */
		GtkTreePath *path;
		
		path = gtk_tree_path_new ();
		gtk_tree_path_append_index (path, GPOINTER_TO_INT (iter->user_data));
		gtk_tree_model_row_changed (GTK_TREE_MODEL (proxy), path, iter);
		gtk_tree_path_free (path);
	}
}

/**
 * gnome_db_data_proxy_get_iter_from_values
 * @proxy: a #GnomeDbDataProxy object
 * @iter: an unset #GtkTreeIter to set to the requested row
 * @values: a list of #GdaValue values
 * @cols_index: an array of #gint containing the column number to match each value of @values
 *
 * Sets @iter to the first row where all the values in @values at the columns identified at
 * @cols_index match. If the row can't be identified, then the contents of @iter is not modified.
 *
 * NOTE: the @cols_index array MUST contain a column index for each value in @values
 *
 * Returns: TRUE if the row has been identified @iter was set
 */
gboolean
gnome_db_data_proxy_get_iter_from_values (GnomeDbDataProxy *proxy, GtkTreeIter *iter,
					  GSList *values, gint *cols_index)
{
	GtkTreeIter intern;
	gboolean found = FALSE;

	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), FALSE);
	g_return_val_if_fail (proxy->priv, FALSE);
	g_return_val_if_fail (iter, FALSE);
	g_return_val_if_fail (values, FALSE);

	/* if there are still some rows waiting to be added in the idle loop, then force them to be added
	 * first, otherwise we might not find what we are looking for!
	 */
	if (proxy->priv->idle_add_event_source) {
		g_idle_remove_by_data (proxy);
		while (idle_add_model_rows (proxy)) ;
	}

	if (! gtk_tree_model_get_iter_first (GTK_TREE_MODEL (proxy), &intern))
		return FALSE;

	do {
		GSList *list;
		gint index;
		gboolean allequal = TRUE;
		GdaValue *value;

		list = values;
		index = 0;
		while (list && allequal) {
			if (cols_index)
				g_return_val_if_fail (cols_index [index] < proxy->priv->model_nb_cols, FALSE);

			gtk_tree_model_get (GTK_TREE_MODEL (proxy), &intern, 
					    cols_index ? cols_index [index] : index, &value, -1);
			if (gda_value_compare_ext ((GdaValue *) (list->data), value))
				allequal = FALSE;

			list = g_slist_next (list);
			index++;
		}

		if (allequal) {
			*iter = intern;
			found = TRUE;
		}
	} 
	while (!found && gtk_tree_model_iter_next (GTK_TREE_MODEL (proxy), &intern));
	
	return found;
}


/**
 * gnome_db_data_proxy_get_model
 * @proxy: a #GnomeDbDataProxy object
 *
 * Get the #GnomeDbDataModel which holds the unmodified (reference) data of @proxy
 *
 * Returns: the #GnomeDbDataModel
 */
GnomeDbDataModel *
gnome_db_data_proxy_get_model (GnomeDbDataProxy *proxy)
{
	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), NULL);
	g_return_val_if_fail (proxy->priv, NULL);

	return proxy->priv->model;
}

/**
 * gnome_db_data_proxy_append
 * @proxy: a #GnomeDbDataProxy object
 * @iter:  an unset #GtkTreeIter to set to the appended row
 *
 * Appends a new row to the proxy.
 */
void
gnome_db_data_proxy_append (GnomeDbDataProxy *proxy, GtkTreeIter *iter)
{
	RowModif *rm;
	GtkTreePath *path;
	gint col;
	GdaValue *value;
	const GdaValue *cvalue;
	
	g_return_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy));
	g_return_if_fail (proxy->priv);
	g_return_if_fail (iter);

	/* RowModif structure */
	rm = row_modifs_new (proxy, NULL);
	rm->model_row = -1;
	rm->orig_values = NULL; /* there is no original value */
	rm->orig_values_size = 0;

	proxy->priv->all_modifs = g_slist_prepend (proxy->priv->all_modifs, rm);
	proxy->priv->new_rows = g_slist_append (proxy->priv->new_rows, rm);

	/* new iter value */
	path = gtk_tree_path_new ();
	gtk_tree_path_append_index (path, proxy->priv->current_nb_rows + (proxy->priv->add_null_entry ? 1 : 0) +
				    g_slist_length (proxy->priv->new_rows) - 1);
	data_proxy_get_iter (GTK_TREE_MODEL (proxy), iter, path);

	/* signal row insertion */
	gtk_tree_model_row_inserted (GTK_TREE_MODEL (proxy), path, iter);
	gtk_tree_path_free (path);

	/* for the columns which allow a default value, set them to the default value */
	value = gda_value_new_uinteger (GNOME_DB_VALUE_IS_DEFAULT);
	for (col = 0; col < proxy->priv->model_nb_cols; col ++) {
		GnomeDbParameter *param;
		
		/* columns attributes */
		if (proxy->priv->cols_non_modif_attrs [col] & GNOME_DB_VALUE_CAN_BE_DEFAULT)
			gnome_db_data_proxy_set_value (proxy, iter, proxy->priv->model_nb_cols + col, value);

		/* columns original values */
		param = gnome_db_data_model_get_param_at_column (proxy->priv->model, proxy->priv->info_data_set, col);
		if (gnome_db_parameter_is_valid (param)) {
			cvalue = gnome_db_parameter_get_value (param);
			gnome_db_data_proxy_set_value (proxy, iter, col, cvalue);
		}
	}
	gda_value_free (value);
}

/**
 * gnome_db_data_proxy_has_been_modified
 * @proxy: a #GnomeDbDataProxy object
 *
 * Tell if any modification has been made to the data, or if the data in @proxy is the same
 * as the one in the #GnomeDbDataModel given when @proxy was created
 *
 * Returns: TRUE if some modifications have been made
 */
gboolean
gnome_db_data_proxy_has_been_modified (GnomeDbDataProxy *proxy)
{
	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), FALSE);
	g_return_val_if_fail (proxy->priv, FALSE);

	return proxy->priv->all_modifs ? TRUE : FALSE;
}

/**
 * gnome_db_data_proxy_reset_all
 * @proxy: a #GnomeDbDataProxy object
 *
 * Resets any modification made to @proxy
 */
void
gnome_db_data_proxy_reset_all (GnomeDbDataProxy *proxy)
{
	g_return_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy));
	g_return_if_fail (proxy->priv);
	GtkTreePath *path;
	GtkTreeIter iter;

	/* new rows are first treated and removed (no memory de-allocation here, though) */
	if (proxy->priv->new_rows) {
		while (proxy->priv->new_rows) {
			path = gtk_tree_path_new ();
			gtk_tree_path_append_index (path, proxy->priv->current_nb_rows + (proxy->priv->add_null_entry ? 1 : 0));
			proxy->priv->new_rows = g_slist_delete_link (proxy->priv->new_rows, proxy->priv->new_rows);

			gtk_tree_model_row_deleted (GTK_TREE_MODEL (proxy), path);
			gtk_tree_path_free (path);
		}

		proxy->priv->stamp = g_random_int ();
	}

	/* all modified rows are then treated (including memory de-allocation for new rows) */
	while (proxy->priv->all_modifs) {
		gint model_row = ROW_MODIF (proxy->priv->all_modifs->data)->model_row;
		
		row_modifs_free (ROW_MODIF (proxy->priv->all_modifs->data));
		if (model_row >= 0)
			g_hash_table_remove (proxy->priv->modif_rows, GINT_TO_POINTER (model_row));
		proxy->priv->all_modifs = g_slist_delete_link (proxy->priv->all_modifs, proxy->priv->all_modifs);
		
		if (model_row >= 0) {
			/* not a new row */
			path = gtk_tree_path_new_from_indices (model_to_iter_numbers (proxy, model_row), -1);
			data_proxy_get_iter (GTK_TREE_MODEL (proxy), &iter, path);
			gtk_tree_model_row_changed (GTK_TREE_MODEL (proxy), path, &iter);
			gtk_tree_path_free (path);
		}
	}
}

/**
 * gnome_db_data_proxy_reset_value
 * @proxy: a #GnomeDbDataProxy object
 * @col:
 * @row:
 *
 * Resets data at the corresponding row and column. @row and @col can be <0 if all the corresponding
 * rows and columns are to be resetted.
 */
void
gnome_db_data_proxy_reset_value (GnomeDbDataProxy *proxy, GtkTreeIter *iter, gint col)
{
	gint row;
	g_return_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy));
	g_return_if_fail (proxy->priv);
	g_return_if_fail (iter);
	g_return_if_fail (iter->stamp == proxy->priv->stamp);

	row = GPOINTER_TO_INT (iter->user_data);

	if ((col >= 0) && (col < proxy->priv->model_nb_cols)) {
		RowModif *rm;
		GtkTreePath *path;

		path = gtk_tree_path_new ();
		gtk_tree_path_append_index (path, row);
		
		rm = find_row_modif_for_iter (proxy, iter);
		if (rm && rm->modif_values) {
			/* there are some modifications to the row */
			GSList *list;
			RowValue *rv = NULL;
			
			list = rm->modif_values;
			while (list && !rv) {
				if (ROW_VALUE (list->data)->model_column == col)
					rv = ROW_VALUE (list->data);
				list = g_slist_next (list);
			}
			if (rv) {
				/* remove this RowValue from the RowList */
				rm->modif_values = g_slist_remove (rm->modif_values, rv);
				if (!rm->modif_values && !rm->to_be_deleted) {
					/* remove this RowList as well */
					proxy->priv->all_modifs = g_slist_remove (proxy->priv->all_modifs, rm);
					if (rm->model_row < 0) 
						proxy->priv->new_rows = g_slist_remove (proxy->priv->new_rows, rm);
					else
						g_hash_table_remove (proxy->priv->modif_rows, GINT_TO_POINTER (rm->model_row));
					row_modifs_free (rm);

					proxy->priv->stamp = g_random_int ();
					gtk_tree_model_row_deleted (GTK_TREE_MODEL (proxy), path);
				}
				else
					gtk_tree_model_row_changed (GTK_TREE_MODEL (proxy), path, iter);	
			}
		}

		gtk_tree_path_free (path);
	}
	else
		g_warning ("Unknown GnomeDbDataProxy column: %d", col);
}

static gboolean commit_row_modif (GnomeDbDataProxy *proxy, RowModif *rm, GError **error);

/**
 * gnome_db_data_proxy_commit_all
 * @proxy: a #GnomeDbDataProxy object
 * @error: place to store the error, or %NULL
 *
 * Commits all the modified data in the proxy back into the #GnomeDbDataModel. 
 * If an error occurs, then the data commiting stops, so
 * some data might still need to be commited after that function returns FALSE.
 *
 * Returns: TRUE if no error occured.
 */
gboolean
gnome_db_data_proxy_commit_all (GnomeDbDataProxy *proxy, GError **error)
{
	gboolean allok = TRUE;
	GSList *list;

	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), FALSE);
	g_return_val_if_fail (proxy->priv, FALSE);

	gda_data_model_freeze (GDA_DATA_MODEL (proxy->priv->model));
	proxy->priv->internal_changes ++;

	list = proxy->priv->all_modifs;
	while (list && allok) {
		allok = commit_row_modif (proxy, ROW_MODIF (list->data), error);
		list = g_slist_next (list);
	}

	proxy->priv->internal_changes --;
	gda_data_model_thaw (GDA_DATA_MODEL (proxy->priv->model)); /* => might make a refresh */

	return allok;
}

/**
 * gnome_db_data_proxy_commit_row
 * @proxy: a #GnomeDbDataProxy object
 * @iter: the modified row to commit, stored as a #GtkTreeIter, or %NULL
 * @error: place to store the error, or %NULL
 * 
 * Commits the modified data in the proxy back into the #GnomeDbDataModel.
 *
 * Returns: TRUE if no error occured.
 */
gboolean
gnome_db_data_proxy_commit_row (GnomeDbDataProxy *proxy, GtkTreeIter *iter, GError **error)
{
	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), FALSE);
	g_return_val_if_fail (proxy->priv, FALSE);
	g_return_val_if_fail (iter, FALSE);
	g_return_val_if_fail (iter->stamp == proxy->priv->stamp, FALSE);

#ifdef debug_NO
	DEBUG_HEADER;
	gnome_db_base_dump (proxy, 5);
#endif

	return commit_row_modif (proxy, find_row_modif_for_iter (proxy, iter), error);
}

/*
 * Commits the modifications held in one single RowModif structure.
 *
 * Returns: TRUE if no error occured
 */
static gboolean
commit_row_modif (GnomeDbDataProxy *proxy, RowModif *rm, GError **error)
{
	gboolean err = FALSE;

	if (!rm)
		return TRUE;

	proxy->priv->internal_changes ++;
	if (rm->to_be_deleted) {
		/* delete the row */
		const GdaRow *gdarow;
		
		g_assert (rm->model_row >= 0);
		gdarow = gda_data_model_get_row (GDA_DATA_MODEL (proxy->priv->model), rm->model_row);
		err = ! gda_data_model_remove_row (GDA_DATA_MODEL (proxy->priv->model), gdarow);
	}
	else {
		if (rm->model_row >= 0) {
			/* update the row */
			GdaRow *gdarow;
			GSList *list;
			GList *values = NULL;
			gint i;
			gboolean newvalue_found;
			GdaValue *newvalue;
			gboolean *is_default;
			
			g_assert (rm->modif_values);
			g_assert (rm->orig_values);
			is_default = g_new0 (gboolean, proxy->priv->model_nb_cols);
			for (i=0; i < rm->orig_values_size; i++) {
				newvalue_found = FALSE;
				newvalue = NULL;
				
				list = rm->modif_values;
				while (list && !newvalue_found) {
					if (ROW_VALUE (list->data)->model_column == i) {
						newvalue = ROW_VALUE (list->data)->value;
						newvalue_found = TRUE;
						is_default [i] = ROW_VALUE (list->data)->attributes & 
							GNOME_DB_VALUE_IS_DEFAULT;
					}
					list = g_slist_next (list);
				}
				if (!newvalue_found)
					newvalue = rm->orig_values[i];
				values = g_list_append (values, newvalue);
			}
			gdarow = gda_row_new_from_list (GDA_DATA_MODEL (proxy->priv->model), values);
			g_list_free (values);
			gda_row_set_number (gdarow, rm->model_row);
			for (i = 0; i < proxy->priv->model_nb_cols; i++) {
				if (is_default [i])
					gda_row_set_is_default (gdarow, i, TRUE);
			}
			err = ! gda_data_model_update_row (GDA_DATA_MODEL (proxy->priv->model), gdarow);
			gda_row_free (gdarow);
			g_free (is_default);
		}
		else {
			/* insert a new row */
			GdaRow *gdarow;
			GSList *list;
			GList *values = NULL;
			gint i;
			GdaValue *newvalue;
			gboolean *is_default;
			
			g_assert (rm->modif_values);
			is_default = g_new0 (gboolean, proxy->priv->model_nb_cols);
			for (i = 0; i < proxy->priv->model_nb_cols; i++) {
				newvalue = NULL;
				
				list = rm->modif_values;
				while (list && !newvalue) {
					if (ROW_VALUE (list->data)->model_column == i) {
						newvalue = ROW_VALUE (list->data)->value;
						is_default [i] = ROW_VALUE (list->data)->attributes & 
							GNOME_DB_VALUE_IS_DEFAULT;
					}
					list = g_slist_next (list);
				}
				values = g_list_append (values, newvalue);
			}
			
			gdarow = gda_row_new_from_list (GDA_DATA_MODEL (proxy->priv->model), values);
			g_list_free (values);
			for (i = 0; i < proxy->priv->model_nb_cols; i++) {
				if (is_default [i]) {
					gda_row_set_is_default (gdarow, i, TRUE);
				}
			}
			err = ! gda_data_model_append_row (GDA_DATA_MODEL (proxy->priv->model), gdarow);
			gda_row_free (gdarow);
			g_free (is_default);
		}
	}
	proxy->priv->internal_changes --;
	
	if (!err) {
		/* data refresh */
		if (proxy->priv->model_refresh_pending)
			model_data_refreshed_cb (proxy->priv->model, proxy);
	}
	else
		g_set_error (error, GNOME_DB_DATA_PROXY_ERROR, GNOME_DB_DATA_PROXY_COMMIT_ERROR,
			     _("Can't commit the modified values"));

	return !err;
}

/*
 * GtkTreeModel interface
 *
 * REM about the GtkTreeIter: only the iter->user_data is used to retreive a row in the data model:
 *     iter->user_data contains the requested row number in the GtkTreeModel numbers
 *     iter->stamp is reset any time the model changes.
 */
static GtkTreeModelFlags
data_proxy_get_flags (GtkTreeModel *tree_model)
{
	g_return_val_if_fail (IS_GNOME_DB_DATA_PROXY (tree_model), 0);

	return (GTK_TREE_MODEL_LIST_ONLY | GTK_TREE_MODEL_ITERS_PERSIST);
}

static gint
data_proxy_get_n_columns (GtkTreeModel *tree_model)
{
	GnomeDbDataProxy *proxy;
	gint nb = PROXY_NB_GEN_COLUMNS;

	g_return_val_if_fail (IS_GNOME_DB_DATA_PROXY (tree_model), 0);
	proxy = GNOME_DB_DATA_PROXY (tree_model);
	g_return_val_if_fail (proxy->priv, 0);

	nb += 3 * proxy->priv->model_nb_cols;
	return nb;
}

static GType
data_proxy_get_column_type (GtkTreeModel *tree_model, gint index)
{
	GnomeDbDataProxy *proxy;
	GType retval = 0;

	g_return_val_if_fail (IS_GNOME_DB_DATA_PROXY (tree_model), 0);
	proxy = GNOME_DB_DATA_PROXY (tree_model);
	g_return_val_if_fail (proxy->priv, 0);

	if (index < 0) {
		switch (index) {
		case PROXY_COL_MODEL_N_COLUMNS:
		case PROXY_COL_MODEL_ROW:
			retval = G_TYPE_INT;
			break;
		case PROXY_COL_MODEL_POINTER:
			retval = G_TYPE_POINTER;
			break;
		case PROXY_COL_MODIFIED:
		case PROXY_COL_TO_DELETE:
			retval = G_TYPE_BOOLEAN;
			break;
		}
	}
	else {
		if ((index < proxy->priv->model_nb_cols) || 
		    ((index >= 2*proxy->priv->model_nb_cols) && (index < 3*proxy->priv->model_nb_cols)))
			retval = G_TYPE_POINTER;
		else
			if (index < 2*proxy->priv->model_nb_cols)
				retval = G_TYPE_UINT;
	}

	if (retval == 0)
		g_warning ("Unknown GnomeDbDataProxy column: %d", index);

	return retval;
}

static gboolean
data_proxy_get_iter (GtkTreeModel *tree_model, GtkTreeIter *iter, GtkTreePath *path)
{
	GnomeDbDataProxy *proxy;
	gint *indices, n, depth;

	g_return_val_if_fail (IS_GNOME_DB_DATA_PROXY (tree_model), FALSE);
	proxy = GNOME_DB_DATA_PROXY (tree_model);
	g_return_val_if_fail (proxy->priv, FALSE);
	g_return_val_if_fail (path, FALSE);
	g_return_val_if_fail (iter, FALSE);
	
	indices = gtk_tree_path_get_indices (path);
	depth = gtk_tree_path_get_depth (path);
	g_return_val_if_fail (depth == 1, FALSE);

	n = indices[0]; /* the n-th top level row */
	if (n < gnome_db_data_proxy_get_n_rows (proxy)) {
		iter->stamp = proxy->priv->stamp;
		iter->user_data = GINT_TO_POINTER (n);
		return TRUE;
	}
	else
		return FALSE;
}

static GtkTreePath *
data_proxy_get_path (GtkTreeModel *tree_model, GtkTreeIter *iter)
{
	GnomeDbDataProxy *proxy;
	GtkTreePath *path;

	g_return_val_if_fail (IS_GNOME_DB_DATA_PROXY (tree_model), NULL);
	proxy = GNOME_DB_DATA_PROXY (tree_model);
	g_return_val_if_fail (proxy->priv, NULL);
	g_return_val_if_fail (iter, NULL);
	g_return_val_if_fail (iter->stamp == proxy->priv->stamp, NULL);

	path = gtk_tree_path_new ();
	gtk_tree_path_append_index (path, GPOINTER_TO_INT (iter->user_data));

	return path;
}

static void
data_proxy_get_value (GtkTreeModel *tree_model, GtkTreeIter *iter, gint column, GValue *value)
{
	gint model_row;
	GnomeDbDataProxy *proxy;
	RowModif *rm;
	gboolean value_set = FALSE;

	g_return_if_fail (IS_GNOME_DB_DATA_PROXY (tree_model));
	proxy = GNOME_DB_DATA_PROXY (tree_model);
	g_return_if_fail (proxy->priv);
	g_return_if_fail (iter);
	g_return_if_fail (iter->stamp == proxy->priv->stamp);
	g_return_if_fail (value);

	g_value_init (value, data_proxy_get_column_type (tree_model, column));

	model_row = iter_to_model_numbers (proxy, GPOINTER_TO_INT (iter->user_data));

	if (column < 0) {
		switch (column) {
		case PROXY_COL_MODEL_N_COLUMNS:
			g_value_set_int (value, proxy->priv->model_nb_cols);
			value_set = TRUE;
			break;
		case PROXY_COL_MODEL_ROW:
			g_value_set_int (value, model_row);
			value_set = TRUE;
			break;
		case PROXY_COL_MODEL_POINTER:
			g_value_set_pointer (value, proxy->priv->model);
			value_set = TRUE;
			break;
		case PROXY_COL_MODIFIED:
			rm = find_row_modif_for_iter (proxy, iter);
			g_value_set_boolean (value, rm && rm->modif_values ? TRUE : FALSE);
			value_set = TRUE;
			break;
		case PROXY_COL_TO_DELETE:
			rm = find_row_modif_for_iter (proxy, iter);
			g_value_set_boolean (value, rm && rm->to_be_deleted ? TRUE : FALSE);
			value_set = TRUE;
			break;
		}
	}
	else {
		if (column < 3 * proxy->priv->model_nb_cols) {
			RowModif *rm = NULL;
			gboolean value_has_modifs = FALSE;
			gint model_col = column % proxy->priv->model_nb_cols;
			gint what = column / proxy->priv->model_nb_cols; /* 0 <=> current GdaValue
									    1 <=> attributes
									    2 <=> original value */

			rm = find_row_modif_for_iter (proxy, iter);
			if (what == 2) { /* original value */
				if (rm) {
					if (rm->orig_values)
						g_value_set_pointer (value, 
								     rm->orig_values [model_col]);
					else
						g_value_set_pointer (value, NULL);
				}
				else {
					g_value_set_pointer (value,
							     gda_data_model_get_value_at (GDA_DATA_MODEL (proxy->priv->model), 
											  model_col, model_row));
					
				}
				value_set = TRUE;
			}
			else {
				if (rm && rm->modif_values) {
					/* there are some modifications to the row */
					GSList *list;
					RowValue *rv = NULL;
					
					list = rm->modif_values;
					while (list && !rv) {
						if (ROW_VALUE (list->data)->model_column == model_col)
							rv = ROW_VALUE (list->data);
						list = g_slist_next (list);
					}
					if (rv) {
						value_has_modifs = TRUE;
						if (what == 0)
							g_value_set_pointer (value, rv->value);
						else 
							g_value_set_uint (value, rv->attributes);
						value_set = TRUE;
					}
				}

				if (!value_has_modifs) {
					value_set = TRUE;
					if (model_row >= 0) {
						/* existing row */
						if (column < proxy->priv->model_nb_cols) 
							g_value_set_pointer (value, 
							gda_data_model_get_value_at (GDA_DATA_MODEL (proxy->priv->model), 
										     column, model_row));
						else {
							const GdaValue *gdavalue;
							guint retval;

							gdavalue = gda_data_model_get_value_at 
								(GDA_DATA_MODEL (proxy->priv->model), model_col, model_row);
							retval = proxy->priv->cols_non_modif_attrs [model_col];
							if (!gdavalue || gda_value_is_null (gdavalue))
								retval |= GNOME_DB_VALUE_IS_NULL;
							
							g_value_set_uint (value, retval);
						}
					}
					else {
						/* new row */
						if (column < proxy->priv->model_nb_cols) 
							g_value_set_pointer (value, NULL);
						else {
							guint retval;

							retval = proxy->priv->cols_non_modif_attrs [model_col];
							retval |= GNOME_DB_VALUE_IS_NULL;
							g_value_set_uint (value, retval);
						}
					}
				}

				if (what == 1) {
					/* compute the GNOME_DB_VALUE_DATA_NON_VALID attribute */
					if (! (proxy->priv->cols_non_modif_attrs[model_col] & GNOME_DB_VALUE_CAN_BE_NULL)) {
						guint retval;
						retval = g_value_get_uint (value);
						if (retval & GNOME_DB_VALUE_IS_NULL) {
							retval |= GNOME_DB_VALUE_DATA_NON_VALID;
							g_value_set_uint (value, retval);
						}
					}
				}
			}
		}
	}

	if (! value_set) 
		g_warning ("Unknown GnomeDbDataProxy column: %d", column);
}

static gboolean
data_proxy_iter_next (GtkTreeModel *tree_model, GtkTreeIter *iter)
{
	GnomeDbDataProxy *proxy;
	gint row;

	g_return_val_if_fail (IS_GNOME_DB_DATA_PROXY (tree_model), FALSE);
	proxy = GNOME_DB_DATA_PROXY (tree_model);
	g_return_val_if_fail (proxy->priv, FALSE);
	g_return_val_if_fail (iter, FALSE);
	g_return_val_if_fail (iter->stamp == proxy->priv->stamp, FALSE);
	
	row = GPOINTER_TO_INT (iter->user_data);
	row++;
	if (row >= gnome_db_data_proxy_get_n_rows (proxy)) 
		return FALSE;
	else {
		iter->user_data = GINT_TO_POINTER (row);
		return TRUE;
	}
}

static gboolean
data_proxy_iter_children (GtkTreeModel *tree_model, GtkTreeIter *iter, GtkTreeIter *parent)
{
	GnomeDbDataProxy *proxy;

	g_return_val_if_fail (IS_GNOME_DB_DATA_PROXY (tree_model), FALSE);
	proxy = GNOME_DB_DATA_PROXY (tree_model);
	g_return_val_if_fail (proxy->priv, FALSE);
	g_return_val_if_fail (iter, FALSE);
	
	if (!parent && (gnome_db_data_proxy_get_n_rows (proxy) > 0)) {
		iter->stamp = proxy->priv->stamp;
		iter->user_data = GINT_TO_POINTER (0);
		return TRUE;
	}
	else
		return FALSE;
}

static gboolean
data_proxy_iter_has_child (GtkTreeModel *tree_model, GtkTreeIter *iter)
{
	return FALSE;
}
static gint
data_proxy_iter_n_children (GtkTreeModel *tree_model, GtkTreeIter *iter)
{
	GnomeDbDataProxy *proxy;

	g_return_val_if_fail (IS_GNOME_DB_DATA_PROXY (tree_model), -1);
	proxy = GNOME_DB_DATA_PROXY (tree_model);
	g_return_val_if_fail (proxy->priv, -1);

	if (!iter)
		return gnome_db_data_proxy_get_n_rows (proxy);
	else
		return 0;
}

static gboolean
data_proxy_iter_nth_child (GtkTreeModel *tree_model, GtkTreeIter *iter, GtkTreeIter *parent, gint n)
{
	GnomeDbDataProxy *proxy;

	g_return_val_if_fail (IS_GNOME_DB_DATA_PROXY (tree_model), FALSE);
	proxy = GNOME_DB_DATA_PROXY (tree_model);
	g_return_val_if_fail (proxy->priv, FALSE);
	g_return_val_if_fail (iter, FALSE);

	if (!parent && (n < gnome_db_data_proxy_get_n_rows (proxy))) {
		iter->stamp = proxy->priv->stamp;
		iter->user_data = GINT_TO_POINTER (n);
		return TRUE;
	}
	else
		return FALSE;
}

static gboolean
data_proxy_iter_parent (GtkTreeModel *tree_model, GtkTreeIter *iter, GtkTreeIter *child)
{
	return FALSE;
}

#ifdef debug
static void
gnome_db_data_proxy_dump (GnomeDbDataProxy *proxy, guint offset)
{
	gchar *offstr, *str;
	gint n_cols;
	gint *cols_size;
	gchar *sep_col  = " | ";
	gchar *sep_row  = "-+-";
	gchar sep_fill = '-';
	gint i;
	GtkTreeIter iter;
	const GdaValue *value;
	
	g_return_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy));
	g_return_if_fail (proxy->priv);

        /* string for the offset */
        offstr = g_new0 (gchar, offset+1);
	memset (offstr, ' ', offset);

	g_print ("%s" D_COL_H1 "GnomeDbDataProxy" D_COL_NOR " %p\n", offstr, proxy);
	
	/* compute the columns widths: using column titles... */
	n_cols = proxy->priv->model_nb_cols;
	cols_size = g_new0 (gint, n_cols);
	
	for (i = 0; i < n_cols; i++) {
		str = gda_data_model_get_column_title (GDA_DATA_MODEL (proxy->priv->model), i);
		if (str && *str)
			cols_size [i] = strlen (str);
		else
			cols_size [i] = 0;	
	}

	/* ... and using column data */
	if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (proxy), &iter)) {
		for (i = 0; i < n_cols; i++) {
			value = gnome_db_data_proxy_get_value (proxy, &iter, i);
			str = value ? gda_value_stringify (value) : g_strdup ("_null_");
			cols_size [i] = MAX (cols_size [i], strlen (str));
			g_free (str);
		}

		while (gtk_tree_model_iter_next (GTK_TREE_MODEL (proxy), &iter)) {
			for (i = 0; i < n_cols; i++) {
				value = gnome_db_data_proxy_get_value (proxy, &iter, i);
				str = value ? gda_value_stringify (value) : g_strdup ("_null_");
				cols_size [i] = MAX (cols_size [i], strlen (str));
				g_free (str);
			}
		}
	}
	
	/* actual dumping of the contents: column titles...*/
	for (i = 0; i < n_cols; i++) {
		str = gda_data_model_get_column_title (GDA_DATA_MODEL (proxy->priv->model), i);
		if (i != 0)
			g_print ("%s", sep_col);
		g_print ("%*s", cols_size [i], str);
	}
	g_print ("\n");

	/* ... separation line ... */
	for (i = 0; i < n_cols; i++) {
		gint j;
		if (i != 0)
			g_print ("%s", sep_row);
		for (j = 0; j < cols_size [i]; j++)
			g_print ("%c", sep_fill);
	}
	g_print ("\n");

	/* ... and data */
	if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (proxy), &iter)) {
		for (i = 0; i < n_cols; i++) {
			value = gnome_db_data_proxy_get_value (proxy, &iter, i);
			str = value ? gda_value_stringify (value) : g_strdup ("_null_");
			if (i != 0)
				g_print ("%s", sep_col);
			g_print ("%*s", cols_size [i], str);
			g_free (str);
		}
		g_print ("\n");
		while (gtk_tree_model_iter_next (GTK_TREE_MODEL (proxy), &iter)) {
			for (i = 0; i < n_cols; i++) {
				value = gnome_db_data_proxy_get_value (proxy, &iter, i);
				str = value ? gda_value_stringify (value) : g_strdup ("_null_");
				if (i != 0)
					g_print ("%s", sep_col);
				g_print ("%*s", cols_size [i], str);
				g_free (str);
			}
			g_print ("\n");
		}
	}

	g_free (cols_size);
	g_free (offstr);
}
#endif

/**
 * gnome_db_data_proxy_set_sample_size
 * @proxy: a #GnomeDbDataProxy object
 * @sample_size: the requested size of a chunck, or 0
 *
 * Sets the size of each chunck of fata to display: the maximum number of rows which
 * can be displayed at a time. The default value is arbitrary 300 as it is big enough to
 * be able to display quite a lot of data, but small enough to avoid too much data
 * displayed at the same time.
 *
 * Note: the rows which have been added but not yet commited will always be displayed
 * regardless of the current chunck of data, and the modified rows which are not visible
 * when the displayed chunck of data changes are still held as modified rows.
 *
 * To remove the chuncking of the data to display, simply pass @sample_size the 0 value.
 */
void
gnome_db_data_proxy_set_sample_size (GnomeDbDataProxy *proxy, gint sample_size)
{
	gint new_sample_size;
	g_return_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy));
	g_return_if_fail (proxy->priv);

	new_sample_size = sample_size <= 0 ? 0 : sample_size;
	if (proxy->priv->sample_size != new_sample_size) {
		proxy->priv->sample_size = new_sample_size;
		adjust_displayed_chunck (proxy);
	}
}

/**
 * gnome_db_data_proxy_get_sample_size
 * @proxy: a #GnomeDbDataProxy object
 *
 * Get the size of each chunk of data displayed at a time.
 *
 * Returns: the chunck (or sample) size, or 0 if chunking is disabled.
 */
gint
gnome_db_data_proxy_get_sample_size (GnomeDbDataProxy *proxy)
{
	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), 0);
	g_return_val_if_fail (proxy->priv, 0);

	return proxy->priv->sample_size;
}

/**
 * gnome_db_data_proxy_set_sample_start
 * @proxy: a #GnomeDbDataProxy object
 * @sample_start: the number of the first row to be displayed
 *
 * Sets the number of the first row to be displayed.
 */
void
gnome_db_data_proxy_set_sample_start (GnomeDbDataProxy *proxy, gint sample_start)
{
	g_return_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy));
	g_return_if_fail (proxy->priv);
	g_return_if_fail (sample_start >= 0);

	if (proxy->priv->sample_first_row != sample_start) {
		proxy->priv->sample_first_row = sample_start;
		adjust_displayed_chunck (proxy);
	}
}

/**
 * gnome_db_data_proxy_get_sample_start
 * @proxy: a #GnomeDbDataProxy object
 *
 * Get the row number of the first row to be displayed.
 *
 * Returns: the number of the first row being displayed.
 */
gint
gnome_db_data_proxy_get_sample_start (GnomeDbDataProxy *proxy)
{
	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), 0);
	g_return_val_if_fail (proxy->priv, 0);

	return proxy->priv->sample_first_row;
}

/**
 * gnome_db_data_proxy_get_sample_end
 * @proxy: a #GnomeDbDataProxy object
 *
 * Get the row number of the last row to be displayed.
 *
 * Returns: the number of the last row being displayed.
 */
gint
gnome_db_data_proxy_get_sample_end (GnomeDbDataProxy *proxy)
{
	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), 0);
	g_return_val_if_fail (proxy->priv, 0);

	return proxy->priv->sample_last_row;	
}

/*
 * Adjusts the values of the first and last rows to be displayed depending
 * on the sample size, and update the GtkTreeModel interface
 */
static void
adjust_displayed_chunck (GnomeDbDataProxy *proxy)
{
	gint i, old_nb_rows, new_nb_rows;
	GtkTreePath *path;
	GtkTreeIter iter;
	gint model_nb_rows;

	g_return_if_fail (proxy->priv->model);

	/*
	 * Stop idle adding of rows if necessary
	 */
	if (proxy->priv->idle_add_event_source) {
		g_idle_remove_by_data (proxy);
		proxy->priv->idle_add_event_source = 0;
	}

	/*
	 * Compute chuncks limits
	 */
	old_nb_rows = proxy->priv->current_nb_rows;
	model_nb_rows = gda_data_model_get_n_rows (GDA_DATA_MODEL (proxy->priv->model));
	if (proxy->priv->sample_size >= 0) {
		if (proxy->priv->sample_first_row >= model_nb_rows) 
			proxy->priv->sample_first_row = proxy->priv->sample_size * ((model_nb_rows - 1) / proxy->priv->sample_size);

		proxy->priv->sample_last_row = proxy->priv->sample_first_row + proxy->priv->sample_size - 1;
		if (proxy->priv->sample_last_row >= model_nb_rows)
			proxy->priv->sample_last_row = model_nb_rows - 1;
		new_nb_rows = proxy->priv->sample_last_row - proxy->priv->sample_first_row + 1;
	}
	else {
		proxy->priv->sample_first_row = 0;
		proxy->priv->sample_last_row = model_nb_rows - 1;
		new_nb_rows = proxy->priv->sample_last_row;
	}

	/*
	 * GtkTreeModel: update all the rows which can be updated:
	 * emit the GtkTreeModel::"row_changed" signal for all the rows which already existed
	 */
	for (i=0; (i < old_nb_rows) && (i < new_nb_rows); i++) {
		path = gtk_tree_path_new_from_indices (model_to_iter_numbers (proxy, proxy->priv->sample_first_row + i), -1);
		data_proxy_get_iter (GTK_TREE_MODEL (proxy), &iter, path);
		gtk_tree_model_row_changed (GTK_TREE_MODEL (proxy), path, &iter);
		gtk_tree_path_free (path);
	}
	
	if (old_nb_rows < new_nb_rows) {
		/*
		 * GtkTreeModel: insert the missing rows in the idle loop
		 */
		proxy->priv->stamp = g_random_int ();
		proxy->priv->idle_add_event_source = g_idle_add ((GSourceFunc) idle_add_model_rows, proxy);

		proxy->priv->current_nb_rows = old_nb_rows;
	}
	else {
		/*
		 * GtkTreeModel: delete all the remaining rows:
		 * emit the GtkTreeModel::"row_deleted" signal for all the new rows 
		 * (using the same row number!) 
		 */
		path = gtk_tree_path_new_from_indices (model_to_iter_numbers (proxy, proxy->priv->sample_first_row + i), -1);
		for (; i < old_nb_rows; i++) 
			gtk_tree_model_row_deleted (GTK_TREE_MODEL (proxy), path);
		gtk_tree_path_free (path);
		proxy->priv->stamp = g_random_int ();

		proxy->priv->current_nb_rows = new_nb_rows;
	}
}

/*
 * Called in an idle loop to "add" rows to the proxy. The only effect is to
 * "declare" more rows in the GtkTreeModel interface. Returns FALSE when
 * it does not need to be called anymore.
 *
 * Rem: must be called only by the adjust_displayed_chunck() function.
 */
static gboolean
idle_add_model_rows (GnomeDbDataProxy *proxy)
{
	gint tmp_current_nb_rows;
	gint model_nb_rows;
	gint i = 0;
	GtkTreePath *path;
	GtkTreeIter iter;

#define IDLE_STEP 50

	g_return_val_if_fail (proxy->priv->model, FALSE);

	model_nb_rows = gda_data_model_get_n_rows (GDA_DATA_MODEL (proxy->priv->model));
	if (proxy->priv->sample_size > 0) {
		tmp_current_nb_rows = proxy->priv->sample_size;
		if (tmp_current_nb_rows >= model_nb_rows)
			tmp_current_nb_rows = model_nb_rows;
	}
	else
		tmp_current_nb_rows = model_nb_rows;
	
	while ((i < IDLE_STEP) && (proxy->priv->current_nb_rows < tmp_current_nb_rows)) {
		proxy->priv->current_nb_rows ++;
		path = gtk_tree_path_new_from_indices (model_to_iter_numbers (proxy, 
									      proxy->priv->sample_first_row + proxy->priv->current_nb_rows - 1), -1);
		data_proxy_get_iter (GTK_TREE_MODEL (proxy), &iter, path);
		gtk_tree_model_row_inserted (GTK_TREE_MODEL (proxy), path, &iter);
		gtk_tree_path_free (path);
		i++;
	}
	proxy->priv->stamp = g_random_int ();

	if (i < IDLE_STEP) {
		proxy->priv->idle_add_event_source = 0;
		return FALSE;
	}
	else
		return TRUE;
}
