/* gnome-db-qf-all.c
 *
 * Copyright (C) 2003 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-qf-all.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-field.h"
#include "gnome-db-entity.h"
#include "gnome-db-renderer.h"
#include "gnome-db-referer.h"
#include "gnome-db-ref-base.h"
#include "marshal.h"
#include "gnome-db-query.h"
#include "gnome-db-target.h"
#include <string.h>

/* 
 * Main static functions 
 */
static void gnome_db_qf_all_class_init (GnomeDbQfAllClass * class);
static void gnome_db_qf_all_init (GnomeDbQfAll *qf);
static void gnome_db_qf_all_dispose (GObject *object);
static void gnome_db_qf_all_finalize (GObject *object);

static void gnome_db_qf_all_set_property (GObject              *object,
				    guint                 param_id,
				    const GValue         *value,
				    GParamSpec           *pspec);
static void gnome_db_qf_all_get_property (GObject              *object,
				    guint                 param_id,
				    GValue               *value,
				    GParamSpec           *pspec);

/* XML storage interface */
static void        gnome_db_qf_all_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_qf_all_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_qf_all_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_qf_all_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

/* Field interface */
static void              gnome_db_qf_all_field_init      (GnomeDbFieldIface *iface);
static GnomeDbEntity         *gnome_db_qf_all_get_entity      (GnomeDbField *iface);
static GnomeDbServerDataType *gnome_db_qf_all_get_data_type   (GnomeDbField *iface);

/* Renderer interface */
static void            gnome_db_qf_all_renderer_init      (GnomeDbRendererIface *iface);
static GdaXqlItem     *gnome_db_qf_all_render_as_xql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error);
static gchar          *gnome_db_qf_all_render_as_sql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error);
static gchar          *gnome_db_qf_all_render_as_str   (GnomeDbRenderer *iface, GnomeDbDataSet *context);

/* Referer interface */
static void        gnome_db_qf_all_referer_init        (GnomeDbRefererIface *iface);
static gboolean    gnome_db_qf_all_activate            (GnomeDbReferer *iface);
static void        gnome_db_qf_all_deactivate          (GnomeDbReferer *iface);
static gboolean    gnome_db_qf_all_is_active           (GnomeDbReferer *iface);
static GSList     *gnome_db_qf_all_get_ref_objects     (GnomeDbReferer *iface);
static void        gnome_db_qf_all_replace_refs        (GnomeDbReferer *iface, GHashTable *replacements);

/* virtual functions */
static GObject    *gnome_db_qf_all_copy           (GnomeDbQfield *orig);
static gboolean    gnome_db_qf_all_is_equal       (GnomeDbQfield *qfield1, GnomeDbQfield *qfield2);


/* When the GnomeDbQuery is nullified */
static void nullified_object_cb (GObject *obj, GnomeDbQfAll *field);
static void target_removed_cb (GnomeDbQuery *query, GnomeDbTarget *target, GnomeDbQfAll *field);

#ifdef debug
static void gnome_db_qf_all_dump (GnomeDbQfAll *field, guint offset);
#endif

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	TEMPL_SIGNAL,
	LAST_SIGNAL
};

static gint gnome_db_qf_all_signals[LAST_SIGNAL] = { 0 };

/* properties */
enum
{
	PROP_0,
	PROP_QUERY
};


/* private structure */
struct _GnomeDbQfAllPrivate
{
	GnomeDbQuery    *query;
	GnomeDbRefBase  *target_ref;
};


/* module error */
GQuark gnome_db_qf_all_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_qf_all_error");
	return quark;
}


GType
gnome_db_qf_all_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbQfAllClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_qf_all_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbQfAll),
			0,
			(GInstanceInitFunc) gnome_db_qf_all_init
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_qf_all_xml_storage_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo field_info = {
			(GInterfaceInitFunc) gnome_db_qf_all_field_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo renderer_info = {
			(GInterfaceInitFunc) gnome_db_qf_all_renderer_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo referer_info = {
			(GInterfaceInitFunc) gnome_db_qf_all_referer_init,
			NULL,
			NULL
		};
		
		
		type = g_type_register_static (GNOME_DB_TYPE_QFIELD, "GnomeDbQfAll", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_TYPE_XML_STORAGE, &xml_storage_info);
		g_type_add_interface_static (type, GNOME_DB_TYPE_FIELD, &field_info);
		g_type_add_interface_static (type, GNOME_DB_TYPE_RENDERER, &renderer_info);
		g_type_add_interface_static (type, GNOME_DB_TYPE_REFERER, &referer_info);
	}
	return type;
}

static void 
gnome_db_qf_all_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_qf_all_get_xml_id;
	iface->save_to_xml = gnome_db_qf_all_save_to_xml;
	iface->load_from_xml = gnome_db_qf_all_load_from_xml;
}

static void
gnome_db_qf_all_field_init (GnomeDbFieldIface *iface)
{
	iface->get_entity = gnome_db_qf_all_get_entity;
	iface->get_data_type = gnome_db_qf_all_get_data_type;
}

static void
gnome_db_qf_all_renderer_init (GnomeDbRendererIface *iface)
{
	iface->render_as_xql = gnome_db_qf_all_render_as_xql;
	iface->render_as_sql = gnome_db_qf_all_render_as_sql;
	iface->render_as_str = gnome_db_qf_all_render_as_str;
	iface->is_valid = NULL;
}

static void
gnome_db_qf_all_referer_init (GnomeDbRefererIface *iface)
{
        iface->activate = gnome_db_qf_all_activate;
        iface->deactivate = gnome_db_qf_all_deactivate;
        iface->is_active = gnome_db_qf_all_is_active;
        iface->get_ref_objects = gnome_db_qf_all_get_ref_objects;
        iface->replace_refs = gnome_db_qf_all_replace_refs;
}

static void
gnome_db_qf_all_class_init (GnomeDbQfAllClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_qf_all_signals[TEMPL_SIGNAL] =
		g_signal_new ("templ_signal",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQfAllClass, templ_signal),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->templ_signal = NULL;

	object_class->dispose = gnome_db_qf_all_dispose;
	object_class->finalize = gnome_db_qf_all_finalize;

	/* Properties */
	object_class->set_property = gnome_db_qf_all_set_property;
	object_class->get_property = gnome_db_qf_all_get_property;
	g_object_class_install_property (object_class, PROP_QUERY,
					 g_param_spec_pointer ("query", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_qf_all_dump;
#endif
	GNOME_DB_QFIELD_CLASS (class)->copy = gnome_db_qf_all_copy;
	GNOME_DB_QFIELD_CLASS (class)->is_equal = gnome_db_qf_all_is_equal;
	GNOME_DB_QFIELD_CLASS (class)->is_list = NULL;
	GNOME_DB_QFIELD_CLASS (class)->get_params = NULL;
}

static void
gnome_db_qf_all_init (GnomeDbQfAll *gnome_db_qf_all)
{
	gnome_db_qf_all->priv = g_new0 (GnomeDbQfAllPrivate, 1);
	gnome_db_qf_all->priv->query = NULL;
	gnome_db_qf_all->priv->target_ref = NULL;
}


/**
 * gnome_db_qf_all_new_with_target
 * @query: a #GnomeDbQuery in which the new object will be
 * @target: a #GnomeDbTarget object
 *
 * Creates a new GnomeDbQfAll object which represents all the fields of the entity represented
 * by @target. For example if @target represents my_table, then the created object would
 * represent 'my_table.*' in SQL notation
 *
 * Returns: the new object
 */
GObject*
gnome_db_qf_all_new_with_target (GnomeDbQuery *query, GnomeDbTarget *target)
{
	GObject   *obj;
	GnomeDbQfAll *gnome_db_qf_all;
	GnomeDbDict *dict;
	guint id;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (target && IS_GNOME_DB_TARGET (target), NULL);
	g_return_val_if_fail (gnome_db_target_get_query (target) == query, NULL);
	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));

	obj = g_object_new (GNOME_DB_TYPE_QF_ALL, "dict", dict, NULL);
	gnome_db_qf_all = GNOME_DB_QF_ALL (obj);
	g_object_get (G_OBJECT (query), "field_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_qf_all), id);

	gnome_db_qf_all->priv->query = query;
	gnome_db_base_connect_nullify (query, 
				 G_CALLBACK (nullified_object_cb), gnome_db_qf_all);
	g_signal_connect (G_OBJECT (query), "target_removed",
			  G_CALLBACK (target_removed_cb), gnome_db_qf_all);

	gnome_db_qf_all->priv->target_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	gnome_db_ref_base_set_ref_object (gnome_db_qf_all->priv->target_ref, GNOME_DB_BASE (target));
	
	return obj;
}

/**
 * gnome_db_qf_all_new_with_xml_id
 * @query: a #GnomeDbQuery in which the new object will be
 * @target_xml_id: the XML Id of a #GnomeDbTarget object
 *
 * Creates a new GnomeDbQfAll object which represents all the fields of the entity represented
 * by the target identified by @target_xml_id. 
 *
 * Returns: the new object
 */
GObject*
gnome_db_qf_all_new_with_xml_id (GnomeDbQuery *query, const gchar *target_xml_id)
{
	GObject   *obj;
	GnomeDbQfAll *gnome_db_qf_all;
	GnomeDbDict *dict;
	gchar *str, *ptr, *tok, *qid;
	guint id;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (target_xml_id && *target_xml_id, NULL);
	qid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (query));
	str = g_strdup (target_xml_id);
	ptr = strtok_r (str, ":", &tok);
        g_return_val_if_fail (!strcmp (ptr, qid), NULL);
	g_free (qid);
	g_free (str);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));
	obj = g_object_new (GNOME_DB_TYPE_QF_ALL, "dict", dict, NULL);
	gnome_db_qf_all = GNOME_DB_QF_ALL (obj);
	g_object_get (G_OBJECT (query), "field_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_qf_all), id);

	gnome_db_qf_all->priv->query = query;
	gnome_db_base_connect_nullify (query, 
				 G_CALLBACK (nullified_object_cb), gnome_db_qf_all);
	g_signal_connect (G_OBJECT (query), "target_removed",
			  G_CALLBACK (target_removed_cb), gnome_db_qf_all);

	gnome_db_qf_all->priv->target_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	gnome_db_ref_base_set_ref_name (gnome_db_qf_all->priv->target_ref, GNOME_DB_TYPE_TARGET, REFERENCE_BY_XML_ID, target_xml_id);
	
	return obj;
}

static void 
nullified_object_cb (GObject *obj, GnomeDbQfAll *field)
{
	gnome_db_base_nullify (GNOME_DB_BASE (field));
}

static void
target_removed_cb (GnomeDbQuery *query, GnomeDbTarget *target, GnomeDbQfAll *field)
{
	GnomeDbBase *base;

	base = gnome_db_ref_base_get_ref_object (field->priv->target_ref);
	if (base && (GNOME_DB_TARGET (base) == target)) 
		gnome_db_base_nullify (GNOME_DB_BASE (field));	
}

static void
gnome_db_qf_all_dispose (GObject *object)
{
	GnomeDbQfAll *gnome_db_qf_all;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_QF_ALL (object));

	gnome_db_qf_all = GNOME_DB_QF_ALL (object);
	if (gnome_db_qf_all->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		if (gnome_db_qf_all->priv->query) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_qf_all->priv->query),
							      G_CALLBACK (nullified_object_cb), gnome_db_qf_all);
			g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_qf_all->priv->query),
							      G_CALLBACK (target_removed_cb), gnome_db_qf_all);
			gnome_db_qf_all->priv->query = NULL;
		}
		if (gnome_db_qf_all->priv->target_ref) {
			g_object_unref (G_OBJECT (gnome_db_qf_all->priv->target_ref));
			gnome_db_qf_all->priv->target_ref = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_qf_all_finalize (GObject   * object)
{
	GnomeDbQfAll *gnome_db_qf_all;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_QF_ALL (object));

	gnome_db_qf_all = GNOME_DB_QF_ALL (object);
	if (gnome_db_qf_all->priv) {
		g_free (gnome_db_qf_all->priv);
		gnome_db_qf_all->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_qf_all_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	GnomeDbQfAll *gnome_db_qf_all;
	gpointer ptr;

	gnome_db_qf_all = GNOME_DB_QF_ALL (object);
	if (gnome_db_qf_all->priv) {
		switch (param_id) {
		case PROP_QUERY:
			ptr = g_value_get_pointer (value);
			g_return_if_fail (ptr && IS_GNOME_DB_QUERY (ptr));

			if (gnome_db_qf_all->priv->query) {
				if (gnome_db_qf_all->priv->query == GNOME_DB_QUERY (ptr))
					return;

				g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_qf_all->priv->query),
								      G_CALLBACK (nullified_object_cb), gnome_db_qf_all);

				g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_qf_all->priv->query),
								      G_CALLBACK (target_removed_cb), gnome_db_qf_all);
			}

			gnome_db_qf_all->priv->query = GNOME_DB_QUERY (ptr);
			gnome_db_base_connect_nullify (ptr, 
						 G_CALLBACK (nullified_object_cb), gnome_db_qf_all);
			g_signal_connect (G_OBJECT (ptr), "target_removed",
					  G_CALLBACK (target_removed_cb), gnome_db_qf_all);
			break;
		}
	}
}

static void
gnome_db_qf_all_get_property (GObject              *object,
			guint                 param_id,
			GValue               *value,
			GParamSpec           *pspec)
{
	GnomeDbQfAll *gnome_db_qf_all;
	gnome_db_qf_all = GNOME_DB_QF_ALL (object);
	
	if (gnome_db_qf_all->priv) {
		switch (param_id) {
		case PROP_QUERY:
			g_value_set_pointer (value, gnome_db_qf_all->priv->query);
			break;
		}	
	}
}

static GObject *
gnome_db_qf_all_copy (GnomeDbQfield *orig)
{
	GnomeDbQfAll *qf;
	GObject *obj;
	g_assert (IS_GNOME_DB_QF_ALL (orig));
	qf = GNOME_DB_QF_ALL (orig);

	obj = gnome_db_qf_all_new_with_xml_id (qf->priv->query, 
					 gnome_db_ref_base_get_ref_name (qf->priv->target_ref, NULL, NULL));
	if (gnome_db_base_get_name (GNOME_DB_BASE (orig)))
		gnome_db_base_set_name (GNOME_DB_BASE (obj), gnome_db_base_get_name (GNOME_DB_BASE (orig)));

	if (gnome_db_base_get_description (GNOME_DB_BASE (orig)))
		gnome_db_base_set_description (GNOME_DB_BASE (obj), gnome_db_base_get_description (GNOME_DB_BASE (orig)));

	return obj;
}

static gboolean
gnome_db_qf_all_is_equal (GnomeDbQfield *qfield1, GnomeDbQfield *qfield2)
{
	const gchar *ref1, *ref2;
	g_assert (IS_GNOME_DB_QF_ALL (qfield1));
	g_assert (IS_GNOME_DB_QF_ALL (qfield2));
	
	/* it is here assumed that qfield1 and qfield2 are of the same type and refer to the same
	   query */
	ref1 = gnome_db_ref_base_get_ref_name (GNOME_DB_QF_ALL (qfield1)->priv->target_ref, NULL, NULL);
	ref2 = gnome_db_ref_base_get_ref_name (GNOME_DB_QF_ALL (qfield2)->priv->target_ref, NULL, NULL);

	return !strcmp (ref1, ref2) ? TRUE : FALSE;
}

/**
 * gnome_db_qf_all_get_target
 * @field: a #GnomeDbQfAll object
 *
 * Get the #GnomeDbTarget object @field 'belongs' to
 *
 * Returns: the #GnomeDbTarget object
 */
GnomeDbTarget *
gnome_db_qf_all_get_target (GnomeDbQfAll *field)
{
	GnomeDbBase *base;
	g_return_val_if_fail (field && IS_GNOME_DB_QF_ALL (field), NULL);
	g_return_val_if_fail (field->priv, NULL);

	base = gnome_db_ref_base_get_ref_object (field->priv->target_ref);
	if (base)
		return GNOME_DB_TARGET (base);
	else
		return NULL;
}

#ifdef debug
static void
gnome_db_qf_all_dump (GnomeDbQfAll *field, guint offset)
{
	gchar *str;
	gint i;

	g_return_if_fail (field && IS_GNOME_DB_QF_ALL (field));
	
        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (field->priv) {
                g_print ("%s" D_COL_H1 "GnomeDbQfAll" D_COL_NOR " \"%s\" (%p, id=%d) ",
                         str, gnome_db_base_get_name (GNOME_DB_BASE (field)), field, gnome_db_base_get_id (GNOME_DB_BASE (field)));
		if (gnome_db_qf_all_is_active (GNOME_DB_REFERER (field)))
			g_print ("Active, ");
		else
			g_print (D_COL_ERR "Inactive" D_COL_NOR ", ");
		g_print ("references %p (%s)\n", 
			 gnome_db_ref_base_get_ref_object (field->priv->target_ref),
			 gnome_db_ref_base_get_ref_name (field->priv->target_ref, NULL, NULL));
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, field);
}
#endif


/* 
 * GnomeDbField interface implementation
 */
static GnomeDbEntity *
gnome_db_qf_all_get_entity (GnomeDbField *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QF_ALL (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_ALL (iface)->priv, NULL);

	return GNOME_DB_ENTITY (GNOME_DB_QF_ALL (iface)->priv->query);
}

static GnomeDbServerDataType *
gnome_db_qf_all_get_data_type (GnomeDbField *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QF_ALL (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_ALL (iface)->priv, NULL);

	return NULL;
}

/* 
 * GnomeDbXmlStorage interface implementation
 */
static gchar *
gnome_db_qf_all_get_xml_id (GnomeDbXmlStorage *iface)
{
	gchar *q_xml_id, *xml_id;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_ALL (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_ALL (iface)->priv, NULL);

	q_xml_id = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (GNOME_DB_QF_ALL (iface)->priv->query));
	xml_id = g_strdup_printf ("%s:QF%d", q_xml_id, gnome_db_base_get_id (GNOME_DB_BASE (iface)));
	g_free (q_xml_id);
	
	return xml_id;
}

static xmlNodePtr
gnome_db_qf_all_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL;
	GnomeDbQfAll *field;
	gchar *str;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_ALL (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_ALL (iface)->priv, NULL);

	field = GNOME_DB_QF_ALL (iface);

	node = xmlNewNode (NULL, "GNOME_DB_QF");
	
	str = gnome_db_qf_all_get_xml_id (iface);
	xmlSetProp (node, "id", str);
	g_free (str);

	xmlSetProp (node, "type", "ALL");
	xmlSetProp (node, "name", gnome_db_base_get_name (GNOME_DB_BASE (field)));
	xmlSetProp (node, "target", gnome_db_ref_base_get_ref_name (field->priv->target_ref, NULL, NULL));
	if (! gnome_db_qfield_is_visible (GNOME_DB_QFIELD (field)))
		xmlSetProp (node, "is_visible",  "f");
	if (gnome_db_qfield_is_internal (GNOME_DB_QFIELD (field)))
		xmlSetProp (node, "is_internal", "t");

	return node;
}

static gboolean
gnome_db_qf_all_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbQfAll *field;
	gchar *prop;
	gboolean target = FALSE;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_ALL (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QF_ALL (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	field = GNOME_DB_QF_ALL (iface);
	if (strcmp (node->name, "GNOME_DB_QF")) {
		g_set_error (error,
			     GNOME_DB_QF_ALL_ERROR,
			     GNOME_DB_QF_ALL_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_QF>"));
		return FALSE;
	}

	prop = xmlGetProp (node, "type");
	if (prop) {
		if (strcmp (prop, "ALL")) {
			g_set_error (error,
				     GNOME_DB_QF_ALL_ERROR,
				     GNOME_DB_QF_ALL_XML_LOAD_ERROR,
				     _("Wrong type of field in <GNOME_DB_QF>"));
			return FALSE;
		}
		g_free (prop);
	}

	prop = xmlGetProp (node, "id");
	if (prop) {
		gchar *ptr, *tok;
		ptr = strtok_r (prop, ":", &tok);
		ptr = strtok_r (NULL, ":", &tok);
		if (strlen (ptr) < 3) {
			g_set_error (error,
				     GNOME_DB_QF_ALL_ERROR,
				     GNOME_DB_QF_ALL_XML_LOAD_ERROR,
				     _("Wrong 'id' attribute in <GNOME_DB_QF>"));
			return FALSE;
		}
		gnome_db_base_set_id (GNOME_DB_BASE (field), atoi (ptr+2));
		g_free (prop);
	}

	prop = xmlGetProp (node, "name");
	if (prop) {
		gnome_db_base_set_name (GNOME_DB_BASE (field), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "target");
	if (prop) {
		target = TRUE;
		gnome_db_ref_base_set_ref_name (field->priv->target_ref, GNOME_DB_TYPE_TARGET, REFERENCE_BY_XML_ID, prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "is_visible");
	if (prop) {
		gnome_db_qfield_set_visible (GNOME_DB_QFIELD (field), (*prop == 't') ? TRUE : FALSE);
		g_free (prop);
	}

	prop = xmlGetProp (node, "is_internal");
	if (prop) {
		gnome_db_qfield_set_internal (GNOME_DB_QFIELD (field), (*prop == 't') ? TRUE : FALSE);
		g_free (prop);
	}

	if (target)
		return TRUE;
	else {
		g_set_error (error,
			     GNOME_DB_QF_ALL_ERROR,
			     GNOME_DB_QF_ALL_XML_LOAD_ERROR,
			     _("Missing required attributes for <GNOME_DB_QF>"));
		return FALSE;
	}
}


/*
 * GnomeDbRenderer interface implementation
 */
static GdaXqlItem *
gnome_db_qf_all_render_as_xql (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error)
{
	GdaXqlItem *node = NULL;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_ALL (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_ALL (iface)->priv, NULL);
	
	TO_IMPLEMENT;
	return node;
}

static gchar *
gnome_db_qf_all_render_as_sql (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error)
{
	gchar *str = NULL;
	GnomeDbBase *base;
	GnomeDbQfAll *field;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_ALL (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_ALL (iface)->priv, NULL);
	field = GNOME_DB_QF_ALL (iface);

	base = gnome_db_ref_base_get_ref_object (field->priv->target_ref);
	if (base) 
		str = g_strdup_printf ("%s.*", gnome_db_target_get_alias (GNOME_DB_TARGET (base)));
	else
		g_set_error (error,
			     GNOME_DB_QF_ALL_ERROR,
			     GNOME_DB_QF_ALL_RENDER_ERROR,
			     _("Can't find target '%s'"), gnome_db_ref_base_get_ref_name (field->priv->target_ref,
										    NULL, NULL));
	
	return str;
}

static gchar *
gnome_db_qf_all_render_as_str (GnomeDbRenderer *iface, GnomeDbDataSet *context)
{
	gchar *str = NULL;
	GnomeDbBase *base;
	GnomeDbQfAll *field;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_ALL (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_ALL (iface)->priv, NULL);
	field = GNOME_DB_QF_ALL (iface);

	base = gnome_db_ref_base_get_ref_object (field->priv->target_ref);
	if (base) {
		GnomeDbEntity *ent = gnome_db_target_get_represented_entity (GNOME_DB_TARGET (base));
		str = g_strdup_printf ("%s(%s).*", gnome_db_base_get_name (GNOME_DB_BASE (ent)), 
				       gnome_db_target_get_alias (GNOME_DB_TARGET (base)));
	}
	else
		str = g_strdup (_("Non activated field"));
	return str;
}


/*
 * GnomeDbReferer interface implementation
 */
static gboolean
gnome_db_qf_all_activate (GnomeDbReferer *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QF_ALL (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QF_ALL (iface)->priv, FALSE);

	return gnome_db_ref_base_activate (GNOME_DB_QF_ALL (iface)->priv->target_ref);
}

static void
gnome_db_qf_all_deactivate (GnomeDbReferer *iface)
{
	g_return_if_fail (iface && IS_GNOME_DB_QF_ALL (iface));
	g_return_if_fail (GNOME_DB_QF_ALL (iface)->priv);

	gnome_db_ref_base_deactivate (GNOME_DB_QF_ALL (iface)->priv->target_ref);
}

static gboolean
gnome_db_qf_all_is_active (GnomeDbReferer *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QF_ALL (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QF_ALL (iface)->priv, FALSE);

	return gnome_db_ref_base_is_active (GNOME_DB_QF_ALL (iface)->priv->target_ref);
}

static GSList *
gnome_db_qf_all_get_ref_objects (GnomeDbReferer *iface)
{
	GSList *list = NULL;
        GnomeDbBase *base;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_ALL (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_ALL (iface)->priv, NULL);

        base = gnome_db_ref_base_get_ref_object (GNOME_DB_QF_ALL (iface)->priv->target_ref);
        if (base)
                list = g_slist_append (list, base);

        return list;
}

static void
gnome_db_qf_all_replace_refs (GnomeDbReferer *iface, GHashTable *replacements)
{
	GnomeDbQfAll *field;

        g_return_if_fail (iface && IS_GNOME_DB_QF_ALL (iface));
        g_return_if_fail (GNOME_DB_QF_ALL (iface)->priv);

        field = GNOME_DB_QF_ALL (iface);
        if (field->priv->query) {
                GnomeDbQuery *query = g_hash_table_lookup (replacements, field->priv->query);
                if (query) {
                        g_signal_handlers_disconnect_by_func (G_OBJECT (field->priv->query),
                                                              G_CALLBACK (nullified_object_cb), field);
                        g_signal_handlers_disconnect_by_func (G_OBJECT (field->priv->query),
                                                              G_CALLBACK (target_removed_cb), field);
                        field->priv->query = query;
			gnome_db_base_connect_nullify (query,
						 G_CALLBACK (nullified_object_cb), field);
                        g_signal_connect (G_OBJECT (query), "target_removed",
                                          G_CALLBACK (target_removed_cb), field);
                }
        }

        gnome_db_ref_base_replace_ref_object (field->priv->target_ref, replacements);
}
