/* gnome-db-raw-grid.c
 *
 * Copyright (C) 2002 - 2007 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include <gdk/gdkkeysyms.h>
#include <glib/gi18n-lib.h>
#include <libgda/libgda.h>
#include "gnome-db-raw-grid.h"
#include "gnome-db-data-widget.h"
#include "utility.h"
#include "gnome-db-tools.h"
#include "gnome-db-util.h"
#include "marshal.h"
#include "data-entries/gnome-db-data-cell-renderer-combo.h"
#include "data-entries/gnome-db-data-cell-renderer-info.h"

static void gnome_db_raw_grid_class_init (GnomeDbRawGridClass * class);
static void gnome_db_raw_grid_init (GnomeDbRawGrid *wid);
static void gnome_db_raw_grid_dispose (GObject *object);

static void gnome_db_raw_grid_set_property (GObject *object,
					    guint param_id,
					    const GValue *value,
					    GParamSpec *pspec);
static void gnome_db_raw_grid_get_property (GObject *object,
					    guint param_id,
					    GValue *value,
					    GParamSpec *pspec);

static void init_tree_view (GnomeDbRawGrid *grid);

static void proxy_destroyed_cb (GdaDataProxy *proxy, GnomeDbRawGrid *grid);
static void proxy_sample_changed_cb (GdaDataProxy *proxy, gint sample_start, gint sample_end, GnomeDbRawGrid *grid);
static void proxy_row_updated_cb (GdaDataProxy *proxy, gint proxy_row, GnomeDbRawGrid *grid);
static void proxy_reset_cb (GdaDataProxy *proxy, GnomeDbRawGrid *grid);
static void paramlist_public_data_changed_cb (GdaParameterList *paramlist, GnomeDbRawGrid *grid);
static void paramlist_param_plugin_changed_cb (GdaParameterList *paramlist, GdaParameter *param, GnomeDbRawGrid *grid);
static gboolean iter_row_to_change_cb (GdaDataModelIter *iter, gint row, GnomeDbRawGrid *grid);
static void iter_row_changed_cb (GdaDataModelIter *iter, gint row, GnomeDbRawGrid *grid);

/* GnomeDbDataWidget interface */
static void            gnome_db_raw_grid_widget_init           (GnomeDbDataWidgetIface *iface);
static GdaDataProxy   *gnome_db_raw_grid_get_proxy             (GnomeDbDataWidget *iface);
static void            gnome_db_raw_grid_col_set_show          (GnomeDbDataWidget *iface, gint column, gboolean shown);
static void            gnome_db_raw_grid_set_column_editable   (GnomeDbDataWidget *iface, gint column, gboolean editable);
static void            gnome_db_raw_grid_show_column_actions   (GnomeDbDataWidget *iface, gint column, gboolean show_actions);
static GtkActionGroup *gnome_db_raw_grid_get_actions_group     (GnomeDbDataWidget *iface);
static GdaDataModelIter *gnome_db_raw_grid_widget_get_data_set   (GnomeDbDataWidget *iface);

static GdaDataModel   *gnome_db_raw_grid_widget_get_gda_model             (GnomeDbDataWidget *iface);
static void            gnome_db_raw_grid_widget_set_gda_model             (GnomeDbDataWidget *iface, GdaDataModel *model);
static gboolean        gnome_db_raw_grid_widget_set_write_mode (GnomeDbDataWidget *iface, GnomeDbDataWidgetWriteMode mode);
static GnomeDbDataWidgetWriteMode gnome_db_raw_grid_widget_get_write_mode (GnomeDbDataWidget *iface);

typedef struct {
	GdaParameterListGroup *group;
	GtkCellRenderer       *data_cell;
	GtkCellRenderer       *info_cell;
	gboolean               info_shown;
	gboolean               data_locked; /* TRUE if no modification allowed on that column */
} ColumnData;

#define COLUMN_DATA(x) ((ColumnData *)(x))

static ColumnData *get_column_data (GnomeDbRawGrid *grid, GdaParameterListGroup *group);

struct _GnomeDbRawGridPriv
{
	GdaDataModel               *data_model;  /* data model provided by set_model() */
	GdaDataModelIter           *iter;        /* iterator for @store, used for its structure */
	GnomeDbDataStore           *store;       /* GtkTreeModel interface, using @proxy */
	GdaDataProxy               *proxy;       /* proxy data model, proxying @data_model */

	GSList                     *columns_data; /* list of ColumnData */

	gboolean                    default_show_info_cell;
	gboolean                    default_show_global_actions;

	GtkActionGroup             *actions_group;

	gint                        export_type; /* used by the export dialog */
	GnomeDbDataWidgetWriteMode  write_mode;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

/* signals */
enum
{
        SELECTION_CHANGED,
	DOUBLE_CLICKED,
	POPULATE_POPUP,
        LAST_SIGNAL
};

static gint gnome_db_raw_grid_signals[LAST_SIGNAL] = { 0, 0, 0 };


/* properties */
enum
{
        PROP_0,
	PROP_MODEL,
	PROP_INFO_CELL_VISIBLE,
	PROP_GLOBAL_ACTIONS_VISIBLE
};

/*Callbacks */

/*
 * Real initialization
 */

static gboolean tree_view_event_cb (GtkWidget *treeview, GdkEvent *event, GnomeDbRawGrid *grid);
static gint     tree_view_popup_button_pressed_cb (GtkWidget *widget, GdkEventButton *event, 
						   GnomeDbRawGrid *grid);

static void     tree_view_selection_changed_cb (GtkTreeSelection *selection, GnomeDbRawGrid *grid);
static void     tree_view_row_activated_cb     (GtkTreeView *tree_view, GtkTreePath *path, 
						GtkTreeViewColumn *column, GnomeDbRawGrid *grid);

static void action_new_cb (GtkAction *action, GnomeDbRawGrid *grid);
static void action_delete_cb (GtkAction *action, GnomeDbRawGrid *grid);
static void action_undelete_cb (GtkAction *action, GnomeDbRawGrid *grid);
static void action_commit_cb (GtkAction *action, GnomeDbRawGrid *grid);
static void action_reset_cb (GtkAction *action, GnomeDbRawGrid *grid);
static void action_first_chunck_cb (GtkAction *action, GnomeDbRawGrid *grid);
static void action_prev_chunck_cb (GtkAction *action, GnomeDbRawGrid *grid);
static void action_next_chunck_cb (GtkAction *action, GnomeDbRawGrid *grid);
static void action_last_chunck_cb (GtkAction *action, GnomeDbRawGrid *grid);

static GtkActionEntry ui_actions[] = {
	{ "ActionNew", GTK_STOCK_ADD, "_New", NULL, "Create a new data entry", G_CALLBACK (action_new_cb)},
	{ "ActionDelete", GTK_STOCK_REMOVE, "_Delete", NULL, "Delete the selected entry", G_CALLBACK (action_delete_cb)},
	{ "ActionUndelete", GTK_STOCK_UNDELETE, "_Undelete", NULL, "Cancels the deletion of the selected entry", 
	  G_CALLBACK (action_undelete_cb)},
	{ "ActionCommit", GTK_STOCK_SAVE, "_Commit", NULL, "Commit the latest changes", G_CALLBACK (action_commit_cb)},
	{ "ActionReset", GTK_STOCK_REFRESH, "_Reset", NULL, "Reset the data", G_CALLBACK (action_reset_cb)},
	{ "ActionFirstChunck", GTK_STOCK_GOTO_FIRST, "_First chunck", NULL, "Go to first chunck of records", 
	  G_CALLBACK (action_first_chunck_cb)},
	{ "ActionLastChunck", GTK_STOCK_GOTO_LAST, "_Last chunck", NULL, "Go to last chunck of records", 
	  G_CALLBACK (action_last_chunck_cb)},
	{ "ActionPrevChunck", GTK_STOCK_GO_BACK, "_Previous chunck", NULL, "Go to previous chunck of records", 
	  G_CALLBACK (action_prev_chunck_cb)},
	{ "ActionNextChunck", GTK_STOCK_GO_FORWARD, "Ne_xt chunck", NULL, "Go to next chunck of records",
	  G_CALLBACK (action_next_chunck_cb)}
};

GType
gnome_db_raw_grid_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbRawGridClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_raw_grid_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbRawGrid),
			0,
			(GInstanceInitFunc) gnome_db_raw_grid_init
		};		

		static const GInterfaceInfo data_widget_info = {
                        (GInterfaceInitFunc) gnome_db_raw_grid_widget_init,
                        NULL,
                        NULL
                };
		
		type = g_type_register_static (GTK_TYPE_TREE_VIEW, "GnomeDbRawGrid", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_TYPE_DATA_WIDGET, &data_widget_info);
	}

	return type;
}

static void
gnome_db_raw_grid_widget_init (GnomeDbDataWidgetIface *iface)
{
	iface->get_proxy = gnome_db_raw_grid_get_proxy;
	iface->col_set_show = gnome_db_raw_grid_col_set_show;
	iface->set_column_editable = gnome_db_raw_grid_set_column_editable;
	iface->show_column_actions = gnome_db_raw_grid_show_column_actions;
	iface->get_actions_group = gnome_db_raw_grid_get_actions_group;
	iface->get_data_set = gnome_db_raw_grid_widget_get_data_set;
	iface->get_gda_model = gnome_db_raw_grid_widget_get_gda_model;
	iface->set_gda_model = gnome_db_raw_grid_widget_set_gda_model;
	iface->set_write_mode = gnome_db_raw_grid_widget_set_write_mode;
	iface->get_write_mode = gnome_db_raw_grid_widget_get_write_mode;	
}

static void
gnome_db_raw_grid_class_init (GnomeDbRawGridClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	gnome_db_raw_grid_signals[SELECTION_CHANGED] = 
		g_signal_new ("selection_changed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbRawGridClass, selection_changed),
                              NULL, NULL,
                              marshal_VOID__BOOLEAN, G_TYPE_NONE,
                              1, G_TYPE_BOOLEAN);
	gnome_db_raw_grid_signals[DOUBLE_CLICKED] = 
		g_signal_new ("double_clicked",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbRawGridClass, double_clicked),
                              NULL, NULL,
                              marshal_VOID__INT, G_TYPE_NONE,
                              1, G_TYPE_INT);
	gnome_db_raw_grid_signals[POPULATE_POPUP] = 
		g_signal_new ("populate_popup",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbRawGridClass, populate_popup),
                              NULL, NULL,
                              marshal_VOID__OBJECT, G_TYPE_NONE,
                              1, GTK_TYPE_MENU);

	object_class->dispose = gnome_db_raw_grid_dispose;

	/* Properties */
        object_class->set_property = gnome_db_raw_grid_set_property;
        object_class->get_property = gnome_db_raw_grid_get_property;
	g_object_class_install_property (object_class, PROP_MODEL,
                                         g_param_spec_object ("model", _("Data to display"), NULL, GDA_TYPE_DATA_MODEL,
							      G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_INFO_CELL_VISIBLE,
                                         g_param_spec_boolean ("info_cell_visible", NULL, _("Info cell visible"), FALSE,
                                                               G_PARAM_READABLE | G_PARAM_WRITABLE));
															   
	g_object_class_install_property (object_class, PROP_GLOBAL_ACTIONS_VISIBLE,
                                         g_param_spec_boolean ("global_actions_visible", NULL, _("Global Actions visible"), FALSE,
                                                               G_PARAM_READABLE | G_PARAM_WRITABLE));
}

static void
gnome_db_raw_grid_init (GnomeDbRawGrid *grid)
{
	GtkTreeView *tree_view;
	GtkTreeSelection *selection;
	
	grid->priv = g_new0 (GnomeDbRawGridPriv, 1);
	grid->priv->store = NULL;
	grid->priv->proxy = NULL;
	grid->priv->default_show_info_cell = FALSE;
	grid->priv->default_show_global_actions = TRUE;
	grid->priv->columns_data = NULL;
	grid->priv->export_type = 2;
	grid->priv->write_mode = GNOME_DB_DATA_WIDGET_WRITE_ON_DEMAND;
	
	tree_view = GTK_TREE_VIEW (grid);
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_view), TRUE);
	gtk_tree_view_set_enable_search (GTK_TREE_VIEW (tree_view), TRUE);
	g_signal_connect (G_OBJECT (tree_view), "event",
			  G_CALLBACK (tree_view_event_cb), grid);
	g_signal_connect (G_OBJECT (tree_view), "button_press_event",
                          G_CALLBACK (tree_view_popup_button_pressed_cb), grid);

	/* selection and signal handling */
	selection = gtk_tree_view_get_selection (tree_view);
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);
	g_signal_connect (G_OBJECT (selection), "changed",
			  G_CALLBACK (tree_view_selection_changed_cb), grid);
	g_signal_connect (G_OBJECT (tree_view), "row_activated",
			  G_CALLBACK (tree_view_row_activated_cb), grid);

	/* action group */
	grid->priv->actions_group = gtk_action_group_new ("Actions");
	gtk_action_group_add_actions (grid->priv->actions_group, ui_actions, G_N_ELEMENTS (ui_actions), grid);
}

/**
 * gnome_db_raw_grid_new
 * @model: a #GdaDataModel
 *
 * Creates a new #GnomeDbRawGrid widget suitable to display the data in @model
 *
 *  Returns: the new widget
 */
GtkWidget *
gnome_db_raw_grid_new (GdaDataModel *model)
{
	GtkWidget *grid;

	g_return_val_if_fail (!model || GDA_IS_DATA_MODEL (model), NULL);

	grid = (GtkWidget *) g_object_new (GNOME_DB_TYPE_RAW_GRID, "model", model, NULL);

	return grid;
}

static void gnome_db_raw_grid_clean (GnomeDbRawGrid *grid);
static void
gnome_db_raw_grid_dispose (GObject *object)
{
	GnomeDbRawGrid *grid;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_RAW_GRID (object));
	grid = GNOME_DB_RAW_GRID (object);

	if (grid->priv) {
		gnome_db_raw_grid_clean (grid);

		if (grid->priv->actions_group) {
			g_object_unref (G_OBJECT (grid->priv->actions_group));
			grid->priv->actions_group = NULL;
		}

		/* the private area itself */
		g_free (grid->priv);
		grid->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static void
gnome_db_raw_grid_set_property (GObject *object,
				guint param_id,
				const GValue *value,
				GParamSpec *pspec)
{
	GnomeDbRawGrid *grid;

        grid = GNOME_DB_RAW_GRID (object);
        if (grid->priv) {
                switch (param_id) {
		case PROP_MODEL:
			{
				GdaDataModel *model = GDA_DATA_MODEL(g_value_get_object (value));
				if (model)
					g_return_if_fail (GDA_IS_DATA_MODEL (model));
	
				gnome_db_raw_grid_clean (grid);
				if (!model)
					return;
	
				grid->priv->store = GNOME_DB_DATA_STORE (gnome_db_data_store_new (model));
				grid->priv->proxy = gnome_db_data_store_get_proxy (grid->priv->store);
				grid->priv->data_model = gda_data_proxy_get_proxied_model (grid->priv->proxy);
				
				gda_object_connect_destroy (grid->priv->proxy,
								G_CALLBACK (proxy_destroyed_cb), grid);
				g_signal_connect (grid->priv->proxy, "sample_changed",
						  G_CALLBACK (proxy_sample_changed_cb), grid);
				g_signal_connect (grid->priv->proxy, "row_updated",
						  G_CALLBACK (proxy_row_updated_cb), grid);
				g_signal_connect (grid->priv->proxy, "reset",
						  G_CALLBACK (proxy_reset_cb), grid);
				
				grid->priv->iter = gda_data_model_create_iter (GDA_DATA_MODEL (grid->priv->proxy));
				
				g_signal_connect (grid->priv->iter, "public_data_changed",
						  G_CALLBACK (paramlist_public_data_changed_cb), grid);
				g_signal_connect (grid->priv->iter, "param_plugin_changed",
						  G_CALLBACK (paramlist_param_plugin_changed_cb), grid);

				g_signal_connect (grid->priv->iter, "row_changed",
						  G_CALLBACK (iter_row_changed_cb), grid);
				g_signal_connect (grid->priv->iter, "row_to_change",
						  G_CALLBACK (iter_row_to_change_cb), grid);

				gda_data_model_iter_invalidate_contents (grid->priv->iter);
							
				gtk_tree_view_set_model ((GtkTreeView *) grid, GTK_TREE_MODEL (grid->priv->store));
				init_tree_view (grid);
	
				g_signal_emit_by_name (object, "proxy_changed", grid->priv->proxy);
				g_signal_emit_by_name (object, "iter_changed", grid->priv->iter);
				
				break;
			}
				
		case PROP_INFO_CELL_VISIBLE: 
			{
				GSList *list = grid->priv->columns_data;
				gboolean show = g_value_get_boolean (value);
				grid->priv->default_show_info_cell = show;
	
				while (list) {
					COLUMN_DATA (list->data)->info_shown = show;
					g_object_set (G_OBJECT (COLUMN_DATA (list->data)->info_cell), "visible", 
						      show, NULL);
					list = g_slist_next (list);
				}
			}
			break;
			
		case PROP_GLOBAL_ACTIONS_VISIBLE:
			gtk_action_group_set_visible (grid->priv->actions_group, g_value_get_boolean (value));
			break;
		
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }
}

static void
gnome_db_raw_grid_get_property (GObject *object,
				 guint param_id,
				 GValue *value,
				 GParamSpec *pspec)
{
	GnomeDbRawGrid *grid;

        grid = GNOME_DB_RAW_GRID (object);
        if (grid->priv) {
                switch (param_id) {
		case PROP_MODEL:
			g_value_set_pointer (value, grid->priv->data_model);
			break;
		case PROP_INFO_CELL_VISIBLE:
			g_value_set_boolean(value, grid->priv->default_show_info_cell);
			break;
		case PROP_GLOBAL_ACTIONS_VISIBLE:
			g_value_set_boolean(value, grid->priv->default_show_global_actions);
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }	
}

/**
 * gnome_db_raw_grid_get_selection
 * @grid: a #GnomeDbRawGrid widget
 *
 * Returns the list of the currently selected rows in a #GnomeDbRawGrid widget. 
 * The returned value is a list of integers, which represent each of the selected rows.
 *
 * If new rows have been inserted, then those new rows will have a row number equal to -1.
 *
 * Returns: a new list, should be freed (by calling g_list_free) when no longer needed.
 */
GList *
gnome_db_raw_grid_get_selection (GnomeDbRawGrid *grid)
{
	GtkTreeSelection *selection;
	GList *selected_rows;

	g_return_val_if_fail (grid && GNOME_DB_IS_RAW_GRID (grid), NULL);
	g_return_val_if_fail (grid->priv, NULL);

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid));
	selected_rows = gtk_tree_selection_get_selected_rows (selection, NULL);
	if (selected_rows) {
		GList *list, *retlist = NULL;
		GtkTreeIter iter;
		gint row;

		list = selected_rows;
		while (list) {
			if (gtk_tree_model_get_iter (GTK_TREE_MODEL (grid->priv->store), &iter,
						     (GtkTreePath *)(list->data))) {
				gtk_tree_model_get (GTK_TREE_MODEL (grid->priv->store), &iter, 
						    DATA_STORE_COL_MODEL_ROW, &row, -1);
				retlist = g_list_prepend (retlist, GINT_TO_POINTER (row));
			}
			list = g_list_next (list);
		}
		g_list_foreach (selected_rows, (GFunc) gtk_tree_path_free, NULL);
		g_list_free (selected_rows);
		return g_list_reverse (retlist);
	}
	else
		return NULL;
}



/*
 * creates a new string where underscores '_' are replaced by double underscores '__'
 * WARNING: the old string is free'ed so it is possible to do "str=double_underscores(str);"
 */
static gchar *
replace_double_underscores (gchar *str)
{
        gchar **arr;
        gchar *ret;
	
        arr = g_strsplit (str, "_", 0);
        ret = g_strjoinv ("__", arr);
        g_strfreev (arr);
	g_free (str);
	
        return ret;
}

static void     cell_renderer_value_set_attributes (GtkTreeViewColumn *tree_column, GtkCellRenderer *cell,
						    GtkTreeModel *tree_model, GtkTreeIter *iter, GnomeDbRawGrid *grid);
static void     cell_renderer_info_set_attributes  (GtkTreeViewColumn *tree_column, GtkCellRenderer *cell,
						    GtkTreeModel *tree_model, GtkTreeIter *iter, GnomeDbRawGrid *grid);

static void     data_cell_value_changed (GtkCellRenderer *renderer, const gchar *path, 
					 const GValue *new_value, GnomeDbRawGrid *grid);
static void     data_cell_values_changed (GtkCellRenderer *renderer, const gchar *path, 
					  GSList *new_values, GSList *all_new_values, GnomeDbRawGrid *grid);
static void     data_cell_status_changed (GtkCellRenderer *renderer, const gchar *path, 
					  GdaValueAttribute requested_action, GnomeDbRawGrid *grid);

/*
 * Creates the GtkTreeView's columns, from the grid->priv->store GtkTreeModel
 */
static void
init_tree_view (GnomeDbRawGrid *grid)
{
	gint i;
	GtkTreeView *tree_view;
	GSList *list;
	GdaDict *dict;

	tree_view = GTK_TREE_VIEW (grid);

	/* Creation of the columns in the treeview, to fit the parameters in grid->priv->iter param list */
	dict = gda_object_get_dict (GDA_OBJECT (grid->priv->iter));
	list = GDA_PARAMETER_LIST (grid->priv->iter)->groups_list;
	i = 0;
	while (list) {
		GdaParameter *param;
		GdaParameterListGroup *group;
		GtkTreeViewColumn *column;
		GtkCellRenderer *renderer;
		ColumnData *column_data;
		gboolean explicit_hide = FALSE;

		group = GDA_PARAMETER_LIST_GROUP (list->data);

		/* update the list of columns data */
		column_data = get_column_data (grid, group);
		if (!column_data) {
			column_data = g_new0 (ColumnData, 1);
			column_data->group = group;
			column_data->info_shown = grid->priv->default_show_info_cell;
			column_data->data_locked = FALSE;
			grid->priv->columns_data = g_slist_append (grid->priv->columns_data, column_data);
		}

		/* create renderers */
		if (group->nodes_source) {
			/* parameters depending on a GdaDataModel */
			gchar *title;
			gboolean nullok = TRUE;
			GSList *nodes;
			
			nodes = group->nodes;
			while (nodes && nullok) {
				if (gda_parameter_get_not_null (GDA_PARAMETER 
								(GDA_PARAMETER_LIST_NODE (nodes->data)->param)))
					nullok = FALSE;
				nodes = g_slist_next (nodes);
			}
			
			/* determine title */
			if (g_slist_length (group->nodes) == 1)
				title = (gchar *) gda_object_get_name (GDA_OBJECT (GDA_PARAMETER_LIST_NODE (group->nodes->data)->param));
			else 
				title = (gchar *) gda_object_get_name (GDA_OBJECT (group->nodes_source->data_model));

			if (title)
				title = replace_double_underscores (g_strdup (title));
			else 
				/* FIXME: find a better label */
				title = g_strdup (_("Value"));

			/* FIXME: if nullok is FALSE, then set the column title in bold */

			renderer = gnome_db_data_cell_renderer_combo_new (GDA_PARAMETER_LIST (grid->priv->iter), group->nodes_source);
			column_data->data_cell = renderer;
			gtk_tree_view_insert_column_with_data_func (tree_view, i, title, renderer,
								    (GtkTreeCellDataFunc) cell_renderer_value_set_attributes, 
								    grid, NULL);
			column = gtk_tree_view_get_column (tree_view, i);
			g_free (title);

			g_signal_connect (G_OBJECT (renderer), "changed", 
					  G_CALLBACK (data_cell_values_changed), grid);

			g_object_set_data (G_OBJECT (column), "source", group->nodes_source);
			/* FIXME: compute @explicit_hide */
		}
		else {
			/* single direct parameter */
			GType g_type;
			gchar *plugin = NULL;
			gchar *title;
			
			param = GDA_PARAMETER (GDA_PARAMETER_LIST_NODE (group->nodes->data)->param);
			g_type = gda_parameter_get_g_type (param);

			title = (gchar *) gda_object_get_name (GDA_OBJECT (param));
			if (title && *title)
				title = replace_double_underscores (g_strdup (title));
			else
				title = NULL;
			if (!title)
				title = g_strdup (_("No title"));
			
			g_object_get (G_OBJECT (param), "entry_plugin", &plugin, NULL);
			renderer = gnome_db_util_new_cell_renderer (dict, g_type, plugin);
			column_data->data_cell = renderer;
			if (! column_data->data_locked)
				column_data->data_locked = GDA_PARAMETER_LIST_NODE (group->nodes->data)->hint & 
					GDA_PARAMETER_LIST_PARAM_READ_ONLY;
			gtk_tree_view_insert_column_with_data_func (tree_view, i, title, renderer,
								    (GtkTreeCellDataFunc) cell_renderer_value_set_attributes, 
								    grid, NULL);
			column = gtk_tree_view_get_column (tree_view, i);
			g_free (title);

			if (GDA_PARAMETER_LIST_NODE (group->nodes->data)->hint & GDA_PARAMETER_LIST_PARAM_HIDE)
				g_object_set_data (G_OBJECT (param), "_gnome_db_explicit_show", GINT_TO_POINTER (1));

			if (GPOINTER_TO_INT (g_object_get_data (G_OBJECT (param), "_gnome_db_explicit_show")) == 1)
				explicit_hide = TRUE;

			g_signal_connect (G_OBJECT (renderer), "changed", 
					  G_CALLBACK (data_cell_value_changed), grid);
		}

		if (explicit_hide)
			gtk_tree_view_column_set_visible (column, FALSE);

		g_object_set_data (G_OBJECT (column), "data_renderer", renderer);
		g_object_set (G_OBJECT (renderer), "editable", !column_data->data_locked, NULL);

		/* settings and signals */
		g_object_set (G_OBJECT (renderer), "editable", !column_data->data_locked, NULL);
		if (g_object_class_find_property (G_OBJECT_GET_CLASS (renderer), "set_default_if_invalid"))
			g_object_set (G_OBJECT (renderer), "set_default_if_invalid", TRUE, NULL);
		g_object_set_data (G_OBJECT (renderer), "group", group);
		g_object_set_data (G_OBJECT (column), "group", group);

		/* Adding the GValue's information cell as another GtkCellRenderer */
		renderer = gnome_db_data_cell_renderer_info_new (grid->priv->store, grid->priv->iter, group);
		column_data->info_cell = renderer;
		gtk_tree_view_column_pack_end (column, renderer, FALSE);
		gtk_tree_view_column_set_cell_data_func (column, renderer, 
							 (GtkTreeCellDataFunc) cell_renderer_info_set_attributes, 
							 grid, NULL);
		g_signal_connect (G_OBJECT (renderer), "status_changed",
				  G_CALLBACK (data_cell_status_changed), grid);
		g_object_set (G_OBJECT (renderer), "visible", column_data->info_shown, NULL);
		g_object_set_data (G_OBJECT (renderer), "group", group);

		/* Sorting data */
		/* gtk_tree_view_column_set_sort_column_id (column, i); */
		/* gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (tree_model), */
		/* i, (GtkTreeIterCompareFunc) tree_sortable_sort_values, node, NULL); */
		list = g_slist_next (list);
		i++;
	}

	/*
	 * Hiding some columns of the grid
	 */
	list = GDA_PARAMETER_LIST (grid->priv->iter)->nodes_list;
	while (list) {
		guint hint = GDA_PARAMETER_LIST_NODE (list->data)->hint;

		if (hint & GDA_PARAMETER_LIST_PARAM_HIDE) {
			GtkTreeViewColumn *viewcol;
			GdaParameterListGroup *group;
			gint pos;

			group = gda_parameter_list_find_group_for_param ((GdaParameterList *) grid->priv->iter, 
									 GDA_PARAMETER_LIST_NODE (list->data)->param);
			pos = g_slist_index (((GdaParameterList *) grid->priv->iter)->groups_list, group);
			g_assert (pos >= 0);

			viewcol = gtk_tree_view_get_column (GTK_TREE_VIEW (grid), pos);
			gtk_tree_view_column_set_visible (viewcol, FALSE);
		}
		list = g_slist_next (list);
	}
}

/*
 * Set the attributes for each cell renderer which is not the information cell renderer, 
 * called by each cell renderer before actually displaying anything.
 */
static void
cell_renderer_value_set_attributes (GtkTreeViewColumn *tree_column,
				    GtkCellRenderer *cell,
				    GtkTreeModel *tree_model,
				    GtkTreeIter *iter, GnomeDbRawGrid *grid)
{
	GdaParameterListGroup *group;
	guint attributes;
	gboolean to_be_deleted = FALSE;
	ColumnData *column_data;

	group = g_object_get_data (G_OBJECT (tree_column), "group");
	column_data = get_column_data (grid, group);
	
	if (group->nodes_source) {
		/* parameters depending on a GdaDataModel */
		GList *values = NULL;
		GdaParameterListSource *source;

		source = g_object_get_data (G_OBJECT (tree_column), "source");

		/* NOTE:
		 * For performances reasons we want to provide, if possible, all the values required by the combo cell
		 * renderer to draw whatever it wants, without further making it search for the values it wants in
		 * source->data_model.
		 *
		 * For this purpose, we try to get a complete list of values making one row of the node->data_for_params
		 * data model, so that the combo cell renderer has all the values it wants.
		 *
		 * NOTE2:
		 * This optimization is required anyway when source->data_model can be changed depending on
		 * external events and we can't know when it has changed.
		 */
		attributes = gnome_db_utility_proxy_compute_attributes_for_group (group, grid->priv->store, grid->priv->iter,
									 iter, &to_be_deleted);
		values = gnome_db_utility_proxy_compute_values_for_group (group, grid->priv->store, grid->priv->iter, iter,
								 TRUE);
		if (!values) {
			values = gnome_db_utility_proxy_compute_values_for_group (group, grid->priv->store, 
									 grid->priv->iter, iter, FALSE);
			g_object_set (G_OBJECT (cell), 
				      "values_display", values,
				      "value_attributes", attributes,
				      "editable", !column_data->data_locked && !(attributes & GDA_VALUE_ATTR_NO_MODIF),
				      "show_expander", !column_data->data_locked && !(attributes & GDA_VALUE_ATTR_NO_MODIF),
				      "cell_background", GNOME_DB_COLOR_NORMAL_MODIF,
				      "cell_background-set", 
				      ! (attributes & GDA_VALUE_ATTR_IS_UNCHANGED) || to_be_deleted,
				      "to_be_deleted", to_be_deleted, 
				      "visible", !(attributes & GDA_VALUE_ATTR_UNUSED),
				      NULL);
			g_list_free (values);
		}
		else {
			g_object_set (G_OBJECT (cell), 
				      "values_display", values,
				      "value_attributes", attributes,
				      "editable", !column_data->data_locked && !(attributes & GDA_VALUE_ATTR_NO_MODIF),
				      "show_expander", !column_data->data_locked && !(attributes & GDA_VALUE_ATTR_NO_MODIF),
				      "cell_background", GNOME_DB_COLOR_NORMAL_MODIF,
				      "cell_background-set", 
				      ! (attributes & GDA_VALUE_ATTR_IS_UNCHANGED) || to_be_deleted,
				      "to_be_deleted", to_be_deleted, 
				      "visible", !(attributes & GDA_VALUE_ATTR_UNUSED),
				      NULL);
			g_list_free (values);
		}
	}
	else {
		/* single direct parameter */
		gint col;
		gint offset;
		GValue *value;
		gint row;

		offset = gda_data_model_get_n_columns (gda_data_proxy_get_proxied_model (grid->priv->proxy));

		g_assert (g_slist_length (group->nodes) == 1);
		col = gda_data_model_iter_get_column_for_param (grid->priv->iter, 
								GDA_PARAMETER_LIST_NODE (group->nodes->data)->param);
		gtk_tree_model_get (GTK_TREE_MODEL (grid->priv->store), iter, 
				    DATA_STORE_COL_TO_DELETE, &to_be_deleted,
				    DATA_STORE_COL_MODEL_ROW, &row,
				    col, &value, 
				    offset + col, &attributes, -1);
		g_object_set (G_OBJECT (cell), 
			      "value", value,
			      "value_attributes", attributes,
			      "editable", !column_data->data_locked && !(attributes & GDA_VALUE_ATTR_NO_MODIF),
			      "cell_background", GNOME_DB_COLOR_NORMAL_MODIF,
			      "cell_background-set", ! (attributes & GDA_VALUE_ATTR_IS_UNCHANGED) || to_be_deleted,
			      "to_be_deleted", to_be_deleted, 
			      "visible", !(attributes & GDA_VALUE_ATTR_UNUSED),
			      NULL);
	}
}

/*
 * Set the attributes for each information cell renderer, 
 * called by each cell renderer before actually displaying anything.
 */
static void
cell_renderer_info_set_attributes (GtkTreeViewColumn *tree_column,
				   GtkCellRenderer *cell,
				   GtkTreeModel *tree_model,
				   GtkTreeIter *iter, GnomeDbRawGrid *grid)
{
	GdaParameterListGroup *group;
	guint attributes;
	gboolean to_be_deleted = FALSE;
	ColumnData *column_data;

	group = g_object_get_data (G_OBJECT (tree_column), "group");
	column_data = get_column_data (grid, group);
	
	if (group->nodes_source) {
		/* parameters depending on a GdaDataModel */
		GdaParameterListSource *source;

		source = g_object_get_data (G_OBJECT (tree_column), "source");

		attributes = gnome_db_utility_proxy_compute_attributes_for_group (group, grid->priv->store, grid->priv->iter, 
									 iter, &to_be_deleted);
		g_object_set (G_OBJECT (cell), 
			      "editable", !column_data->data_locked && !(attributes & GDA_VALUE_ATTR_NO_MODIF),
			      "value_attributes", attributes,
			      "cell_background", GNOME_DB_COLOR_NORMAL_MODIF,
			      "cell_background-set", ! (attributes & GDA_VALUE_ATTR_IS_UNCHANGED) || to_be_deleted,
			      "to_be_deleted", to_be_deleted, 
			      "visible", column_data->info_shown && !(attributes & GDA_VALUE_ATTR_UNUSED),
			      NULL);
	}
	else {
		/* single direct parameter */
		gint col;
		gint offset;
		gint row;
		
		offset = gda_data_model_get_n_columns (gda_data_proxy_get_proxied_model (grid->priv->proxy));

		g_assert (g_slist_length (group->nodes) == 1);
		col = gda_data_model_iter_get_column_for_param (grid->priv->iter, 
								GDA_PARAMETER_LIST_NODE (group->nodes->data)->param);
		gtk_tree_model_get (GTK_TREE_MODEL (grid->priv->store), iter, 
				    DATA_STORE_COL_TO_DELETE, &to_be_deleted,
				    DATA_STORE_COL_MODEL_ROW, &row,
				    offset + col, &attributes, -1);
		g_object_set (G_OBJECT (cell), 
			      "editable", !column_data->data_locked && !(attributes & GDA_VALUE_ATTR_NO_MODIF),
			      "value_attributes", attributes,
			      "cell_background", GNOME_DB_COLOR_NORMAL_MODIF,
			      "cell_background-set", ! (attributes & GDA_VALUE_ATTR_IS_UNCHANGED) || to_be_deleted,
			      "to_be_deleted", to_be_deleted, 
			      "visible", column_data->info_shown && !(attributes & GDA_VALUE_ATTR_UNUSED),
			      NULL);
	}
}

static gboolean set_iter_from_path (GnomeDbRawGrid *grid, const gchar *path, GtkTreeIter *iter);

/*
 * Callback when a value displayed as a GtkCellRenderer has been changed, for single parameter cell renderers
 *
 * Apply the new_value to the GTkTreeModel (grid->priv->store)
 */
static void
data_cell_value_changed (GtkCellRenderer *renderer, const gchar *path, const GValue *new_value, GnomeDbRawGrid *grid)
{
	GtkTreeIter iter;
	GdaParameterListGroup *group;
	
	group = g_object_get_data (G_OBJECT (renderer), "group");
	g_assert (g_slist_length (group->nodes) == 1);

	if (set_iter_from_path (grid, path, &iter)) {
		    gint col;
		    col = gda_data_model_iter_get_column_for_param (grid->priv->iter, 
								    GDA_PARAMETER_LIST_NODE (group->nodes->data)->param);
		    gnome_db_data_store_set_value (grid->priv->store, &iter, col, new_value);
	}

#ifdef debug_no
	DEBUG_HEADER;
	gda_object_dump (grid->priv->proxy, 5);
#endif
}


/*
 * Callback when a value displayed as a GnomeDbDataCellRendererCombo has been changed.
 *
 * Apply the new_value to the GTkTreeModel (grid->priv->store)
 */
static void
data_cell_values_changed (GtkCellRenderer *renderer, const gchar *path, 
			  GSList *new_values, GSList *all_new_values, GnomeDbRawGrid *grid)
{
	GtkTreeIter iter;
	GdaParameterListGroup *group;
	
	group = g_object_get_data (G_OBJECT (renderer), "group");
	g_assert (group->nodes_source);

	if (new_values)
		g_return_if_fail (g_slist_length (group->nodes) == g_slist_length (new_values));
	else
		/* the reason for not having any value is that the GnomeDbDataCellRendererCombo had no selected item */
		return;
	
#ifdef debug_NO
	DEBUG_HEADER;
	gda_object_dump (grid->priv->proxy, 0);
#endif

	if (set_iter_from_path (grid, path, &iter)) {
		GSList *list, *params;
		gint col, proxy_row;

		proxy_row = gnome_db_data_store_get_row_from_iter (grid->priv->store, &iter);

		/* update the GnomeDbDataStore */
		params = group->nodes;
		list = new_values;
		while (list) {
			col = gda_data_model_iter_get_column_for_param (grid->priv->iter, 
									GDA_PARAMETER_LIST_NODE (params->data)->param);
			gnome_db_data_store_set_value (grid->priv->store, &iter, col, (GValue *)(list->data));
			list = g_slist_next (list);
			params = g_slist_next (params);
		}
		
#ifdef PROXY_STORE_EXTRA_VALUES
		/* call gda_data_proxy_set_model_row_value() */
		gint i;
		for (i = 0; i < group->nodes_source->shown_n_cols; i++) {
			GValue *value;

			col = group->nodes_source->shown_cols_index[i];
			value = (GValue *) g_slist_nth_data (all_new_values, col);
			gda_data_proxy_set_model_row_value (grid->priv->proxy,
							    group->nodes_source->data_model,
							    proxy_row, col, value);
		}
#endif
	}

#ifdef debug_NO
	DEBUG_HEADER;
	gda_object_dump (grid->priv->proxy, 0);
#endif
}

static gboolean
set_iter_from_path (GnomeDbRawGrid *grid, const gchar *path, GtkTreeIter *iter)
{
	GtkTreePath *treepath;

	g_assert (path);

	treepath = gtk_tree_path_new_from_string (path);
	if (! gtk_tree_model_get_iter (GTK_TREE_MODEL (grid->priv->store), iter, treepath)) {
		gtk_tree_path_free (treepath);
		g_warning ("Can't get iter for path %s", path);
		return FALSE;
	}
	gtk_tree_path_free (treepath);
	
	return TRUE;
}


/*
 * Actions
 */
static void
data_cell_status_changed (GtkCellRenderer *renderer, const gchar *path, GdaValueAttribute requested_action,
			  GnomeDbRawGrid *grid)
{
	GtkTreePath *treepath;
	GtkTreeIter iter;
	GdaParameterListGroup *group;
	GtkTreeModel *tree_model;
	gint col;
	gint offset;
	GValue *attribute;

	offset = gda_data_model_get_n_columns (gda_data_proxy_get_proxied_model (grid->priv->proxy));

	group = g_object_get_data (G_OBJECT (renderer), "group");
	tree_model = GTK_TREE_MODEL (grid->priv->store);

	treepath = gtk_tree_path_new_from_string (path);
	if (! gtk_tree_model_get_iter (tree_model, &iter, treepath)) {
		gtk_tree_path_free (treepath);
		g_warning ("Can't get iter for path %s", path);
		return;
	}
	gtk_tree_path_free (treepath);
	
	g_value_set_uint (attribute = gda_value_new (G_TYPE_UINT), requested_action);
	if (group->nodes_source) {
		/* parameters depending on a GdaDataModel */
		gint proxy_row;
		GSList *list = group->nodes;
		proxy_row = gnome_db_data_store_get_row_from_iter (grid->priv->store, &iter);

		while (list) {
			col = gda_data_model_iter_get_column_for_param (grid->priv->iter, 
									GDA_PARAMETER_LIST_NODE (list->data)->param);
			gnome_db_data_store_set_value (grid->priv->store, &iter, offset + col, attribute);
			list = g_slist_next (list);
		}

#ifdef PROXY_STORE_EXTRA_VALUES
		/* call gda_data_proxy_set_model_row_value() */
		gint i;
		for (i = 0; i < group->nodes_source->shown_n_cols; i++) {
			col = group->nodes_source->shown_cols_index[i];
			
			if (requested_action & GDA_VALUE_ATTR_IS_NULL)
				gda_data_proxy_set_model_row_value (grid->priv->proxy, 
								    group->nodes_source->data_model,
								    proxy_row, col, NULL);
			else {
				if (requested_action & GDA_VALUE_ATTR_IS_UNCHANGED)
					gda_data_proxy_clear_model_row_value (grid->priv->proxy, 
									      group->nodes_source->data_model,
									      proxy_row, col);
				else {
					if (requested_action & GDA_VALUE_ATTR_IS_DEFAULT) {
						TO_IMPLEMENT;
					}
					else
						TO_IMPLEMENT;
				}
			}
		}
#endif
	}
	else {
		/* single direct parameter */
		gint col;

		g_assert (g_slist_length (group->nodes) == 1);
		col = gda_data_model_iter_get_column_for_param (grid->priv->iter, 
								GDA_PARAMETER_LIST_NODE (group->nodes->data)->param);
		gnome_db_data_store_set_value (grid->priv->store, &iter, offset + col, attribute);
	}
	gda_value_free (attribute);

#ifdef debug_no
	DEBUG_HEADER;
	gda_object_dump (grid->priv->proxy, 0);
#endif
}

static void
action_new_cb (GtkAction *action, GnomeDbRawGrid *grid)
{
	GtkTreeIter iter;
	GtkTreePath *path;

	if (gnome_db_data_store_append (grid->priv->store, &iter)) {
		path = gtk_tree_model_get_path (GTK_TREE_MODEL (grid->priv->store), &iter);
		gtk_tree_view_set_cursor (GTK_TREE_VIEW (grid), path, NULL, FALSE);
		gtk_tree_path_free (path);
	}
}

static void
action_delete_cb (GtkAction *action, GnomeDbRawGrid *grid)
{
	GtkTreeIter iter;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GList *sel_rows;
	GdaDataProxy *proxy;
	
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid));
	sel_rows = gtk_tree_selection_get_selected_rows (select, &model);
	proxy = gnome_db_data_store_get_proxy (GNOME_DB_DATA_STORE (model));

	/* rem: get the list of selected rows after each row deletion because the data model might have changed and
	 * row numbers might also have changed */
	while (sel_rows) {
		gtk_tree_model_get_iter (model, &iter, (GtkTreePath *) (sel_rows->data));
		if (!gda_data_proxy_row_is_deleted (proxy, 
						    gnome_db_data_store_get_row_from_iter (GNOME_DB_DATA_STORE (model), 
											   &iter))) {
			gnome_db_data_store_delete (grid->priv->store, &iter);
			g_list_foreach (sel_rows, (GFunc) gtk_tree_path_free, NULL);
			g_list_free (sel_rows);
			sel_rows = gtk_tree_selection_get_selected_rows (select, &model); 
		}
		else
			sel_rows = sel_rows->next;
	}
}

static void
action_undelete_cb (GtkAction *action, GnomeDbRawGrid *grid)
{
	GtkTreeIter iter;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GList *sel_rows, *cur_row;
	
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid));
	sel_rows = gtk_tree_selection_get_selected_rows (select, &model);
	cur_row = sel_rows;
	while (cur_row) {
		gtk_tree_model_get_iter (model, &iter, (GtkTreePath *) (cur_row->data));
		gnome_db_data_store_undelete (grid->priv->store, &iter);
		cur_row = g_list_next (cur_row);
	}
	g_list_foreach (sel_rows, (GFunc) gtk_tree_path_free, NULL);
	g_list_free (sel_rows);
}

static void
action_commit_cb (GtkAction *action, GnomeDbRawGrid *grid)
{
	gint row;
	GError *error = NULL;
	gboolean allok = TRUE;
	gint mod1, mod2;

	mod1 = gda_data_proxy_get_n_modified_rows (grid->priv->proxy);
	row = gda_data_model_iter_get_row (grid->priv->iter);
	if (grid->priv->write_mode >= GNOME_DB_DATA_WIDGET_WRITE_ON_ROW_CHANGE) {
		gint newrow;

		allok = gda_data_proxy_apply_row_changes (grid->priv->proxy, row, &error);
		if (allok) {
			newrow = gda_data_model_iter_get_row (grid->priv->iter);
			if (row != newrow) /* => current row has changed because the proxy had to emit a "row_removed" when
					      actually succedded the commit 
					      => we need to come back to that row
					   */
				gda_data_model_iter_set_at_row (grid->priv->iter, row);
		}
	}
	else
		allok = gda_data_proxy_apply_all_changes (grid->priv->proxy, &error);

	mod2 = gda_data_proxy_get_n_modified_rows (grid->priv->proxy);
	if (!allok) {
		if (mod1 != mod2)
			/* the data model has changed while doing the writing */
			gnome_db_utility_display_error ((GnomeDbDataWidget *) grid, FALSE, error);
		else
			gnome_db_utility_display_error ((GnomeDbDataWidget *) grid, TRUE, error);
		g_error_free (error);
	}
}

static void
action_reset_cb (GtkAction *action, GnomeDbRawGrid *grid)
{
	gda_data_proxy_cancel_all_changes (grid->priv->proxy);
	gda_data_model_send_hint (GDA_DATA_MODEL (grid->priv->proxy), GDA_DATA_MODEL_HINT_REFRESH, NULL);
}

static void
action_first_chunck_cb (GtkAction *action, GnomeDbRawGrid *grid)
{
	gda_data_proxy_set_sample_start (grid->priv->proxy, 0);
}

static void
action_prev_chunck_cb (GtkAction *action, GnomeDbRawGrid *grid)
{
	gint sample_size, sample_start;

	sample_size = gda_data_proxy_get_sample_size (grid->priv->proxy);
	if (sample_size > 0) {
		sample_start = gda_data_proxy_get_sample_start (grid->priv->proxy);
		sample_start -= sample_size;
		gda_data_proxy_set_sample_start (grid->priv->proxy, sample_start);
	}
}

static void
action_next_chunck_cb (GtkAction *action, GnomeDbRawGrid *grid)
{
	gint sample_size, sample_start;

	sample_size = gda_data_proxy_get_sample_size (grid->priv->proxy);
	if (sample_size > 0) {
		sample_start = gda_data_proxy_get_sample_start (grid->priv->proxy);
		sample_start += sample_size;
		gda_data_proxy_set_sample_start (grid->priv->proxy, sample_start);
	}
}

static void
action_last_chunck_cb (GtkAction *action, GnomeDbRawGrid *grid)
{
	gda_data_proxy_set_sample_start (grid->priv->proxy, G_MAXINT);
}

/*
 * Catch any event in the GtkTreeView widget
 */
static gboolean 
tree_view_event_cb (GtkWidget *treeview, GdkEvent *event, GnomeDbRawGrid *grid)
{
	gboolean done = FALSE;

	if (event->type == GDK_KEY_PRESS) {
		GdkEventKey *ekey = (GdkEventKey *) event;
		guint modifiers = gtk_accelerator_get_default_mod_mask ();		

		/* Tab to move one column left or right */
		if (ekey->keyval == GDK_Tab) {
			GtkTreeViewColumn *column;
			GtkTreePath *path;

			/* FIXME: if a column is currently edited, then make sure the editing of that cell is not cancelled */
			gtk_tree_view_get_cursor (GTK_TREE_VIEW (treeview), &path, &column);
			if (column && path) {
				GList *columns = gtk_tree_view_get_columns (GTK_TREE_VIEW (treeview));
				GList *col;
				GtkCellRenderer *renderer;

				/* change column */
				col = g_list_find (columns, column);
				g_return_val_if_fail (col, FALSE);

				if (((ekey->state & modifiers) == GDK_SHIFT_MASK) || ((ekey->state & modifiers) == GDK_CONTROL_MASK))
					col = g_list_previous (col); /* going to previous column */
				else 
					col = g_list_next (col); /* going to next column */

				if (col) {
					renderer = g_object_get_data (G_OBJECT (col->data), "data_renderer");
					gtk_tree_view_set_cursor_on_cell (GTK_TREE_VIEW (treeview), path, 
									  GTK_TREE_VIEW_COLUMN (col->data), 
									  renderer, FALSE);
					gtk_widget_grab_focus (treeview);
					done = TRUE;
				}
				g_list_free (columns);
			}
			if (path)
				gtk_tree_path_free (path);
		}
		
		/* DELETE to delete the selected row */
		if (ekey->keyval == GDK_Delete) {
			GtkTreeIter iter;
			GtkTreeSelection *selection;
			GtkTreeModel *model;
			GList *sel_rows, *cur_row;

			selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid));
			sel_rows = gtk_tree_selection_get_selected_rows (selection, &model);
			cur_row = sel_rows;
			while (cur_row) {
				gtk_tree_model_get_iter (model, &iter, (GtkTreePath *) (cur_row->data));
				if (((ekey->state & modifiers) == GDK_SHIFT_MASK) || 
				    ((ekey->state & modifiers) == GDK_CONTROL_MASK))
					gnome_db_data_store_undelete (grid->priv->store, &iter);
				else
					gnome_db_data_store_delete (grid->priv->store, &iter);
				cur_row = g_list_next (cur_row);
			}
			g_list_foreach (sel_rows, (GFunc) gtk_tree_path_free, NULL);
			g_list_free (sel_rows);
			
			done = TRUE;
		}
	}

	return done;
}

static void menu_select_all_cb (GtkWidget *widget, GnomeDbRawGrid *grid);
static void menu_unselect_all_cb (GtkWidget *widget, GnomeDbRawGrid *grid);
static void menu_show_columns_cb (GtkWidget *widget, GnomeDbRawGrid *grid);
static void menu_save_as_cb (GtkWidget *widget, GnomeDbRawGrid *grid);

static gint
tree_view_popup_button_pressed_cb (GtkWidget *widget, GdkEventButton *event, GnomeDbRawGrid *grid)
{
	GtkWidget *menu;
	GtkTreeView *tree_view;
	GtkTreeSelection *selection;
	GtkSelectionMode sel_mode;

        if (event->button != 3)
                return FALSE;

	tree_view = GTK_TREE_VIEW (grid);
	selection = gtk_tree_view_get_selection (tree_view);
	sel_mode = gtk_tree_selection_get_mode (selection);

        /* create the menu */
        menu = gtk_menu_new ();
	if (sel_mode == GTK_SELECTION_MULTIPLE)
		gtk_menu_append (GTK_MENU (menu),
				 gnome_db_new_menu_item (_("Select _All"), FALSE, 
							 G_CALLBACK (menu_select_all_cb), grid));

	if ((sel_mode == GTK_SELECTION_SINGLE) || (sel_mode == GTK_SELECTION_MULTIPLE))
		gtk_menu_append (GTK_MENU (menu),
				 gnome_db_new_menu_item (_("_Clear Selection"), FALSE,
							 G_CALLBACK (menu_unselect_all_cb), grid));
        gtk_menu_append (GTK_MENU (menu),
			 gnome_db_new_check_menu_item (_("Show Column _Titles"),
						       gtk_tree_view_get_headers_visible (tree_view),
						       G_CALLBACK (menu_show_columns_cb), grid));

	if (sel_mode != GTK_SELECTION_NONE) {
		gtk_menu_append (GTK_MENU (menu), gtk_separator_menu_item_new ());
		gtk_menu_append (GTK_MENU (menu), gnome_db_new_menu_item (GTK_STOCK_SAVE_AS, TRUE,
					       G_CALLBACK (menu_save_as_cb), grid));
	}

	/* allow listeners to add their custom menu items */
        g_signal_emit (G_OBJECT (grid), gnome_db_raw_grid_signals [POPULATE_POPUP], 0, GTK_MENU (menu));
        gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, event->button, event->time);
        gtk_widget_show_all (menu);
	
        return TRUE;
}

static void
menu_select_all_cb (GtkWidget *widget, GnomeDbRawGrid *grid)
{
	GtkTreeSelection *selection;
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid));
	
	gtk_tree_selection_select_all (selection);
}

static void
menu_unselect_all_cb (GtkWidget *widget, GnomeDbRawGrid *grid)
{
	GtkTreeSelection *selection;
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid));
	
	gtk_tree_selection_unselect_all (selection);
}

static void
menu_show_columns_cb (GtkWidget *widget, GnomeDbRawGrid *grid)
{
        GtkCheckMenuItem *item;

        item = (GtkCheckMenuItem *) widget;

        g_return_if_fail (GTK_IS_CHECK_MENU_ITEM (item));
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (grid),
					   gtk_check_menu_item_get_active (item));
}

static void save_as_response_cb (GtkDialog *dialog, guint response_id, GnomeDbRawGrid *grid);

static void
menu_save_as_cb (GtkWidget *widget, GnomeDbRawGrid *grid)
{
	GtkWidget *dialog;
	GtkWidget *label;
	GtkWidget *filename;
	GtkWidget *types;
	GtkWidget *hbox, *table, *check;
	char *str;
	GtkTreeSelection *sel;
	gint selrows;

	/* create dialog box */
	dialog = gtk_dialog_new_with_buttons (_("Saving Data"),
					      GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (grid))), 0,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_SAVE, GTK_RESPONSE_OK,
					      NULL);
	gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);

	str = g_strdup_printf ("<big><b>%s:</b></big>\n%s", _("Saving data to a file"), 
			       _("The data will be exported without any of the modifications which may "
				 "have been made and have not been committed."));
        label = gtk_label_new ("");
        gtk_label_set_markup (GTK_LABEL (label), str);
        gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
        g_free (str);
        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, FALSE, TRUE, 2);

	str = g_strdup_printf ("<b>%s:</b>", _("File name"));
        label = gtk_label_new ("");
        gtk_label_set_markup (GTK_LABEL (label), str);
        gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
        g_free (str);
        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, FALSE, TRUE, 2);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, FALSE, FALSE, 5);
        gtk_widget_show (hbox);
        label = gtk_label_new ("    ");
        gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
        gtk_widget_show (label);

	filename = gtk_file_chooser_widget_new (GTK_FILE_CHOOSER_ACTION_SAVE);
	g_object_set_data (G_OBJECT (dialog), "filename", filename);
	gtk_box_pack_start (GTK_BOX (hbox), filename, TRUE, TRUE, 0);

	str = g_strdup_printf ("<b>%s:</b>", _("Details")); 
        label = gtk_label_new ("");
        gtk_label_set_markup (GTK_LABEL (label), str);
        gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
        g_free (str);
        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, FALSE, TRUE, 2);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, FALSE, FALSE, 5);
        gtk_widget_show (hbox);
        label = gtk_label_new ("    ");
        gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
        gtk_widget_show (label);
	
	table = gtk_table_new (2, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), 5);
	gtk_table_set_col_spacings (GTK_TABLE (table), 5);
	gtk_box_pack_start (GTK_BOX (hbox), table, TRUE, TRUE, 0);
	gtk_widget_show (table);

	label = gtk_label_new (_("Limit to selection?"));
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, 0, 0, 0);
	gtk_widget_show (label);

	sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid));
	selrows = gtk_tree_selection_count_selected_rows (sel);
	if (selrows <= 0)
		gtk_widget_set_sensitive (label, FALSE);

	check = gtk_check_button_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), check, 1, 2, 0, 1);
	gtk_widget_show (check);
	if (selrows <= 0)
		gtk_widget_set_sensitive (check, FALSE);
	else
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), TRUE);
	g_object_set_data (G_OBJECT (dialog), "sel_only", check);

	label = gtk_label_new (_("File type:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, 0, 0, 0);
	gtk_widget_show (label);

	types = gtk_combo_box_new_text ();
	gtk_table_attach_defaults (GTK_TABLE (table), types, 1, 2, 1, 2);
	gtk_widget_show (label);
	g_object_set_data (G_OBJECT (dialog), "types", types);

	gtk_combo_box_append_text (GTK_COMBO_BOX (types), _("Tab-delimited"));
	gtk_combo_box_append_text (GTK_COMBO_BOX (types), _("Comma-delimited"));
	gtk_combo_box_append_text (GTK_COMBO_BOX (types), _("XML"));
	gtk_combo_box_set_active (GTK_COMBO_BOX (types), grid->priv->export_type);
	
	/* run the dialog */
	g_signal_connect (G_OBJECT (dialog), "response", G_CALLBACK (save_as_response_cb), grid);
	gtk_widget_show_all (dialog);
}

static gboolean confirm_file_overwrite (GtkWindow *parent, const gchar *path);

static void
save_as_response_cb (GtkDialog *dialog, guint response_id, GnomeDbRawGrid *grid)
{
	GtkWidget *types;
	gint export_type;
	GtkWidget *filename;
	gboolean selection_only = FALSE;
	
	if (response_id == GTK_RESPONSE_OK) {
		gchar *body;
		gchar *path;
		GList *columns, *list;
		gint *cols, nb_cols;
		gint *rows = NULL, nb_rows = 0;
		GdaParameter *param;
		GdaParameterList *paramlist;

		types = g_object_get_data (G_OBJECT (dialog), "types");
		filename = g_object_get_data (G_OBJECT (dialog), "filename");
		selection_only = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON 
							       (g_object_get_data (G_OBJECT (dialog), "sel_only")));

		/* output columns computation */
		columns = gtk_tree_view_get_columns (GTK_TREE_VIEW (grid));
		cols = g_new (gint, gda_data_model_get_n_columns (GDA_DATA_MODEL (grid->priv->data_model)));
		nb_cols = 0;
		list = columns;
		while (list) {
			if (gtk_tree_view_column_get_visible (GTK_TREE_VIEW_COLUMN (list->data))) {
				GdaParameterListGroup *group;
				GSList *params;

				group = g_object_get_data (G_OBJECT (list->data), "group");
				params = group->nodes;
				while (params) {
					cols [nb_cols] = gda_data_model_iter_get_column_for_param (grid->priv->iter, 
									GDA_PARAMETER_LIST_NODE (params->data)->param);
					nb_cols ++;
					params = g_slist_next (params);
				}
			}
			list = g_list_next (list);
		}
		g_list_free (columns);

		/* output rows computation */
		if (selection_only) {
			GtkTreeSelection *selection;
			GList *sel_rows, *list;
			gint pos;

			selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid));
			sel_rows = gtk_tree_selection_get_selected_rows (selection, NULL);

			nb_rows = g_list_length (sel_rows);
			rows = g_new0 (gint, nb_rows);

			for (pos = 0, list = sel_rows; list; list = list->next, pos++) {
				gint *ind = gtk_tree_path_get_indices ((GtkTreePath*) list->data);
				rows [pos] = *ind;
				gtk_tree_path_free ((GtkTreePath*) list->data);
			}
			g_list_free (sel_rows);
		}
		
		/* Actual ouput computations */
		export_type = gtk_combo_box_get_active (GTK_COMBO_BOX (types));
		grid->priv->export_type = export_type;
		switch (export_type) {
		case 0:
			param = gda_parameter_new_string ("SEPARATOR", "\t");
			paramlist = gda_parameter_list_new (NULL);
			gda_parameter_list_add_param (paramlist, param);
			g_object_unref (param);
			body = gda_data_model_export_to_string (GDA_DATA_MODEL (grid->priv->data_model), 
								GDA_DATA_MODEL_IO_TEXT_SEPARATED, 
								cols, nb_cols, rows, nb_rows, paramlist);
			g_object_unref (paramlist);
			break;
		case 1:
			param = gda_parameter_new_string ("SEPARATOR", ",");
			paramlist = gda_parameter_list_new (NULL);
			gda_parameter_list_add_param (paramlist, param);
			g_object_unref (param);
			body = gda_data_model_export_to_string (GDA_DATA_MODEL (grid->priv->data_model), 
								GDA_DATA_MODEL_IO_TEXT_SEPARATED, 
								cols, nb_cols, rows, nb_rows, paramlist);
			g_object_unref (paramlist);
			break;
		case 2:
			param = NULL;
			body = (gchar *) gda_object_get_name (GDA_OBJECT (grid->priv->data_model));
			if (body)
				param = gda_parameter_new_string ("NAME", body);
			paramlist = gda_parameter_list_new (NULL);
			gda_parameter_list_add_param (paramlist, param);
			g_object_unref (param);
			body = gda_data_model_export_to_string (GDA_DATA_MODEL (grid->priv->data_model), 
								GDA_DATA_MODEL_IO_DATA_ARRAY_XML, 
								cols, nb_cols, rows, nb_rows, paramlist);
			g_object_unref (paramlist);
			break;
		default:
			g_assert_not_reached ();
			break;
		}
		g_free (cols);
		if (rows)
			g_free (rows);

		/* saving */
		if (body) {
			path = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (filename));
			if (path) {
				if (g_file_test (path, G_FILE_TEST_EXISTS)) {
					if (! confirm_file_overwrite (GTK_WINDOW (dialog), path)) {
						g_free (body);
						g_free (path);
						return;
					}
				}

				if (! gda_file_save (path, body, strlen (body))) {
					gnome_db_show_error (NULL, _("Could not save file %s"), path);
					g_free (body);
					g_free (path);
					return;
				}
				g_free (path);
			} 
			else {
				gnome_db_show_error (NULL, _("You must specify a file name"));
				g_free (body);
				return;
			}
			g_free (body);
		} else
			gnome_db_show_error (NULL,_("Got empty file while converting the data"));
	}

	gtk_widget_destroy (GTK_WIDGET (dialog));
}

static gboolean
confirm_file_overwrite (GtkWindow *parent, const gchar *path)
{
	GtkWidget *dialog, *button;
	gboolean yes;
	gchar *msg;

	msg = g_strdup_printf (_("File '%s' already exists.\n"
				 "Do you want to overwrite it?"), path);

	/* build the dialog */
	dialog = gnome_db_new_alert (
		NULL,
		GTK_MESSAGE_QUESTION,
		msg,
		_("If you choose yes, the contents will be lost."));
	button = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_dialog_add_action_widget (GTK_DIALOG (dialog),
				      button,
				      GTK_RESPONSE_NO);
	gtk_dialog_add_action_widget (GTK_DIALOG (dialog),
				      gtk_button_new_from_stock (GTK_STOCK_YES),
				      GTK_RESPONSE_YES);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog),
					 GTK_RESPONSE_NO);

	/* run the dialog */
	gtk_widget_show_all (dialog);
	yes = gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_YES;

	/* free memory */
	gtk_widget_destroy (dialog);
	g_free (msg);
	return yes;
}


static void
tree_view_row_activated_cb (GtkTreeView *tree_view, GtkTreePath *path, GtkTreeViewColumn *column, GnomeDbRawGrid *grid)
{
	gint *indices;

	indices = gtk_tree_path_get_indices (path);
#ifdef debug_signal
	g_print (">> 'DOUBLE_CLICKED' from %s %p\n", G_OBJECT_TYPE_NAME (grid), grid);
#endif
	g_signal_emit (G_OBJECT (grid), gnome_db_raw_grid_signals[DOUBLE_CLICKED], 0, *indices);
#ifdef debug_signal
	g_print ("<< 'DOUBLE_CLICKED' from %s %p\n", G_OBJECT_TYPE_NAME (grid), grid);
#endif	
}

/*
 * Synchronize the values of the parameters in priv->iter with the currently selected row
 */
static void
tree_view_selection_changed_cb (GtkTreeSelection *selection, GnomeDbRawGrid *grid)
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	gboolean row_selected = FALSE;
	gint has_selection;
	
	/* block the GdaDataModelIter' "changed" signal */
	g_signal_handlers_block_by_func (grid->priv->iter,
					 G_CALLBACK (iter_row_changed_cb), grid);

	if (gtk_tree_selection_get_mode (selection) == GTK_SELECTION_MULTIPLE) {
		has_selection = gtk_tree_selection_count_selected_rows (selection);
		if (has_selection == 1) {
			GList *sel_rows;

			sel_rows = gtk_tree_selection_get_selected_rows (selection, &model);
			has_selection = gtk_tree_model_get_iter (model, &iter, (GtkTreePath *) (sel_rows->data)) ? 1 : 0;
			g_list_foreach (sel_rows, (GFunc) gtk_tree_path_free, NULL);
			g_list_free (sel_rows);
		}
	}
	else
		has_selection = gtk_tree_selection_get_selected (selection, &model, &iter) ? 1 : 0;

	if (has_selection == 1) {
		if (!gda_data_model_iter_set_at_row (grid->priv->iter, 
						     gnome_db_data_store_get_row_from_iter (grid->priv->store, 
											    &iter))) {
			/* selection changing is refused, return to the current selected row */
			GtkTreePath *path;
			path = gtk_tree_path_new_from_indices (gda_data_model_iter_get_row (grid->priv->iter), -1);
			gtk_tree_selection_unselect_all (selection);
			gtk_tree_selection_select_path (selection, path);
			gtk_tree_path_free (path);
		}
		row_selected = TRUE;
	}
	else {
		/* render all the parameters invalid, and make the iter point to row -1 */
		gda_data_model_iter_invalidate_contents (grid->priv->iter);
		gda_data_model_iter_set_at_row (grid->priv->iter, -1);
	}

#ifdef debug_signal
	g_print (">> 'SELECTION_CHANGED' from %s %p\n", G_OBJECT_TYPE_NAME (grid), grid);
#endif
	g_signal_emit (G_OBJECT (grid), gnome_db_raw_grid_signals[SELECTION_CHANGED], 0, row_selected);
#ifdef debug_signal
	g_print ("<< 'SELECTION_CHANGED' from %s %p\n", G_OBJECT_TYPE_NAME (grid), grid);
#endif

	/* unblock the GdaDataModelIter' "changed" signal */
	g_signal_handlers_unblock_by_func (grid->priv->iter,
					   G_CALLBACK (iter_row_changed_cb), grid);
}

static ColumnData *
get_column_data (GnomeDbRawGrid *grid, GdaParameterListGroup *group)
{
	ColumnData *retval = NULL;
	GSList *list = grid->priv->columns_data;
	while (list && !retval) {
		if (COLUMN_DATA (list->data)->group == group)
			retval = COLUMN_DATA (list->data);

		list = g_slist_next (list);
	}

	return retval;
}

/**
 * gnome_db_raw_grid_set_sample_size
 * @grid:
 * @sample_size:
 *
 */
void
gnome_db_raw_grid_set_sample_size (GnomeDbRawGrid *grid, gint sample_size)
{
	g_return_if_fail (grid && GNOME_DB_IS_RAW_GRID (grid));
	g_return_if_fail (grid->priv);

	gda_data_proxy_set_sample_size (grid->priv->proxy, sample_size);
}

/**
 * gnome_db_raw_grid_set_sample_start
 * @grid:
 * @sample_start:
 *
 */
void
gnome_db_raw_grid_set_sample_start (GnomeDbRawGrid *grid, gint sample_start)
{
	g_return_if_fail (grid && GNOME_DB_IS_RAW_GRID (grid));
	g_return_if_fail (grid->priv);
	
	gda_data_proxy_set_sample_start (grid->priv->proxy, sample_start);
}


/*
 * GnomeDbDataWidget interface
 */

static GdaDataProxy *
gnome_db_raw_grid_get_proxy (GnomeDbDataWidget *iface)
{
	GnomeDbRawGrid *grid;
	g_return_val_if_fail (iface && GNOME_DB_IS_RAW_GRID (iface), NULL);
	grid = GNOME_DB_RAW_GRID (iface);
	g_return_val_if_fail (grid->priv, NULL);

	return grid->priv->proxy;
}

static void
gnome_db_raw_grid_set_column_editable (GnomeDbDataWidget *iface, gint column, gboolean editable)
{
	GnomeDbRawGrid *grid;
	GdaParameter *param;
	ColumnData *column_data;
	GdaParameterListGroup *group;

	g_return_if_fail (iface && GNOME_DB_IS_RAW_GRID (iface));
	grid = GNOME_DB_RAW_GRID (iface);
	g_return_if_fail (grid->priv);
	
	if (grid->priv->data_model) {
		editable = editable && !gda_data_proxy_is_read_only (grid->priv->proxy);

		param = gda_data_model_iter_get_param_for_column (grid->priv->iter, column);
		g_return_if_fail (param);

		group = gda_parameter_list_find_group_for_param (GDA_PARAMETER_LIST (grid->priv->iter), param);
		g_return_if_fail (group);

		column_data = get_column_data (grid, group);
		g_return_if_fail (column_data);
		
		if (editable && !gda_data_proxy_is_read_only (grid->priv->proxy))
			column_data->data_locked = FALSE;
		else
			column_data->data_locked = TRUE;
	}
}

static void
gnome_db_raw_grid_show_column_actions (GnomeDbDataWidget *iface, gint column, gboolean show_actions)
{
	GnomeDbRawGrid *grid;

	g_return_if_fail (iface && GNOME_DB_IS_RAW_GRID (iface));
	grid = GNOME_DB_RAW_GRID (iface);
	g_return_if_fail (grid->priv);

	if (column >= 0) {
		GdaParameter *param;
		GdaParameterListGroup *group;
		ColumnData *cdata;

		/* setting applies only to the @column column */
		param = gda_data_model_iter_get_param_for_column (grid->priv->iter, column);
		g_return_if_fail (param);
		
		group = gda_parameter_list_find_group_for_param (GDA_PARAMETER_LIST (grid->priv->iter), param);
		g_return_if_fail (group);

		cdata = get_column_data (grid, group);
		g_return_if_fail (cdata);
		
		if (show_actions != cdata->info_shown) {
			cdata->info_shown = show_actions;
			g_object_set (G_OBJECT (cdata->info_cell), "visible", cdata->info_shown, NULL);
		}
	}
	else {
		/* setting applies to all columns */
		GSList *list;
		for (list = grid->priv->columns_data; list; list = list->next) {
			ColumnData *cdata = (ColumnData*)(list->data);
			if (show_actions != cdata->info_shown) {
				cdata->info_shown = show_actions;
				g_object_set (G_OBJECT (cdata->info_cell), "visible", cdata->info_shown, NULL);
			}
		}
		grid->priv->default_show_info_cell = show_actions;
	}
}

static void
gnome_db_raw_grid_col_set_show (GnomeDbDataWidget *iface, gint column, gboolean shown)
{
	GnomeDbRawGrid *grid;
	gint pos = -1;
	GtkTreeViewColumn *viewcol;
	GdaParameterListGroup *group;
	GdaParameter *param;

	g_return_if_fail (iface && GNOME_DB_IS_RAW_GRID (iface));
	grid = GNOME_DB_RAW_GRID (iface);
	g_return_if_fail (grid->priv);

	param = gda_data_model_iter_get_param_for_column (grid->priv->iter, column);
	g_return_if_fail (param);

	group = gda_parameter_list_find_group_for_param ((GdaParameterList *)grid->priv->iter, param);	
	pos = g_slist_index (((GdaParameterList *)grid->priv->iter)->groups_list, group);	
	g_assert (pos >= 0);

	viewcol = gtk_tree_view_get_column (GTK_TREE_VIEW (grid), pos);

	/* Sets the column's visibility */
	gtk_tree_view_column_set_visible (viewcol, shown);
	g_object_set_data (G_OBJECT (param), "_gnome_db_explicit_show", GINT_TO_POINTER (shown ? 2 : 1));
}


static GtkActionGroup *
gnome_db_raw_grid_get_actions_group (GnomeDbDataWidget *iface)
{
	GnomeDbRawGrid *grid;
	
	g_return_val_if_fail (iface && GNOME_DB_IS_RAW_GRID (iface), NULL);
	grid = GNOME_DB_RAW_GRID (iface);
	g_return_val_if_fail (grid->priv, NULL);

	return grid->priv->actions_group;
}

static GdaDataModelIter *
gnome_db_raw_grid_widget_get_data_set (GnomeDbDataWidget *iface)
{
	GnomeDbRawGrid *grid;
	
	g_return_val_if_fail (iface && GNOME_DB_IS_RAW_GRID (iface), NULL);
	grid = GNOME_DB_RAW_GRID (iface);
	g_return_val_if_fail (grid->priv, NULL);

	return grid->priv->iter;
}


static GdaDataModel *
gnome_db_raw_grid_widget_get_gda_model (GnomeDbDataWidget *iface)
{
	GnomeDbRawGrid *grid;
	g_return_val_if_fail (iface && GNOME_DB_IS_RAW_GRID (iface), NULL);
	grid = GNOME_DB_RAW_GRID (iface);
	g_return_val_if_fail (grid->priv, NULL);

	/* warning: here we are returning the *original* GdaDataModel */

	return GDA_DATA_MODEL (grid->priv->data_model);
}

static void
paramlist_public_data_changed_cb (GdaParameterList *paramlist, GnomeDbRawGrid *grid)
{
	GList *columns, *list;

	/* info cells */
	if (grid->priv->columns_data) {
		GSList *list = grid->priv->columns_data;
		while (list) {
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (grid->priv->columns_data);
		grid->priv->columns_data = NULL;
	}

	columns = gtk_tree_view_get_columns (GTK_TREE_VIEW (grid));
	if (columns) {
		list = columns;
		while (list) {
			gtk_tree_view_remove_column (GTK_TREE_VIEW (grid),
						     (GtkTreeViewColumn*) (list->data));
			list = g_list_next (list);
		}
		g_list_free (columns);
	}

	init_tree_view (grid);
}

static void
paramlist_param_plugin_changed_cb (GdaParameterList *paramlist, GdaParameter *param, GnomeDbRawGrid *grid)
{
	/* TODO: be more specific and change only the cell renderer corresponding to @param */
	paramlist_public_data_changed_cb (paramlist, grid);
}

static gboolean
iter_row_to_change_cb (GdaDataModelIter *iter, gint row, GnomeDbRawGrid *grid)
{
	gboolean allow_change = TRUE;
	if (row >= 0) {
		if (grid->priv->write_mode >= GNOME_DB_DATA_WIDGET_WRITE_ON_ROW_CHANGE) {
			/* write back the current row */
			if (gda_data_proxy_row_has_changed (grid->priv->proxy, row)) {
				GError *error = NULL;
				if (!gda_data_proxy_apply_row_changes (grid->priv->proxy, row, &error)) {
					allow_change = gnome_db_utility_display_error_with_keep_or_discard_choice ((GnomeDbDataWidget *) grid, 
													  error);
					if (allow_change)
						gda_data_proxy_cancel_row_changes (grid->priv->proxy, row, -1);
					g_error_free (error);
				}
			}
		}
	}
	
	return allow_change;
}

static void
iter_row_changed_cb (GdaDataModelIter *iter, gint row, GnomeDbRawGrid *grid)
{
	GtkTreeSelection *selection;
	GtkTreePath *path;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid));
	
	if (row >= 0) {
		GtkSelectionMode mode;
		GtkTreeIter treeiter;

		mode = gtk_tree_selection_get_mode (selection);
		if (mode != GTK_SELECTION_SINGLE)
			gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
		path = gtk_tree_path_new_from_indices (row, -1);
		if (gtk_tree_model_get_iter (GTK_TREE_MODEL (grid->priv->store), &treeiter, path)) {
			gtk_tree_selection_select_path (selection, path);
			gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (grid), path, NULL,
						      FALSE, 0., 0.);
		}
		gtk_tree_path_free (path);
		if (mode != GTK_SELECTION_SINGLE)
			gtk_tree_selection_set_mode (selection, mode);
	}
	else
		gtk_tree_selection_unselect_all (selection);
}

static void
proxy_sample_changed_cb (GdaDataProxy *proxy, gint sample_start, gint sample_end, GnomeDbRawGrid *grid)
{
	/* bring back the vertical scrollbar to the top */
	gtk_adjustment_set_value (gtk_tree_view_get_vadjustment (GTK_TREE_VIEW (grid)), 0.);
}

static void
proxy_row_updated_cb (GdaDataProxy *proxy, gint proxy_row, GnomeDbRawGrid *grid)
{
	if (grid->priv->write_mode == GNOME_DB_DATA_WIDGET_WRITE_ON_VALUE_ACTIVATED) {
		gint row;
		
		/* REM: this may lead to problems in the proxy because it is not re-entrant, solution: add a very short timeout */
		row = gda_data_model_iter_get_row (grid->priv->iter);
		if ((row >= 0) && (row == proxy_row)) {
			/* write back the current row */
			if (gda_data_proxy_row_has_changed (grid->priv->proxy, row)) {
				GError *error = NULL;
				if (!gda_data_proxy_apply_row_changes (grid->priv->proxy, row, &error)) {
					gboolean discard;
					discard = gnome_db_utility_display_error_with_keep_or_discard_choice ((GnomeDbDataWidget *) grid, 
												     error);
					if (discard)
						gda_data_proxy_cancel_row_changes (grid->priv->proxy, row, -1);
					g_error_free (error);
				}
			}
		}
	}
}

static void
proxy_reset_cb (GdaDataProxy *proxy, GnomeDbRawGrid *grid)
{
	g_object_ref (G_OBJECT (proxy));
	g_object_set (G_OBJECT (grid), "model", proxy, NULL);
	g_object_unref (G_OBJECT (proxy));
}

static void
proxy_destroyed_cb (GdaDataProxy *proxy, GnomeDbRawGrid *grid)
{
	g_assert (proxy == grid->priv->proxy);
	g_signal_handlers_disconnect_by_func (proxy, G_CALLBACK (proxy_destroyed_cb), grid);
	g_signal_handlers_disconnect_by_func (proxy, G_CALLBACK (proxy_sample_changed_cb), grid);
	g_signal_handlers_disconnect_by_func (proxy, G_CALLBACK (proxy_row_updated_cb), grid);
	g_signal_handlers_disconnect_by_func (proxy, G_CALLBACK (proxy_reset_cb), grid);
	grid->priv->proxy = NULL;
}

static void
gnome_db_raw_grid_widget_set_gda_model (GnomeDbDataWidget *iface, GdaDataModel *model)
{
	GnomeDbRawGrid *grid;

	g_return_if_fail (iface && GNOME_DB_IS_RAW_GRID (iface));
	grid = GNOME_DB_RAW_GRID (iface);
	g_return_if_fail (grid->priv);
	
	g_object_set (grid, "model", model, NULL);
}

static void
gnome_db_raw_grid_clean (GnomeDbRawGrid *grid)
{
	GList *columns, *list;

	/* columns */
	columns = gtk_tree_view_get_columns (GTK_TREE_VIEW (grid));
	list = columns;
	while (list) {
		gtk_tree_view_remove_column ((GtkTreeView *) grid,
					     (GtkTreeViewColumn *) list->data);
		list = g_list_next (list);
	}
	g_list_free (columns);

	/* info cells */
	if (grid->priv->columns_data) {
		GSList *list = grid->priv->columns_data;
		while (list) {
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (grid->priv->columns_data);
		grid->priv->columns_data = NULL;
	}
	
	/* private data set */
	if (grid->priv->iter) {
		g_signal_handlers_disconnect_by_func (grid->priv->iter,
						      G_CALLBACK (paramlist_public_data_changed_cb), grid);
		g_signal_handlers_disconnect_by_func (grid->priv->iter,
						      G_CALLBACK (paramlist_param_plugin_changed_cb), grid);
		g_signal_handlers_disconnect_by_func (grid->priv->iter,
						      G_CALLBACK (iter_row_changed_cb), grid);
		g_signal_handlers_disconnect_by_func (grid->priv->iter,
						      G_CALLBACK (iter_row_to_change_cb), grid);
		g_object_unref (grid->priv->iter);
		grid->priv->iter = NULL;
	}
	
	/* proxy */
	if (grid->priv->proxy)
		proxy_destroyed_cb (grid->priv->proxy, grid);

	/* store */
	if (grid->priv->store) {
		g_object_unref (grid->priv->store);
		grid->priv->store = NULL;
	}
}

static gboolean
gnome_db_raw_grid_widget_set_write_mode (GnomeDbDataWidget *iface, GnomeDbDataWidgetWriteMode mode)
{
	GnomeDbRawGrid *grid;
	
	g_return_val_if_fail (GNOME_DB_IS_RAW_GRID (iface), FALSE);
	grid = GNOME_DB_RAW_GRID (iface);
	g_return_val_if_fail (grid->priv, FALSE);

	grid->priv->write_mode = mode;
	if (mode == GNOME_DB_DATA_WIDGET_WRITE_ON_VALUE_CHANGE) {
		grid->priv->write_mode = GNOME_DB_DATA_WIDGET_WRITE_ON_VALUE_ACTIVATED;
		return FALSE;
	}

	if (mode == GNOME_DB_DATA_WIDGET_WRITE_ON_ROW_CHANGE) {
		GtkTreeSelection *selection;
		selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid));
		gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
	}
	return TRUE;
}

static GnomeDbDataWidgetWriteMode
gnome_db_raw_grid_widget_get_write_mode (GnomeDbDataWidget *iface)
{
	GnomeDbRawGrid *grid;
	
	g_return_val_if_fail (GNOME_DB_IS_RAW_GRID (iface), GNOME_DB_DATA_WIDGET_WRITE_ON_DEMAND);
	grid = GNOME_DB_RAW_GRID (iface);
	g_return_val_if_fail (grid->priv, GNOME_DB_DATA_WIDGET_WRITE_ON_DEMAND);

	return grid->priv->write_mode;
}
