/* gnome-db-goo-fkconstraint.c
 *
 * Copyright (C) 2004 - 2007 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gtk/gtk.h>
#include <math.h>
#include <libgda/libgda.h>
#include <glib/gi18n-lib.h>
#include "gnome-db-goo.h"
#include "gnome-db-goo-fkconstraint.h"
#include "gnome-db-goo-entity.h"
#include "gnome-db-goo-text.h"
#include "gnome-db-goo-utility.h"

static void gnome_db_goo_fkconstraint_class_init (GnomeDbGooFkconstraintClass * class);
static void gnome_db_goo_fkconstraint_init       (GnomeDbGooFkconstraint * cc);
static void gnome_db_goo_fkconstraint_dispose    (GObject   * object);
static void gnome_db_goo_fkconstraint_finalize   (GObject   * object);

static void gnome_db_goo_fkconstraint_set_property    (GObject *object,
						    guint param_id,
						    const GValue *value,
						    GParamSpec *pspec);
static void gnome_db_goo_fkconstraint_get_property    (GObject *object,
						    guint param_id,
						    GValue *value,
						    GParamSpec *pspec);

static void gnome_db_goo_fkconstraint_get_edge_nodes (GnomeDbGooItem *citem, 
							 GnomeDbGooItem **from, GnomeDbGooItem **to);

static void clean_items (GnomeDbGooFkconstraint *cc);
static void create_items (GnomeDbGooFkconstraint *cc);
static void update_items (GnomeDbGooFkconstraint *cc);

enum
{
	PROP_0,
	PROP_FK_CONSTRAINT
};

struct _GnomeDbGooFkconstraintPrivate
{
	GSList             *constraints;
	GnomeDbGooEntity   *fk_entity_item;
	GnomeDbGooEntity   *ref_pk_entity_item;
	GSList             *shapes; /* list of GnomeDbGooCanvasShape structures */
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

GType
gnome_db_goo_fkconstraint_get_type (void)
{
	static GType type = 0;

        if (G_UNLIKELY (type == 0)) {
		static const GTypeInfo info = {
			sizeof (GnomeDbGooFkconstraintClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_goo_fkconstraint_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbGooFkconstraint),
			0,
			(GInstanceInitFunc) gnome_db_goo_fkconstraint_init
		};		

		type = g_type_register_static (GNOME_DB_TYPE_GOO_ITEM, "GnomeDbGooFkconstraint", &info, 0);
	}

	return type;
}	

static void
gnome_db_goo_fkconstraint_class_init (GnomeDbGooFkconstraintClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	GNOME_DB_GOO_ITEM_CLASS (class)->get_edge_nodes = gnome_db_goo_fkconstraint_get_edge_nodes;

	object_class->dispose = gnome_db_goo_fkconstraint_dispose;
	object_class->finalize = gnome_db_goo_fkconstraint_finalize;

	/* Properties */
	object_class->set_property = gnome_db_goo_fkconstraint_set_property;
	object_class->get_property = gnome_db_goo_fkconstraint_get_property;
	g_object_class_install_property (object_class, PROP_FK_CONSTRAINT,
					 g_param_spec_pointer ("fk_constraint", "Latest FK constraint to add", 
							       NULL, G_PARAM_WRITABLE));
       
}

static void
gnome_db_goo_fkconstraint_init (GnomeDbGooFkconstraint *cc)
{
	cc->priv = g_new0 (GnomeDbGooFkconstraintPrivate, 1);
	cc->priv->constraints = NULL;
	cc->priv->fk_entity_item = NULL;
	cc->priv->ref_pk_entity_item = NULL;
	cc->priv->shapes = NULL;
}


static void entity_destroy_cb (GnomeDbGooEntity *entity, GnomeDbGooFkconstraint *cc);
static void constraint_destroyed_cb (GdaDictConstraint *fkcons, GnomeDbGooFkconstraint *cc);

static void
gnome_db_goo_fkconstraint_dispose (GObject *object)
{
	GnomeDbGooFkconstraint *cc;
	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_GOO_FKCONSTRAINT (object));

	cc = GNOME_DB_GOO_FKCONSTRAINT (object);

	clean_items (cc);
	if (cc->priv->constraints) {
		GSList *list = cc->priv->constraints;
		while (list) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (list->data),
							      G_CALLBACK (constraint_destroyed_cb), cc);
			list = g_slist_next (list);
		}
		g_slist_free (cc->priv->constraints);
		cc->priv->constraints = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}


static void
gnome_db_goo_fkconstraint_finalize (GObject *object)
{
	GnomeDbGooFkconstraint *cc;
	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_GOO_FKCONSTRAINT (object));

	cc = GNOME_DB_GOO_FKCONSTRAINT (object);
	if (cc->priv) {
		g_free (cc->priv);
		cc->priv = NULL;
	}

	/* for the parent class */
	parent_class->finalize (object);
}

static void 
gnome_db_goo_fkconstraint_set_property    (GObject *object,
					guint param_id,
					const GValue *value,
					GParamSpec *pspec)
{
	GnomeDbGooFkconstraint *cc;

	cc = GNOME_DB_GOO_FKCONSTRAINT (object);

	switch (param_id) {
	case PROP_FK_CONSTRAINT:
		gnome_db_goo_fkconstraint_add_constraint (cc, g_value_get_pointer (value));
		break;
	}
}

static void 
gnome_db_goo_fkconstraint_get_property    (GObject *object,
					guint param_id,
					GValue *value,
					GParamSpec *pspec)
{
	GnomeDbGooFkconstraint *cc;

	cc = GNOME_DB_GOO_FKCONSTRAINT (object);

	switch (param_id) {
	default:
		g_warning ("No such property!");
		break;
	}
}

static void
gnome_db_goo_fkconstraint_get_edge_nodes (GnomeDbGooItem *citem, 
					  GnomeDbGooItem **from, GnomeDbGooItem **to)
{
	GnomeDbGooFkconstraint *cc;

	cc = GNOME_DB_GOO_FKCONSTRAINT (citem);

	if (from)
		*from = (GnomeDbGooItem*) cc->priv->fk_entity_item;
	if (to)
		*to = (GnomeDbGooItem*) cc->priv->ref_pk_entity_item;
}

static void
constraint_destroyed_cb (GdaDictConstraint *fkcons, GnomeDbGooFkconstraint *cc)
{
	g_assert (g_slist_find (cc->priv->constraints, fkcons));
	cc->priv->constraints = g_slist_remove (cc->priv->constraints, fkcons);
	g_signal_handlers_disconnect_by_func (G_OBJECT (fkcons),
					      G_CALLBACK (constraint_destroyed_cb), cc);
	
	/* destroy itself if there are no more constraint left in the end */
	if (!cc->priv->constraints)
		goo_canvas_item_remove (GOO_CANVAS_ITEM (cc));
	else {
		clean_items (cc);
		create_items (cc);
	}
}

static void
entity_destroy_cb (GnomeDbGooEntity *entity, GnomeDbGooFkconstraint *cc)
{
	goo_canvas_item_remove (GOO_CANVAS_ITEM (cc));
}


static gboolean single_item_enter_notify_event_cb (GooCanvasItem *ci, GooCanvasItem *target_item,
						   GdkEventCrossing *event, GnomeDbGooFkconstraint *cc);
static gboolean single_item_leave_notify_event_cb (GooCanvasItem *ci, GooCanvasItem *target_item,
						   GdkEventCrossing *event, GnomeDbGooFkconstraint *cc);
static gboolean single_item_button_press_event_cb (GooCanvasItem *ci, GooCanvasItem *target_item,
						   GdkEventButton *event, GnomeDbGooFkconstraint *cc);
static void entity_item_moved_cb (GooCanvasItem *entity, GnomeDbGooFkconstraint *cc);

/* 
 * destroy any existing GooCanvasItem objects 
 */
static void 
clean_items (GnomeDbGooFkconstraint *cc)
{
	if (cc->priv->fk_entity_item) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->fk_entity_item),
						      G_CALLBACK (entity_item_moved_cb), cc);
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->fk_entity_item),
						      G_CALLBACK (entity_destroy_cb), cc);
		cc->priv->fk_entity_item = NULL;
	}

	if (cc->priv->ref_pk_entity_item) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->ref_pk_entity_item),
						      G_CALLBACK (entity_item_moved_cb), cc);
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->ref_pk_entity_item),
						      G_CALLBACK (entity_destroy_cb), cc);
		cc->priv->ref_pk_entity_item = NULL;
	}
	
	/* remove all the GooCanvasItem objects */
	gnome_db_goo_canvas_shapes_remove_all (cc->priv->shapes);
	cc->priv->shapes = NULL;
}

/*
 * create new GooCanvasItem objects
 *
 * It is assumed that the list of FK constraints to render is coherent (that all the constraints are
 * FK constraints, for the same table, referencing the same FK table).
 */
static void 
create_items (GnomeDbGooFkconstraint *cc)
{
	GooCanvasItem *item;
	GSList *fklist = cc->priv->constraints, *list, *canvas_shapes, *list2;
	GdaDictConstraint *fkcons;
	GdaDictTable *table;
	GnomeDbGooItem *entity_item;
	GnomeDbGoo *canvas = gnome_db_goo_item_get_canvas (GNOME_DB_GOO_ITEM (cc));
	gboolean user_constraint;

	if (!fklist)
		return;

	/* Analyse FK constraint */
	fkcons = GDA_DICT_CONSTRAINT (fklist->data);
	item = GOO_CANVAS_ITEM (cc);
	table = gda_dict_constraint_get_table (fkcons);
	entity_item = gnome_db_goo_get_item_for_object (canvas, GDA_OBJECT (table));
	g_return_if_fail (entity_item);
	cc->priv->fk_entity_item = GNOME_DB_GOO_ENTITY (entity_item);
	g_signal_connect (G_OBJECT (entity_item), "moving",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "moved",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "shifted",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "destroy",
			  G_CALLBACK (entity_destroy_cb), cc);

	table = gda_dict_constraint_fkey_get_ref_table (fkcons);
	entity_item = gnome_db_goo_get_item_for_object (canvas, GDA_OBJECT (table));
	g_return_if_fail (entity_item);
	cc->priv->ref_pk_entity_item = GNOME_DB_GOO_ENTITY (entity_item);
	g_signal_connect (G_OBJECT (entity_item), "moving",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "moved",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "shifted",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "destroy",
			  G_CALLBACK (entity_destroy_cb), cc);

	/* actual line(s) */
	g_assert (!cc->priv->shapes);
	canvas_shapes = gnome_db_goo_util_compute_anchor_shapes (GOO_CANVAS_ITEM (cc), NULL,
								 cc->priv->fk_entity_item, 
								 cc->priv->ref_pk_entity_item, 
								 g_slist_length (fklist), 0, TRUE);
	cc->priv->shapes = gnome_db_goo_canvas_shapes_remove_obsolete_shapes (canvas_shapes);
	list2 = canvas_shapes;
	list = fklist;
	while (list && list2) {
		GooCanvasItem *item = GNOME_DB_GOO_CANVAS_SHAPE (list2->data)->item;
		gchar *color = "black";
		fkcons = GDA_DICT_CONSTRAINT (list->data);
		g_object_get (G_OBJECT (fkcons), "user_constraint", &user_constraint, NULL);
		if (user_constraint)
			color = "#B2B2B2";

		g_object_set (G_OBJECT (item), 
			      "stroke-color", color,
			      NULL);
		
		if (G_OBJECT_TYPE (item) == GOO_TYPE_CANVAS_POLYLINE) {
			g_object_set (G_OBJECT (item), 
				      "start-arrow", TRUE,
				      "arrow-tip-length", 4.,
				      "arrow-length", 5.,
				      "arrow-width", 4.,
				      NULL);
			if (list2 != canvas_shapes)
				list = g_slist_next (list); /* next constraint */
		}
		else if (G_OBJECT_TYPE (item) == GOO_TYPE_CANVAS_ELLIPSE)
			g_object_set (G_OBJECT (item), 
				      "fill-color", color,
				      NULL);

		g_object_set_data (G_OBJECT (item), "fkcons", fkcons);
		g_signal_connect (G_OBJECT (item), "enter-notify-event", 
				  G_CALLBACK (single_item_enter_notify_event_cb), cc);
		g_signal_connect (G_OBJECT (item), "leave-notify-event", 
				  G_CALLBACK (single_item_leave_notify_event_cb), cc);
		g_signal_connect (G_OBJECT (item), "button-press-event",
				  G_CALLBACK (single_item_button_press_event_cb), cc);
		
		list2 = list2->next;
	}
	g_return_if_fail (!list2);
}

/*
 * update GooCanvasItem objects
 *
 * It is assumed that the list of FK constraints to render is coherent (that all the constraints are
 * FK constraints, for the same table, referencing the same FK table).
 */
static void 
update_items (GnomeDbGooFkconstraint *cc)
{
	GSList *fklist = cc->priv->constraints;

	if (!fklist)
		return;

	cc->priv->shapes = gnome_db_goo_util_compute_anchor_shapes (GOO_CANVAS_ITEM (cc), cc->priv->shapes,
								    cc->priv->fk_entity_item, 
								    cc->priv->ref_pk_entity_item, 
								    g_slist_length (fklist), 0, TRUE);
	cc->priv->shapes = gnome_db_goo_canvas_shapes_remove_obsolete_shapes (cc->priv->shapes);
}

static void popup_delete_cb (GtkMenuItem *mitem, GnomeDbGooFkconstraint *cc);

/*
 * item is for a single FK constraint
 */
static gboolean
single_item_enter_notify_event_cb (GooCanvasItem *ci, GooCanvasItem *target_item,
				   GdkEventCrossing *event, GnomeDbGooFkconstraint *cc)
{
	GdaDictConstraint *fkcons = g_object_get_data (G_OBJECT (ci), "fkcons");
	GSList *list, *pairs;

	pairs = gda_dict_constraint_fkey_get_fields (fkcons);
	list = pairs;
	while (list) {
		GdaDictConstraintFkeyPair *pair = GDA_DICT_CONSTRAINT_FK_PAIR (list->data);
		GnomeDbGooField *field;
		
		field = gnome_db_goo_entity_get_field_item (cc->priv->fk_entity_item, 
							    GDA_ENTITY_FIELD (pair->fkey));
		gnome_db_goo_text_set_highlight (GNOME_DB_GOO_TEXT (field), TRUE);
		field = gnome_db_goo_entity_get_field_item (cc->priv->ref_pk_entity_item, 
							    GDA_ENTITY_FIELD (pair->ref_pkey));
		gnome_db_goo_text_set_highlight (GNOME_DB_GOO_TEXT (field), TRUE);
		
		g_free (pair);
		list = g_slist_next (list);
	}
	g_slist_free (pairs);

	return FALSE;
}

static gboolean
single_item_leave_notify_event_cb (GooCanvasItem *ci, GooCanvasItem *target_item,
				   GdkEventCrossing *event, GnomeDbGooFkconstraint *cc)
{
	GdaDictConstraint *fkcons = g_object_get_data (G_OBJECT (ci), "fkcons");
	GSList *list, *pairs;

	pairs = gda_dict_constraint_fkey_get_fields (fkcons);
	list = pairs;
	while (list) {
		GdaDictConstraintFkeyPair *pair = GDA_DICT_CONSTRAINT_FK_PAIR (list->data);
		GnomeDbGooField *field;
		
		field = gnome_db_goo_entity_get_field_item (cc->priv->fk_entity_item, 
							    GDA_ENTITY_FIELD (pair->fkey));
		gnome_db_goo_text_set_highlight (GNOME_DB_GOO_TEXT (field), FALSE);
		field = gnome_db_goo_entity_get_field_item (cc->priv->ref_pk_entity_item, 
							    GDA_ENTITY_FIELD (pair->ref_pkey));
		gnome_db_goo_text_set_highlight (GNOME_DB_GOO_TEXT (field), FALSE);
		
		g_free (pair);
		list = g_slist_next (list);
	}
	g_slist_free (pairs);

	return FALSE;
}

static gboolean
single_item_button_press_event_cb (GooCanvasItem *ci, GooCanvasItem *target_item,
				   GdkEventButton *event, GnomeDbGooFkconstraint *cc)
{
	GdaDictConstraint *fkcons = g_object_get_data (G_OBJECT (ci), "fkcons");
	gboolean is_user_constraint = FALSE;
	GtkWidget *menu, *entry;

	menu = gtk_menu_new ();
	entry = gtk_menu_item_new_with_label (_("Remove"));
	g_object_get (G_OBJECT (fkcons), "user_constraint", &is_user_constraint, NULL);
	gtk_widget_set_sensitive (entry, is_user_constraint);
	g_object_set_data (G_OBJECT (entry), "fkcons", fkcons);
	g_signal_connect (G_OBJECT (entry), "activate", G_CALLBACK (popup_delete_cb), cc);
	gtk_menu_append (GTK_MENU (menu), entry);
	gtk_widget_show (entry);
	gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
			NULL, NULL, ((GdkEventButton *)event)->button,
			((GdkEventButton *)event)->time);
	return TRUE;
}


static void
popup_delete_cb (GtkMenuItem *mitem, GnomeDbGooFkconstraint *cc)
{
	GdaDictConstraint *fkcons = g_object_get_data (G_OBJECT (mitem), "fkcons");
	gboolean is_user_constraint;

	g_object_get (G_OBJECT (fkcons), "user_constraint", &is_user_constraint, NULL);
	if (is_user_constraint)
		gda_object_destroy (GDA_OBJECT (fkcons));
}

static void
entity_item_moved_cb (GooCanvasItem *entity, GnomeDbGooFkconstraint *cc)
{
	update_items (cc);
}


/**
 * gnome_db_goo_fkconstraint_add_constraint
 * @cc:
 * @fkcons:
 * 
 * Add @fkcons to the list of displayed constraints
 */
void
gnome_db_goo_fkconstraint_add_constraint (GnomeDbGooFkconstraint *cc, GdaDictConstraint *fkcons)
{
	g_return_if_fail (cc && GNOME_DB_IS_GOO_FKCONSTRAINT (cc));
	g_return_if_fail (cc->priv);
	g_return_if_fail (fkcons && GDA_IS_DICT_CONSTRAINT (fkcons));
	g_return_if_fail (gda_dict_constraint_get_constraint_type (GDA_DICT_CONSTRAINT (fkcons)) == 
			  CONSTRAINT_FOREIGN_KEY);

	if (g_slist_find (cc->priv->constraints, fkcons))
		return;

	if (cc->priv->constraints) {
		/* there are already some FK constraints there, so test that the new one is
		 * compatible with the existing ones*/
		
	}

	cc->priv->constraints = g_slist_append (cc->priv->constraints, fkcons);
	gda_object_connect_destroy (fkcons, G_CALLBACK (constraint_destroyed_cb), cc);

	clean_items (cc);
	create_items (cc);
}

/**
 * gnome_db_goo_fkconstraint_new
 * @parent: the parent item, or NULL. 
 * @fkcons: the #GdaDictConstraint to represent
 * @...: optional pairs of property names and values, and a terminating NULL.
 *
 * Creates a new canvas item to represent the @fkcons FK constraint
 *
 * Returns: a new #GooCanvasItem object
 */
GooCanvasItem *
gnome_db_goo_fkconstraint_new (GooCanvasItem *parent, GdaDictConstraint *fkcons, ...)
{
	GooCanvasItem *item;
	const char *first_property;
	va_list var_args;
		
	item = g_object_new (GNOME_DB_TYPE_GOO_FKCONSTRAINT, NULL);

	if (parent) {
		goo_canvas_item_add_child (parent, item, -1);
		g_object_unref (item);
	}

	g_object_set (item, "fk_constraint", fkcons, NULL);

	va_start (var_args, fkcons);
	first_property = va_arg (var_args, char*);
	if (first_property)
		g_object_set_valist ((GObject*) item, first_property, var_args);
	va_end (var_args);

	return item;
}
