/* gnome-db-canvas-fkconstraint.c
 *
 * Copyright (C) 2004 - 2006 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gtk/gtk.h>
#include <math.h>
#include <libgda/libgda.h>
#include <glib/gi18n-lib.h>
#include "gnome-db-canvas.h"
#include "gnome-db-canvas-fkconstraint.h"
#include "gnome-db-canvas-entity.h"
#include "gnome-db-canvas-text.h"
#include "graph-utility.h"

static void gnome_db_canvas_fkconstraint_class_init (GnomeDbCanvasFkconstraintClass * class);
static void gnome_db_canvas_fkconstraint_init       (GnomeDbCanvasFkconstraint * cc);
static void gnome_db_canvas_fkconstraint_dispose    (GObject   * object);
static void gnome_db_canvas_fkconstraint_finalize   (GObject   * object);

static void gnome_db_canvas_fkconstraint_set_property    (GObject *object,
						    guint param_id,
						    const GValue *value,
						    GParamSpec *pspec);
static void gnome_db_canvas_fkconstraint_get_property    (GObject *object,
						    guint param_id,
						    GValue *value,
						    GParamSpec *pspec);

static void gnome_db_canvas_fkconstraint_get_edge_nodes (GnomeDbCanvasItem *citem, 
							 GnomeDbCanvasItem **from, GnomeDbCanvasItem **to);

static void clean_items (GnomeDbCanvasFkconstraint *cc);
static void create_items (GnomeDbCanvasFkconstraint *cc);

enum
{
	PROP_0,
	PROP_FK_CONSTRAINT
};

struct _GnomeDbCanvasFkconstraintPrivate
{
	GSList                *constraints;
	GnomeDbCanvasEntity   *fk_entity_item;
	GnomeDbCanvasEntity   *ref_pk_entity_item;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

GType
gnome_db_canvas_fkconstraint_get_type (void)
{
	static GType type = 0;

        if (G_UNLIKELY (type == 0)) {
		static const GTypeInfo info = {
			sizeof (GnomeDbCanvasFkconstraintClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_canvas_fkconstraint_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbCanvasFkconstraint),
			0,
			(GInstanceInitFunc) gnome_db_canvas_fkconstraint_init
		};		

		type = g_type_register_static (GNOME_DB_TYPE_CANVAS_ITEM, "GnomeDbCanvasFkconstraint", &info, 0);
	}

	return type;
}	

static void
gnome_db_canvas_fkconstraint_class_init (GnomeDbCanvasFkconstraintClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	GNOME_DB_CANVAS_ITEM_CLASS (class)->get_edge_nodes = gnome_db_canvas_fkconstraint_get_edge_nodes;

	object_class->dispose = gnome_db_canvas_fkconstraint_dispose;
	object_class->finalize = gnome_db_canvas_fkconstraint_finalize;

	/* Properties */
	object_class->set_property = gnome_db_canvas_fkconstraint_set_property;
	object_class->get_property = gnome_db_canvas_fkconstraint_get_property;
	g_object_class_install_property (object_class, PROP_FK_CONSTRAINT,
					 g_param_spec_pointer ("fk_constraint", "Latest FK constraint to add", 
							       NULL, G_PARAM_WRITABLE));
       
}

static void
gnome_db_canvas_fkconstraint_init (GnomeDbCanvasFkconstraint *cc)
{
	cc->priv = g_new0 (GnomeDbCanvasFkconstraintPrivate, 1);
	cc->priv->constraints = NULL;
	cc->priv->fk_entity_item = NULL;
	cc->priv->ref_pk_entity_item = NULL;
}


static void entity_destroy_cb (GnomeDbCanvasEntity *entity, GnomeDbCanvasFkconstraint *cc);
static void constraint_destroyed_cb (GdaDictConstraint *fkcons, GnomeDbCanvasFkconstraint *cc);

static void
gnome_db_canvas_fkconstraint_dispose (GObject *object)
{
	GnomeDbCanvasFkconstraint *cc;
	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_CANVAS_FKCONSTRAINT (object));

	cc = GNOME_DB_CANVAS_FKCONSTRAINT (object);

	clean_items (cc);
	if (cc->priv->constraints) {
		GSList *list = cc->priv->constraints;
		while (list) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (list->data),
							      G_CALLBACK (constraint_destroyed_cb), cc);
			list = g_slist_next (list);
		}
		g_slist_free (cc->priv->constraints);
		cc->priv->constraints = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}


static void
gnome_db_canvas_fkconstraint_finalize (GObject *object)
{
	GnomeDbCanvasFkconstraint *cc;
	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_CANVAS_FKCONSTRAINT (object));

	cc = GNOME_DB_CANVAS_FKCONSTRAINT (object);
	if (cc->priv) {
		g_free (cc->priv);
		cc->priv = NULL;
	}

	/* for the parent class */
	parent_class->finalize (object);
}

static void 
gnome_db_canvas_fkconstraint_set_property    (GObject *object,
					guint param_id,
					const GValue *value,
					GParamSpec *pspec)
{
	GnomeDbCanvasFkconstraint *cc;

	cc = GNOME_DB_CANVAS_FKCONSTRAINT (object);

	switch (param_id) {
	case PROP_FK_CONSTRAINT:
		gnome_db_canvas_fkconstraint_add_constraint (cc, g_value_get_pointer (value));
		break;
	}
}

static void 
gnome_db_canvas_fkconstraint_get_property    (GObject *object,
					guint param_id,
					GValue *value,
					GParamSpec *pspec)
{
	GnomeDbCanvasFkconstraint *cc;

	cc = GNOME_DB_CANVAS_FKCONSTRAINT (object);

	switch (param_id) {
	default:
		g_warning ("No such property!");
		break;
	}
}

static void
gnome_db_canvas_fkconstraint_get_edge_nodes (GnomeDbCanvasItem *citem, 
					     GnomeDbCanvasItem **from, GnomeDbCanvasItem **to)
{
	GnomeDbCanvasFkconstraint *cc;

	cc = GNOME_DB_CANVAS_FKCONSTRAINT (citem);

	if (from)
		*from = (GnomeDbCanvasItem*) cc->priv->fk_entity_item;
	if (to)
		*to = (GnomeDbCanvasItem*) cc->priv->ref_pk_entity_item;
}

static void
constraint_destroyed_cb (GdaDictConstraint *fkcons, GnomeDbCanvasFkconstraint *cc)
{
	g_assert (g_slist_find (cc->priv->constraints, fkcons));
	cc->priv->constraints = g_slist_remove (cc->priv->constraints, fkcons);
	g_signal_handlers_disconnect_by_func (G_OBJECT (fkcons),
					      G_CALLBACK (constraint_destroyed_cb), cc);
	
	/* destroy itself if there are no more constraint left in the end */
	if (!cc->priv->constraints)
		gtk_object_destroy (GTK_OBJECT (cc));
	else {
		clean_items (cc);
		create_items (cc);
	}
}

static void
entity_destroy_cb (GnomeDbCanvasEntity *entity, GnomeDbCanvasFkconstraint *cc)
{
	gtk_object_destroy (GTK_OBJECT (cc));
}


static gboolean single_item_event_cb (GnomeCanvasItem *ci, GdkEvent *event, GnomeDbCanvasFkconstraint *cc);
static void entity_item_moved_cb (GnomeCanvasItem *entity, GnomeDbCanvasFkconstraint *cc);

/* 
 * destroy any existing GnomeCanvasItem objects 
 */
static void 
clean_items (GnomeDbCanvasFkconstraint *cc)
{
	if (cc->priv->fk_entity_item) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->fk_entity_item),
						      G_CALLBACK (entity_item_moved_cb), cc);
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->fk_entity_item),
						      G_CALLBACK (entity_destroy_cb), cc);
		cc->priv->fk_entity_item = NULL;
	}

	if (cc->priv->ref_pk_entity_item) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->ref_pk_entity_item),
						      G_CALLBACK (entity_item_moved_cb), cc);
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->ref_pk_entity_item),
						      G_CALLBACK (entity_destroy_cb), cc);
		cc->priv->ref_pk_entity_item = NULL;
	}

	/* remove all the GnomeCanvasItem objects */
	while (GNOME_CANVAS_GROUP (cc)->item_list)
		gtk_object_destroy (GTK_OBJECT (GNOME_CANVAS_GROUP (cc)->item_list->data));
	
}

/*
 * create new GnomeCanvasItem objects
 *
 * It is assumed that the list of FK constraints to render is coherent (that all the constraints are
 * FK constraints, for the same table, referencing the same FK table).
 */
static void 
create_items (GnomeDbCanvasFkconstraint *cc)
{
	GnomeCanvasItem *item;
	GSList *fklist = cc->priv->constraints, *list, *canvas_shapes, *list2, *list3;
	GdaDictConstraint *fkcons;
	GdaDictTable *table;
	GnomeDbCanvasItem *entity_item;
	GnomeDbCanvas *canvas = gnome_db_canvas_item_get_canvas (GNOME_DB_CANVAS_ITEM (cc));
	gboolean user_constraint;

	if (!fklist)
		return;

	/* Analyse FK constraint */
	fkcons = GDA_DICT_CONSTRAINT (fklist->data);
	item = GNOME_CANVAS_ITEM (cc);
	table = gda_dict_constraint_get_table (fkcons);
	entity_item = gnome_db_canvas_get_item_for_object (canvas, GDA_OBJECT (table));
	g_return_if_fail (entity_item);
	cc->priv->fk_entity_item = GNOME_DB_CANVAS_ENTITY (entity_item);
	g_signal_connect (G_OBJECT (entity_item), "moving",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "moved",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "shifted",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "destroy",
			  G_CALLBACK (entity_destroy_cb), cc);

	table = gda_dict_constraint_fkey_get_ref_table (fkcons);
	entity_item = gnome_db_canvas_get_item_for_object (canvas, GDA_OBJECT (table));
	g_return_if_fail (entity_item);
	cc->priv->ref_pk_entity_item = GNOME_DB_CANVAS_ENTITY (entity_item);
	g_signal_connect (G_OBJECT (entity_item), "moving",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "moved",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "shifted",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "destroy",
			  G_CALLBACK (entity_destroy_cb), cc);

	/* actual line(s) */
	canvas_shapes = graph_util_compute_anchor_shapes (cc->priv->fk_entity_item, cc->priv->ref_pk_entity_item, 
							  g_slist_length (fklist), 0);

	list3 = graph_util_draw_canvas_shapes (GNOME_CANVAS_GROUP (cc), canvas_shapes);
	graph_util_free_canvas_shapes (canvas_shapes);
	list2 = list3;
	list = fklist;
	while (list && list2) {
		gchar *color = "black";
		fkcons = GDA_DICT_CONSTRAINT (list->data);
		g_object_get (G_OBJECT (fkcons), "user_constraint", &user_constraint, NULL);
		if (user_constraint)
			color = "#B2B2B2";

		g_object_set (G_OBJECT (list2->data), 
			      "fill_color", color,
			      NULL);

		if (G_OBJECT_TYPE (list2->data) == GNOME_TYPE_CANVAS_LINE) {
			g_object_set (G_OBJECT (list2->data), 
				      "first_arrowhead", TRUE,
				      "arrow-shape-a", 8.,
				      "arrow-shape-b", 12.,
				      "arrow-shape-c", 5.,
				      NULL);
			if (list2 != list3)
				list = g_slist_next (list); /* next constraint */
		}

		g_object_set_data (G_OBJECT (list2->data), "fkcons", fkcons);
		g_signal_connect (G_OBJECT (list2->data), "event", 
				  G_CALLBACK (single_item_event_cb), cc);
		
		list2 = g_slist_next (list2);
	}
	g_slist_free (list3);
	g_return_if_fail (!list2);
}

static void popup_delete_cb (GtkMenuItem *mitem, GnomeDbCanvasFkconstraint *cc);


/*
 * item is for a single FK constraint
 */
static gboolean
single_item_event_cb (GnomeCanvasItem *ci, GdkEvent *event, GnomeDbCanvasFkconstraint *cc)
{
	GdaDictConstraint *fkcons = g_object_get_data (G_OBJECT (ci), "fkcons");
	gboolean highlight = FALSE, is_user_constraint = FALSE;
	GtkWidget *menu, *entry;
	gboolean done = FALSE;
	GSList *list, *pairs;

	switch (event->type) {
	case GDK_ENTER_NOTIFY:
		highlight = TRUE;
	case GDK_LEAVE_NOTIFY:
		pairs = gda_dict_constraint_fkey_get_fields (fkcons);
		list = pairs;
		while (list) {
			GdaDictConstraintFkeyPair *pair = GDA_DICT_CONSTRAINT_FK_PAIR (list->data);
			GnomeDbCanvasField *field;
			
			field = gnome_db_canvas_entity_get_field_item (cc->priv->fk_entity_item, 
								 GDA_ENTITY_FIELD (pair->fkey));
			gnome_db_canvas_text_set_highlight (GNOME_DB_CANVAS_TEXT (field), highlight);
			field = gnome_db_canvas_entity_get_field_item (cc->priv->ref_pk_entity_item, 
								 GDA_ENTITY_FIELD (pair->ref_pkey));
			gnome_db_canvas_text_set_highlight (GNOME_DB_CANVAS_TEXT (field), highlight);
			
			g_free (pair);
			list = g_slist_next (list);
		}
		g_slist_free (pairs);
		break;
	case GDK_BUTTON_PRESS:
                menu = gtk_menu_new ();
                entry = gtk_menu_item_new_with_label (_("Remove"));
		g_object_get (G_OBJECT (fkcons), "user_constraint", &is_user_constraint, NULL);
		gtk_widget_set_sensitive (entry, is_user_constraint);
		g_object_set_data (G_OBJECT (entry), "fkcons", fkcons);
                g_signal_connect (G_OBJECT (entry), "activate", G_CALLBACK (popup_delete_cb), cc);
                gtk_menu_append (GTK_MENU (menu), entry);
                gtk_widget_show (entry);
                gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
                                NULL, NULL, ((GdkEventButton *)event)->button,
                                ((GdkEventButton *)event)->time);
		done = TRUE;
                break;
	default:
		break;
	}

	return done;
}


static void
popup_delete_cb (GtkMenuItem *mitem, GnomeDbCanvasFkconstraint *cc)
{
	GdaDictConstraint *fkcons = g_object_get_data (G_OBJECT (mitem), "fkcons");
	gboolean is_user_constraint;

	g_object_get (G_OBJECT (fkcons), "user_constraint", &is_user_constraint, NULL);
	if (is_user_constraint)
		gda_object_destroy (GDA_OBJECT (fkcons));
}

static void
entity_item_moved_cb (GnomeCanvasItem *entity, GnomeDbCanvasFkconstraint *cc)
{
	clean_items (cc);
	create_items (cc);
}


/**
 * gnome_db_canvas_fkconstraint_add_constraint
 * @cc:
 * @fkcons:
 * 
 * Add @fkcons to the list of displayed constraints
 */
void
gnome_db_canvas_fkconstraint_add_constraint (GnomeDbCanvasFkconstraint *cc, GdaDictConstraint *fkcons)
{
	g_return_if_fail (cc && GNOME_DB_IS_CANVAS_FKCONSTRAINT (cc));
	g_return_if_fail (cc->priv);
	g_return_if_fail (fkcons && GDA_IS_DICT_CONSTRAINT (fkcons));
	g_return_if_fail (gda_dict_constraint_get_constraint_type (GDA_DICT_CONSTRAINT (fkcons)) == 
			  CONSTRAINT_FOREIGN_KEY);

	if (g_slist_find (cc->priv->constraints, fkcons))
		return;

	if (cc->priv->constraints) {
		/* there are already some FK constraints there, so test that the new one is
		 * compatible with the existing ones*/
		
	}

	cc->priv->constraints = g_slist_append (cc->priv->constraints, fkcons);
	gda_object_connect_destroy (fkcons, G_CALLBACK (constraint_destroyed_cb), cc);

	clean_items (cc);
	create_items (cc);
}
