/*
 * gnome-file-selector-util.h - functions for getting files from a
 * selector
 *
 * Authors:
 *    Jacob Berkman  <jacob@ximian.com>
 *
 * Copyright 2001 Ximian, Inc.
 *
 */

#include <config.h>

#include "gnome-file-selector-util.h"

#include <bonobo/bonobo-exception.h>
#include <bonobo/bonobo-listener.h>
#include <bonobo/bonobo-widget.h>

#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>

#include <libgnome/libgnome.h>
#include <libgnome/gnome-i18n.h>

#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-window-icon.h>

typedef enum {
	FILESEL_OPEN,
	FILESEL_OPEN_MULTI,
	FILESEL_SAVE
} FileselMode;

typedef struct {
	Bonobo_EventSource es;
	Bonobo_EventSource_ListenerId id;

	GtkWidget *dialog;
	GtkWidget *control;

	BonoboListener *listener;
	gpointer data;

	FileselMode mode;
} FileSelector;

static void
destroy_file_selector (FileSelector *filesel)
{
	if (filesel->dialog)
		gtk_widget_destroy (filesel->dialog);
	else if (filesel->control)
		gtk_widget_destroy (filesel->control);

	if (filesel->es != CORBA_OBJECT_NIL) {
		CORBA_Environment ev;

		CORBA_exception_init (&ev);
		if (filesel->id)
			Bonobo_EventSource_removeListener (
				filesel->es, filesel->id, &ev);
		bonobo_object_release_unref (filesel->es, &ev);
		CORBA_exception_free (&ev);
	}

	if (filesel->listener)
		bonobo_object_unref (BONOBO_OBJECT (filesel->listener));

	/* don't free data since that is returned to the caller */

	g_free (filesel);
}

static gint
delete_file_selector (GtkWidget *d, GdkEventAny *e, gpointer data)
{
	gtk_widget_hide (d);
	gtk_main_quit ();
	return TRUE;
}

static void
listener_cb (BonoboListener *listener, 
	     gchar *event_name,
	     CORBA_any *any,
	     CORBA_Environment *ev,
	     gpointer data)
{
	FileSelector *filesel;
	CORBA_sequence_CORBA_string *seq;
	char *subtype;

	filesel = data;
	gtk_widget_hide (filesel->dialog);

	subtype = bonobo_event_subtype (event_name);
	if (!strcmp (subtype, "Cancel"))
		goto cancel_clicked;

	seq = any->_value;
	if (seq->_length < 1)
		goto cancel_clicked;

	if (filesel->mode == FILESEL_OPEN_MULTI) {
		char **strv;
		int i;

		if (seq->_length == 0) {
			filesel->data = NULL;
			goto cancel_clicked;
		}

		strv = g_new (char *, seq->_length + 1);
		for (i = 0; i < seq->_length; i++)
			strv[i] = g_strdup (seq->_buffer[i]);
		strv[i] = NULL;
		filesel->data = strv;
	} else {
		filesel->data = g_strdup (seq->_buffer[0]);
	}

 cancel_clicked:
	g_free (subtype);
	gtk_main_quit ();
}

/* FIXME: break up into smaller functions */
static gpointer
run_file_slector (GtkWindow  *parent,
		  gboolean    enable_vfs,
		  FileselMode mode, 
		  const char *title,
		  const char *mime_types,
		  const char *default_path, 
		  const char *default_filename)

{
	FileSelector      *filesel;
	Bonobo_Unknown     corba_control;
	CORBA_Environment  ev;
	char              *moniker;
	gpointer           retval;

	filesel = g_new0 (FileSelector, 1);
	filesel->mode = mode;

	moniker = g_strdup_printf (
		"OAFIID:GNOME_FileSelector_Control!"
		"Application=%s;"
		"EnableVFS=%d;"
		"MultipleSelection=%d;"
		"SaveMode=%d",
		gnome_app_id,
		enable_vfs,
		mode == FILESEL_OPEN_MULTI, 
		mode == FILESEL_SAVE);

	filesel->control = bonobo_widget_new_control (moniker, CORBA_OBJECT_NIL);
	g_free (moniker);

	if (!filesel->control)
		goto error_out;

	if (mime_types)
		bonobo_widget_set_property (BONOBO_WIDGET (filesel->control),
					    "MimeTypes", mime_types,
					    NULL);

	if (default_path)
		bonobo_widget_set_property (BONOBO_WIDGET (filesel->control),
					    "DefaultLocation", default_path,
					    NULL);

	if (default_filename)
		bonobo_widget_set_property (BONOBO_WIDGET (filesel->control),
					    "DefaultFileName", default_filename,
					    NULL);

	corba_control = bonobo_widget_get_objref (BONOBO_WIDGET (filesel->control));

	CORBA_exception_init (&ev);
	filesel->es = Bonobo_Unknown_queryInterface (corba_control,
						     "IDL:Bonobo/EventSource:1.0",
						     &ev);
	if (BONOBO_EX (&ev)) {
		filesel->es = CORBA_OBJECT_NIL;
		goto error_out;
	}

	filesel->listener = bonobo_listener_new (listener_cb, filesel);
	filesel->id = Bonobo_EventSource_addListenerWithMask (
		filesel->es, BONOBO_OBJREF (filesel->listener),
		"GNOME/FileSelector/Control:ButtonClicked",
		&ev);

	if (BONOBO_EX (&ev)) {
		filesel->id = 0;
		goto error_out;
	}

	CORBA_exception_free (&ev);

	filesel->dialog = gtk_window_new (GTK_WINDOW_DIALOG);

	gtk_container_add (GTK_CONTAINER (filesel->dialog), 
			   filesel->control);
	gtk_widget_set_usize (filesel->dialog, 560, 450);
	gnome_window_icon_set_from_default (GTK_WINDOW (filesel->dialog));
	gtk_window_set_title (GTK_WINDOW (filesel->dialog), title);
	if (parent)
		gtk_window_set_transient_for (GTK_WINDOW (filesel->dialog), 
					      parent);
	gtk_window_set_modal (GTK_WINDOW (filesel->dialog), TRUE);

	gtk_signal_connect (GTK_OBJECT (filesel->dialog), "delete_event",
			    GTK_SIGNAL_FUNC (delete_file_selector),
			    filesel);

	gtk_widget_show_all (filesel->dialog);
	gtk_main ();

	retval = filesel->data;
	destroy_file_selector (filesel);

	return retval;

 error_out: 
	{
		GtkWidget *d;
		
		d = gnome_error_dialog (_("There was an error creating a file selector.\n"
					  "Please check your GNOME installation and make\n"
					  "sure the gnome-file-selector package is\n"
					  "correctly installed."));
		gnome_dialog_run_and_close (GNOME_DIALOG (d));
	}

	destroy_file_selector (filesel);
	return NULL;
}

/**
 * gnome_file_selector_open:
 * @parent: optional window the dialog should be a transient for.
 * @enable_vfs: if FALSE, restrict files to local paths.
 * @mime_types: optional list of mime types to provide filters for.
 *   These are of the form: "HTML Files:text/html|Text Files:text/html,text/plain"
 * @default_path: optional directory to start in
 *
 * Creates and shows a modal open file dialog, waiting for the user to
 * select a file or cancel before returning.
 *
 * Return value: the URI (or plain file path if @enable_vfs is FALSE)
 * of the file selected, or NULL if cancel was pressed.
 **/
char *
gnome_file_selector_open (GtkWindow  *parent,
			  gboolean    enable_vfs,
			  const char *mime_types,
			  const char *default_path)
{
	return run_file_slector (parent, enable_vfs, FILESEL_OPEN, 
				 _("Select a file to open"),
				 mime_types, default_path, NULL);
}

/**
 * gnome_file_selector_open_multi:
 * @parent: optional window the dialog should be a transient for
 * @enable_vfs: if FALSE, restrict files to local paths.
 * @mime_types: optional list of mime types to provide filters for.
 *   These are of the form: "HTML Files:text/html|Text Files:text/html,text/plain"
 * @default_path: optional directory to start in
 *
 * Creates and shows a modal open file dialog, waiting for the user to
 * select a file or cancel before returning.
 *
 * Return value: a NULL terminated string array of the selected URIs
 * (or local file paths if @enable_vfs is FALSE), or NULL if cancel
 * was pressed.
 **/
char **
gnome_file_selector_open_multi (GtkWindow  *parent,
				gboolean    enable_vfs,
				const char *mime_types,
				const char *default_path)
{
	return run_file_slector (parent, enable_vfs, FILESEL_OPEN_MULTI,
				 _("Select files to open"),
				 mime_types, default_path, NULL);
}

/**
 * gnome_file_selector_save:
 * @parent: optional window the dialog should be a transient for
 * @enable_vfs: if FALSE, restrict files to local paths.
 * @mime_types: optional list of mime types to provide filters for.
 *   These are of the form: "HTML Files:text/html|Text Files:text/html,text/plain"
 * @default_path: optional directory to start in
 * @default_filename: optional file name to default to
 *
 * Creates and shows a modal save file dialog, waiting for the user to
 * select a file or cancel before returning.
 *
 * Return value: the URI (or plain file path if @enable_vfs is FALSE)
 * of the file selected, or NULL if cancel was pressed.
 **/
char *
gnome_file_selector_save (GtkWindow  *parent,
			  gboolean    enable_vfs,
			  const char *mime_types,
			  const char *default_path, 
			  const char *default_filename)
{
	return run_file_slector (parent, FILESEL_SAVE, enable_vfs,
				 _("Select a filename to save"),
				 mime_types, default_path, default_filename);
}
