/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  gpa-transport-selector.c: A print transport selector
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors :
 *    Lauris Kaplinski <lauris@ximian.com>
 *
 *  Copyright (C) 2000-2001 Ximian, Inc. 
 *
 */

#include "config.h"

#include <string.h>
#include <gtk/gtk.h>

#include "gnome-print-i18n.h"
#include "gpa-transport-selector.h"

static void gpa_transport_selector_class_init (GPATransportSelectorClass *klass);
static void gpa_transport_selector_init (GPATransportSelector *selector);
static void gpa_transport_selector_finalize (GObject *object);

static gint gpa_transport_selector_construct (GPAWidget *widget);

static void gpa_transport_selector_rebuild_widget (GPATransportSelector *ts);

static void gpa_transport_selector_filename_entry_changed       (GtkEntry *entry, GPAWidget *gpw);
static void gpa_transport_selector_custom_command_entry_changed (GtkEntry *entry, GPAWidget *gpw);
static void gpa_transport_selector_printer_entry_changed        (GtkEntry *entry, GPAWidget *gpw);
static void gpa_transport_selector_default_printer_rb_toggled   (GtkRadioButton *radiobutton, GPATransportSelector *ts);
static void gpa_transport_selector_custom_printer_rb_toggled    (GtkRadioButton *radiobutton, GPATransportSelector *ts);
static void gpa_transport_selector_menuitem_activate       (GtkWidget *widget, gint index);
static void gpa_transport_selector_select_transport        (GPATransportSelector *ts, const gchar *id);

static GPAWidgetClass *parent_class;

GtkType
gpa_transport_selector_get_type (void)
{
	static GType type = 0;
	if (!type) {
		static const GTypeInfo info = {
			sizeof (GPATransportSelectorClass),
			NULL, NULL,
			(GClassInitFunc) gpa_transport_selector_class_init,
			NULL, NULL,
			sizeof (GPATransportSelector),
			0,
			(GInstanceInitFunc) gpa_transport_selector_init,
		};
		type = g_type_register_static (GPA_TYPE_WIDGET, "GPATransportSelector", &info, 0);
	}
	return type;
}

static void
gpa_transport_selector_class_init (GPATransportSelectorClass *klass)
{
	GObjectClass *object_class;
	GPAWidgetClass *gpa_class;

	object_class = (GObjectClass *) klass;
	gpa_class = (GPAWidgetClass *) klass;

	parent_class = gtk_type_class (GPA_TYPE_WIDGET);

	object_class->finalize = gpa_transport_selector_finalize;

	gpa_class->construct = gpa_transport_selector_construct;
}

static void
gpa_transport_selector_filename_entry_changed (GtkEntry *entry, GPAWidget *gpw)
{
	const guchar *text;

	text = gtk_entry_get_text (entry);

	gnome_print_config_set (gpw->config, "Settings.Transport.Backend.FileName", text);
}

static void
gpa_transport_selector_printer_entry_changed (GtkEntry *entry, GPAWidget *gpw)
{
	const guchar *text;

	text = gtk_entry_get_text (entry);

	gnome_print_config_set (gpw->config, "Settings.Transport.Backend.Printer", text);
}

static void
gpa_transport_selector_custom_command_entry_changed (GtkEntry *entry, GPAWidget *gpw)
{
	const guchar *text;

	text = gtk_entry_get_text (entry);

	gnome_print_config_set (gpw->config, "Settings.Transport.Backend.Command", text);
}

static void
gpa_transport_selector_default_printer_rb_toggled (GtkRadioButton *radiobutton, GPATransportSelector *ts)
{
	gtk_widget_set_sensitive (ts->printer_entry, FALSE);
}

static void
gpa_transport_selector_custom_printer_rb_toggled (GtkRadioButton *radiobutton, GPATransportSelector *ts)
{
	gtk_widget_set_sensitive (ts->printer_entry, TRUE);
}

static void
gpa_transport_selector_init (GPATransportSelector *ts)
{
	ts->hbox = gtk_hbox_new (FALSE, 4);
	gtk_container_add (GTK_CONTAINER (ts), ts->hbox);
	gtk_widget_show (ts->hbox);

	/* Create transports option menu */
	ts->menu = gtk_option_menu_new ();
	gtk_box_pack_start (GTK_BOX (ts->hbox), ts->menu, FALSE, FALSE, 0);

	/* Create filename entry */
	ts->filename_entry = gtk_entry_new ();
	g_signal_connect (G_OBJECT (ts->filename_entry), "changed", GTK_SIGNAL_FUNC (gpa_transport_selector_filename_entry_changed), ts);
	gtk_box_pack_start (GTK_BOX (ts->hbox), ts->filename_entry, TRUE, TRUE, 0);

	/* Create printname selector */
	ts->printer_hbox = gtk_hbox_new (FALSE, 4);
	ts->default_printer_rb = gtk_radio_button_new_with_label (NULL,  _("Default printer"));
	g_signal_connect (G_OBJECT (ts->default_printer_rb), "toggled",
			  (GCallback) gpa_transport_selector_default_printer_rb_toggled, ts);
	gtk_box_pack_start (GTK_BOX (ts->printer_hbox), ts->default_printer_rb, TRUE, TRUE, 0);
	gtk_widget_show (ts->default_printer_rb);
	ts->custom_printer_rb = gtk_radio_button_new_from_widget (GTK_RADIO_BUTTON (ts->default_printer_rb));
	g_signal_connect (G_OBJECT (ts->custom_printer_rb), "toggled",
			  (GCallback) gpa_transport_selector_custom_printer_rb_toggled, ts);
	gtk_box_pack_start (GTK_BOX (ts->printer_hbox), ts->custom_printer_rb, TRUE, TRUE, 0);
	gtk_widget_show (ts->custom_printer_rb);
	
	ts->printer_entry = gtk_entry_new ();
	gtk_widget_set_sensitive (ts->printer_entry, FALSE);
	g_signal_connect (G_OBJECT (ts->printer_entry), "changed",
			  (GCallback) gpa_transport_selector_printer_entry_changed, ts);
	gtk_box_pack_end (GTK_BOX (ts->printer_hbox), ts->printer_entry, TRUE, TRUE, 0);
	gtk_widget_show (ts->printer_entry);

	gtk_box_pack_start (GTK_BOX (ts->hbox), ts->printer_hbox, TRUE, TRUE, 0);

	/* Create custom command entry */
	ts->custom_command_entry = gtk_entry_new ();
	g_signal_connect (G_OBJECT (ts->custom_command_entry), "changed",
			  (GCallback) gpa_transport_selector_custom_command_entry_changed, ts);
	gtk_box_pack_end (GTK_BOX (ts->hbox), ts->custom_command_entry, TRUE, TRUE, 0);
	

	ts->transportlist = NULL;
}

/* fixme: */
static void
gpa_transport_selector_finalize (GObject *object)
{
	GPATransportSelector *ts;

	ts = (GPATransportSelector *) object;

	if (ts->handler) {
		g_signal_handler_disconnect (G_OBJECT (ts->printer), ts->handler);
		ts->handler = 0;
	}
	
	if (ts->printer) {
		gpa_node_unref (ts->printer);
		ts->printer = NULL;
	}

	while (ts->transportlist) {
		gpa_node_unref (GPA_NODE (ts->transportlist->data));
		ts->transportlist = g_slist_remove (ts->transportlist, ts->transportlist->data);
	}

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gpa_transport_selector_rebuild_widget_cb (GPANode *node, guint flags, gpointer ts_ptr)
{
	GPATransportSelector *ts = ts_ptr;

	gpa_node_unref (ts->printer);
	ts->printer = gpa_node_get_child_from_path (GNOME_PRINT_CONFIG_NODE (GPA_WIDGET (ts)->config), "Printer");
	gpa_transport_selector_rebuild_widget (ts);
}

static gint
gpa_transport_selector_construct (GPAWidget *gpaw)
{
	GPATransportSelector *ts;

	ts = GPA_TRANSPORT_SELECTOR (gpaw);

	ts->printer = gpa_node_get_child_from_path (GNOME_PRINT_CONFIG_NODE (gpaw->config), "Printer");
	ts->handler = g_signal_connect (G_OBJECT (ts->printer), "modified",
					(GCallback) gpa_transport_selector_rebuild_widget_cb, ts);
	gpa_transport_selector_rebuild_widget (ts);

	return TRUE;
}

static void
gpa_transport_selector_rebuild_widget (GPATransportSelector *ts)
{
	GPANode *node;
	GPANode *option, *child;
	GtkWidget *menu, *item;
	gint idx, def;
	guchar *defid;
	GSList *l;

	node = GNOME_PRINT_CONFIG_NODE (GPA_WIDGET (ts)->config);

	/* Cleanup old state */
	while (ts->transportlist) {
		gpa_node_unref (GPA_NODE (ts->transportlist->data));
		ts->transportlist = g_slist_remove (ts->transportlist, ts->transportlist->data);
	}

	gtk_option_menu_remove_menu (GTK_OPTION_MENU (ts->menu));

	/* Construct new list */
	option = gpa_node_get_child_from_path (node, "Settings.Transport.Option.Backend");
	if (!option) {
		/* No transport node */
		gtk_widget_hide (ts->menu);
		gtk_widget_hide (ts->filename_entry);
		gtk_widget_activate (ts->default_printer_rb);
		gtk_widget_hide (ts->custom_command_entry);
		gtk_widget_show (ts->printer_hbox);
		return;
	}
	for (child = gpa_node_get_child (option, NULL); child != NULL; child = gpa_node_get_child (option, child)) {
		ts->transportlist = g_slist_prepend (ts->transportlist, child);
	}
	if (!ts->transportlist) {
		/* Empty transport list */
		gtk_widget_hide (ts->menu);
		gtk_widget_hide (ts->filename_entry);
		gtk_widget_activate (ts->default_printer_rb);
		gtk_widget_hide (ts->custom_command_entry);
		gtk_widget_show (ts->printer_hbox);
		return;
	}

	menu = gtk_menu_new ();
	ts->transportlist = g_slist_reverse (ts->transportlist);

	idx = 0;
	def = 0;
	defid = gpa_node_get_path_value (node, "Settings.Transport.Backend");
	/* Construct transport list */
	for (l = ts->transportlist; l != NULL; l = l->next) {
		gchar *val;
		val = gpa_node_get_path_value (GPA_NODE (l->data), "Name");
		if (!val) {
			g_warning ("Transport does not have 'Name' attribute");
		} else {
			item = gtk_menu_item_new_with_label (val);
			g_object_set_data (G_OBJECT (item), "GPAWidget", ts);
			g_signal_connect (G_OBJECT (item), "activate",
					  (GCallback) gpa_transport_selector_menuitem_activate, GINT_TO_POINTER (idx));
			gtk_widget_show (item);
			gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
			g_free (val);
			if (defid) {
				guchar *id;
				id = gpa_node_get_value (GPA_NODE (l->data));
				if (id && !strcmp (id, defid))
					def = idx;
				g_free (id);
			}
			idx += 1;
		}
	}
	if (idx == 0) {
		/* No valid transports */
		gtk_widget_destroy (menu);
		gtk_widget_hide (ts->menu);
		gtk_widget_hide (ts->filename_entry);
		gtk_widget_activate (ts->default_printer_rb);
		gtk_widget_hide (ts->custom_command_entry);
		gtk_widget_show (ts->printer_hbox);
		return;
	}

	gtk_widget_show (menu);
	gtk_option_menu_set_menu (GTK_OPTION_MENU (ts->menu), menu);
	gtk_option_menu_set_history (GTK_OPTION_MENU (ts->menu), def);

	if (defid) {
		gpa_transport_selector_select_transport (ts, defid);
		g_free (defid);
	}

	gtk_widget_set_sensitive (ts->menu, TRUE);
}

static void
gpa_transport_selector_menuitem_activate (GtkWidget *widget, gint index)
{
	GPAWidget *gpaw;
	GPANode *transport;
	gchar *value;

	gpaw = g_object_get_data (G_OBJECT (widget), "GPAWidget");
	g_return_if_fail (gpaw != NULL);
	g_return_if_fail (GPA_IS_WIDGET (gpaw));

	transport = g_slist_nth_data (GPA_TRANSPORT_SELECTOR (gpaw)->transportlist, index);
	g_return_if_fail (transport != NULL);
	g_return_if_fail (GPA_IS_NODE (transport));

	value = gpa_node_get_value (transport);
	g_return_if_fail (value != NULL);

	gpa_transport_selector_select_transport (GPA_TRANSPORT_SELECTOR (gpaw), value);

	g_free (value);
}

static void
gpa_transport_selector_select_transport (GPATransportSelector *ts, const gchar *id)
{
	GPAWidget *gpaw;

	gpaw = GPA_WIDGET (ts);

	gnome_print_config_set (gpaw->config, "Settings.Transport.Backend", id);

	if (!strcmp (id, "file")) {
		guchar *filename;
		/* Current transport is file */
		filename = gnome_print_config_get (gpaw->config, "Settings.Transport.Backend.FileName");
		if (filename) {
			gtk_entry_set_text (GTK_ENTRY (ts->filename_entry), filename);
			g_free (filename);
		} else {
			gtk_entry_set_text (GTK_ENTRY (ts->filename_entry), "gnome-print.out");
		}
		gtk_widget_show (ts->menu);
		gtk_widget_show (ts->filename_entry);
		gtk_widget_hide (ts->printer_hbox);
		gtk_widget_hide (ts->custom_command_entry);
	} else if (!strcmp (id, "lpr")) {
		guchar *lp;
		/* Current transport is lpr */
		lp = gnome_print_config_get (gpaw->config, "Settings.Transport.Backend.Printer");
		if (lp && *lp) {
			gtk_widget_activate (ts->custom_printer_rb);
			gtk_entry_set_text (GTK_ENTRY (ts->printer_entry), lp);
		} else {
			gtk_widget_activate (ts->default_printer_rb);
		}
		if (lp)
			g_free (lp);
		gtk_widget_show (ts->menu);
		gtk_widget_hide (ts->filename_entry);
		gtk_widget_show (ts->printer_hbox);
		gtk_widget_hide (ts->custom_command_entry);
	} else if (!strcmp (id, "custom")) {
		guchar *command;
		/* Current transport is a custom command */
		command = gnome_print_config_get (gpaw->config, "Settings.Transport.Backend.Command");
		if (command) {
			gtk_entry_set_text (GTK_ENTRY (ts->custom_command_entry), command);
			g_free (command);
		} else {
			gtk_entry_set_text (GTK_ENTRY (ts->custom_command_entry), "lpr");
		}
		gtk_widget_show (ts->menu);
		gtk_widget_hide (ts->filename_entry);
		gtk_widget_hide (ts->printer_hbox);
		gtk_widget_show (ts->custom_command_entry);
	} else {
		gtk_widget_show (ts->menu);
		gtk_widget_hide (ts->filename_entry);
		gtk_widget_hide (ts->printer_hbox);
		gtk_widget_hide (ts->custom_command_entry);
	}
}
