/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * gpa-option-menu.c:
 *
 * Libgnomeprint is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * Libgnomeprint is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with the libgnomeprint; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors :
 *   Chema Celorio <chema@ximian.com>
 *   Lauris Kaplinski <lauris@ximian.com>
 *
 * Copyright (C) 2001-2002 Ximian, Inc. 
 *
 */

#include <config.h>

#include <string.h>
#include <gtk/gtk.h>

#include "gpa-widget.h"
#include "gpa-option-menu.h"
#include "gnome-print-i18n.h"

static void gpa_option_menu_class_init (GPAOptionMenuClass *klass);
static void gpa_option_menu_init (GPAOptionMenu *om);
static void gpa_option_menu_finalize (GObject *object);

static GPAWidgetClass *parent_class;

GtkType
gpa_option_menu_get_type (void)
{
	static GType type = 0;
	if (!type) {
		static const GTypeInfo info = {
			sizeof (GPAOptionMenuClass),
			NULL, NULL,
			(GClassInitFunc) gpa_option_menu_class_init,
			NULL, NULL,
			sizeof (GPAOptionMenu),
			0,
			(GInstanceInitFunc) gpa_option_menu_init
		};
		type = g_type_register_static (GPA_TYPE_WIDGET, "GPAOptionMenu", &info, 0);
	}
	return type;
}

static void
gpa_option_menu_class_init (GPAOptionMenuClass *klass)
{
	GObjectClass *object_class;

	object_class = (GObjectClass *) klass;
	
	parent_class = gtk_type_class (GTK_TYPE_WIDGET);

	object_class->finalize = gpa_option_menu_finalize;
}

static void
gpa_option_menu_init (GPAOptionMenu *om)
{
	om->menu = gtk_option_menu_new ();
	gtk_container_add (GTK_CONTAINER (om), om->menu);
	gtk_widget_show (GTK_WIDGET (om->menu));
	
	om->key    = NULL;
	om->shell  = NULL;
	om->updating = FALSE;
	om->node   = NULL;
	om->handler = 0;
	om->parent = NULL;
	om->parent_handler = 0;
}

static void
gpa_option_menu_finalize (GObject *object)
{
	GPAOptionMenu *om;

	om = (GPAOptionMenu *) object;
	
	if (om->key)
		g_free (om->key);

	if (om->handler) {
		g_signal_handler_disconnect (om->node, om->handler);
		gpa_node_unref (om->node);
		om->handler = 0;
		om->node = NULL;
	}

	if (om->parent_handler) {
		g_signal_handler_disconnect (om->parent, om->parent_handler);
		gpa_node_unref (om->parent);
		om->parent_handler = 0;
		om->parent = NULL;
	}
	
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gpa_option_menu_item_activate_cb (GtkMenuItem *item, GPAOptionMenu *om)
{
	GPANode *parent;
	GPANode *node;

	if (om->updating) {
		g_print ("Avoid updating item_activated_cb\n");
		return;
	}

	om->updating = TRUE;
	
	parent = gpa_node_get_child_from_path (GNOME_PRINT_CONFIG_NODE (GPA_WIDGET (om)->config), om->key);
	node = g_object_get_data (G_OBJECT (item), "node");
	gpa_node_set_value (parent, gpa_node_id (node));

	om->updating = FALSE;
}

static void
gpa_option_menu_node_modified_cb (GPANode *node, guint flags, GPAOptionMenu *om)
{
	gchar *current;
	GPANode *option, *item;
	gint sel = 0, pos;

	if (om->updating)
		return;

	current = gpa_node_get_value (node);
	option = gpa_node_get_child_from_path (node, "Option");
	g_assert (option);
	g_assert (current);

	pos = 0;
	for (item = gpa_node_get_child (option, NULL); item != NULL; item = gpa_node_get_child (option, item)) {
		gchar *id;
		id = gpa_node_get_value (item);
		if (*id && strcmp (id, current) == 0) {
			sel = pos;
			g_free (id);
			break;
		}
		if (id)
			g_free (id);
		pos++;
		gpa_node_unref (item);
	}

	om->updating = TRUE;
	gtk_option_menu_set_history (GTK_OPTION_MENU (om->menu), sel);
	om->updating = FALSE;

	gpa_node_unref (option);
}

static void
gpa_option_menu_rebuild_menu (GPAOptionMenu *om)
{
	GtkWidget *menu_shell, *item;
	gint pos, sel;

	pos = 0;
	sel = 0;

	/* Remove our previous links */
	gtk_option_menu_remove_menu (GTK_OPTION_MENU (om->menu));
	if (om->handler) {
		g_signal_handler_disconnect (om->node, om->handler);
		om->handler = 0;
	}

	menu_shell = gtk_menu_new ();
	gtk_widget_show (menu_shell);

	om->node = gpa_node_get_child_from_path (GNOME_PRINT_CONFIG_NODE (GPA_WIDGET (om)->config), om->key);
	
	if (om->node) {
		GPANode *option;
		guchar *current;
		om->handler = g_signal_connect (G_OBJECT (om->node), "modified",
						(GCallback) gpa_option_menu_node_modified_cb, om);
		current = gpa_node_get_value (om->node);
		option = gpa_node_get_child_from_path (om->node, "Option");
		if (option) {
			GPANode *item;
			for (item = gpa_node_get_child (option, NULL); item != NULL; item = gpa_node_get_child (option, item)) {
				guchar *id, *name;
				id = gpa_node_get_value (item);
				name = gpa_node_get_path_value (item, "Name");
				if (id && *id && name && *name) {
					GtkWidget *i;
					gpa_node_ref (item);
					i = gtk_menu_item_new_with_label (name);
					g_object_set_data_full (G_OBJECT (i), "node", item,
								(GtkDestroyNotify) gpa_node_unref);
					g_signal_connect (G_OBJECT (i), "activate",
							  (GCallback) gpa_option_menu_item_activate_cb, om);
					gtk_widget_show (i);
					gtk_menu_shell_append (GTK_MENU_SHELL (menu_shell), i);
					if (current && !strcmp (id, current))
						sel = pos;
					pos += 1;
				}
				if (name)
					g_free (name);
				if (id)
					g_free (id);
				gpa_node_unref (item);
			}
			gpa_node_unref (option);
		}
		if (current)
			g_free (current);
	}

	if (pos < 1) {
		item = gtk_menu_item_new_with_label (_("No options are defined"));
		gtk_widget_set_sensitive (item, FALSE);
		gtk_widget_show (item);
		gtk_menu_shell_append (GTK_MENU_SHELL (menu_shell), item);
	}

	gtk_widget_show (menu_shell);
	gtk_option_menu_set_menu (GTK_OPTION_MENU (om->menu), menu_shell);
	gtk_option_menu_set_history (GTK_OPTION_MENU (om->menu), sel);
}

static void
gpa_option_menu_parent_modified_cb (GPANode *node, guint flags, GPAOptionMenu *om)
{
	if (om->updating) {
		g_print ("Avoid updating parent_modified_cb\n");
		return;
	}

	om->updating = TRUE;
	gpa_option_menu_rebuild_menu (om);
	om->updating = FALSE;
}

static gboolean
gpa_option_menu_construct (GPAOptionMenu *om, const guchar *key, GPANode *parent)
{
	g_return_val_if_fail (GPA_IS_NODE (parent), FALSE);

	om->key = g_strdup (key);

	gpa_option_menu_rebuild_menu (om);

	om->parent = gpa_node_ref (parent);
	om->parent_handler = g_signal_connect (G_OBJECT (parent), "modified",
					       (GCallback) gpa_option_menu_parent_modified_cb, om);

	return TRUE;
}

GtkWidget *
gpa_option_menu_new (GnomePrintConfig *config, const guchar *key, GPANode *parent)
{
	GtkWidget *gpa_menu;

	g_return_val_if_fail (config != NULL, NULL);
	g_return_val_if_fail (key != NULL, NULL);
	g_return_val_if_fail (parent != NULL, NULL);

	gpa_menu = gpa_widget_new (GPA_TYPE_OPTION_MENU, config);

	gpa_option_menu_construct (GPA_OPTION_MENU (gpa_menu), key, parent);

	gtk_widget_show (gpa_menu);

	return gpa_menu;
}


