/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import org.gnu.gdk.Atom;
import org.gnu.gdk.Pixbuf;
import org.gnu.glib.GObject;
import org.gnu.glib.Handle;


/**
 * Data used in Drag and drop operations
 */
public class SelectionData extends GObject
{
	public SelectionData( Handle handle ){
		super(handle);
	}

	public Atom getSelection(){
		return new Atom( getSelection(getHandle()) );
	}

	public Atom getTarget(){
		return new Atom( getTarget(getHandle()) );
	}
	
	public Atom getType(){
		return new Atom( getType(getHandle()) );
	}

	public int getFormat(){
		return getFormat(getHandle());
	}

	public int getLength(){
		return getLength( getHandle() );
	}
	
	public String getText(){
		return gtk_selection_data_get_text( getHandle() );
	}
	public boolean setText( String text ){
		return gtk_selection_data_set_text( getHandle(), text );
	}
	
    /**
     * Given that this SelectionData object holds a list of targets, 
     * determines if any of the targets in targets can be used to provide text.
     *
     * @return TRUE if this SelectionData holds a list of targets, and a 
     * suitable target for text is included, otherwise FALSE.
     */
    public boolean targetsIncludText() {
        return gtk_selection_data_targets_include_text(getHandle());
    }

    /**
     * Given that this SelectionData object holds a list of targets,
     * determines if any of the targets in targets can be used to
     * provide a {@link org.gnu.gdk.Pixbuf}.
     *
     * @param writable Whether to accept only targets for which GTK+ knows 
     * how to convert a pixbuf into the format.
     * @return TRUE if this SelectionData holds a list of targets, and a 
     * suitable target for images is included, otherwise FALSE.
     */
    public boolean targetsIncludeImage( boolean writable ) {
        return gtk_selection_data_targets_include_image( getHandle(), writable );
    }

	public Atom[] getTargets() {
	    Handle hndls[] = gtk_selection_data_get_targets(getHandle());
		if (null == hndls)
			return null;
		Atom[] value = new Atom[hndls.length];
		for (int i = 0; i < hndls.length; i++) {
			value[i] = new Atom(hndls[i]);
		}
		return value;
	}
	
	public boolean setPixbuf(Pixbuf pixbuf) {
	    return gtk_selection_data_set_pixbuf(getHandle(), pixbuf.getHandle());
	}
	
	public Pixbuf getPixbuf() {
	    return new Pixbuf(gtk_selection_data_get_pixbuf(getHandle()));
	}
	
	public boolean setUris(String[] uris) {
	    return gtk_selection_data_set_uris(getHandle(), uris);
	}
	
	public String[] getUris() {
	    return gtk_selection_data_get_uris(getHandle());
	}
	

	native static protected final Handle getSelection( Handle selectionData );
	native static protected final Handle getTarget( Handle selectionData );
	native static protected final Handle getType( Handle selectionData );
	native static protected final int getFormat( Handle selectionData );
	native static protected final String getData( Handle selectionData );
	native static protected final int getLength( Handle selectionData );
	native static protected final Handle getDisplay( Handle selectionData );
	native static protected final String  gtk_selection_data_get_text( Handle selectionData );
	native static protected final boolean gtk_selection_data_set_text( Handle selectionData, String str );
	native static protected final Handle[] gtk_selection_data_get_targets(Handle selectionData);
	native static protected final boolean gtk_selection_data_targets_include_text(Handle selectionData);
    native static final private boolean gtk_selection_data_targets_include_image(Handle selection_data, boolean writable);
	native static protected final boolean gtk_selection_data_set_pixbuf(Handle selectionData, Handle pixbuf); 
	native static protected final Handle gtk_selection_data_get_pixbuf(Handle selectionData);
	native static protected final boolean gtk_selection_data_set_uris(Handle selectionData, String[] uris);
	native static protected final String[] gtk_selection_data_get_uris(Handle selectionData);

}

