/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import org.gnu.glib.Type;
import org.gnu.javagnome.Handle;

/**
 * This is a container widget that is capable of controlling the size
 * of a single child widget. The scale values are from 0.0 to 1.0,
 * indicating the maximum amount the child can expand to fill the space
 * allocated to the Alignment widget. The align values determine the
 * x and y positions relative to the top left and bottom right corners
 * of the Alignment rectangle. The align values are from 0.0 to the
 * top or left side, and 1.0 for the bottom or right side.
 */
public class Alignment extends Bin {
	
	/**
	 * Construct a new Alignment.
	 * @param xAlign The horizontal alignment of the child widget from 0 (left)
	 * to 1 (right).
	 * @param yAlign The vertical alignment of the child widget from 0 (top)
	 * to 1 (bottom).
	 * @param xScale The amount that the child widget expands horizontally to
	 * fill up unused space, from 0 to 1.  A value of 0 indicates that the child
	 * widget should never expand.  A value of 1 indicates that the child widget
	 * will expand to fill all of the space allocated for the Alignment.
	 * @param yScale The amount that the child widget expands vertically to
	 * fill up unused space, from 0 to 1.  The values are similar to xScale.
	 */
	public Alignment(double xAlign, double yAlign, double xScale, double yScale) {
		super(gtk_alignment_new(xAlign, yAlign, xScale, yScale));
	}
	
	/**
	 * Construct an alignment using a handle to a native resource.
	 */
	public Alignment(Handle handle) {
	    super(handle);
	}

	/**
	 * Set the alignment and scale for an already created Alignment object.
	 * @param xAlign The horizontal alignment of the child widget from 0 (left)
	 * to 1 (right).
	 * @param yAlign The vertical alignment of the child widget from 0 (top)
	 * to 1 (bottom).
	 * @param xScale The amount that the child widget expands horizontally to
	 * fill up unused space, from 0 to 1.  A value of 0 indicates that the child
	 * widget should never expand.  A value of 1 indicates that the child widget
	 * will expand to fill all of the space allocated for the Alignment.
	 * @param yScale The amount that the child widget expands vertically to
	 * fill up unused space, from 0 to 1.  The values are similar to xScale.
	 */
	public void set(double xAlign, double yAlign, double xScale, double yScale) {
		Alignment.gtk_alignment_set(getHandle(), xAlign, yAlign, xScale, yScale);
	}

	/**
	 * Sets the padding on the different sides of the widget. The padding adds
	 * blank space to the sides of the widget. For instance, this can be used to
	 * indent the child widget towards the right by adding padding on the left.
	 * 
	 * @param top the padding at the top of the widget
	 * @param bottom the padding at the bottom of the widget
	 * @param left the padding at the left of the widget
	 * @param right	the padding at the right of the widget.
	 */
	public void setPadding(int top, int bottom, int left, int right){
		gtk_alignment_set_padding(getHandle(), top, bottom, left, right);
	}
	
	/**
	 * Gets the size of the padding at the top of the widget.
	 * @return The padding at the top of the widget.
	 */
	public int getTopPadding() {
		int[] top = new int[1];
		int[] bottom = new int[1];
		int[] left = new int[1];
		int[] right = new int[1];
		gtk_alignment_get_padding(getHandle(), top, bottom, left, right);
		return top[0];
	}
	
	/**
	 * Gets the size of the padding at the bottom of the widget.
	 * @return The padding at the bottom of the widget.
	 */
	public int getBottomPadding() {
		int[] top = new int[1];
		int[] bottom = new int[1];
		int[] left = new int[1];
		int[] right = new int[1];
		gtk_alignment_get_padding(getHandle(), top, bottom, left, right);
		return bottom[0];
	}
	
	/**
	 * Gets the size of the padding at the left of the widget.
	 * @return The padding at the left of the widget.
	 */
	public int getLeftPadding() {
		int[] top = new int[1];
		int[] bottom = new int[1];
		int[] left = new int[1];
		int[] right = new int[1];
		gtk_alignment_get_padding(getHandle(), top, bottom, left, right);
		return left[0];
	}
	
	/**
	 * Gets the size of the padding at the right of the widget.
	 * @return The padding at the right of the widget.
	 */
	public int getRightPadding() {
		int[] top = new int[1];
		int[] bottom = new int[1];
		int[] left = new int[1];
		int[] right = new int[1];
		gtk_alignment_get_padding(getHandle(), top, bottom, left, right);
		return right[0];
	}
	
	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gtk_alignment_get_type());
	}
	

	native static final protected int gtk_alignment_get_type();
	native static final protected Handle gtk_alignment_new(double xalign, double yalign, double xscale, double yscale);
	native static final protected void gtk_alignment_set(Handle alignment, double xalign, double yalign, double xscale, double yscale);
	native static final protected void gtk_alignment_set_padding(Handle alignment, int top, int bottom, int left, int right);
	native static final protected void gtk_alignment_get_padding(Handle alignment, int[] top, int[] bottom, int[] left, int[] right);

}
