/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * This library is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libical-glib/i-cal-timezone.h>
#include <libical-glib/i-cal-component.h>
#include <libical-glib/i-cal-array.h>
#include <libical-glib/i-cal-timetype.h>

G_DEFINE_TYPE (ICalTimezone, i_cal_timezone, I_CAL_TYPE_OBJECT)

static void
i_cal_timezone_class_init (ICalTimezoneClass *klass)
{	
	g_type_class_add_private (klass, sizeof (ICalTimezone));
}

static void
i_cal_timezone_init (ICalTimezone *self)
{
}

/**
 * i_cal_timezone_new_full: (skip)
 * @native: The native libical object.
 * @owner: The parent.
 * @is_global_memory: Whether it is allocated in the global memory.
 *
 * Create a new libical-glib object from the native libical object and the owner.
 * 
 * Returns: (transfer full): The newly create libical-glib object.
 *
 * Since: 1.0
 **/
ICalTimezone *
i_cal_timezone_new_full (icaltimezone *native, 
                         GObject *owner, 
                         gboolean is_global_memory)
{
	ICalTimezone *object;
	if (native == NULL)
		return NULL;
	object = g_object_new (I_CAL_TIMEZONE_TYPE, NULL);
	i_cal_object_construct ((ICalObject *)object,
		        	(gpointer) native,
		        	(GDestroyNotify) i_cal_timezone_destroy,
		        	is_global_memory,
		        	owner);
    							
	return object;
}

/**
 * i_cal_timezone_new:
 *
 * The constructor of the type #ICalTimezone
 *
 * Returns: (transfer full) (allow-none): The newly created object of the type #ICalTimezone.
 *
 * Since: 1.0
 *
 **/
ICalTimezone *
i_cal_timezone_new (void)
{
	return i_cal_timezone_new_full (icaltimezone_new (), NULL, FALSE) ;
}

/**
 * i_cal_timezone_destroy: (skip)
 * @zone: The #ICalTimezone to be freed
 *
 * The destructor of the type #ICalTimezone
 *
 * Since: 1.0
 *
 **/
void
i_cal_timezone_destroy (icaltimezone *zone)
{
	icaltimezone_free (zone, 1);
}

/**
 * i_cal_timezone_copy:
 * @zone: The #ICalTimezone needs to be cloned.
 *
 * The clone method for #ICalTimezone
 *
 * Returns: (transfer none): The newly created #ICalTimezone with the same values as @zone
 *
 * Since: 1.0
 *
 **/
ICalTimezone *
i_cal_timezone_copy (const ICalTimezone *zone)
{
	g_return_val_if_fail (I_CAL_IS_TIMEZONE (zone), NULL);
	g_return_val_if_fail (zone != NULL, NULL);

	return i_cal_timezone_new_full (icaltimezone_copy ((icaltimezone *)i_cal_object_get_native (I_CAL_OBJECT (zone))), NULL, FALSE) ;
}

/**
 * i_cal_timezone_free: (skip)
 * @zone: The #ICalTimezone to be freed
 * @free_struct: Whether to free it or reset it. 1 to free and 0 to reset
 *
 * The destructor of #ICalTimezone
 *
 * Since: 1.0
 *
 **/
void
i_cal_timezone_free (ICalTimezone *zone, 
                     gint free_struct)
{
	g_return_if_fail (I_CAL_IS_TIMEZONE (zone));
	g_return_if_fail (zone != NULL);

	icaltimezone_free ((icaltimezone *)i_cal_object_get_native (I_CAL_OBJECT (zone)), free_struct);
}

/**
 * i_cal_timezone_set_tzid_prefix:
 * @new_prefix: The #ICalTimezone to be set
 *
 * Sets the prefix to be used for tzid's generated from system tzdata. Must be globally unique (such as
 * a domain name owned by the developer of the calling application), and begin and end with forward slashes.
 * Do not change or de-allocate the string buffer after calling this.
 *
 * Since: 1.0
 *
 **/
void
i_cal_timezone_set_tzid_prefix (const gchar *new_prefix)
{
	g_return_if_fail (new_prefix != NULL);

	icaltimezone_set_tzid_prefix (new_prefix);
}

/**
 * i_cal_timezone_free_builtin_timezones:
 *
 * Free any builtin timezone information
 *
 * Since: 1.0
 *
 **/
void
i_cal_timezone_free_builtin_timezones (void)
{
	icaltimezone_free_builtin_timezones ();
}

/**
 * i_cal_timezone_get_builtin_timezone:
 * @location: The location representing the timezone.
 *
 * Returns a single builtin timezone, given its Olson city name.
 *
 * Returns: (transfer full): The builtin #ICalTimezone with the name of @location
 *
 * Since: 1.0
 *
 **/
ICalTimezone *
i_cal_timezone_get_builtin_timezone (const gchar *location)
{
	g_return_val_if_fail (location != NULL, NULL);

	return i_cal_timezone_new_full (icaltimezone_get_builtin_timezone (location), NULL, TRUE) ;
}

/**
 * i_cal_timezone_get_builtin_timezone_from_offset:
 * @offset: The offset used to get the #ICalTimezone
 * @tzname: The reference #ICalTimezone name
 *
 * Returns a single builtin timezone, given its offset.
 *
 * Returns: (transfer full)
 *
 * Since: 1.0
 *
 **/
ICalTimezone *
i_cal_timezone_get_builtin_timezone_from_offset (gint offset, 
                                                 const gchar *tzname)
{
	g_return_val_if_fail (tzname != NULL, NULL);

	return i_cal_timezone_new_full (icaltimezone_get_builtin_timezone_from_offset (offset, tzname), NULL, TRUE) ;
}

/**
 * i_cal_timezone_get_builtin_timezone_from_tzid:
 * @tzid: The tzid name
 *
 * Returns a single builtin timezone, given its TZID.
 *
 * Returns: (transfer full)
 *
 * Since: 1.0
 *
 **/
ICalTimezone *
i_cal_timezone_get_builtin_timezone_from_tzid (const gchar *tzid)
{
	g_return_val_if_fail (tzid != NULL, NULL);

	return i_cal_timezone_new_full (icaltimezone_get_builtin_timezone_from_tzid (tzid), NULL, TRUE) ;
}

/**
 * i_cal_timezone_get_builtin_timezones:
 *
 * Returns a list of builtin timezone.
 *
 * Returns: (transfer full): The builtin #ICalTimezone.
 *
 * Since: 1.0
 *
 **/
ICalArray *
i_cal_timezone_get_builtin_timezones (void)
{
	return i_cal_array_new_full (icaltimezone_get_builtin_timezones (), NULL) ;
}

/**
 * i_cal_timezone_get_utc_timezone:
 *
 * Returns the UTC timezone.
 *
 * Returns: (transfer full): The utc #ICalTimezone
 *
 * Since: 1.0
 *
 **/
ICalTimezone *
i_cal_timezone_get_utc_timezone (void)
{
	return i_cal_timezone_new_full (icaltimezone_get_utc_timezone (), NULL, TRUE) ;
}

/**
 * i_cal_timezone_get_tzid:
 * @zone: The #ICalTimezone to be queried
 *
 * Returns the TZID of a timezone.
 *
 * Returns: The timezone id
 *
 * Since: 1.0
 *
 **/
const gchar *
i_cal_timezone_get_tzid (const ICalTimezone *zone)
{
	g_return_val_if_fail (I_CAL_IS_TIMEZONE (zone), NULL);
	g_return_val_if_fail (zone != NULL, NULL);

	return icaltimezone_get_tzid ((icaltimezone *)i_cal_object_get_native (I_CAL_OBJECT (zone)));
}

/**
 * i_cal_timezone_get_location:
 * @zone: The #ICalTimezone to be queried
 *
 * Returns the city name of a timezone.
 *
 * Returns: The location of the #ICalTimezone
 *
 * Since: 1.0
 *
 **/
const gchar *
i_cal_timezone_get_location (const ICalTimezone *zone)
{
	g_return_val_if_fail (I_CAL_IS_TIMEZONE (zone), NULL);
	g_return_val_if_fail (zone != NULL, NULL);

	return icaltimezone_get_location ((icaltimezone *)i_cal_object_get_native (I_CAL_OBJECT (zone)));
}

/**
 * i_cal_timezone_get_tznames:
 * @zone: The #ICalTimezone to be queried
 *
 * Returns the TZNAME properties used in the latest STANDARD and DAYLIGHT components. If they are the same
 * it will return just one, e.g. "LMT". If they are different it will format them like "EST/EDT". Note that
 * this may also return NULL.
 *
 * Returns: The timezone name
 *
 * Since: 1.0
 *
 **/
const gchar *
i_cal_timezone_get_tznames (const ICalTimezone *zone)
{
	g_return_val_if_fail (I_CAL_IS_TIMEZONE (zone), NULL);
	g_return_val_if_fail (zone != NULL, NULL);

	return icaltimezone_get_tznames ((icaltimezone *)i_cal_object_get_native (I_CAL_OBJECT (zone)));
}

/**
 * i_cal_timezone_get_latitude:
 * @zone: The #ICalTimezone to be queried
 *
 * Returns the latitude of a builtin timezone.
 *
 * Returns: The latitude of the #ICalTimezone
 *
 * Since: 1.0
 *
 **/
gdouble
i_cal_timezone_get_latitude (const ICalTimezone *zone)
{
	g_return_val_if_fail (I_CAL_IS_TIMEZONE (zone), 0);
	g_return_val_if_fail (zone != NULL, 0);

	return icaltimezone_get_latitude ((icaltimezone *)i_cal_object_get_native (I_CAL_OBJECT (zone)));
}

/**
 * i_cal_timezone_get_longitude:
 * @zone: The #ICalTimezone to be queried
 *
 * Returns the longitude of a builtin timezone.
 *
 * Returns: The longitude of the #ICalTimezone.
 *
 * Since: 1.0
 *
 **/
gdouble
i_cal_timezone_get_longitude (const ICalTimezone *zone)
{
	g_return_val_if_fail (I_CAL_IS_TIMEZONE (zone), 0);
	g_return_val_if_fail (zone != NULL, 0);

	return icaltimezone_get_longitude ((icaltimezone *)i_cal_object_get_native (I_CAL_OBJECT (zone)));
}

/**
 * i_cal_timezone_get_component:
 * @zone: The #ICalTimezone to be queried
 *
 * Returns the VTIMEZONE component of a timezone.
 *
 * Returns: (transfer full): the VTIMEZONE component of the @zone.
 *
 * Since: 1.0
 *
 **/
ICalComponent *
i_cal_timezone_get_component (const ICalTimezone *zone)
{
	g_return_val_if_fail (I_CAL_IS_TIMEZONE (zone), NULL);
	g_return_val_if_fail (zone != NULL, NULL);

	return i_cal_component_new_full (icaltimezone_get_component ((icaltimezone *)i_cal_object_get_native (I_CAL_OBJECT (zone))), (GObject *)zone) ;
}

/**
 * i_cal_timezone_set_component:
 * @zone: The #ICalTimezone to be set
 * @comp: The VTIMEZONE component of an #ICalTimezone, initializing the tzid, location and tzname fields.
 *
 * Sets the VTIMEZONE component of #ICalTimezone, initializing the tzid, location and tzname fields. It
 * returns 1 on success or 0 on failure, i.e. no TZID was found.
 *
 * Returns: Whether the action is successful. 1 for success, 0 for failure.
 *
 * Since: 1.0
 *
 **/
gint
i_cal_timezone_set_component (ICalTimezone *zone, 
                              ICalComponent *comp)
{
	g_return_val_if_fail (I_CAL_IS_TIMEZONE (zone), 0);
	g_return_val_if_fail (zone != NULL, 0);
	g_return_val_if_fail (I_CAL_IS_COMPONENT (comp), 0);
	g_return_val_if_fail (comp != NULL, 0);

	return icaltimezone_set_component ((icaltimezone *)i_cal_object_get_native (I_CAL_OBJECT (zone)), (icalcomponent *)i_cal_object_get_native (I_CAL_OBJECT (comp)));
}

/**
 * i_cal_timezone_get_display_name:
 * @zone: The #ICalTimezone to be queried
 *
 * Get the display name of the @zone.
 *
 * Returns: The display name of @zone
 *
 * Since: 1.0
 *
 **/
const gchar *
i_cal_timezone_get_display_name (const ICalTimezone *zone)
{
	g_return_val_if_fail (I_CAL_IS_TIMEZONE (zone), NULL);
	g_return_val_if_fail (zone != NULL, NULL);

	return icaltimezone_get_display_name ((icaltimezone *)i_cal_object_get_native (I_CAL_OBJECT (zone)));
}

/**
 * i_cal_timezone_convert_time:
 * @tt: The time to be converted
 * @from_zone: from timezone
 * @to_zone: to timezone
 *
 * Convert time from one timezone to another.
 *
 * Since: 1.0
 *
 **/
void
i_cal_timezone_convert_time (ICalTimetype *tt, 
                             ICalTimezone *from_zone, 
                             ICalTimezone *to_zone)
{
	g_return_if_fail (I_CAL_IS_TIMETYPE (tt));
	g_return_if_fail (tt != NULL);
	g_return_if_fail (I_CAL_IS_TIMEZONE (from_zone));
	g_return_if_fail (from_zone != NULL);
	g_return_if_fail (I_CAL_IS_TIMEZONE (to_zone));
	g_return_if_fail (to_zone != NULL);

	icaltimezone_convert_time ((struct icaltimetype *)i_cal_object_get_native (I_CAL_OBJECT (tt)), (icaltimezone *)i_cal_object_get_native (I_CAL_OBJECT (from_zone)), (icaltimezone *)i_cal_object_get_native (I_CAL_OBJECT (to_zone)));
}

/**
 * i_cal_timezone_get_utc_offset:
 * @zone: The given #ICalTimezone.
 * @tt: The local time.
 * @is_daylight: whether it is day light.
 *
 * Calculates the UTC offset of a given local time in the given timezone.  It is the number of seconds to
 * add to UTC to get local time.  The is_daylight flag is set to 1 if the time is in daylight-savings time.
 *
 * Since: 1.0
 *
 **/
void
i_cal_timezone_get_utc_offset (ICalTimezone *zone, 
                               ICalTimetype *tt, 
                               gint *is_daylight)
{
	g_return_if_fail (I_CAL_IS_TIMEZONE (zone));
	g_return_if_fail (zone != NULL);
	g_return_if_fail (I_CAL_IS_TIMETYPE (tt));
	g_return_if_fail (tt != NULL);
	g_return_if_fail (is_daylight != NULL);

	icaltimezone_get_utc_offset ((icaltimezone *)i_cal_object_get_native (I_CAL_OBJECT (zone)), (struct icaltimetype *)i_cal_object_get_native (I_CAL_OBJECT (tt)), is_daylight);
}

/**
 * i_cal_timezone_get_utc_offset_of_utc_time:
 * @zone: The given #ICalTimezone.
 * @tt: The local time.
 * @is_daylight: whether it is day light.
 *
 * Calculates the UTC offset of a given UTC time in the given timezone.  It is the number of seconds to
 * add to UTC to get local time.  The is_daylight flag is set to 1 if the time is in daylight-savings time.
 *
 * Since: 1.0
 *
 **/
void
i_cal_timezone_get_utc_offset_of_utc_time (ICalTimezone *zone, 
                                           ICalTimetype *tt, 
                                           gint *is_daylight)
{
	g_return_if_fail (I_CAL_IS_TIMEZONE (zone));
	g_return_if_fail (zone != NULL);
	g_return_if_fail (I_CAL_IS_TIMETYPE (tt));
	g_return_if_fail (tt != NULL);
	g_return_if_fail (is_daylight != NULL);

	icaltimezone_get_utc_offset_of_utc_time ((icaltimezone *)i_cal_object_get_native (I_CAL_OBJECT (zone)), (struct icaltimetype *)i_cal_object_get_native (I_CAL_OBJECT (tt)), is_daylight);
}

/**
 * i_cal_timezone_array_new:
 *
 * Create a new array of timezones.
 *
 * Returns: (transfer full): Create a new array.
 *
 * Since: 1.0
 *
 **/
ICalArray *
i_cal_timezone_array_new (void)
{
	return i_cal_array_new_full (icaltimezone_array_new (), NULL) ;
}

/**
 * i_cal_timezone_array_append_from_vtimezone:
 * @timezones: The timezones to be populated
 * @child: The component to be appended to @timezones.
 *
 * Populate the array of timezones with component.
 *
 * Since: 1.0
 *
 **/
void
i_cal_timezone_array_append_from_vtimezone (ICalArray *timezones, 
                                            ICalComponent *child)
{
	g_return_if_fail (I_CAL_IS_ARRAY (timezones));
	g_return_if_fail (timezones != NULL);
	g_return_if_fail (I_CAL_IS_COMPONENT (child));
	g_return_if_fail (child != NULL);

	icaltimezone_array_append_from_vtimezone ((icalarray *)i_cal_object_get_native (I_CAL_OBJECT (timezones)), (icalcomponent *)i_cal_object_get_native (I_CAL_OBJECT (child)));
}

/**
 * i_cal_timezone_array_free: (skip)
 * @timezones: Free an array.
 *
 * Free an array of timezones.
 *
 * Since: 1.0
 *
 **/
void
i_cal_timezone_array_free (ICalArray *timezones)
{
	g_return_if_fail (I_CAL_IS_ARRAY (timezones));
	g_return_if_fail (timezones != NULL);

	icaltimezone_array_free ((icalarray *)i_cal_object_steal_native (I_CAL_OBJECT (timezones)));
}

/**
 * i_cal_timezone_set_zone_directory:
 * @path: The path to look for the zonefiles
 *
 * Set the directory to look for the zonefiles
 *
 * Since: 1.0
 *
 **/
void
i_cal_timezone_set_zone_directory (gchar *path)
{
	g_return_if_fail (path != NULL);

	set_zone_directory (path);
}

/**
 * i_cal_timezone_free_zone_directory:
 *
 * Free memory dedicated to the zonefile directory
 *
 * Since: 1.0
 *
 **/
void
i_cal_timezone_free_zone_directory (void)
{
	free_zone_directory ();
}

/**
 * i_cal_timezone_release_zone_tab:
 *
 * Free memory dedicated to the zonefile directory
 *
 * Since: 1.0
 *
 **/
void
i_cal_timezone_release_zone_tab (void)
{
	icaltimezone_release_zone_tab ();
}

/**
 * i_cal_timezone_dump_changes:
 * @zone: The timezone to be queried.
 * @max_year: max year
 * @fp: The file handle.
 *
 * This outputs a list of timezone changes for the given timezone to the given file, up to the maximum year
 * given.
 *
 * Returns: 1 if success.
 *
 * Since: 1.0
 *
 **/
gint
i_cal_timezone_dump_changes (ICalTimezone *zone, 
                             gint max_year, 
                             FILE *fp)
{
	g_return_val_if_fail (I_CAL_IS_TIMEZONE (zone), 0);
	g_return_val_if_fail (zone != NULL, 0);
	g_return_val_if_fail (fp != NULL, 0);

	return icaltimezone_dump_changes ((icaltimezone *)i_cal_object_get_native (I_CAL_OBJECT (zone)), max_year, fp);
}

/**
 * i_cal_timezone_array_element_at:
 * @timezones: The array to be visited.
 * @index: The index
 *
 * Get the #ICalTimezone at specified position in array.
 *
 * Returns: (transfer full): The #ICalTimezone at the position @index in @timezones.
 *
 * Since: 1.0
 *
 **/
ICalTimezone *
i_cal_timezone_array_element_at (ICalArray *timezones, 
                                 guint index)
{
	return i_cal_timezone_new_full ((gpointer)i_cal_array_element_at (timezones, index), (GObject *)timezones, FALSE);
}