'''
Defines an L{AEEvent} indicating a gesture has been performed on an L{AEInput} 
device.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import Base, Constants

class InputGesture(Base.AccessEngineEvent):
  '''  
  Event that fires when some L{AEInput.Gesture} is detected on an L{AEInput}
  device. Always defaults to the L{Constants.FOCUS_LAYER}.
  
  This class registers its name and whether it should be monitored by default in
  an L{AEMonitor.EventMonitor} using the L{Constants.registerEventType} function
  when this module is first imported. The L{AEMonitor.EventMonitor} can use this
  information to build its menus.
  
  @ivar gesture: Gesture detected
  @type gesture: L{AEInput.Gesture}
  '''
  Constants.registerEventType('InputGesture', False)
  def __init__(self, gesture, **kwargs):
    '''
    Calls the base class (which sets the event priority) and then stores the 
    L{AEInput.GestureList} that will be used to trigger a L{Task}.
    
    @param gesture: Gestures detected on an L{AEInput} device
    @type gesture: L{AEInput.Gesture}
    '''
    Base.AccessEngineEvent.__init__(self, focused=True, **kwargs)
    self.gesture = gesture
    
  def __str__(self):
    '''
    Returns a human readable representation of this event including its name,
    its gesture codes, and its device.
    
    @return: Information about this event
    @rtype: string
    '''
    name = Base.AccessEngineEvent.__str__(self)
    return '%s:\n\tgesture: %s\n\tdevice: %s' % \
           (name, self.gesture, self.gesture.getDevice().getName())

  def execute(self, tier_manager, **kwargs):
    '''
    Contacts the L{TierManager} and asks it to manage this event as a gesture.
    
    @param tier_manager: TierManager that will handle the event
    @type tier_manager: L{TierManager}
    @param kwargs: Packed references to other managers not of interest here
    @type kwargs: dictionary
    @return: True to indicate the event executed properly
    @rtype: boolean
    '''
    tier_manager.manageGesture(self)
    return True
  
  def getGesture(self):
    '''
    Gets the L{AEInput.Gesture} that triggered this event.
    
    @return: Gesture seen on an L{AEInput} device
    @rtype: L{AEInput.Gesture}
    '''
    return self.gesture
  
  def getDataForTask(self):
    '''
    Provides no information to the L{Task} about this event.
    
    @return: Empty dict
    @rtype: dictionary
    '''
    return {}