'''
Defines the base class for all L{AEOutput} devices.

@author: Larry Weiss
@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import Constants

class AEOutput(object):
  '''
  Defines the base class for all output devices. All output devices used by
  LSR should derive from this class and should implement the methods defined
  here.

  All methods defined here raise L{NotImplementedError} to ensure that
  derived classes create appropriate implementions.
  '''

  def init(self):
    '''
    Called after the instance is created to initialize the device.

    If called when already initialized, this will restore the device to it's
    initialized state. May also be called to re-initialize the device after
    a close if output was suspended.

    @raise NotImplementedError: When not overridden in a subclass.
    @raise Base.InitError: When a communication or state problem exists for 
      the specific device.
    '''
    raise NotImplementedError

  def close(self):
    '''
    Closes an initialized output device.

    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError

  def getName(self):
    '''
    Gives the user displayable (localized) name for this output device.
    Relevant version and device status should be included.

    @return: The localized name for the device
    @rtype: string
    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError

  def getCommandChars(self):
    '''
    Returns a string of characters that may be interpreted as a command if
    sent to the device.

    @return: The string of characters that may be interpreted as a command.
    @rtype: string
    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError

  def getCommandName(self, cmd, value):
    '''
    Gets the user displayable (localized) name for the specified command at
    the specified value.

    @param cmd: The device specific command to query
    @type cmd: integer
    @param value: The current value of the specified command
    @type value: float
    @return: The localized name for the command at the specified value
    @rtype: string
    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError

  def getCommandLevel(self, cmd):
    '''
    Gets the current value of the specified command.

    The default value should be returned when it has not been changed by
    L{sendCommand}. A return of -1 indicates that the specified command is not
    implemented. A return of 0 indicates the the specified command has only 
    a single value.

    @param cmd: The device specific command to query
    @type cmd: integer
    @return: The current value of the specified command
    @rtype: object
    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError

  def getLevelCount(self, cmd):
    ''' 
    Gets the maximum settable value for the specified command. A return of 0 or
    1 indicates that a single value is available.

    @param cmd: The device specific command to query
    @type cmd: integer
    @return: Maximum and minimum settable level
    @rtype: 2-tuple of integer
    @raise NotImplementedError: When not overridden in a subclass.
    @see: L{sendCommand}
    '''
    raise NotImplementedError

  def sendCommand(self, cmd, value):
    '''
    Sends the specified command with the specified value to the device.

    Depending on the device, the command specified, and the features of the
    device, this command may be buffered until L{sendTalk} has been called.
    When the first parameter is valid and has only a single value, the second
    parameter is ignored. When the command specified is invalid or not
    implemented, this method will raise an exception. When the value is outside
    the valid range, the nearest valid value will be substituted.

    This only impacts content sent after this command. Text sent prior to this
    command should not be modified by this command.

    @param cmd: The device specific command to send.
    @type cmd: integer
    @param value: The value of the specified command to send.
    @type value: float
    @raise NotImplementedError: When not overridden in a subclass.
    @see: L{getLevelCount}
    '''
    raise NotImplementedError

  def sendString(self, string):
    '''
    Sends a string of one or more characters to the device.

    Depending on the device and it's features, the character may first be
    buffered until either "enough" characters and commands have been
    received, or L{sendTalk} has been called. To guard against sending
    unintended commands, the caller can use L{getCommandChars} for a string of
    characters to avoid.

    @param string: The string to send to the device
    @type string: string
    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError

  def sendStop(self):
    '''
    Purges buffered text and commands, and interrupts current output.

    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError

  def sendTalk(self):
    '''
    Indicates to the output implementation that buffered content should be
    presented to the user. For speech, this may be sending some of the
    buffered content and sending a command. For Braille, this may "packaging" a
    portion of the buffered content for the display.

    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError

  def sendStringSync(self, string, stop=True):
    '''
    Sends a complete string to a device followed by a talk command to speak it
    immediately. Blocks until the string has finished being spoken.
    
    Do B{not} use this method in place of sendString for normal device 
    operation. It is intended as a method for doing device output pre-runtime
    or when debugging.
    
    @param string: The string to send to the device
    @type string: string
    @param stop: True by default to interrupt current output
    @type stop: boolean
    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError

  def setLocale(self, loc):
    '''
    Sets the locale used for presenting content to the user.

    When the exact locale provided is not supported, this will attempt to set
    an available locale based on the language. And will use the device default
    when the language is not available. This returns the exact locale set.

    Depending on the device, this should also modify the text content
    presented to the user. For Braille devices, this can change which table is
    used. For speech devices, this can change which language rules are used to
    pronounce the received text.

    When a supported locale is specified, the strings returned from
    L{getCommandName} will reflect this command.

    @param loc: The name of the locale to set
    @type loc: string
    @return: The locale actually set
    @rtype: string
    @raise NotImplementedError: When not overridden in a subclass
    @see: L{getLocale}
    '''
    raise NotImplementedError

  def getLocale(self):
    '''
    Returns the current locale in use for the output device.

    @return: The name of the current locale
    @rtype: string
    @raise NotImplementedError: When not overridden in a subclass
    @see: L{setLocale}
    '''
    raise NotImplementedError