'''
Defines a class responsible for managing all input and output devices and
monitors.

@author: Larry Weiss
@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import logging, sys
import AEOutput, AEInput, AEEvent
from LSRInterfaces import implements
from UIRegistrar import UIRegistrar, DEVICE, MONITOR, STARTUP
from i18n import _

log = logging.getLogger('Device')

class DeviceManager(object):
  '''
  Creates and manages the devices and monitors use by LSR. Keeps a list of each
  and also defines a "default" for devices. Provides a common interface to all
  output and input devices and mirrors I/O to the monitors.
  
  @ivar event_manager: L{EventManager} to which this L{DeviceManager} can post 
    L{AEEvent}s 
  @type event_manager: L{EventManager}
  @ivar out_devs: All output devices registered and initialized
  @type out_devs: list
  @ivar in_devs: All input devices registered and initialized
  @type in_devs: list
  @ivar out_mons: All output monitors registered and initialized
  @type out_mons: list
  @ivar in_mons: All input monitors registered and initialized
  @type in_mons: list
  @ivar def_output: The "default" output device
  @type def_output: L{AEOutput}
  @ivar def_input: The "default" input device
  @type def_input: L{AEInput}
  @ivar acc_eng: The AccessEngine reference that created this class
  @type acc_eng: L{AccessEngine}
  '''
  def __init__(self, acc_eng):
    '''
    Creates the empty lists for devices and monitors.

    @param acc_eng: The AccessEngine reference that created this class
    @type acc_eng: L{AccessEngine}
    '''
    self.out_devs = []
    self.in_devs = []
    self.out_mons = []
    self.in_mons = []
    self.def_output = None
    self.def_input = None
    self.acc_eng = acc_eng
    self.event_manager = None

  def init(self, event_man, **kwargs):
    '''
    Loads output and input devices. Called by L{AccessEngine} at startup.
    
    @todo: PP: how do we unload these devices when they're no longer desired?

    @param event_man: L{EventManager} to which this L{DeviceManager} can
      post L{AEEvent}s 
    @type event_man: L{EventManager}
    @param kwargs: References to managers not of interest here
    @type kwargs: dictionary
    '''
    self.event_manager = event_man
    reg = UIRegistrar()
    # load all device classes to be created at startup
    devs = reg.loadAssociated(DEVICE, self.acc_eng.getProfile(), STARTUP)
    # register all devices, making the first input and output the defaults
    for d in devs:
      # call registerDevice which will automatically set defaults
      try:
        self.registerDevice(d)
      except (AEOutput.AEOutputError, AEInput.AEInputError):
        log.info('Could not initialize device: %s', str(d))
      except NotImplementedError:
        log.info('Device does not provide an I/O interface: %s', str(d))
    # load all I/O monitor classes to be created at startup
    mons = reg.loadAssociated(MONITOR, self.acc_eng.getProfile(), STARTUP)
    for m in mons:
      # call registerDevice which will weed out non-IO monitors for us
      try:
        self.registerMonitor(m)
      except NotImplementedError:
        # ignore NotImplementedErrors if we get other monitor types in the mix
        pass
  
    # make sure we initialized at least one input and one output device
    if self.def_input is None:
      self.close()
      raise ValueError(_('No input device available'))
    if self.def_output is None:
      self.close()
      raise ValueError(_('No output device available'))
    
  def close(self):
    '''
    Shuts down this manager and all its registered L{AEOutput} and L{AEInput}
    devices.
    '''
    for d in self.out_devs:
      try:
        d.close()
      except:
        # ignore all exceptions
        pass
    for d in self.in_devs:
      try:
        d.close()
      except:
        # ignore all exceptions
        pass
      
  def stopNow(self, dev):
    '''
    Sends the stop command to the referenced output device. When the device has
    not been registered or is None, the stop command is sent to the default
    output device. After sending the stop, L{showStop} is called.

    @param dev: The device to send the stop command or None to send to L{the 
      default output device<setDefaultOutput>}.
    @type dev: L{AEOutput}
    '''
    # when dev is None assume defaultOutput (which may also be None)
    dev = dev or self.def_output
    dev.sendStop()
    # always send the command to the registered monitors
    self.showStop(dev)

  def showStop(self, dev):
    '''
    Calls all registered output monitors to show that a stop command has been
    sent to the referenced output device. This does nothing when no monitors
    are registered.

    @param dev: The device that was sent the stop command.
    @type dev: L{AEOutput}
    '''
    for mon in self.out_mons:
      mon.showStop(dev)
    # for demonstration purposes, just print if no monitors
    # if not self.out_mons:
    #  sys.stdout.write('{stop}')

  def talkNow(self, dev):
    '''
    Tells the specified output device to send buffered data. When the device
    has not been registered or is None, the talk command is sent to the
    default output device. After sending the talk, L{showTalk} is called.

    @param dev: The device to send the talk command or None to send to L{the 
      default output device<setDefaultOutput>}
    @type dev: L{AEOutput}
    '''
    # when dev is None assume defaultOutput (which may also be None)
    dev = dev or self.def_output
    dev.sendTalk()
    # always send the command to the registered monitors
    self.showTalk(dev)

  def showTalk(self, dev):
    '''
    Calls all registered output monitors to show the talk command sent to the
    referenced output device. This does nothing when no monitors are
    registered.

    @param dev: The device that was sent the talk command.
    @type dev: L{AEOutput}
    '''
    for mon in self.out_mons:
      mon.showTalk(dev)
    # for demonstration purposes, just print if no monitors
    #if not self.out_mons:
    #  print '{talk}'
    
  def sendString(self, dev, string, talk=True):
    '''
    Sends the string to the specified output device. When the device has not
    been registered or is None, the string is sent to the default output
    device. The string is sent with a L{AEOutput.Base.AEOutput.sendString} call
    followed by L{talkNow} when the talk parameter is true.
    
    @note: On 12/13/05 PP changed this method to send more than one character at
    a time to each device. Whatever string is passed to this method is now 
    sent as-is to each device. The default behavior of any Speech device is
    to add whatever it receives to its output thread queue. In this case, each
    string is added as an element to the queue. The thread then dumps whatever
    is in its queue to the output device driver. In this case, the driver is
    now receiving entire words to buffer and later send when the buffer is full
    or a talk is received. This small change results in drastic performance
    improvements to LSR as the call overhead is now greatly reduced from here
    to each device, from each device object to its output thread, and from the
    output thread to the device hardware.

    @param dev: The device to send the string or None to send to L{the default 
      output device<setDefaultOutput>}
    @type dev: L{AEOutput}
    @param string: The string to send to the referenced device
    @type string: string
    @param talk: Defaults to True to indicate that L{talkNow} should be called
      after all characters are sent
    @type talk: boolean
    '''
    # when dev is None assume defaultOutput (which may also be None)
    dev = dev or self.def_output
    dev.sendString(string)
    if talk:
       self.talkNow(dev)
    self.showString(dev, string, talk)
 
  def showString(self, dev, string, talk):
    '''
    Sends a string to all registered output monitors. This method should be
    used when only monitor output is desired without sending to any devices.

    This is also called by L{sendString} when no output devices have been
    registered to support testers with no other output.

    @param dev: The device that was sent the string.
    @type dev: L{AEOutput}
    @param string: The string that was sent to the referenced device
    @type string: string
    @param talk: Defaults to True to indicate that L{talkNow} should be called
    after all characters are sent
    @type talk: boolean
    '''
    for mon in self.out_mons:
      mon.showString(dev, string, talk)
    # for demonstration purposes, just print if no monitors
    #if not self.out_mons:
      #print string

  def sendCmd(self, dev, cmd, value):
    '''
    Sends the referenced command to the referenced device. When the device has
    not been registered or is None, the command is sent to the default output
    device. After sending the command, L{showCmd} is called.

    @param dev: The device to send the command or None to send to L{the default 
      output device<setDefaultOutput>}
    @type dev: L{AEOutput}
    @param cmd: The device specific command that should be sent
    @type cmd: integer
    @param value: The device specific value for the specified command
    @type value: object
    @raise ValueError: When the command or value is invalid
    '''
    # when dev is None assume defaultOutput (which may also be None)
    dev = dev or self.def_output
    dev.sendCommand(cmd, value)
    # always send the command to the registered monitors
    self.showCmd(dev, cmd, value)

  def showCmd(self, dev, cmd, value):
    '''
    Calls all registered output monitors to show the command sent to the
    referenced output device. This does nothing when no output monitors are
    registered.

    @param dev: The device that was sent the command
    @type dev: L{AEOutput}
    @param cmd: The device specific command that should be sent
    @type cmd: int
    @param value: The device specific value for the specified command
    @type value: float
    '''
    for mon in self.out_mons:
      mon.showCmd(dev, cmd, value)
    # for demonstration purposes, just print if no monitors
    #if not self.out_mons:
    #  sys.stdout.write('{CMD:%s(%d,%d)}' % (dev.getName(), cmd, value))

  def getCmdLevel(self, dev, cmd):
    '''
    Gets the current value of the specified command.

    The default value should be returned when it has not been changed by
    L{sendCmd}. A return of 0 indicates the the specified command has only a
    single value. A return of -1 indicates that either the device is not
    registered, or the referenced command is not available.

    @param dev: The device to query or None to query L{the default output 
      device<setDefaultOutput>}
    @type dev: L{AEOutput}
    @param cmd: The device specific command to query
    @type cmd: int
    @return: The current value of the specified command
    @rtype: int
    '''
    value = -1
    # when dev is None assume defaultOutput (which may also be None)
    if dev is None:
      dev = self.def_output
    if dev in self.out_devs:
      try:
        value = dev.getCommandLevel(cmd)
      except ValueError:
        pass  # value already set to -1
    # return the value
    return value

  def getMaxLevel(self, dev, cmd):
    '''
    Gets the maximum value of the specified command.

    A return of -1 indicates that either the device is not registered, or the 
    referenced command is not available.

    @param dev: The device to query or None to query L{the default output 
      device<setDefaultOutput>}
    @type dev: L{AEOutput}
    @param cmd: The device specific command to query
    @type cmd: int
    @return: The maximum value of the specified command
    @rtype: int
    '''
    value = -1
    # when dev is None assume defaultOutput (which may also be None)
    if dev is None:
      dev = self.def_output
    if dev in self.out_devs:
      try:
        value = dev.getLevelCount(cmd)
      except ValueError:
        pass  # value already set to -1
    # return the value
    return value

  def sendIndex(self, dev, marker=0):
    '''
    Sends the referenced index marker to the referenced device. When the
    device has not been registered or is None, the index marker is sent to the
    default output device. After sending the command, L{showCmd} is called.

    @param dev: The device to send the index marker or None to send to L{the 
      default output device<setDefaultOutput>}
    @type dev: L{AEOutput}
    @param marker: The marker value that should be sent or 0 when unspecified
    @type marker: int
    '''
    # when dev is None assume defaultOutput (which may also be None)
    if dev is None:
      dev = self.def_output
    try:
      if dev in self.out_devs:
        dev.sendIndex(marker)
      # always send the command to the registered monitors
      self.showIndex(dev, marker)
    except NotImplementedError:
      # do nothing when Indexing not implemented
      pass

  def showIndex(self, dev, marker):
    '''
    Calls all registered output monitors to show the index marker sent to the
    referenced output device. This does nothing when no output monitors are
    registered.

    @param dev: The device that was sent the index marker
    @type dev: L{AEOutput}
    @param marker: The marker value that was sent
    @type marker: int
    '''
    for mon in self.out_mons:
      mon.showIndex(dev, marker)
    # for demonstration purposes, just print if no monitors
    #if not self.out_mons:
      #sys.stdout.write('{I:%s:%d}' % (dev, marker))
      
  def _isDuplicateDevice(self, dev, output):
    '''
    Checks if the given device is a duplicate in that it provides no new 
    interfaces beyond those provided by the devices already registered.
    
    @param dev: A device class
    @type dev: L{AEInput} or L{AEOutput} class
    @param output: Is the device an output device (True) or input (False)?
    @type output: boolean
    '''
    if output:
      dt = AEOutput.DEVICE_TYPES
      reg_devs = self.out_devs
    else:
      dt = AEInput.DEVICE_TYPES
      reg_devs = self.in_devs
    # get interfaces device implements
    dev_int = (i for i in dt if implements(dev, i))
    # check if some registered device already implements all of these interfaces
    for reg_dev in reg_devs:
      reg_int = (i for i in dt if implements(reg_dev, i))
      if set(reg_int).issuperset(dev_int):
        return True
    return False

  def _registerInputDevice(self, dev):
    '''
    Registers the given device as an input device if the device implements the
    L{AEInput} base class and provides some subinterface not already provided
    by another registered input device. For instance, if a device implementing
    L{AEInput.SystemInput} is already registered, another device implementing 
    just L{AEInput.SystemInput} will not be registered by this method.
    
    @param dev: Device to attempt to register
    @type dev: L{AEInput.AEInput} class
    @return: True if the device was registered, False if it is not an 
      L{AEInput} device, and None if it provides no new interfaces
    @rtype: boolean or None
    '''
    if not implements(dev, AEInput.AEInput):
      # don't register something that isn't an input device
      return False
    if self._isDuplicateDevice(dev, False):
      # don't register a device that provides all the same interfaces as a
      # previous device
      return False
    if dev not in self.out_devs:
      # don't reinitialize a device that was added to out_devs
      dev.init()
    # add DeviceManager as a gesture listener
    dev.addInputListener(self._onGesture)
    # append to the in_devs list after successfully initialized
    self.in_devs.append(dev)
    log.info('Added input device %s', str(dev))
    # first input device registered is default
    if self.def_input is None:
      self.def_input = dev
    return True
  
  def _registerOutputDevice(self, dev):
    '''
    Registers the given device as an output device if the device implements the
    L{AEOutput} base class and provides some subinterface not already provided
    by another registered output device. For instance, if a device implementing
    L{AEOutput.Speech} is already registered, another device implementing just
    L{AEOutput.Speech} will not be registered by this method.
    
    @param dev: Device to attempt to register
    @type dev: L{AEOutput.AEOutput} class
    @return: True if the device was registered, False if it is not an 
      L{AEOutput} device, and None if it provides no new interfaces
    @rtype: boolean or None
    '''
    if not implements(dev, AEOutput.AEOutput):
      # don't register something that isn't an output device
      return False
    if self._isDuplicateDevice(dev, True):
      # don't register a device that provides all the same interfaces as a
      # previous device
      return None
    if dev not in self.in_devs:
      # don't reinitialize a device that was added to in_devs
      dev.init()
    # add DeviceManager as an index listener when implemented
    try:
      dev.addIndexListener(self._onIndex)
    except NotImplementedError:
      # do nothing when indexing not implemented
      pass
    # append to the out_devs list after successfully initialized
    self.out_devs.append(dev)
    log.info('Added output device %s', str(dev))
    # first output device registered is default
    if self.def_output is None:
      self.def_output = dev
    return True

  def registerDevice(self, dev):
    '''    
    Registers the referenced device based on its one or more interfaces.
    When the interface is determined, the init() method is called on the device.
    Returns true when the reference is of a known type that initializes
    successfully.
    
    @param dev: The device to register
    @type dev: L{AEOutput} or L{AEInput} class
    @raise NotImplementedError: When the device implements none of the required
      interfaces
    @raise AEOutputError: When output device initialization fails
    @raise AEInputError: When input device initialization fails
    '''
    is_out = self._registerOutputDevice(dev)
    is_in = self._registerInputDevice(dev)

    if is_out == False and is_in == False:
      # raise an exception if none of the required interfaces was implemented
      raise NotImplementedError
  
  def registerMonitor(self, mon):
    '''
    @todo: PP: implement when we have input and output monitors
    '''
    pass
    #implemented = False
    #registered = False
    ## maybe register as output monitor
    #if isinstance(mon, OutputMonitor):
      #implemented = True
      #if mon not in self.out_mons:
        #if not registered:
          ## no need to multi-init
          #mon.init()
        ## append to the in_devs list after successfully initialized
        #self.out_mons.append(mon)
        #registered = True
      #else:
        #registered = True   # indicate already initialized

    ## maybe register as input monitor
    #if isinstance(mon, InputMonitor):
      #implemented = True
      #if mon not in self.in_mons:
        #if not registered:
          ## no need to multi-init
          #mon.init()
        ## append to the in_devs list after successfully initialized
        #self.in_mons.append(mon)
        #registered = True
      #else:
        #registered = True   # indicate already initialized
    #if not implemented:
      ## raise an exception if none of the required interfaces was implemented
      #raise NotImplementedError
    ## return whether the device/monitor was registered
    #return registered

  def setDefaultOutput(self, dev):
    '''
    Sets the referenced output device as the default output device. When the
    referenced device is not registered, or does not implement the
    L{AEOutput} interface, this does nothing. 

    @param dev: The input device
    @type dev: L{AEOutput}
    @raise NotImplementedError: When the provided device does not implement
        L{AEOutput.AEOutput}
    '''
    if dev in self.out_devs:
      self.def_output = dev
    elif implements(dev, AEOutput.AEOutput):
      if self.register(dev):
        self.def_output = dev
    else:
      raise NotImplementedError(_('Not an AEOutput instance'))

  def getDefaultOutput(self):
    '''
    Returns the current default output device or None when no output devices
    have been registered.

    @return: The current default output device
    @rtype: L{AEOutput}
    '''
    return self.def_output
  
  def getOutputByName(self, name):
    '''
    Gets the L{AEOutput} device registered under the given name.
    
    @param name: Name of the output device
    @type name: string
    @return: Output device or None if not registered
    @rtype: L{AEOutput}
    '''
    for dev in self.out_devs:
      if dev.__class__.__name__ == name:
        return dev
    return None

  def setDefaultInput(self, dev):
    '''
    Sets the referenced input device as the default input device. When the
    referenced device is not registered, or does not implement the
    L{AEInput} interface, this does nothing. 

    @param dev: The input device
    @type dev: L{AEInput}
    @raise NotImplementedError: When the provided device does not implement
        L{AEInput.AEInput}
    '''
    if dev in self.in_devs:
      self.def_input = dev
    elif implements(dev, AEInput.AEInput):
      if self.register(dev):
        self.def_input = dev
    else:
      raise NotImplementedError(_('Not an AEInput instance'))

  def getDefaultInput(self):
    '''
    Returns the current default input device or None when no input devices
    have been registered.

    @return: The current default input device
    @rtype: L{AEInput}
    '''
    return self.def_input
  
  def getInputByName(self, name):
    '''
    Gets the L{AEInput} device registered under the given name.
    
    @param name: Name of the input device
    @type name: string
    @return: Input device or None if not registered
    @rtype: L{AEInput}
    '''
    for dev in self.in_devs:
      if dev.__class__.__name__ == name:
        return dev
    return None
  
  def _onGesture(self, gesture):
    '''
    Creates an L{AEEvent} indicating the given gesture was found on a registered
    input device. When executed, the L{AEEvent} will notify the L{TierManager}
    about the gesture and allow it to activate the appropriate L{Task}
    registered to respond to the gesture in the active L{Tier}.
    
    @param gesture: Gesture seen on an input device
    @type gesture: L{AEInput.Gesture}
    '''
    event = AEEvent.InputGesture(gesture)
    self.event_manager.postEvents(event)
    
  def _onIndex(self, index):
    '''
    @todo: PP: handle index callback markers
    '''
    pass

if __name__ == '__main__':
  # test this class
  import DeviceManager
  dm = DeviceManager.DeviceManager('not used')
  dm.stopNow(None)
  for ch in 'Hello World':
    dm.sendChar(None, ch)
  dm.talkNow(None)
  dm.sendIndex(None,42)
  dm.sendString(None,'Again')
  

