'''
Defines a L{Perk} class that can register L{Task}s which will be executed in
response to L{AEEvent}s and L{AEInput.Gesture}s.

@author: Peter Parente
@author: Pete Brunet
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import logging, weakref
import Task, AEInput, AEEvent
from i18n import _

log = logging.getLogger('Perk')

class Perk(Task.Tools.All):
  '''  
  Registers and maintains a collection of L{Task}s that will execute in response
  to L{AccessEngine} events and commands (L{AEEvent}s and L{AEInput.Gesture}s.).
  This class should be sublclassed by L{Perk} developers who wish to define,
  modify, or extend the LSR user interface. This class derives from
  L{Task.Tools.All} to allow the use of the convenience methods defined in that
  class during L{Perk} intialization.
  
  @ivar named_tasks: L{Task}s registered by name
  @type named_tasks: dictionary (string : Task instance)
  @ivar event_tasks: L{Task}s registered by type
  @type event_tasks: dictionary of (Task subclass : list of Task instance)
  @ivar commands: L{Task}s to execute in response to a L{AEInput.GestureList} 
    sequence on some L{AEInput} device. When a L{Task} is unregistered from the
    L{named_tasks} dictionary, its corresponding entry in this dictionary (if 
    it exists) will automatically be removed because of the weak reference
  @type commands: weakref.WeakValueDictionary of (GestureList : Task)
  '''
  def __init__(self):
    '''
    Creates empty dictionaries for named and event L{Task}s.
    '''
    Task.Tools.All.__init__(self)
    self.named_tasks = {}
    self.event_tasks = {}
    self.commands = weakref.WeakValueDictionary()
    
  def init(self):
    '''
    Initializes the L{Perk}. Called by the owning L{Tier} to execute any setup 
    code specified in a L{Perk} subclass just after the L{Perk} is pushed onto
    the L{Tier}.
    
    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError
  
  def registerCommandTask(self, device, codes, name, propogate=False):
    '''
    Registers a L{Task} in this L{Perk} to be executed in response to an 
    L{AEEvent} indicating that the given action codes were input on the given 
    L{AEInput} device. 
    
    @param device: Input device to monitor
    @type device: L{AEInput}
    @param codes: List of lists of action codes forming the L{AEInput.Gesture} 
      that will trigger the execution of the named L{Task}. For example, 
      codes=[[Keyboard.AEK_CTRL, Keyboard.AEK_TILDE]] indicates the single
      gesture of simultaneously pressing Ctrl and ~ on the keyboard device.
    @type codes: list of list of integer
    @param name: Name of the L{Task} registered via L{registerNamedTask} to 
      execute when the input gesture is detected on the device
    @type name: string
    @param propogate: Should the input gesture be allowed to propogate to the OS
      after we receive it?
    @type propogate: boolean
    @raise ValueError: When a L{Task} with the given name is not registered
    '''
    # construct the key list and find the Task with the given name
    gl = AEInput.GestureList(device, codes)
    task = self.getNamedTask(name)
    if task is None:
      # raise a value error if there is no Task with that name
      raise ValueError(_('A Task with name %s is not registered') % name)
    if self.tier.getCommandPerkAndTask(gl) != (None, None):
      # raise a value error if there is already a command registered for this
      # sequence of keys in this TIer
      raise ValueError(
        _('A command with action codes %s on device %s is already registered') % 
        (codes, device.getName()))
    # add key list to our dictionary of commands and store the task to execute
    # as its value
    self.commands[gl] = task
    if not propogate:
      # try to add a filter to the device, ignore if not supported
      try:
        device.addFilter(gl)
      except NotImplementedError:
        pass
    
  def registerNamedTask(self, task, name):
    '''
    Registers a new L{Task} under the given name if no L{Task} is already 
    registered with that name in this L{Tier}.
    
    Only one L{Task} can be registered under a name in a L{Tier}. If a L{Task} 
    is already registered under the given name, any other registration with that
    name is ignored.
    
    @param task: L{Task} to register
    @type task: L{Task.Base.Task}
    @param name: Name to associate with the L{Task}
    @type name: string
    @raise ValueError: When a L{Task} with the given name is already registered
      in this L{Tier}
    '''
    if self.getNamedTask(name) is None:
      self.named_tasks[name] = task
    else:
      raise ValueError(name)
    
  def registerEventTask(self, task, focus=False, tier=False, background=False):
    '''
    Registers a new L{Task} in this L{Perk} under the L{Task} type. The type
    determines which kind of L{AEEvent} will trigger the execution of the
    registered L{Task}. If one or more Tasks are already registered for this 
    type, the given L{Task} will be inserted at the top of the registered stack
    of L{Task}s (i.e. it will be executed first for the appropriate event).
    
    The L{focus}, L{tier}, and L{background} parameters specify on which layer
    the L{Task} will handle events. If focus is True, the L{Task} will be 
    executed in response to an event from a focused control within this L{Tier}.
    If tier is True, the L{Task} will be executed in response to an event from
    an unfocused control within this L{Tier}. If background is True, the L{Task}
    will be executed in response to an event from any control within the tier 
    when the L{Tier} is not active.
    
    The three layers are mutually exclusive. You may set any combination of
    focus, tier, and background to True to register the given L{task} on each
    selected layer in one call. If all three parameters are False, the 
    registration defaults to the focus layer.
    
    The L{Task} passed to this method must implement L{Task.Base.Task.getType}.

    @param task: L{Task} to register
    @type task: L{Task.Base.Task}
    @param focus: Should this L{Task} handle events from focused accessibles in 
      this L{Tier}?
    @type focus: boolean
    @param tier: Should this L{Task} handle events from unfocused accessibles in 
      this L{Tier}?
    @type tier: boolean
    @param background: Should this L{Task} handle events from any accessible in 
      this L{Tier} when the L{Tier} is inactive?
    @type background: boolean
    @raise NotImplementedError: When the L{Task} does not implement 
      L{Task.Base.Task.getType}
    '''
    # default to the focus layer if no layer is specified    
    default = not (focus or tier or background)
    d = {AEEvent.FOCUS_LAYER : focus or default, AEEvent.TIER_LAYER : tier, 
         AEEvent.BACKGROUND_LAYER : background}
    # get the kind of AEEvent the Task 
    kind = task.getType()
    # opt: inform the Tier of the kind of event that is now desired by this Perk
    self.tier.setEventInterest(kind, True)
    for layer, val in d.items():
      if val:
        # get the list of Task registered for this type and layer or insert an
        # empty list and get it instead
        curr = self.event_tasks.setdefault((kind, layer), [])
        # insert the task at the front of the list
        curr.insert(0, task)
    
  def unregisterCommandTask(self, device, codes):
    '''
    Unregisters a L{Task} set to execute in response to the given action codes 
    on the given device B{from this L{Perk} only}.
    
    @param device: Input device to monitor
    @type device: L{AEInput}
    @param codes: List of lists of action codes forming the L{AEInput.Gesture} 
      that will trigger the execution of the named L{Task}. For example, 
      codes=[[Keyboard.AEK_CTRL, Keyboard.AEK_TILDE]] indicates the single
      gesture of simultaneously pressing Ctrl and ~ on the keyboard device.
    @type codes:  list of list of integer
    @raise KeyError: When a L{AEInput.GestureList} is not registered
    '''
    # GestureList has __eq__ and __hash__ overridden so two different instances 
    # can hash to the same dictionary location as long as their devices and 
    # codes are the same
    gl = AEInput.GestureList(device, codes)
    del self.commands[gl]
    # try to remove a filter from the device, ignore if not supported
    try:
      device.removeFilter(gl)
    except NotImplementedError:
      pass
    
  def unregisterNamedTaskLocal(self, name):
    '''
    Unregisters a named L{Task} B{from this L{Perk} only}. If a Task with the 
    given name is not found in this L{Perk}, an exception is raised.
    
    @param name: Name of the L{Task} to unregister
    @type name: string
    @raise KeyError: When a L{Task} with the given name is not registered
    '''
    del self.named_tasks[name]
    
  def unregisterEventTaskLocal(self, task, focus=False, tier=False, 
                               background=False):
    '''
    Unregisters the given L{Task} instance B{from this L{Perk} only}. If the 
    given L{Task} instance was not registered for an event in this L{Perk}, an 
    exception is raised. The L{focus}, L{tier}, and L{background} parameters
    state from which layer(s) this L{Task} should be unregistered.
    
    The L{Task} passed to this method must implement L{Task.Base.Task.getType}.
    
    @param task: L{Task} to unregister
    @type task: L{Task.Base.Task}
    @raise KeyError: When there are no L{Task}s registered with the type of the 
        given L{Task}
    @raise ValueError: When the given L{Task} is not registered on one of the
      specified layers
    @raise NotImplementedError: When the L{Task} does not implement 
      L{Task.Base.Task.getType}
    @see: L{registerEventTask}
    '''
    # default to the focus layer if no layer is specified    
    default = not (focus or tier or background)
    d = {AEEvent.FOCUS_LAYER : focus or default, AEEvent.TIER_LAYER : tier, 
         AEEvent.BACKGROUND_LAYER : background}
    # get the task type
    kind = task.getType()
    # opt: inform the Tier of the kind of event that is no longer
    # desired by this Perk
    self.tier.setEventInterest(kind, False)
    for layer, val in d.items():
      if val:
        # get the list of Task registered for this type and layer or insert an
        # empty list and get it instead
        curr = self.event_tasks[(kind, layer)]
        curr.remove(task)
    
  def getCommandTask(self, device, codes):
    '''
    Gets the L{Task} registered anywhere in this L{Tier} to execute in response 
    to the given action codes on the given device.
    
    Calls L{Tier.Tier.getCommandPerkAndTask} to search across all L{Perk}s in
    the owning L{Tier}.
    
    @param device: Input device to monitor
    @type device: L{AEInput}
    @param codes: List of lists of action codes forming the L{AEInput.Gesture} 
      that will trigger the execution of the named L{Task}. For example, 
      codes=[[Keyboard.AEK_CTRL, Keyboard.AEK_TILDE]] indicates the single
      gesture of simultaneously pressing Ctrl and ~ on the keyboard device.
    @type codes: list of list of integer
    @return: L{Task} set to execute in response to the input gesture or None
    @rtype: L{Task.Base.Task}
    '''
    gl = AEInput.GestureList(device, codes)
    perk, task = self.tier.getCommandPerkAndTask(gl)
    return task
    
  def getCommandTaskLocal(self, gesture_list):
    '''
    Gets the L{Task} registered B{in this L{Perk} only} to execute in response
    to the given L{AEInput.GestureList}.
    
    Called by L{Tier.Tier.getCommandPerkAndTask} during a search through all
    L{Perk}s in the owning L{Tier} for the given key list. L{Perk} writers
    should use L{getCommandTask} to search for a L{Task} across the entire
    L{Tier} instead of this method.
    
    @param gesture_list: Gestures and device on which they were performed
    @type gesture_list: L{AEInput.GestureList}
    @return: L{Task} set to execute in response to the input gesture or None
    @rtype: L{Task.Base.Task}
    '''
    return self.commands.get(gesture_list)
    
  def getNamedTask(self, name):
    '''
    Gets the L{Task} with the given name if it is registered anywhere in the
    current L{Tier}. If no L{Task} is registered under the given name, returns 
    None.
    
    Calls L{Tier.Tier.getNamedTask} to search across all L{Perk}s in the owning
    L{Tier}.
    
    @param name: Name of the L{Task} to locate
    @type name: string
    @return: L{Task} with the given name or None
    @rtype: L{Task.Base.Task}
    '''
    # Note: self.tier is an attribute of the Task.Tools.TaskTools superclass.
    return self.tier.getNamedTask(name)
    
  def getNamedTaskLocal(self, name):
    '''
    Gets the L{Task} with the given name if it is registered B{in this L{Perk}
    only}. If no L{Task} is registered under the given name in this L{Perk}, 
    returns None.
    
    Called by L{Tier.Tier.getNamedTask} during a search through all L{Perk}s in
    the owning L{Tier} for the given name. L{Perk} writers should probably use
    L{getNamedTask} to search for a L{Task} across the entire L{Tier} instead.
    
    @param name: Name of the L{Task} to locate
    @type name: string
    @return: L{Task} with the given name or None
    @rtype: L{Task.Base.Task}
    @see: L{getNamedTask}
    '''
    return self.named_tasks.get(name)
    
  def getAllEventTasks(self, event_type, task_layer):
    '''
    Gets all registered L{Task}s registered to handle the given L{AEEvent} type
    on the given layer from the current L{Tier}. 
    
    Calls L{Tier.Tier.getAllEventTasks} to search across all L{Perk}s in the 
    owning L{Tier}.  
    
    @param event_type: Desired type of L{AEEvent}
    @type event_type: L{AEEvent} class
    @param task_layer: Layer on which the desired L{Task}s are registered, one 
      of L{Task.FOCUS_LAYER}, L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type task_layer: integer
    @return: List of all L{Task}s that handle the given event type and on 
      the given layer in the L{Tier} which owns this L{Perk}
    @rtype: list of L{Task.Base.Task}
    '''
    # Note: self.tier is an attribute of the Task.Tools.TaskTools superclass.
    return self.tier.getAllEventTasks(event_type, task_layer)
    
  def getEventTasksLocal(self, event_type, task_layer):
    '''
    Get all registered L{Task}s registered to handle the given L{AEEvent} type 
    B{in this L{Perk} only}.
    
    Called by L{Tier.Tier.getAllEventTasks} during a search through all L{Perk}s
    in the owning L{Tier} for the given task type. L{Perk} writers should
    probably use L{getAllEventTasks} to search for a L{Task} across the entire
    L{Tier} instead.
    
    @param event_type: Desired type of L{AEEvent}
    @type event_type: L{AEEvent} class
    @param task_layer: Layer on which the desired L{Task}s are registered, one 
      of L{Task.FOCUS_LAYER}, L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type task_layer: integer
    @return: List of all L{Task}s that handle the given event type on the given
      layer in this L{Perk}
    @rtype: list of L{Task.Base.Task}
    @see: L{getAllEventTasks}
    '''
    return self.event_tasks.get((event_type, task_layer), [])

  def getDescription(self):
    '''
    @return: Description of this task
    @rtype: string
    '''
    return ''
    
  def getName(self):
    '''
    @return: Name of this task
    @rtype: string
    '''
    return str(self.__class__.__name__)  