'''
Defines L{AccAdapt.Adapter}s for AT-SPI list accessibles. Lists implement both
the Table and the Selection interfaces.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import pyLinAcc
from AEEvent import *
from DefaultEventHandler import *
from DefaultNav import *
from LSRInterfaces import *
from pyLinAcc import Interfaces, Constants

FUDGE_PX = 5

class ListEventHandlerAdapter(DefaultEventHandlerAdapter):
  '''
  Overrides L{DefaultEventHandlerAdapter} to generate selector events on focus
  and on selection. Expects the subject to be a raw L{pyLinAcc.Accessible}.
  
  Adapts subject accessibles that provide the L{pyLinAcc.Interfaces.ISelection},
  L{pyLinAcc.Interfaces.ITable} interfaces and have ROLE_TABLE.
  '''  
  provides = [IEventHandler]

  @staticmethod
  def when(subject):
    '''
    Tests if the given subject can be adapted by this class.
    
    @param subject: Accessible to test
    @type subject: L{pyLinAcc.Accessible}
    @return: True when the subject meets the condition named in the docstring
    for this class, False otherwise
    @rtype: boolean
    '''
    i = Interfaces
    c = Constants
    r = subject.getRole()
    return (r == c.ROLE_TABLE and i.ISelection(subject) and i.ITable(subject)) 

  def _handleFocusEvent(self, event, **kwargs):
    '''
    Creates an L{AEEvent.FocusChange} indicating that the accessible being
    adapted has gained the focus. Also creates a L{AEEvent.SelectorChange}.
    These two L{AEEvent}s will be posted by the caller.
    
    @param event: Raw focus change event
    @type event: L{pyLinAcc.Event.Event}
    @param kwargs: Parameters to be passed to any created L{AEEvent}
    @type kwargs: dictionary
    @return: L{AEEvent.FocusChange} and L{AEEvent.SelectorChange}
    @rtype: tuple of L{AEEvent}
    '''
    # build a focus event
    por = POR(self.subject, None, 0)
    focus_event = FocusChange(por, True, **kwargs)

    # get the selection interface
    selection = Interfaces.ISelection(self.subject)
    # ignore selector events when nothing is selected
    if selection.nSelectedChildren == 0:
      return (focus_event,)
    # get the selected object, for now, just the first item in the selection
    acc_child = selection.getSelectedChild(0)
    item_offset = acc_child.getIndexInParent()
    # build and return a selector event
    return focus_event, self._getSelectorEvent(acc_child, item_offset, **kwargs)
  
  def _handleDescendantEvent(self, event, **kwargs):
    '''
    Creates an L{AEEvent.SelectorChange} indicating the "selector" moved in this
    accessible.
    
    @param event: Raw decendent changed event
    @type event: L{pyLinAcc.Event.Event}
    @param kwargs: Parameters to be passed to any created L{AEEvent}
    @type kwargs: dictionary
    @return: L{AEEvent.SelectorChange}
    @rtype: tuple of L{AEEvent} 
    '''
    return (self._getSelectorEvent(event.any_data, event.detail1, **kwargs),)
  
  def _getSelectorEvent(self, accessible, item_offset, **kwargs):
    '''
    Creates an L{AEEvent.SelectorChange} indicating the selector moved in this
    accessible. 

    This method corrects for the possibility that the selected item actually
    have children that have the important information which are themselves not
    selected but returned as children of the even source. Right now, the last
    child in such a case appears to carry the information. More robust
    processing may be needed in the future.
    
    @param accessible: Accessible that generated this event
    @type accessible: L{pyLinAcc.Accessible}
    @param item_offset: Offset of item involved in the selection event
    @type item_offset: integer
    @param kwargs: Parameters to be passed to any created L{AEEvent}
    @type kwargs: dictionary
    @return: Selection event 
    @rtype: L{AEEvent.SelectorChange}
    
    @todo: PB: Is it correct to always skip combo boxes?  I seem to remember one
      or more cases where the combo box got the event but the contained 
      list/menu did not, i.e. there is some inconsistency in this area and bugs
      may need to be filed.
    @todo: PB: handle ctrl+arrow - changes active decendant
    @todo: PB: handle ctrl+spacebar - selects active decendant
    @todo: PB: handle shift+arrow - selects/deselects
    '''
    if accessible.childCount > 0:
      accessible = accessible.getChildAtIndex(accessible.childCount - 1)    
    
    # Notes:
    #   When adding support for more than one selection, we'll probably have
    #   to keep track of each state change (SELECTED or not) and report the most
    #   recently changed item or pair of items.  Some examples:
    #   - a single item became selected or unselected
    #   - one item became unselected and an adjacent one became selected
    #   In the second case the perk should probably receive both items
    #   The Selection interface has a list of one or more selected items.
    #   selection.nSelectedChildren indicates how many there are and 
    #   selection.getSelectedChild(index) accesses one of them.
    
    # determine if there is a Text interface
    try:
      text = Interfaces.IText(accessible)
      acc_child_text = text.getText(0, text.characterCount)   
    except NotImplementedError:      
      acc_child_text = accessible.name

    # create a POR, pass it and the item text at the POR to the Tier
    por = POR(self.subject, item_offset, 0)
    return SelectorChange(por, unicode(acc_child_text, 'utf-8'), 
                          **kwargs)
  
class ListNavAdapter(DefaultNavAdapter):
  '''
  Overrides L{DefaultNavAdapter} to provide navigation over list entries as 
  items and to avoid traversing entries as separate accessible children in the
  L{IAccessibleNav} interface. Expects the subject to be a L{POR}/
  
  Adapts accessibles that provide the L{pyLinAcc.Interfaces.ISelection} and
  L{pyLinAcc.Interfaces.ITable} interfaces, and have ROLE_TABLE.
  '''
  provides = [IAccessibleNav, IItemNav]

  @staticmethod
  def when(subject):
    '''
    Tests if the given subject can be adapted by this class.
    
    @param subject: L{POR} containing an accessible to test
    @type subject: L{POR}
    @return: True when the subject meets the condition named in the docstring
      for this class, False otherwise
    @rtype: boolean
    '''
    acc = subject.accessible
    r = acc.getRole()
    c = Constants
    i = Interfaces
    return (r == c.ROLE_TABLE and i.ISelection(acc) and i.ITable(acc))
      
  @pyLinAcc.errorToLookupError
  def _getVisibleItemExtents(self):
    '''
    Gets the item offsets of the first and last visible items in a 1D list of 
    items.
    
    @return: First and last item offsets
    @rtype: 2-tuple of integer
    @raise LookupError: When the first or last item or parent accessible is
      not available
    '''
    acc = self.accessible
    comp = Interfaces.IComponent(acc)
    e = comp.getExtents(Constants.WINDOW_COORDS)
    # get the first item
    x, y = e.x+FUDGE_PX, e.y+FUDGE_PX
    first = comp.getAccessibleAtPoint(x, y, Constants.WINDOW_COORDS)
    # get the last item
    x, y = e.x+e.width-FUDGE_PX, e.y+e.height-FUDGE_PX
    last = comp.getAccessibleAtPoint(x, y, Constants.WINDOW_COORDS)
    if first is None or last is None:
      raise IndexError
    # get their indicies
    return first.getIndexInParent(), last.getIndexInParent()
  
  @pyLinAcc.errorToLookupError
  def getNextItem(self, only_visible=True): 
    '''
    Gets the next item relative to the one indicated by the L{POR}
    providing this interface.

    Currently, this assumes only_visible is True.
    
    @param only_visible: True when Item in the returned L{POR} must be visible
    @type only_visible: boolean
    @return: Point of regard to the next item in the same accessible
    @rtype: L{POR}
    @raise IndexError: When there is no next item
    @raise LookupError: When lookup for the next item fails even though it may 
      exist
    '''
    acc = self.accessible
    off = self.item_offset
    comp = Interfaces.IComponent(acc)
    # get the first and last indicies
    i, j = self._getVisibleItemExtents()
    # and check if the first item is visible since it might be a header
    v = IAccessibleInfo(POR(acc.getChildAtIndex(0))).isAccVisible()
    if off is None and v:
      # return the possible header
      return POR(acc, 0, 0)
    elif off < i:
      # return the first visible, non-header item
      return POR(acc, i, 0)
    elif off+1 >= i and off+1 <= j:
      # return the next item offset
      return POR(acc, off+1, 0)
    else:
      # no more visible items
      raise IndexError
    
  @pyLinAcc.errorToLookupError
  def getPrevItem(self, only_visible=True):
    '''
    Gets the previous item relative to the one indicated by the L{POR} providing
    this interface. 
    
    Currently, this assumes only_visible is True.
    
    @param only_visible: True when Item in the returned L{POR} must be visible
    @type only_visible: boolean
    @return: Point of regard to the previous item in the same accessible
    @rtype: L{POR}
    @raise IndexError: When there is no previous item
    @raise LookupError: When lookup for the previous item fails even though it
      may exist
    '''
    acc = self.accessible
    off = self.item_offset
    comp = Interfaces.IComponent(acc)
    # get the first and last indicies
    i, j = self._getVisibleItemExtents()
    # and check if the first item is visible since it might be a header
    v = IAccessibleInfo(POR(acc.getChildAtIndex(0))).isAccVisible()
    if off is None:
      # no more visible items
      raise IndexError
    elif off > j:
      # return the last visible item
      return POR(acc, j, 0)
    elif off-1 >= i and off-1 <= j:
      # return the previous visible item
      return POR(acc, off-1, 0)
    elif off > 0 and v:
      # return the visible header item
      return POR(acc, 0, 0)
    elif off == 0:
      # return the list iteself
      return POR(acc, None, 0)
    
  @pyLinAcc.errorToLookupError
  def getLastItem(self, only_visible=True):
    '''
    Gets the last item relative to the one indicated by the L{POR} 
    providing this interface.
    
    @param only_visible: True when Item in the returned L{POR} must be visible
    @type only_visible: boolean
    @return: Point of regard to the last item in the same accessible
    @rtype: L{POR}
    @raise LookupError: When lookup for the last item fails even though it may 
      exist
    '''
    acc = self.accessible
    comp = Interfaces.IComponent(acc)
    # try getting the last item by index first
    child = acc.getChildAtIndex(acc.childCount-1)
    if IAccessibleInfo(POR(child)).isAccVisible() or not only_visible:
      return POR(acc, acc.childCount-1, 0)
    # use coords to get the last visible item
    i, j = self._getVisibleItemExtents()
    return POR(acc, j, 0)
    
  @pyLinAcc.errorToLookupError
  def getFirstItem(self, only_visible=True):
    '''
    Gets the first item relative to the one indicated by the L{POR} 
    providing this interface.
    
    @param only_visible: True when Item in the returned L{POR} must be visible
    @type only_visible: boolean
    @return: Point of regard to the last item in the same accessible
    @rtype: L{POR}
    @raise LookupError: When lookup for the last item fails even though it may 
      exist
    '''
    acc = self.accessible
    comp = Interfaces.IComponent(acc)
    # try getting the first item by index first
    child = acc.getChildAtIndex(0)
    if IAccessibleInfo(POR(child)).isAccVisible() or not only_visible:
      return POR(acc, 0, 0)
    # use coords to get the last visible item
    i, j = self._getVisibleItemExtents()
    return POR(acc, i, 0)
  
  @pyLinAcc.errorToLookupError 
  def getFirstAccChild(self):
    '''
    Always raises LookupError. Tables have items but no children.
    
    @raise LookupError: Always
    '''
    raise LookupError

  @pyLinAcc.errorToLookupError 
  def getLastAccChild(self):
    '''
    Always raises LookupError. Tables have items but no children.
    
    @raise LookupError: Always
    '''
    raise LookupError

  @pyLinAcc.errorToLookupError
  def getChildAcc(self, index):
    '''
    Always raises LookupError. Tables have items but no children.
    
    @raise LookupError: Always
    '''
    raise LookupError
