'''
Defines the base class for all L{Task}s.

@author: Peter Parente
@author: Larry Weiss
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import Tools, Perk, weakref

class Task(Tools.All):
  '''
  Most base class for all L{Task}s that execute in response to L{AEEvent}s.
  
  The implementer should override either the values in name and description or 
  the methods that return those values. They are used to provide a name and 
  description of this L{Task} to the user.
  
  @ivar this_perk: A weak reference to the L{Perk} that instantiated this 
      L{Task}
  @type this_perk: weakref.proxy to L{Perk}
  @ivar name: Name of this L{Task}
  @type name: string
  @ivar description: Extended description of this L{Task}
  @type description: string
  '''
  def __init__(self, perk):
    '''
    Sets the name and description of this L{Task} to an empty string by default.
    
    @param perk: L{Perk} that created this L{Task}
    @type perk: L{Perk}
    '''
    Tools.All.__init__(self)
    self.this_perk = weakref.proxy(perk)
    self.name = str(self.__class__.__name__)
    self.description = ''
    
  def execute(self, **kwargs):
    '''
    Executes this L{Task} in response to an event. Called by 
    L{Tier.Tier._executeTask}.
    
    @param kwargs: Dictionary containing parameters passed to a task.
    @type kwargs: dictionary 
    @return: Should processing continue? Always returns True by default.
    @rtype: boolean
    '''
    return True
  
  def update(self, **kwargs):
    '''
    Allows this L{Task} to update itself in response to an event that was 
    consumed by a L{Task} that executed higher in the L{Perk} stack. Called by 
    L{Tier.Tier._executeTask}.
    
    @param kwargs: Dictionary containing parameters passed to a task.
    @type kwargs: dictionary 
    '''
    pass
    
  def getType(self):
    '''
    Gets the type of a subclass of L{Task} for use by a L{Tier} in determining
    which L{Task}s to execute in response to some L{AEEvent}. This method must
    be implemented by any class that wants to be registered to handle events 
    via L{Perk.Perk.registerEventTask}.
    
    @raise NotImplementedError: When not overidden by a subclass
    '''
    raise NotImplementedError
    
  def getDescription(self):
    '''
    @return: Description of this task
    @rtype: string
    '''
    return ''
    
  def getName(self):
    '''
    @return: Name of this task
    @rtype: string
    '''
    return str(self.__class__.__name__)