'''
Defines the most base L{Tools} class from which all other L{Task.Tools} should
derive.

@author: Peter Parente
@author: Pete Brunet
@author: Larry Weiss
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import LSRConstants

class TaskTools(object):
  '''  
  Provides initialize and finalize methods for preparing and unpreparing the
  data required by L{Tools} subclasses while a L{Task} or L{Perk} is executing.
  Temporarily stores information such as the L{Tier}, L{LSRSettings}, and focus 
  L{POR} of the owning L{Tier}.
  
  @ivar acc_eng: Reference to L{AccessEngine} for task context
  @type acc_eng: L{AccessEngine}
  @ivar event_man: Reference to the L{EventManager}
  @type event_man: L{EventManager}
  @ivar view_man: Reference to the L{ViewManager}
  @type view_man: L{ViewManager}
  @ivar tier_man: Reference to the L{TierManager}
  @type tier_man: L{TierManager}
  @ivar device_man: Reference to the L{DeviceManager}
  @type device_man: L{DeviceManager}
  @ivar tier: The L{Tier} owning the L{Task} or L{Perk} subclass
  @type tier: L{Tier}
  @ivar state: The L{LSRSettings} for the Task or Perk subclass, set by 
    L{preExecute}
  @type state: L{LSRSettings}
  @ivar task_por:  The L{POR} for the Task or Perk subclass, set by 
    L{preExecute}
  @type task_por: L{POR}
  @ivar curr_item_text: The string of text starting at L{curr_item_por}
    and including L{task_por}. 
  @type curr_item_text: string
  @ivar curr_item_por: The L{POR} of the start of L{curr_item_text}. 
  @type curr_item_por: L{POR}
  @ivar error_code: Error code for last trapped exception, None if no error
  @type error_code: integer
  '''
  def __init__(self):
    '''
    Create the ivars and initialize them to None.
    '''
    self.event_man = None
    self.device_man = None
    self.tier_man = None
    self.view_man = None
    self.state = None
    self.task_por = None
    self.tier = None
    self.acc_eng = None
    self.curr_item_text = None
    self.curr_item_por = None
    self.error_code = None

  def preExecute(self, ae, tier, event_por=None):
    '''
    Configures this L{TaskTools} object before execution.

    @param ae: Reference to AccessEngine for event context
    @type ae: L{AccessEngine}
    @param tier: The owning L{Tier}
    @type tier: L{Tier}
    @param event_por: Point of regard received with the last event
    @type event_por: L{POR}
    '''
    self.acc_eng = ae
    self.tier = tier
    self.state = tier.getState()
    if self.state.Mode == LSRConstants.MODE_POINTER:
      self.task_por = self.state.Pointer
    elif self.state.Mode == LSRConstants.MODE_EVENT:
      self.task_por = event_por or self.state.Pointer
    else:
      self.task_por = tier.getFocusPOR()
    # get all manager references from AccessEngine very quickly
    self.acc_eng.loanManagers(self)

  def postExecute(self, save_por=True):
    '''
    Unconfigures this L{TaskTools} object after execution. Updates the Pointer 
    and saves the held state object back in the L{Tier} if L{save_por} is
    True and L{LSRSettings}.Mode is either L{LSRConstants.MODE_EVENT} or 
    L{LSRConstants.MODE_POINTER}.
    
    @param save_por: Store the L{task_por} in L{LSRSettings}?
    @type save_por: boolean
    '''
    # update the Pointer if needed
    if (save_por and self.state.Mode in 
        (LSRConstants.MODE_EVENT, LSRConstants.MODE_POINTER)):
      self.state.Pointer = self.task_por
    # store any state changes in the Tier
    self.tier.setState(self.state)
    # unset all the manager variables borrowed from AccessEngine very quickly
    self.acc_eng.unloanManagers(self)
    self.acc_eng = None
    self.tier = None
    self.state = None
    self.task_por = None
    self.error_code = None
    
  def getErrorCode(self):
    '''
    Gets the value of the last trapped error or None if no error occurred.
    
    @return: Error code
    @rtype: integer
    '''
    return self.error_code