'''
Defines L{Tools} for doing output.

@todo: PP: remove use of AEOutput.isSpeech; prefer try/except for non-supported
  methods
@todo: PP: use speech style objects instead

@author: Peter Parente
@author: Pete Brunet
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import Base, View, Word, AEOutput

class Output(Base.TaskTools):
  '''
  Provides methods for doing output on any L{AEOutput} device.
  '''
  # Constants used for getting text attributes
  # attribute and value must observe i8n, need to be able to translate both
  FONT_SIZE = 'size'
  FONT_COLOR = 'fg-color'
  FONT_WEIGHT = 'weight'
  FONT_STYLE = 'style'
  FONT_UNDERLINE = 'underline'
  FONT_STYLES = {'italic':_('italic'), 'bold':_('bold'), 'single':_('single')}

  FONT_ATTRS = {FONT_SIZE:_('size'), FONT_COLOR:_('color'), 
                FONT_WEIGHT:_('weight'), FONT_STYLE:_('style'), 
                FONT_UNDERLINE:('underline')}
  FONT_NON_NUMERIC = [FONT_COLOR, FONT_STYLE, FONT_UNDERLINE] 
  
  def say(self, text, por=None, dev=None):
    '''
    Outputs the text while respecting settings.

    @todo: LW: Send index marker before sending each word.
    @param text: Text to output
    @type text: string
    @param por: The point of regard of the start of text, or None when the text 
      is not from the screen
    @type por: L{POR}
    @param dev: The device to send the text to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    '''
    por = por or self.task_por
    # first, parse into words
    words = Word.buildWordsFromString(text, self.state, por)
    # handle blank strings
    if len(words) == 1 and self.state.Blank and words[0].isAllBlank():
      self.device_man.sendString(dev, _('blank'), True)
    else:
      # send each word, sending talk command at end
      for w in words:
        # send index markers
        self.device_man.sendString(dev, unicode(w), False)
      self.device_man.talkNow(dev)
      
  def sayItem(self, por=None, dev=None):
    '''
    Ouputs the text from the given L{POR} or the one stored in L{task_por} to 
    the end of the Item while respecting settings. 
    
    Note: I{This does not modify L{task_por}, L{curr_item_text}, or 
    L{curr_item_por}.}
    
    @param por: The point of regard of the start of text, or None when the text 
      is not from the screen or L{task_por} should be used.
    @type por: L{POR}
    @param dev: The device to send the item to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    '''
    por = por or self.task_por 
    if self.state.Wrap:
      # treating the rest of the app as 1 long item, so speak them all.
      item = self.getItemText(por)
      while item:
        self.say(item, self.task_por, dev)
        self.moveToNextItem(self.task_por)
        item = self.getItemText(self.task_por)
    else:
      item = self.getItemText(por)
      self.say(item)

  def sayWord(self, text=None, por=None, dev=None):
    '''
    Outputs the first L{Word} of L{text} while respecting settings. When text 
    is None, the text at L{task_por} is used. 
    
    Note: I{This does not modify L{task_por}, L{curr_item_text}, or 
    L{curr_item_por}.}

    @todo: LW: Send index marker before sending word.
    @param text: Text to output, or None when text at the L{task_por} should 
      be used.
    @type text: string
    @param por: The point of regard of the start of text, or None when the text 
      is not from the screen or L{task_por} should be used.
    @type por: L{POR}
    @param dev: The device to send the word to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    '''
    por = por or self.task_por
    w = Word.Word(self.state, por)
    if text is None:
      # use text at task_por
      text = self.getItemText(por)
      w.append(text[por.char_offset:])
    else:
      w.append(text)
    if self.state.Blank and w.isAllBlank():
      self.device_man.sendString(dev, _('blank'), True)
    else:
      self.device_man.sendString(dev, unicode(w), True)

  def sayChar(self, text=None, por=None, dev=None):
    '''
    Outputs the first character of L{text} while respecting settings. When text 
    is None, the text at L{task_por} is used.  
    
    Note: I{This does not modify L{task_por}, L{curr_item_text}, or 
    L{curr_item_por}.}

    @param text: Text to output, or None when text at the L{task_por} should 
      be used.
    @type text: string
    @param por: The point of regard of the start of text, or None when the text 
      is not from the screen or L{task_por} should be used.
    @type por: L{POR}
    @param dev: The device to send the char to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    '''
    por = por or self.task_por
    w = Word.Word(self.state, por)
    if text is None:
      # use text at task_por
      text = self.getItemText(por)
      c = text[por.char_offset]
    else:
      c = text[0]
    w.append(c)
    if self.state.Blank and w.isAllBlank():
      self.device_man.sendString(dev, _('blank'), True)
    else:
      self.device_man.sendString(dev, unicode(w), True)
      
  def sayTextAttributes(self, por=None):
    '''
    Say font attributes (size, style, color, etc) of a given por.
    
    Attributes of note:
    (http://developer.gnome.org/doc/API/2.0/atk/AtkText.html#AtkTextAttribute)
    ATK_TEXT_ATTR_EDITABLE Either "true" or "false" indicating whether text is
    editable or not
    ATK_TEXT_ATTR_UNDERLINE "none", "single", "double" or "low"
    ATK_TEXT_ATTR_STRIKETHROUGH "true" or "false" whether text is strikethrough
    ATK_TEXT_ATTR_SIZE 	The size of the characters.
    ATK_TEXT_ATTR_SCALE The scale of the characters. The value is a string 
    representation of a double
    ATK_TEXT_ATTR_WEIGHT The weight of the characters.  
    ATK_TEXT_ATTR_LANGUAGE The language used
    ATK_TEXT_ATTR_BG_COLOR The background color. The value is an RGB value of 
    the format "u,u,u"
    ATK_TEXT_ATTR_FG_COLOR The foreground color. The value is an RGB value of 
    the format "u,u,u"
    ATK_TEXT_ATTR_DIRECTION The direction of the text, if set. Values are 
    "none", "ltr" or "rtl"
    ATK_TEXT_ATTR_JUSTIFICATION The justification of the text, if set. Values
    are "left", "right", "center" or "fill"
    ATK_TEXT_ATTR_STYLE The slant style of the text, if set. Values are 
    "normal", "oblique" or "italic"
    
    @todo: DBC: Output message if not mapped?
    '''
    por = por or self.task_por
    attrs = self.getAccAllTextAttrs(por)
    self.stopNow() 
    for s in attrs:
      n, v = s.split(':')
      # we only want to report certain font attributes
      if n in self.FONT_ATTRS:
        # non-numeric output (such as color) needs own check for i8n
        if n in self.FONT_NON_NUMERIC:
          if n == 'fg-color':
            v = self.getColorString(v)
          elif v in self.FONT_STYLES:
            v = self.FONT_STYLES[v]
        # else: leave v as it is (a numeric value)
        self.say('%s %s' %(self.FONT_ATTRS[n], v))

  def msg(self, text, dev=None):
    '''
    Outputs the text while disregarding settings.

    @param text: Text to output
    @type text: string
    @param dev: The device to send the text to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    '''
    self.device_man.sendString(dev, text, True)

  def out(self, text, dev=None):
    '''
    Outputs the text while disregarding settings and not sending a flush.

    @param text: Text to output
    @type text: string
    @param dev: The device to send the text to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    '''
    self.device_man.sendString(dev, text, False)

  def stop(self, dev=None):
    '''
    Tells the referenced output device to interrupt current output and clear 
    buffered data. When Stopping is True, this has no effect.
    
    @param dev: The device to send the stop to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    '''
    if self.state.Stopping:
      self.device_man.stopNow(dev)
      self.state.MayStop = True
  
  def stopNow(self, dev=None):
    '''
    Tells the referenced output device to interrupt current output and clear 
    buffered data. Unlike L{stop}, this ignores the Stopping setting.
    
    @todo: LW: discard cached index markers
    @param dev: The device to send the stop to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    '''
    self.device_man.stopNow(dev)
    self.state.MayStop = True
    
  def mayStop(self, dev=None):
    '''
    Stops the device if L{LSRSettings}.MayStop is True. When False, resets that flag.

    @param dev: The device to send the stop to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    @return: Was the stop processed (True) or inhibited (False)?
    @rtype: boolean
    '''
    if self.state.MayStop:
      self.stop(dev)
      return True
    else:
      self.state.MayStop = True
      return False
      
  def inhibitMayStop(self):
    '''
    Prevents the next call to L{mayStop} from stopping the device provided to
    that call. Sets the L{LSRSettings}.MayStop flag to False.    
    '''
    self.state.MayStop = False
    
  def setRate(self, newRate, dev=None):
    '''
    Synchronously set the new speech rate on the referenced device. When no 
    device is specified, the default output device is used.
    
    This command does not take effect until subsequent text has been processed.
    The effect is that a call to L{getRate} immediately following this method 
    will generally B{not} reflect the new value.
    
    The newRate value should be between 1 and the value returned by 
    L{getMaxRate}. New values less than 1 will be set to 1, and new values 
    greater than maximum will be set to the maximum. When the device does not 
    support rate commands, this will have no effect.
    
    @param newRate: The integer value of the new rate.
    @type newRate: integer
    @param dev: The device to change that defaults to None for the default 
      output device.
    @type dev: L{AEOutput.Speech}
    '''
    if dev is None:
      dev = self.device_man.getDefaultOutput()
    # rate is only valid for speech devices
    if AEOutput.isSpeech(dev):
      # adjust newRate within range
      if newRate < 1:
        newRate = 1
      maxValue = self.device_man.getMaxLevel(dev, 
                                             AEOutput.Constants.CONTEXT_RATE)
      # max is -1 when parm not supported
      if maxValue > -1:
        if newRate > maxValue:
          newRate = maxValue
        try:
          self.device_man.sendCmd(dev, AEOutput.Constants.CONTEXT_RATE, newRate)
        except ValueError:
          pass
  
  def getRate(self, dev=None):
    '''
    Gets the current speech rate on the referenced device. When no device 
    is specified, the default output device is used.
    
    The value returned does B{not reflect pending changes}, only the current 
    value. A return of -1 indicates the device does not have a rate value, 
    e.g. a Braille device.

    @param dev: The device to change that defaults to None for the default 
      output device.
    @type dev: L{AEOutput.Speech}
    @return: The current rate.
    @rtype: integer
    '''
    curValue = -1
    if dev is None:
      dev = self.device_man.getDefaultOutput()
    # rate is only valid for speech devices
    if AEOutput.isSpeech(dev):
      curValue = self.device_man.getCmdLevel(dev, 
                                             AEOutput.Constants.CONTEXT_RATE)
    return curValue
  
  def getMaxRate(self, dev=None):
    '''
    Gets the maximum speech rate on the referenced device. When no device 
    is specified, the default output device is used.
    
    A return of -1 indicates the device does not have a rate value, 
    e.g. a Braille device.

    @param dev: The device to change that defaults to None for the default 
      output device.
    @type dev: L{AEOutput.Speech}
    @return: The current maximum rate.
    @rtype: integer
    '''
    maxValue = -1
    if dev is None:
      dev = self.device_man.getDefaultOutput()
    # rate is only valid for speech devices
    if AEOutput.isSpeech(dev):
      maxValue = self.device_man.getMaxLevel(dev, 
                                             AEOutput.Constants.CONTEXT_RATE)
    return maxValue
  
  def setPitch(self, newPitch, dev=None):
    '''
    Synchronously set the new speech pitch on the referenced device. When no 
    device is specified, the default output device is used.
    
    This command does not take effect until subsequent text has been processed.
    The effect is that a call to L{getPitch} immediately following this method 
    will generally B{not} reflect the new value.
    
    The newPitch value should be between 1 and the value returned by 
    L{getMaxPitch}. New values less than 1 will be set to 1, and new values 
    greater than maximum will be set to the maximum. When the device does not 
    support pitch commands, this will have no effect.
    
    @param newPitch: The integer value of the new pitch.
    @type newPitch: integer
    @param dev: The device to change that defaults to None for the default 
      output device.
    @type dev: L{AEOutput.Speech}
    '''
    if dev is None:
      dev = self.device_man.getDefaultOutput()
    # pitch is only valid for speech devices
    if AEOutput.isSpeech(dev):
      # adjust newPitch within range
      if newPitch < 1:
        newPitch = 1
      maxValue = self.device_man.getMaxLevel(dev, 
                                             AEOutput.Constants.CONTEXT_PITCH)
      # max is -1 when parm not supported
      if maxValue > -1:
        if newPitch > maxValue:
          newPitch = maxValue
        try:
          self.device_man.sendCmd(dev, AEOutput.Constants.CONTEXT_PITCH, 
                                  newPitch)
        except ValueError:
          pass
  
  def getPitch(self, dev=None):
    '''
    Gets the current speech pitch on the referenced device. When no device 
    is specified, the default output device is used.
    
    The value returned does B{not reflect pending changes}, only the current 
    value. A return of -1 indicates the device does not have a pitch value, 
    e.g. a Braille device.

    @param dev: The device to change that defaults to None for the default 
      output device.
    @type dev: L{AEOutput.Speech}
    @return: The current pitch.
    @rtype: integer
    '''
    curValue = -1
    if dev is None:
      dev = self.device_man.getDefaultOutput()
    # rate is only valid for speech devices
    if AEOutput.isSpeech(dev):
      curValue = self.device_man.getCmdLevel(dev, 
                                             AEOutput.Constants.CONTEXT_PITCH)
    return curValue
  
  def getMaxPitch(self, dev=None):
    '''
    Gets the maximum speech pitch on the referenced device. When no device 
    is specified, the default output device is used.
    
    A return of -1 indicates the device does not have a pitch value, 
    e.g. a Braille device.

    @param dev: The device to change that defaults to None for the default 
      output device.
    @type dev: L{AEOutput.Speech}
    @return: The current maximum pitch.
    @rtype: integer
    '''
    maxValue = -1
    if dev is None:
      dev = self.device_man.getDefaultOutput()
    # rate is only valid for speech devices
    if AEOutput.isSpeech(dev):
      maxValue = self.device_man.getMaxLevel(dev, 
                                             AEOutput.Constants.CONTEXT_PITCH)
    return maxValue
  
  def setVolume(self, newVol, dev=None):
    '''
    Synchronously set the new speech volume on the referenced device. When no 
    device is specified, the default output device is used.
    
    This command does not take effect until subsequent text has been processed.
    The effect is that a call to L{getVolume} immediately following this method 
    will generally B{not} reflect the new value.
    
    The newVol value should be between 1 and the value returned by 
    L{getMaxVolume}. New values less than 1 will be set to 1, and new values 
    greater than maximum will be set to the maximum. When the device does not 
    support volume commands, this will have no effect.
    
    @param newVol: The integer value of the new volume.
    @type newVol: integer
    @param dev: The device to change that defaults to None for the default 
      output device.
    @type dev: L{AEOutput.Speech}
    '''
    if dev is None:
      dev = self.device_man.getDefaultOutput()
    # rate is only valid for speech devices
    if AEOutput.isSpeech(dev):
      # adjust newVol within range
      if newVol < 1:
        newVol = 1
      maxValue = self.device_man.getMaxLevel(dev, 
                                             AEOutput.Constants.CONTEXT_VOLUME)
      # max is -1 when parm not supported
      if maxValue > -1:
        if newVol > maxValue:
          newVol = maxValue
        try:
          self.device_man.sendCmd(dev, AEOutput.Constants.CONTEXT_VOLUME, 
                                  newVol)
        except ValueError:
          pass
  
  def getVolume(self, dev=None):
    '''
    Gets the current speech volume on the referenced device. When no device 
    is specified, the default output device is used.
    
    The value returned does B{not reflect pending changes}, only the current 
    value. A return of -1 indicates the device does not have a volume value, 
    e.g. a Braille device.

    @param dev: The device to change that defaults to None for the default 
      output device.
    @type dev: L{AEOutput.Speech}
    @return: The current volume.
    @rtype: integer
    '''
    curValue = -1
    if dev is None:
      dev = self.device_man.getDefaultOutput()
    # rate is only valid for speech devices
    if AEOutput.isSpeech(dev):
      curValue = self.device_man.getCmdLevel(dev, 
                                             AEOutput.Constants.CONTEXT_VOLUME)
    return curValue
  
  def getMaxVolume(self, dev=None):
    '''
    Gets the maximum speech volume on the referenced device. When no device 
    is specified, the default output device is used.
    
    A return of -1 indicates the device does not have a volume value, 
    e.g. a Braille device.

    @param dev: The device to change that defaults to None for the default 
      output device.
    @type dev: L{AEOutput.Speech}
    @return: The current maximum volume.
    @rtype: integer
    '''
    maxValue = -1
    if dev is None:
      dev = self.device_man.getDefaultOutput()
    # rate is only valid for speech devices
    if AEOutput.isSpeech(dev):
      maxValue = self.device_man.getMaxLevel(dev, 
                                             AEOutput.Constants.CONTEXT_VOLUME)
    return maxValue
  
  def setVoice(self, newVoice, dev=None):
    '''
    Asynchronously set the new speech voice on the referenced device. When no 
    device is specified, the default output device is used.
    
    This command does not take effect until subsequent text has been processed.
    The effect is that a call to L{getVoice} immediately following this method 
    will generally B{not} reflect the new value.
    
    The newVoice value should be between 1 and the value returned by 
    L{getMaxVoice}. New values less than 1 will be set to 1, and new values 
    greater than maximum will be set to the maximum. When the device does not 
    support voice commands, this will have no effect.
    
    @param newVoice: The integer value of the new voice.
    @type newVoice: integer
    @param dev: The device to change that defaults to None for the default 
      output device.
    @type dev: L{AEOutput.Speech}
    '''
    if dev is None:
      dev = self.device_man.getDefaultOutput()
    # rate is only valid for speech devices
    if AEOutput.isSpeech(dev):
      # adjust newVoice within range
      if newVoice < 1:
        newVoice = 1
      maxValue = self.device_man.getMaxLevel(dev, 
                                             AEOutput.Constants.CONTEXT_VOICE)
      # max is -1 when parm not supported
      if maxValue > -1:
        if newVoice > maxValue:
          newVoice = maxValue
        try:
          self.device_man.sendCmd(dev, AEOutput.Constants.CONTEXT_VOICE, 
                                  newVoice)
        except ValueError, e:
          print e
  
  def getVoice(self, dev=None):
    '''
    Gets the current speech voice on the referenced device. When no device 
    is specified, the default output device is used.
    
    The value returned does B{not reflect pending changes}, only the current 
    value. A return of None indicates the device does not have a voice value, 
    e.g. a Braille device.

    @param dev: The device to change that defaults to None for the default 
      output device.
    @type dev: L{AEOutput.Speech}
    @return: The current voice.
    @rtype: integer
    '''
    curValue = None
    if dev is None:
      dev = self.device_man.getDefaultOutput()
    # rate is only valid for speech devices
    if AEOutput.isSpeech(dev):
      curValue = self.device_man.getCmdLevel(dev, 
                                             AEOutput.Constants.CONTEXT_VOICE)
    return curValue
  
  def getMaxVoice(self, dev=None):
    '''
    Gets the maximum speech voice on the referenced device. When no device 
    is specified, the default output device is used.
    
    A return of -1 indicates the device does not have a voice value, 
    e.g. a Braille device.

    @param dev: The device to change that defaults to None for the default 
      output device.
    @type dev: L{AEOutput.Speech}
    @return: The current maximum voice.
    @rtype: integer
    '''
    maxValue = -1
    if dev is None:
      dev = self.device_man.getDefaultOutput()
    # rate is only valid for speech devices
    if AEOutput.isSpeech(dev):
      maxValue = self.device_man.getMaxLevel(dev, 
                                             AEOutput.Constants.CONTEXT_VOICE)
    return maxValue
  