'''
Defines a class responsible for handling events in the currently focused view.

@var VIEW_EVENTS: Event names that indicate a change of view
@type VIEW_EVENTS: list
@var DEFAULT_EVENTS: Names of events that will be dispatched by the 
  L{ViewManager} by default
@type DEFAULT_EVENTS: list
@var AE_MAP: Mapping from L{AEEvent}s to raw events that must be registered
  to generate them
@type AE_MAP: dictionary

@todo: PP: the AT-SPI specific code in this module should be moved to an adapter
  supporting an interface for mapping L{AEEvent}s to raw event names (e.g. 
  IEventFactory)

@author: Peter Parente
@author: Pete Brunet
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import logging
import pyLinAcc
from AEEvent import *
from POR import POR
from LSRInterfaces import *
from Walker import AccessibleWalker

log = logging.getLogger('View')

# events that almost certainly indicate a view change
VIEW_EVENTS = ['window:activate', 'window:deactivate', 'window:create',
               'window:destroy']

# events that we should always watch, regardless of Task interests
DEFAULT_EVENTS = [ViewChange, FocusChange, CaretChange, PropertyChange]

# opt: AE to raw event mapping, only registers for raw events when desired
AE_MAP = {FocusChange : ['focus'],
          CaretChange : ['object:text-caret-moved',
                         'object:text-changed'],
          ChildrenChange : ['object:children-changed'],
          PropertyChange : ['object:property-change'],
          SelectorChange : ['object:selection-changed',
                            'object:active-descendant-changed'],
          StateChange : ['object:state-changed'],
          TableChange : ['object:row-inserted',
                         'object:row-deleted',
                         'object:row-reordered',
                         'object:column-inserted',
                         'object:column-deleted',
                         'object:column-reordered'],
          ViewChange : VIEW_EVENTS
          }

# unmapped
# object:text-attributes-changed (maybe to PropertyChange)
# object:attribute-changed (maybe to PropertyChange)
# object:bounds-changed (maybe to ScreenChange)
# object:visible-data-changed (maybe to ScreenChange)
# object:text-bounds-changed (maybe to ScreenChange????)
# object:model-changed (maybe to ModelChange?)
# document:load-complete (all to DocumentChange? or maybe ModelChange?)
# document:reload
# document:load-stopped
# document:content-changed
# document:attributes-changed
# terminal:line-changed (should probably map to CaretChange)
# terminal:columncount-changed (this is really like object:col-inserted)
# terminal:linecount-changed (and this like object:row-inserted)
# terminal:application-changed (maybe ViewChange?)
# terminal:charwidth-changed (ScreenChange?)

class ViewManager(object):
  '''
  Stores a root accessible representing the active view when an event occurs
  indicating that the view may have changed.
  
  @ivar event_manager: Reference to the L{EventManager}
  @type event_manager: L{EventManager}
  @ivar active_view: Root L{POR} representing the active view
  @type active_view: L{POR}
  '''
  def __init__(self, ae):
    '''
    Initializes instance variables that will store a reference to the 
    L{EventManager} and active view to None.
    
    @param ae: Reference to the L{AccessEngine}, currently unused
    @type ae: L{AccessEngine}
    '''
    self.event_manager = None
    self.active_view = None
      
  def init(self, event_man, **kwargs):
    '''
    Stores a reference to the L{EventManager}. Registers with the 
    L{EventManager} using the L{EventManager.EventManager.addClient} to receive 
    raw L{pyLinAcc} events related to the active view. Also registers 
    for raw events that may indicate that this L{ViewManager} needs to manage a
    new view. Called by L{AccessEngine} at startup.
    
    @param event_man: Reference to the L{EventManager}
    @type event_man: L{EventManager}
    '''
    # store the event manager so we can post events later
    self.event_manager = event_man
    # register for events that almost certainly mean the view has changed
    self.event_manager.addClient(self.onProbableViewChange, *VIEW_EVENTS)
    # track how many dependencies we have on a certain kind of event
    #self.event_manager.addClient(self.onRawEvent, *ALL_EVENTS)
    for kind in DEFAULT_EVENTS:
      self.setEventInterest(kind, True)
    
  def close(self):
    '''Does nothing.'''
    pass
    
  def setEventInterest(self, kind, wants):
    '''
    Sets or unsets an interest in a particular kind of L{AEEvent}. This
    information is used to register or unregister for raw events on-demand as
    an optimization.
    
    @param kind: Indicates the type of L{AEEvent} some part of the system wants
      to be able to process
    @type kind: L{AEEvent} class
    @param wants: Does the system want to process the given kind of L{AEEvent}?
    @type wants: boolean
    ''' 
    if wants:
      self.event_manager.addClient(self.onRawEvent, *AE_MAP[kind])
    else:
      self.event_manager.removeClient(self.onRawEvent, *AE_MAP[kind])

  def getView(self):
    '''
    Gets the root accessible of the active view.
    
    @return: Root L{POR} of the active view
    @rtype: L{POR}
    '''
    return self.active_view
  
  def setView(self, accessible):
    '''    
    Stores the root accessible corresponding to the received event to
    represent the active view.
    
    @param accessible: Event source that triggered the change of view. Usually a
      top level window or panel.
    @type accessible: L{pyLinAcc.Accessible}
    @return: Was a new view root set or not?
    @rtype: boolean
    '''
    if accessible is None:
      if self.active_view is None:
        return False
      # the view was lost, set it to None
      self.active_view = None
      return True
    else:
      # walk to the first POR
      root = AccessibleWalker(POR(accessible)).getFirstPOR()
      # see if it is different from our current root POR
      if root != self.active_view:
        # save a new view because the current view is invalid
        self.active_view = root
        return True
    return False
   
  def onRawEvent(self, event):
    '''
    Based on a raw L{pyLinAcc.Event.Event} posts L{AEEvent}s to the active
    L{Tier} through the L{EventManager}.
    
    @param event: Event indicating some change
    @type event: L{pyLinAcc.Event.Event}
    '''
    # only forward events if there is a view
    if self.active_view is not None:
      eh = IEventHandler(event.source)
      ae_events = eh.getAEEvents(event)
      if ae_events is not None:
        # getAEEvents always returns a tuple now
        self.event_manager.postEvents(*ae_events)
    
  def onProbableViewChange(self, event):
    '''        
    Creates a L{AEEvent.ViewChange} event in response to a window activation
    or deactivation. Also responds to create or destroy events from metacity.
    Called in response to any of the raw events listed in
    L{VIEW_EVENTS} and registered with the L{EventManager} in the
    L{ViewManager.init} method when this object was created. All events in
    L{VIEW_EVENTS} are monitored as indicating view changes.

    @param event: Event that may indicate the view has changed
    @type event: L{pyLinAcc.Event.Event}
    '''
    try:
      # make sure we can get the event type and role
      et = event.type.major
      role = event.source.getRole()
    except:
      return
    
    if et == 'activate':
      if self.setView(event.source):
        e = ViewChange(self.active_view, True)
        self.event_manager.postEvents(e)
      return
    elif self.active_view is not None and et == 'deactivate':
      e = ViewChange(self.active_view, False)
      self.event_manager.postEvents(e)
      return
    
    # metacity brings up a "special little window" that does not fire an 
    # activate or deactivate so we have to watch for a create event specifically
    # for the switch app and switch desktop; handling the window manager 
    # specifically here since it is such a weird case
    try:
      app_name = event.source.getApplication().name
      is_window = (role == pyLinAcc.Constants.ROLE_WINDOW)
    except AttributeError:
      return
    if app_name == 'metacity' and is_window:
      if et == 'create':
        if self.setView(event.source):
          e = ViewChange(self.active_view, True)
          self.event_manager.postEvents(e)
        elif et == 'destroy':
          e = ViewChange(self.active_view, False)
          self.event_manager.postEvents(e)