'''
Defines the base class for all speech output devices.

@author: Brett Clippingdale
@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import time
import Base
from ThreadProxy import *

class Audio(Base.AEOutput):
  ''' 
  Defines the base class for all speech output devices.
  '''
  def getProxy(self):
    '''
    Looks at the L{USE_THREAD} flag to see if the device implementing this 
    interface wants a thread proxy or not.
    
    @return: self or L{ThreadProxy.AudioThreadProxy}
    @rtype: L{AEOutput}
    '''
    if self.USE_THREAD == True:
      return AudioThreadProxy(self)
    elif self.USE_THREAD == False:
      return self
    else:
      raise NotImplementedError
  
  def sendStringSync(self, text, style, stop=True):
    '''
    Buffers a complete string to send to the device synchronously.

    This should B{not} be used in place of L{sendString} since this will not
    return until the string is finished being output. This is provided B{only}
    for the convenience of utility writers. Uses L{sendStop}, L{sendString}, 
    L{sendTalk}, and then sleeps until L{isActive} returns False.

    @param text: String to send to the device
    @type text: string
    @param style: Style on which this string should be output; None implies some
      reasonable default should be used
    @type style: L{AEOutput.Style}
    @param stop: Stop current output before doing this output?
    @type stop: boolean
    '''
    proxy = self.getProxy()
    if stop:
      proxy.sendStop()
    proxy.sendString(text, style)
    proxy.sendTalk()

    # wait until done speaking; this blocks the main thread -- be careful
    while proxy.isActive():
      time.sleep(0.1)
      
