'''
Defines classes for representing device capabilities and styles. 

@author: Brett Clippingdale
@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import AEState

class Style(AEState.AEState):
  '''
  Base class for all style classes that will contain a set of properties 
  defining how output should be presented on an L{AEOutput} device. Implements 
  the flyweight pattern such that if this style object does not define a 
  particular property, it may be retrieved from another style object instead.
  
  Derives from L{AEState} such that all style properties can be persisted to
  disk or exposed as configurable user settings if desired.
  
  @ivar style: Another style object to which this one should dispatch attribute
    lookups if attributes are not defined in this object
  @type style: L{Style}
  '''
  def __init__(self, style=None):
    '''
    Initializes the style object to which to dispatch.
    
    @param style: 
    @type style: L{Style}
    '''
    self.style = style
    
  def __getattr__(self, name):
    '''
    Tries to get the named attribute. Dispatches the request to L{style} if it
    is not found in this object and L{style} is not None.
    
    @param name: Name of the attribute to fetch
    @type name: string
    @return: Attribute value
    @rtype: object
    '''
    try:
      return self.__dict__[name]
    except KeyError:
      return getattr(self.style, name)

  def __str__(self):
    '''
    @return: Contents of this object's __dict__
    @rtype: string
    '''
    return str(self.__dict__)
  
  def copy(self):
    '''
    Makes a fast copy of this object's __dict__.
    
    @return: New style object
    @rtype: L{Style}
    '''
    other = self.__class__()
    other.__dict__.update(self.__dict__)
    return other

class AudioStyleDefault(Style):
  '''
  Defines the basic style attributes for all L{AEOutput.Audio} devices. Most
  attributes are defaulted to None indicating they are not supported. It is
  the job for a specific output device definition to override these values with
  its own defaults.
  
  @ivar Volume: Current volume
  @type Volume: integer
  @ivar MaxVolume: Maximum supported volume
  @type MaxVolume: integer
  @ivar MinVolume: Minimum supported volume
  @type MinVolume: integer
  @ivar Rate: Current speech rate
  @type Rate: integer
  @ivar MaxRate: Maximum supported speech rate
  @type MaxRate: integer
  @ivar MinRate: Minimum supported speech rate
  @type MinRate: integer
  @ivar Pitch: Current baseline pitch
  @type Pitch: integer
  @ivar MaxPitch: Maximum supported pitch
  @type MaxPitch: integer
  @ivar MinPitch: Minimum supported pitch
  @type MinPitch: integer
  @ivar Voice: Current voice number
  @type Voice: integer
  @ivar MaxVoice: Maximum voice number
  @type MaxVoice: integer
  @ivar MinVoice: Minimum voice number
  @type MinVoice: integer
  @ivar Position: Current spatial position
  @type Position: 3-tuple of float
  @ivar Language: Current language specified as ISO language and country codes
  @type Language: string
  @type Instrument: Current MIDI instrument number
  @ivar Instrument: integer
  @ivar MaxInstrument: Maximum supported instrument number
  @type MaxInstrument: integer
  @ivar MinInstrument: Minimum supported instrument number
  @type MinInstrument: integer
  @ivar Continuous: Is output continuous (i.e. looping)?
  @type Continuous: boolean
  @ivar Channel: Channel number indicating concurrency of output. Styles 
    sharing a channel number are output sequentially. Styles having different 
    channel numbers are played concurrently if possible.
  @type Channel: integer
  @ivar Stoppable: Can output be stopped?
  @type Stoppable: boolean
  @ivar Speakable: Can output be spoken?
  @type Speakable: boolean
  @ivar Soundable: Can output be played as a non-speech sound?
  @type Soundable: boolean
  @ivar Mute: Is all output inhibited?
  @type Mute: boolean
  @ivar Gender: Speech vocal tract gender
  @type Gender: integer
  @ivar MaxGender: Maximum supported gender constant
  @type MaxGender: integer
  @ivar MinGender: Minimum supported gender constant
  @type MinGender: integer
  @ivar Aspiration: Speech breathiness
  @type Aspiration: integer
  @ivar MaxAspiration: Maximum supported aspiration
  @type MaxAspiration: integer
  @ivar MinAspiration: Minimum supported aspiration
  @type MinAspiration: integer
  @ivar Frication: Emphasis of fricatives in speech
  @type Frication: integer
  @ivar MaxFrication: Maximum supported frication
  @type MaxFrication: integer
  @ivar MinFrication: Minimum supported frication
  @type MinFrication: integer
  @ivar Intonation: Speech pitch variation
  @type Intonation: integer
  @ivar MinIntonation: Maximum supported pitch variation
  @type MinIntonation: integer
  @ivar MaxIntonation: Minimum supported pitch variation
  @type MaxIntonation: integer
  @ivar HeadSize: Speech speaker head size for reverberation
  @type HeadSize: integer
  @ivar MaxHeadSize: Maximum supported head size
  @type MaxHeadSize: integer
  @ivar MinHeadSize: Minimum supported head size
  @type MinHeadSize: integer
  '''
  Position = None
  Language = None
  Channel = 0
  MinChannel = 0
  MaxChannel = 0
  Stoppable = True
  Speakable = True
  Soundable = True
  Mute = False
  Continuous = False
  Instrument = None
  MaxInstrument = None
  MinInstrument = None
  Volume = None
  MaxVolume = None
  MinVolume = None
  Rate = None
  MaxRate = None
  MinRate = None
  Pitch = None
  MaxPitch = None
  MinPitch = None
  Voice = None
  MaxVoice = None
  MinVoice = None
  Gender = None
  MaxGender = None
  MinGender = None
  Aspiration = None
  MaxAspiration = None
  MinAspiration = None
  Frication = None
  MaxFrication = None
  MinFrication = None
  Intonation = None
  MaxIntonation = None
  MinIntonation = None
  HeadSize = None
  MinHeadSize = None
  MaxHeadSize = None
