'''
Defines the base class for all choosers in LSR.

@todo: PP: support singleton flag

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

class AEChooser(object):
  '''  
  Most abstract base class for all L{AEChooser} dialogs.
  
  This class is abstract as most of its methods raise NotImplementedError and 
  need to be overriden in subclasses.
  
  @cvar SINGLETON: Should only one instance of this chooser live at a time? 
    This flag should be overriden by a subclass to indicate whether the
    chooser is a singleton or not.
  @type SINGLETON: boolean
  @cvar OK: Indicates the chooser is completing and its options should be
    applied
  @type OK: integer
  @cvar CANCEL: Indicates the chooser is canceling and its options should be
    ignored
  @type CANCEL: integer
  @cvar APPLY: Indicates the chooser options should be applied immediately 
    with no changes to its state
  @type APPLY: integer
  @ivar event_man: Reference to the event manager to which an event will be
    posted when the user indicates the dialog is completed, cancelled, or
    the current optionss applied
  @type event_man: L{EventManager}
  '''
  SINGLETON = False
  CANCEL = 0
  APPLY = 1
  OK = 2
  
  def __init__(self):
    '''
    Initializes the L{event_man} variable to None.
    '''
    self.event_man = None
    
  def __call__(self, event_man):
    '''
    Stores a reference the L{EventManager} that will be notified by 
    L{_postEvent).
    
    @param event_man: Reference to the event manager to which an event will be
      posted when the user indicates the dialog is completed, cancelled, or
      the current optionss applied
    @type event_man: L{EventManager}
    @raise ValueError: When the L{SINGLETON} flag is set and an instance of 
      this class already exists
    '''
    self.event_man = event_man
  
  def init(self):
    '''
    Initializes the chooser. Should enable the chooser for interaction with the
    user.
    
    @raise NotImplementedError: When not overridden by a subclass
    '''
    raise NotImplementedError
    
  def close(self):
    '''
    Closes the chooser. Should prevent further chooser interaction with the 
    user.
    
    @raise NotImplementedError: When not overridden by a subclass
    '''
    raise NotImplementedError
  
  def getName(self):
    '''
    Gets the name of the chooser.    
    
    @return: Human readable name of the chooser
    @rtype: string
    @raise NotImplementedError: When not overriden by a subclass
    '''
    raise NotImplementedError
  
  def update(self, **kwargs):
    '''
    Does an update of some aspect of the L{AEChooser}. A subclass can override
    this method to support updates from observers of L{AEEvent.ChooserChange}
    events.
    
    @param kwargs: Arbitrary data given by the observer. The L{AEChooser}
      implementor should strong-name keyword params of interest.
    @type kwargs: dictionary
    '''
    pass
  
  def _postEvent(self, kind, **kwargs):
    '''    
    Posts an L{AEEvent.ChooserChange} event to the L{EventManager}. The kind of
    change can be one of L{OK}, L{APPLY}, and L{CANCEL}. Any keyword arguments
    will be delivered to an observer. The event will also include a reference
    to this L{AEChooser} such that an observer can call the L{update} method on
    it.
    
    @param kind: Kind of event, one of L{OK}, L{APPLY}, L{CANCEL}
    @type kind: integer
    @param kwargs: Arbitrary data to be delivered with the event
    @type kwargs: dictionary
    '''
    self.event_man.postEvent(AEEvent.ChooserChange(self, kind, **kwargs))