'''
Defines a L{Task} to execute in response to a L{AEEvent.ViewChange}.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import Base, Constants
import AEEvent

class ViewTask(Base.Task):
  '''
  Executed when a view change occurs.
  
  This class registers its name and whether it should be monitored by default in
  an L{AEMonitor.EventMonitor} using the L{Constants.registerTaskType} function
  when this module is first imported. The L{AEMonitor.EventMonitor} can use this
  information to build its menus.
  '''
  Constants.registerTaskType('ViewTask', True)
  
  def getType(self):
    '''
    @return: Type of L{AEEvent} this L{Task} wants to handle
    @rtype: class
    '''
    return AEEvent.ViewChange
  
  def update(self, por, title, gained, layer, **kwargs):
    '''
    Updates this L{Task} in response to a consumed view change event. Called by 
    L{Tier.Tier._executeTask}.
    
    @param por: Point of regard to the root of the new view
    @type por: L{POR}
    @param title: Title of the view
    @type title: string
    @param gained: True if gained, False if lost, None if mutated
    @type gained: boolean
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    pass

  def execute(self, por, title, gained, layer, **kwargs):
    '''
    Executes this L{Task} in response to a view change event. Called by 
    L{Tier.Tier._executeTask}.
    
    @param por: Point of regard to the root of the new view
    @type por: L{POR}
    @param title: Title of the view
    @type title: string
    @param gained: True if gained, False if lost, None if mutated
    @type gained: boolean
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: True to allow other L{Task}s to process this event
    @rtype: boolean
    '''
    if gained:
      return self.executeGained(por=por, title=title, layer=layer, **kwargs)
    elif gained is None:
      return self.executeMutated(por=por, title=title, layer=layer, **kwargs)
    else:
      return self.executeLost(por=por, title=title, layer=layer, **kwargs)
    
  def executeGained(self, por, title, layer, **kwargs):
    '''
    Executes this L{Task} in response to a view gained event. Called by 
    L{execute}.
    
    @param por: Point of regard to the root of the new view
    @type por: L{POR}
    @param title: Title of the view
    @type title: string
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: True to allow other L{Task}s to process this event
    @rtype: boolean
    '''
    return True
  
  def executeLost(self, por, title, layer, **kwargs):
    '''    
    Executes this L{Task} in response to a view lost event. Called by
    L{execute}.
    
    @param por: Point of regard to the root of the lost view
    @type por: L{POR}
    @param title: Title of the view
    @type title: string
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: True to allow other L{Task}s to process this event
    @rtype: boolean
    '''
    return True
  
  def executeMutated(self, por, title, layer, **kwargs):
    '''
    Executes this L{Task} in response to a view mutated event. Called by 
    L{execute}.
    
    @param por: Point of regard to the root of the mutated view
    @type por: L{POR}
    @param title: Title of the view
    @type title: string
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: True to allow other L{Task}s to process this event
    @rtype: boolean
    '''
    return True