'''
Defines a class responsible for managing all input and output devices and
monitors.

@author: Larry Weiss
@author: Peter Parente
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import logging, sys
import AEOutput, AEInput, AEEvent, AEMonitor, UIRegistrar
from LSRInterfaces import implements
from UIRegistrar import DEVICE, MONITOR
from i18n import _
import LSRConstants

log = logging.getLogger('Device')

class DeviceManager(object):
  '''
  Creates and manages the devices and monitors use by LSR. Keeps a list of each
  and also defines a "default" for devices. Provides a common interface to all
  output and input devices and mirrors I/O to the monitors. Provides a mapping
  from output styles to semantic concepts in the user interface to allow
  customization of how information is presented.
  
  @ivar event_manager: L{EventManager} to which this L{DeviceManager} can post 
    L{AEEvent}s 
  @type event_manager: L{EventManager}
  @ivar out_devs: All output devices registered and initialized
  @type out_devs: list
  @ivar in_devs: All input devices registered and initialized
  @type in_devs: list
  @ivar in_mons: Collection of monitors to notify about input
  @type in_mons: L{AEMonitor.MonitorCollection}
  @ivar out_mons: Collection of monitors to notify about output
  @type out_mons: L{AEMonitor.MonitorCollection}
  @ivar def_output: The "default" output device
  @type def_output: L{AEOutput}
  @ivar def_input: The "default" input device
  @type def_input: L{AEInput}
  @ivar acc_eng: The AccessEngine reference that created this class
  @type acc_eng: L{AccessEngine}
  @ivar marks: Maps index markers on devices to 
  @type marks: dictionary
  '''
  def __init__(self, acc_eng):
    '''
    Creates the empty lists for devices and monitors.

    @param acc_eng: The AccessEngine reference that created this class
    @type acc_eng: L{AccessEngine}
    '''
    self.out_devs = []
    self.in_devs = []
    self.in_mons = AEMonitor.MonitorCollection()
    self.out_mons = AEMonitor.MonitorCollection()
    self.def_output = None
    self.def_input = None
    self.acc_eng = acc_eng
    self.event_manager = None
    self.settings_manager = None
    self.marks = {}

  def init(self, event_man, sett_man, **kwargs):
    '''
    Called by L{AccessEngine} at startup.

    @param event_man: L{EventManager} to which this L{DeviceManager} can
      post L{AEEvent}s 
    @type event_man: L{EventManager}
    @param sett_man: L{SettingsManager} which this L{DeviceManager} can use to
      load and persist device settings
    @type sett_man: L{EventManager}
    @param kwargs: References to managers not of interest here
    @type kwargs: dictionary
    '''
    self.event_manager = event_man
    self.settings_manager = sett_man
    
    # load all I/O monitor classes to be created at startup
    mons = UIRegistrar.loadAssociated(MONITOR, self.acc_eng.getProfile())
    self.addMonitors(*mons)
        
    # load all device classes to be created at startup
    self.loadDevices()
    
  def loadDevices(self):
    '''
    Iterates over all devices in the active profile and loads ones that provide
    functionality different from those already loaded.
    '''
    devs = UIRegistrar.loadAssociated(DEVICE, self.acc_eng.getProfile())
    # register all devices, making the first input and output the defaults
    for d in devs:
      # call registerDevice which will automatically set defaults
      try:
        self.registerDevice(d)
      except (AEOutput.AEOutputError, AEInput.AEInputError):
        log.debug('could not initialize device: %s', d.getName())
      except NotImplementedError, e:
        log.debug('device %s does not provide a new interface: %s' % 
                 (d.getName(), str(e)))
    
    # warn if we didn't initialize at least one output device
    if self.def_output is None:
      log.warn(_('no output available'))
    if self.def_input is None:
      log.warn(_('no input available'))
      
  def unloadDevices(self):
    '''
    Unloads all current devices.
    '''
    for d in self.out_devs:
      self.unregisterDevice(d)
    # @todo: PP: support unloading of input devices
    #for d in self.in_devs:
      #self.unregisterDevice(d)
    self.def_output = None
    #self.def_input = None
    
  def close(self):
    '''
    Shuts down this manager and all its registered L{AEOutput} and L{AEInput}
    devices and L{AEMonitor}s. Saves all style information for currently 
    loaded devices.
    '''
    for d in self.out_devs:
      self.unregisterDevice(d)
    for d in self.in_devs:
      self.unregisterDevice(d)
    self.def_output = None
    self.def_input = None
    self.in_mons.clear()
    self.out_mons.clear()

  def sendStop(self, dev, sem, layer):
    '''
    Sends the stop command to the referenced output device. When the device has
    not been registered or is None, the stop command is sent to the default
    output device.

    @param dev: The device to send the stop command or None to send to L{the 
      default output device<setDefaultOutput>}.
    @type dev: L{AEOutput}
    @param sem: Semantic description of the information to stop None to 
      indicate stopping all output
    @type sem: integer
    @param layer: Layer on which the event occurred, one of 
      L{Task.LAYER_FOCUS}, L{Task.LAYER_TIER}, or L{Task.LAYER_BACKGROUND}
    @type layer: integer
    '''
    # when dev is None assume default output
    dev = dev or self.def_output
    # notify monitors
    self.out_mons.show(event=OutputEvent.STOP, dev=dev, sem=sem, layer=layer)
    if dev is None: return
    if sem is None and layer is None:
      # send a global stop
      dev.sendStop(None)
    elif sem is None:
      # find all styles associated with the given layer
      all = set([dev.getStyle((sem, layer)) for sem in
                 LSRConstants.SEMANTIC_STYLES.keys()])
      map(dev.sendStop, all)
    else:
      s = dev.getStyle((sem, layer))
      dev.sendStop(s)

  def sendTalk(self, dev, sem, layer):
    '''
    Tells the specified output device to send buffered data. When the device
    has not been registered or is None, the talk command is sent to the
    default output device.

    @param dev: The device to send the talk command or None to send to L{the 
      default output device<setDefaultOutput>}
    @type dev: L{AEOutput}
    @param sem: Semantic information to start outputting or None to indicate 
      that all buffered information should be output
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.LAYER_FOCUS},
      L{Task.LAYER_TIER}, or L{Task.LAYER_BACKGROUND}
    @type layer: integer
    '''
    # when dev is None assume default output
    dev = dev or self.def_output
    # notify monitors
    self.out_mons.show(event=OutputEvent.TALK, dev=dev, sem=sem, layer=layer)
    if dev is None: return
    if sem is None:
      s = None
    else:
      s = dev.getStyle((sem, layer))
    dev.sendTalk(s)

  def sendFilename(self, dev, filename, sem, layer, talk=True):
    '''
    Sends the filename to the specified output device. When the device has not
    been registered or is None, the string is sent to the default output
    device.

    @param dev: The device to send the string or None to send to L{the default 
      output device<setDefaultOutput>}
    @type dev: L{AEOutput}
    @param filename: The filename to send to the referenced device
    @type filename: string
    @param sem: Semantic description of the text to send or None to indicate the
      information is void of any known semantic
    @type sem: integer 
    @param layer: Layer on which the event occurred, one of L{Task.LAYER_FOCUS},
      L{Task.LAYER_TIER}, or L{Task.LAYER_BACKGROUND}
    @type layer: integer
    @param talk: Defaults to True to indicate that L{sendTalk} should be called
      after all characters are sent
    @type talk: boolean
    '''
    # when dev is None assume default output
    dev = dev or self.def_output
    # notify monitors
    self.out_mons.show(event=OutputEvent.FILENAME, text=filename, dev=dev, 
                       sem=sem, layer=layer)
    if dev is None: return
    # find the style
    style = dev.getStyle((sem, layer))
    # respect muting
    if style.Mute: return
    try:
      dev.sendFilename(filename, style)
      if talk:
        self.sendTalk(dev, sem, layer)
    except NotImplementedError:
      pass
    # make the style as clean
    style.makeClean()
    
  def sendString(self, dev, text, sem, layer, talk=True, por=None):
    '''
    Sends the string to the specified output device. When the device has not
    been registered or is None, the string is sent to the default output
    device. The string is sent with a L{AEOutput.Base.AEOutput.sendString} call
    followed by L{sendTalk} when the talk parameter is True.
    
    @todo: PP: collect word markers

    @param dev: The device to send the string or None to send to L{the default 
      output device<setDefaultOutput>}
    @type dev: L{AEOutput}
    @param text: The text to send to the referenced device
    @type text: string
    @param sem: Semantic description of the text to send or None to indicate the
      information is void of any known semantic
    @type sem: integer 
    @param layer: Layer on which the event occurred, one of 
      L{Task.LAYER_FOCUS}, L{Task.LAYER_TIER}, or L{Task.LAYER_BACKGROUND}
    @type layer: integer
    @param talk: Defaults to True to indicate that L{sendTalk} should be called
      after all characters are sent
    @type talk: boolean
    @param por: Point of regard to the start of the string if one exists or 
      None if the string did not originate from a L{POR}
    @type por: L{POR}
    '''
    # when dev is None assume default output
    dev = dev or self.def_output
    # notify monitors
    self.out_mons.show(event=OutputEvent.STRING, text=text, dev=dev,
                       sem=sem, layer=layer)  
    if dev is None: return
    # get the style
    style = dev.getStyle((sem, layer))
    # respect muting
    if style.Mute: return
    # parse the string using the current style
    parsed = dev.parseString(text, style, por)
    # send all words, pors, and styles
    for word, por, new_style in parsed:
      # send the index and collect the marker
      #try:
        #marker = dev.sendIndex(style)
      #except NotImplementedError:
        #pass
      # send the word string
      dev.sendString(word, new_style)
    if talk:
       self.sendTalk(dev, sem, layer)
    # mark the style as clean
    style.makeClean()

  def sendIndex(self, dev, sem, layer, data=None):
    '''
    Sends the referenced index marker to the referenced device. When the
    device has not been registered or is None, the index marker is sent to the
    default output device.

    @param dev: The device to send the index marker or None to send to L{the 
      default output device<setDefaultOutput>}
    @type dev: L{AEOutput}
    @param sem: Semantic description of the index text to send or None to 
      indicate the information is void of any known semantic
    @type sem: integer
    @param layer: Layer on which the event occurred, one of 
      L{Task.LAYER_FOCUS}, L{Task.LAYER_TIER}, or L{Task.LAYER_BACKGROUND}
    @type layer: integer
    @param data: Data to be forwarded to any observers when the index is 
      reached
    @type data: object
    '''
    # when dev is None assume default output
    dev = dev or self.def_output
    # notify registered monitors
    self.out_mons.show(event=OutputEvent.INDEX, dev=dev, sem=sem,layer=layer)
    if dev is None: return
    # get the style
    s = dev.getStyle((sem, layer))
    # respect muting
    if s.Mute: return
    try:
      # generate a new marker on the device
      mark = dev.sendIndex(s)
    except NotImplementedError:
      # do nothing when indexing not implemented
      return
    # store the mark and associated data for later retrieval
    #self.marks[(dev, mark)] = (sem, layer, data)

  def sendWelcome(self):
    '''Outputs the LSR welcome message on the default output device.'''
    msg = _('welcome to %s version %s, revision %s') % \
        (LSRConstants.NAME, LSRConstants.PROG_VERSION, LSRConstants.PROG_DATE)
    try:
      st = self.def_output.getStyle((LSRConstants.SEM_ITEM, 
                                     AEEvent.LAYER_FOCUS))
    except AttributeError:
      # no output available
      return
    self.def_output.sendString(msg, st)
    self.def_output.sendTalk()
    
  def sendGoodbye(self):
    '''Outputs the LSR goodbye message on the default output device.'''
    try:
      st = self.def_output.getStyle((LSRConstants.SEM_ITEM, 
                                     AEEvent.LAYER_FOCUS))
      self.def_output.sendStringSync(_('lsr exiting'), st)
    except AttributeError:
      pass

  def _isDuplicateDevice(self, dev, output):
    '''
    Checks if the given device is a duplicate in that it provides no new 
    interfaces beyond those provided by the devices already registered.
    
    @param dev: A device class
    @type dev: L{AEInput} or L{AEOutput} class
    @param output: Is the device an output device (True) or input (False)?
    @type output: boolean
    '''
    if output:
      dt = AEOutput.DEVICE_TYPES
      reg_devs = self.out_devs
    else:
      dt = AEInput.DEVICE_TYPES
      reg_devs = self.in_devs
    # get interfaces device implements
    dev_int = (i for i in dt if implements(dev, i))
    # check if some registered device already implements all of these interfaces
    for reg_dev in reg_devs:
      reg_int = (i for i in dt if implements(reg_dev, i))
      if set(reg_int).issuperset(dev_int):
        return True
    return False

  def _registerInputDevice(self, dev):
    '''
    Registers the given device as an input device if the device implements the
    L{AEInput} base class and provides some subinterface not already provided
    by another registered output device. For instance, if a device implementing
    L{AEInput.SystemInput} is already registered, another device implementing 
    just L{AEInput.SystemInput} will not be registered by this method.
    
    @param dev: Device to attempt to register
    @type dev: L{AEInput.AEInput} class
    @return: True if the device was registered, False if it is not an 
      L{AEInput} device, and None if it provides no new interfaces
    @rtype: boolean or None
    '''
    if not implements(dev, AEInput.AEInput):
      # don't register something that isn't an input device
      return False
    if self._isDuplicateDevice(dev, False):
      # don't register a device that provides all the same interfaces as a
      # previous device
      return False
    if dev not in self.out_devs:
      # don't reinitialize a device that was added to out_devs
      dev.init()
    # add DeviceManager as a gesture listener
    dev.addInputListener(self._onGesture)
    # append to the in_devs list after successfully initialized
    self.in_devs.append(dev)
    log.debug('added input device %s', str(dev))
    # first input device registered is default
    if self.def_input is None:
      self.def_input = dev
    return True
  
  def _registerOutputDevice(self, dev):
    '''
    Registers the given device as an output device if the device implements the
    L{AEOutput} base class and provides some subinterface not already provided
    by another registered output device. For instance, if a device implementing
    L{AEOutput.Audio} is already registered, another device implementing just
    L{AEOutput.Audio} will not be registered by this method.
    
    @param dev: Device to attempt to register
    @type dev: L{AEOutput.AEOutput} class
    @return: True if the device was registered, False if it is not an 
      L{AEOutput} device, and None if it provides no new interfaces
    @rtype: boolean or None
    '''
    if not implements(dev, AEOutput.AEOutput):
      # don't register something that isn't an output device
      return False
    dev = dev.getProxy()
    if self._isDuplicateDevice(dev, True):
      # don't register a device that provides all the same interfaces as a
      # previous device
      return None
    if dev not in self.in_devs:
      # don't reinitialize a device that was added to in_devs
      dev.init()
    # add DeviceManager as an index listener when implemented
    try:
      dev.addIndexListener(self._onIndex)
    except NotImplementedError:
      # do nothing when indexing not implemented
      pass
    # append to the out_devs list after successfully initialized
    self.out_devs.append(dev)
    log.debug('added output device %s', str(dev))
    # first output device registered is default
    if self.def_output is None:
      self.def_output = dev

    try:
      # try to load styles for the device
      dev.postInitOutput(self.settings_manager)
    except KeyError:
      # or initialize new output styles
      self._initOutputStyles(dev)
    return True
  
  def _initOutputStyles(self, dev):
    '''
    Initializes styles for an L{AEOutput} device. Calls createDistinctStyles
    on the device to get an initial batch of styles to use to distinguish
    some types of information.
    
    @param dev: Device reference
    @type dev: L{AEOutput}
    '''
    # create styles based on the abilities of this device
    num_layers = len(AEEvent.LAYERS_ALL)
    num_groups = len(LSRConstants.STYLE_GROUP_ALL)
    styles = dev.createDistinctStyles(num_groups, num_layers)
    # create semantics, layer to styles mapping for device
    for layer, i in enumerate(AEEvent.LAYERS_ALL):
      # provide a default style for a semantic of None
      s = styles[i*num_groups]
      dev.setStyle((None, layer), s)
      for sem, grp in LSRConstants.SEMANTIC_STYLES.items():
        # provide a style for (semantic, layer, device)
        s = styles[i*num_groups + (grp % num_groups)]
        dev.setStyle((sem, layer), s)

  def registerDevice(self, dev):
    '''    
    Registers the referenced device based on its one or more interfaces.
    When the interface is determined, the init() method is called on the device.
    Returns true when the reference is of a known type that initializes
    successfully.
    
    @param dev: The device to register
    @type dev: L{AEOutput} or L{AEInput} class
    @raise NotImplementedError: When the device implements none of the required
      interfaces
    @raise AEOutputError: When output device initialization fails
    @raise AEInputError: When input device initialization fails
    '''
    is_out = self._registerOutputDevice(dev)
    is_in = self._registerInputDevice(dev)

    if is_out == False and is_in == False:
      # raise an exception if none of the required interfaces was implemented
      raise NotImplementedError
    
  def unregisterDevice(self, dev):
    '''
    Unregisters a device from both the input and output lists based on its
    capabilities.
    
    @param dev: The device to unregister
    @type dev: L{AEOutput} or L{AEInput} class
    '''
    try:
      dev.close()
    except Exception:
      pass
    try:
      # let an output device save its styles
      dev.postCloseOutput(self.settings_manager)
    except AttributeError:
      pass
    # @todo: PP: allow input devices to save state too
    try:
      self.out_devs.remove(dev)
      log.debug('removed output device %s', dev.getName())
    except ValueError:
      pass
    # @todo: PP: support unloading of input devices
    try:
      self.in_devs.remove(dev)
      log.debug('removed input device %s', dev.getName())
    except ValueError:
      pass
    if self.out_devs:
      self.def_output = self.out_devs[0]
    if self.in_devs:
      self.def_input = self.in_devs[0]

  def addMonitors(self, *monitors):
    '''
    Adds one or more L{AEMonitor}s to the list of monitors to be notified about
    IO events. 
    
    @param monitors: L{AEMonitor}s to notify
    @type monitors: tuple of L{AEMonitor}s
    '''
    self.in_mons.add(InputEvent, monitors)
    self.out_mons.add(OutputEvent, monitors)

  def getMonitors(self):
    '''
    @return: Collections of all loaded input and output L{AEMonitor}s in that
      order
    @rtype: 2-tuple L{AEMonitor.MonitorCollection}
    '''
    return self.in_mons, self.out_mons

  def setDefaultOutput(self, dev):
    '''
    Sets the referenced output device as the default output device. When the
    referenced device is not registered, or does not implement the
    L{AEOutput} interface, this does nothing. 

    @param dev: The input device
    @type dev: L{AEOutput}
    @raise NotImplementedError: When the provided device does not implement
        L{AEOutput.AEOutput}
    '''
    if dev in self.out_devs:
      self.def_output = dev
    elif implements(dev, AEOutput.AEOutput):
      if self.registerDevice(dev):
        self.def_output = dev
    else:
      raise NotImplementedError('Not an output device')

  def getDefaultOutput(self):
    '''
    Returns the current default output device or None when no output devices
    have been registered.

    @return: The current default output device
    @rtype: L{AEOutput}
    '''
    return self.def_output
  
  def getOutputByName(self, name):
    '''
    Gets the L{AEOutput} device registered under the given name.
    
    @param name: Name of the output device
    @type name: string
    @return: Output device or None if not registered
    @rtype: L{AEOutput}
    '''
    for dev in self.out_devs:
      if dev.getClassName() == name:
        return dev
    return None

  def setDefaultInput(self, dev):
    '''
    Sets the referenced input device as the default input device. When the
    referenced device is not registered, or does not implement the
    L{AEInput} interface, this does nothing. 

    @param dev: The input device
    @type dev: L{AEInput}
    @raise NotImplementedError: When the provided device does not implement
        L{AEInput.AEInput}
    '''
    if dev in self.in_devs:
      self.def_input = dev
    elif implements(dev, AEInput.AEInput):
      if self.register(dev):
        self.def_input = dev
    else:
      raise NotImplementedError('Not an input device')

  def getDefaultInput(self):
    '''
    Returns the current default input device or None when no input devices
    have been registered.

    @return: The current default input device
    @rtype: L{AEInput}
    '''
    return self.def_input
  
  def getInputByName(self, name):
    '''
    Gets the L{AEInput} device registered under the given name.
    
    @param name: Name of the input device
    @type name: string
    @return: Input device or None if not registered
    @rtype: L{AEInput}
    '''
    for dev in self.in_devs:
      if dev.getClassName() == name:
        return dev
    return None
  
  def _onGesture(self, gesture, timestamp):
    '''    
    Creates an L{AEEvent} indicating the given gesture was found on a
    registered input device. When executed, the L{AEEvent} will notify the
    L{TierManager} about the gesture and allow it to activate the appropriate
    L{Task} registered to respond to the gesture in the active L{Tier}.
    
    @param gesture: Gesture seen on an input device
    @type gesture: L{AEInput.Gesture}
    @param timestamp: Time at which the event occurred
    @type timestamp: float
    '''
    event = AEEvent.InputGesture(gesture, timestamp)
    self.event_manager.postEvents(event)
    # notify monitors
    self.in_mons.show(event=InputEvent.GESTURE, dev=gesture.getDevice(), 
                      gesture=gesture)
    
  def _onIndex(self, dev, index):
    '''
    @todo: PP: handle index callback markers
    '''
    # notify monitors
    self.out_mons.show(event=OutputEvent.MARKER)
    
  def getDevices(self):
    '''
    @return: References to all loaded devices
    @rtype: list of L{UIElement}
    '''
    return list(set(self.in_devs).union(self.out_devs))
  
  def getStyle(self, dev, sem, layer):
    '''
    Gets the style for semantic/layer of a given device.
    
    @param dev: The device from which to get the style.
    @type dev: L{AEOutput}
    @param sem: The device semantic from which to get the style, or None to
      get the device's default style.
    @type sem: integer 
    @param layer: Layer on which the event occurred, one of 
      L{Task.LAYER_FOCUS}, L{Task.LAYER_TIER}, or L{Task.LAYER_BACKGROUND}
    @type layer: integer
    @return: the L{AEOutput.Style} for the given semantic of a device
    @rtype: L{AEOutput.Style}
    @raise KeyError: When the semantic or layer is invalid
    '''
    dev = dev or self.def_output
    if dev is None: return None
    if sem is None or layer is None:
      return dev.getDefaultStyle()
    return dev.getStyle((sem, layer))

class DeviceEvent(object):
  '''
  Base type for indicating IO events to monitors such that a monitor may 
  select whether to monitor input, output, both, or neither based on this
  type and its subtypes. An L{AEMonitor} may query a subclass of this class
  for its event types and their names in order to construct menus allowing the
  user to select which events should be logged.
  
  @cvar default_types: List of device events that should be monitored by 
    default
  @type default_types: list of string
  @cvar all_types: List of all device events
  @type all_types: list of string
  '''
  default_types = []
  all_types = []
  
  @classmethod
  def getNames(cls):
    '''
    @return: Names of all events
    @rtype: list of string
    '''
    return cls.all_types
  
  @classmethod
  def getDefaults(cls):
    '''
    @return: Recommended names of events to be logged by default
    @rtype: list of string
    '''
    return cls.default_types

class InputEvent(DeviceEvent):
  '''
  Subtype defining constants for input events.
  
  @cvar GESTURE: Gesture was seen on an input device
  @type GESTURE: string
  '''
  GESTURE = 'gesture'
  default_types = []
  all_types = [GESTURE]

class OutputEvent(DeviceEvent):
  '''
  Subtype defining constants for output events.
  
  @cvar STRING: String was sent to an output device
  @type STRING: string
  @cvar FILENAME: Filename was sent to an output device
  @type FILENAME: string
  @cvar TALK: Talk was sent to an output device
  @type TALK: string
  @cvar INDEX: Index was sent to an output device
  @type INDEX: string
  @cvar MARKER: Marker was encountered during output
  @type MARKER: string
  '''
  STRING = 'string'
  TALK = 'talk'
  INDEX = 'index'
  FILENAME = 'filename'
  MARKER = 'marker'
  STOP = 'stop'
  default_types = [STRING, TALK, FILENAME, STOP]
  all_types = [STRING, TALK, FILENAME, MARKER, STOP]