'''
Defines L{Tools} for registering, unregistering, introspecting, and otherwise
manging L{Tier}s, L{Perk}s, and L{Task}s.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import weakref
from Error import *
import Base
import LSRConstants, UIRegistrar

class System(Base.TaskTools):
  '''
  Provides methods for managing L{Tier}s, L{Perk}s, and L{Task}s.
  
  @cvar LOADED: Constant indicating L{UIElement}s loaded
  @type LOADED: integer
  @cvar INSTALLED: Constant indicating L{UIElement}s installed
  @type INSTALLED: integer
  @cvar TIER_ASSOCIATED: Constant indicating L{Perk}s associated with the 
    current L{Tier} in the active profile
  @type TIER_ASSOCIATED: integer
  @cvar ALL_ASSOCIATED: Constant indicating L{UIElement}s associated with the 
    active profile
  @type ALL_ASSOCIATED: integer
  '''
  LOADED = 0
  INSTALLED = 1
  TIER_ASSOCIATED = 2
  ALL_ASSOCIATED = 3
  
  def getProfileName(self):
    '''
    Gets the name of the profile LSR is running under.
    
    @return: Profile name
    @rtype: string
    '''
    return self.acc_eng.getProfile()
  
  def pushPerk(self, *names):
    '''
    Adds one or more L{Perk}s to the top of the stack. If more than one L{Perk}
    is specified, the last specified L{Perk} will be at the top of the stack 
    when this method completes. That is, the behavior of pushing more than one
    L{Perk} at a time is the same as if each L{Perk} were pushed individually.
    
    @param names: Names of L{Perk}s to add
    @type names: string
    '''
    reg = UIRegistrar
    perks = []
    for name in names:
      perk = reg.loadOne(name)
      if perk is not None:
        perks.append(perk)
    self.tier.pushPerk(self.acc_eng, *perks)
    
  def insertPerk(self, index, perk):
    '''
    Adds one L{Perk} to the stack at the insertion index. Negative indices are
    valid per Python list convention.
    
    @param index: Index at which the L{Perk} should be inserted into the stack
      where zero is the bottom
    @type index: integer
    @param perk: L{Perk} to add
    @type perk: L{Perk}
    '''
    reg = UIRegistrar
    perk = reg.loadOne(perk)
    if perk is not None:
      self.tier.insertPerk(self.acc_eng, index, perk)
  
  def popPerk(self, *indices):
    '''
    Removes one or more L{Perk}s from the stack given their indices.
    
    @param indices: Indices of L{Perk}s to remove
    @type indices: list of integer
    @raise IndexError: When the stack is empty
    @raise ValueError: When specified L{Perk} is not found
    '''
    self.tier.popPerk(*indices)
    
  def reloadPerks(self):
    '''
    Reloads all L{Perk}s in the current L{Tier}. Any changes made to the 
    L{Perk} logic will take effect immediately after the reload. This method
    does not refer to the L{Perk}s associated with the current profile. It 
    simply reloads all currently loaded L{Perk}s.
    '''
    reg = UIRegistrar
    # get official registered Perk names for all existing Perks in this Tier
    names = self.getPerkClassNames()
    names.reverse()
    # store Tier and AccessEngine references before deletion
    tier = self.tier
    acc_eng = self.acc_eng
    # clear out existing Perks
    self.tier.clearPerks()
    # restore the Tier reference
    self.tier = tier
    self.acc_eng = acc_eng
    # push all new Perks
    self.pushPerk(*names)
    # throw the refs away permanently
    self.tier = None
    self.acc_eng = None
    
  def _getClassNames(self, kind, which):
    '''
    Internal method for getting class names of all installed or associated
    L{UIElement}s for the given kind.
    
    @note: Installed UIEs that are not importable because of missing 
      dependencies are not listed as installed to maintain parity with 
      L{_getNames} and L{_getDescriptions}.
    @param kind: Kind of L{UIElement} to fetch metadata for, one of 
      L{UIRegistrar.ALL_KINDS}
    @type kind: string
    @param which: Which names to get: L{INSTALLED}, L{TIER_ASSOCIATED}, 
      or L{ALL_ASSOCIATED}
    @type which: integer
    @return: All descriptions
    @rtype: list of string
    '''
    reg = UIRegistrar
    if which == self.INSTALLED:
      names = reg.listInstalled(kind)
    elif which == self.ALL_ASSOCIATED:
      names = reg.listAssociated(kind, self.acc_eng.getProfile())
    elif which == self.TIER_ASSOCIATED and kind == reg.PERK:
      names = reg.listAssociated(kind, self.acc_eng.getProfile(),
                             self.tier.getName())
    # make sure the UIE is loadable before listing it
    return filter(lambda x: reg.loadOne(x) is not None, names)
    
  def _getNames(self, kind, which):
    '''
    Internal method for getting names of all installed or associated
    L{UIElement}s for the given kind.
    
    @param kind: Kind of L{UIElement} to fetch metadata for, one of 
      L{UIRegistrar.ALL_KINDS}
    @type kind: string
    @param which: Which names to get: L{INSTALLED}, L{TIER_ASSOCIATED}, 
      or L{ALL_ASSOCIATED}
    @type which: integer
    @return: All descriptions
    @rtype: list of string
    '''
    reg = UIRegistrar
    if which == self.INSTALLED:
      return [uie.getName() for uie in (reg.loadOne(name) for name in 
                                        reg.listInstalled(kind))
                                        if uie is not None]
    elif which in (self.ALL_ASSOCIATED, self.TIER_ASSOCIATED):
      names = self._getClassNames(kind, which)
      if names is None:
        return None
      return [uie.getName() for uie in (reg.loadOne(name) for name in names)
              if uie is not None]
  
  def _getDescriptions(self, kind, which):
    '''
    Internal method for getting descriptions of all installed or associated
    L{UIElement}s for the given kind.
    
    @param kind: Kind of L{UIElement} to fetch metadata for, one of 
      L{UIRegistrar.ALL_KINDS}
    @type kind: string
    @param which: Which descriptions to get: L{INSTALLED}, L{TIER_ASSOCIATED}, 
      or L{ALL_ASSOCIATED}
    @type which: integer
    @return: All descriptions
    @rtype: list of string
    '''
    reg = UIRegistrar
    if which == self.INSTALLED:
      return [uie.getDescription() for uie in (reg.loadOne(name) for name in 
                                               reg.listInstalled(kind))
                                               if uie is not None]
    elif which in (self.ALL_ASSOCIATED, self.TIER_ASSOCIATED):
      names = self._getClassNames(kind, which)
      if names is None:
        return None
      return [uie.getDescription() for uie in (reg.loadOne(name) 
                                               for name in names)
                                               if uie is not None]

  def getPerkMetadata(self, which=LOADED):
    '''
    Gets metadata for all L{Perk}s in this L{Tier}, all installed L{Perk}s,
    L{Perk}s associated with this L{Tier}, or all L{Perk}s associated with this
    profile depending on the which flag. Includes L{Perk} class names, human
    readable names, and descriptions.
    
    The default value is used if which is invalid.
    
    @todo: PP: optimize
    
    @param which: Which names to get: L{LOADED}, L{INSTALLED}, 
      L{TIER_ASSOCIATED}, or L{ALL_ASSOCIATED}
    @type which: integer
    @return: Class names, names, descriptions
    @rtype: list of 3-tuple of string
    '''
    return zip(self.getPerkClassNames(which),
               self.getPerkNames(which),
               self.getPerkDescriptions(which))

  def getDeviceMetadata(self, which=LOADED):
    '''    
    Gets metadata for all L{AEOutput} and L{AEInput} devices currently loaded
    in, installed in, or associated with the active profile depending on the
    which flag.

    The default value is used if which is invalid.
    
    @todo: PP: optimize
    
    @param which: Which names to get: L{LOADED}, L{INSTALLED}, or
      L{ALL_ASSOCIATED}
    @type which: integer
    @return: Class names, names, descriptions
    @rtype: list of 3-tuple of string
    '''
    return zip(self.getDeviceClassNames(which), 
               self.getDeviceNames(which),
               self.getDeviceDescriptions(which))

  def getMonitorMetadata(self, which=LOADED):
    '''    
    Gets metadata for all L{AEMonitor}s currently loaded in, installed in, or
    associated with the active profile depending on the which flag.

    The default value is used if which is invalid.
    
    @todo: PP: optimize
    
    @param which: Which names to get: L{LOADED}, L{INSTALLED}, or
      L{ALL_ASSOCIATED}
    @type which: integer
    @return: Class names, names, descriptions
    @rtype: list of 3-tuple of string
    '''
    return zip(self.getMonitorClassNames(which), 
               self.getMonitorNames(which),
               self.getMonitorDescriptions(which))
  
  def getChooserMetadata(self, which=ALL_ASSOCIATED):
    '''        
    Gets metadata for all L{AEChooser}s installed in or associated with the
    active profile depending on the which flag.
    
    The default value is used if which is invalid.

    @todo: PP: optimize
    
    @note: Getting loaded chooser names is not currently supported.   
    @param which: Which names to get: L{LOADED}, L{INSTALLED}, or
      L{ALL_ASSOCIATED}
    @type which: integer
    @return: Class names, names, descriptions
    @rtype: list of 3-tuple of string
    '''
    return zip(self.getChooserClassNames(which), 
               self.getChooserNames(which),
               self.getChooserDescriptions(which))

  def getDeviceClassNames(self, which=LOADED):
    '''
    Gets the programmatic class names of all L{AEOutput} and L{AEInput} devices
    currently loaded in, installed in, or associated with the active profile
    depending on the which flag.
    
    The default value is used if which is invalid.
    
    @param which: Which names to get: L{LOADED}, L{INSTALLED}, or 
      L{ALL_ASSOCIATED}
    @type which: integer
    @return: Names of all L{AEOutput} and L{AEInput} classes
    @rtype: list of string
    '''
    if which != self.LOADED:
      return self._getClassNames(UIRegistrar.DEVICE, which)
    else:
      return [dev.getClassName() for dev in self.device_man.getDevices()]
    
  def getDeviceNames(self, which=LOADED):
    '''    
    Gets the human readable names of all L{AEOutput} and L{AEInput} devices
    currently loaded in, installed in, or associated with the active profile
    depending on the which flag.
    
    The default value is used if which is invalid.
    
    @param which: Which names to get: L{LOADED}, L{INSTALLED}, or
      L{ALL_ASSOCIATED}
    @type which: integer
    @return: Names of all L{AEOutput} and L{AEInput} devices
    @rtype: list of string
    '''
    if which != self.LOADED:
      return self._getNames(UIRegistrar.DEVICE, which)
    else:
      return [dev.getName() for dev in self.device_man.getDevices()]
    
  def getDeviceDescriptions(self, which=LOADED):
    '''    
    Gets the descriptions of all L{AEOutput} and L{AEInput} devices
    currently loaded in, installed in, or associated with the active profile
    depending on the which flag.
    
    The default value is used if which is invalid.
    
    @param which: Which descriptions to get: L{LOADED}, L{INSTALLED}, or
      L{ALL_ASSOCIATED}
    @type which: integer
    @return: Names of all L{AEOutput} and L{AEInput} devices
    @rtype: list of string
    '''
    if which != self.LOADED:
      return self._getDescriptions(UIRegistrar.DEVICE, which)
    else:
      return [dev.getDescription() for dev in self.device_man.getDevices()]
  
  def getMonitorClassNames(self, which=LOADED):
    '''    
    Gets the programmatic class names of all L{AEMonitor}s currently loaded in,
    installed in, or associated with the active profile depending on the which 
    flag.
    
    The default value is used if which is invalid.
    
    @param which: Which names to get: L{LOADED}, L{INSTALLED}, or 
      L{ALL_ASSOCIATED}
    @type which: integer
    @return: Names of all L{AEMonitor} classes
    @rtype: list of string
    '''
    if which != self.LOADED:
      return self._getClassNames(UIRegistrar.MONITOR, which)
    else:
      text = []
      imons, omons = self.device_man.getMonitors()
      mons = (imons, omons, self.tier_man.getMonitors(), 
              self.event_man.getMonitors())
      for col in mons:
        for mon in col:
          text.append(mon.getClassNames())
      return text
    
  def getMonitorNames(self, which=LOADED):
    '''    
    Gets the human readable names of all L{AEMonitor}s currently loaded in,
    installed in, or associated with the active profile depending on the which 
    flag.
    
    The default value is used if which is invalid.
    
    @param which: Which names to get: L{LOADED}, L{INSTALLED}, or
      L{ALL_ASSOCIATED}
    @type which: integer
    @return: Names of all L{AEMonitor}s
    @rtype: list of string
    '''
    if which != self.LOADED:
      return self._getNames(UIRegistrar.MONITOR, which)
    else:
      text = []
      imons, omons = self.device_man.getMonitors()
      mons = (imons, omons, self.tier_man.getMonitors(), 
              self.event_man.getMonitors())
      for col in mons:
        for mon in col:
          text.append(mon.getNames())
      return text
    
  def getMonitorDescriptions(self, which=LOADED):
    '''    
    Gets the human readable descriptions of all L{AEMonitor}s currently loaded
    in, installed in, or associated with the active profile depending on the 
    which flag.
    
    The default value is used if which is invalid.
    
    @param which: Which descriptions to get: L{LOADED}, L{INSTALLED}, or
      L{ALL_ASSOCIATED}
    @type which: integer
    @return: Descriptions of all L{AEMonitor}s
    @rtype: list of string
    '''
    if which != self.LOADED:
      return self._getDescriptions(UIRegistrar.MONITOR, which)
    else:
      text = []
      imons, omons = self.device_man.getMonitors()
      mons = (imons, omons, self.tier_man.getMonitors(), 
              self.event_man.getMonitors())
      for col in mons:
        for mon in col:
          text.append(mon.getDescription())
      return text
  
  def getChooserClassNames(self, which=ALL_ASSOCIATED):
    '''    
    Gets the programmatic class names of all L{AEChooser}s installed in or
    associated with the active profile depending on the which flag.
    
    The default value is used if which is invalid.
    
    @note: Getting loaded chooser names is not currently supported.
    @param which: Which names to get: L{INSTALLED} or L{ALL_ASSOCIATED}
    @type which: integer
    @return: Names of all L{AEChooser} classes
    @rtype: list of string
    '''
    if which != self.LOADED:
      return self._getClassNames(UIRegistrar.CHOOSER, which)
    
  def getChooserNames(self, which=ALL_ASSOCIATED):
    '''    
    Gets the human readable names of all L{AEChooser}s installed in or
    associated with the active profile depending on the which flag.
    
    The default value is used if which is invalid.
    
    @note: Getting loaded chooser names is not currently supported.
    @param which: Which names to get: L{INSTALLED} or L{ALL_ASSOCIATED}
    @type which: integer
    @return: Names of all L{AEChooser}s
    @rtype: list of string
    '''
    if which != self.LOADED:
      return self._getNames(UIRegistrar.CHOOSER, which)
    
  def getChooserDescriptions(self, which=ALL_ASSOCIATED):
    '''
    Gets the descriptions of all L{AEChooser}s installed in or associated with
    the active profile depending on the which flag.
    
    The default value is used if which is invalid.
    
    @note: Getting loaded chooser names is not currently supported.
    @param which: Which descriptions to get: L{INSTALLED} or L{ALL_ASSOCIATED}
    @type which: integer
    @return: Descriptions of all L{AEChooser}s
    @rtype: list of string
    '''
    if which != self.LOADED:
      return self._getDescriptions(UIRegistrar.CHOOSER, which)

  def getPerkClassNames(self, which=LOADED):
    '''
    Gets the programmatic class names of all L{Perk}s in this L{Tier}, all
    installed L{Perk}s, L{Perk}s associated with this L{Tier}, or all L{Perk}s 
    associated with this profile depending on the which flag.
    
    The default value is used if which is invalid.
    
    @param which: Which names to get: L{LOADED}, L{INSTALLED}, 
      L{TIER_ASSOCIATED}, or L{ALL_ASSOCIATED}
    @type which: integer
    @return: Names of all L{Perk} classes
    @rtype: list of string
    '''
    if which != self.LOADED:
      return self._getClassNames(UIRegistrar.PERK, which)
    else:
      return [perk.getClassName() for perk in self.tier.getPerks()]

  def getPerkNames(self, which=LOADED):
    '''    
    Gets the human readable names of all L{Perk}s in this L{Tier}, all
    installed L{Perk}s, L{Perk}s associated with this L{Tier}, or all L{Perk}s 
    associated with this profile depending on the which flag. 
    
    The default value is used if which is invalid.
    
    @param which: Which names to get: L{LOADED}, L{INSTALLED}, 
      L{TIER_ASSOCIATED}, or L{ALL_ASSOCIATED}
    @type which: integer
    @return: Names of all L{Perk}s
    @rtype: list of string
    '''
    if which != self.LOADED:
      return self._getNames(UIRegistrar.PERK, which)
    else:
      return [perk.getName() for perk in self.tier.getPerks()]
  
  def getPerkDescriptions(self, which=LOADED):
    '''    
    Gets the human readable descriptions of all L{Perk}s in this L{Tier}, all
    installed L{Perk}s, L{Perk}s associated with this L{Tier}, or all L{Perk}s 
    associated with this profile depending on the which flag.
    
    The default value is used if which is invalid.
    
    @param which: Which descriptions to get: L{LOADED}, L{INSTALLED}, 
      L{TIER_ASSOCIATED}, or L{ALL_ASSOCIATED}
    @type which: integer
    @return: Descriptions of all L{Perk}s
    @rtype: list of string
    '''
    if which != self.LOADED:
      return self._getDescriptions(UIRegistrar.PERK, which)
    else:
      return [perk.getDescription() for perk in self.tier.getPerks()]    
    
  def registerTask(self, task):
    '''
    Universal task registration method. Uses known interfaces to try to 
    register the given L{Task} as an event task, an input task, a timer task,
    etc. All information for configuring the L{Task} is contained in the 
    task itself.
    
    @param task: Task to register
    @type task: L{Task.Base.Task}
    @raise NotImplementedError: When the L{Task} does not properly implement
      the interfaces it should
    '''
    count = 0
    try:
      # first try as an event task
      self.perk.registerEventTask(task, *task.getEventLayers())
      count += 1
    except AttributeError:
      pass
    try:
      # then try as a timer task
      self.perk.registerTimerTask(task, task.getTimerInterval())
      count += 1
    except AttributeError:
      pass
    try:
      # now try as a named task
      self.perk.registerNamedTask(task, task.getIdentity())
      count += 1
    except (AttributeError, ValueError):
      pass
    if count == 0:
      # not any kind of known task
      raise NotImplementedError
    
  def unregisterTask(self, task):
    '''
    Universal task unregistration method. Uses known interfaces to try to 
    unregister the given L{Task} as an event task, an input task, a timer task,
    etc. All information for configuring the L{Task} is contained in the 
    task itself.
    
    @param task: Task to unregister
    @type task: L{Task.Base.Task}
    @raise NotImplementedError: When the L{Task} does not properly implement
      the interfaces it should
    '''
    count = 0
    perk = self.perk
    try:
      # first try as an event task
      perk.unregisterEventTask(task, *task.getEventLayers())
      count += 1
    except AttributeError:
      pass
    try:
      # then try as a timer task
      # make sure it's a timer task first
      task.getInterval
      perk.unregisterTimerTask(task)
    except AttributeError:
      pass
    try:
      # now try as a named task
      perk.unregisterNamedTask(task.getIdentity())
      count += 1
    except (AttributeError, ValueError):
      pass
    if not count:
      # not any kind of known task
      raise NotImplementedError
    
  def getTierNamedTask(self, name):
    '''
    Gets the L{Task} with the given name if it is registered anywhere in the
    current L{Tier}. If no L{Task} is registered under the given name, returns 
    None.
    
    Calls L{Tier.Tier.getNamedTask} to search across all L{Perk}s in the owning
    L{Tier}.
    
    @param name: Name of the L{Task} to locate
    @type name: string
    @return: L{Task} with the given name or None
    @rtype: L{Task.Base.Task}
    '''
    return self.tier.getNamedTask(name)
    
  def getTierEventTasks(self, event_type, task_layer):
    '''
    Gets all registered L{Task}s registered to handle the given L{AEEvent} type
    on the given layer from the current L{Tier}. 
    
    Calls L{Tier.Tier.getEventTasks} to search across all L{Perk}s in the 
    owning L{Tier}.  
    
    @param event_type: Desired type of L{AEEvent}
    @type event_type: L{AEEvent} class
    @param task_layer: Layer on which the desired L{Task}s are registered, one 
      of L{Task.LAYER_FOCUS}, L{Task.LAYER_TIER}, or L{Task.LAYER_BACKGROUND}
    @type task_layer: integer
    @return: List of all L{Task}s that handle the given event type and on 
      the given layer in the L{Tier} which owns this L{Perk}
    @rtype: list of L{Task.Base.Task}
    '''
    return self.tier.getEventTasks(event_type, task_layer)
  
  def getNamedTask(self, name):
    '''
    Gets the L{Task} with the given name if it is registered B{in this L{Perk}
    only}. If no L{Task} is registered under the given name in this L{Perk}, 
    returns None.
    
    @param name: Name of the L{Task} to locate
    @type name: string
    @return: L{Task} with the given name or None
    @rtype: L{Task.Base.Task}
    @see: L{getTierNamedTask}
    '''
    return self.perk.getNamedTask(name)
    
  def getEventTasks(self, event_type, task_layer):
    '''
    Get all registered L{Task}s registered to handle the given L{AEEvent} type 
    B{in this L{Perk} only}.
    
    @param event_type: Desired type of L{AEEvent}
    @type event_type: L{AEEvent} class
    @param task_layer: Layer on which the desired L{Task}s are registered, one 
      of L{Task.LAYER_FOCUS}, L{Task.LAYER_TIER}, or L{Task.LAYER_BACKGROUND}
    @type task_layer: integer
    @return: List of all L{Task}s that handle the given event type on the given
      layer in this L{Perk}
    @rtype: list of L{Task.Base.Task}
    @see: L{getTierEventTasks}
    '''
    return self.perk.getEventTasks(event_type, task_layer)
  
  def getDeviceState(self, device_name):
    '''
    Gets the default L{AEState} associated with an L{AEOutput} or L{AEInput} 
    device currently loaded in the L{DeviceManager}.
    
    @todo: PP: currently coded to support output devices only and then only
      their default styles
    
    @param device_name: Name of the device
    @type device_name: string
    @return: Loaded state for the device
    @rtype: L{AEState}
    '''
    try:
      dev = self.getOutputDevice(device_name)
    except InvalidDeviceError:
      return None
    return dev.getDefaultStyle()
  
  def getPerkState(self, perk_name):
    '''
    Gets the L{AEState} associated with a L{Perk}. If the state was previously
    persisted to disk, it is loaded from disk. If not, a new state object
    is created for the named Perk and immediately persisted in the user's 
    profile.
    
    @param perk_name: Name of the L{Perk}
    @type perk_name: string
    @return: Loaded state for the L{Perk}
    @rtype: L{AEState}
    '''
    # load an instance of the perk
    perk = UIRegistrar.loadOne(perk_name)
    if perk is None:
      # quit if we couldn't load the perk
      return None
    # create a new empty state object
    state = perk.getState()
    # try to load previously persisted state
    try:
      return self.sett_man.loadState(perk_name, state)
    except KeyError:
      pass
    # persist a new state instance if it does not exist
    self.sett_man.saveState(perk_name, state)
    return state
  
  def loadAllMonitors(self):
    '''
    Loads and shows all L{AEMonitor}s associated with this profile.
    
    @return: Was at least one monitor loaded?
    @rtype: boolean
    '''
    loaded = False
    reg = UIRegistrar
    mons = reg.loadAssociated(UIRegistrar.MONITOR, self.acc_eng.getProfile())
    emons = self.event_man.getMonitors()
    imons, omons = self.device_man.getMonitors()
    tmons = self.tier_man.getMonitors()
    if not len(emons):
      self.event_man.addMonitors(*mons)
      loaded = True
    if not len(tmons):
      self.tier_man.addMonitors(*mons)
      loaded = True
    if not (len(imons) + len(omons)):
      self.device_man.addMonitors(*mons)
      loaded = True
    return loaded
  
  def unloadAllMonitors(self):
    '''
    Hides and unloads all L{AEMonitor}s associated with this profile.
    
    @return: Was at least one monitor unloaded?
    @rtype: boolean
    '''
    unloaded = False
    emons = self.event_man.getMonitors()
    imons, omons = self.device_man.getMonitors()
    tmons = self.tier_man.getMonitors()
    for mon in (emons, imons, omons, tmons):
      if len(mon):
        mon.clear()
        unloaded = True
    return unloaded
  
  def loadMonitor(self, name):
    '''
    Loads and shows one L{AEMonitor}.
    
    @param name: Class name of the L{AEMonitor}
    @type name: string
    '''
    mon = UIRegistrar.loadOne(name)
    self.event_man.addMonitors(mon)
    self.tier_man.addMonitors(mon)
    self.device_man.addMonitors(mon)
    
  def unloadMonitor(self, name):
    '''
    Hides and unloads one L{AEMonitor}.
    
    @param name: Class name of the L{AEMonitor}
    @type name: string
    '''
    emons = self.event_man.getMonitors()
    tmons = self.tier_man.getMonitors()
    imons, omons = self.device_man.getMonitors()
    for mons in (emons, tmons, imons, omons):
      mons.removeByClassName(name)

  def loadChooser(self, name, **kwargs):
    '''
    Loads an L{AEChooser} dialog with the given name from disk. Passes the
    keyword arguments to the init method of the chooser.
    
    @param name: UIE name of the chooser
    @type name: string
    @param kwargs: Keyword arguments to initialize the chooser
    @type kwargs: dictionary
    '''
    reg = UIRegistrar
    chooser = reg.loadOne(name)
    if chooser is None:
      return
    try:
      # call the chooser to initialize it with a reference to the event manager
      chooser(self.event_man, self.tier.getIdentity())
    except ValueError, ex:
      # singleton already exists, reactivate
      ex.args[0].activate(timestamp=self.timestamp, **kwargs)
    else:
      # call init to let the chooser initialize itself
      chooser.init(timestamp=self.timestamp, **kwargs)
      # register to handle events from this chooser
      self.perk.registerChooserTask(chooser, self)
    
  def unloadChooser(self, chooser):
    '''
    Unloads the given L{AEChooser} dialog.
    
    @param chooser: The chooser to unload
    @type chooser: L{AEChooser}
    '''
    # unregister the task observing this chooser
    self.perk.unregisterChooserTask(chooser)

  def doTask(self, task_name, **kwargs):
    '''
    Immediately executes a L{Task} registered under the given name in this
    L{Tier}. Keyword arguments for L{Task} execution may be provided. The 
    L{task_por} of this L{Task} is always provided to the L{Task} being 
    executed and this L{Task}'s L{task_por} is updated with the value of the
    pointer after the other L{Task} has finished execution.
    
    @param task_name: Name of the L{Task} to execute
    @type task_name: string
    @param kwargs: Arbitrary keyword arguments to pass to the L{Task}, some of
      which may be required, others which may be optional, and others which
      may be completely ignored
    @type kwargs: dictionary
    '''
    self.tier.manageNamedTask(self.task_por, task_name, self.layer, kwargs)
    self.task_por = self.getPointerPOR()
    
  def blockNTasks(self, n, task_type, condition=lambda **x: True):
    '''
    Blocks L{Task}s of the given type from executing in response to the next
    n events. This is a convenience shortcut to avoid registering one-shot
    L{Task}s in L{Perk}s to do exactly the same thing.
    
    @param n: Number of events to block
    @type n: integer
    @param task_type: Kind of L{Task} to block
    @type task_type: L{Task} class
    @param condition: Condition statement to check if the event is actually
      one to be consumed. The callable is called with the keyword arguments
      passed to the L{Task} execute method.
    @type condition: callable
    '''
    # define an anonymous class of the given type
    class consumer(task_type):
      def init(self):
        self.count = n
      def execute(self, **kwargs):
        if not condition(**kwargs):
          return
        self.count -= 1
        if self.count == 0:
          self.unregisterTask(self)
        return False
    # register an instance as a blocker
    self.registerTask(consumer())
    
  def associatePerk(self, name, tier=None, all_tiers=False, index=None):
    '''
    Associates a L{Perk} with the active profile.
    
    @todo: PP: push instances of this Perk onto appropriate tiers?
    
    @param name: Name of the UIE to associate
    @type name: string
    @param tier: Name of the L{Tier} with which the L{Perk} will be associated.
      Defaults to the name of the L{Tier} specified by the L{Perk} itself.
    @type tier: string
    @param all_tiers: Should the L{Perk} be loaded on every L{Tier}? Defaults
      to the value specified by the L{Perk} itself.
    @type all_tiers: boolean
    @param index: Load order index of the L{Perk}. Lower means sooner to load
      but also later to handle events (i.e. bottom of the L{Perk} stack).
    @type index: integer
    '''
    UIRegistrar.associate(name, [self.getProfileName()], tier, all_tiers,index)
  
  def disassociatePerk(self, name, tier=None, all_tiers=False):
    '''
    Disassociates a L{Perk} from the active profile.
    
    @todo: PP: remove currently loaded instances of this Perk?
    
    @param name: Name of the UIE to disassociate
    @type name: string
    @param tier: Name of the L{Tier} from which the L{Perk} will be 
      disassociated. Defaults to the name of the L{Tier} specified by the 
      L{Perk} itself.
    @type tier: string
    @param all_tiers: Should the L{Perk} be removed from the list of Perks to
      load for every L{Tier}? Defaults to the value specified by the L{Perk} 
      itself.
    '''
    UIRegistrar.disassociate(name, [self.getProfileName()], tier, all_tiers)
    
  def associateMonitor(self, name):
    '''
    Associates a L{AEMonitor} with the active profile. Loads it immediately.
    
    @param name: Name of the UIE to associate
    @type name: string
    '''
    UIRegistrar.associate(name, [self.getProfileName()])
    self.loadMonitor(name)
  
  def disassociateMonitor(self, name):
    '''
    Associates a L{AEMonitor} with the active profile. Unloads it immediately.
    
    @param name: Name of the UIE to associate
    @type name: string
    '''
    self.unloadMonitor(name)
    UIRegistrar.disassociate(name, [self.getProfileName()])
  
  def associateDevice(self, name, index=None):
    '''
    Associates a L{AEOutput} or L{AEInput} device with the active profile.
    Does not load it immediately. See L{refreshDevices} for that functionality.
    
    @param name: Name of the UIE to associate
    @type name: string
    @param index: Load order index of the device where lower means sooner, None
      means last.
    @type index: integer
    '''
    UIRegistrar.associate(name, [self.getProfileName()], index=index)
  
  def disassociateDevice(self, name):
    '''
    Disassociates a L{AEOutput} or L{AEInput} device from the active profile. 
    Does not unload it immediately. See L{refreshDevices} for that 
    functionality.
    
    @param name: Name of the UIE to associate
    @type name: string
    '''
    UIRegistrar.disassociate(name, [self.getProfileName()])

  def refreshDevices(self):
    '''
    Unloads all existing devices then immediately loads all devices associated
    with the active profile.
    '''
    self.device_man.unloadDevices()
    self.device_man.loadDevices()