'''
Defines classes for representing device capabilities and styles. 

@var FORMAT_TEXT: Words will be output without spelling
@type FORMAT_TEXT: integer
@var FORMAT_PRONOUNCE: Punctuation will be pronounced
@type FORMAT_PRONOUNCE: integer
@var FORMAT_SPELL: All characters in a word will be spelled
@type FORMAT_SPELL: integer
@var FORMAT_PHONETIC: All characters in a word will be spelled using a phonetic
  spelling table
@type FORMAT_PHONETIC: integer

@author: Brett Clippingdale
@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
from i18n import _
import Word, AEState

# constants for SpellFormat
FORMAT_TEXT = 0
FORMAT_PRONOUNCE = 1
FORMAT_SPELL = 2
FORMAT_PHONETIC = 3

class StyleFlyweight(AEState.AEState):
  '''
  Base class for all style classes that will contain a set of properties 
  defining how output should be presented on an L{AEOutput} device. Implements 
  the flyweight pattern such that if this style object does not define a 
  particular property, it may be retrieved from another style object instead.
    
  @ivar _style: Another style object to which this one should dispatch 
    attribute lookups if attributes are not defined in this object
  @type _style: L{Style}
  @cvar ADDITIVE: List of attribute names that are additive rather than 
    replacements for their parent L{_style} attributes
  @type ADDITIVE: tuple
  '''
  ADDITIVE = tuple()
  
  def __init__(self, style=None):
    '''
    Initializes the style object to which to dispatch requests for attributes
    that are not defined in this object.
    
    @param style: Parent style object
    @type style: L{Style}
    '''
    super(StyleFlyweight, self).__init__()
    self._style = style
    
  def __getattribute__(self, name):
    '''
    Tries to get the named attribute. Dispatches the request to L{_style} if it
    is not found in this object and L{_style} is not None.

    @param name: Name of the attribute to fetch
    @type name: string
    @return: Attribute value
    @rtype: object
    '''
    if name.startswith('_'):
      # opt: shortcut for protected vars
      return object.__getattribute__(self, name)
    try:
      # get the list of additive attributes
      add = object.__getattribute__(self, 'ADDITIVE')
    except AttributeError:
      add = None
    # get the parent style
    st = object.__getattribute__(self, '_style')
    if add and name in add:
      # additive
      val = object.__getattribute__(st, name)
      return val + object.__getattribute__(self, name)
    else:
      try:
        # exists in this object
        return object.__getattribute__(self, name)
      except AttributeError:
        # exists in the parent
        return object.__getattribute__(st, name)
  
  def copy(self):
    '''
    Makes a fast copy of this object's __dict__.
    
    @return: New style object
    @rtype: L{Style}
    '''
    other = super(StyleFlyweight, self).copy()
    st = self._style
    if st:
      parent = st.copy()
      other.__dict__['_style'] = parent
    return other
  
  def isDirty(self):
    '''
    @return: Is there anything in the L{_dirty} set on this style or its 
      parent?
    @rtype: boolean
    '''
    d = super(StyleFlyweight, self).isDirty()
    if self._style:
      d = d or self._style.isDirty()
    return d
  
  def makeClean(self):
    '''Resets the L{_dirty} set.'''
    if self._style is not None:
      self._style.makeClean()
    self._dirty = set()    

class StyleDefault(Word.WordState):
  '''
  Derives from L{Word.WordState} such that the style inherits options for 
  parsing text. Text parsing options that are ubiquitous for all output devices
  but are not part of the core parser are defined as class variables here.
  
  @cvar SpellFormat: Set to FORMAT_TEXT to present words without additional
    spelling. Set to FORMAT_PRONOUNCE to spell punctuation characters for
    presentation. Set to FORMAT_SPELL to spell all characters for 
    presentation. Set to FORMAT_PHONETIC to spell all characters phonetically 
    for presentation. Defaults to None which means the setting cannot be 
    changed for a device and is the equivalent of FORMAT_TEXT. Any values other
    than FORMAT_TEXT may be ignored by a device.
  @type SpellFormat: integer
  @cvar CapString: Set to the string that should be presented prior to the
    spelling of capitalized letters and presented twice before spelling fully
    capitalized words. Defaults to 'cap'.
  @type CapString: string
  '''
  SpellFormat = None
  CapString = 'cap'

class AudioStyleDefault(StyleDefault):
  '''
  Defines the basic style attributes for all L{AEOutput.Audio} devices. Most
  attributes are defaulted to None indicating they are not supported. It is
  the job for a specific output device definition to override these values with
  its own defaults. Text parsing options are also included here for use by
  the default implementation of parseString in L{AEOutput.Audio}.
  
  @cvar Volume: Current volume
  @type Volume: integer
  @cvar MaxVolume: Maximum supported volume
  @type MaxVolume: integer
  @cvar MinVolume: Minimum supported volume
  @type MinVolume: integer
  @cvar Rate: Current speech rate
  @type Rate: integer
  @cvar MaxRate: Maximum supported speech rate
  @type MaxRate: integer
  @cvar MinRate: Minimum supported speech rate
  @type MinRate: integer
  @cvar Pitch: Current baseline pitch
  @type Pitch: integer
  @cvar MaxPitch: Maximum supported pitch
  @type MaxPitch: integer
  @cvar MinPitch: Minimum supported pitch
  @type MinPitch: integer
  @cvar Voice: Current voice number
  @type Voice: integer
  @cvar MaxVoice: Maximum voice number
  @type MaxVoice: integer
  @cvar MinVoice: Minimum voice number
  @type MinVoice: integer
  @cvar Position: Current spatial position
  @type Position: 3-tuple of float
  @cvar Language: Current language specified as ISO language and country codes
  @type Language: string
  @type Instrument: Current MIDI instrument number
  @cvar Instrument: integer
  @cvar MaxInstrument: Maximum supported instrument number
  @type MaxInstrument: integer
  @cvar MinInstrument: Minimum supported instrument number
  @type MinInstrument: integer
  @cvar Continuous: Is output continuous (i.e. looping)?
  @type Continuous: boolean
  @cvar Channel: Channel number indicating concurrency of output. Styles 
    sharing a channel number are output sequentially. Styles having different 
    channel numbers are played concurrently if possible.
  @type Channel: integer
  @cvar Stoppable: Can output be stopped?
  @type Stoppable: boolean
  @cvar Speakable: Can output be spoken?
  @type Speakable: boolean
  @cvar Soundable: Can output be played as a non-speech sound?
  @type Soundable: boolean
  @cvar Mute: Is all output inhibited?
  @type Mute: boolean
  @cvar Gender: Speech vocal tract gender
  @type Gender: integer
  @cvar MaxGender: Maximum supported gender constant
  @type MaxGender: integer
  @cvar MinGender: Minimum supported gender constant
  @type MinGender: integer
  @cvar Aspiration: Speech breathiness
  @type Aspiration: integer
  @cvar MaxAspiration: Maximum supported aspiration
  @type MaxAspiration: integer
  @cvar MinAspiration: Minimum supported aspiration
  @type MinAspiration: integer
  @cvar Frication: Emphasis of fricatives in speech
  @type Frication: integer
  @cvar MaxFrication: Maximum supported frication
  @type MaxFrication: integer
  @cvar MinFrication: Minimum supported frication
  @type MinFrication: integer
  @cvar Intonation: Speech pitch variation
  @type Intonation: integer
  @cvar MinIntonation: Maximum supported pitch variation
  @type MinIntonation: integer
  @cvar MaxIntonation: Minimum supported pitch variation
  @type MaxIntonation: integer
  @cvar HeadSize: Speech speaker head size for reverberation
  @type HeadSize: integer
  @cvar MaxHeadSize: Maximum supported head size
  @type MaxHeadSize: integer
  @cvar MinHeadSize: Minimum supported head size
  @type MinHeadSize: integer
  @cvar SpellCaps: When set to True, fully capitalized words will be spelled
    for presentation if the device supports spelling.
  @type SpellCaps: boolean
  @cvar Blank: String to substitute for blank output. Defaults to 'blank' 
    localized.
  @type Blank: string
  @cvar CapExpand:  When set to True, a space will be inserted before
    embedded capital letters in the presentation of a word. Defaults to True.
  @type CapExpand: boolean
  @cvar NumExpand:  When set to True, a space will be inserted before embedded
    numbers in the presentation of a word. Defaults to False.
  @type NumExpand: boolean
  '''
  Position = None
  Language = None
  Channel = 0
  MinChannel = 0
  MaxChannel = 0
  Stoppable = True
  Speakable = True
  Soundable = True
  Mute = False
  Continuous = False
  Instrument = None
  MaxInstrument = None
  MinInstrument = None
  Volume = None
  MaxVolume = None
  MinVolume = None
  Rate = None
  MaxRate = None
  MinRate = None
  Pitch = None
  MaxPitch = None
  MinPitch = None
  Voice = None
  MaxVoice = None
  MinVoice = None
  Gender = None
  MaxGender = None
  MinGender = None
  Aspiration = None
  MaxAspiration = None
  MinAspiration = None
  Frication = None
  MaxFrication = None
  MinFrication = None
  Intonation = None
  MaxIntonation = None
  MinIntonation = None
  HeadSize = None
  MinHeadSize = None
  MaxHeadSize = None
  SpellCaps = None
  Blank = _('blank')
  CapExpand = True
  NumExpand = False
  
  def _newWordGroup(self, group):
    '''
    Builds a group containing settings that should be available for all audio
    devices given that the L{Word} parser and the L{Audio.parseString} methods
    support them without any extra work on the part of the device.
    
    @param group: Parent group for the word settings group
    @type group: L{AEState.Setting.Group}
    @return: Group containing the word settings, can be extended as needed
    @rtype: L{AEState.Setting.Group}
    '''
    w = group.newGroup(_('Words'))
    w.newEnum('WordDef', _('Word definition'), 
              {_('Non-blank') : Word.WORD_NON_BLANK,
               _('Alphanumeric and punct.') : Word.WORD_ALPHA_NUMERIC_PUNCT,
               _('Alphanumeric') : Word.WORD_ALPHA_NUMERIC,
               _('Alpha and punct.') : Word.WORD_ALPHA_PUNCT,
               _('Alphabetic') : Word.WORD_ALPHABETIC},
              _('Defines the characters that comprise a word.'))
    #w.newString('CapString', _('Cap string'))
    w.newString('Blank', _('Blank string'), 
                _('String to speak when a blank or ignored character is '
                  'output on its own.'))
    w.newBool('Caps', _('Preserve caps?'), 
              _('When set, capitalization is preserved in the output. '
                'Otherwise, all strings are lowercased.'))
    w.newBool('CapExpand', _('Expand caps?'), 
              _('When set, insert a space before each capital letter in the '
                'output.'))
    w.newBool('NumExpand', _('Expand numbers?'), 
              _('When set, insert a space before each number in the output.'))
    #w.newNumeric('MaxRepeat', _('Repeat count'), 1, 80, 0)
    #w.newString('Ignore', _('Ignored characters'))
    return w