'''
Defines a class responsible for handling events in the currently focused view.

@var CHECK_INTERVAL: Time in ms that the L{ViewManager} will iterate over all
  top level applications and provide them in a list to L{TierManager}
@type CHECK_INTERVAL: integer
@var DEFAULT_EVENTS: Names of events that will be dispatched by the 
  L{ViewManager} by default
@type DEFAULT_EVENTS: list

@todo: PP: the AT-SPI specific code for walking the desktop should be moved to
  an adapter 

@author: Peter Parente
@author: Pete Brunet
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import logging
import pyLinAcc
from AEEvent import *
from POR import POR
from LSRInterfaces import *
from Walker import AccessibleWalker

log = logging.getLogger('View')

# interval on which the manager will search for dead views
CHECK_INTERVAL = 60000

# events that we should always watch, regardless of Task interests
DEFAULT_EVENTS = [ViewChange, FocusChange, SelectorChange, CaretChange, 
                  PropertyChange]

class ViewManager(object):
  '''
  Stores a root accessible representing the active view when an event occurs
  indicating that the view may have changed.
  
  @ivar event_manager: Reference to the L{EventManager}
  @type event_manager: L{EventManager}
  @ivar active_view: Root L{POR} representing the active view
  @type active_view: L{POR}
  @ivar tier_manager: Reference to the L{TierManager} for clearing dead L{Tier}
    objects on a set interval
  @type tier_manager: L{TierManager}
  @ivar wanted_events: Lookup table for what L{AEEvent}s are desired by any
    L{Task} in any L{Perk} in all L{Tier}s. Used to optimize event dispatch and
    processing. Unwanted events are ignored.
  @type wanted_events: dictionary
  @ivar activated: Is there an activated view somewhere?
  @type activated: boolean
  '''
  def __init__(self, ae):
    '''
    Initializes instance variables that will store a reference to the 
    L{EventManager} and active view to None.
    
    @param ae: Reference to the L{AccessEngine}, currently unused
    @type ae: L{AccessEngine}
    '''
    self.tier_manager = None
    self.event_manager = None
    self.active_view = None
    self.wanted_events = {}
    self.activated = False
    # register for a callback to check living applications
    ae.addTimer(self.onIterateApps, CHECK_INTERVAL)
      
  def init(self, event_man, tier_man, **kwargs):
    '''
    Stores a reference to the L{EventManager}. Registers with the 
    L{EventManager} using the L{EventManager.EventManager.addClient} to receive 
    raw events related to the active view. Also registers 
    for raw events that may indicate that this L{ViewManager} needs to manage a
    new view. Called by L{AccessEngine} at startup.
    
    @param event_man: Reference to the L{EventManager}
    @type event_man: L{EventManager}
    '''
    # store the tier manager so we can note running applications at startup
    self.tier_manager = tier_man
    # store the event manager so we can post events later
    self.event_manager = event_man
    # register for events that almost certainly mean the view has changed
    self.event_manager.addClient(self.onRawViewEvent, 
                                 *IEventHandler().getRawEvents(ViewChange))
    # track how many dependencies we have on a certain kind of event
    for kind in DEFAULT_EVENTS:
      self.setEventInterest(kind, True)
    
  def close(self):
    '''Does nothing.'''
    pass
  
  def initViews(self):
    '''   
    Walks through the top level applications on the first desktop. Calls
    L{TierManager.TierManager.createTier} for application encountered. If the
    L{TierManager} determines that an app has at least one L{Perk} written for
    that application, creates a L{Tier} for the application and loads all of
    its L{Perk}s. If the L{Tier} is created, this method sends
    L{AEEvent.ViewChange} messages to the L{Tier} with a flag of
    L{AEEvent.ViewChange.ViewChange.STARTUP} such that L{Perk}s in the L{Tier}
    may initialize themselves and begin any desired background processes.
    '''
    d = pyLinAcc.Registry.getDesktop(0)
    ai = IAccessibleInfo(POR(d))
    an = IAccessibleNav(POR(d))
    events = []
    # iterate over all children of the desktop
    for i in xrange(ai.getAccChildCount()):
      try:
        app = an.getChildAcc(i)
      except (LookupError, IndexError):
        continue
      # get application names and IDs
      aai = IAccessibleInfo(app)
      aid = aai.getAccAppID()
      name = aai.getAccAppName()
      # inform the tier manager
      if self.tier_manager.createTier(name, aid, app, True) is not None:
        events.append(ViewChange(app, ViewChange.STARTUP))
      # if we've found the active app, just continue
      if self.active_view is not None:
        continue
      aan = IAccessibleNav(app)
      # iterate over all top level windows in the app
      for x in xrange(aai.getAccChildCount()):
        try:
          win = aan.getChildAcc(x)
        except (LookupError, IndexError):
          continue
        # look for one that has state active
        wai = IAccessibleInfo(win)
        if wai.hasAccState('active'):
          self.active_view = win
          break
    # send a view change gained event to the active tier
    if self.active_view is not None:
      events.append(ViewChange(self.active_view, ViewChange.FIRST_GAINED))
    self.event_manager.postEvents(*events)
      
  def setEventInterest(self, kind, wants):
    '''
    Sets or unsets an interest in a particular kind of L{AEEvent}. This
    information is used to register or unregister for raw events on-demand as
    an optimization.
    
    @param kind: Indicates the type of L{AEEvent} some part of the system wants
      to be able to process
    @type kind: L{AEEvent} class
    @param wants: Does the system want to process the given kind of L{AEEvent}?
    @type wants: boolean
    ''' 
    count = self.wanted_events.setdefault(kind, 0)
    if wants:
      count += 1
    else:
      count -= 1
    if count <= 0:
      del self.wanted_events[kind]
      self.event_manager.removeClient(self.onRawEvent, 
                                      *IEventHandler().getRawEvents(kind))
      return
    elif count == 1:
      self.event_manager.addClient(self.onRawEvent,
                                   *IEventHandler().getRawEvents(kind))
    self.wanted_events[kind] = count
    
  def getActivated(self):
    '''
    Gets the whether some view is active or not.
    
    @return: Value of L{activated}
    @rtype: boolean
    '''
    return self.activated
  
  def setActivated(self, val):
    '''
    Sets the whether some view is active or not.
    
    @param val: Value to store in L{activated}
    @type val: boolean
    '''
    self.activated = val

  def getView(self):
    '''
    Gets the root accessible of the active view.
    
    @return: Root L{POR} of the active view
    @rtype: L{POR}
    '''
    return self.active_view
  
  def setView(self, accessible):
    '''    
    Stores the root accessible corresponding to the received event to
    represent the active view.
    
    @param accessible: Event source that triggered the change of view. Usually
      a top level window or panel.
    @type accessible: L{pyLinAcc.Accessible}
    @return: Was a new view root set or not?
    @rtype: boolean
    '''
    # walk to the first POR
    root = POR(accessible) #AccessibleWalker(POR(accessible)).getFirstPOR()
    # see if it is different from our current root POR
    if root != self.active_view:
      # store the new view
      self.active_view = root
      return True
    return False
  
  def onIterateApps(self):
    '''
    Collects all application names and IDs and gives them to the L{TierManager}
    so it can free any L{Tier}s that are no longer associated with running
    applications.
    
    @return: True to continue receiving notifications
    @rtype: boolean
    '''
    d = pyLinAcc.Registry.getDesktop(0)
    ai = IAccessibleInfo(POR(d))
    an = IAccessibleNav(POR(d))
    aids = []
    # iterate over all children of the desktop
    for i in xrange(ai.getAccChildCount()):
      try:
        app = an.getChildAcc(i)
      except (LookupError, IndexError):
        continue
      # get all application identifiers
      aai = IAccessibleInfo(app)
      aids.append(aai.getAccAppID())
    # inform the TierManager about the living applications
    self.tier_manager.freeDeadTiers(aids)
    return True
   
  def onRawEvent(self, event):
    '''
    Based on a raw L{pyLinAcc.Event.Event} posts L{AEEvent}s to the active
    L{Tier} through the L{EventManager}.
    
    @param event: Event indicating some change
    @type event: L{pyLinAcc.Event.Event}
    '''
    # only forward events if there is a view
    if self.active_view is not None:
      eh = IEventHandler(event.source)
      eh.getAEEvents(event, self.event_manager.postEvents)
    
  def onRawViewEvent(self, event):
    '''    
    Creates a L{AEEvent.ViewChange} event in response to a window activation or
    deactivation. Also responds to create or destroy events from floating
    windows. Called in response to any of the raw events listed in
    L{VIEW_EVENTS} and registered with the L{EventManager} in the
    L{ViewManager.init} method when this object was created. All events in
    L{VIEW_EVENTS} are monitored as indicating view changes.

    @param event: Event that may indicate the view has changed
    @type event: L{pyLinAcc.Event.Event}
    '''
    eh = IEventHandler(event.source)
    eh.getAEViewEvents(event, self.event_manager.postEvents, self)