'''
Defines a gtk dialog showing the roles and names of CVIs in the active 
view of the L{ViewManager} in a gtk.TreeView control.

@todo: PP: this module needs to change now that we no longer have CVIs; use
  IAccessibleNav interface to build the gtk.GenericTreeModel dynamically

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made 
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''

import pygtk
pygtk.require('2.0')
import gtk
import weakref
from i18n import _

class ViewAdapter(gtk.GenericTreeModel):
  '''
  Adapts a view object to the gtk.GenericTreeModel interface so that it can
  be displayed in a gtk.TreeView control. Only weak references to CVIs are 
  used internally because the gtk.TreeView seems to hold onto the model even 
  when it is replaced with a new one. If strong references to CVIs are used,
  the CVIs will never be properly garbage collected.
  
  See U{http://www.pygtk.org/pygtk2reference/class-pygtkgenerictreemodel.html}
  for details about this interface.
  
  @ivar cvi_view: view to adapt to the gtk.GenericTreeModel interface
  @type cvi_view: view
  @cvar columns: Names of the columns in the gtk.TreeView
  @type columns: list
  '''
  columns = [_('Role'), _('Name')]
  
  def __init__(self, cvi_view):
    '''
    Initializes the parent class and stores a reference to the view.
    '''
    gtk.GenericTreeModel.__init__(self)
    self.cvi_view = cvi_view

  def on_get_flags(self):
    '''
    Returns flags indicating the behavior of the model.
    
    @return: Flags indicating the behavior of the model
    @rtype: integer
    '''
    return 0
    
  def on_get_n_columns(self):
    '''
    Returns the number of columns to show in the tree widget by checking how
    many column names are defined in L{ViewAdapter.columns}.
    
    @return: Number of columns
    @rtype: integer
    '''
    return len(ViewAdapter.columns)
    
  def on_get_column_type(self, i):
    '''
    Gets the type of data to be displayed in the column at index i.
    
    @param i: Column index
    @type i: integer
    @return: Always the type str indicating strings
    @rtype: type
    '''
    return str
    
  def on_get_iter(self, path):
    '''
    Gets the CVI at the given path in the view.
    
    @param path: Tuple pointing to a unique CVI in the view
    @type path: tuple
    @return: CVI at the given location or None if the path is invalid
    @rtype: weakref.proxy to CVI
    '''
    try:
      return weakref.proxy(self.cvi_view.getCVIFromPath(path))
    except IndexError:
      return None
    
  def on_get_path(self, cvi):
    '''
    Gets the path to the given CVI.
    
    @param cvi: CVI in the current view
    @type cvi: weakref.proxy to CVI
    @return: Path to given CVI
    @rtype: tuple
    '''
    return cvi.path
    
  def on_get_value(self, cvi, i):
    '''
    Gets the value of the CVI to be shown in the column at index i.
    
    @param cvi: CVI in the current view
    @type cvi: CVI
    @param i: Column index of the requested value
    @type i: integer
    @return: Value of the CVI for the given column
    @rtype: object
    '''
    mthd = [cvi.getRoleName, cvi.getName]
    try:
      # if accessible goes away, ignore all errors for now
      return mthd[i]()
    except Exception:
      return None
    
  def on_iter_next(self, cvi):
    '''
    Gets the next sibling CVI of the given CVI or None if there is no
    next sibling.
    
    @param cvi: CVI in the current view
    @type cvi: CVI
    @return: Next sibling CVI or None
    @rtype: weakref.proxy to CVI
    '''
    try:
      return weakref.proxy(cvi.next)
    except TypeError:
      # if we can't make a weakref, assume cvi.next was None
      return None
    
  def on_iter_children(self, cvi):
    '''
    Gets the first child CVI of the given CVI or None if there is no
    first child.
    
    @param cvi: CVI in the current view
    @type cvi: CVI
    @return: First child CVI
    @rtype: weakref.proxy to CVI
    ''' 
    try:
      return weakref.proxy(cvi.children[0])
    except (AttributeError, IndexError):
      return None
    
  def on_iter_has_child(self, cvi):
    '''
    Checks if the give CVI has at least one child CVI.
    
    @param cvi: CVI in the current view
    @type cvi: CVI
    @return: Does the CVI have children?
    @rtype: boolean
    ''' 
    return cvi.hasChildren()
    
  def on_iter_n_children(self, cvi):
    '''
    Gets the number of child CVIs of the given CVI.
    
    @param cvi: CVI in the current view
    @type cvi: CVI
    @return: Number of child CVIs
    @rtype: integer
    '''
    try:
      return cvi.getChildCount()
    except AttributeError:
      return 0
    
  def on_iter_nth_child(self, cvi, i):
    '''
    Gets the ith child of the given CVI. If no CVI is provided, assume
    the request if for a root CVI. Since a view can only have one root,
    ignore all requests with no given CVI and an index greater than zero.
    
    @param cvi: CVI in the current view
    @type cvi: CVI
    @param i: Child index of the requested CVI
    @type i: integer
    @return: ith child of the CVI or the root CVI or None
    @rtype: weakref.proxy to CVI
    ''' 
    if cvi is None:
      # requests with a None cvi are asking for root objects
      if i == 0:
        # we only ever have a single root
        return weakref.proxy(self.cvi_view.root)
      else:
        # all other requests at the root level are bogus
        return None
    try:
      # if a cvi is given, get the nth child
      return weakref.proxy(cvi.children[i])
    except (AttributeError, IndexError):
      return None
    
  def on_iter_parent(self, cvi):
    '''
    Gets the parent CVI of the given CVI or None if there is no parent.
    
    @param cvi: CVI in the current view
    @type cvi: CVI
    @return: Parent CVI or None
    @rtype: weakref.proxy to CVI
    '''
    try:
      return weakref.proxy(cvi.parent)
    except TypeError:
      return None
    
class Dialog(object):
  '''
  Renders a view adapted by L{ViewAdapter} in a gtk.TreeView widget.
  
  @ivar window: Top level window housing the tree widget
  @type window: gtk.Window
  @ivar alive: Should this monitor be showing updates to the view?
  @type alive: boolean
  @ivar tree_view: Tree widget rendering an L{ViewAdapter}
  @type tree_view: gtk.TreeView
  '''
  def __init__(self, ae):
    '''
    Creates a top-level gtk.Window and a gtk.TreeView. Builds 
    gtk.CellRendererText and gtk.TreeViewColumn objects to show the labels of
    the various columns in the tree widget. Makes the first column searchable.
    
    See L{ViewAdapter.columns} for the column definitions.

    @param ae: Reference to the L{AccessEngine}, currently unused
    @type ae: L{AccessEngine}
    '''
    # create a new window
    self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
    self.window.set_title(_('View Monitor'))
    self.window.set_size_request(400, 600)
    
    # set monitor for window closing
    self.window.connect('destroy', self.onDestroy)
    self.alive = True
 
    # create the TreeView
    self.tree_view = gtk.TreeView()
    
    for i in xrange(len(ViewAdapter.columns)):
      # create a CellRendererText to render the data
      cell = gtk.CellRendererText()
      # create the TreeViewColumn to display the data
      tv_column = gtk.TreeViewColumn(ViewAdapter.columns[i], cell, text=i)
      # add tvcolumn to treeview
      self.tree_view.append_column(tv_column)

    # make it searchable
    self.tree_view.set_search_column(1)

    # show all components
    scrolled_window = gtk.ScrolledWindow()
    scrolled_window.add(self.tree_view)
    self.window.add(scrolled_window)
    self.window.show_all()
    
  def onDestroy(self, widget):
    '''
    Destroys the window and sets the L{Dialog.alive} variable to False so any 
    future changes of view are ignored (i.e. if this window is in the process
    of dying and the view changes).

    @param widget: Source of the event
    @type widget: gtk.Widget
    '''
    self.alive = False
    
  def refreshView(self, cvi_view):
    '''
    Creates a L{ViewAdapter} for the given view and sets it as the model for
    the gtk.TreeView widget. Ignores any requests if the L{Dialog.alive}
    variable is set to False.
    
    @param cvi_view: View of CVIs to render
    @type cvi_view: view
    '''
    if not self.alive: return
    if cvi_view is None:
      self.tree_view.set_model(None)
    else:
      model = ViewAdapter(cvi_view)
      self.tree_view.set_model(model)
      self.tree_view.expand_to_path((0,))