'''
Defines an L{AEEvent} indicating a setting changed.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''

import Base, Constants
import AEChooser

class SettingChange(Base.AccessEngineEvent):
  '''  
  Event that fires when a L{AEChooser} or L{Task.Tools} method indicates a 
  setting changed.
  
  This class registers its name and whether it should be monitored by default
  in an L{AEMonitor} using the L{Constants.registerEventType} function when
  this module is first imported. The L{AEMonitor} can use this information to
  build its menus.
  
  @ivar aid: Unique identifier for the application L{Tier} with which the 
    L{AEChooser} that fired this event is associated
  @type aid: opaque
  @ivar source: Object that caused the setting change
  @type source: L{UIElement}
  @ivar state: State object having the setting that changed
  @type state: L{AEState}
  @ivar name: Name of the setting that changed
  @type name: string
  @ivar value: New value for the setting
  @type value: object
  @ivar kwargs: Aribitrary data to be passed to the handler of this event
  @type kwargs: dictionary
  '''
  Constants.registerEventType('SettingChange', False)
  def __init__(self, aid, source, state, name, value, **kwargs):
    '''
    Stores important references.
    
    @param aid: Unique identifier for the application L{Tier} with which the 
    L{AEChooser} that fired this event is associated
    @type aid: opaque
    @param source: Object that caused the setting change
    @type source: L{UIElement}
    @param state: State object having the setting that changed
    @type state: L{AEState}
    @param name: Name of the setting that changed
    @type name: string
    @param value: New value for the setting
    @type value: object
    '''
    Base.AccessEngineEvent.__init__(self, focused=True, **kwargs)
    self.aid = aid
    self.kwargs = kwargs
    self.source = source
    self.state = state
    self.name = name
    self.value = value
    
  def __str__(self):
    '''
    Returns a human readable representation of this event including its name,
    the setting name, the setting value, and the source of the event.
    
    @return: Information about this event
    @rtype: string
    '''
    name = Base.AccessEngineEvent.__str__(self)
    return '%s:\n\tname: %s\n\tvalue: %s\n\tsource: %s' % \
           (name, self.name, self.value, self.source.getName())

  def execute(self, tier_manager, **kwargs):
    '''
    Contacts the L{TierManager} and asks it to manage this setting event.
    
    @param tier_manager: TierManager that will handle the event
    @type tier_manager: L{TierManager}
    @param kwargs: Packed references to other managers not of interest here
    @type kwargs: dictionary
    @return: True to indicate the event executed properly
    @rtype: boolean
    '''
    tier_manager.manageSetting(self)
    return True
  
  def getAppID(self):
    '''
    @return: Unique application ID identifying the top most container for the
      source of this event (i.e. the application)
    @rtype: opaque object
    '''
    return self.aid
  
  def getDataForTask(self):
    '''
    Fetches data out of this L{SettingChange} for use by a L{Task.SettingTask}.
    
    @return: Dictionary of parameters to be passed to a L{Task.SettingTask} as 
      follows:
        - source: The source object of the change event
        - state: The L{AEState} object containing the setting that changed
        - name: The name of the setting
        - value: The value of the setting
        - any additional data in the L{kwargs} instance variable
    @rtype: dictionary
    '''
    d = {}
    d.update(self.kwargs)
    d.update(dict(source=self.source, state=self.state, name=self.name, 
                  value=self.value))
    return d