'''
Defines the default user interface for LSR.

@author: Peter Parente
@author: Pete Brunet
@author: Larry Weiss
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made 
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''
import Perk, Task, LSRConstants
from POR import POR
import unicodedata
from i18n import _

__uie__ = dict(kind='perk', all_tiers=True)

class DefaultPerkState(Perk.PerkState):
  '''
  Defines a single set of state variables for all instances of the 
  L{DefaultPerk} and their L{Task}s. The variables are configuration settings
  that should be respected in all instances of L{DefaultPerk}.
  
  @cvar OnlyVisible: Should review keys only walk visible items or hidden items
    also?
  @type OnlyVisible: boolean
  @cvar Wrap: Should movement of the Pointer L{POR} wrap at item boundaries?
  @type Wrap: boolean
  @cvar RepeatRole: Should successively visited items with the same role
    have their role announced (True) or not (False)?
  @type RepeatRole: boolean
  '''
  OnlyVisible = True
  Wrap = True
  RepeatRole = False
  WordEcho = False
  CharEcho = True
  
  def getSettings(self):
    '''
    Gets configurable settings for this L{Perk}.
    
    @return: Group of all configurable settings
    @rtype: L{AEState.Setting.Group}
    '''
    g = self._newGroup()
    e = g.newGroup(_('Echo'))
    e.newBool('WordEcho', _('Echo words?'), 
              _('When set, entire words are spoken when editing text.'))
    e.newBool('CharEcho', _('Echo characters?'), 
            _('When set, individual characters are spoken when editing text.'))
    v = g.newGroup(_('Verbosity'))
    v.newBool('RepeatRole', _('Always say role?'),
            _('When set, the role name of every widget is announced. '
              'Otherwise, the role name is only announced when a new role is '
              'encountered.'))
    r = g.newGroup(_('Review mode'))
    r.newBool('OnlyVisible', _('Review visible items only?'), 
              _('When set, reviewing only includes visible items. Otherwise, '
                'invisible items are also included.'))
    r.newBool('Wrap', _('Wrap pointer across items?'),
            _('When set, next and previous, word and character navigation '
              'can cross item boundaries. Otherwise, first and last, word '
              'and character are announced instead.'))
    return g
  
class DefaultPerk(Perk.Perk):
  '''
  Defines a default user interface that makes LSR act like a typical screen 
  reader.
  
  @ivar last_caret: Stores the previous caret L{POR}
  @type last_caret: L{POR}
  @ivar last_role: Stores the previous role to avoid duplicate announcements
  @type last_role: string
  @ivar last_level: Stores the previous level to avoid duplicate announcements
  @type last_level: integer
  @ivar last_container: Stores the previously announced container name
  @type last_container: string
  @ivar last_sel_len: Length of the selected text when the last selection event
    was received
  @type last_sel_len: integer
  @ivar last_row: Index of the last row activated
  @type last_row: integer
  @ivar last_col: Index of the last column activated
  @type last_col: integer
  @ivar last_count: Last count of selected characters
  @type last_count: integer 
  '''
  # defines the class to use for state information in this Perk
  STATE = DefaultPerkState
  
  def init(self):
    '''
    Registers L{Task}s to handle focus, view, caret, selector, state, and 
    property change events. Registers named L{Task}s that can be mapped to 
    input gestures.
    '''
    # set an audio device as the default output
    self.setPerkIdealOutput('audio')
    # register event handlers
    self.registerTask(HandleFocusChange('default focus'))
    self.registerTask(HandleViewChange('default view'))
    self.registerTask(HandleCaretChange('default caret'))
    self.registerTask(HandleSelectorChange('default selector'))
    self.registerTask(HandleStateChange('default state'))
    self.registerTask(HandlePropertyChange('default property'))
    # register named tasks
    # these will be mapped to input gestures in other Perks
    self.registerTask(Stop('stop now'))
    self.registerTask(IncreaseRate('increase rate'))
    self.registerTask(DecreaseRate('decrease rate'))
    self.registerTask(PreviousItem('previous item'))
    self.registerTask(CurrentItem('current item'))
    self.registerTask(NextItem('next item'))
    self.registerTask(PreviousWord('previous word'))
    self.registerTask(CurrentWord('current word'))
    self.registerTask(NextWord('next word'))
    self.registerTask(PreviousChar('previous char'))
    self.registerTask(CurrentChar('current char'))
    self.registerTask(NextChar('next char'))
    self.registerTask(WhereAmI('where am i'))
    self.registerTask(FocusToPointer('focus to pointer'))
    self.registerTask(PointerToFocus('pointer to focus'))
    self.registerTask(ReadTop('read top'))
    self.registerTask(ReadTextColor('read text color'))
    self.registerTask(ReadTextAttributes('read text attributes'))
    self.registerTask(ReadDescription('read description'))
    self.registerTask(ReadMessage('read message'))
    self.registerTask(ReadNewLabel('read new label'))
    self.registerTask(ReadNewRole('read new role'))
    self.registerTask(ReadNewLevel('read new level'))
    self.registerTask(ReadNewHeaders('read new headers'))
    self.registerTask(ReadNewContainer('read new container'))
    self.registerTask(ReadItemIndex('read item index'))
    self.registerTask(ReadItemDetails('read item details'))
    self.registerTask(ReadAll('read all'))
    
    # cyclic tasks
    self.registerTask(Task.CyclicInputTask('cycle text attributes',
                                           'read text color',
                                           'read text attributes'))
    
    # stateful variables; OK to access from other Task classes in this module
    self.resetLasts()
  
  def resetLasts(self, container=''):
    '''
    Resets variables tracking the last container, caret position, tree level, 
    selection length, row/col offsets, and row/col count on view change.
    
    @param container: Last container name announced
    @type container: string
    '''
    self.last_container = container
    self.last_caret = POR(item_offset=0)
    self.last_role = None
    self.last_level = None
    self.last_sel_len = 0
    self.last_row = None
    self.last_col = None
    self.last_count = False

class Stop(Task.InputTask):
  '''
  Task to stop speech immediately, ignoring the value of the Stopping setting.
  '''
  def execute(self, **kwargs):
    self.stopAll()
    task = self.getNamedTask('continuous read')
    if task is not None:
      self.unregisterTask(task)

class IncreaseRate(Task.InputTask):
  '''
  Increase the speech rate. The maximum rate is announced when reached.

  @see: L{DecreaseRate}
  '''
  def execute(self, **kwargs):
    self.stopNow()
    rate = self.getRate()
    max_rate = self.getMaxRate()
    if rate < max_rate:
      rate += 10
      self.setRate(rate)
      self.sayInfo(text=(rate), template=('rate %d'))
      # notify about a change in setting in case the settings dialog is open
      style = self.getStyle()
      self.signalSetting(style, 'Rate', rate)
    else:
      self.sayInfo(text=_('maximum rate'))

class DecreaseRate(Task.InputTask):
  '''
  Decrease the speech rate. The minimum rate is announced when reached.

  @see: L{IncreaseRate}
  '''
  def execute(self, **kwargs):
    self.stopNow()
    rate = self.getRate()
    min_rate = self.getMinRate()
    if rate > min_rate:
      rate -= 10
      self.setRate(rate)
      self.sayInfo(text=rate, template=_('rate %d'))
      # notify about a change in setting in case the settings dialog is open
      style = self.getStyle()
      self.signalSetting(style, 'Rate', rate)
    else:
      self.sayInfo(text=_('minimum rate'))
      
class PreviousItem(Task.InputTask):
  '''
  Moves the POR to the beginning of the previous item and speaks it.
  '''
  def execute(self, stop=True, **kwargs):
    if stop: self.stopNow()
    # try to get the previous item
    por = self.getPrevItem(wrap=True, only_visible=self.perk_state.OnlyVisible)
    if por is None:
      # announce there is no prior item
      self.sayInfo(text=_('first item'))
      return
    # store the new POR and report info about it
    self.moveToPOR(por)
    text = self.getItemText()
    if not text.strip() and not self.hasAccState('focusable'):
      # try again, nothing of interest to report for item content
      self.doTask('read new role', por=por)
      return self.execute(False)
    else:
      self.doTask('read item details', por=por, text=text)

class CurrentItem(Task.InputTask):
  '''
  Moves the POR to the beginning of the current Item and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    por = self.getCurrItem()
    # move to the start of this item
    self.moveToPOR(por)
    # say information about the item
    self.doTask('read item details', por=por)

class NextItem(Task.InputTask):
  '''
  Moves the POR to the beginning of the next Item and speaks it.
  '''
  def execute(self, stop=True, **kwargs):
    if stop: self.stopNow()
    # try to get the next item
    por = self.getNextItem(wrap=True, only_visible=self.perk_state.OnlyVisible)
    if por is None:
      # announce there is no next item
      self.sayInfo(text=_('last item'))
      return
    # store the new POR and report info about it
    self.moveToPOR(por)
    text = self.getItemText()
    if not text.strip() and not self.hasAccState('focusable'):
      # try again, nothing of interest to report for item content
      self.doTask('read new role', por=por)
      return self.execute(False)
    else:
      self.doTask('read item details', por=por, text=text)
    
class PreviousWord(Task.InputTask):
  '''
  Moves the POR to the beginning of the previous word and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    # try to get the previous word
    por = self.getPrevWord()
    if por is not None:
      self.moveToPOR(por)
      self.sayWord()
      return
    if not self.perk_state.Wrap:
      # don't move to the previous item if wrapping is off
      self.sayInfo(text=_('first word'))
      return
    # get the last word of the previous item
    por = self.getPrevItem(wrap=True, only_visible=self.perk_state.OnlyVisible)
    if por is None:
      # bail if this is the first item also
      self.sayInfo(text=_('first item'))
      return
    por = self.getLastWord(por)
    # if it's still None, there's a problem
    if por is None:
      raise Task.PORError
    # move and do the announcement
    self.moveToPOR(por)
    # always announce the role as a way of indicating a new item
    self.sayRole()
    self.sayWord()

class CurrentWord(Task.InputTask):
  '''
  Moves the POR to the beginning of the current word and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    por = self.getCurrWord()
    # if it's None, there's a problem
    if por is None:
      raise Task.PORError
    self.moveToPOR(por)
    self.sayWord()

class NextWord(Task.InputTask):
  '''
  Moves the POR to the beginning of the next word and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    # try to get the next word
    por = self.getNextWord()
    if por is not None:
      self.moveToPOR(por)
      self.sayWord()
      return
    if not self.perk_state.Wrap:
      # don't move to the next item if wrapping is off
      self.sayInfo(text=_('last word'))
      return
    # get the first word of the next item
    por = self.getNextItem(wrap=True, only_visible=self.perk_state.OnlyVisible)
    if por is None:
      # bail if this is the last item also
      self.sayInfo(text=_('last item'))
      return
    # move and do the announcement
    self.moveToPOR(por)
    # always announce the role as a way of indicating a new item
    self.sayRole()
    self.sayWord()
    
class PreviousChar(Task.InputTask):
  '''
  Moves the POR to the beginning of the previous word and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    # try to get the previous character
    por = self.getPrevChar()
    if por is not None:
      self.moveToPOR(por)
      self.sayChar()
      return
    if not self.perk_state.Wrap:
      # don't move to the previous item if wrapping is off
      self.sayInfo(text=_('first character'))
      return
    # get the last char of the previous item
    por = self.getPrevItem(wrap=True, only_visible=self.perk_state.OnlyVisible)
    if por is None:
      # bail if this is the first item also
      self.sayInfo(text=_('first item'))
      return
    # get the last character
    por = self.getLastChar(por)
    # if it's still None, there's a problem
    if por is None:
      raise Task.PORError
    # move and do the announcement
    self.moveToPOR(por)
    # always announce the role as a way of indicating a new item
    self.sayRole()
    self.sayChar()

class CurrentChar(Task.InputTask):
  '''
  Moves the POR to the beginning of the current word and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    self.sayChar()

class NextChar(Task.InputTask):
  '''
  Moves the POR to the beginning of the next word and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    # try to get the next character
    por = self.getNextChar()
    if por is not None:
      self.moveToPOR(por)
      self.sayChar()
      return
    if not self.perk_state.Wrap:
      # don't move to the next item if wrapping is off
      self.sayInfo(text=_('last character'))
      return
    # get the last char of the next item
    por = self.getNextItem(wrap=True, only_visible=self.perk_state.OnlyVisible)
    if por is None:
      # bail if this is the last item also
      self.sayInfo(text=_('last item'))
      return
    # move and do the announcement
    self.moveToPOR(por)
    # always announce the role as a way of indicating a new item
    self.sayRole()
    self.sayChar()
    
class WhereAmI(Task.InputTask):
  '''
  Reports the location of the current L{task_por} by announcing all control and
  container names in child-to-parent order.
  '''
  def execute(self, **kwargs):
    # stop now and indicate the next stop is fine
    self.stopNow()
    # announce role and text
    self.sayRole()
    self.sayItem()
    for curr in self.iterAncestorAccs():
      # announce role and text
      self.sayRole(curr)
      self.sayItem(curr)

class FocusToPointer(Task.InputTask):
  '''
  Attempts to switch focus to try to set focus to L{POR} of accessible that was
  last reviewed in passive mode. Set the selection and caret offset properly 
  too if supported.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    self.setAccPOR()
    
class PointerToFocus(Task.InputTask):
  '''
  Jumps the L{task_por} back to the L{POR} of the last focus event.
  '''
  def execute(self, **kwargs):
    # set the Task POR to the focus POR
    self.task_por = self.getAccFocus()
    # stop all output and execute the default focus and selector handlers
    self.doTask('default focus', por=self.task_por, gained=True)
    self.doTask('read item details')    

class ReadItemIndex(Task.Task):
  '''
  Reads the row and/or column indices of the current item followed by the 
  total number of items. The total number are only announced if a new control
  has received focus.
  '''
  def execute(self, por=None, **kwargs):
    '''    
    Speaks the row and column indices of the given L{POR} or the L{task_por} if
    no L{POR} is given. Also speaks the total number of items if it has not
    been announced previously.
    
    @todo: PP: improve non-table cases where index is important (e.g. all menu
      items, radio groups)
    
    @param por: Point of regard to an object possibly having an index of 
      interest
    @type por: L{POR}
    '''
    # get the number of rows and cols
    exts = self.getAccTableExtents(por)
    if exts is None:
      if self.hasAccRole('menu item'):
        # accessible that does not implement the table interface
        i = self.getAccIndex(por)
        # get the parent count
        parent = self.getParentAcc()
        count = self.getAccCount(parent)
        if not self.perk.last_count:
          text = (i+1, count)
          template = _('item %d of %d')
        else:
          text = i+1
          template = _('item %d')
        # first is 1, not 0
        self.sayIndex(text=text, template=template)
        self.perk.last_count = True
    else:
      rows, cols = exts
      # always get the current row number
      row = self.getAccRow(por)
      print cols
      if cols <= 1:
        # lists
        if not self.perk.last_count:
          text = (row+1, rows)
          template = _('item %d of %d')
        else:
          text = row+1
          template = _('item %d')
        # first is 1, not 0
        self.sayIndex(text=text, template=template)
      else:
        # tables
        col = self.getAccColumn(por)
        if not self.perk.last_count:
          try:
            text = (row+1, col+1, rows, cols)
          except TypeError:
            # not in a cell
            return
          template = _('item %d, %d, of %d, %d')
        else:
          try:
            text = (row+1, col+1)
          except TypeError:
            # not in a cell
            return
          template = _('item %d, %d')
          # first is 1,1 , not 0,0
        self.sayIndex(text=text, template=template)
      self.perk.last_count = True

class ReadNewHeaders(Task.Task):
  '''
  Reads the row and/or column headers if they differ from the last ones
  announced.
  '''
  def execute(self, por=None, **kwargs):
    '''    
    Speaks the row and column headers of the given L{POR} or the L{task_por} if
    no L{POR} is given. Only makes an announcement if the last row/col
    encountered by this L{Task} or L{ReadNewHeaders} is different from the
    current.
    
    @param por: Point of regard to a table cell possibly having a header
    @type por: L{POR}
    '''
    row = self.getAccRow(por)
    if row is not None and row != self.perk.last_row:
      header = self.getAccRowHeader(por)
      if header:
        self.saySection(text=header, template=_('row %s'))
      self.perk.last_row = row
    col = self.getAccColumn(por)
    if col is not None and col != self.perk.last_col:
      header = self.getAccColumnHeader(por)
      if header:
        self.saySection(text=header, template=_('column %s'))
      self.perk.last_col = col

class ReadNewRole(Task.Task):
  '''
  Reads the role of a widget if it is different from the last role read.
  '''
  def execute(self, por=None, **kwargs):
    '''
    Speaks the role of the given L{POR} or the L{task_por} if no L{POR} is 
    given only if it is different than the last role spoken by this method or
    if the L{DefaultPerkState.RepeatRole} flag is True.
    
    @param por: Point of regard to the accessible whose role should be said, 
      defaults to L{task_por} if None
    @type por: L{POR}
    '''
    role = self.getAccRoleName(por)
    if self.perk_state.RepeatRole or role != self.perk.last_role:
      self.sayRole(text=role)
      self.perk.last_role = role
      
class ReadNewLevel(Task.Task):
  '''
  Reads the tree level of a widget if it is different from the last level read.
  '''
  def execute(self, por=None, **kwargs):
    '''
    Speaks the level of the given L{POR} or the L{task_por} if no L{POR} is 
    given only if it is different than the last level spoken by this method.
    
    @param por: Point of regard to the accessible whose role should be said, 
      defaults to L{task_por} if None
    @type por: L{POR}
    '''
    level = self.getAccLevel(por)
    if level is not None and level != self.perk.last_level:
      # root is 1
      self.sayLevel(text=level+1, template=_('level %d'))
      self.perk.last_level = level

class ReadNewLabel(Task.Task):
  '''
  Read the label on a widget if it is different from the item text of the
  widget and the last container label read.
  '''
  def execute(self, por=None, **kwargs):
    '''
    Speaks the text of a new label at the given L{POR} or at the L{task_por} 
    if no L{POR} is given only if it is different than the item text and the
    last container announced by L{ReadNewContainer}.
    
    @param por: Point of regard to an accessible that should be announced if it
      is a new label
    @type por: L{POR}
    '''
    label = self.getAccLabel(por)
    if (label and label != self.perk.last_container):
      self.sayLabel(text=label)

class ReadNewContainer(Task.Task):
  '''
  Read the label of the container of the current item as long as it is not the
  same as the label of the container last announced.
  '''
  def execute(self, por=None, **kwargs):
    '''
    Speaks the first interesting ancestor of the given L{POR} if it is 
    different from the last interesting ancestor announced.

    @param por: Point of regard to an accessible that should be announced if it
      is a new menu.
    @type por: L{POR}
    '''
    # get the root accessible
    root = self.getRootAcc()
    for anc in self.iterAncestorAccs(por, allow_trivial=True):
      # don't count the application itself
      if self.hasAccRole('application', anc):
        break
      to_check = [anc]
      # try the previous peer too if it's a label
      prev = self.getPrevPeerAcc(anc)
      if (prev and self.hasAccRole('label', prev) and 
          not self.getAccRelations('label for', prev)):
        to_check.append(prev)
      for acc in to_check:
        name_text = self.getAccName(acc)
        label_text = self.getAccLabel(acc)
        last = self.perk.last_container
        if (((name_text and last.startswith(name_text)) or
            (label_text and last.startswith(label_text)) or
            self.hasAccRole('menu bar', acc))):
          return
        if name_text and name_text.strip():
          # try to announce the name text first
          self.saySection(text=name_text)
          self.perk.last_container = name_text
          return
        if label_text and label_text.strip():
          # then try the label
          self.saySection(text=label_text)
          self.perk.last_container = label_text
          return
          
class ReadMessage(Task.Task):
  '''
  Reads a programmatic LSR announcement.
  '''
  def execute(self, text, sem=LSRConstants.SEM_INFO, **kwargs):
    '''
    Makes an arbitrary announcement using the given semantic for styling.
    
    @param text: Text to announce
    @type text: string
    @param sem: Semantic of the announcement. Defaults to info.
    @type sem: integer
    '''
    self.say(text, sem=sem)

class ReadAll(Task.InputTask):
  '''
  Read the remaining contents of the active view starting at the current
  point of regard.
  '''
  def execute(self, interval=None, **kwargs):
    self.stopNow()
    self.registerTask(_ContinueReadAll(1, 'continuous read'))
    self.doTask('continuous read')

class _ContinueReadAll(Task.TimerTask):
  '''
  Continue queuing up contents in the view on a timed interval.
  
  @todo: PP: if index markers are supported, update the start_por accordingly
  so a stop leaves the user near the last reviewed location, or maybe this is
  an option?
  
  @ivar start_por: Starting point of regard for the read operation
  @type start_por: L{POR}
  @ivar continue_por: Point of regard where reading should continue
  @type continue_por: L{POR}
  '''
  def init(self):
    self.start_por = None
    self.continue_por = None
  
  def execute(self, **kwargs):
    # initialize the start point
    self.start_por = self.start_por or self.task_por
    # always read as if focused
    self.layer = Task.LAYER_FOCUS
    end = self.getEndAcc()
    if self.task_por == self.start_por:
      self.task_por = self.continue_por or self.task_por
      # continue where we left off
      for i in xrange(5):
        self.doTask('next item', stop=False)
        if self.task_por == end:
          # no movement, end of view
          self.unregisterTask(self)
          break
    else:
      # task por changed since last continuation, abort
      self.unregisterTask(self)
    # store the last review position
    self.continue_por = self.task_por
    # restore the starting por
    self.task_por = self.start_por

class ReadTop(Task.InputTask):
  '''
  Read top of the active view, typically the title of the foreground window.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    self.sayWindow()
    
class _BaseAttributes(Task.InputTask):
  '''
  Base class for other Tasks that will read text attributes.
  '''
  def _getAttrs(self):
    '''
    Reusable method for getting default and current text attributes and 
    merging them into one dictionary.
    
    @return: Dictionary of name/value attribute pairs
    @rtype: dictionary
    '''
    attrs = self.getAccAllTextAttrs()
    dattrs = self.getAccDefTextAttrs()
    
    if not attrs and not dattrs:
      # no atttribute information
      self.sayInfo(text=_('text attributes unavailable'))
      return
    elif not attrs:
      # only default attribute information
      attrs = dattrs
    elif not dattrs:
      # only current attribute information
      pass
    else:
      # overwrite default with current
      attrs = dattrs.update(attrs)
    return attrs
      
class ReadTextColor(_BaseAttributes):
  '''
  Read the text foreground and background color names followed by their values.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    # get merged attributes
    attrs = self._getAttrs()

    # get foreground value
    fgval = attrs.get('fg-color')
    fgname = self.getColorString(fgval) or _('default foreground') 
    
    # get background value
    bgval = attrs.get('bg-color')
    bgname = self.getColorString(bgval) or _('default background')

    # i18n: foreground on background color (e.g. "black on white")
    template = _('%s on %s.') 
    names = template % (fgname, bgname)
    if bgval is None and fgval is None:
      self.sayColor(text=names)
    elif bgval is not None and fgval is not None:
      self.sayColor(text=(fgval, bgval), template=names + ' ' + template)
    elif fgval is not None:
      self.sayColor(text=fgval, template=names + ' %s')
    elif bgval is not None:
      self.sayColor(text=bgval, template=names + ' %s')
      
class ReadTextAttributes(_BaseAttributes):
  '''
  Read all remaining text attribute name/value pairs sans color information.
  
  @todo: PP: need localizations for all attribute names and text values, or
    some reasonable subset thereof
  @todo: PP: probably want names for numbers that have standard levels (e.g. 
    font weight 400 is bold)
  '''
  def execute(self, **kwargs):
    self.stopNow()
    attrs = self._getAttrs()
    for name, value in attrs.items():
      if not name.endswith('color'):
        self.sayTextAttrs(text=(name, value), template='%s: %s')

class ReadDescription(Task.InputTask):
  '''
  Read the description of the accessible at the current point of regard.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    self.sayDesc()
  
class ReadItemDetails(Task.Task):
  '''
  Reads all details of the current item.
  '''
  def execute(self, text=None, **kwargs):
    if text is None:
      text = self.getItemText()
    # announce the item role
    #if self.getAccFocus() != self.task_por:
    self.doTask('read new role')
    # announce the item text only if it is different than the label
    label = self.getAccLabel()
    if (label != text):
      self.sayItem(text=text)
    # try to announce the headers on the row and column
    self.doTask('read new headers')
    # see if there are any states worth reporting
    self.sayState()
    # try to announce the level
    self.doTask('read new level')
    # try to announce the item index
    self.doTask('read item index')
    # try to announce the total number of items
    self.doTask('read item count')
    # get the hotkey for this item
    self.sayHotkey()
    if not text:
      # only announce value if it isn't encoded in the item text
      self.sayValue()
    # i18n hint: a numeric range by a step (e.g. 0.0 to 100.0 by 2.0)
    self.sayValueExtents(template=_('%.1f to %.1f by %.1f'))

class HandleViewChange(Task.ViewTask):
  '''
  Task that handles a L{AEEvent.ViewChange}.
  
  @todo: PP: Pause before saying 'No view'
  '''
  def executeLost(self, por, title, **kwargs):  
    return True
  
  def executeFirstGained(self, por, title, **kwargs):
    '''    
    Announces the title of the first view to be activated, but without stopping
    previous speech which is likely to be the LSR welcome message. Inhibits the
    next stop to avoid never announcing the title because of another immediate
    event following this one (i.e. focus).
    
    @param por: Point of regard to the root of the new view
    @type por: L{POR}
    @param title: Title of the newly activated view
    @type title: string
    '''
    # say the title of the new view
    if not self.sayWindow(text=title):
      self.sayWindow(text=_('no title'))
    # reset stateful vars
    self.perk.resetLasts(title)
    return True
  
  def executeGained(self, por, **kwargs):
    '''
    Stops output then calls L{executeFirstGained} to reuse its code for 
    announcing the new view.
    '''
    # stop output
    self.stopNow()
    rv = self.executeFirstGained(por, **kwargs)
    # prevent the next event (focus?) from stopping this announcement
    self.inhibitMayStop()
    # see if we want to handle this new view in a special way
    if self.hasAccRole('alert', por):
      self.registerTask(ReadAlert())
    return rv
    
class HandleFocusChange(Task.FocusTask):
  '''  
  Task that handles a L{AEEvent.FocusChange}.
  '''  
  def executeGained(self, por, **kwargs):
    '''
    Annouces the label and role of the newly focused widget. The role is output
    only if it is different from the last one announced.
    
    @param por: Point of regard for the new focus
    @type por: L{POR}
    '''
    self.mayStop()
    # announce containers when they change
    self.doTask('read new container', por=por)
    # announce the label when present if it is not the same as the name or the
    # container
    self.doTask('read new label', por=por)
    # only announce role when it has changed
    self.doTask('read new role', por=por)
    # prevent the next selector or caret from stopping this announcement
    self.inhibitMayStop()
    # always reset last caret position
    self.perk.last_caret = POR(item_offset=0)
    self.perk.last_count = False
    return True

class HandleSelectorChange(Task.SelectorTask):
  ''' 
  Task that handles a L{AEEvent.SelectorChange}.
  '''
  def executeActive(self, por, text, **kwargs):
    '''
    Announces the text of the active item.
    
    @param por: Point of regard where the selection event occurred
    @type por: L{POR}
    @param text: The text of the active item
    @type text: string
    '''
    self.mayStop()
    self.doTask('read item details', por=por, text=text)
    
  def executeText(self, por, text, **kwargs):
    '''
    Announces the localized word selected or unselected followed by the total 
    number of characters in the selection.
    
    @param por: Point of regard where the selection event occurred
    @type por: L{POR}
    @param text: The currently selected text
    @type text: string
    '''
    n = len(text)
    if n > self.perk.last_sel_len:
      self.sayItem(text=n, template=_('selected %d'))
    else:
      self.sayItem(text=n, template=_('unselected %d'))
    self.perk.last_sel_len = n

class HandleCaretChange(Task.CaretTask):
  '''  
  Task that handles a L{AEEvent.CaretChange}. The L{execute} method performs
  actions that should be done regardless of whether text was inserted or
  deleted or the caret moved. The other three methods handle these more
  specific cases.
  
  @ivar changed: Tracks whether the last caret move event was a caused by a
    change (insert/delete) or just navigation
  @type changed: boolean
  '''
  def init(self):
    self.changed = False
  
  def execute(self, por, text, text_offset, added, **kwargs):
    '''
    Stores the point of regard to the caret event location. Tries to stop
    current output.
    
    @param por: Point of regard where the caret event occurred
    @type por: L{POR}
    @param text: The text inserted, deleted or the line of the caret
    @type text: string
    @param text_offset: The offset of the inserted/deleted text or the line 
      offset when movement only
    @type text_offset: integer
    @param added: True when text added, False when text deleted, and None 
      (the default) when event is for caret movement only
    @type added: boolean
    '''
    # let the base class call the appropriate method for inset, move, delete
    rv = Task.CaretTask.execute(self, por, text, text_offset, added, **kwargs)
    # store for next comparison
    self.perk.last_caret = por
    return rv
    
  def executeInserted(self, por, text, text_offset, **kwargs):
    '''
    Announces the inserted text.
    
    @param por: Point of regard where the caret event occurred
    @type por: L{POR}
    @param text: The text inserted
    @type text: string
    @param text_offset: The offset of the inserted text
    @type text_offset: integer
    '''
    #print 'CHANGED: insert'
    self.changed = True
    if len(text) == 1:
      w = (self.perk_state.WordEcho and 
           unicodedata.category(text)[0] in ('C', 'Z'))
      c = self.perk_state.CharEcho
      if w and c:
        # word and character echo
        self.mayStop()
        self.sayChar(text=text)
        self.sayWord(por)
      elif w:
        # word echo
        self.mayStop()
        self.sayWord(por)
      elif c:
        # character echo
        self.mayStop()
        self.sayChar(text=text)
    else:
      # chunk of text, say the whole thing for now
      self.mayStop()
      self.sayItem(text=text)
    
  def executeMoved(self, por, text, text_offset, **kwargs):
    '''
    Announces the text passed in the caret movement.
    
    @param por: Point of regard where the caret event occurred
    @type por: L{POR}
    @param text: The text passed during the move
    @type text: string
    @param text_offset: The offset of the new caret position
    @type text_offset: integer
    '''
    # don't echo again if a change just happened
    if self.changed:
      #print 'UNCHANGED: moved'
      self.changed = False
      return
    
    if not por.isSameAcc(self.perk.last_caret):
      # say the current line
      self.mayStop()
      self.sayItem(text=text)
      if self.getAccTextSelectionCount() > 0:
        # and the total number of selected characters if there's a selection
        self.sayTextAttrs(text=len(self.getAccTextSelection()[0]),
                          template=_('selected %s'))
    elif not por.isSameItem(self.perk.last_caret):
      # say the new line when item_offset changes
      self.mayStop()
      self.sayItem(text=text)
    else:
      #print 'move:', por, self.perk.last_caret
      # respond to caret movement on same line
      if por.isCharBefore(self.perk.last_caret):
        # moved left, say text between positions
        diff = text[por.char_offset : self.perk.last_caret.char_offset]
        self.mayStop()
        self.sayItem(text=diff)
      elif por.isCharAfter(self.perk.last_caret):
        # moved right, say text between positions
        diff = text[self.perk.last_caret.char_offset : por.char_offset]
        self.mayStop()
        self.sayItem(text=diff)
    
  def executeDeleted(self, por, text, text_offset, **kwargs):
    '''
    Announces the deleted text. Prepends delete and backspace text to indicate
    which way the caret moved.
    
    @param por: Point of regard where the caret event occurred
    @type por: L{POR}
    @param text: The text delete
    @type text: string
    @param text_offset: The offset of the delete text
    @type text_offset: integer
    '''
    self.mayStop()
    if (self.perk.last_caret.item_offset + 
        self.perk.last_caret.char_offset) > text_offset:
      # say backspace if we're backing up
      self.sayItem(text=text, template=_('backspace %s'))
      self.changed = True
      #print 'CHANGED: delete'
    else:
      # say delete if we're deleting the current
      self.sayItem(text=text, template=_('delete %s'))
      self.changed = False

  def update(self, por, text, text_offset, added, **kwargs):
    '''
    Updates the L{DefaultPerk}.last_caret with the current L{por}.
    
    @param por: Point of regard where the caret event occurred
    @type por: L{POR}
    @param text: The text inserted, deleted or the line of the caret
    @type text: string
    @param text_offset: The offset of the inserted/deleted text or the line 
      offset when movement only
    @type text_offset: integer
    @param added: True when text added, False when text deleted, and None 
      when event is for caret movement only
    @type added: boolean
    '''
    # store for next comparison
    self.perk.last_caret = por
    self.changed = False
    #print 'UNCHANGED: update'
  
class HandleStateChange(Task.StateTask):
  '''  
  Task that handles a L{AEEvent.StateChange}.
  '''
  def execute(self, por, name, value, **kwargs):
    '''
    Announces state changes according to the ones of interest defined by
    L{getStateText}.
    
    @param por: Point of regard where the state change occurred
    @type por: L{POR}
    @param name: Name of the state that changed
    @type name: string
    @param value: Is the state set (True) or unset (False)?
    @type value: boolean
    '''
    text = self.getStateText(por, name, value)
    if text:
      # try a may stop
      self.mayStop()
      self.sayState(text=text)
    return True

class HandlePropertyChange(Task.PropertyTask):
  '''
  Task that handles a L{AEEvent.PropertyChange}.
  '''
  def execute(self, por, name, value, **kwargs):
    '''
    Announces value changes.
    
    @param por: Point of regard where the property change occurred
    @type por: L{POR}
    @param name: Name of the property that changed
    @type name: string
    @param value: New value of the property
    @type value: object
    '''
    if name == 'value':
      if not self.getItemText(por):
        # only announce value if we're not going to get a text change event
        # immediately after saying the same thing
        self.mayStop()
        self.sayItem(text=value)
    
class ReadAlert(Task.FocusTask):
  '''
  Task registered temporarily when an alert dialog is shown. Collects all 
  labels of interest in the dialog and says them in order.
  '''
  def executeGained(self, **kwargs):
    por = self.getViewRootAcc()
    # start iteration at the root
    for por in self.iterNextAccs(por):
      # announce all labels
      if self.hasAccRole('label', por):
        self.sayItem(por)
    # unregister this task
    self.unregisterTask(self)
    return True
