'''
Defines a L{Perk} that keeps track of user assigned bookmarks to L{POR}s in 
an application.

@author: Scott Haeger
@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2007 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made 
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''
# import useful modules for Perks
import Perk, Task
from POR import POR
from i18n import _
from AEConstants import *

__uie__ = dict(kind='perk', tier=None, all_tiers=True)

class BookmarkPerk(Perk.Perk):
  '''
  Defines a user interface into a collection of bookmarked L{POR}s.
  
  @todo: PP: how to support bindings to other devices? methods/vars on this
    perk which do the registration and keep track of what's bound when we get
    an event later?
  
  @ivar bookmarks: collection of PORs
  @type bookmarks: dictionary
  @ivar keys: List of keys to which the bookmark commands are bound
  @type keys: list
  
  '''
  def init(self):
    self.bookmarks = {}
    self.keys = []
    
    self.registerTask(HandleAddBookmark('bookmark add bookmark'))
    self.registerTask(HandleGotoBookmark('bookmark goto bookmark'))
    self.registerTask(HandleReviewBookmark('bookmark review bookmark'))
 
    # register keyboard bindings
    kbd = self.getInputDevice(None, 'keyboard')
    # set Alt+CapsLock as a modifier
    self.addInputModifiers(kbd, kbd.AEK_ALT_L, kbd.AEK_ALT_R, 
                           kbd.AEK_CAPS_LOCK)
    # set Alt+Shift as a modifier
    self.addInputModifiers(kbd, kbd.AEK_ALT_L, kbd.AEK_SHIFT_L, kbd.AEK_ALT_R,
                           kbd.AEK_SHIFT_R)
                                                  
    self.keys = [kbd.AEK_2, kbd.AEK_3, kbd.AEK_4, kbd.AEK_5, kbd.AEK_6,
                 kbd.AEK_7, kbd.AEK_8, kbd.AEK_9, kbd.AEK_0]
    modifiers = [[kbd.AEK_ALT_L, kbd.AEK_CAPS_LOCK],
                 [kbd.AEK_ALT_R, kbd.AEK_CAPS_LOCK]]
    # Register for keyboard input command Alt-CapsLock-(2-0).
    # Keyboard keys 2 - 0 (9 keys in kb layout order)
    # Adds bookmark          
    for modifier in modifiers:
      for key in self.keys:
        self.registerCommand(kbd, 'bookmark add bookmark', False,
                             modifier+[key])
        
    # Register for keyboard input command Alt-Shift-(2-0).
    # Keyboard keys 2 - 0 (9 keys in kb layout order)
    # Goto bookmark
    modifiers = [[kbd.AEK_ALT_L, kbd.AEK_SHIFT_L], 
             [kbd.AEK_ALT_R, kbd.AEK_SHIFT_R]]
    for modifier in modifiers:
      for key in self.keys:
        self.registerCommand(kbd, 'bookmark goto bookmark', False,
                             modifier+[key])
    # Register for keyboard input command Alt-Shift-tilde.
    # Review bookmark
    for modifier in modifiers:
      self.registerCommand(kbd, 'bookmark review bookmark', False,
                           modifier+[kbd.AEK_TILDE])
      
  def sayBookmark(self, por):
    '''
    Speaks information about the given L{POR} using an audio output device.
    
    @todo: PP: should probably call an existing task to report item information
      like 'read item details' or something
    
    @param por: Point of regard for a bookmark
    @type por: L{POR}
    '''
    self.sayApp(por=por, talk=False)
    self.sayItem(por=por, talk=False)
    self.sayRole(por=por)
    
  def getName(self):
    return _('Bookmarks')
  
  def getDescription(self):
    return _('Allows the user to bookmark locations in an application and '
             'return to them using hotkeys at a later time.')
  
class HandleAddBookmark(Task.CyclicInputTask):
  '''
  Adds a new bookmark. If a current, different bookmark will be overwritten, 
  warns the user first and requires they activate this task again.
  '''
  def execute(self, cycle_count, gesture=None, **kwargs):
    if gesture is None: return
    keys = gesture.getSortedActionCodes()
    index = keys[0]
      
    self.stopNow()
    if cycle_count == 0:
      if self.perk.bookmarks.has_key(index):
        self.sayInfo(text=_('activate again to overwrite bookmark %d') %
                     (self.perk.keys.index(index)+2))
      else:
        self.sayInfo(text=_('adding bookmark %d') %
                     (self.perk.keys.index(index)+2))
        self.perk.bookmarks[index] = self.task_por
    elif self.perk.bookmarks[index] == self.task_por:
      self.sayInfo(text=_('already bookmarked'))
    else:
      self.sayInfo(text=_('overwriting bookmark %d.') %
                   (self.perk.keys.index(index)+2))
      self.perk.bookmarks[index] = self.task_por
    
class HandleGotoBookmark(Task.InputTask):
  '''
  Triggers a 'pointer to por' event using the bookmarked L{POR}.
  '''
  def execute(self, gesture=None, **kwargs):
    if gesture is None: return
    keys = gesture.getSortedActionCodes()
    index = keys[0]
    
    self.stopNow()
    if self.perk.bookmarks.has_key(index):
     # self.moveToPOR(self.perk.bookmarks[index])
      self.doTask('pointer to por', por=self.perk.bookmarks[index])
    else:
      self.sayError(text=_('no bookmark registered'))
      
class HandleReviewBookmark(Task.InputTask):
  '''
  Reviews all bookmarks saved in the current application.
  '''
  def execute(self, gesture=None, **kwargs):
    self.stopNow()
    if len(self.perk.bookmarks) == 0:
      self.sayInfo(text=_("no bookmarks are registered"))
    else:
      # get handle to keyboard constants
      bm_num = 2
      unreg = []
      for i in self.perk.keys:
        if self.perk.bookmarks.has_key(i): 
          self.sayInfo(text=_("bookmark %d: ") % bm_num, talk=False)
          self.perk.sayBookmark(self.perk.bookmarks[i])
        else:
          unreg.append(str(bm_num))
        bm_num += 1
      if len(unreg) > 0:
        self.sayInfo(text=_("Bookmarks %s not registered.") % ', '.join(unreg))
