'''
Defines L{Task}s for searching through accessible tree.

@author: Scott Haeger
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2007 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made 
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''
# import useful modules for Perks
import Perk, Task, AEConstants
from POR import POR
from i18n import _

__uie__ = dict(kind='perk', all_tiers=True)

class SearchPerk(Perk.Perk):
  '''
  Manages the search dialog and performs search through accessible tree using
  query input by the user.
  '''
  def init(self):
    '''
    
    '''
    self.chooser = None
    
    # Register for search specific tasks
    self.registerTask(SearchChooserTask('search show chooser'))
    self.registerTask(SearchFindNext('search find next'))
    self.registerTask(SearchFindNext('search find prev'))
    
  def getName(self):
    return _('Search')
  
  
class SearchFindNext(Task.InputTask):
  '''
  Search through the accessible tree in the forward (down the tree) direction.
  '''
  def execute(self, **kwargs):
    if self.perk.chooser is None:
      return False
    query = self.perk.chooser.getQuery()
    if query is None or query == '':
      return False
    # define predicates
    if self.perk.chooser.isMatchCaseSet():
      def pred(acc):
        return self.getAccName(por=acc) == query
    else:
      querylower = query.lower()
      def pred(acc):
        return self.getAccName(por=acc).lower() == querylower
    # do search
    for acc in self.iterNextItems(wrap=True):
      if pred(acc):
        # need to turn autopointer on for 'pointer to por'
        self.tier.setAutoPointer(True)  
        self.doTask('pointer to por', por=acc)
        self.tier.setAutoPointer(False)  
        return
    if self.perk.chooser.isWrapSet():
      for acc in self.iterNextItems(por=self.getRootAcc(), wrap=True):
        if pred(acc):
          self.tier.setAutoPointer(True)  
          self.doTask('pointer to por', por=acc)
          self.tier.setAutoPointer(False)  
          return    
    self.doTask('read message', text=_('%s not found') %query) 
    
    
class SearchFindPrev(Task.InputTask):
  '''
  Search through the accessible tree in the backward (up the tree) direction.
  '''
  def execute(self, **kwargs):
    if self.perk.chooser is None:
      return False
    query = self.perk.chooser.getQuery()
    if query is None or query == '':
      return False
    # define predicates
    if self.perk.chooser.isMatchCaseSet():
      def pred(acc):
        return self.getAccName(por=acc) == query
    else:
      querylower = query.lower()
      def pred(acc):
        return self.getAccName(por=acc).lower() == querylower
    
    for acc in self.iterPrevItems(wrap=True):
      if pred(acc):
        self.tier.setAutoPointer(True)
        self.doTask('pointer to por', por=acc)
        self.tier.setAutoPointer(False)
        return
    if self.perk.chooser.isWrapSet():
      for acc in self.iterPrevItems(por=self.getEndAcc(), wrap=True):
        if pred(acc):
          self.tier.setAutoPointer(True)
          self.doTask('pointer to por', por=acc)
          self.tier.setAutoPointer(False)
          return    
    self.doTask('read message', text=_('%s not found') %query) 
    

class SearchChooserTask(Task.ChooserTask):
  '''  
  Shows the Search chooser which allows a user to search the current view for
  a given accessible.
  '''
  def executeStart(self, timestamp, **kwargs):
    '''
    Runs when an input gesture is given to start the chooser. 
    
    @param timestamp: Time to provide to the chooser dialog so it can raise 
      itself to the foreground properly
    @type timestamp: float
    '''
    # load the chooser if it is not loaded
    if self.perk.chooser is None:
      self.loadChooser('SearchChooser', app_name=self.getAppName(), 
                                      timestamp=timestamp)  
      # Turn off pointer updating   
      self.tier.setAutoPointer(False)
    # close the search dialog on second gesture
    else:
      self.perk.chooser.close()
      self.perk.chooser = None
      
  def executeSignal(self, chooser, kind, layer, **kwargs):
    '''    
    Runs when the chooser sends a signal to this L{Perk} indicating some
    change in the dialog that the L{Perk} should be aware of.  In this 
    chooser, the signal will only be used to update the perk instance 
    variable "chooser".
    
    @param chooser: Reference to the chooser object that sent the signal
    @type chooser: L{AEChooser}
    @param kind: Signal name
    @type kind: string
    @param layer: Ignored. Pulled out of kwargs so they can be passed to
      L{doTask} without modification
    @type layer: integer
    @param kwargs: Keyword arguments specific to each signal type
    @type kwargs: dictionary
    '''
    if kind == chooser.START:
      self.perk.chooser = chooser
  
  def executeEnd(self, chooser, **kwargs):
    # turn pointer updating back on.
    self.perk.unregisterChooserTask(chooser)
    self.tier.setAutoPointer(True)
      