/* mainpagetable.c
 *
 * Copyright (C) 1999 - 2002 Vivien Malerba
 * Copyright (C) 2002 Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <libgnomedb/gnome-db-grid.h>
#include <libgnomedb/gnome-db-stock.h>
#include <libgnomedb/gnome-db-util.h>
#include "mainpagetable.h"
#include "tableedit.h"
#include "query.h"
#include "query-env.h"
#include "query-exec.h"

static void main_page_table_class_init (MainPageTableClass * class);
static void main_page_table_init (MainPageTable * wid);
static void main_page_table_initialize (MainPageTable * wid);
static void main_page_table_finalize (GObject *object);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

typedef struct {
	ConfManager *conf;
	DbTable     *table;
	Query       *query;
	QueryEnv    *env;
	GtkWidget   *props_dlg;
	GtkWidget   *exec_dlg;
} Row_Data;


/*
 * static functions 
 */
static void selection_made (GnomeDbGrid * grid, gint row, gpointer data);
static void selection_cleared (GnomeDbGrid * grid, gpointer data);
static void grid_double_clicked_cb (GnomeDbGrid * grid, gint row, MainPageTable * mpt);
static void grid_create_popup_cb (GnomeDbGrid * grid, GtkWidget *menu, MainPageTable * mpt);
static void remove_table_cb (GObject   * obj, MainPageTable * wid);
static void edit_table_cb (GObject   * obj, MainPageTable * wid);
static void view_records_button_cb (GObject   * obj, MainPageTable * mpt);
static void database_added_cb (ConfManager *conf, Database *db, MainPageTable * mpt);
static void database_removed_cb (ConfManager *conf, Database *db, MainPageTable * mpt);
static void main_page_table_conn_close_cb (GObject   * obj,
					   MainPageTable * mpt);
static void main_page_table_add_cb (GObject   * obj, DbTable * table,
				    MainPageTable * wid);
static void main_page_table_drop_cb (GObject   * obj, DbTable * table,
				     MainPageTable * wid);
static void main_page_db_updated_cb (Database * db, MainPageTable * mpt);


guint
main_page_table_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MainPageTableClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) main_page_table_class_init,
			NULL,
			NULL,
			sizeof (MainPageTable),
			0,
			(GInstanceInitFunc) main_page_table_init
		};		

		type = g_type_register_static (GTK_TYPE_VBOX, "MainPageTable", &info, 0);
	}
	return type;
}

static void
main_page_table_class_init (MainPageTableClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	parent_class = g_type_class_peek_parent (class);

	object_class->finalize = main_page_table_finalize;
}

static void
main_page_table_init (MainPageTable * wid)
{
	GtkWidget *bb;
	GdaDataModel *model;

	/* Toolbar */
	bb = gnome_db_new_toolbar_widget (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
	gtk_box_pack_start (GTK_BOX (wid), bb, FALSE, TRUE, GNOME_PAD / 2);
	gtk_widget_show (bb);

	wid->new_table = gtk_toolbar_insert_stock (GTK_TOOLBAR (bb), GTK_STOCK_NEW,
						   _("Create a new table"), NULL,
						   NULL, NULL, -1); /* FIXME */
	wid->remove_table = gtk_toolbar_insert_stock (GTK_TOOLBAR (bb), GTK_STOCK_DELETE,
						      _("Drop selected table"), NULL,
						      GTK_SIGNAL_FUNC (remove_table_cb), wid, -1);
	wid->edit_table = gtk_toolbar_insert_stock (GTK_TOOLBAR (bb), GTK_STOCK_PROPERTIES,
						    _("View selected table properties"), NULL,
						    GTK_SIGNAL_FUNC (edit_table_cb), wid, -1);
	wid->view_table = gtk_toolbar_insert_stock (GTK_TOOLBAR (bb), GNOME_DB_STOCK_TABLES,
						    _("View records for the selected table"), NULL,
						    GTK_SIGNAL_FUNC (view_records_button_cb), wid, -1);

	/* table list */
	model = gda_data_model_array_new (4);
	gda_data_model_set_column_title (model, 0, _("Name"));
	gda_data_model_set_column_title (model, 1, _("Type"));
	gda_data_model_set_column_title (model, 2, _("Owner"));
	gda_data_model_set_column_title (model, 3, _("Comments"));

	wid->list = gnome_db_grid_new_with_model (model);
	g_object_unref (G_OBJECT (model));
	gnome_db_grid_set_selection_mode (GNOME_DB_GRID (wid->list), GTK_SELECTION_SINGLE);
	gtk_box_pack_start (GTK_BOX (wid), wid->list, TRUE, TRUE, 2);
	gtk_widget_show (wid->list);
	g_signal_connect (G_OBJECT (wid->list), "row_selected",
			  G_CALLBACK (selection_made), wid);
	g_signal_connect (G_OBJECT (wid->list), "selection_cleared",
			  G_CALLBACK (selection_cleared), wid);
	g_signal_connect (G_OBJECT (wid->list), "double_clicked",
			  G_CALLBACK (grid_double_clicked_cb), wid);
	g_signal_connect (G_OBJECT (wid->list), "create_popup_menu",
			  G_CALLBACK (grid_create_popup_cb), wid);

	wid->sel_row = -1;
}

GtkWidget *
main_page_table_new (ConfManager * conf)
{
	GObject   *obj;
	MainPageTable *wid;

	obj = g_object_new (MAIN_PAGE_TABLE_TYPE, NULL);
	wid = MAIN_PAGE_TABLE (obj);
	wid->conf = conf;

	main_page_table_initialize (wid);

	g_signal_connect (G_OBJECT (conf), "database_added",
			  G_CALLBACK (database_added_cb), wid);

	g_signal_connect (G_OBJECT (conf), "database_removed",
			  G_CALLBACK (database_removed_cb), wid);

	g_signal_connect (G_OBJECT (conf->srv), "conn_closed",
			  G_CALLBACK (main_page_table_conn_close_cb), obj);

	return GTK_WIDGET (obj);
}

static void main_page_table_finalize (GObject *object)
{
	/* FIXME: think about all the Row_Data with Query objects... */
	
	parent_class->finalize (object);
}

static void database_added_cb (ConfManager *conf, Database *db, MainPageTable * mpt)
{
	g_signal_connect (G_OBJECT (conf->db), "table_created",
			  G_CALLBACK (main_page_table_add_cb),
			  mpt);
	g_signal_connect (G_OBJECT (conf->db), "table_dropped",
			  G_CALLBACK (main_page_table_drop_cb),
			  mpt);
	g_signal_connect (G_OBJECT (conf->db), "updated",
			  G_CALLBACK (main_page_db_updated_cb),
			  mpt);
}

static void database_removed_cb (ConfManager *conf, Database *db, MainPageTable * mpt)
{
	/* nothing to do about it */
}


static void
main_page_table_initialize (MainPageTable * wid)
{
	gtk_widget_set_sensitive (wid->remove_table, FALSE);
	gtk_widget_set_sensitive (wid->edit_table, FALSE);
	gtk_widget_set_sensitive (wid->view_table, FALSE);
	conf_manager_register_sensitive_on_connect (wid->conf,
						    GTK_WIDGET (wid->new_table));
}

void
main_page_table_set_data_dlg (MainPageTable * mpt, guint row, GtkWidget * dlg)
{
	Row_Data *rdata = NULL;

	rdata = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (mpt->list), row);
	if (rdata)
		rdata->props_dlg = dlg;
}

GtkWidget *
main_page_table_get_data_dlg (MainPageTable * mpt, guint row)
{
	Row_Data *rdata = NULL;

	rdata = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (mpt->list), row);
	if (rdata)
		return rdata->props_dlg;
	else
		return NULL;
}

gint
main_page_table_get_row (MainPageTable * mpt, DbTable * table)
{
	gint i = 0;
	gboolean found = FALSE;
	Row_Data *rdata;

	while ((i < gda_data_model_get_n_rows (gnome_db_grid_get_model (GNOME_DB_GRID (mpt->list)))) && !found) {
		rdata = gnome_db_grid_get_row_data (GNOME_DB_GRID (mpt->list), i);
		g_assert (rdata);
		if (rdata->table == table)
			found = TRUE;
		else
			i++;
	}
	if (found)
		return i;
	else
		return -1;
}



static void
selection_made (GnomeDbGrid * grid, gint row, gpointer data)
{
	MainPageTable *mpt = MAIN_PAGE_TABLE (data);

	mpt->sel_row = row;

	/* set sensitiveness of buttons */
	gtk_widget_set_sensitive (mpt->remove_table, TRUE);
	gtk_widget_set_sensitive (mpt->edit_table, TRUE);
	gtk_widget_set_sensitive (mpt->view_table, TRUE);
}

static void
selection_cleared (GnomeDbGrid * grid, gpointer data)
{
	MainPageTable *mpt = MAIN_PAGE_TABLE (data);

	mpt->sel_row = -1;
	gtk_widget_set_sensitive (mpt->remove_table, FALSE);
	gtk_widget_set_sensitive (mpt->edit_table, FALSE);
	gtk_widget_set_sensitive (mpt->view_table, FALSE);
}

static void
grid_double_clicked_cb (GnomeDbGrid * grid, gint row, MainPageTable * mpt)
{
	if (GNOME_DB_IS_GRID (grid)) {
		Row_Data *rd;

		rd = (Row_Data *) gnome_db_grid_get_row_data (grid, row);
		view_records_button_cb (NULL, mpt);
	}
}

static void
grid_create_popup_cb (GnomeDbGrid * grid, GtkWidget * menu, MainPageTable * mpt)
{
	GtkWidget *wid;

	wid = gtk_separator_menu_item_new ();
	gtk_menu_append (GTK_MENU (menu), wid);
	gtk_widget_show (wid);
	
	wid = gtk_image_menu_item_new_from_stock (GTK_STOCK_PROPERTIES, NULL);
	g_signal_connect (G_OBJECT (wid), "activate",
			  G_CALLBACK (edit_table_cb),
			  mpt);
	gtk_menu_append (GTK_MENU (menu), wid);
	gtk_widget_show (wid);
	if (mpt->sel_row == -1)
		gtk_widget_set_sensitive (wid, FALSE);

	wid = gtk_menu_item_new_with_label (_("View records"));
	g_signal_connect (G_OBJECT (wid), "activate",
			  G_CALLBACK (view_records_button_cb), mpt);
	gtk_menu_append (GTK_MENU (menu), wid);
	gtk_widget_show (wid);
	if (mpt->sel_row == -1)
		gtk_widget_set_sensitive (wid, FALSE);
}

static void remove_table_dlg_response_cb (GtkDialog *dlg, gint reply, MainPageTable * wid);
static void remove_table_dlg_destroy_cb (GtkDialog *dlg, MainPageTable * wid);
static void remove_table_dlg_conn_close_cb (GObject *srv, GtkDialog *dlg);
static void
remove_table_cb (GObject   * obj, MainPageTable * wid)
{
	Row_Data *rdata = NULL;
	gchar *str;
	GtkWidget *dlg;

	if (wid->sel_row >= 0)
		rdata = (Row_Data *)
			gnome_db_grid_get_row_data (GNOME_DB_GRID (wid->list), wid->sel_row);
	if (rdata) {
		if (rdata->table->is_view)
			str = g_strdup_printf (_("Do you really wish to drop the view '%s'?"),
					       rdata->table->name);
		else
			str = g_strdup_printf (_("Do you really wish to drop the table '%s'?"),
					       rdata->table->name);

		dlg = gtk_message_dialog_new (GTK_WINDOW (rdata->conf->app),
					      0,
					      GTK_MESSAGE_QUESTION,
					      GTK_BUTTONS_YES_NO,
					      str);

		g_signal_connect (G_OBJECT (wid->conf->srv), "conn_to_close",
				  G_CALLBACK (remove_table_dlg_conn_close_cb), dlg);

		g_signal_connect (G_OBJECT (dlg), "response",
				  G_CALLBACK (remove_table_dlg_response_cb), wid);
		g_signal_connect (G_OBJECT (dlg), "destroy",
				  G_CALLBACK (remove_table_dlg_destroy_cb), wid);

		g_free (str);
	}
}

static void
remove_table_dlg_response_cb (GtkDialog *dlg, gint reply, MainPageTable * wid)
{
	Row_Data *rdata = NULL;
	gchar *query;

	if (wid->sel_row >= 0)
		rdata = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (wid->list), wid->sel_row);

	switch (reply) {
	case GTK_RESPONSE_YES:
		if (rdata) {
			ServerResultset *rs;
			if (rdata->table->is_view)
				query = g_strdup_printf ("drop view %s\n",
							 rdata->table->name);
			else
				query = g_strdup_printf ("drop table %s\n",
							 rdata->table->name);
			rs = server_access_do_query (rdata->conf->srv, query, SERVER_ACCESS_QUERY_SQL);
			if (rs)
				g_object_unref (G_OBJECT (rs));
			g_free (query);
			database_refresh (rdata->conf->db, rdata->conf->srv);
		}
		break;
	case GTK_RESPONSE_NO:
		break;
	}
	gtk_widget_destroy (GTK_WIDGET (dlg));
}

static void
remove_table_dlg_destroy_cb (GtkDialog *dlg, MainPageTable * wid)
{
	/* disconnect callback */
	g_signal_handlers_disconnect_by_func (G_OBJECT (wid->conf->srv),
					      G_CALLBACK (remove_table_dlg_conn_close_cb), dlg);
}

static void
remove_table_dlg_conn_close_cb (GObject *srv, GtkDialog *dlg)
{
	gtk_widget_destroy (GTK_WIDGET (dlg));
}


static void
edit_table_cb (GObject   * obj, MainPageTable * wid)
{
	Row_Data *rdata = NULL;
	GtkWidget *dlg;

	if (wid->sel_row >= 0)
		rdata = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (wid->list), wid->sel_row);
	if (rdata) {
		dlg = table_edit_dialog_new (wid->conf, rdata->table);
		if (dlg)
			gtk_widget_show (dlg);
	}
}

static void
view_records_button_cb (GObject   * obj, MainPageTable * mpt)
{
	Row_Data *rdata = NULL;
	gchar *str;

	if (mpt->sel_row >= 0)
		rdata = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (mpt->list), mpt->sel_row);
	if (rdata) {
		QueryExec *qx;

		if (!rdata->exec_dlg) {
			GtkWidget *dlg;
			if (!rdata->query) {	/* build the query here */
				QueryField *qf;
				
				if (rdata->table->is_view)
					str = g_strdup_printf (_("contents of view %s"), rdata->table->name);
				else
					str = g_strdup_printf (_("contents of table %s"), rdata->table->name);
				rdata->query = QUERY (query_new (str, NULL, mpt->conf));
				g_free (str);
				
				/* QueryField */
				qf = query_field_allfields_dbtable_new (rdata->query, rdata->table->name,
									NULL, G_OBJECT (rdata->table));
				query_field_set_is_printed (qf, TRUE);
				query_add_field (QUERY (rdata->query), qf);
				
				rdata->env = QUERY_ENV (query_env_new (rdata->query));
				query_env_set_modif_table (rdata->env, rdata->table);
				rdata->env->actions =
					QUERY_ACTION_INSERT | QUERY_ACTION_REFRESH |
					QUERY_ACTION_DELETE |
					QUERY_ACTION_COMMIT | QUERY_ACTION_FIRST |
					QUERY_ACTION_LAST | QUERY_ACTION_PREV |
					QUERY_ACTION_NEXT;
			}
#ifdef debug
			query_dump_contents (rdata->query);
#endif
			qx = QUERY_EXEC (query_exec_new (rdata->env));
			dlg = query_exec_get_edit_form (qx, 0);
			if (dlg) {
				rdata->exec_dlg = dlg;
				gtk_widget_show (dlg);
				g_object_add_weak_pointer (G_OBJECT (dlg), (gpointer) &(rdata->exec_dlg));
			}
			else
				query_exec_free (QUERY_EXEC (qx));
		}
		else
			gdk_window_raise (rdata->props_dlg->window);
	}
}


static void drop_associated_query (Row_Data * rdata);
static void
main_page_table_conn_close_cb (GObject   * obj, MainPageTable * mpt)
{
	Row_Data *rdata;
	gint i, nb = gda_data_model_get_n_rows (gnome_db_grid_get_model (GNOME_DB_GRID (mpt->list)));

	/* Because when the connexion is closed, there should be no DbTable left, 
	   normally nb sould be equal to 0, but just in case */
	for (i = nb - 1; i >= 0; i++) {
		rdata = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (mpt->list), i);
		drop_associated_query (rdata);
		g_free (rdata);

		gda_data_model_remove_row (gnome_db_grid_get_model (GNOME_DB_GRID (mpt->list)), 
					   gda_data_model_get_row (gnome_db_grid_get_model (GNOME_DB_GRID (mpt->list)), i));
	}
	gtk_widget_set_sensitive (mpt->remove_table, FALSE);
	gtk_widget_set_sensitive (mpt->edit_table, FALSE);
	gtk_widget_set_sensitive (mpt->view_table, FALSE);
	gtk_widget_set_sensitive (mpt->new_table, FALSE);
	mpt->sel_row = -1;
}


/* this CB is intended to be connected to the "table_created" signal of the 
   Database objects it represents.  So:
   - obj is a Database object
*/
static void comments_table_changed_cb (DbTable * table, Row_Data * rdata);
static void
main_page_table_add_cb (GObject * obj, DbTable * table, MainPageTable * wid)
{
	GList *cols = NULL;
	Row_Data *rdata;

	cols = g_list_append (cols, gda_value_new_string (table->name));
	if (table->is_view)
		cols = g_list_append (cols, gda_value_new_string (_("View")));
	else
		cols = g_list_append (cols, gda_value_new_string (_("Table")));
	cols = g_list_append (cols, gda_value_new_string (table->owner));
	cols = g_list_append (cols, gda_value_new_string (table->comments));

	gda_data_model_append_row (gnome_db_grid_get_model (GNOME_DB_GRID (wid->list)), cols);
	g_list_foreach (cols, (GFunc) gda_value_free, NULL);
	g_list_free (cols);

	rdata = g_new0 (Row_Data, 1);
	rdata->conf = wid->conf;
	rdata->table = table;
	rdata->query = NULL;
	rdata->env = NULL;
	rdata->props_dlg = NULL;
	rdata->exec_dlg = NULL;
	gnome_db_grid_set_row_data (GNOME_DB_GRID (wid->list),
				    gda_data_model_get_n_rows (gnome_db_grid_get_model (GNOME_DB_GRID (wid->list))) - 1,
				    rdata);
	g_signal_connect (G_OBJECT (table), "comments_changed",
			  G_CALLBACK (comments_table_changed_cb),
			  rdata);

	g_print ("ADDED TABLE %s, rdata=%p\n", table->name, rdata);
}

static void
comments_table_changed_cb (DbTable * table, Row_Data * rdata)
{
	gint row;
	GtkWidget *list;

	list = MAIN_PAGE_TABLE (rdata->conf->tables_page)->list;
	row = gnome_db_grid_find_row_from_data (GNOME_DB_GRID (list), rdata);
	if (row >= 0) {
		gda_value_set_string (
			gda_data_model_get_value_at (gnome_db_grid_get_model (GNOME_DB_GRID (list)), 3, row),
			table->comments);
		gda_data_model_row_updated (gnome_db_grid_get_model (GNOME_DB_GRID (list)), row);
	}
}



/* this CB is intended to be connected to the "table_dropped" signal of the 
   Database objects it represents.  So:
   - obj is a Database object
*/

static void
main_page_table_drop_cb (GObject   * obj, DbTable * table, MainPageTable * wid)
{
	gint i = 0;
	gboolean found = FALSE;
	Row_Data *rdata;
	GdaDataModel *model;

	g_print ("REMOVED TABLE %s\n", table->name);

	model = gnome_db_grid_get_model (GNOME_DB_GRID (wid->list));
	while ((i < gda_data_model_get_n_rows (model)) && !found) {
		rdata = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (wid->list), i);
		g_assert (rdata);

		if (rdata->table == table) {
			found = TRUE;
			/* the associated query is not valid anymore */
			drop_associated_query (rdata);
			g_free (rdata);
			gda_data_model_remove_row (model,
						   gda_data_model_get_row (model, i));
			if (wid->sel_row == i) {
				wid->sel_row = -1;
				/* un sensitive buttons */
				gtk_widget_set_sensitive (wid->remove_table, FALSE);
				gtk_widget_set_sensitive (wid->edit_table, FALSE);
				gtk_widget_set_sensitive (wid->view_table, FALSE);
			}
		}
		i++;
	}

	if (!found)
		g_warning ("main_page_table_drop_cb: table '%s' not found!\n", table->name);
}
 
static void
drop_associated_query (Row_Data * rdata)
{
	g_return_if_fail (rdata);

	if (rdata->env) 
		/* don't free the env, that will be done by the query which holds
		   a ref on it. */
		rdata->env = NULL;

	if (rdata->query) {
		g_object_unref (G_OBJECT (rdata->query));
		rdata->query = NULL;
	}
}


static void
main_page_db_updated_cb (Database * db, MainPageTable * mpt)
{
	gint row;
	Row_Data *rd;
	GdaDataModel *model;

	model = gnome_db_grid_get_model (GNOME_DB_GRID (mpt->list));
	for (row = 0; row < gda_data_model_get_n_rows (model); row++) {
		rd = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (mpt->list), row);
		g_assert (rd);

		if (rd->table->is_view) {
			gda_value_set_string (
				gda_data_model_get_value_at (model, 1, row), _("View"));
		} 
		else {
			gda_value_set_string (gda_data_model_get_value_at (model, 1, row), _("Table"));
		}

		gda_value_set_string (gda_data_model_get_value_at (model, 2, row), rd->table->owner);
		gda_data_model_row_updated (model, row);

		/* the associated query is not valid anymore */
		drop_associated_query (rd);
	}
}
