/* field.c
 *
 * Copyright (C) 2001 - 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


/*
 * This module represents QueryField of type QUERY_FIELD_FIELD:
 * a field of a QueryView (DbTable (table or view) or Query).
 */

#include <config.h>
#include "../query.h"
#include "../database.h"
#include "../query-field-private.h"
#include "../object-selector.h"

static void        q_init            (QueryField *qf);
static void        q_finalize        (QueryField *qf);
static void        q_deactivate      (QueryField *qf);
static void        q_activate        (QueryField *qf);
static GtkWidget * q_get_edit_widget (QueryField *qf);
static GtkWidget * q_get_sel_widget  (QueryField *qf, QueryField *sel_qf, 
				      GCallback callback, gpointer data);
static gchar     * q_render_as_sql   (QueryField *qf, GSList * missing_values);
static xmlNodePtr  q_render_as_xml   (QueryField *qf, GSList * missing_values);
static gchar     * q_render_as_string(QueryField *qf, GSList * missing_values);
static void        q_save_to_xml     (QueryField *qf, xmlNodePtr node);
static void        q_load_from_xml   (QueryField *qf, xmlNodePtr node);
static void        q_copy_other_field(QueryField *qf, QueryField *other);
static gboolean    q_is_equal_to     (QueryField *qf, QueryField *other);
static GSList    * q_get_monitored_objects (QueryField *qf);
static void        q_replace_comp    (QueryField *qf, gint ref, GObject   *old, GObject   *new);

/* Weak ref from the refering object */
static void        q_obj_destroyed_cb (QueryField *qf, GObject *obj);

static void query_field_field_set_dbfield (QueryField *qf, QueryView *qv, DbField *field);
static void query_field_field_set_queryfield (QueryField *qf, QueryView *qv, QueryField *field);

typedef struct {
	/* The view and field attributs should either be both NULL or both non NULL */
	QueryView   *view;
	GObject     *field; /* can be either DbField if view is DbTable or QueryField if view is Query */
	gboolean     is_field_destroyed; /* TRUE if the weak ref on the field is no more */

	/* XML field name if we did not find it in the first place (like TVxxx:FIyyy) */
	gchar       *field_name;
} private_data;

#define QF_PRIVATE_DATA(qf) ((private_data *) qf->priv->private_data)


QueryFieldIface * 
query_field_field_get_iface()
{
	QueryFieldIface *iface;

	iface = g_new0 (QueryFieldIface, 1);
	iface->field_type = QUERY_FIELD_FIELD;
	iface->name = "field";
	iface->pretty_name = _("Field of a table or sub-query");
	iface->init = q_init;
	iface->destroy = q_finalize;
	iface->deactivate = q_deactivate;
	iface->activate = q_activate;
	iface->get_edit_widget = q_get_edit_widget;
	iface->get_sel_widget = q_get_sel_widget;
	iface->render_as_sql = q_render_as_sql;
	iface->render_as_xml = q_render_as_xml;
	iface->render_as_string = q_render_as_string;
	iface->save_to_xml = q_save_to_xml;
	iface->load_from_xml = q_load_from_xml;
	iface->copy_other_field = q_copy_other_field;
	iface->is_equal_to = q_is_equal_to;
	iface->get_monitored_objects = q_get_monitored_objects;
	iface->replace_comp = q_replace_comp;

	return iface;
}

static void        
q_init            (QueryField *qf)
{
	private_data *data;
	data = g_new0 (private_data, 1);
	qf->priv->private_data = (gpointer) data;
	QF_PRIVATE_DATA(qf)->view = NULL;
	QF_PRIVATE_DATA(qf)->field = NULL;
	QF_PRIVATE_DATA(qf)->field_name = NULL;
	QF_PRIVATE_DATA(qf)->is_field_destroyed = FALSE;
}

static void        
q_finalize         (QueryField *qf)
{
	query_field_deactivate (qf);
	if (qf->priv->private_data) {
		if (QF_PRIVATE_DATA(qf)->field_name)
			g_free (QF_PRIVATE_DATA(qf)->field_name);
		g_free (qf->priv->private_data);
		qf->priv->private_data = NULL;
	}
}

static void view_status_changed_cb (QueryView *qv, QueryField *qf);
static gchar * qf_get_xml_id (QueryField *qf);
static void
q_deactivate      (QueryField *qf)
{
	if (! qf->priv->activated)
		return;

	/* This function disconnects any event handler from any object
	   this QueryField wants to receive events from.
	   Here we disconnect from the view if we are connected */

	g_signal_handlers_disconnect_by_func (G_OBJECT (QF_PRIVATE_DATA(qf)->view),
					      G_CALLBACK (view_status_changed_cb), qf);

	if (! QF_PRIVATE_DATA(qf)->is_field_destroyed) {
		g_object_weak_unref (G_OBJECT (QF_PRIVATE_DATA(qf)->field),
				     (GWeakNotify) q_obj_destroyed_cb, qf);		
	}
	QF_PRIVATE_DATA(qf)->is_field_destroyed = FALSE;
	
	if (QF_PRIVATE_DATA(qf)->field_name) {
		g_free (QF_PRIVATE_DATA(qf)->field_name);
		QF_PRIVATE_DATA(qf)->field_name = NULL;
	}

	QF_PRIVATE_DATA(qf)->field_name = qf_get_xml_id (qf);

	QF_PRIVATE_DATA(qf)->view = NULL;
	QF_PRIVATE_DATA(qf)->field = NULL;

	query_field_set_activated (qf, FALSE);
}

static gchar *
qf_get_xml_id (QueryField *qf)
{
	gchar *str = NULL;

	if (!query_view_get_is_activated (QF_PRIVATE_DATA(qf)->view))
		return NULL;

	if (QF_PRIVATE_DATA(qf)->field) {
		if (IS_QUERY_FIELD (QF_PRIVATE_DATA(qf)->field))
			str = g_strdup_printf ("%s/%s", 
					       query_view_get_xml_id (QF_PRIVATE_DATA(qf)->view),
					       query_field_get_xml_id (QUERY_FIELD (QF_PRIVATE_DATA(qf)->field)));
		else {
			DbTable *table;

			table = database_find_table_from_field (qf->priv->query->conf->db, 
								DB_FIELD (QF_PRIVATE_DATA(qf)->field));
			if (table) /* table can be NULL if we are destroying it or the DbField */
				str = g_strdup_printf ("%s/%s", 
						       query_view_get_xml_id (QF_PRIVATE_DATA(qf)->view),
						       db_field_get_xml_id (DB_FIELD (QF_PRIVATE_DATA(qf)->field),
									    qf->priv->query->conf->db));
		}
	}

	return str;
}

static void
q_activate        (QueryField *qf)
{
	/* this function gets references to any object this QueryField wants to 
	   receive events from. */

	GObject   *field;
	QueryView *view;
	
	if (qf->priv->activated)
		return;

	field = QF_PRIVATE_DATA(qf)->field;
	view = QF_PRIVATE_DATA(qf)->view;
	
	if (!field && QF_PRIVATE_DATA(qf)->field_name) {
		gchar *str, *ptr;

		str = g_strdup (QF_PRIVATE_DATA(qf)->field_name);
		ptr = strtok (str, "/");
		ptr = strtok (NULL, "/");

		view = query_view_find_from_xml_name (qf->priv->query->conf, qf->priv->query, str);
		if (view) {
			DbField *dfield = NULL;
			QueryField *qfield = NULL;

			dfield = database_find_field_from_xml_name (qf->priv->query->conf->db, ptr);
			if (!dfield)
				qfield = query_get_field_by_xmlid (qf->priv->query, ptr);
										       
			if ((IS_QUERY (query_view_get_view_obj (view)) && qfield) ||
			    (IS_DB_TABLE (query_view_get_view_obj (view)) && dfield)) {
				g_free (QF_PRIVATE_DATA(qf)->field_name);
				QF_PRIVATE_DATA(qf)->field_name = NULL;	
				field = (qfield != NULL) ? G_OBJECT (qfield) : G_OBJECT (dfield);
			}
		}
		g_free (str);
	}


	if (field && view) {
		QF_PRIVATE_DATA(qf)->view = view;
		QF_PRIVATE_DATA(qf)->field = field;

		g_signal_connect (G_OBJECT (view), "status_changed",
				  G_CALLBACK (view_status_changed_cb), qf);

		g_object_weak_ref (G_OBJECT (QF_PRIVATE_DATA(qf)->field),
				   (GWeakNotify) q_obj_destroyed_cb, qf);

		/* now it's activated */ 
		query_field_set_activated (qf, TRUE);
	}
}

static void
view_status_changed_cb (QueryView *qv, QueryField *qf)
{
	if (query_view_get_activated (qv) == FALSE)
		q_deactivate (qf);
}

static void        
q_obj_destroyed_cb (QueryField *qf, GObject *obj)
{
	if (G_OBJECT (QF_PRIVATE_DATA(qf)->field) == obj)
		QF_PRIVATE_DATA(qf)->is_field_destroyed = TRUE;

	q_deactivate (qf);
	query_field_nullify (qf);
}


static void widget_field_sel_cb (ObjectSelector *os, GObject *sel, QueryField *qf);
static GtkWidget * 
q_get_edit_widget (QueryField *qf)
{
	GtkWidget *vbox, *os;

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 0);

	os = object_selector_new (qf->priv->query->conf, 
				  OBJECT_SELECTOR_QVIEWS | OBJECT_SELECTOR_QVIEW_FS,
				  OBJECT_SELECTOR_PARAM_TOP_QUERY, qf->priv->query,
				  OBJECT_SELECTOR_PARAM_SUB_QUERIES, FALSE, -1);
	object_selector_set_column_label (OBJECT_SELECTOR (os), 0, _("Source"));
	object_selector_set_comments_visible (OBJECT_SELECTOR (os), FALSE);

	if (QF_PRIVATE_DATA(qf)->field)
		object_selector_set_selection_parent (OBJECT_SELECTOR (os), 
						      G_OBJECT (QF_PRIVATE_DATA(qf)->field),
						      G_OBJECT (QF_PRIVATE_DATA(qf)->view));
	gtk_box_pack_start (GTK_BOX (vbox), os, TRUE, TRUE, 0);
	gtk_widget_show (os);

	g_signal_connect (G_OBJECT (os), "selection_changed",
			  G_CALLBACK (widget_field_sel_cb), qf);
	
	return vbox;
}



/* we consider the selected field is the the one being
   represented in the QueryField */
static void
widget_field_sel_cb (ObjectSelector *os, GObject *sel, QueryField *qf)
{
	GObject *parent;

	/* test if anything has changed */
	if (QF_PRIVATE_DATA(qf)->field == sel) 
		return;

	query_field_deactivate (qf);

	if (sel) {
		parent = object_selector_get_selected_object_parent (os);
		if (parent && IS_QUERY_VIEW (parent)) {
			if (IS_DB_FIELD (sel)) 
				query_field_field_set_dbfield (qf, QUERY_VIEW (parent), DB_FIELD (sel));
			if (IS_QUERY_FIELD (sel))
				query_field_field_set_queryfield (qf, QUERY_VIEW (parent), QUERY_FIELD (sel));
			query_field_activate (qf);
		}
	}

#ifdef debug_signal
	g_print (">> 'FIELD_MODIFIED' from widget_field_sel_cb\n");
#endif
	g_signal_emit_by_name (G_OBJECT (qf), "field_modified");
#ifdef debug_signal
	g_print ("<< 'FIELD_MODIFIED' from widget_field_sel_cb\n");
#endif	
}

static GtkWidget * 
q_get_sel_widget  (QueryField *qf, QueryField *sel_qf, GCallback callback, gpointer data)
{
	GtkWidget *button;
	gchar *str;

	if (qf->priv->activated) {
		const gchar *str1;
		gchar *str2;
		str1 = query_view_get_name (QF_PRIVATE_DATA(qf)->view);
		str2 = query_view_get_field_name (QF_PRIVATE_DATA(qf)->view, QF_PRIVATE_DATA(qf)->field);
		str = g_strdup_printf ("%s.%s", str1, str2);
		g_free (str2);
	}
	else
		str = g_strdup (_("Table or sub-query's FIELD"));

	button = gtk_button_new_with_label (str);	
	g_free (str);
	g_signal_connect (G_OBJECT (button), "clicked", callback, data);
	g_object_set_data (G_OBJECT (button), "qf", qf);

	/* Set the "QF_obj_emit_sig" attribute so that we can attach attributes to that button
	   which will be transmitted when the user clicks on it */
	g_object_set_data (G_OBJECT (button), "QF_obj_emit_sig", button);

	/* state */
	if (sel_qf == qf) 
		gtk_widget_set_sensitive (button, FALSE);

	return button;
}


static gchar * 
q_render_as_sql   (QueryField *qf, GSList * missing_values)
{
	gchar *str = NULL;

	if (qf->priv->activated) {
		const gchar *str1;
		gchar *str2;
		str1 = query_view_get_name (QF_PRIVATE_DATA(qf)->view);
		/* FIXME: find a more clean rule for the QueryField of the sub-query */
		str2 = query_view_get_field_name (QF_PRIVATE_DATA(qf)->view, QF_PRIVATE_DATA(qf)->field);
		str = g_strdup_printf ("%s.%s", str1, str2);
		g_free (str2);
	}

	return str;
}

static xmlNodePtr  
q_render_as_xml   (QueryField *qf, GSList * missing_values)
{
	return NULL;
}

static gchar * 
q_render_as_string(QueryField *qf, GSList * missing_values)
{
	gchar *str = NULL;
	
	str = q_render_as_sql (qf, missing_values);
	if (!str) 
		str = g_strdup ("A table's field");

	return str;
}

static void  
q_save_to_xml     (QueryField *qf, xmlNodePtr node)
{
	if (qf->priv->activated) {
		gchar *str;
	
		/* node object ref */
		str = query_view_get_xml_id (QF_PRIVATE_DATA(qf)->view);
		xmlSetProp (node, "object", str);
		g_free (str);

		if (IS_QUERY_FIELD (QF_PRIVATE_DATA(qf)->field))
			str = query_field_get_xml_id (QUERY_FIELD (QF_PRIVATE_DATA(qf)->field));
		else
			str = db_field_get_xml_id (DB_FIELD (QF_PRIVATE_DATA(qf)->field),
						   qf->priv->query->conf->db);
		xmlSetProp (node, "object_ext", str);
		g_free (str);
		
		xmlSetProp (node, "type", "field");
	}
	else
		g_warning ("QueryField not activated; can't save\n");
}

static void        
q_load_from_xml   (QueryField *qf, xmlNodePtr node)
{
	query_field_deactivate (qf);
	
	/* check we have a QueryField */
	if (!strcmp (node->name, "QueryField")) {
		gchar *str, *str2;

		str = xmlGetProp (node, "type");
		if (!str || (str && strcmp (str, "field"))) {
			if (str) g_free (str);
			return;
		}

		str = xmlGetProp (node, "object");
		str2 = xmlGetProp (node, "object_ext");
		if (str2) {
			if (QF_PRIVATE_DATA(qf)->field_name)
				g_free (QF_PRIVATE_DATA(qf)->field_name);
			QF_PRIVATE_DATA(qf)->field_name = g_strdup_printf ("%s/%s", str, str2);;
			g_free (str2);
		}
		g_free (str);
		query_field_activate (qf);
	}
}

static void        
q_copy_other_field(QueryField *qf, QueryField *other)
{
	/* we can't call q_destroy(qf) because we don't know what the type
	   of QueryField it was before. This is normally done by the
	   QueryField object before the copy */

	if (QF_PRIVATE_DATA(other)->field) {
		QF_PRIVATE_DATA(qf)->view = QF_PRIVATE_DATA(other)->view;
		QF_PRIVATE_DATA(qf)->field = QF_PRIVATE_DATA(other)->field;
		query_field_activate (qf);
	}
	else {
		if (QF_PRIVATE_DATA(other)->field_name) {
			QF_PRIVATE_DATA(qf)->field_name = g_strdup (QF_PRIVATE_DATA(other)->field_name);
			query_field_activate (qf);
		}
	}
}

static gboolean
q_is_equal_to (QueryField *qf, QueryField *other)
{
	gboolean retval = FALSE;

	if (qf->priv->activated && other->priv->activated) {
		if ((QF_PRIVATE_DATA(qf)->view == QF_PRIVATE_DATA(other)->view) &&
		    (QF_PRIVATE_DATA(qf)->field == QF_PRIVATE_DATA(other)->field))
			retval = TRUE;
	}

	return retval;
}


static GSList *
q_get_monitored_objects (QueryField *qf)
{
	GSList *list = NULL;

	if (qf->priv->activated) {
		list = g_slist_prepend (NULL, QF_PRIVATE_DATA(qf)->field);
		list = g_slist_prepend (list, QF_PRIVATE_DATA(qf)->view);
	}

	return list;
}

static void
q_replace_comp (QueryField *qf, gint ref, GObject   *old, GObject   *new)
{
	/* References can be to QueryViews can change. The obj attribute of the
	 new and the old QueryViews can either:
	 1) be the same and the the field WILL NOT change
	 2) be different and in this case things get complicated...
	*/
	if ((G_OBJECT (QF_PRIVATE_DATA(qf)->view) == old) && 
	    (IS_QUERY_VIEW (old) && IS_QUERY_VIEW (new))) {
		if (query_view_get_view_obj (QUERY_VIEW (old)) == 
		    query_view_get_view_obj (QUERY_VIEW (new))) { /* case 1 */
			/* we can keep it activated because the field does not change */
			g_signal_handlers_disconnect_by_func (G_OBJECT (QF_PRIVATE_DATA(qf)->view),
							      G_CALLBACK (view_status_changed_cb), qf);
			QF_PRIVATE_DATA(qf)->view = QUERY_VIEW (new);
			g_signal_connect (G_OBJECT (QF_PRIVATE_DATA(qf)->view), "status_changed",
					  G_CALLBACK (view_status_changed_cb), qf);
		}
		else { /* case 2 */
			gboolean done = FALSE;
			if (qf->priv->activated) {
				gint i;
				GSList *list;
				GObject   *field;
				
				if (IS_DB_TABLE (query_view_get_view_obj (QUERY_VIEW (old))))
					list = DB_TABLE (query_view_get_view_obj (QUERY_VIEW (old)))->fields;
				else
					list = QUERY (query_view_get_view_obj (QUERY_VIEW (old)))->fields;
				
				i = g_slist_index (list, QF_PRIVATE_DATA(qf)->field);

				if (IS_DB_TABLE (query_view_get_view_obj (QUERY_VIEW (new))))
					list = DB_TABLE (query_view_get_view_obj (QUERY_VIEW (new)))->fields;
				else
					list = QUERY (query_view_get_view_obj (QUERY_VIEW (new)))->fields;
				field = g_slist_nth_data (list, i);

				query_field_deactivate (qf);
				if (field && 
				    ((IS_QUERY_FIELD (field) && IS_QUERY_FIELD (QF_PRIVATE_DATA(qf)->field)) &&
				     (IS_DB_FIELD (field) && IS_DB_FIELD (QF_PRIVATE_DATA(qf)->field)))) {
					gchar *str1, *str2, *str3;
					str1 = query_view_get_xml_id (QUERY_VIEW (new));
				
					if (IS_QUERY_FIELD (field))
						str2 = query_field_get_xml_id (QUERY_FIELD (field));
					else
						str2 = db_field_get_xml_id (DB_FIELD (field), qf->priv->query->conf->db);
					str3 = g_strdup_printf ("%s/%s", str1, str2);
					g_free (str1);
					g_free (str2);
					g_print ("===> %s\n", str3);
					if (QF_PRIVATE_DATA(qf)->field_name)
						g_free (QF_PRIVATE_DATA(qf)->field_name);
					QF_PRIVATE_DATA(qf)->field_name = str3;

					query_field_activate (qf);
					done = TRUE;
				}
			}

			if (!done) 
				g_warning ("Cound not replace %p by %p in Field QueryField %p (named %s/%s)\n",
					   old, new, qf, qf->priv->name, qf->priv->alias);
		}
	}
	return;
}




/* 
 * 
 * Real QueryField object's different methods
 * 
 *
 */


QueryField *
query_field_field_dbfield_new (Query *q, gchar * name, QueryView *qv, DbField *field)
{
	QueryField *qf;
	QueryView *realqv;

	g_return_val_if_fail (q && IS_QUERY (q), NULL);
	g_return_val_if_fail (field && IS_DB_FIELD (field), NULL);
	if (qv) {
		g_return_val_if_fail (IS_QUERY_VIEW (qv) && (query_view_get_query (qv) == q), NULL);
		realqv = qv;
	}
	else { 
		/* try to find a suitable QV... */
		QueryView *qv2 = NULL;
		GSList *list;
		DbTable *table;

		table = database_find_table_from_field (q->conf->db, field);

		list = q->views;
		while (list && !qv2) {
			if (IS_DB_TABLE (query_view_get_view_obj (QUERY_VIEW (list->data))) &&
			    (DB_TABLE (query_view_get_view_obj (QUERY_VIEW (list->data))) == table))
				qv2 = QUERY_VIEW (list->data);
			list = g_slist_next (list);
		}

		/* ...and otherwise make one */
		if (!qv2)
			realqv = QUERY_VIEW (query_add_view_with_obj (q, G_OBJECT (table)));
		else
			realqv = qv2;
	}

	qf = QUERY_FIELD (query_field_new (q, name, QUERY_FIELD_FIELD));
	query_field_field_set_dbfield (qf, realqv, field);
	query_field_activate (qf);

	return qf;
}

QueryField *
query_field_field_queryfield_new (Query *q, gchar * name, QueryView *qv, QueryField *field)
{
	QueryField *qf;
	QueryView *realqv;

	g_return_val_if_fail (q && IS_QUERY (q), NULL);
	g_return_val_if_fail (field && IS_QUERY_FIELD (field), NULL);
	if (qv) {
		g_return_val_if_fail (IS_QUERY_VIEW (qv) && (query_view_get_query (qv) == q), NULL);
		realqv = qv;
	}
	else { 
		/* try to find a suitable QV... */
		QueryView *qv2 = NULL;
		GSList *list;
		Query *query;
		
		query = field->priv->query;

		list = q->views;
		while (list && !qv2) {
			if (IS_QUERY (query_view_get_view_obj (QUERY_VIEW (list->data))) &&
			    (QUERY (query_view_get_view_obj (QUERY_VIEW (list->data))) == query))
				qv2 = QUERY_VIEW (list->data);
			list = g_slist_next (list);
		}

		/* ...and otherwise make one */
		if (!qv2)
			realqv = QUERY_VIEW (query_add_view_with_obj (q, G_OBJECT (query)));
		else
			realqv = qv2;
	}

	qf = QUERY_FIELD (query_field_new (q, name, QUERY_FIELD_FIELD));
	query_field_field_set_queryfield (qf, realqv, field);
	query_field_activate (qf);

	return qf;
}


ServerDataType  *
query_field_field_get_data_type (QueryField *qf)
{
	g_assert (qf);
	g_assert (IS_QUERY_FIELD (qf));
	g_assert (qf->priv->field_type == QUERY_FIELD_FIELD);

	if (! qf->priv->activated)
		return NULL;
	else {
		if (IS_DB_FIELD (QF_PRIVATE_DATA(qf)->field))
			return DB_FIELD (QF_PRIVATE_DATA(qf)->field)->type;
		else
			return NULL; /* FIXME */
	}
}

static void
query_field_field_set_dbfield (QueryField *qf, QueryView *qv, DbField *field)
{
	gboolean done = FALSE;
	
	g_return_if_fail (qf && IS_QUERY_FIELD (qf));
	g_return_if_fail (qf->priv->field_type == QUERY_FIELD_FIELD);
	if (qv) 
		g_return_if_fail (IS_QUERY_VIEW (qv));

	if (qf->priv->activated)
		query_field_deactivate (qf);

	/* Test to find a QueryView for that field, otherwise do nothing */
	if (field) {
		if (qv && IS_DB_TABLE (query_view_get_view_obj (qv)) && 
		    g_slist_find (DB_TABLE (query_view_get_view_obj (qv))->fields, field)) {
			QF_PRIVATE_DATA(qf)->view = qv;
			QF_PRIVATE_DATA(qf)->field = G_OBJECT (field);
			qf->priv->query = query_view_get_query (qv);
			done = TRUE;
		}
		if (!done) {
			/* ignoring qv now, try to find a good QueryView */
			QueryView *qv2 = NULL;
			GSList *list;

			list = qf->priv->query->views;
			while (list && !qv2) {
				if (IS_DB_TABLE (query_view_get_view_obj (QUERY_VIEW (list->data))) &&
				    g_slist_find (DB_TABLE (query_view_get_view_obj (QUERY_VIEW (list->data)))->fields, field))
					qv2 = QUERY_VIEW (list->data);
				list = g_slist_next (list);
			}
			if (qv2) {
				QF_PRIVATE_DATA(qf)->view = qv2;
				QF_PRIVATE_DATA(qf)->field = G_OBJECT (field);
				qf->priv->query = query_view_get_query (qv2);
				done = TRUE;
			}
		}
	}

	if (done)
		query_field_activate (qf);
}

static void
query_field_field_set_queryfield (QueryField *qf, QueryView *qv, QueryField *field)
{
	gboolean done = FALSE;
	
	g_return_if_fail (qf && IS_QUERY_FIELD (qf));
	g_return_if_fail (qf->priv->field_type == QUERY_FIELD_FIELD);
	if (qv) 
		g_return_if_fail (IS_QUERY_VIEW (qv));

	if (qf->priv->activated)
		query_field_deactivate (qf);

	/* Test to find a QueryView for that field, otherwise do nothing */
	if (field) {
		if (qv && IS_QUERY (query_view_get_view_obj (qv)) && 
		    g_slist_find (QUERY (query_view_get_view_obj (qv))->fields, field)) {
			QF_PRIVATE_DATA(qf)->view = qv;
			QF_PRIVATE_DATA(qf)->field = G_OBJECT (field);
			qf->priv->query = query_view_get_query (qv);
			done = TRUE;
		}
		if (!done) {
			/* ignoring qv now, try to find a good QueryView */
			QueryView *qv2 = NULL;
			GSList *list;

			list = qf->priv->query->views;
			while (list && !qv2) {
				if (IS_QUERY (query_view_get_view_obj (QUERY_VIEW (list->data))) &&
				    g_slist_find (QUERY (query_view_get_view_obj (QUERY_VIEW (list->data)))->fields, field))
					qv2 = QUERY_VIEW (list->data);
				list = g_slist_next (list);
			}
			if (qv2) {
				QF_PRIVATE_DATA(qf)->view = qv2;
				QF_PRIVATE_DATA(qf)->field = G_OBJECT (field);
				qf->priv->query = query_view_get_query (qv2);
				done = TRUE;
			}
		}
	}

	if (done)
		query_field_activate (qf);
}

QueryView *
query_field_field_get_queryview    (QueryField *qf)
{
	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), NULL);
	g_return_val_if_fail (qf->priv->field_type == QUERY_FIELD_FIELD, NULL);

	if (qf->priv->activated && QF_PRIVATE_DATA(qf)->view)
		return QF_PRIVATE_DATA(qf)->view;
	else
		return NULL;
}

gboolean
query_field_field_is_db_field (QueryField *qf)
{
	gboolean retval = FALSE;

	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), FALSE);
	if (qf->priv->activated) {
		retval = IS_DB_FIELD (QF_PRIVATE_DATA (qf)->field) ? TRUE : FALSE;
	}

	return retval;
}

DbField *
query_field_field_get_db_field (QueryField *qf)
{
	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), NULL);
	g_return_val_if_fail (qf->priv->field_type == QUERY_FIELD_FIELD, NULL);

	if (qf->priv->activated && QF_PRIVATE_DATA(qf)->field && IS_DB_FIELD (QF_PRIVATE_DATA(qf)->field))
		return DB_FIELD (QF_PRIVATE_DATA(qf)->field);
	else
		return NULL;
}

QueryField *
query_field_field_get_query_field (QueryField *qf)
{
	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), NULL);
	g_return_val_if_fail (qf->priv->field_type == QUERY_FIELD_FIELD, NULL);

	if (qf->priv->activated && QF_PRIVATE_DATA(qf)->field && IS_QUERY_FIELD (QF_PRIVATE_DATA(qf)->field))
		return QUERY_FIELD (QF_PRIVATE_DATA(qf)->field);
	else
		return NULL;
}
