/* mainpagesql.c
 *
 * Copyright (C) 2002 Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <libgda/gda-data-model-array.h>
#include <gtk/gtkframe.h>
#include <gtk/gtktable.h>
#include <gtk/gtkvpaned.h>
#include <libgnomedb/gnome-db-grid.h>
#include <libgnomedb/gnome-db-editor.h>
#include <libgnomedb/gnome-db-util.h>
#include "mainpagesql.h"

static void main_page_sql_class_init (MainPageSqlClass *klass);
static void main_page_sql_init (MainPageSql *psql, MainPageSqlClass *klass);
static void main_page_sql_finalize (GObject *object);

static GObjectClass *parent_class = NULL;

static void
clear_sql_cb (GtkWidget *w, gpointer user_data)
{
	MainPageSql *psql = (MainPageSql *) user_data;

	g_return_if_fail (IS_MAIN_PAGE_SQL (psql));

	gnome_db_editor_set_text (GNOME_DB_EDITOR (psql->sql_editor), "", 0);
	gnome_db_grid_set_model (GNOME_DB_GRID (psql->grid), NULL);
	gnome_appbar_set_status (GNOME_APPBAR (psql->conf->appbar), _("Ready."));
}

static void
connection_closed_cb (GObject *obj, gpointer user_data)
{
	MainPageSql *psql = (MainPageSql *) user_data;

	g_return_if_fail (IS_MAIN_PAGE_SQL (psql));

	gnome_db_editor_set_text (GNOME_DB_EDITOR (psql->sql_editor), "", 0);
	gnome_db_grid_set_model (GNOME_DB_GRID (psql->grid), NULL);
}

static void
copy_sql_cb (GtkWidget *w, gpointer user_data)
{
	MainPageSql *psql = (MainPageSql *) user_data;

	g_return_if_fail (IS_MAIN_PAGE_SQL (psql));

	gnome_db_editor_copy_clipboard (GNOME_DB_EDITOR (psql->sql_editor));
	gnome_appbar_set_status (GNOME_APPBAR (psql->conf->appbar), _("Selection copied to clipboard"));
}

static void
cut_sql_cb (GtkWidget *w, gpointer user_data)
{
	MainPageSql *psql = (MainPageSql *) user_data;

	g_return_if_fail (IS_MAIN_PAGE_SQL (psql));

	gnome_db_editor_cut_clipboard (GNOME_DB_EDITOR (psql->sql_editor));
	gnome_appbar_set_status (GNOME_APPBAR (psql->conf->appbar), _("Selection moved to clipboard"));
}

static void
open_sql_cb (GtkWidget *w, gpointer user_data)
{
	gchar *filename;
	MainPageSql *psql = (MainPageSql *) user_data;

	g_return_if_fail (IS_MAIN_PAGE_SQL (psql));

	/* prompt the user for the file to open */
	filename = gnome_db_select_file_dialog (gtk_widget_get_toplevel (w),
						_("Open SQL file"));
	if (!filename)
		return;

	if (!gnome_db_editor_load_from_file (GNOME_DB_EDITOR (psql->sql_editor), filename))
		gnome_db_show_error (_("Could not open file %s for reading"), filename);
	else
		gnome_appbar_set_status (GNOME_APPBAR (psql->conf->appbar), _("File loaded"));

	g_free (filename);
}

static void
paste_sql_cb (GtkWidget *w, gpointer user_data)
{
	MainPageSql *psql = (MainPageSql *) user_data;

	g_return_if_fail (IS_MAIN_PAGE_SQL (psql));

	gnome_db_editor_paste_clipboard (GNOME_DB_EDITOR (psql->sql_editor));
	gnome_appbar_set_status (GNOME_APPBAR (psql->conf->appbar), _("Selection pasted from clipboard"));
}

static GdaCommand *
manage_placeholders (MainPageSql *psql, const gchar *sql)
{
	GdaCommand *cmd = NULL;
	gchar *s, *tmp;
	GtkWidget *dialog, *table = NULL, *label, *entry;
	GList *entry_list = NULL, *placeholder_positions = NULL, *placeholder_names = NULL, *l;
	GList *pl, *nl;
	gchar *head, *tail, *value;
	gint placeholder_count = 0, cnt, pos_span;
	GString *str = NULL;

	/* create the dialog */
	dialog = gtk_dialog_new_with_buttons (_("Enter values"),
					      GTK_WINDOW (gtk_widget_get_toplevel (psql->grid)), 0,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OK, GTK_RESPONSE_OK, NULL);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
	label = gnome_db_new_label_widget (_("There are some placeholders in the command you\n"
					     "are trying to execute. Please fill in the values\n"
					     " for each of them."));
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_CENTER);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, FALSE, FALSE, 2);

	for (s = sql, cnt = 0; s && *s; s++, cnt++) {
		if (*s == ':') {
			GString *fname = g_string_new ("");

			tmp = s + 1;
			while (tmp && *tmp && g_ascii_isalnum (*tmp)) {
				fname = g_string_append_c (fname, *tmp);	
				tmp++;
			}

			placeholder_names = g_list_append (placeholder_names, fname->str);
			placeholder_count++;

			/* create widgets for this placeholder */
			if (!table) {
				table = gnome_db_new_table_widget (1, 2, FALSE);
				gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), table, TRUE, TRUE, 2);
			} else {
				gtk_table_resize (GTK_TABLE (table), placeholder_count, 2);
			}

			label = gnome_db_new_label_widget (fname->str);
			gtk_table_attach (GTK_TABLE (table), label, 0, 1, placeholder_count - 1, placeholder_count,
					  GTK_FILL, GTK_FILL, 2, 2);
			entry = gnome_db_new_entry_widget (0, TRUE);
			gtk_table_attach (GTK_TABLE (table), entry, 1, 2, placeholder_count - 1, placeholder_count,
					  GTK_FILL, GTK_FILL, 2, 2);

			entry_list = g_list_append (entry_list, entry);
			placeholder_positions = g_list_append (placeholder_positions, GINT_TO_POINTER (cnt));

			s = tmp;
			cnt += fname->len + 1;
			g_string_free (fname, FALSE);
		}
	}

	if (placeholder_count > 0) {
		switch (gtk_dialog_run (GTK_DIALOG (dialog))) {
		case GTK_RESPONSE_OK :
			pos_span = 0;
			str = g_string_new (sql);
			for (l = entry_list; l != NULL; l = l->next) {
				pl = g_list_nth (placeholder_positions, g_list_position (entry_list, l));
				nl = g_list_nth (placeholder_names, g_list_position (entry_list, l));

				head = g_malloc0 (GPOINTER_TO_INT (pl->data) + pos_span + 1);
				strncpy (head, str->str, GPOINTER_TO_INT (pl->data) + pos_span);

				value = gtk_entry_get_text (GTK_ENTRY (l->data));

				if (sql[GPOINTER_TO_INT (pl->data) + strlen (nl->data) + 1] != '\0') {
					tail = g_malloc0 (strlen (sql) - GPOINTER_TO_INT (pl->data) - strlen (nl->data));
					strncpy (tail, sql + (GPOINTER_TO_INT (pl->data) + strlen (nl->data) + 1),
						 strlen (sql) - GPOINTER_TO_INT (pl->data) - strlen (nl->data) -1);
				} else
					tail = g_strdup ("");

				tmp = g_strdup_printf ("%s%s%s", head, value, tail);
				g_string_free (str, TRUE);
				str = g_string_new (tmp);

				g_free (head);
				g_free (tail);
				g_free (tmp);

				pos_span += strlen (value) - (strlen (nl->data) + 1);
			}

			cmd = gda_command_new (str->str, GDA_COMMAND_TYPE_SQL, 0);
			g_string_free (str, TRUE);
			break;
		case GTK_RESPONSE_CANCEL :
			cmd = NULL;
			break;
		}
	} else
		cmd = gda_command_new (sql, GDA_COMMAND_TYPE_SQL, 0);

	/* free memory */
	gtk_widget_destroy (dialog);
	g_list_free (entry_list);
	g_list_free (placeholder_positions);
	g_list_foreach (placeholder_names, (GFunc) g_free, NULL);
	g_list_free (placeholder_names);

	return cmd;
}

static void
run_sql_cb (GtkWidget *w, gpointer user_data)
{
	GdaConnection *cnc;
	GdaDataModel *model;
	GdaCommand *cmd;
	gchar *sql;
	gchar *msg;
	GList *value_list = NULL;
	GdaValue *tmval;
	GTimer *timer;
	MainPageSql *psql = (MainPageSql *) user_data;

	g_return_if_fail (IS_MAIN_PAGE_SQL (psql));

	cnc = psql->conf->srv->cnc;

	sql = gnome_db_editor_get_all_text (GNOME_DB_EDITOR (psql->sql_editor));
	if (!sql || !*sql)
		return;

	/* replace input values placeholders */
	if (!(cmd = manage_placeholders (psql, sql)))
		return;

	value_list = g_list_append (value_list, gda_value_new_string (gda_command_get_text (cmd)));

	/* execute command */
	timer = g_timer_new ();
	tmval = gda_value_new_timestamp_from_timet (time (NULL));
	g_timer_start (timer);

	model = gda_connection_execute_single_command (cnc, cmd, NULL);
	g_timer_stop (timer);
	if (model)
		value_list = g_list_append (value_list, gda_value_new_string (_("Query executed")));
	else
		value_list = g_list_append (value_list, gda_value_new_string (_("Error")));
	gnome_db_grid_set_model (GNOME_DB_GRID (psql->grid), model);

	value_list = g_list_append (value_list, tmval);
	value_list = g_list_append (value_list, gda_value_new_double (g_timer_elapsed (timer, NULL)));

	gda_data_model_append_row (gnome_db_grid_get_model (psql->log_list), value_list);
	g_list_foreach (value_list, (GFunc) gda_value_free, NULL);
	g_list_free (value_list);
	g_timer_destroy (timer);

	/* update UI */
	msg = g_strdup_printf (_("%d rows returned"), gda_data_model_get_n_rows (model));
	gnome_appbar_set_status (GNOME_APPBAR (psql->conf->appbar), msg);

	/* add command to history */
	psql->command_history = g_list_prepend (psql->command_history, sql);

	/* free memory */
	g_object_unref (G_OBJECT (model));
	gda_command_free (cmd);
	g_free (msg);
}

static void
save_sql_cb (GtkWidget *w, gpointer user_data)
{
	gchar *sql;
	gchar *filename;
	MainPageSql *psql = (MainPageSql *) user_data;

	g_return_if_fail (IS_MAIN_PAGE_SQL (psql));

	sql = gnome_db_editor_get_all_text (GNOME_DB_EDITOR (psql->sql_editor));
	if (!sql || !*sql)
		return;

	filename = gnome_db_select_file_dialog (gtk_widget_get_toplevel (w),
						_("Save as SQL file"));
	if (!filename)
		return;

	if (!gnome_db_editor_save_to_file (GNOME_DB_EDITOR (psql->sql_editor), filename))
		gnome_db_show_error (_("Could not save to file %s"), filename);
	else
		gnome_appbar_set_status (GNOME_APPBAR (psql->conf->appbar), _("File saved"));

	g_free (filename);
}

static void
sql_history_cb (GtkWidget *w, gpointer user_data)
{
	GtkWidget *dialog;
	GtkWidget *grid;
	GdaDataModel *model;
	GList *l;
	gint cnt;
	MainPageSql *psql = (MainPageSql *) user_data;

	g_return_if_fail (IS_MAIN_PAGE_SQL (psql));

	/* create the data model for the command history */
	if (!psql->command_history)
		return;

	model = gda_data_model_array_new (2);
	gda_data_model_set_column_title (model, 0, "#");
	gda_data_model_set_column_title (model, 1, _("SQL command"));

	cnt = 1;
	for (l = psql->command_history; l != NULL; l = l->next, cnt++) {
		GList *value_list = NULL;

		value_list = g_list_append (value_list, gda_value_new_integer (cnt));
		value_list = g_list_append (value_list, gda_value_new_string ((const gchar *) l->data));

		gda_data_model_append_row (model, value_list);

		g_list_foreach (value_list, (GFunc) gda_value_free, NULL);
		g_list_free (value_list);
	}

	/* create the dialog box */
	dialog = gtk_dialog_new_with_buttons (
		_("SQL command history"),
		GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (psql))), 0,
		GTK_STOCK_CLOSE, GTK_RESPONSE_CANCEL,
		GTK_STOCK_OK, GTK_RESPONSE_OK,
		NULL);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
	gtk_widget_set_usize (dialog, 350, 250);

	grid = gnome_db_grid_new ();
	gtk_widget_show (grid);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), grid, TRUE, TRUE, 0);
	gnome_db_grid_set_model (GNOME_DB_GRID (grid), model);

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
		GList *selection = gnome_db_grid_get_selection (GNOME_DB_GRID (grid));

		if (selection != NULL) {
			gchar *sql;

			sql = gda_value_stringify (gda_data_model_get_value_at (
							   model, 1, GPOINTER_TO_INT (selection->data)));
			gnome_db_editor_set_text (GNOME_DB_EDITOR (psql->sql_editor), sql, strlen (sql));
			g_free (sql);
		}
	}

	/* free memory */
	gtk_widget_destroy (dialog);
	g_object_unref (G_OBJECT (model));
}

static void
main_page_sql_class_init (MainPageSqlClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = main_page_sql_finalize;
}

static void
main_page_sql_init (MainPageSql *psql, MainPageSqlClass *klass)
{
	GtkWidget *table, *paned, *toolbar, *notebook;
	GdaDataModel *model;

	/* FIXME: load from configuration */
	psql->command_history = NULL;

	/* create main table */
	table = gtk_table_new (4, 2, FALSE);
	gtk_widget_show (table);
	gtk_box_pack_start (GTK_BOX (psql), table, TRUE, TRUE, GNOME_PAD / 2);

	paned = gtk_vpaned_new ();
	gtk_widget_show (paned);
	gtk_table_attach (GTK_TABLE (table), paned, 0, 2, 1, 4,
			  GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			  GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			  1, 1);

	/* create toolbar */
	toolbar = gnome_db_new_toolbar_widget (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
	gtk_table_attach (GTK_TABLE (table), toolbar, 0, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 1, 1);

	gtk_toolbar_insert_stock (GTK_TOOLBAR (toolbar), GTK_STOCK_EXECUTE,
				  _("Execute current command"), _("Execute current command"),
				  GTK_SIGNAL_FUNC (run_sql_cb), psql, -1);
	gtk_toolbar_insert_stock (GTK_TOOLBAR (toolbar), GTK_STOCK_GO_UP,
				  _("Select SQL command from history"), _("Select SQL command from history"),
				  GTK_SIGNAL_FUNC (sql_history_cb), psql, -1);
	gtk_toolbar_insert_stock (GTK_TOOLBAR (toolbar), GTK_STOCK_CLEAR,
				  _("Clear current results"), _("Clear current results"),
				  GTK_SIGNAL_FUNC (clear_sql_cb), psql, -1);
	gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));
	gtk_toolbar_insert_stock (GTK_TOOLBAR (toolbar), GTK_STOCK_OPEN,
				  _("Open SQL file"), _("Open SQL file"),
				  GTK_SIGNAL_FUNC (open_sql_cb), psql, -1);
	gtk_toolbar_insert_stock (GTK_TOOLBAR (toolbar), GTK_STOCK_SAVE,
				  _("Save SQL to file"), _("Save SQL to file"),
				  GTK_SIGNAL_FUNC (save_sql_cb), psql, -1);
	gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));
	gtk_toolbar_insert_stock (GTK_TOOLBAR (toolbar), GTK_STOCK_CUT,
				  _("Cut selected text"), _("Cut selected text"),
				  GTK_SIGNAL_FUNC (cut_sql_cb), psql, -1);
	gtk_toolbar_insert_stock (GTK_TOOLBAR (toolbar), GTK_STOCK_COPY,
				  _("Copy selected text to clipboard"), _("Copy selected text to clipboard"),
				  GTK_SIGNAL_FUNC (copy_sql_cb), psql, -1);
	gtk_toolbar_insert_stock (GTK_TOOLBAR (toolbar), GTK_STOCK_PASTE,
				  _("Paste clipboard contents in current position"),
				  _("Paste clipboard contents in current position"),
				  GTK_SIGNAL_FUNC (paste_sql_cb), psql, -1);

	/* SQL editor */
	psql->sql_editor = gnome_db_editor_new ();
	gtk_widget_show (psql->sql_editor);
	gtk_paned_add1 (GTK_PANED (paned), psql->sql_editor);

	/* notebook to contain the grid and the other information types */
	notebook = gnome_db_new_notebook_widget ();
	gtk_paned_add2 (GTK_PANED (paned), notebook);

	/* grid to display results */
	psql->grid = gnome_db_grid_new ();
	gtk_widget_show (psql->grid);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), psql->grid, gtk_label_new (_("Results")));

	/* log window */
	model = gda_data_model_array_new (4);
	gda_data_model_set_column_title (model, 0, "SQL");
	gda_data_model_set_column_title (model, 1, _("Result"));
	gda_data_model_set_column_title (model, 2, _("Timestamp"));
	gda_data_model_set_column_title (model, 3, _("Duration"));

	psql->log_list = gnome_db_grid_new_with_model (model);
	gtk_widget_show (psql->log_list);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), psql->log_list, gtk_label_new (_("Logging")));

	g_object_unref (G_OBJECT (model));
}

static void
main_page_sql_finalize (GObject *object)
{
	MainPageSql *psql = (MainPageSql *) object;

	g_return_if_fail (IS_MAIN_PAGE_SQL (psql));

	if (psql->command_history != NULL) {
		g_list_foreach (psql->command_history, (GFunc) g_free, NULL);
		g_list_free (psql->command_history);
		psql->command_history = NULL;
	}

	parent_class->finalize (object);
}

GType
main_page_sql_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MainPageSqlClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) main_page_sql_class_init,
			NULL,
			NULL,
			sizeof (MainPageSql),
			0,
			(GInstanceInitFunc) main_page_sql_init
		};		

		type = g_type_register_static (GTK_TYPE_VBOX, "MainPageSql", &info, 0);
	}
	return type;
}

GtkWidget *
main_page_sql_new (ConfManager *conf)
{
	MainPageSql *psql;

	psql = g_object_new (MAIN_PAGE_SQL_TYPE, NULL);
	psql->conf = conf;

	g_signal_connect (G_OBJECT (conf->srv), "conn_closed",
			  G_CALLBACK (connection_closed_cb), psql);

	return GTK_WIDGET (psql);
}
