/* query-editor-expr.c
 *
 * Copyright (C) 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "query.h"
#include "query-field-private.h"
#include "query-editor-fields.h"
#include "marshal.h"

/*
 *  
 *
 * Implementation of the QueryEditorExpr Widget
 * 
 *
 */

enum {
	STATUS,
	LAST_EXPR_SIGNAL
};

static gint expr_signals[LAST_EXPR_SIGNAL] = { 0 };

struct _QueryEditorExprPrivate
{
	gboolean           qef_in_destroy;	
	gboolean           qee_in_dispose;
	gboolean           data_detached;

	/* information about the selected QF for edition */
	QueryField        *sel_qf;
	QueryField        *parent_sel_qf;
	gint               parent_sel_ref;

	/* independant list of QueryFields while the edition takes place */
	GSList            *fields;

	/* Widgets */
	GtkWidget         *frame;
	GtkWidget         *current_expr;
	GtkWidget         *expr_container;
	GtkWidget         *current_area;
	GtkWidget         *area_container;
	GtkWidget         *name_entry;
	GtkWidget         *printed_cb;
	GtkWidget         *print_name_label;
	GtkWidget         *print_name_entry;
	GtkWidget         *type_omenu;
};


static void query_editor_expr_class_init (QueryEditorExprClass * class);
static void query_editor_expr_init (QueryEditorExpr * qee);
static void query_editor_expr_initialize (QueryEditorExpr * qee);
static void query_editor_expr_dispose (GObject   *obj);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *qex_parent_class = NULL;

guint
query_editor_expr_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (QueryEditorExprClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) query_editor_expr_class_init,
			NULL,
			NULL,
			sizeof (QueryEditorExpr),
			0,
			(GInstanceInitFunc) query_editor_expr_init
		};		

		type = g_type_register_static (GTK_TYPE_VBOX, "QueryEditorExpr", &info, 0);
	}

	return type;
}

static void
query_editor_expr_class_init (QueryEditorExprClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	qex_parent_class = g_type_class_peek_parent (class);
	expr_signals[STATUS] =
		g_signal_new ("status",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryEditorExprClass, status),
			      NULL, NULL,
			      marshal_VOID__BOOLEAN, G_TYPE_NONE, 1,
			      G_TYPE_BOOLEAN);

	class->status = NULL;

	object_class->dispose = query_editor_expr_dispose;
}

static void
query_editor_expr_init (QueryEditorExpr * qee)
{
	qee->priv = g_new0 (QueryEditorExprPrivate, 1);

	qee->priv->qef_in_destroy = FALSE;
	qee->priv->qee_in_dispose = FALSE;

	qee->qef = NULL;
	qee->orig_qf = NULL;
	qee->top_qf = NULL;
	qee->priv->sel_qf = NULL;
	qee->priv->fields = NULL;
	qee->priv->data_detached = FALSE; /* => QueryFields managed by this widget */

	qee->priv->parent_sel_qf = NULL;
	qee->priv->parent_sel_ref = -1;

	qee->priv->frame = NULL;
	qee->priv->current_expr = NULL;
	qee->priv->expr_container = NULL;
	qee->priv->current_area = NULL;
	qee->priv->area_container = NULL;
	qee->priv->name_entry = NULL;
	qee->priv->printed_cb = NULL;
	qee->priv->print_name_label = NULL;
	qee->priv->print_name_entry = NULL;
	qee->priv->type_omenu = NULL;
}

static void qee_refresh (QueryEditorExpr * qee);
static void set_top_qf (QueryEditorExpr *qee, QueryField *top_qf);
static void qef_destroy_cb (QueryEditorExpr *qee, QueryEditorFields *qef); /* GWeakNotify */
static void expr_query_field_dropped_cb (Query *q, QueryField *old_field, QueryEditorExpr *qee);
GtkWidget *
query_editor_expr_new (QueryEditorFields * qef, QueryField *qf)
{
	GObject   *obj;
	QueryEditorExpr *qee;

	g_return_val_if_fail (qef, NULL);
	g_return_val_if_fail (IS_QUERY_EDITOR_FIELDS (qef), NULL);
	if (qf)
		g_return_val_if_fail (IS_QUERY_FIELD (qf), NULL);

	obj = g_object_new (QUERY_EDITOR_EXPR_TYPE, NULL);
	qee = QUERY_EDITOR_EXPR (obj);
	qee->qef = qef;
	qee->orig_qf = qf;

	query_editor_expr_initialize (qee);

	if (qf) {
		/* make a copy of qf to be the QueryField we are working on 
		   => we now have a default reference on ALL the objects in the fields list */
		set_top_qf (qee, QUERY_FIELD (query_field_new_copy_all (qf, &(qee->priv->fields))));
		qee->priv->sel_qf = qee->top_qf;
	}

	/* initial refresh */
	qee_refresh (qee);

	/* signals */
	g_object_weak_ref (G_OBJECT (qef), (GWeakNotify) qef_destroy_cb, qee);

	g_signal_connect (G_OBJECT (query_editor_fields_get_query (qee->qef)), "field_dropped",
			  G_CALLBACK (expr_query_field_dropped_cb), qee);

	return GTK_WIDGET (obj);
}

static void top_qf_status_changed (QueryField *qf, QueryEditorExpr *qee);
static void 
set_top_qf (QueryEditorExpr *qee, QueryField *top_qf)
{
	if (qee->top_qf != top_qf) {
		if (qee->top_qf) 
			g_signal_handlers_disconnect_by_func (G_OBJECT (qee->top_qf),
							      G_CALLBACK (top_qf_status_changed), qee);

		qee->top_qf = top_qf;
		if (top_qf) 
			g_signal_connect (G_OBJECT (qee->top_qf), "status_changed",
					  G_CALLBACK (top_qf_status_changed), qee);

		qee_refresh (qee);
	}
}


static void 
top_qf_status_changed (QueryField *qf, QueryEditorExpr *qee)
{
#ifdef debug_signal
	g_print (">> 'STATUS' from top_qf_status_changed\n");
#endif
	g_signal_emit (G_OBJECT (qee), expr_signals[STATUS], 0, query_field_get_activated (qee->top_qf));
#ifdef debug_signal
	g_print ("<< 'STATUS' from top_qf_status_changed\n");
#endif
}

static void 
query_editor_expr_dispose (GObject   *object)
{
	QueryEditorExpr *qee;
	GSList *list;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_QUERY_EDITOR_EXPR (object));

	qee = QUERY_EDITOR_EXPR (object);

	if (qee->priv) {
		qee->priv->qee_in_dispose = TRUE;
		if (!qee->priv->data_detached) {
			if (qee->top_qf) 
				set_top_qf (qee, NULL); /* to disconnect signals properly */
			
			/* free the list of QueryFields */
			list = qee->priv->fields;
			while (list) {
				guint sig;
				
				/* disconnects the "field_modified" signal from the QueryFields */
				if ((sig = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (list->data), "sig")))) {
					g_signal_handler_disconnect (G_OBJECT (list->data), sig);
						g_object_set_data (G_OBJECT (list->data), "sig", NULL);
				}
				
				/* drop the QueryField's reference */
				g_object_unref (G_OBJECT (list->data));

				list = g_slist_next (list);
			}
			g_slist_free (qee->priv->fields);
			qee->priv->fields = NULL;
			
			qee->priv->parent_sel_qf = NULL;
			qee->priv->parent_sel_ref = -1;
		}
		
		/* signal handlers */
		g_signal_handlers_disconnect_by_func (G_OBJECT (query_editor_fields_get_query (qee->qef)),
						      G_CALLBACK (expr_query_field_dropped_cb), qee);
		
		if (! qee->priv->qef_in_destroy)
			g_object_weak_unref (G_OBJECT (qee->qef), (GWeakNotify) qef_destroy_cb, qee);

		g_free (qee->priv);
		qee->priv = NULL;
	}

	/* parent class */
	qex_parent_class->dispose (object);
}


static void 
qef_destroy_cb (QueryEditorExpr *qee, QueryEditorFields *qef)
{
	qee->priv->qef_in_destroy = TRUE;
	gtk_widget_destroy (GTK_WIDGET (qee));
}


static void 
expr_query_field_dropped_cb (Query *q, QueryField *old_field, QueryEditorExpr *qee)
{
	if (old_field == qee->orig_qf)
		qee->orig_qf = NULL;
}


#ifdef debug
static void dump_fields (GtkWidget *button, QueryEditorExpr * qee);
#endif
static void expr_name_changed_cb (GtkEntry *entry, QueryEditorExpr * qee); 
static void print_toggled_cb (GtkToggleButton *togglebutton, QueryEditorExpr * qee);
static void expr_alias_changed_cb (GtkEntry *entry, QueryEditorExpr * qee); 
static void
query_editor_expr_initialize (QueryEditorExpr * qee)
{
	GtkWidget *vb, *hb, *label, *frame, *table, *entry, *cb;
	GtkWidget *menu, *optionmenu;

#ifdef debug
	GtkWidget *button;
#endif
	
	/* Expression part */
	hb = gtk_hbox_new (FALSE, GNOME_PAD/2.);
	gtk_box_pack_start (GTK_BOX (qee), hb, FALSE, TRUE, 0);
	
	label = gtk_label_new (_("Complete expression:"));
	gtk_box_pack_start (GTK_BOX (hb), label, FALSE, TRUE, GNOME_PAD/2.);

	vb = gtk_vbox_new (FALSE, GNOME_PAD/2.);
	gtk_box_pack_start (GTK_BOX (hb), vb, FALSE, TRUE, GNOME_PAD/2.);
	qee->priv->expr_container = vb;

#ifdef debug
	button = gtk_button_new_with_label ("Dump");
	gtk_box_pack_start (GTK_BOX (hb), button, FALSE, TRUE, GNOME_PAD/2.);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (dump_fields), qee);
#endif

	/* Properties */
	frame = gtk_frame_new (_("Selected part's details"));
	gtk_box_pack_start (GTK_BOX (qee), frame, TRUE, TRUE, GNOME_PAD);
	qee->priv->frame = frame;

	vb = gtk_vbox_new (FALSE, GNOME_PAD/2.);
	gtk_container_add (GTK_CONTAINER (frame), vb);

	table = gtk_table_new (5, 2, FALSE);
	
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD/2.);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD/2.);
	gtk_box_pack_start (GTK_BOX (vb), table, TRUE, TRUE, 0);
	label = gtk_label_new (_("Type:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
			  GTK_FILL, 0, 0, 0);

	optionmenu = gtk_option_menu_new ();
	qee->priv->type_omenu = optionmenu;
	menu = gtk_menu_new ();

	gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu), menu);
	gtk_table_attach (GTK_TABLE (table), optionmenu, 1, 2, 0, 1,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0, 0);

	label = gtk_label_new (_("Name:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
			  GTK_FILL, 0, 0, 0);

	entry = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), entry, 1, 2, 1, 2,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0, 0);
	qee->priv->name_entry = entry;
	g_signal_connect (G_OBJECT (entry), "changed",
			  G_CALLBACK (expr_name_changed_cb), qee);

	label = gtk_label_new (_("Printed:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
			  GTK_FILL, 0, 0, 0);

	cb = gtk_check_button_new ();
	gtk_table_attach (GTK_TABLE (table), cb, 1, 2, 2, 3,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0, 0);
	qee->priv->printed_cb = cb;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (cb), TRUE);
	g_signal_connect (G_OBJECT (cb), "toggled",
			  G_CALLBACK (print_toggled_cb), qee);

	label = gtk_label_new (_("Print name:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 3, 4,
			  GTK_FILL, 0, 0, 0);
	qee->priv->print_name_label = label;

	entry = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), entry, 1, 2, 3, 4,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0, 0);
	qee->priv->print_name_entry = entry;
	g_signal_connect (G_OBJECT (entry), "changed",
			  G_CALLBACK (expr_alias_changed_cb), qee);


	label = gtk_label_new (_("Definition:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_misc_set_padding (GTK_MISC (label), 0, GNOME_PAD);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 4, 5,
			  GTK_FILL, GTK_FILL, 0, 0);

	qee->priv->area_container = gtk_vbox_new (FALSE, 0);
	gtk_table_attach (GTK_TABLE (table), qee->priv->area_container, 1, 2, 4, 5,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 
			  0, 0);

	/* show all the sub widgets of qee */
	gtk_widget_show_all (hb);
	gtk_widget_show_all (frame);

	gtk_widget_set_size_request (GTK_WIDGET (qee), 380, 400);
}

static void 
expr_name_changed_cb (GtkEntry *entry, QueryEditorExpr * qee)
{
	if (qee->priv->sel_qf)
		query_field_set_name (qee->priv->sel_qf, gtk_entry_get_text (entry));
}

static void 
expr_alias_changed_cb (GtkEntry *entry, QueryEditorExpr * qee)
{
	if (qee->priv->sel_qf)
		query_field_set_alias (qee->priv->sel_qf, gtk_entry_get_text (entry));
}

static void 
print_toggled_cb (GtkToggleButton *togglebutton, QueryEditorExpr * qee)
{
	gboolean active = gtk_toggle_button_get_active (togglebutton);

	if (qee->priv->sel_qf) 
		query_field_set_is_printed (qee->priv->sel_qf, active);
	gtk_widget_set_sensitive (qee->priv->print_name_label, active);
	gtk_widget_set_sensitive (qee->priv->print_name_entry, active);
}

static void qee_refresh_sel_area (QueryEditorExpr * qee);
static void expr_type_act_cb (GtkMenuItem *mitem, QueryEditorExpr *qee);
static void expr_alias_act_cb (GtkMenuItem *mitem, QueryEditorExpr *qee);
static void expr_selected_cb (GtkWidget *wid, QueryEditorExpr *qee);
static void expr_qf_changed_cb (QueryField *qf, QueryEditorExpr *qee);
static void 
qee_refresh (QueryEditorExpr * qee)
{
	GtkWidget *area;
	GtkWidget *menu, *smenu, *mitem, *smitem;
	GSList *list;
	QueryField *qf;
	gint i=1, pos=0;

	if (qee->priv->qee_in_dispose)
		return;

	/* Set the Type's option menu's possible values */
	gtk_option_menu_remove_menu (GTK_OPTION_MENU (qee->priv->type_omenu));
	menu = gtk_menu_new ();
	gtk_option_menu_set_menu (GTK_OPTION_MENU (qee->priv->type_omenu), menu);

	mitem = gtk_menu_item_new_with_label (_("Undefined!"));
	gtk_menu_append (GTK_MENU (menu), mitem);
	gtk_widget_set_sensitive (mitem , FALSE);
	g_object_set_data (G_OBJECT (mitem), "qftype", GINT_TO_POINTER (-1));

	list = QUERY_FIELD_CLASS(gtk_type_class (query_field_get_type ()))->field_types;
	while (list) {
		mitem = gtk_menu_item_new_with_label (_(((QueryFieldIface *)(list->data))->pretty_name));
		gtk_menu_append (GTK_MENU (menu), mitem);
		g_object_set_data (G_OBJECT (mitem), "qftype", 
				   GINT_TO_POINTER (((QueryFieldIface *)(list->data))->field_type));
		g_signal_connect (G_OBJECT (mitem), "activate",
				  G_CALLBACK (expr_type_act_cb), qee);
		if ((qee->priv->sel_qf) && 
		    ((QueryFieldIface *)(list->data))->field_type == query_field_get_ftype (qee->priv->sel_qf))
			pos = i;

		list = g_slist_next (list);
		i++;
	}
	gtk_option_menu_set_history (GTK_OPTION_MENU (qee->priv->type_omenu), pos);

	/* sub menu for the named objects */
	mitem = NULL;
	smenu = NULL;

	list = qee->priv->fields;
	while (list) { /* named objects already in the working list */
		qf = QUERY_FIELD (list->data);
		if ((qf != qee->priv->sel_qf) && query_field_get_name (qf)) {
			gchar *str;

			if (!mitem) {
				mitem = gtk_menu_item_new_with_label (_("From named expressions"));
				gtk_menu_append (GTK_MENU (menu), mitem);
				smenu = gtk_menu_new ();
				gtk_menu_item_set_submenu (GTK_MENU_ITEM (mitem), smenu);
			}
			str = g_strdup_printf ("%s (*)", query_field_get_name (qf));
			smitem = gtk_menu_item_new_with_label (str);
			g_free (str);

			g_object_set_data (G_OBJECT (smitem), "qfalias", qf);
			g_signal_connect (G_OBJECT (smitem), "activate",
					  G_CALLBACK (expr_alias_act_cb), qee); 
			gtk_menu_append (GTK_MENU (smenu), smitem);
		}
		list = g_slist_next (list);
	}

	list = query_editor_fields_get_query (qee->qef)->fields;
	while (list) { /* named objects still not copied into the working list */
		qf = QUERY_FIELD (list->data);
		if (query_field_get_name (qf)) {
			QueryField *otherqf;
			GSList *there;
			gboolean found = FALSE;
			
			there = qee->priv->fields;
			while (there && !found) { /* look for duplicates */
				otherqf = QUERY_FIELD (there->data);
				if (query_field_get_name (otherqf) && 
				    !strcmp (query_field_get_name (otherqf), 
					     query_field_get_name (qf)))
					found = TRUE;
				there = g_slist_next (there);
			}
			
			if (!found) {
				if (!mitem) {
					mitem = gtk_menu_item_new_with_label (_("From named expressions"));
					gtk_menu_append (GTK_MENU (menu), mitem);
					smenu = gtk_menu_new ();
					gtk_menu_item_set_submenu (GTK_MENU_ITEM (mitem), smenu);
				}
				smitem = gtk_menu_item_new_with_label (query_field_get_name (qf));
				g_object_set_data (G_OBJECT (smitem), "qfalias", qf);
				g_signal_connect (G_OBJECT (smitem), "activate",
						  G_CALLBACK (expr_alias_act_cb), qee); 
				gtk_menu_append (GTK_MENU (smenu), smitem);
			}
		}
		list = g_slist_next (list);
	}
	gtk_widget_show_all (menu);

	/* selection area */
	qee_refresh_sel_area (qee);

	/* fill in the entries: name, is_printed, alias */
	g_signal_handlers_block_by_func (G_OBJECT (qee->priv->name_entry),
					 G_CALLBACK (expr_name_changed_cb), qee);
	g_signal_handlers_block_by_func (G_OBJECT (qee->priv->print_name_entry),
					 G_CALLBACK (expr_alias_changed_cb), qee);
	if (qee->priv->sel_qf) {
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (qee->priv->printed_cb), 
					      query_field_get_is_printed (qee->priv->sel_qf));
		if (query_field_get_name (qee->priv->sel_qf))
			gtk_entry_set_text (GTK_ENTRY (qee->priv->name_entry), 
					    query_field_get_name (qee->priv->sel_qf));
		else
			gtk_entry_set_text (GTK_ENTRY (qee->priv->name_entry), "");
		if (query_field_get_alias (qee->priv->sel_qf))
			gtk_entry_set_text (GTK_ENTRY (qee->priv->print_name_entry), 
					    query_field_get_alias (qee->priv->sel_qf));
		else
			gtk_entry_set_text (GTK_ENTRY (qee->priv->print_name_entry), "");
	}
	else {
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (qee->priv->printed_cb), FALSE);
		gtk_entry_set_text (GTK_ENTRY (qee->priv->name_entry), "");
		gtk_entry_set_text (GTK_ENTRY (qee->priv->print_name_entry), "");
	}
	g_signal_handlers_unblock_by_func (G_OBJECT (qee->priv->name_entry),
					   G_CALLBACK (expr_name_changed_cb), qee);
	g_signal_handlers_unblock_by_func (G_OBJECT (qee->priv->print_name_entry),
					   G_CALLBACK (expr_alias_changed_cb), qee);


	/* display the QueryField's specific part */
	if (qee->priv->current_area) {
		gtk_widget_destroy (qee->priv->current_area);
		qee->priv->current_area = NULL;
	}
	if (qee->priv->sel_qf && qee->top_qf) {
		area = query_field_get_edit_widget (qee->priv->sel_qf);
	}
	else {
		area = gtk_label_new ("");
		gtk_widget_set_size_request (area, 200, 100);
	}
	gtk_box_pack_start (GTK_BOX (qee->priv->area_container), area, TRUE, TRUE, GNOME_PAD);	
	gtk_widget_show (area);
	qee->priv->current_area = area;

	/* connects the "field_modified" signal from the QueryFields in the list of managed fields */
	list = qee->priv->fields;
	while (list) {
		if (! GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (list->data), "sig"))) {
			guint sig;		
			sig = g_signal_connect (G_OBJECT (list->data), "field_modified",
						G_CALLBACK (expr_qf_changed_cb), qee);
			g_object_set_data (G_OBJECT (list->data), "sig", GUINT_TO_POINTER (sig));
		}

		list = g_slist_next (list);
	}
}

static void
qee_refresh_sel_area (QueryEditorExpr * qee)
{
	GtkWidget *expr;

	/* The expression display is created/replaced */
	if (qee->priv->current_expr) {
		gtk_widget_destroy (qee->priv->current_expr);
		qee->priv->current_expr = NULL;
	}

	expr = query_field_get_select_widget (qee->top_qf, qee->priv->sel_qf, 
					      G_CALLBACK (expr_selected_cb), qee);
	gtk_box_pack_start (GTK_BOX (qee->priv->expr_container), expr, FALSE, TRUE, GNOME_PAD/2.);
	qee->priv->current_expr = expr;
	gtk_widget_show (expr);
}


/* 
 * the user has selected a QueryField from the complete expression 
 */
static void 
expr_selected_cb (GtkWidget *wid, QueryEditorExpr *qee)
{
	QueryField *qf;
	gpointer ptr;

	/* QueryField to be edited */
	ptr = g_object_get_data (G_OBJECT (wid), "qf");
	if (!ptr) {
		/* the user can create a new QueryField */
		gtk_widget_set_sensitive (qee->priv->frame, TRUE);

		/* clean the display */
		qee->priv->sel_qf = NULL;
		qee_refresh (qee);
	}
	else {
		gtk_widget_set_sensitive (qee->priv->frame, TRUE);
		qf = QUERY_FIELD (ptr);
		if (qf != qee->priv->sel_qf) {
			qee->priv->sel_qf = qf;
			qee_refresh (qee);
		}
	}

	/* get some info on an upper level QueryField which needs to
	   be told if the selected QueryField is replaced by another */
	ptr = g_object_get_data (G_OBJECT (wid), "pqf");
	if (ptr) {
		gpointer ref;
		ref = g_object_get_data (G_OBJECT (wid), "ref");

		qee->priv->parent_sel_qf = QUERY_FIELD (ptr);
		qee->priv->parent_sel_ref = GPOINTER_TO_INT (ref);
	}
	else {
		qee->priv->parent_sel_qf = NULL;
		qee->priv->parent_sel_ref = -1;
	}
}

static void 
expr_qf_changed_cb (QueryField *qf, QueryEditorExpr *qee)
{
	qee_refresh_sel_area (qee);

	/* simple rules to set default names and print names */
	if (qee->priv->sel_qf) {
		const gchar *cstr = NULL;
		QueryFieldType ftype = query_field_get_ftype (qee->priv->sel_qf);
		QueryView *view;
		ServerFunction *function;

		switch (ftype) {
		case QUERY_FIELD_FIELD:
			if (query_field_field_is_db_field (qee->priv->sel_qf)) {
				DbField *field = query_field_field_get_db_field (qee->priv->sel_qf);
				if (field)
					cstr = field->name;
			}
			else {
				QueryField *field = query_field_field_get_query_field (qee->priv->sel_qf);
				if (field)
					cstr = query_field_get_name (field);
			}
			if (cstr) {
				gtk_entry_set_text (GTK_ENTRY (qee->priv->name_entry), cstr);
				gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (qee->priv->printed_cb), TRUE);
				gtk_entry_set_text (GTK_ENTRY (qee->priv->print_name_entry), cstr);
			}
			break;
		case QUERY_FIELD_ALLFIELDS:
			view = query_field_allfields_get_queryview (qee->priv->sel_qf);
			if (view) {
				cstr = query_view_get_name (view);
				
				gtk_entry_set_text (GTK_ENTRY (qee->priv->name_entry), cstr);
				gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (qee->priv->printed_cb), TRUE);
				gtk_entry_set_text (GTK_ENTRY (qee->priv->print_name_entry), "");
			}
			break;
		case QUERY_FIELD_VALUE:
			cstr = query_field_get_name (qee->priv->sel_qf);
			if (!cstr || (cstr && !*cstr)) {
				if (query_field_value_is_parameter (qee->priv->sel_qf))
					cstr = _("Parameter");
				else
					cstr = _("Value");
				gtk_entry_set_text (GTK_ENTRY (qee->priv->name_entry), cstr);
				gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (qee->priv->printed_cb), FALSE);
				gtk_entry_set_text (GTK_ENTRY (qee->priv->print_name_entry), "");
			}
			break;
		case QUERY_FIELD_FUNCTION:
			function = query_field_func_get_function (qee->priv->sel_qf);
			if (function) {
				cstr = function->sqlname;
				
				gtk_entry_set_text (GTK_ENTRY (qee->priv->name_entry), cstr);
				gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (qee->priv->printed_cb), TRUE);
				gtk_entry_set_text (GTK_ENTRY (qee->priv->print_name_entry), cstr);
			}
			break;
		default:
			break;
		}
	}
}

static void replace_query_field (QueryEditorExpr *qee, QueryField *new_field);

/* This function is called when a menu item from the menu of the different types of QF available
   for creation is activated */
static void 
expr_type_act_cb (GtkMenuItem *mitem, QueryEditorExpr *qee)
{
	gint type;
	QueryField *qf = NULL;
	QueryFieldType qftype;

	type = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (mitem), "qftype"));
	if (type == -1)
		qf = NULL;
	else {
		/* create a new QF */
		qftype = (QueryFieldType) type;
		qf = QUERY_FIELD (query_field_new (query_editor_fields_get_query (qee->qef), NULL, qftype));
		g_object_set_data (G_OBJECT (qf), "qf_list", &(qee->priv->fields));
		qee->priv->fields = g_slist_append (qee->priv->fields, qf);
	}
	
	replace_query_field (qee, qf);

	/* finaly refresh! */
	qee->priv->sel_qf = qf;
	qee_refresh (qee);
}

static void 
expr_alias_act_cb (GtkMenuItem *mitem, QueryEditorExpr *qee)
{
	gpointer ptr;
	
	ptr = g_object_get_data (G_OBJECT (mitem), "qfalias");
	if (ptr) {
		QueryField *alias;

		alias = QUERY_FIELD (ptr);
		
		if (! g_slist_find (qee->priv->fields, alias)) {
			QueryField *newalias;

			/* copy recursively that QueryField into qee->priv->fields 
			 => we have a default reference on the copied QueryField objects */
			newalias = QUERY_FIELD (query_field_new_copy_all (alias, &(qee->priv->fields)));

			/* if the original QF is alreday in the Query, set the "orig_qf" */
			if (g_slist_find (query_editor_fields_get_query (qee->qef)->fields, alias))
				g_object_set_data (G_OBJECT (newalias), "orig_qf", alias);

			alias = newalias;
		}

		replace_query_field (qee, alias);

		/* finaly refresh! */
		qee->priv->sel_qf = alias;
		qee_refresh (qee);
	}
}


/* 
 * replace the current selected QueryField (qee->priv->sel_qf, which can be NULL) 
 * with the new one given as argument
 */
static void
replace_query_field (QueryEditorExpr *qee, QueryField *new_field)
{
	/* if we are editing the main QueryField, then update qee->top_qf */
	if (qee->priv->sel_qf == qee->top_qf) {
		gboolean status = FALSE;

		if (qee->top_qf)
			status = query_field_get_activated (qee->top_qf);

		set_top_qf (qee, new_field);
		qee->priv->sel_qf = new_field;

		if (status != query_field_get_activated (qee->top_qf))
			top_qf_status_changed (qee->top_qf, qee);
	}

	/* now do the replacement in a parent QueryField if applicable */
	if (qee->priv->parent_sel_qf) 
		query_field_replace_ref_int (qee->priv->parent_sel_qf, qee->priv->parent_sel_ref, new_field);
}



/* At the end of this function, we consider that we don't actually
   hold any reference to any of the fields, but that they are transmitted
   to the object calling this function
*/
GSList *
query_editor_expr_fetch_fields (QueryEditorExpr * qee)
{
	GSList *list, *retval;
	guint sig;

	g_return_val_if_fail (qee && IS_QUERY_EDITOR_EXPR (qee), NULL);

#ifdef debug
	AAA (1000);
	g_print ("query_editor_expr_fetch_fields() fields are:\n");
	dump_fields (NULL, qee);
	AAA (1001);
#endif

	/* disconnects the "field_modified" signal from the QueryFields */
	list = qee->priv->fields;
	while (list) {
		if ((sig = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (list->data), "sig")))) {
			g_signal_handler_disconnect (G_OBJECT (list->data), sig);
			g_object_set_data (G_OBJECT (list->data), "sig", NULL);
		}

		list = g_slist_next (list);
	}

	/* DON'T FREE THE qee->priv->fields LIST HERE */
	retval = qee->priv->fields;
	qee->priv->fields = NULL;

	/* Selection and top_qf */
	set_top_qf (qee, NULL);
	qee->priv->parent_sel_ref = -1;
	qee->priv->parent_sel_qf = NULL;

	/* Widget status */
	qee->priv->data_detached = TRUE;

	return retval;
}








/* 
 * Debug functions 
 */
#ifdef debug
static gchar* field_type_to_str (QueryFieldType t) 
{
	gchar *str;

	switch (t) {
	case QUERY_FIELD_FIELD:
		str = "QUERY_FIELD_FIELD";
		break;
	case QUERY_FIELD_ALLFIELDS:
		str = "QUERY_FIELD_ALLFIELDS";
		break;
	case QUERY_FIELD_AGGREGATE:
		str = "QUERY_FIELD_AGGREGATE";
		break;
	case QUERY_FIELD_FUNCTION:
		str = "QUERY_FIELD_FUNCTION";
		break;
	case QUERY_FIELD_VALUE:
		str = "QUERY_FIELD_VALUE";
		break;
	case QUERY_FIELD_QUERY:
		str = "QUERY_FIELD_QUERY";
		break;
	case QUERY_FIELD_QUERY_FIELD:
		str = "QUERY_FIELD_QUERY_FIELD";
		break;
	default:
		str = "UNKNOWN TYPE";
	}
	return str;
}

static void 
dump_fields (GtkWidget *button, QueryEditorExpr * qee)
{
	GSList *list = qee->priv->fields;
	gchar *str="";
	GSList *monitored, *iter;

	g_print ("%s" D_COL_H2 "--------------------------------------\n" D_COL_NOR, str);

	g_print ("%s" D_COL_H2 "Top Query fields:%p\n" D_COL_NOR, str, qee->top_qf);
	g_print ("%s" D_COL_H2 "Sel Query fields:%p\n" D_COL_NOR, str, qee->priv->sel_qf);
	while (list) {
		QueryField *qf;
		qf = QUERY_FIELD (list->data);
		g_print ("%s" D_COL_H2 "Query field:%p\n" D_COL_NOR, str, qf);

		g_print ("%s  " D_COL_H2 "* field:" D_COL_NOR " %s\n", str, query_field_get_name (qf));
		g_print ("%s\tid: %d\n", str, query_field_get_id (qf));
		g_print ("%s\talias : %s\n", str, query_field_get_alias (qf));
		g_print ("%s\tfield type: %s\n", str, field_type_to_str (query_field_get_ftype (qf)));
		if (query_field_get_ftype (qf) == QUERY_FIELD_VALUE) {
			const GdaValue *val;
			g_print ("%s\tVALUE: ServerDataType GDA value = %d\n", str,
				 query_field_value_get_data_type (qf)->gda_type);
			val = query_field_value_get_value (qf);
			if (val)
				g_print ("%s\tVALUE: GdaValue = %p, type = %d\n", str, val,
					 gda_value_get_type (val));
			else
				g_print ("%s\tVALUE: GdaValue = %p\n", str, val);
		}
		monitored = query_field_get_monitored_objects (qf);
		iter = monitored;
		while (iter) {
			if (iter == monitored)
				g_print ("%s\tmonitored objects:", str);
			g_print (" %p", iter->data);
			iter = g_slist_next (iter);
		}
		if (monitored) {
			g_print ("\n");
			g_slist_free (monitored);
		}
		if (query_field_get_activated (qf)) {
			gchar *sql;
			
			g_print ("%s\t" D_COL_OK "activated\n" D_COL_NOR , str);
			sql = query_field_render_as_sql (qf, NULL);
			g_print ("%s\tSQL= %s\n", str, sql);
			g_free (sql);
		}
		else
			g_print("%s\t" D_COL_ERR "not activated\n" D_COL_NOR, str);
		list = g_slist_next (list);
	}
}
#endif
