/* mg-data-cell-renderer-boolean.c
 * Copyright (C) 2000  Red Hat, Inc.,  Jonathan Blandford <jrb@redhat.com>
 * Copyright (C) 2003  Vivien Malerba <malerba@gnome-db.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <libgda/libgda.h>
#include "mg-data-cell-renderer-boolean.h"
#include <libmergeant/marshal.h>
#include <libmergeant/mg-data-handler.h>


static void mg_data_cell_renderer_boolean_get_property  (GObject                    *object,
							 guint                       param_id,
							 GValue                     *value,
							 GParamSpec                 *pspec);
static void mg_data_cell_renderer_boolean_set_property  (GObject                    *object,
							 guint                       param_id,
							 const GValue               *value,
							 GParamSpec                 *pspec);
static void mg_data_cell_renderer_boolean_init       (MgDataCellRendererBoolean      *celltext);
static void mg_data_cell_renderer_boolean_class_init (MgDataCellRendererBooleanClass *class);
static void mg_data_cell_renderer_boolean_render     (GtkCellRenderer            *cell,
						      GdkWindow                  *window,
						      GtkWidget                  *widget,
						      GdkRectangle               *background_area,
						      GdkRectangle               *cell_area,
						      GdkRectangle               *expose_area,
						      GtkCellRendererState        flags);
static void mg_data_cell_renderer_boolean_get_size   (GtkCellRenderer            *cell,
                                                      GtkWidget                  *widget,
                                                      GdkRectangle               *cell_area,
                                                      gint                       *x_offset,
                                                      gint                       *y_offset,
                                                      gint                       *width,
                                                      gint                       *height);
static gboolean mg_data_cell_renderer_boolean_activate  (GtkCellRenderer            *cell,
							 GdkEvent                   *event,
							 GtkWidget                  *widget,
							 const gchar                *path,
							 GdkRectangle               *background_area,
							 GdkRectangle               *cell_area,
							 GtkCellRendererState        flags);

enum {
	CHANGED,
	LAST_SIGNAL
};


struct _MgDataCellRendererBooleanPrivate 
{
	MgDataHandler        *dh;
        GdaValueType          type;
        GdaValue             *value;
	gboolean              to_be_deleted;

	gboolean              editable;
	gboolean              active;
	gboolean              null;
};

enum {
	PROP_0,
	PROP_VALUE,
	PROP_VALUE_ATTRIBUTES,
	PROP_EDITABLE,
	PROP_TO_BE_DELETED
};

static guint toggle_cell_signals[LAST_SIGNAL] = { 0 };


GType
mg_data_cell_renderer_boolean_get_type (void)
{
	static GType cell_type = 0;

	if (!cell_type) {
		static const GTypeInfo cell_info = {
			sizeof (MgDataCellRendererBooleanClass),
			NULL,		/* base_init */
			NULL,		/* base_finalize */
			(GClassInitFunc) mg_data_cell_renderer_boolean_class_init,
			NULL,		/* class_finalize */
			NULL,		/* class_data */
			sizeof (MgDataCellRendererBoolean),
			0,              /* n_preallocs */
			(GInstanceInitFunc) mg_data_cell_renderer_boolean_init,
		};
		
		cell_type =
			g_type_register_static (GTK_TYPE_CELL_RENDERER_TOGGLE, "MgDataCellRendererBoolean",
						&cell_info, 0);
	}

	return cell_type;
}

static void
mg_data_cell_renderer_boolean_init (MgDataCellRendererBoolean *cell)
{
	cell->priv = g_new0 (MgDataCellRendererBooleanPrivate, 1);
	cell->priv->dh = NULL;
	cell->priv->type = GDA_VALUE_TYPE_BOOLEAN;
	cell->priv->editable = FALSE;
	GTK_CELL_RENDERER (cell)->mode = GTK_CELL_RENDERER_MODE_ACTIVATABLE;
	GTK_CELL_RENDERER (cell)->xpad = 2;
	GTK_CELL_RENDERER (cell)->ypad = 2;
}

static void
mg_data_cell_renderer_boolean_class_init (MgDataCellRendererBooleanClass *class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (class);
	GtkCellRendererClass *cell_class = GTK_CELL_RENDERER_CLASS (class);

	object_class->get_property = mg_data_cell_renderer_boolean_get_property;
	object_class->set_property = mg_data_cell_renderer_boolean_set_property;

	cell_class->get_size = mg_data_cell_renderer_boolean_get_size;
	cell_class->render = mg_data_cell_renderer_boolean_render;
	cell_class->activate = mg_data_cell_renderer_boolean_activate;
  
	g_object_class_install_property (object_class,
					 PROP_VALUE,
					 g_param_spec_pointer ("value",
                                                               _("Value"),
                                                               _("GdaValue to render"),
                                                               G_PARAM_READWRITE));
  
	g_object_class_install_property (object_class,
					 PROP_VALUE_ATTRIBUTES,
					 g_param_spec_uint ("value_attributes", NULL, NULL,
                                                            0, G_MAXUINT, 0, G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
					 PROP_EDITABLE,
					 g_param_spec_boolean ("editable",
							       _("Editable"),
							       _("The toggle button can be activated"),
							       TRUE,
							       G_PARAM_READABLE |
							       G_PARAM_WRITABLE));

	g_object_class_install_property (object_class,
					 PROP_TO_BE_DELETED,
					 g_param_spec_boolean ("to_be_deleted", NULL, NULL, FALSE,
                                                               G_PARAM_WRITABLE));

	toggle_cell_signals[CHANGED] =
		g_signal_new ("changed",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (MgDataCellRendererBooleanClass, changed),
			      NULL, NULL,
			      marshal_VOID__STRING_POINTER,
			      G_TYPE_NONE, 2,
			      G_TYPE_STRING,
			      G_TYPE_POINTER);
}

static void
mg_data_cell_renderer_boolean_get_property (GObject     *object,
					    guint        param_id,
					    GValue      *value,
					    GParamSpec  *pspec)
{
	MgDataCellRendererBoolean *cell = MG_DATA_CELL_RENDERER_BOOLEAN (object);
  
	switch (param_id) {
	case PROP_VALUE:
		g_value_set_pointer (value, cell->priv->value);
		break;
	case PROP_VALUE_ATTRIBUTES:
		break;
	case PROP_EDITABLE:
		g_value_set_boolean (value, cell->priv->editable);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}


static void
mg_data_cell_renderer_boolean_set_property (GObject      *object,
					    guint         param_id,
					    const GValue *value,
					    GParamSpec   *pspec)
{
	MgDataCellRendererBoolean *cell = MG_DATA_CELL_RENDERER_BOOLEAN (object);
  
	switch (param_id) {
	case PROP_VALUE:
		/* Because we don't have a copy of the value, we MUST NOT free it! */
                cell->priv->value = NULL;
		if (value) {			
                        GdaValue *gval = g_value_get_pointer (value);
			if (gval && !gda_value_is_null (gval)) {
				g_return_if_fail (gda_value_get_type (gval) == cell->priv->type);
				if (! gda_value_isa (gval, GDA_VALUE_TYPE_BOOLEAN))
					g_warning ("MgDataCellRendererBoolean can only handle boolean values");
				else 
					g_object_set (G_OBJECT (object),
						      "inconsistent", FALSE,
						      "active", gda_value_get_boolean (gval), NULL);
			}
			else
				g_object_set (G_OBJECT (object), 
					      "inconsistent", TRUE,
					      "active", FALSE, NULL);

                        cell->priv->value = gval;
                }
		else
			g_object_set (G_OBJECT (object), 
				      "inconsistent", TRUE,
				      "active", FALSE, NULL);

                g_object_notify (object, "value");
		break;
	case PROP_VALUE_ATTRIBUTES:
		break;
	case PROP_EDITABLE:
		cell->priv->editable = g_value_get_boolean (value);
		g_object_set (G_OBJECT (object), "activatable", cell->priv->editable, NULL);
		g_object_notify (G_OBJECT(object), "editable");
		break;
	case PROP_TO_BE_DELETED:
		cell->priv->to_be_deleted = g_value_get_boolean (value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

/**
 * mg_data_cell_renderer_boolean_new:
 * 
 * Creates a new #MgDataCellRendererBoolean. Adjust rendering
 * parameters using object properties. Object properties can be set
 * globally (with g_object_set()). Also, with #GtkTreeViewColumn, you
 * can bind a property to a value in a #GtkTreeModel. For example, you
 * can bind the "active" property on the cell renderer to a boolean value
 * in the model, thus causing the check button to reflect the state of
 * the model.
 * 
 * Return value: the new cell renderer
 **/
GtkCellRenderer *
mg_data_cell_renderer_boolean_new (MgDataHandler *dh, GdaValueType type)
{
	GObject *obj;
        MgDataCellRendererBoolean *cell;

        g_return_val_if_fail (dh && IS_MG_DATA_HANDLER (dh), NULL);
        obj = g_object_new (MG_DATA_CELL_RENDERER_BOOLEAN_TYPE, NULL);
	
        cell = MG_DATA_CELL_RENDERER_BOOLEAN (obj);
        cell->priv->dh = dh;
        g_object_ref (G_OBJECT (dh));
        cell->priv->type = type;
	
        return GTK_CELL_RENDERER (obj);
}

static void
mg_data_cell_renderer_boolean_get_size (GtkCellRenderer *cell,
					GtkWidget       *widget,
					GdkRectangle    *cell_area,
					gint            *x_offset,
					gint            *y_offset,
					gint            *width,
					gint            *height)
{
	GtkCellRendererClass *toggle_class = g_type_class_peek (GTK_TYPE_CELL_RENDERER_TOGGLE);

	(toggle_class->get_size) (cell, widget, cell_area, x_offset, y_offset, width, height);
}

static void
mg_data_cell_renderer_boolean_render (GtkCellRenderer      *cell,
				      GdkWindow            *window,
				      GtkWidget            *widget,
				      GdkRectangle         *background_area,
				      GdkRectangle         *cell_area,
				      GdkRectangle         *expose_area,
				      GtkCellRendererState  flags)
{
	GtkCellRendererClass *toggle_class = g_type_class_peek (GTK_TYPE_CELL_RENDERER_TOGGLE);

	(toggle_class->render) (cell, window, widget, background_area, cell_area, expose_area, flags);

	if (MG_DATA_CELL_RENDERER_BOOLEAN (cell)->priv->to_be_deleted)
		gtk_paint_hline (widget->style,
				 window, GTK_STATE_SELECTED,
				 cell_area, 
				 widget,
				 "hline",
				 cell_area->x + cell->xpad, cell_area->x + cell_area->width - cell->xpad,
				 cell_area->y + cell_area->height / 2.);

}

static gboolean
mg_data_cell_renderer_boolean_activate  (GtkCellRenderer            *cell,
					 GdkEvent                   *event,
					 GtkWidget                  *widget,
					 const gchar                *path,
					 GdkRectangle               *background_area,
					 GdkRectangle               *cell_area,
					 GtkCellRendererState        flags)
{
	if (MG_DATA_CELL_RENDERER_BOOLEAN (cell)->priv->editable) {
		GtkCellRendererClass *toggle_class = g_type_class_peek (GTK_TYPE_CELL_RENDERER_TOGGLE);
		gboolean retval, active;
		GdaValue *value;

		retval = (toggle_class->activate) (cell, event, widget, path, background_area, cell_area, flags);
		active = gtk_cell_renderer_toggle_get_active (GTK_CELL_RENDERER_TOGGLE (cell));
		value = gda_value_new_boolean (! active);
		g_signal_emit (G_OBJECT (cell), toggle_cell_signals[CHANGED], 0, path, value);
		gda_value_free (value);
		return retval;
	}

	return FALSE;
}


