/* utility.c
 * Copyright (C) 2003 - 2004 Vivien Malerba <malerba@gnome-db.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "mg-defs.h"
#include "utility.h"
#include <libgda/libgda.h>
#include "mg-data-entry.h"
#include "mg-result-set.h"
#include "mg-parameter.h"
#include "mg-server-data-type.h"
#include "mg-server.h"
#include "mg-query.h"
#include "mg-renderer.h"
#include "mg-entity.h"
#include "mg-field.h"
#include "mg-qfield.h"
#include "mg-qf-field.h"


/**
 * utility_entry_build_actions_menu
 * @obj_data:
 * @attrs:
 * @function:
 *
 * Creates a GtkMenu widget which contains the possible actions on a data entry which 
 * attributes are @attrs. After the menu has been displayed, and when an action is selected,
 * the @function callback is called with the 'user_data' being @obj_data.
 *
 * The menu item which emits the signal has a "action" attribute which describes what action the
 * user has requested.
 *
 * Returns: the new menu
 */
GtkWidget *
utility_entry_build_actions_menu (GObject *obj_data, guint attrs, GCallback function)
{
	GtkWidget *menu, *mitem;
	gchar *str;
	gboolean nullact = FALSE;
	gboolean defact = FALSE;
	gboolean reset = FALSE;

	gboolean value_is_null;
	gboolean value_is_modified;
	gboolean value_is_default;

	menu = gtk_menu_new ();

	/* which menu items to make sensitive ? */
	value_is_null = attrs & MG_DATA_ENTRY_IS_NULL;
	value_is_modified = ! (attrs & MG_DATA_ENTRY_IS_UNCHANGED);
	value_is_default = attrs & MG_DATA_ENTRY_IS_DEFAULT;

	if ((attrs & MG_DATA_ENTRY_CAN_BE_NULL) && 
	    !(attrs & MG_DATA_ENTRY_IS_NULL))
		nullact = TRUE;
	if ((attrs & MG_DATA_ENTRY_CAN_BE_DEFAULT) && 
	    !(attrs & MG_DATA_ENTRY_IS_DEFAULT))
		defact = TRUE;
	if (!(attrs & MG_DATA_ENTRY_IS_UNCHANGED)) {
		if (attrs & MG_DATA_ENTRY_HAS_VALUE_ORIG) 
			reset = TRUE;
	}

	/* set to NULL item */
	str = g_strdup (_("Unset"));
	mitem = gtk_check_menu_item_new_with_label (str);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem),
					value_is_null);
	gtk_widget_show (mitem);
	g_object_set_data (G_OBJECT (mitem), "action", GUINT_TO_POINTER (MG_DATA_ENTRY_IS_NULL));
	g_signal_connect (G_OBJECT (mitem), "activate",
			  G_CALLBACK (function), obj_data);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
	g_free (str);
	gtk_widget_set_sensitive (mitem, nullact);

	/* default value item */
	str = g_strdup (_("Set to default value"));
	mitem = gtk_check_menu_item_new_with_label (str);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), 
					value_is_default);
	gtk_widget_show (mitem);
	g_object_set_data (G_OBJECT (mitem), "action", GUINT_TO_POINTER (MG_DATA_ENTRY_IS_DEFAULT));
	g_signal_connect (G_OBJECT (mitem), "activate",
			  G_CALLBACK (function), obj_data);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
	g_free (str);
	gtk_widget_set_sensitive (mitem, defact);
		
	/* reset to original value item */
	str = g_strdup (_("Reset to original value"));
	mitem = gtk_check_menu_item_new_with_label (str);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), 
					!value_is_modified);
	gtk_widget_show (mitem);
	g_object_set_data (G_OBJECT (mitem), "action", GUINT_TO_POINTER (MG_DATA_ENTRY_IS_UNCHANGED));
	g_signal_connect (G_OBJECT (mitem), "activate",
			  G_CALLBACK (function), obj_data);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
	g_free (str);
	gtk_widget_set_sensitive (mitem, reset);

	return menu;
}


/**
 * utility_entry_build_info_colors_array
 * 
 * Creates an array of colors for the different states of an entry:
 *    Valid   <-> No special color
 *    Null    <-> Green
 *    Default <-> Blue
 *    Invalid <-> Red
 * Each status (except Valid) is represented by two colors for GTK_STATE_NORMAL and
 * GTK_STATE_PRELIGHT.
 *
 * Returns: a new array of 6 colors
 */
GdkColor **utility_entry_build_info_colors_array ()
{
	GdkColor **colors;
	GdkColor *color;
	
	colors = g_new0 (GdkColor *, 6);
	
	/* Green color */
	color = g_new0 (GdkColor, 1);
	gdk_color_parse ("#00cd66", color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[0] = color;
	
	color = g_new0 (GdkColor, 1);
	gdk_color_parse ("#00ef77", color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[1] = color;
	
	
	/* Blue color */
	color = g_new0 (GdkColor, 1);
	gdk_color_parse ("#6495ed", color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[2] = color;
	
	color = g_new0 (GdkColor, 1);
	gdk_color_parse ("#75a6fe", color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[3] = color;
	
	
	/* Red color */
	color = g_new0 (GdkColor, 1);
	gdk_color_parse ("#ff6a6a", color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[4] = color;
	
	color = g_new0 (GdkColor, 1);
	gdk_color_parse ("#ff7b7b", color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[5] = color;

	return colors;
}


static const GdaValue *get_value_from_recordset (GtkTreeModel *tree_model, GtkTreeIter *iter, 
						 MgWorkCore *core, MgParameter *context_param);
static ModelUserModifiedValue *get_user_modifs (GtkTreeModel *tree_model, GtkTreeIter *iter, MgContextNode *context_node);

/**
 * utility_grid_model_get_value
 * @tree_model:
 * @iter:
 * @core:
 * @context_node:
 * @pk_values_only:
 * @attributes:
 *
 * Fetch the GdaValue which is in @core's resultset corresponding to the @node column
 * and the row described through @iter in the @tree_model model.
 *
 * If @attributes is not %NULL, then after the call, it will contain the #MgDataEntry attributes
 * corresponding to the value.
 *
 * The returned value can be of type GDA_VALUE_TYPE_LIST if several values are to be stored.
 *
 * Returns: the GdaValue (a newly allocated one), never returns NULL.
 */
GdaValue*
utility_grid_model_get_value (GtkTreeModel *tree_model, GtkTreeIter *iter,
			      MgWorkCore *core, MgContextNode *context_node,
			      gboolean pk_values_only, guint *attributes)
{
	guint attrs = 0;
	GdaValue *retval = NULL;
	
	/* signle direct parameter */
	if (context_node->param) {
		const GdaValue *value = NULL;
		ModelUserModifiedValue *user_modif;
		gboolean is_default = FALSE;
		gboolean is_unchanged = TRUE;
		gint row;	
		const GdaValue *default_val;

		/* if (row >= 0) then the data row originally exists into the resultset, and
		 * otherwise it is a new row, to be inserted */
		gtk_tree_model_get (tree_model, iter, COLUMN_ROW_NUM, &row, -1);

		default_val = mg_parameter_get_default_value (context_node->param);
		user_modif = get_user_modifs (tree_model, iter, context_node);
		if (user_modif) {
			value = user_modif->value;
			attrs = user_modif->attributes;
			is_default = attrs & MG_DATA_ENTRY_IS_DEFAULT;
			if (row >= 0) {
				const GdaValue *orig_value;

				orig_value = get_value_from_recordset (tree_model, iter, core, context_node->param);
				is_unchanged = FALSE;
				if (((!value || gda_value_is_null (value)) && 
				     (!orig_value || gda_value_is_null (orig_value))) ||
				    (value && orig_value && 
				     (gda_value_get_type (value) == gda_value_get_type (orig_value)) &&
				     !gda_value_compare (value, orig_value)))
					is_unchanged = TRUE;
			}
			else
				is_unchanged = FALSE;
		}
		else {
			if (!g_slist_find (core->params_in_data_rs, context_node->param)) 
				value = mg_parameter_get_value (context_node->param);
			else {
				if (row >= 0)
					value = get_value_from_recordset (tree_model, iter, core, context_node->param);
				else
					value = NULL;
			}
		}
		
		
		/* Same attributes setting as for mg-form.c mg_form_initialize() around line 296 */
		attrs = attrs | ((!value || gda_value_is_null (value)) ? MG_DATA_ENTRY_IS_NULL : 0);
		attrs = attrs | (mg_parameter_get_not_null (context_node->param) ? 0 : MG_DATA_ENTRY_CAN_BE_NULL);
		attrs = attrs | (default_val ? MG_DATA_ENTRY_CAN_BE_DEFAULT : 0);
		attrs = attrs | (is_unchanged ? MG_DATA_ENTRY_IS_UNCHANGED : 0);
		
		/* Validity calculation: the same as for mg-entry-wrapper mg_entry_wrapper_get_attributes() */
		if (! (is_default && (attrs & MG_DATA_ENTRY_CAN_BE_DEFAULT))) {
			if (/*(gda_value_is_null (value) && !null_forced) ||*/
			    ((!value || gda_value_is_null (value)) && !(attrs & MG_DATA_ENTRY_CAN_BE_NULL)))
				attrs = attrs | MG_DATA_ENTRY_DATA_NON_VALID;
		}
		
		/* if we are not inserting a new row, then we consider that we have an original value */
		if (row > -1)
			attrs = attrs | MG_DATA_ENTRY_HAS_VALUE_ORIG;
		
		if (!value)
			retval = gda_value_new_null ();
		else
			retval = gda_value_copy (value);

		if (attributes)
			*attributes = attrs;

		return retval;
	}



	/* Parameter(s) constrained by a query, we only want the PK fields for the query */
	if (pk_values_only) {
		GList *values = NULL;
		ModelUserModifiedValue *user_modif;
		gboolean is_default = TRUE;
		gboolean is_unchanged = TRUE;
		gboolean is_null = TRUE;
		gboolean can_be_default = TRUE;
		gboolean can_be_null = TRUE;
		gint row;	
		GSList *params;
		GList *list = NULL;
		
		/* if (row >= 0) then the data row originally exists into the resultset, and
		 * otherwise it is a new row, to be inserted */
		gtk_tree_model_get (tree_model, iter, COLUMN_ROW_NUM, &row, -1);

		user_modif = get_user_modifs (tree_model, iter, context_node);
		
		/* building the 'values' list */
		if (user_modif) {
			if (user_modif->value && !gda_value_is_null (user_modif->value)) {
				GList *tmplist = (GList *) gda_value_get_list (user_modif->value);
				g_assert (gda_value_isa (user_modif->value, GDA_VALUE_TYPE_LIST));

				params = context_node->params;
				while (params) {
					gint pos;

					pos = GPOINTER_TO_INT (g_hash_table_lookup (context_node->params_pos_in_query, 
										    params->data));
					values = g_list_append (values, g_list_nth_data (tmplist, pos));
					params = g_slist_next (params);
				}
			}

			attrs = user_modif->attributes;
		}
		else {
			if (row >= 0) {
				params = context_node->params;
				while (params) {
					values = g_list_append (values, 
								get_value_from_recordset (tree_model, iter, core, 
											  MG_PARAMETER (params->data)));
					params = g_slist_next (params);
				}
			}
		}
		list = values;

		/* computing the attributes if necessary */
		if (attributes) {
			if (row < 0)
				is_unchanged = FALSE;

			params = context_node->params;
			while (params) {
				const GdaValue *value = NULL;
				
				if (values) {
					g_assert (list);
					value = (GdaValue *) (list->data);
					list = g_list_next (list);
				}
				
				if (can_be_default)
					can_be_default = mg_parameter_get_default_value (MG_PARAMETER (params->data)) ? TRUE : FALSE;
				
				if (can_be_null)
					can_be_null = mg_parameter_get_not_null (MG_PARAMETER (params->data)) ? FALSE : TRUE;
				
				if (is_unchanged) {
					const GdaValue *orig_value;
					
					orig_value = get_value_from_recordset (tree_model, iter, core, 
									       MG_PARAMETER (params->data));
					is_unchanged = FALSE;
					if (((!value || gda_value_is_null (value)) && (!orig_value || gda_value_is_null (orig_value))) ||
					    (value && orig_value && 
					     (gda_value_get_type (value) == gda_value_get_type (orig_value)) &&
					     !gda_value_compare (value, orig_value)))
						is_unchanged = TRUE;
				}
				
				if (is_null)
					is_null = (!value || gda_value_is_null (value)) ? TRUE : 0;
				
				params = g_slist_next (params);
			}
			
			/* Same attributes setting as for mg-form.c mg_form_initialize() around line 296 */
			attrs = attrs | (is_null ? MG_DATA_ENTRY_IS_NULL : 0);
			attrs = attrs | (can_be_null ? MG_DATA_ENTRY_CAN_BE_NULL : 0);
			attrs = attrs | (can_be_default ? MG_DATA_ENTRY_CAN_BE_DEFAULT : 0);
			attrs = attrs | (is_unchanged ? MG_DATA_ENTRY_IS_UNCHANGED : 0);
			
			/* Validity calculation: the same as for mg-entry-wrapper mg_entry_wrapper_get_attributes() */
			if (! (is_default && (attrs & MG_DATA_ENTRY_CAN_BE_DEFAULT))) {
				if (is_null && !can_be_null)
					attrs = attrs | MG_DATA_ENTRY_DATA_NON_VALID;
			}
			
			/* if we are not inserting a new row, then we consider that we have an original value */
			if (row > -1)
				attrs = attrs | MG_DATA_ENTRY_HAS_VALUE_ORIG;
		}

		if (values) {
			retval = gda_value_new_list ((GdaValueList *) values);
			g_list_free (values);
		}
		else
			retval = gda_value_new_null ();

		if (attributes)
			*attributes = attrs;
		
		return retval;
	}


	/* Parameter(s) constrained by a query, we want all the fields for the query */
	if (!pk_values_only) {
		const GdaValue *value = NULL;
		GList *values = NULL;
		ModelUserModifiedValue *user_modif;
		gboolean is_default = TRUE;
		gboolean is_unchanged = TRUE;
		gboolean is_null = TRUE;
		gboolean can_be_default = TRUE;
		gboolean can_be_null = TRUE;
		gint row;	
		GSList *params;
		GList *list = NULL;
		
		/* if (row >= 0) then the data row originally exists into the resultset, and
		 * otherwise it is a new row, to be inserted */
		gtk_tree_model_get (tree_model, iter, COLUMN_ROW_NUM, &row, -1);

		user_modif = get_user_modifs (tree_model, iter, context_node);
		
		/* building the 'values' list, and set 'value' if user_modif does exist */
		if (user_modif) {
			if (user_modif->value && !gda_value_is_null (user_modif->value)) {
				g_assert (gda_value_isa (user_modif->value, GDA_VALUE_TYPE_LIST));

				value = user_modif->value;
				values = (GList *) gda_value_get_list (user_modif->value);
			}

			attrs = user_modif->attributes;
		}
		else {
			if (row >= 0) {
				GSList *fields = mg_entity_get_visible_fields (MG_ENTITY (context_node->query));
				GSList *list;
				gint pos;
				
				list = fields;
				while (list) {
					if (IS_MG_QF_FIELD (list->data)) {
						pos = GPOINTER_TO_INT (g_hash_table_lookup (core->work_context_qf_position, 
											    list->data));
						g_assert (pos >= 0);
						values = g_list_append (values, mg_resultset_get_gdavalue (core->data_rs, row, pos));
					}
					list = g_slist_next (list);
				}
				g_slist_free (fields);
			}
		}
		list = values;

		/* computing the attributes if necessary */
		if (attributes) {
			if (row < 0)
				is_unchanged = FALSE;

			params = context_node->params;
			while (params) {
				const GdaValue *value = NULL;
				
				if (values) {
					g_assert (list);
					value = (GdaValue *) (list->data);
					list = g_list_next (list);
				}
				
				if (can_be_default)
					can_be_default = mg_parameter_get_default_value (MG_PARAMETER (params->data)) ? TRUE : FALSE;
				
				if (can_be_null)
					can_be_null = mg_parameter_get_not_null (MG_PARAMETER (params->data)) ? FALSE : TRUE;
				
				if (is_unchanged) {
					const GdaValue *orig_value;
					
					orig_value = get_value_from_recordset (tree_model, iter, core, MG_PARAMETER (params->data));
					is_unchanged = FALSE;
					if (((!value || gda_value_is_null (value)) && (!orig_value || gda_value_is_null (orig_value))) ||
					    (value && orig_value && 
					     (gda_value_get_type (value) == gda_value_get_type (orig_value)) &&
					     !gda_value_compare (value, orig_value)))
						is_unchanged = TRUE;
				}
				
				if (is_null)
					is_null = (!value || gda_value_is_null (value)) ? TRUE : 0;
				
				params = g_slist_next (params);
			}
			
			/* Same attributes setting as for mg-form.c mg_form_initialize() around line 296 */
			attrs = attrs | (is_null ? MG_DATA_ENTRY_IS_NULL : 0);
			attrs = attrs | (can_be_null ? MG_DATA_ENTRY_CAN_BE_NULL : 0);
			attrs = attrs | (can_be_default ? MG_DATA_ENTRY_CAN_BE_DEFAULT : 0);
			attrs = attrs | (is_unchanged ? MG_DATA_ENTRY_IS_UNCHANGED : 0);
			
			/* Validity calculation: the same as for mg-entry-wrapper mg_entry_wrapper_get_attributes() */
			if (! (is_default && (attrs & MG_DATA_ENTRY_CAN_BE_DEFAULT))) {
				if (is_null && !can_be_null)
					attrs = attrs | MG_DATA_ENTRY_DATA_NON_VALID;
			}
			
			/* if we are not inserting a new row, then we consider that we have an original value */
			if (row > -1)
				attrs = attrs | MG_DATA_ENTRY_HAS_VALUE_ORIG;
		}

		if (value)
			retval = gda_value_copy (value);
		else {
			if (values) {
				retval = gda_value_new_list ((GdaValueList *) values);
				g_list_free (values);
			}
			else
				retval = gda_value_new_null ();
		}
		if (attributes)
			*attributes = attrs;
		
		return retval;
	}

	/* not reached */
	return NULL;
}

static const GdaValue *
get_value_from_recordset (GtkTreeModel *tree_model, GtkTreeIter *iter, MgWorkCore *core, MgParameter *context_param)
{
	gint col;
	MgWorkCoreNode *cnode;
	gint row;
	
	g_return_val_if_fail (context_param, NULL);

	gtk_tree_model_get (tree_model, iter, COLUMN_ROW_NUM, &row, -1);
	cnode = mg_work_core_find_core_node (core, context_param);
	g_assert (cnode);

	col = cnode->position;
	g_assert (col >= 0);
		
	return mg_resultset_get_gdavalue (core->data_rs, row, col);	
}

static ModelUserModifiedValue *
get_user_modifs (GtkTreeModel *tree_model, GtkTreeIter *iter, MgContextNode *context_node)
{
	ModelUserModifiedRow *user_modifs;
	ModelUserModifiedValue *user_value = NULL;
	
	gtk_tree_model_get (tree_model, iter, COLUMN_USER_MODIFS_ROW, &user_modifs, -1);
	if (user_modifs) {
		GSList *list;
		list = user_modifs->user_values;
		while (list && !user_value) {
			if (MODEL_USER_MODIFIED_VALUE (list->data)->context_node == context_node)
				user_value = MODEL_USER_MODIFIED_VALUE (list->data);
			list = g_slist_next (list);
		}		
	}

	return user_value;
}





static void combo_core_nullified_query_cb (MgQuery *query, ComboCore *ccore);
static void combo_core_nullified_param_cb (MgParameter *param, ComboCore *ccore);
static gint *utility_combo_compute_choice_columns_mask (ComboCore *core, gint *mask_size);
/**
 * utility_combo_initialize_core
 * @conf:
 * @context:
 * @node:
 * @dependency_param_callback:
 * @data:
 *
 * Returns: a new ComboCore
 */
ComboCore *
utility_combo_initialize_core (MgConf *conf, MgContext *context, MgContextNode *node,
			       GCallback dependency_param_callback, gpointer data)
{
	ComboCore *ccore;
	GSList *list;

	g_return_val_if_fail (node && node->query, NULL);

	ccore = g_new0 (ComboCore, 1);

	/* main fields */
	ccore->conf = conf;
	g_object_add_weak_pointer (G_OBJECT (ccore->conf), (gpointer) &(ccore->conf));

	ccore->context = context;
	g_object_add_weak_pointer (G_OBJECT (ccore->context), (gpointer) &(ccore->context));

	ccore->query = node->query;
	g_object_ref (G_OBJECT (ccore->query));
	g_signal_connect (G_OBJECT (ccore->query), "nullified",
			  G_CALLBACK (combo_core_nullified_query_cb), ccore);
	list = mg_entity_get_visible_fields (MG_ENTITY (ccore->query));
	ccore->nb_visible_cols = g_slist_length (list);
	g_slist_free (list);
	ccore->dependency_param_callback = dependency_param_callback;
	ccore->dependency_data = data;

	/* parameters */
	list = node->params;
	while (list) {
		MgField *source_field;
		ComboNode *cnode = g_new0 (ComboNode, 1);

		source_field = MG_FIELD (mg_parameter_get_source_field (MG_PARAMETER (list->data)));
		cnode->param = MG_PARAMETER (list->data);
		cnode->position = mg_entity_get_field_index (MG_ENTITY (ccore->query), source_field);
		cnode->value_orig = NULL;
		cnode->value_default = NULL;
		ccore->nodes = g_slist_append (ccore->nodes, cnode);
		g_object_ref (G_OBJECT (list->data));
		g_signal_connect (G_OBJECT (list->data), "nullified",
				  G_CALLBACK (combo_core_nullified_param_cb), ccore);

		list = g_slist_next (list);
	}

	/* connect the parameters dependencies */
	list = ccore->nodes;
	while (list) {
		GSList *depend = mg_parameter_get_dependencies (COMBO_NODE (list->data)->param);
		while (depend) {
			g_signal_connect (G_OBJECT (depend->data), "changed",
					  G_CALLBACK (dependency_param_callback), data);
			depend = g_slist_next (depend);
		}
		list = g_slist_next (list);
	}

	/* compute the mask and its size */
	ccore->mask = utility_combo_compute_choice_columns_mask (ccore, &(ccore->masksize));

	return ccore;
}

static void
combo_core_nullified_query_cb (MgQuery *query, ComboCore *ccore)
{
	g_assert (ccore->query == query);
	g_signal_handlers_disconnect_by_func (G_OBJECT (query),
					      G_CALLBACK (combo_core_nullified_query_cb), ccore);
	g_object_unref (G_OBJECT (query));
	ccore->query = NULL;
}

static void
combo_core_nullified_param_cb (MgParameter *param, ComboCore *ccore)
{
	/* remove the associated ComboNode */
	GSList *list;
	ComboNode *node = NULL;

	list = ccore->nodes;
	while (list) {
		if (COMBO_NODE (list->data)->param == param)
			node = COMBO_NODE (list->data);
		list = g_slist_next (list);
	}

	g_assert (node);
	ccore->nodes = g_slist_remove (ccore->nodes, node);
	g_signal_handlers_disconnect_by_func (G_OBJECT (param),
					      G_CALLBACK (combo_core_nullified_param_cb), ccore);
	list = mg_parameter_get_dependencies (node->param);
	while (list) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (list->data),
						      G_CALLBACK (ccore->dependency_param_callback), ccore->dependency_data);
		list = g_slist_next (list);
	}

	g_object_unref (G_OBJECT (param));
	if (node->value)
		node->value = NULL; /* don't free that value since we have not copied it */
	if (node->value_orig)
		gda_value_free (node->value_orig);
	if (node->value_default)
		gda_value_free (node->value_default);
	g_free (node);
}


/**
 * utility_combo_free_core
 *
 * Free @core and cleans the associated memory allocations made when utility_combo_initialize_core()
 * was called.
 */
void
utility_combo_free_core (ComboCore *ccore)
{
	while (ccore->nodes) {
		ComboNode *node = COMBO_NODE (ccore->nodes->data);
		g_assert (node->param);
		combo_core_nullified_param_cb (node->param, ccore);
	}

	if (ccore->mask)
		g_free (ccore->mask);

	if (ccore->query)
		combo_core_nullified_query_cb (ccore->query, ccore);

	if (ccore->conf)
		g_object_remove_weak_pointer (G_OBJECT (ccore->conf), 
					      (gpointer) &(ccore->conf));
	
	if (ccore->context)
		g_object_remove_weak_pointer (G_OBJECT (ccore->context),
					      (gpointer) &(ccore->context));
	
	utility_combo_destroy_model (ccore);

	g_free (ccore);
}

static GdaDataModel *make_single_line_model (const gchar *line);

/**
 * utility_combo_compute_model
 * @core:
 *
 * Computes a new GdaDataModel model (eventualy re-runs 'core->query') if necessary.
 * Some special 1 line models are created in case of errors.
 */
void
utility_combo_compute_model (ComboCore *core)
{
	MgServer *srv;
	gboolean valid_data;
	GdaDataModel *data_model = NULL;

	srv = mg_conf_get_server (core->conf);
	utility_combo_destroy_model (core);

	if (mg_server_conn_is_opened (srv)) {
		MgResultSet *rs = NULL;
		gchar *sql;
		GError *error = NULL;
		
		sql = mg_renderer_render_as_sql (MG_RENDERER (core->query), core->context, 0, &error);
		if (!sql) {
			valid_data = FALSE;
			data_model = make_single_line_model (_("No value available"));
			if (error)
				g_error_free (error);
		}
		else {
			rs = mg_server_do_query (srv, sql, MG_SERVER_QUERY_SQL, &error);
			if (!rs) {
				valid_data = FALSE;
				data_model = make_single_line_model (error->message);
				g_error_free (error);
			}
			else {
				if (mg_resultset_get_nbtuples (rs) == 0) {
					valid_data = FALSE;
					data_model = make_single_line_model (_("No value available"));
				}
				else {
					valid_data = TRUE;
					core->resultset = rs;
					data_model = mg_resultset_get_data_model (rs);

					/* keep our own reference on the data model */
					g_object_ref (G_OBJECT (data_model));
				}
			}
		}
	}
	else {
		valid_data = FALSE;
		data_model = make_single_line_model (_("Connection not opened"));
	}

	/* setting the new model */
	core->data_model = data_model;
	core->data_model_valid = valid_data;
}

/*
 * Creates a MgDataModel object with a single line and column
 * containing the provided message
 */
static GdaDataModel *
make_single_line_model (const gchar *line)
{
	GdaValue *value;
	GdaDataModel *data_model;
	GList *rowvalues;
	
	data_model = gda_data_model_array_new (1);
	value = gda_value_new_string (line);
	rowvalues = g_list_append (NULL, value);
	gda_data_model_append_row (GDA_DATA_MODEL (data_model), rowvalues);
	g_list_free (rowvalues);
	gda_value_free (value);

	return data_model;
}

/**
 * utility_combo_destroy_model
 * @core:
 *
 * Free the memory associated with @core's model
 */
void
utility_combo_destroy_model (ComboCore *core)
{
	if (core->data_model) {
		g_object_unref (core->data_model);
		core->data_model = NULL;
		core->data_model_valid = FALSE;
	}

	if (core->resultset) {
		g_object_unref (G_OBJECT (core->resultset));
		core->resultset = NULL;
	}
}


/**
 * utility_combo_compute_choice_strings
 * @core:
 *
 * Creates a list of (allocated strings), one for each row of the data model stored within @core
 * ('core->data_model')
 *
 * Returns: a new list of strings
 */
GList *
utility_combo_compute_choice_strings (ComboCore *core)
{
	GList *strings = NULL;
	const GdaValue *value;
	gint nrows, i, j;
	GString *str;
	gchar *tmpstr;
	GdaDataModel *model = core->data_model;
	
	/* actual string rendering */
	nrows = gda_data_model_get_n_rows (model);
	for (i=0; i<nrows; i++) {
		str = g_string_new ("");
		if (core->mask) {
			for (j=0; j<core->masksize; j++) {
				value = gda_data_model_get_value_at (model, core->mask[j], i);
				if (value && !gda_value_is_null (value))
					tmpstr = gda_value_stringify (value);
				else
					tmpstr = g_strdup ("---");

				if (j>0)
					g_string_append (str, " / " );
				g_string_append (str, tmpstr);
				g_free (tmpstr);
			}
		}

		strings = g_list_append (strings, str->str);
		g_string_free (str, FALSE);
	}
	
	return strings;
}

/**
 * utility_combo_compute_display_string
 * @core:
 *
 * Creates a new string for @core (no query is run and no value is searched for 
 * in any resultset)
 *
 * Returns: a new string to display
 */
gchar *
utility_combo_compute_display_string (ComboCore *core, GList *values)
{
	gchar *retval = NULL;
	const GdaValue *value;
	gint j;
	GString *str;
	gchar *tmpstr;
	
	g_return_val_if_fail (values && (g_list_length (values) == core->nb_visible_cols), NULL);

	/* actual string rendering */
	str = g_string_new ("");
	if (core->mask) {
		for (j=0; j<core->masksize; j++) {
			value = (GdaValue*) g_list_nth_data (values, core->mask[j]);
			if (value && !gda_value_is_null (value))
				tmpstr = gda_value_stringify (value);
			else
				tmpstr = g_strdup ("---");
			if (j>0)
				g_string_append (str, " / " );
			if (tmpstr) {
				g_string_append (str, tmpstr);
				g_free (tmpstr);
			}
		}
	}

	retval = str->str;
	g_string_free (str, FALSE);
	
	return retval;
}


/**
 * utility_combo_compute_choice_columns_mask
 * @core:
 * @mask_size:
 *
 * Create a mask of which column of the resultset needs to be displayed in a combo box.
 * The mask is an array of the column numbers to be displayed, and must be de-allocated after usage.
 *
 * Returns: a new array of columns to display
 */
static gint *
utility_combo_compute_choice_columns_mask (ComboCore *core, gint *mask_size)
{
	gint ncols;
	gint *mask = NULL, masksize = 0;
	
	/* mask making: we only want columns which are not parameter values and if not internal MgQField */
	ncols = core->nb_visible_cols;
	if (ncols - g_slist_length (core->nodes) > 0) {
		gint i, current = 0;
		
		masksize = ncols - g_slist_length (core->nodes);
		mask = g_new0 (gint, masksize);
		for (i=0; i<ncols ; i++) {
			GSList *list = core->nodes;
			gboolean found = FALSE;
			while (list && !found) {
				if (COMBO_NODE (list->data)->position == i)
					found = TRUE;
				else
					list = g_slist_next (list);
			}
			if (!found) {
				MgField *field = mg_entity_get_field_by_index (MG_ENTITY (core->query), i);

				if (!mg_qfield_is_internal (MG_QFIELD (field))) {
					mask[current] = i;
					current ++;
				}
			}
		}
		masksize = current;
	}
	else {
		gint i;

		masksize = ncols;
		mask = g_new0 (gint, masksize);
		for (i=0; i<ncols; i++) {
			mask[i] = i;
		}
	}

	if (mask_size)
		*mask_size = masksize;

	return mask;
}
