/* data-handler-default.c
 *
 * Copyright (C) 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <string.h>
#include "data-entry.h"
#include "data-handler-default.h"
#include "data-handler-private.h"

/* the returned char is allocated and the one in argument freed */
static gchar *server_access_escape_chars (gchar * str);
/* the returned char is allocated and the one in argument freed */
static gchar *server_access_unescape_chars (gchar * str);







static void
emit_contents_modified (GObject * obj, GtkWidget * dd)
{
#ifdef debug_signal
	g_print (">> 'CONTENTS_MODIFIED' from emit_contents_modified de=%p\n", dd);
#endif
	g_signal_emit_by_name (G_OBJECT (dd), "contents_modified");
#ifdef debug_signal
	g_print ("<< 'CONTENTS_MODIFIED' from emit_contents_modified\n");
#endif
}

static void 
contents_changed_cb (GObject *obj, GtkWidget *dd)
{
	g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	emit_contents_modified (obj, dd);
}



/*****************************************************************************/
/*                                                                           */
/* Default functions => Read only                                            */
/*                                                                           */
/*****************************************************************************/
static GtkWidget *
widget_from_value_default (DataHandler *dh, const GdaValue * value)
{
	DataEntry *dd;
	GtkWidget *wid, *frame;

	dd = DATA_ENTRY (data_entry_new (dh));
	data_entry_set_orig_value (dd, value);

	frame = gtk_frame_new (NULL);
	data_entry_pack_default (dd, frame);

	wid = gtk_label_new (_("This data cannot be displayed or modified\n"
			       "NOTE: it is possible to write a plugin for it\n"
			       "and display it normally"));
	gtk_container_add (GTK_CONTAINER (frame), wid);

	gtk_widget_show_all (frame);

	if (value && !gda_value_is_null (value))
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	else
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (FALSE));


	return GTK_WIDGET (dd);
}

static GdaValue *
value_from_widget_default (DataHandler *dh, GtkWidget * wid)
{
	GdaValue *value;
	gboolean set;

	g_return_val_if_fail (wid && IS_DATA_ENTRY (wid), NULL);

	set = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "is_set"));

	if (! set)
		value = gda_value_new_null ();
	else {
		switch (data_entry_get_orig_type (DATA_ENTRY (wid))) {
		case GDA_VALUE_TYPE_BINARY:
		case GDA_VALUE_TYPE_GEOMETRIC_POINT:
		case GDA_VALUE_TYPE_LIST:
		case GDA_VALUE_TYPE_TYPE:
			value = gda_value_copy (data_entry_get_orig_value (DATA_ENTRY (wid)));
			break;
		default:
			value = gda_value_new_null ();
		}
	}

	return value;
}

static void
widget_update_default (DataHandler *dh, GtkWidget * wid, const GdaValue * value, gboolean value_as_default)
{
	g_return_if_fail (wid && IS_DATA_ENTRY (wid));

	if (value_as_default)
		data_entry_set_orig_value (DATA_ENTRY (wid), value);
	
	if (value && !gda_value_is_null (value))
		g_object_set_data (G_OBJECT (wid), "is_set", GINT_TO_POINTER (TRUE));
	else
		g_object_set_data (G_OBJECT (wid), "is_set", GINT_TO_POINTER (FALSE));

	emit_contents_modified (NULL, wid);
}

static gchar *
sql_from_value_default (DataHandler *dh, const GdaValue * value)
{
	gchar *str, *str2, *retval;

	if (!value || gda_value_is_null (value))
		return NULL;

	str = server_resultset_stringify (value);
	if (str) {
		str2 = server_access_escape_chars (str);
		retval = g_strdup_printf ("'%s'", str2);
		g_free (str2);
	}
	else
		retval = NULL;

	return retval;
}

static gchar *
str_from_value_default (DataHandler *dh, const GdaValue * value)
{
	if (value || gda_value_is_null (value)) 
		return NULL;
	else
		return g_strdup (_("Binary data"));
}

/*****************************************************************************/
/*                                                                           */
/* String functions                                                          */
/*                                                                           */
/*****************************************************************************/
struct string_extension {
	GtkWidget *entry;
};
#define STRING_EXTENSION(de) ((struct string_extension *) (DATA_ENTRY (de)->extension))

void string_free_extension (DataEntry *de) {
	g_free (de->extension);
	de->extension = NULL;
}

static GdaValue *
value_from_widget_string (DataHandler *dh, GtkWidget * wid)
{
	GdaValue *value;
	gboolean set;

	g_return_val_if_fail (wid && IS_DATA_ENTRY (wid), NULL);

	set = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "is_set"));

	if (!set)
		value = gda_value_new_null ();
	else
		value = gda_value_new_string (gtk_entry_get_text (
			GTK_ENTRY (STRING_EXTENSION (wid)->entry)));

	return value;
}


static void widget_update_string (DataHandler *dh, GtkWidget * dd, 
				  const GdaValue * value, gboolean value_as_default);
static GtkWidget *
widget_from_value_string (DataHandler *dh, const GdaValue * value)
{
	DataEntry *dd;
	GtkWidget *wid;

	dd = DATA_ENTRY (data_entry_new (dh));
	wid = gtk_entry_new ();
	data_entry_pack_default (dd, wid);
	gtk_widget_show (wid);

	dd->free_extension = string_free_extension;
	dd->extension = g_new0 (struct string_extension, 1);
	STRING_EXTENSION (dd)->entry = wid;

	if (value)
		widget_update_string (dh, GTK_WIDGET (dd), value, TRUE);

	g_signal_connect (G_OBJECT (wid), "changed",
			  G_CALLBACK (contents_changed_cb), dd);

	return GTK_WIDGET (dd);
}


static gchar *str_from_value_string (DataHandler *dh, const GdaValue * value);
static void
widget_update_string (DataHandler *dh, GtkWidget * dd, 
		      const GdaValue * value, gboolean value_as_default)
{
	gchar *str;

	g_return_if_fail (dd && IS_DATA_ENTRY (dd));

	g_signal_handlers_block_by_func (G_OBJECT (STRING_EXTENSION (dd)->entry), 
					 G_CALLBACK (contents_changed_cb), dd);

	if (value_as_default)
		data_entry_set_orig_value (DATA_ENTRY (dd), value);

	if (value && !gda_value_is_null (value)) {
		str = str_from_value_string (dh, value);
		if (str) {
			gtk_entry_set_text (GTK_ENTRY (STRING_EXTENSION (dd)->entry), str);
			g_free (str);
		}
		else 
			gtk_entry_set_text (GTK_ENTRY (STRING_EXTENSION (dd)->entry), "");
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	}
	else {
		gtk_entry_set_text (GTK_ENTRY (STRING_EXTENSION (dd)->entry), "");
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (FALSE));
	}
	
	emit_contents_modified (G_OBJECT (STRING_EXTENSION (dd)->entry), dd);
	g_signal_handlers_unblock_by_func (G_OBJECT (STRING_EXTENSION (dd)->entry), 
					   G_CALLBACK (contents_changed_cb), dd);
}

static gchar *
sql_from_value_string (DataHandler *dh, const GdaValue * value)
{
	gchar *str, *str2, *retval;

	if ((value->type == GDA_VALUE_TYPE_STRING) &&
	    !gda_value_get_string (value))
		return NULL;

	str = server_resultset_stringify (value);
	if (str) {
		str2 = server_access_escape_chars (str);
		retval = g_strdup_printf ("'%s'", str2);
		g_free (str2);
	}
	else
		retval = NULL;

	return retval;
}


static gchar *
str_from_value_string (DataHandler *dh, const GdaValue * value)
{
	if (value) {
		if ((value->type == GDA_VALUE_TYPE_STRING) &&
		    !gda_value_get_string (value))
			return NULL;


		return server_resultset_stringify (value);
	}
	else
		return g_strdup ("");
}

/*****************************************************************************/
/*                                                                           */
/* Functions for the numbers                                                 */
/*                                                                           */
/*****************************************************************************/
struct number_extension {
	GtkWidget *entry;
};
#define NUMBER_EXTENSION(de) ((struct number_extension *) (DATA_ENTRY (de)->extension))

void number_free_extension (DataEntry *de) {
	g_free (de->extension);
	de->extension = NULL;
}


static GdaValue *value_from_widget_number (DataHandler *dh, GtkWidget * wid);
static void widget_update_number (DataHandler *dh, GtkWidget * dd, 
				  const GdaValue * value, gboolean value_as_default);
static GtkWidget *
widget_from_value_number (DataHandler *dh, const GdaValue * value)
{
	DataEntry *dd;
	GtkWidget *wid;

	dd = DATA_ENTRY (data_entry_new (dh));

	wid = gtk_entry_new ();
	data_entry_pack_default (dd, wid);
	gtk_widget_show (wid);

	dd->free_extension = number_free_extension;
	dd->extension = g_new0 (struct number_extension, 1);
	NUMBER_EXTENSION (dd)->entry = wid;

	if (value)
		widget_update_number (dh, GTK_WIDGET (dd), value, TRUE);
	g_signal_connect (G_OBJECT (wid), "changed",
			  G_CALLBACK (contents_changed_cb), dd);

	return GTK_WIDGET (dd);
}

static GdaValue *
value_from_widget_number (DataHandler *dh, GtkWidget * wid)
{
	GdaValue *value;
	gboolean set;

	g_return_val_if_fail (wid && IS_DATA_ENTRY (wid), NULL);

	set = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "is_set"));
	
	if (!set)
		value = gda_value_new_null ();
	else {
		GdaNumeric gnum;
		gint64 bigint;
		gint vint;
		gfloat vsingle;
		gdouble vdouble;
		gshort vsmall;
		gchar vtiny;

		switch (data_entry_get_orig_type (DATA_ENTRY (wid))) {
		case GDA_VALUE_TYPE_NUMERIC:
			gnum.number = g_strdup (gtk_entry_get_text (
				GTK_ENTRY (NUMBER_EXTENSION (wid)->entry)));
			gnum.precision = 10; /* FIXME */
			gnum.width = 10;     /* FIXME */
			value = gda_value_new_numeric (&gnum);
			g_free (gnum.number);
			break;
		case GDA_VALUE_TYPE_BIGINT:
			bigint = atoi (gtk_entry_get_text (GTK_ENTRY (NUMBER_EXTENSION (wid)->entry)));
			value = gda_value_new_bigint (bigint);
			break;
		case GDA_VALUE_TYPE_DOUBLE:
			vdouble = atof (gtk_entry_get_text (GTK_ENTRY (NUMBER_EXTENSION (wid)->entry)));
			value = gda_value_new_double (vdouble);
			break;
		case GDA_VALUE_TYPE_INTEGER:
		case GDA_VALUE_TYPE_UNKNOWN:
		default:
			vint = atoi (gtk_entry_get_text (GTK_ENTRY (NUMBER_EXTENSION (wid)->entry)));
			value = gda_value_new_integer (vint);
			break;
		case GDA_VALUE_TYPE_SINGLE:
			vsingle = atof (gtk_entry_get_text (GTK_ENTRY (NUMBER_EXTENSION (wid)->entry)));
			value = gda_value_new_single (vsingle);
			break;
		case GDA_VALUE_TYPE_SMALLINT:
			vsmall = atoi (gtk_entry_get_text (GTK_ENTRY (NUMBER_EXTENSION (wid)->entry)));
			value = gda_value_new_smallint (vsmall);
			break;
		case GDA_VALUE_TYPE_TINYINT:
			vtiny = *(gtk_entry_get_text (GTK_ENTRY (NUMBER_EXTENSION (wid)->entry)));
			value = gda_value_new_tinyint (vtiny);
			break;
		}
	}		

	return value;	
}

static gchar *
str_from_value_number (DataHandler *dh, const GdaValue * value)
{
	return server_resultset_stringify (value);
}

static void
widget_update_number (DataHandler *dh, GtkWidget * dd, 
		      const GdaValue * value, gboolean value_as_default)
{
	gchar *str;

	g_signal_handlers_block_by_func (G_OBJECT (NUMBER_EXTENSION (dd)->entry),
					 G_CALLBACK (contents_changed_cb), dd);

	if (value_as_default)
		data_entry_set_orig_value (DATA_ENTRY (dd), value);

	if (value && !gda_value_is_null (value)) {
		str = str_from_value_number (dh, value);
		if (str) {
			gtk_entry_set_text (GTK_ENTRY (NUMBER_EXTENSION (dd)->entry), str);
			g_free (str);
		}
		else
			gtk_entry_set_text (GTK_ENTRY (NUMBER_EXTENSION (dd)->entry), "");
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	}
	else {
		gtk_entry_set_text (GTK_ENTRY (NUMBER_EXTENSION (dd)->entry), "");
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (FALSE));
	}

	emit_contents_modified (G_OBJECT (NUMBER_EXTENSION (dd)->entry), dd);
	g_signal_handlers_unblock_by_func (G_OBJECT (NUMBER_EXTENSION (dd)->entry),
					   G_CALLBACK (contents_changed_cb), dd);
}

static gchar *
sql_from_value_number (DataHandler *dh, const GdaValue * value)
{
	gchar *retval, *ptr;

	if (!value || gda_value_is_null (value))
		return NULL;

	retval = server_resultset_stringify (value);
	/* FIXME: take better care of the locales here */
	ptr = retval;
	while (*ptr != '\0') {
		if (*ptr == ',')
			*ptr = '.';
		ptr++;
	}

	return retval;
}



/*****************************************************************************/
/*                                                                           */
/* Functions for the booleans                                                */
/*                                                                           */
/*****************************************************************************/
struct bool_extension {
	GtkWidget *toggle;
};
#define BOOL_EXTENSION(de) ((struct bool_extension *) (DATA_ENTRY (de)->extension))

void bool_free_extension (DataEntry *de) {
	g_free (de->extension);
	de->extension = NULL;
}


static void
bool_emit_contents_modified (GObject * obj, GtkWidget * dd)
{
	gboolean set;

	set = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (dd), "is_set"));

	if (set) {
		if (GTK_TOGGLE_BUTTON (obj)->active)
			gtk_label_set_text (GTK_LABEL (GTK_BIN (obj)->child), _("Yes"));
		else
			gtk_label_set_text (GTK_LABEL (GTK_BIN (obj)->child), _("No"));
	}
	else
		gtk_label_set_text (GTK_LABEL (GTK_BIN (obj)->child), _("Unset"));

#ifdef debug_signal
	g_print (">> 'CONTENTS_MODIFIED' from bool_emit_contents_modified de=%p\n", dd);
#endif
	g_signal_emit_by_name (G_OBJECT (dd), "contents_modified");
#ifdef debug_signal
	g_print (">> 'CONTENTS_MODIFIED' from bool_emit_contents_modified\n");
#endif
}

static void
bool_contents_changed_cb (GObject * obj, GtkWidget * dd)
{
	g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	bool_emit_contents_modified (obj, dd);
}


static void widget_update_bool (DataHandler *dh, GtkWidget * dd, 
				const GdaValue * value, gboolean value_as_default);

static GtkWidget *
widget_from_value_bool (DataHandler *dh, const GdaValue * value)
{
	DataEntry *dd;
	GtkWidget *wid, *hb;

	dd = DATA_ENTRY (data_entry_new (dh));

	hb = gtk_hbox_new (FALSE, 0);
	data_entry_pack_default (dd, hb);
	gtk_widget_show (hb);
	wid = gtk_toggle_button_new_with_label (_("No"));
	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);

	dd->free_extension = bool_free_extension;
	dd->extension = g_new0 (struct bool_extension, 1);
	BOOL_EXTENSION (dd)->toggle = wid;

	widget_update_bool (dh, GTK_WIDGET (dd), value, TRUE);

	g_signal_connect (G_OBJECT (wid), "toggled",
			  G_CALLBACK (bool_contents_changed_cb), dd);

	return GTK_WIDGET (dd);
}

static GdaValue *
value_from_widget_bool (DataHandler *dh, GtkWidget * wid)
{
	GdaValue *value;
	gboolean set;

	g_return_val_if_fail (wid && IS_DATA_ENTRY (wid), NULL);
	set = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "is_set"));

	if (!set)
		value = gda_value_new_null ();
	else {
		gboolean bool;

		bool = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (BOOL_EXTENSION (wid)->toggle));
		value = gda_value_new_boolean (bool);
	}

	return value;	
}

static void
widget_update_bool (DataHandler *dh, GtkWidget * dd, 
		    const GdaValue * value, gboolean value_as_default)
{
	g_signal_handlers_block_by_func (G_OBJECT (BOOL_EXTENSION (dd)->toggle),
					 G_CALLBACK (bool_contents_changed_cb), dd);
	
	if (value_as_default)
		data_entry_set_orig_value (DATA_ENTRY (dd), value);

	if (value && !gda_value_is_null (value)) {
		if (gda_value_get_boolean (value))
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (BOOL_EXTENSION (dd)->toggle),
						      TRUE);
		else
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (BOOL_EXTENSION (dd)->toggle),
						      FALSE);
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	}
	else {

		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (BOOL_EXTENSION (dd)->toggle), FALSE);
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (FALSE));
	}

	bool_emit_contents_modified (G_OBJECT (BOOL_EXTENSION (dd)->toggle), dd);
	g_signal_handlers_unblock_by_func (G_OBJECT (BOOL_EXTENSION (dd)->toggle), 
					   G_CALLBACK (bool_contents_changed_cb), dd);
}

static gchar *
sql_from_value_bool (DataHandler *dh, const GdaValue * value)
{
	gchar *retval = NULL;

	if (!value || gda_value_is_null (value))
		retval = NULL;
	else {
		if (gda_value_get_boolean (value))
			retval = g_strdup ("'t'");
		else
			retval = g_strdup ("'f'");
	}

	return retval;
}

static gchar *
str_from_value_bool (DataHandler *dh, const GdaValue * value)
{
	return server_resultset_stringify (value);
}

/*****************************************************************************/
/*                                                                           */
/* Functions for the dates                                                   */
/*                                                                           */
/*****************************************************************************/
/* General notes about months representations:
   GtkCalendar gets months in [0-11]
   GDate represents months in [1-12]
   struct tm represents months in [0-11]
   GdaDate represents months in [1-12]
*/
struct date_extension {
	GtkWidget *entry;
	GtkWidget *date;
	GtkWidget *window;
	GtkWidget *date_button;
};
#define DATE_EXTENSION(de) ((struct date_extension *) (DATA_ENTRY (de)->extension))

void date_free_extension (DataEntry *de) {
	gtk_widget_destroy (DATE_EXTENSION (de)->window);
	g_free (de->extension);
	de->extension = NULL;
}


static void widget_update_date (DataHandler *dh, GtkWidget * dd, 
				const GdaValue * value, gboolean value_as_default);

/* callbacks for the Date GUI */
static gint date_delete_popup (GtkWidget *widget, DataEntry *de);
static gint date_key_press_popup (GtkWidget *widget, GdkEventKey *event, DataEntry *de);
static gint date_button_press_popup (GtkWidget *widget, GdkEventButton *event, DataEntry *de);
static void date_day_selected (GtkCalendar *calendar, DataEntry *de);
static void date_day_selected_double_click (GtkCalendar *calendar, DataEntry *de);
static void date_calendar_choose_cb (GtkWidget *button, DataEntry *de);

static GtkWidget *
widget_from_value_date (DataHandler *dh, const GdaValue * value)
{
	DataEntry *de;
	GtkWidget *wid, *hb, *window;

	de = DATA_ENTRY (data_entry_new (dh));
	de->free_extension = date_free_extension;
	de->extension = g_new0 (struct date_extension, 1);

	hb = gtk_hbox_new (FALSE, 0);
	data_entry_pack_default (de, hb);
	gtk_widget_show (hb);

	/* text entry */
	wid = gtk_entry_new ();
	gtk_entry_set_max_length (GTK_ENTRY (wid), 10); /* for a numerical date format */
	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);
	g_signal_connect (G_OBJECT (wid), "changed",
			  G_CALLBACK (contents_changed_cb), de);
	DATE_EXTENSION (de)->entry = wid;

	/* window to hold the calendar popup */
	window = gtk_window_new (GTK_WINDOW_POPUP);
        gtk_widget_set_events (window, gtk_widget_get_events (window) | GDK_KEY_PRESS_MASK);
	gtk_window_set_resizable (GTK_WINDOW (window), FALSE);
	g_signal_connect (G_OBJECT (window), "delete_event",
                          G_CALLBACK (date_delete_popup), de);
        g_signal_connect (G_OBJECT (window), "key_press_event",
                          G_CALLBACK (date_key_press_popup), de);
        g_signal_connect (G_OBJECT (window), "button_press_event",
                          G_CALLBACK (date_button_press_popup), de);
	DATE_EXTENSION (de)->window = window;

	/* calendar */
	wid = gtk_calendar_new ();
	DATE_EXTENSION (de)->date = wid;
	gtk_container_add (GTK_CONTAINER (window), wid);
	gtk_widget_show (wid);
	g_signal_connect (G_OBJECT (wid), "day_selected",
                          G_CALLBACK (date_day_selected), de);
        g_signal_connect (G_OBJECT (wid), "day_selected_double_click",
                          G_CALLBACK (date_day_selected_double_click), de);

	/* button to pop up the calendar */
	wid = gtk_button_new_with_label (_("Choose"));
	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);
	g_signal_connect (G_OBJECT (wid), "clicked",
			  G_CALLBACK (date_calendar_choose_cb), de);
	DATE_EXTENSION (de)->date_button = wid;
	
	/* update the widget */
	widget_update_date (dh, GTK_WIDGET (de), value, TRUE);


	return GTK_WIDGET (de);
}

static void hide_popup (DataEntry *de)
{
	gtk_widget_hide (DATE_EXTENSION (de)->window);
	gtk_grab_remove (DATE_EXTENSION (de)->window);
}

static gint
date_delete_popup (GtkWidget *widget, DataEntry *de)
{
	hide_popup (de);
	return TRUE;
}

static gint
date_key_press_popup (GtkWidget *widget, GdkEventKey *event, DataEntry *de)
{
	if (event->keyval != GDK_Escape)
                return FALSE;

        g_signal_stop_emission_by_name (widget, "key_press_event");
        hide_popup (de);

        return TRUE;
}

static gint
date_button_press_popup (GtkWidget *widget, GdkEventButton *event, DataEntry *de)
{
	GtkWidget *child;

        child = gtk_get_event_widget ((GdkEvent *) event);

        /* We don't ask for button press events on the grab widget, so
         *  if an event is reported directly to the grab widget, it must
         *  be on a window outside the application (and thus we remove
         *  the popup window). Otherwise, we check if the widget is a child
         *  of the grab widget, and only remove the popup window if it
         *  is not.
         */
        if (child != widget) {
                while (child) {
                        if (child == widget)
                                return FALSE;
                        child = child->parent;
                }
        }

        hide_popup (de);

        return TRUE;
}

static void
date_day_selected (GtkCalendar *calendar, DataEntry *de)
{
	char buffer [256];
        guint year, month, day;
        struct tm mtm = {0};
        char *str_utf8;

        gtk_calendar_get_date (calendar, &year, &month, &day);

        mtm.tm_mday = day;
        mtm.tm_mon = month;
        if (year > 1900)
                mtm.tm_year = year - 1900;
        else
                mtm.tm_year = year;

        if (strftime (buffer, sizeof (buffer), "%x", &mtm) == 0)
                strcpy (buffer, "???");
        buffer[sizeof(buffer)-1] = '\0';

        str_utf8 = g_locale_to_utf8 (buffer, -1, NULL, NULL, NULL);
        gtk_entry_set_text (GTK_ENTRY (DATE_EXTENSION (de)->entry),
                            str_utf8 ? str_utf8 : "");
        g_free (str_utf8);
}

static void
date_day_selected_double_click (GtkCalendar *calendar, DataEntry *de)
{
	hide_popup (de);
}

static gboolean popup_grab_on_window (GdkWindow *window, guint32 activate_time);
static void position_popup (DataEntry *de);
static GdaValue *value_from_widget_date (DataHandler *dh, GtkWidget * wid);
static void
date_calendar_choose_cb (GtkWidget *button, DataEntry *de)
{
	GdaValue *value;
	guint year, month, day;
	
	/* setting the calendar to the latest displayed date */
	value = value_from_widget_date (data_entry_get_data_handler (de), GTK_WIDGET (de));
	if (value && !gda_value_is_null (value)) {
		const GdaDate *date;
		
		date = gda_value_get_date (value);
		year = date->year;
		month = date->month - 1;
		day = date->day;
	}
	else {
		time_t now;
		struct tm *stm;
		
		now = time (NULL);
		stm = localtime (&now);
		year = stm->tm_year + 1900;
		month = stm->tm_mon;
		day = stm->tm_mday;
	}
	
	gtk_calendar_select_month (GTK_CALENDAR (DATE_EXTENSION (de)->date), month, year);
	gtk_calendar_select_day (GTK_CALENDAR (DATE_EXTENSION (de)->date), day);
	
	/* popup window */
	/* Temporarily grab pointer and keyboard, copied from GnomeDateEdit */
	if (!popup_grab_on_window (button->window,
                                   gtk_get_current_event_time ()))
                return;
	position_popup (de);
	gtk_grab_add (DATE_EXTENSION (de)->window);
	gtk_widget_show (DATE_EXTENSION (de)->window);
	gtk_widget_grab_focus (DATE_EXTENSION (de)->date);
	popup_grab_on_window (DATE_EXTENSION (de)->window->window,
                              gtk_get_current_event_time ());
}

static gboolean 
popup_grab_on_window (GdkWindow *window, guint32 activate_time)
{
	if ((gdk_pointer_grab (window, TRUE,
                               GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
                               GDK_POINTER_MOTION_MASK,
                               NULL, NULL, activate_time) == 0)) {
                if (gdk_keyboard_grab (window, TRUE,
                                       activate_time) == 0)
                        return TRUE;
                else {
                        gdk_pointer_ungrab (activate_time);
                        return FALSE;
                }
        }
        return FALSE;
}

static void 
position_popup (DataEntry *de)
{
	gint x, y;
        gint bwidth, bheight;
        GtkRequisition req;

        gtk_widget_size_request (DATE_EXTENSION (de)->window, &req);

        gdk_window_get_origin (DATE_EXTENSION (de)->date_button->window, &x, &y);

        x += DATE_EXTENSION (de)->date_button->allocation.x;
        y += DATE_EXTENSION (de)->date_button->allocation.y;
        bwidth = DATE_EXTENSION (de)->date_button->allocation.width;
        bheight = DATE_EXTENSION (de)->date_button->allocation.height;

        x += bwidth - req.width;
        y += bheight;

        if (x < 0)
                x = 0;

        if (y < 0)
                y = 0;

        gtk_window_move (GTK_WINDOW (DATE_EXTENSION (de)->window), x, y);
}

static GdaValue *
value_from_widget_date (DataHandler *dh, GtkWidget * wid)
{
	GdaValue *value;
	gboolean set;

	g_return_val_if_fail (wid && IS_DATA_ENTRY (wid), NULL);

	set = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "is_set"));

	if (!set)
		value = gda_value_new_null ();
	else {
		GDate *gdate;
		
		gdate = g_date_new ();
		g_date_set_parse (gdate, gtk_entry_get_text (GTK_ENTRY (DATE_EXTENSION (wid)->entry)));
		
		if (g_date_valid (gdate)) {
			GdaDate date;
			
			date.year = g_date_get_year (gdate);
			date.month = g_date_get_month (gdate);
			date.day = g_date_get_day (gdate);
			
			value = gda_value_new_date (&date);
		}
		else 
			value = gda_value_new_null ();	
		
		g_date_free (gdate);
	}

	return value;
}

static void
widget_update_date (DataHandler *dh, GtkWidget * dd, const GdaValue * value, gboolean value_as_default)
{

	GDate *gdate;
	gchar str[256];
	size_t size;

	if (value_as_default) 
		data_entry_set_orig_value (DATA_ENTRY (dd), value);

	if (value && !gda_value_is_null (value)) {
		const GdaDate *date;
		guint year, month, day;

		date = gda_value_get_date (value);
		year = date->year;
		month = date->month;
		day = date->day;

		gdate = g_date_new ();
		g_date_set_dmy (gdate, day, month, year);

		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	}
	else {
		guint year, month, day;
		time_t now;
		struct tm *stm;

		now = time (NULL);
		stm = localtime (&now);
		year = stm->tm_year + 1900;
		month = stm->tm_mon;
		day = stm->tm_mday;

		gdate = g_date_new ();
		g_date_set_dmy (gdate, day, month+1, year);

		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (FALSE));
	}

	g_signal_handlers_block_by_func (G_OBJECT (DATE_EXTENSION (dd)->entry),
					 G_CALLBACK (contents_changed_cb), dd);
	
	size = g_date_strftime (str, sizeof (str), "%x", gdate);
	g_date_free (gdate);
	if (size == 0)
		str[0] = 0;

	gtk_entry_set_text (GTK_ENTRY (DATE_EXTENSION (dd)->entry), str);

	emit_contents_modified (G_OBJECT (DATE_EXTENSION (dd)->entry), dd);

	g_signal_handlers_unblock_by_func (G_OBJECT (DATE_EXTENSION (dd)->entry),
					   G_CALLBACK (contents_changed_cb), dd);
}

static gchar *
sql_from_value_date (DataHandler *dh, const GdaValue * value)
{
	gchar *str;
	const GdaDate *date;

	if (!value || gda_value_is_null (value))
		return NULL;

	date = gda_value_get_date (value);
	str = g_strdup_printf ("'%02d-%02d-%04d'",
			       date->month,
			       date->day,
			       date->year);
	return str;
}

static gchar *
str_from_value_date (DataHandler *dh, const GdaValue * value)
{
	return server_resultset_stringify (value);
}


/*****************************************************************************/
/*                                                                           */
/* Functions for the times                                                   */
/*                                                                           */
/*****************************************************************************/
struct time_extension {
	GtkWidget *entry;
	glong      timezone; /* since we don't manage it, we need to keep its default value */
};
#define TIME_EXTENSION(de) ((struct time_extension *) (DATA_ENTRY (de)->extension))

void time_free_extension (DataEntry *de) {
	g_free (de->extension);
	de->extension = NULL;
}


static void widget_update_time (DataHandler *dh, GtkWidget * dd, 
				const GdaValue * value, gboolean value_as_default);


static GtkWidget *
widget_from_value_time (DataHandler *dh, const GdaValue * value)
{
	DataEntry *dd;
	GtkWidget *wid, *hb;

	dd = DATA_ENTRY (data_entry_new (dh));

	hb = gtk_hbox_new (FALSE, 0);
	data_entry_pack_default (dd, hb);
	gtk_widget_show (hb);

	/* text entry */
	wid = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	gtk_entry_set_max_length (GTK_ENTRY (wid), 8); /* max for HH:MM:SS format */
	gtk_widget_show (wid);

	dd->free_extension = time_free_extension;
	dd->extension = g_new0 (struct time_extension, 1);
	TIME_EXTENSION (dd)->entry = wid;
	TIME_EXTENSION (dd)->timezone = TIMEZONE_INVALID;

	/* small label */
	wid = gtk_label_new (_("hh:mm:ss"));
	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);

	widget_update_time (dh, GTK_WIDGET (dd), value, TRUE);

	g_signal_connect (G_OBJECT (TIME_EXTENSION (dd)->entry), "changed",
			  G_CALLBACK (contents_changed_cb), dd);

	return GTK_WIDGET (dd);
}


static GdaValue *text_to_time (DataEntry *de);

static GdaValue *
value_from_widget_time (DataHandler *dh, GtkWidget * wid)
{
	GdaValue *value;
	gboolean set;

	g_return_val_if_fail (wid && IS_DATA_ENTRY (wid), NULL);

	set = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "is_set"));

	if (!set)
		value = gda_value_new_null ();
	else 
		value = text_to_time (DATA_ENTRY (wid));

	return value;
}

static GdaValue *
text_to_time (DataEntry *de)
{
	GdaValue *value = NULL;
	GdaTime tim;
	gboolean error = FALSE;
	gchar *str;
	
	str = gtk_entry_get_text (GTK_ENTRY (TIME_EXTENSION (de)->entry));
	tim.timezone = TIME_EXTENSION (de)->timezone;

	if (!str || !(*str))
		error = TRUE;
	else {
		gchar *text, *ptr;
		text = g_strdup (str);
		ptr = (gchar *) strtok (text, "-/.:");
		tim.hour = atoi (ptr);

		if (!(ptr = (gchar *) strtok (NULL, "-/.:")))
			error = TRUE;
		else
			tim.minute = atoi (ptr);

		if (!error) {
			if (!(ptr = (gchar *) strtok (NULL, "-/.:")))
				error = TRUE;
			else
				tim.second = atoi (ptr);
		}

		g_free (text);
	}

	if (error)
		value = gda_value_new_null ();
	else 
		value = gda_value_new_time (&tim);	

	return value;
}


static void
widget_update_time (DataHandler *dh, GtkWidget * dd, const GdaValue * value, gboolean value_as_default)
{
	gchar *strtime, *str_utf8;

	if (value_as_default) 
		data_entry_set_orig_value (DATA_ENTRY (dd), value);

	g_signal_handlers_block_by_func (G_OBJECT (TIME_EXTENSION (dd)->entry),
					 G_CALLBACK (contents_changed_cb), dd);

	if (value && !gda_value_is_null (value)) {
		const GdaTime *tim;

		tim = gda_value_get_time (value);
		strtime = g_strdup_printf ("%02d:%02d:%02d", tim->hour, tim->minute, tim->second);
		if (value_as_default)
			TIME_EXTENSION (dd)->timezone = tim->timezone;
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	}
	else {
		time_t now;
		struct tm *stm;

		now = time (NULL);
		stm = localtime (&now);
		strtime = g_strdup_printf ("%02d:%02d:%02d", stm->tm_hour, stm->tm_min, stm->tm_sec);
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (FALSE));
	}

	str_utf8 = g_locale_to_utf8 (strtime, -1, NULL, NULL, NULL);
	g_free (strtime);
	gtk_entry_set_text (GTK_ENTRY (TIME_EXTENSION (dd)->entry), str_utf8 ? str_utf8 : "");
        g_free (str_utf8);
	
	emit_contents_modified (G_OBJECT (TIME_EXTENSION (dd)->entry), dd);

	g_signal_handlers_unblock_by_func (G_OBJECT (TIME_EXTENSION (dd)->entry),
					   G_CALLBACK (contents_changed_cb), dd);
}

static gchar *
sql_from_value_time (DataHandler *dh, const GdaValue * value)
{
	gchar *str;
	const GdaTime *tim;

	if (!value || gda_value_is_null (value))
		return NULL;

	tim = gda_value_get_time (value);
	str = g_strdup_printf ("'%02d:%02d:%02d'",
			       tim->hour,
			       tim->minute,
			       tim->second);
	return str;
}

static gchar *
str_from_value_time (DataHandler *dh, const GdaValue * value)
{
	return server_resultset_stringify (value);
}




/*****************************************************************************/
/*                                                                           */
/* Functions for the timestamps                                              */
/*                                                                           */
/*****************************************************************************/
struct ts_extension {
	GtkWidget   *date;
	DataHandler *date_handler;
	GtkWidget   *time;
	DataHandler *time_handler;
	gulong       fraction; /* not handled => keep its original value here */
};
#define TS_EXTENSION(de) ((struct ts_extension *) (DATA_ENTRY (de)->extension))

void ts_free_extension (DataEntry *de) {
	g_free (de->extension);
	de->extension = NULL;
}


static void widget_update_ts (DataHandler *dh, GtkWidget * dd, 
			      const GdaValue * value, gboolean value_as_default);
static GtkWidget *
widget_from_value_ts (DataHandler *dh, const GdaValue * value)
{
	DataEntry *dd;
	GtkWidget *wid, *hb;
	DataHandler *handler;
	GdaValue *subvalue;
	GdaDate date;
	GdaTime tim;
	const GdaTimestamp *tims = NULL;

	if (value && gda_value_isa (value, GDA_VALUE_TYPE_TIMESTAMP))
		tims = gda_value_get_timestamp (value);

	dd = DATA_ENTRY (data_entry_new (dh));
	dd->free_extension = time_free_extension;
	dd->extension = g_new0 (struct time_extension, 1);

	hb = gtk_hbox_new (FALSE, 0);
	data_entry_pack_default (dd, hb);
	gtk_widget_show (hb);

	/* date part */
	handler = server_access_get_handler_from_gda (dh->priv->srv, GDA_VALUE_TYPE_DATE);
	if (tims) {
		date.year = tims->year;
		date.month = tims->month;
		date.day = tims->day;
		subvalue = gda_value_new_date (&date);
	}
	else
		subvalue = gda_value_new_null ();
	wid = data_handler_get_widget_from_value (handler, subvalue);
	data_entry_set_show_actions_menu (DATA_ENTRY (wid), FALSE);
	gda_value_free (subvalue);
	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);

	TS_EXTENSION (dd)->date = wid;
	TS_EXTENSION (dd)->date_handler = handler;

	/* time part */
	handler = server_access_get_handler_from_gda (dh->priv->srv, GDA_VALUE_TYPE_TIME);
	if (tims) {
		tim.hour = tims->hour;
		tim.minute = tims->minute;
		tim.second = tims->second;
		tim.timezone = tims->timezone;
		subvalue = gda_value_new_time (&tim);
	}
	else
		subvalue = gda_value_new_null ();
	wid = data_handler_get_widget_from_value (handler, subvalue);
	data_entry_set_show_actions_menu (DATA_ENTRY (wid), FALSE);
	gda_value_free (subvalue);
	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);

	TS_EXTENSION (dd)->time = wid;
	TS_EXTENSION (dd)->time_handler = handler;

	/* widget update */
	widget_update_ts (dh, GTK_WIDGET (dd), value, TRUE);

	g_signal_connect (G_OBJECT (TS_EXTENSION (dd)->date), "contents_modified",
			  G_CALLBACK (contents_changed_cb), dd);
	g_signal_connect (G_OBJECT (TS_EXTENSION (dd)->time), "contents_modified",
			  G_CALLBACK (contents_changed_cb), dd);

	return GTK_WIDGET (dd);
}


static GdaValue *
value_from_widget_ts (DataHandler *dh, GtkWidget * wid)
{
	GdaValue *value;
	gboolean set;
	gboolean isnull = FALSE;

	g_return_val_if_fail (wid && IS_DATA_ENTRY (wid), NULL);

	set = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "is_set"));

	if (!set)
		value = gda_value_new_null ();
	else {
		GdaTimestamp tims;
		const GdaDate *date;
		const GdaTime *tim;
		GdaValue *subvalue;

		tims.fraction = TS_EXTENSION (wid)->fraction;

		subvalue = data_handler_get_value_from_widget (TS_EXTENSION (wid)->date_handler,
							       TS_EXTENSION (wid)->date);
		if (subvalue && !gda_value_is_null (subvalue)) {
			date = gda_value_get_date (subvalue);
			tims.year = date->year;
			tims.month = date->month;
			tims.day = date->day;
		}
		else
			isnull = TRUE;
		if (subvalue)
			gda_value_free (subvalue);

		subvalue = data_handler_get_value_from_widget (TS_EXTENSION (wid)->time_handler,
							       TS_EXTENSION (wid)->time);
		if (subvalue && !gda_value_is_null (subvalue)) {
			tim = gda_value_get_time (subvalue);
			tims.hour = tim->hour;
			tims.minute = tim->minute;
			tims.second = tim->second;
			tims.timezone = tim->timezone;
		}
		else
			isnull = TRUE;
		if (subvalue)
			gda_value_free (subvalue);

		if (isnull)
			value = gda_value_new_null ();
		else
			value = gda_value_new_timestamp (&tims);
	}

	return value;
}



static void
widget_update_ts (DataHandler *dh, GtkWidget * dd, const GdaValue * value, gboolean value_as_default)
{
	if (value_as_default) 
		data_entry_set_orig_value (DATA_ENTRY (dd), value);

	g_signal_handlers_block_by_func (G_OBJECT (TS_EXTENSION (dd)->date),
					 G_CALLBACK (contents_changed_cb), dd);
	g_signal_handlers_block_by_func (G_OBJECT (TS_EXTENSION (dd)->time),
					 G_CALLBACK (contents_changed_cb), dd);

	if (value && !gda_value_is_null (value)) {
		const GdaTimestamp *tims;
		GdaDate date;
		GdaTime tim;
		GdaValue *subvalue;

		tims = gda_value_get_timestamp (value);
		if (value_as_default)
			TS_EXTENSION (dd)->fraction = tims->fraction;

		/* date part */
		date.year = tims->year;
		date.month = tims->month;
		date.day = tims->day;
		subvalue = gda_value_new_date (&date);
		data_handler_widget_update (TS_EXTENSION (dd)->date_handler, TS_EXTENSION (dd)->date,
					    subvalue, value_as_default);
		gda_value_free (subvalue);

		/* time part */
		tim.hour = tims->hour;
		tim.minute = tims->minute;
		tim.second = tims->second;
		tim.timezone = tims->timezone;
		subvalue = gda_value_new_time (&tim);
		data_handler_widget_update (TS_EXTENSION (dd)->time_handler, TS_EXTENSION (dd)->time,
					    subvalue, value_as_default);
		gda_value_free (subvalue);

		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	}
	else {
		GdaValue *subvalue;

		subvalue = gda_value_new_null ();
		data_handler_widget_update (TS_EXTENSION (dd)->date_handler, TS_EXTENSION (dd)->date,
					    subvalue, value_as_default);
		data_handler_widget_update (TS_EXTENSION (dd)->time_handler, TS_EXTENSION (dd)->time,
					    subvalue, value_as_default);
		gda_value_free (subvalue);
		

		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (FALSE));
	}

	emit_contents_modified (G_OBJECT (TS_EXTENSION (dd)->date), dd);

	g_signal_handlers_unblock_by_func (G_OBJECT (TS_EXTENSION (dd)->date),
					   G_CALLBACK (contents_changed_cb), dd);
	g_signal_handlers_unblock_by_func (G_OBJECT (TS_EXTENSION (dd)->time),
					   G_CALLBACK (contents_changed_cb), dd);
}

gchar *strip_date_time_quotes (gchar *str);
static gchar *
sql_from_value_ts (DataHandler *dh, const GdaValue * value)
{
	gchar *str, *strdate, *strtime;
	gchar *s_strdate, *s_strtime;
	const GdaTimestamp *tims;
	GdaDate date;
	GdaTime tim;
	GdaValue *subvalue;
	DataHandler *handler;

	if (!value || gda_value_is_null (value))
		return NULL;

	tims = gda_value_get_timestamp (value);
	/* date part */
	handler = server_access_get_handler_from_gda (dh->priv->srv, GDA_VALUE_TYPE_DATE);
	date.year = tims->year;
	date.month = tims->month;
	date.day = tims->day;
	subvalue = gda_value_new_date (&date);
	strdate = data_handler_get_sql_from_value (handler, subvalue);
	gda_value_free (subvalue);
	
	/* time part */
	handler = server_access_get_handler_from_gda (dh->priv->srv, GDA_VALUE_TYPE_TIME);
	tim.hour = tims->hour;
	tim.minute = tims->minute;
	tim.second = tims->second;
	tim.timezone = tims->timezone;
	subvalue = gda_value_new_time (&tim);
	strtime = data_handler_get_sql_from_value (handler, subvalue);

	s_strdate = strip_date_time_quotes (strdate);
	s_strtime = strip_date_time_quotes (strtime);
	str = g_strdup_printf ("'%s %s'", s_strdate, s_strtime);
	g_free (strdate);
	g_free (strtime);

	return str;
}

gchar *
strip_date_time_quotes (gchar *str)
{
	gchar *ptr = str;

	if (*str == '\'') {
		ptr++;
	}
	
	if (*(str+(strlen (str)-1)) == '\'') {
		*(str+(strlen (str)-1)) = 0;
	}

	return ptr;
}

static gchar *
str_from_value_ts (DataHandler *dh, const GdaValue * value)
{
	return server_resultset_stringify (value);
}


/***************************************************************************/
/*                                                                         */
/* Some usefull functions                                                  */
/*                                                                         */
/***************************************************************************/


/* the returned char is allocated and the one in argument freed */
static gchar *
server_access_escape_chars (gchar * str)
{
	gchar *ptr = str, *ret, *retptr;
	gint size;

	/* determination of the new string size */
	size = 1;
	while (*ptr != '\0') {
		if (*ptr == '\'') {
			if (ptr == str)
				size += 2;
			else {
				if (*(ptr - 1) == '\\')
					size += 1;
				else
					size += 2;
			}
		}
		else
			size += 1;
		ptr++;
	}

	ptr = str;
	ret = (gchar *) malloc (sizeof (gchar) * size);
	retptr = ret;
	while (*ptr != '\0') {
		if (*ptr == '\'') {
			if (ptr == str) {
				*retptr = '\\';
				retptr++;
			}
			else if (*(ptr - 1) != '\\') {
				*retptr = '\\';
				retptr++;
			}
		}
		*retptr = *ptr;
		retptr++;
		ptr++;
	}
	*retptr = '\0';
	g_free (str);
	return ret;
}

/* the returned char is allocated and the one in argument freed */
static gchar *
server_access_unescape_chars (gchar * str)
{
	gchar *ptr = str, *ret, *retptr;
	gint size;

	/* determination of the new string size */
	size = 1;
	while (*ptr != '\0') {
		if ((*ptr == '\\') && (*(ptr + 1) == '\'')) {
			ptr++;
		}
		else {
			size += 1;
			ptr++;
		}
	}

	ptr = str;
	ret = (gchar *) malloc (sizeof (gchar) * size);
	retptr = ret;
	while (*ptr != '\0') {
		if ((*ptr == '\\') && (*(ptr + 1) == '\''))
			ptr++;
		else {
			*retptr = *ptr;
			retptr++;
			ptr++;
		}
	}
	*retptr = '\0';
	g_free (str);
	return ret;
}


GSList *
data_handlers_get_initial_list (ServerAccess *srv)
{
	GSList *list = NULL;
	DataHandler *dh;

	/* 
	 * WARNING: The order here is important, because in the 
	 * sqldata::get_display_fns_from_gda(), the returned value is the first one
	 * which matches the gda type of the given ServerDataType.
	 */

	/* FIXME: the other ones here (only builting ones, not plugins) */

	/* 1 - numerical values */
	dh = DATA_HANDLER (data_handler_new ());
	dh->priv->descr = _("Numerical data representation");
	dh->priv->plugin_name = NULL;
	dh->priv->plugin_file = NULL;
	dh->priv->lib_handle = NULL;
	dh->priv->version = NULL;
	dh->priv->get_unique_key = NULL;
	dh->priv->nb_gda_type = 7;
	dh->priv->valid_gda_types = g_new0 (GdaValueType, 7);
	dh->priv->valid_gda_types[0] = GDA_VALUE_TYPE_BIGINT;
	dh->priv->valid_gda_types[1] = GDA_VALUE_TYPE_DOUBLE;
	dh->priv->valid_gda_types[2] = GDA_VALUE_TYPE_INTEGER;
	dh->priv->valid_gda_types[3] = GDA_VALUE_TYPE_NUMERIC;
	dh->priv->valid_gda_types[4] = GDA_VALUE_TYPE_SINGLE;
	dh->priv->valid_gda_types[5] = GDA_VALUE_TYPE_SMALLINT;
	dh->priv->valid_gda_types[6] = GDA_VALUE_TYPE_TINYINT;
	dh->priv->expand_widget = FALSE;
	dh->priv->sql_from_value = sql_from_value_number;
	dh->priv->str_from_value = str_from_value_number;
	dh->priv->widget_from_value = widget_from_value_number;
	dh->priv->value_from_widget = value_from_widget_number;
	dh->priv->widget_update = widget_update_number;
	dh->priv->srv = srv;
	list = g_slist_append (list, dh);

	/* 2 - Boolean values */
	dh = DATA_HANDLER (data_handler_new ());
	dh->priv->descr = _("Boolean data representation");
	dh->priv->plugin_name = NULL;
	dh->priv->plugin_file = NULL;
	dh->priv->lib_handle = NULL;
	dh->priv->version = NULL;
	dh->priv->get_unique_key = NULL;
	dh->priv->nb_gda_type = 1;
	dh->priv->valid_gda_types = g_new0 (GdaValueType, 1);
	dh->priv->valid_gda_types[0] = GDA_VALUE_TYPE_BOOLEAN;
	dh->priv->expand_widget = FALSE;
	dh->priv->sql_from_value = sql_from_value_bool;
	dh->priv->str_from_value = str_from_value_bool;
	dh->priv->widget_from_value = widget_from_value_bool;
	dh->priv->value_from_widget = value_from_widget_bool;
	dh->priv->widget_update = widget_update_bool;
	dh->priv->srv = srv;
	list = g_slist_append (list, dh);

	/* 3 - Dates values */
	dh = DATA_HANDLER (data_handler_new ());
	dh->priv->descr = _("Dates representation");
	dh->priv->plugin_name = NULL;
	dh->priv->plugin_file = NULL;
	dh->priv->lib_handle = NULL;
	dh->priv->version = NULL;
	dh->priv->get_unique_key = NULL;
	dh->priv->nb_gda_type = 1;
	dh->priv->valid_gda_types = g_new0 (GdaValueType, 1);
	dh->priv->valid_gda_types[0] = GDA_VALUE_TYPE_DATE;
	dh->priv->expand_widget = FALSE;
	dh->priv->sql_from_value = sql_from_value_date;
	dh->priv->str_from_value = str_from_value_date;
	dh->priv->widget_from_value = widget_from_value_date;
	dh->priv->value_from_widget = value_from_widget_date;
	dh->priv->widget_update = widget_update_date;
	dh->priv->srv = srv;
	list = g_slist_append (list, dh);

	/* 4 - Times values */
	dh = DATA_HANDLER (data_handler_new ());
	dh->priv->descr = _("Dates representation");
	dh->priv->plugin_name = NULL;
	dh->priv->plugin_file = NULL;
	dh->priv->lib_handle = NULL;
	dh->priv->version = NULL;
	dh->priv->get_unique_key = NULL;
	dh->priv->nb_gda_type = 1;
	dh->priv->valid_gda_types = g_new0 (GdaValueType, 1);
	dh->priv->valid_gda_types[0] = GDA_VALUE_TYPE_TIME;
	dh->priv->expand_widget = FALSE;
	dh->priv->sql_from_value = sql_from_value_time;
	dh->priv->str_from_value = str_from_value_time;
	dh->priv->widget_from_value = widget_from_value_time;
	dh->priv->value_from_widget = value_from_widget_time;
	dh->priv->widget_update = widget_update_time;
	dh->priv->srv = srv;
	list = g_slist_append (list, dh);

	/* 5 - Times Stamp values */
	dh = DATA_HANDLER (data_handler_new ());
	dh->priv->descr = _("TimeStamp representation");
	dh->priv->plugin_name = NULL;
	dh->priv->plugin_file = NULL;
	dh->priv->lib_handle = NULL;
	dh->priv->version = NULL;
	dh->priv->get_unique_key = NULL;
	dh->priv->nb_gda_type = 1;
	dh->priv->valid_gda_types = g_new0 (GdaValueType, 1);
	dh->priv->valid_gda_types[0] = GDA_VALUE_TYPE_TIMESTAMP;
	dh->priv->expand_widget = FALSE;
	dh->priv->sql_from_value = sql_from_value_ts;
	dh->priv->str_from_value = str_from_value_ts;
	dh->priv->widget_from_value = widget_from_value_ts;
	dh->priv->value_from_widget = value_from_widget_ts;
	dh->priv->widget_update = widget_update_ts;
	dh->priv->srv = srv;
	list = g_slist_append (list, dh);


	/* 6 - String values */
	dh = DATA_HANDLER (data_handler_new ());
	dh->priv->descr = _("Strings representation");
	dh->priv->plugin_name = NULL;
	dh->priv->plugin_file = NULL;
	dh->priv->lib_handle = NULL;
	dh->priv->version = NULL;
	dh->priv->get_unique_key = NULL;
	dh->priv->nb_gda_type = 1;
	dh->priv->valid_gda_types = g_new0 (GdaValueType, 1);
	dh->priv->valid_gda_types[0] = GDA_VALUE_TYPE_STRING;
	dh->priv->expand_widget = FALSE;
	dh->priv->sql_from_value = sql_from_value_string;
	dh->priv->str_from_value = str_from_value_string;
	dh->priv->widget_from_value = widget_from_value_string;
	dh->priv->value_from_widget = value_from_widget_string;
	dh->priv->widget_update = widget_update_string;
	dh->priv->srv = srv;
	list = g_slist_append (list, dh);

	/* LAST - default values */
	dh = DATA_HANDLER (data_handler_new ());
	dh->priv->descr = _("Default data representation (read only)");
	dh->priv->plugin_name = NULL;
	dh->priv->plugin_file = NULL;
	dh->priv->lib_handle = NULL;
	dh->priv->version = NULL;
	dh->priv->get_unique_key = NULL;
	dh->priv->nb_gda_type = 0;	/* means ALL GDA types */
	dh->priv->valid_gda_types = NULL;	/* means ALL GDA types */
	dh->priv->expand_widget = FALSE;
	dh->priv->sql_from_value = sql_from_value_default;
	dh->priv->str_from_value = str_from_value_default;
	dh->priv->widget_from_value = widget_from_value_default;
	dh->priv->value_from_widget = value_from_widget_default;
	dh->priv->widget_update = widget_update_default;
	dh->priv->srv = srv;
	list = g_slist_append (list, dh);

	return list;
}

