/* object-selector.c
 *
 * Copyright (C) 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdarg.h>

#include <config.h>
#include "object-selector-ext.h"
#include "object-selector-priv.h"
#include "marshal.h"
#include "database.h"
#include "server-access.h"
#include "query.h"
#include "query-env.h"

static void object_selector_class_init (ObjectSelectorClass * class);
static void object_selector_init (ObjectSelector * wid);
static void object_selector_dispose (GObject   * object);

/* to store in the model */
typedef struct _TreeItem {
	gchar                *name;
	gchar                *cmp_name;
	gchar                *comment;
	GObject              *obj;
	const ManagerInfo    *manager_info;
	GdkPixbuf            *pixbuf;
} TreeItem;


/* columns names for the GtkTreeView */
enum {
	NAME_COLUMN,
	COMMENT_COLUMN,
	REF_COLUMN,
	INFO_COLUMN,
	PIXBUF_COLUMN,
	NUM_COLUMN,
};



enum
{
	C_SELECTION_CHANGED,
	LAST_SIGNAL
};

static gint object_selector_signals[LAST_SIGNAL] = { 0 };

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;


guint
object_selector_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (ObjectSelectorClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) object_selector_class_init,
			NULL,
			NULL,
			sizeof (ObjectSelector),
			0,
			(GInstanceInitFunc) object_selector_init
		};		
		
		type = g_type_register_static (GTK_TYPE_VBOX, "ObjectSelector", &info, 0);
	}

	return type;
}

static void
object_selector_class_init (ObjectSelectorClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	object_selector_signals[C_SELECTION_CHANGED] =
		g_signal_new ("selection_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (ObjectSelectorClass, selection_changed),
			      NULL, NULL,
			      marshal_VOID__OBJECT, G_TYPE_NONE, 1,
			      G_TYPE_OBJECT);

	class->selection_changed = NULL;
	object_class->dispose = object_selector_dispose;
}


/*
 * Defines a new relation between a kind of object ('manager_object') and objects of kind 'child_info'
 * which, once created are reported by the manager object. Their destruction is also reported.
 */
void 
object_selector_add_manager_class (ObjectSelector * os,
				   ObjectDescriptorMode ifmode,
				   const ObjectInfo *manager_info,
				   const gchar *child_creation_signal_name,
				   const gchar *child_destroy_signal_name,
				   const ObjectInfo *child_info)
{
	InnerObjectLogic *logic;
	logic = g_new0 (InnerObjectLogic, 1);
	
	logic->ifmode = ifmode;
	logic->manager_info = object_selector_find_object_info (os, manager_info);
	logic->child_creation_signal_name = g_strdup (child_creation_signal_name);
	logic->child_destroy_signal_name = g_strdup (child_destroy_signal_name);
	logic->child_info = object_selector_find_object_info (os, child_info);

	os->priv->inner_logic = g_slist_append (os->priv->inner_logic, logic);
}

static void
object_selector_init (ObjectSelector * wid)
{
	wid->priv = g_new0 (ObjectSelectorPriv, 1);
	wid->priv->conf = NULL;
	wid->priv->mode = 0;
	wid->priv->treeview = NULL;
	wid->priv->selection = NULL;
	wid->priv->objects_infos = NULL;
	wid->priv->weak_ref_objects = NULL;
	wid->priv->inner_logic = NULL;
}

static void conn_to_close (ServerAccess *srv, ObjectSelector *os);
static void object_selector_initialize (ObjectSelector *os, va_list args);
GtkWidget *
object_selector_new (ConfManager *conf, ObjectDescriptorMode mode, ...)
{
	va_list args;
	GObject   *obj;
	ObjectSelector *os;

	g_return_val_if_fail (conf && IS_CONF_MANAGER (conf), NULL);

	obj = g_object_new (OBJECT_SELECTOR_TYPE, NULL);
	os = OBJECT_SELECTOR (obj);

	os->priv->conf = conf;
	os->priv->mode = mode;

	va_start (args, mode);
	object_selector_initialize (os, args);
	va_end (args);
	/* fetch the "conn_to_close" signal to purge everything */
	g_signal_connect (G_OBJECT (conf->srv), "conn_to_close",
			  G_CALLBACK (conn_to_close), os);

	return GTK_WIDGET (obj);
}

static void weak_ref_object_not (ObjectSelector *os, GObject *obj);
static void 
conn_to_close (ServerAccess *srv, ObjectSelector *os)
{
	if (os->priv) {
		GSList *list;
		GtkTreeModel *model;
		
		model = gtk_tree_view_get_model (os->priv->treeview);
		gtk_tree_store_clear (GTK_TREE_STORE (model));

		/* managers infos */
		list = os->priv->managers_infos;
		while (list) {
			GSList *list2 = MANAGER_INFO (list->data)->signals;

			while (list2) {
				g_signal_handler_disconnect (MANAGER_INFO (list->data)->manager, 
							     MANAGER_SIGNAL (list2->data)->sigid);
				g_free (list2->data);
				list2 = g_slist_next (list2);
			}
			
			g_slist_free (MANAGER_INFO (list->data)->signals);
			g_slist_free (MANAGER_INFO (list->data)->children_info);
			
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (os->priv->managers_infos);
		
		/* weak ref objects */
		list = os->priv->weak_ref_objects;
		while (list) {
			g_object_weak_unref (G_OBJECT (list->data), (GWeakNotify) weak_ref_object_not, os);
			list = g_slist_next (list);
		}
		g_slist_free (os->priv->weak_ref_objects);
		os->priv->weak_ref_objects = NULL;
	}
	else
		g_warning ("ObjectSelector::conn_to_close() os->priv is NULL!\n");
}

static void
object_selector_dispose (GObject   * object)
{
	ObjectSelector *os;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_OBJECT_SELECTOR (object));
	os = OBJECT_SELECTOR (object);

	if (os->priv) {
		GSList *list;
		/* signals, etc */
		conn_to_close (NULL, os);
		
		/* disconnect the "conn_to_close" signal */
		g_signal_handlers_disconnect_by_func (G_OBJECT (os->priv->conf->srv), G_CALLBACK (conn_to_close), os);
		
		/* logic */
		list = os->priv->inner_logic;
		while (list) {
			g_free (INNER_OBJECT_LOGIC (list->data)->child_creation_signal_name);
			g_free (INNER_OBJECT_LOGIC (list->data)->child_destroy_signal_name);
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (os->priv->inner_logic);

		/* objects infos */
		list = os->priv->objects_infos;
		while (list) {
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (os->priv->objects_infos);

		/* the private area itself */
		g_free (os->priv);
		os->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}



static void tree_selection_changed_cb (GtkTreeSelection *select, ObjectSelector *os);
static void tree_value_set_func (GtkTreeViewColumn *tree_column,
				 GtkCellRenderer *cell,
				 GtkTreeModel *model,
				 GtkTreeIter *iter,
				 gpointer user_data);
/*
 * create the model, the columns and the treeview widget itself
 */
static void 
object_selector_initialize (ObjectSelector *os, va_list args)
{
	GtkTreeModel *model;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTreeSelection *select;
	GtkWidget *sw, *treeview;
	gint n_top_levels;
	gint param;

	/* parameters */
	Query *top_query = NULL;
	gboolean sub_queries = TRUE;

	while ((param = va_arg (args, gint)) != -1) {
		switch (param) {
		case OBJECT_SELECTOR_PARAM_TOP_QUERY:
			top_query = va_arg (args, gpointer);
			if (!IS_QUERY (top_query))
				g_warning ("object_selector_initialize(): TOP_QUERY param is not a Query\n");
			break;
		case OBJECT_SELECTOR_PARAM_SUB_QUERIES:
			sub_queries = va_arg (args, gboolean);
			break;
		default:
			g_warning ("object_selector_initialize(): undefined parameter\n");
			break;
		}
	}

	gtk_container_set_border_width (GTK_CONTAINER (os), 0);

	/* model creation */
	model = GTK_TREE_MODEL (gtk_tree_store_new (NUM_COLUMN, G_TYPE_STRING, G_TYPE_STRING, 
						    G_TYPE_POINTER, G_TYPE_POINTER, GDK_TYPE_PIXBUF));

	/* widget */
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start (GTK_BOX (os), sw, TRUE, TRUE, 0);
	treeview = gtk_tree_view_new_with_model (model);
	g_object_unref (G_OBJECT (model));
	gtk_container_add (GTK_CONTAINER (sw), treeview);
	os->priv->treeview = GTK_TREE_VIEW (treeview);

	gtk_widget_set_size_request (sw, 100, 150);
        gtk_widget_show_all (sw); 

	/* calculates if there is more than one top level entry, and if that is
	   the case, don't display the "manager" */
	n_top_levels = 0;
	n_top_levels += ((os->priv->mode & OBJECT_SELECTOR_TABLES) ||
			 (os->priv->mode & OBJECT_SELECTOR_VIEWS)) ? 1 : 0;
	n_top_levels += os->priv->mode & OBJECT_SELECTOR_SEQUENCES ? 1 : 0;
	n_top_levels += os->priv->mode & OBJECT_SELECTOR_DATA_TYPES ? 1 : 0;
	n_top_levels += os->priv->mode & OBJECT_SELECTOR_PROCS ? 1 : 0;
	n_top_levels += os->priv->mode & OBJECT_SELECTOR_AGGS ? 1 : 0;
	n_top_levels += os->priv->mode & OBJECT_SELECTOR_QUERIES ? 1 : 0;


	/* model population */
	object_selector_create_model_tables_views (os, n_top_levels, NULL);
	object_selector_create_model_sequences (os, n_top_levels, NULL);
	object_selector_create_model_queries (os, n_top_levels, NULL, top_query, sub_queries);
	object_selector_create_model_datatypes (os, n_top_levels, NULL);
	object_selector_create_model_functions (os, n_top_levels, NULL);
	object_selector_create_model_aggregates (os, n_top_levels, NULL);

	/* adding columns */
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Name"));
	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	renderer = gtk_cell_renderer_text_new ();
        gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
                                                 renderer,
                                                 tree_value_set_func,
                                                 NULL, NULL);

	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Comments"), renderer,
							   "text", COMMENT_COLUMN, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	/* selection signal */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
	g_signal_connect (G_OBJECT (select), "changed",
			  G_CALLBACK (tree_selection_changed_cb), os);

}

static void
tree_value_set_func (GtkTreeViewColumn *tree_column,
		     GtkCellRenderer *cell,
		     GtkTreeModel *model,
		     GtkTreeIter *iter,
		     gpointer user_data)
{
        GdkPixbuf *pixbuf;
        gchar *text;
        GList *renderers;
	
        gtk_tree_model_get (model, iter, PIXBUF_COLUMN, &pixbuf, NAME_COLUMN, &text, -1);
        renderers = gtk_tree_view_column_get_cell_renderers (tree_column);
        cell = renderers->data;
	g_object_set (G_OBJECT (cell), "pixbuf", pixbuf, NULL);
        cell = renderers->next->data;
        g_object_set (G_OBJECT (cell), "text", text, NULL);
	g_free (text);
        g_list_free (renderers);
}


static void
tree_selection_changed_cb (GtkTreeSelection *select, ObjectSelector *os) 
{
	GtkTreeIter iter;
	GtkTreeModel *model;

	if (gtk_tree_selection_get_selected (select, &model, &iter)) {
		GObject *sel_obj;

		gtk_tree_model_get (model, &iter, REF_COLUMN, &sel_obj, -1);
		g_signal_emit (G_OBJECT (os), object_selector_signals[C_SELECTION_CHANGED], 0, sel_obj);
	}
}


/*
 * Returns a pointer to the OS's managed copy of the ObjectInfo equal to 'info'.
 * If it does not exist, then it is copied, added to the 'objects_infos' list and returned
 */
ObjectInfo  *
object_selector_find_object_info (ObjectSelector *os, const ObjectInfo *info)
{
	GSList *list;
	ObjectInfo *retval = NULL;

	g_return_val_if_fail (os && IS_OBJECT_SELECTOR (os), NULL);
	g_return_val_if_fail (info, NULL);

	list = os->priv->objects_infos;
	while (list && !retval) {
		if ((OBJECT_INFO (list->data)->type == info->type) && 
		    (OBJECT_INFO (list->data)->get_icon == info->get_icon) &&
		    (OBJECT_INFO (list->data)->get_name == info->get_name) && 
		    (OBJECT_INFO (list->data)->get_cmp_name == info->get_cmp_name) &&
		    (OBJECT_INFO (list->data)->get_cols == info->get_cols)) 
			retval = OBJECT_INFO (list->data);
		list = g_slist_next (list);
	}

	if (!retval) {
		ObjectInfo *oi;

		oi = g_new0 (ObjectInfo, 1);
		*oi = *info;
		os->priv->objects_infos = g_slist_append (os->priv->objects_infos, oi);
		retval = oi;
	}

	return retval;
}

/* FIXME: sub categories */
/*
 * Returns the path to an object as an allocated string, or NULL
 * if the object has not been found
 */
static GtkTreePath *
find_model_path_real (GtkTreeModel *model, GObject *obj, 
		      GType children_type, GtkTreeIter *start_iter)
{
	GtkTreePath * path = NULL;
	GtkTreeIter iter;
	GObject *value;
	ManagerInfo *minfo;
	gboolean res;

	g_return_val_if_fail (obj && G_IS_OBJECT (obj), NULL);

	/* get a valid iterator */
	if (!start_iter) {
		res = gtk_tree_model_get_iter_first (model, &iter);
		if (!res)
			return NULL;
	}
	else
		iter = *start_iter;
	
	/* looking at the node itself */
	gtk_tree_model_get (model, &iter, REF_COLUMN, &value, INFO_COLUMN, &minfo, -1);
	if (value == obj) {
		if (children_type == G_TYPE_INVALID)
			path = gtk_tree_model_get_path (model, &iter);
		else {
			if (minfo) {
				GSList *list = minfo->children_info;

				while (list && !path) {
					if (OBJECT_INFO (list->data)->type == children_type)
						path = gtk_tree_model_get_path (model, &iter);
					list = g_slist_next (list);
				}
			}
		}
	}

	/* looking at the next (sibling) nodes */
	if (!path) {
		GtkTreeIter next_iter = iter;
		res = gtk_tree_model_iter_next (model, &next_iter);
		if (res)
			path = find_model_path_real (model, obj, children_type, &next_iter);
	}

	/* looking at the sub nodes */
	if (!path && gtk_tree_model_iter_has_child (model, &iter)) {
		GtkTreeIter child_iter;

		res = gtk_tree_model_iter_children (model, &child_iter, &iter);
		if (res)
		    path = find_model_path_real (model, obj, children_type, &child_iter);
	}

	return path;
}


GtkTreePath *
object_selector_find_model_path (GtkTreeModel *model, GObject *obj, GType children_type)
{
	return find_model_path_real (model, obj, children_type, NULL);
}


static void remove_object (ObjectSelector *os, GObject *object, gboolean unweakref);
/*
 * removes the object from the weak references list,
 * updates the model (the object's node and its sub nodes recursively)
*/
static void
weak_ref_object_not (ObjectSelector *os, GObject *obj)
{
	GSList *list;
	
	/* remove object from weak reference objects list */
	list = g_slist_find (os->priv->weak_ref_objects, obj);
	if (list) {
		os->priv->weak_ref_objects = g_slist_delete_link (os->priv->weak_ref_objects, list);
	}
	else 
		g_warning ("weak_ref_object_not(): can't find object");

	/* remove object from the treeview if it was present */
	remove_object (os, obj, FALSE);
	g_print ("Object removed: %p\n", obj);
}



/* Removes all the signal handlers (and NOT weak references) that 
 * can exist on the object pointer by the iterator,
 * and recursivaly in the sub tree and siblings (if no_sibling is FALSE)
 * Also frees memory associated with the ObjectAttributes structure
 * of each node
 */
static void
remove_managers (ObjectSelector *os, GtkTreeIter *iter, gboolean no_sibling) 
{
	GtkTreeModel *model;
	GObject *obj;
	ManagerInfo *minfo;
	gboolean res;
	GSList *list;

	model = gtk_tree_view_get_model (os->priv->treeview);
	gtk_tree_model_get (model, iter, REF_COLUMN, &obj, INFO_COLUMN, &minfo, -1);

	/* free the associated ManagerInfo */
	if (minfo) {
		/* signals connected to the object */
		list = minfo->signals;
		while (list) {
			ManagerSignal *sig = MANAGER_SIGNAL (list->data);

			g_signal_handler_disconnect (obj, sig->sigid);
			g_free (sig);
		       
			list = g_slist_next (list);
		}
		g_slist_free (minfo->signals);
		minfo->signals = NULL;
		
		/* free the info */
		os->priv->managers_infos = g_slist_remove (os->priv->managers_infos, minfo);
		g_free (minfo);
		gtk_tree_store_set (GTK_TREE_STORE (model), iter,
				    INFO_COLUMN, NULL, -1);
	}
	

	/* recursive call to the next (sibling) nodes */
	if (!no_sibling) {
		GtkTreeIter next_iter = *iter;
		res = gtk_tree_model_iter_next (model, &next_iter);
		if (res)
			remove_managers (os, &next_iter, FALSE);
	}

	/* recursive call to the sub nodes */
	if (gtk_tree_model_iter_has_child (model, iter)) {
		GtkTreeIter child_iter;

		res = gtk_tree_model_iter_children (model, &child_iter, iter);
		if (res)
			remove_managers (os, &child_iter, FALSE);
	}	
}



/*
 * Called to update the model by removing
 * the object given as argument. the bool tells it the object must explicitely
 * unref'd and removed from the weak ref list as well (not the case if we already have received
 * the weak_unref "signal").
 */
static void
remove_object (ObjectSelector *os, GObject *object, gboolean unweakref)
{
	GtkTreeModel *model;
	GtkTreePath *path;

	if (unweakref) {
		GSList *list;
		list = g_slist_find (os->priv->weak_ref_objects, object);
		if (list)
			os->priv->weak_ref_objects = g_slist_delete_link (os->priv->weak_ref_objects, list);
	}

	/* model update */
	model = gtk_tree_view_get_model (os->priv->treeview);
	path = object_selector_find_model_path (model, object, G_TYPE_INVALID);
	while (path) { /* we want to get ALL the objects in the tree */
		GtkTreeIter iter;

		/* gchar *str = gtk_tree_path_to_string (path); */
/* 		g_print ("REMOVED %s\n", str); */
/* 		g_free (str); */
		
		if (gtk_tree_model_get_iter (model, &iter, path)) {
			remove_managers (os, &iter, TRUE);
			gtk_tree_store_remove (GTK_TREE_STORE (gtk_tree_view_get_model (os->priv->treeview)),
					       &iter);
		}			
		gtk_tree_path_free (path);
		
		path = object_selector_find_model_path (model, object, G_TYPE_INVALID);
	}
}


static void object_created_cb (GObject *obj, GObject *new_obj, ObjectSelector *os);
static void object_destroyed_cb (GObject *obj, GObject *old_obj, ObjectSelector *os);


/* 
 * Add a weak reference on the 'manager' object, and parse the inner logic to see if
 * any signals have been defined for that object in the logic.
 * Creates a ManagerInfo structure if the object can be a manager.
 */
void object_selector_add_manager (ObjectSelector * os, 
				  GObject *manager,
				  const ObjectInfo *manager_info)
{
	GSList *list;
	ObjectInfo *localinfo;
	ManagerInfo *minfo = NULL;

	g_return_if_fail (os && IS_OBJECT_SELECTOR (os));
	g_return_if_fail (manager && G_IS_OBJECT (manager));
	
	
	/* Find the correct logic */
	if (manager_info)
		localinfo = object_selector_find_object_info (os, manager_info);
	else
		localinfo = NULL;

	/* iterate through logic */
	list = os->priv->inner_logic;
	while (list) {
		InnerObjectLogic *logic = INNER_OBJECT_LOGIC (list->data);
		if ((os->priv->mode & logic->ifmode) &&
		    ((localinfo && (logic->manager_info == localinfo)) || 
		     (!localinfo && (logic->manager_info->type == G_OBJECT_TYPE (manager))))) {
			ManagerSignal *csig, *dsig;
			gboolean already_conn = FALSE;
		       
			/* add to the list of manager objects */
			if (!minfo) {
				minfo = g_new0 (ManagerInfo, 1);
				minfo->manager = manager;
				minfo->manager_info = logic->manager_info;
				minfo->children_info = NULL;
				minfo->signals = NULL;
			}

			/* make sure we are not yet connected with this signal */
			if (minfo->signals) {
				GSList *list2 = minfo->signals;

				while (list2 && !already_conn) {
					if (!strcmp (MANAGER_SIGNAL (list2->data)->signal_name, 
						     logic->child_creation_signal_name)) {
						already_conn = TRUE;
					}
					list2 = g_slist_next (list2);
				}
			}

			if (!already_conn) {
				/* connecting the signals */
				csig = g_new0 (ManagerSignal, 1);
				csig->sigid = g_signal_connect (G_OBJECT (manager), 
								logic->child_creation_signal_name,
								G_CALLBACK (object_created_cb), os);
				csig->signal_name = logic->child_creation_signal_name;
				csig->managed_object_type = logic->manager_info->type;
				
				dsig = g_new0 (ManagerSignal, 1);
				dsig->sigid = g_signal_connect (G_OBJECT (manager), 
								logic->child_destroy_signal_name,
								G_CALLBACK (object_destroyed_cb), os);
				dsig->signal_name = logic->child_destroy_signal_name;
				dsig->managed_object_type = logic->manager_info->type;
				
				minfo->children_info = g_slist_append (minfo->children_info, logic->child_info);
				minfo->signals = g_slist_append (minfo->signals, csig);
				minfo->signals = g_slist_append (minfo->signals, dsig);
			}
			
			/* weak reference on the object */
			if (!g_slist_find (os->priv->weak_ref_objects, manager)) {
				g_object_weak_ref (G_OBJECT (manager), (GWeakNotify) weak_ref_object_not, os);
				os->priv->weak_ref_objects = g_slist_append (os->priv->weak_ref_objects, manager);
			}

		}
		list = g_slist_next (list);
	}
	
	if (minfo) {
		GSList *list2 = minfo->signals;
		
		while (list2) {
			g_print ("Manager %p, signal: %s\n", manager, 
				 MANAGER_SIGNAL (list2->data)->signal_name);
			list2 = g_slist_next (list2);
		}
		os->priv->managers_infos = g_slist_append (os->priv->managers_infos, minfo);
		
	}
}

/*
 * Add an object in the treeview. If the object will be a manager itself, then
 * the function object_selector_add_manager() MUST be called before this one.
 */
void 
object_selector_add_object (ObjectSelector * os, 
			    GtkTreeIter *parent_iter, 
			    GObject *parent,
			    GObject *obj, ObjectInfo *oinfo,
			    GtkTreeIter *obj_iter, gboolean sort)
{
	GtkTreeModel *model;
	GtkTreeIter iter, newcat;
	TreeItem *item;
	GtkTreeIter before_iter, *sibling, *cat;
	const ObjectInfo *info = NULL;
	const ManagerInfo *minfo = NULL;
	GSList *list;

	g_return_if_fail (os && IS_OBJECT_SELECTOR (os));
	if (parent)
		g_return_if_fail (G_IS_OBJECT (parent));
	g_return_if_fail (obj && G_IS_OBJECT (obj));

	model = gtk_tree_view_get_model (os->priv->treeview);

	
	/* find the right ObjectInfo */
	if (!oinfo) {
		if (parent) {
			/* the parent, if present, MUST be a manager */
			list = os->priv->managers_infos;
			while (list && !info) {
				if (MANAGER_INFO (list->data)->manager == parent) {
					GSList *list2 = MANAGER_INFO (list->data)->children_info;
					while (list2 && !info) {
						if (OBJECT_INFO (list2->data)->type == G_OBJECT_TYPE (obj))
							info = OBJECT_INFO (list2->data);
						list2 = g_slist_next (list2);
					}
				}
				list = g_slist_next (list);
			}
		}
		else {
			/* try to find any ObjectInfo */
			list = os->priv->objects_infos;
			while (list && !info) {
				if (OBJECT_INFO (list->data)->type == G_OBJECT_TYPE (obj))
					info = OBJECT_INFO (list->data);
				list = g_slist_next (list);
			}
			g_message ("object_selector_add_object(): using a default ObjectInfo!");
		}

		if (!info) {
			g_warning ("object_selector_add_object(): can't find information for object!");
			return;
		}
	}
	else
		info = object_selector_find_object_info (os, oinfo);

	/* find the right info for potential children, if obj is a manager */
	list = os->priv->managers_infos;
	while (list && !minfo) {
		if ((MANAGER_INFO (list->data)->manager == obj) &&
		    (MANAGER_INFO (list->data)->manager_info == info))
			minfo = MANAGER_INFO (list->data);
		list = g_slist_next (list);
	}
	
	/* creating the tree item */
	item = g_new0 (TreeItem, 1);
	if (info->get_name) 
		item->name = (info->get_name)(obj);
	else 
		item->name = g_strdup ("");
	if (info->get_cmp_name)
		item->cmp_name = (info->get_cmp_name)(obj);
	else 
		item->cmp_name = g_strdup ("");
	if (info->get_cols) {
		list = (info->get_cols)(obj, 1);
		item->comment = list->data;
		g_slist_free (list);
	}
	else 
		item->comment = g_strdup ("");
	item->obj = obj;
	item->manager_info = minfo;
	if (info->get_icon) 
		item->pixbuf = (info->get_icon)(obj);

	/* finding where to insert the item (alphabetical order) */
	sibling = NULL;
	if (sort) {
		if (parent_iter && gtk_tree_model_iter_children (model, &before_iter, parent_iter)) {
			gchar *str;
			gboolean res = TRUE;
			
			while (res && !sibling) {
				gtk_tree_model_get (model, &before_iter, NAME_COLUMN, &str, -1);
				if (strcmp (item->name, str) < 0) 
					sibling = &before_iter;
				else
					res = gtk_tree_model_iter_next (model, &before_iter);
				g_free (str);
			}
		}
	}
	
	/* grouping of entries together */
	cat = sibling;
	if (!cat) {
		/* fetch the last item */
		gboolean res;
		gint nb;
		nb = gtk_tree_model_iter_n_children (model, parent_iter);
		
		if (nb > 0) {
			res = gtk_tree_model_iter_nth_child (model, &before_iter, parent_iter, nb-1); 
			if (res)
				cat = &before_iter;
		}
	}

	if (cat) {
		GObject *object;

		gtk_tree_model_get (model, cat, REF_COLUMN, &object, -1);
		if (! object ||
		    (obj && (G_OBJECT_TYPE (obj) == G_OBJECT_TYPE (object)))) {
			gchar *str, *itemstr;
			gchar *name;

			gtk_tree_model_get (model, cat, NAME_COLUMN, &name, -1);
			/* strings to decide if we need to make a sub category */
			if (info->get_cmp_name && object) 
				str = (*info->get_cmp_name) (object);
			else
				str = g_strdup (name);
			
			if (info->get_cmp_name) 
				itemstr = (*info->get_cmp_name) (item->obj);
			else
				itemstr = g_strdup (item->name);

			if (!strcmp (itemstr, str)) {
				gchar *comment;
				const ObjectInfo *tmpinfo;
				GdkPixbuf *pixbuf;
				
				/* Do we really need a sub category? */
				gtk_tree_model_get (model, cat, COMMENT_COLUMN, &comment, 
						    INFO_COLUMN, &tmpinfo, PIXBUF_COLUMN, &pixbuf, -1);
				if (object) {
					GtkTreeIter move;
					
					/* new item */
					gtk_tree_store_insert_after (GTK_TREE_STORE (model), 
								     &newcat, parent_iter, cat);
					gtk_tree_store_set (GTK_TREE_STORE (model), &newcat,
							    NAME_COLUMN, itemstr,
							    REF_COLUMN, NULL,
							    INFO_COLUMN, item->manager_info,
							    PIXBUF_COLUMN, item->pixbuf,
							    -1);
					
					/* "move" the *cat item below this new one */
					gtk_tree_store_append (GTK_TREE_STORE (model), &move, &newcat);
					gtk_tree_store_set (GTK_TREE_STORE (model), &move,
							    NAME_COLUMN, name,
							    COMMENT_COLUMN, comment,
							    REF_COLUMN, obj,
							    INFO_COLUMN, tmpinfo,
							    PIXBUF_COLUMN, pixbuf,
							    -1);
					remove_managers (os, cat, TRUE);
					gtk_tree_store_remove (GTK_TREE_STORE (model), cat);
					
					parent_iter = &newcat;
				}
				else
					parent_iter = cat;
				sibling = NULL;
			}
			g_free (str);
			g_free (itemstr);
		}
	}
		
	/* adding to the model */
	gtk_tree_store_insert_before (GTK_TREE_STORE (model), &iter, parent_iter, sibling);
	if (obj_iter)
		*obj_iter = iter;
	gtk_tree_store_set (GTK_TREE_STORE (model), &iter,
			    NAME_COLUMN, item->name,
			    COMMENT_COLUMN, item->comment,
			    REF_COLUMN, item->obj,
			    INFO_COLUMN, item->manager_info,
			    PIXBUF_COLUMN, item->pixbuf,
			    -1);

	/* free the item (don't free the attributes: attached to the model) */
	g_free (item->name);
	g_free (item->cmp_name);
	g_free (item->comment);
	g_free (item);
}

#ifdef debug
static void
print_path (GtkTreePath *path) {
	if (path) {
		gchar *str = gtk_tree_path_to_string (path);
		g_print ("PATH= %s\n", str);
		g_free (str);
	}
	else
		g_print ("PATH is NULL!\n");
}
#endif

static void
object_created_cb (GObject *obj, GObject *new_obj, ObjectSelector *os)
{
	GtkTreeModel *model;
	GtkTreePath *path;

	/* if the object should be a manager, then set it as a manager */
	object_selector_add_manager (os, new_obj, NULL);

	/* add the object in the tree */
	model = gtk_tree_view_get_model (os->priv->treeview);
	path = object_selector_find_model_path (model, obj, G_OBJECT_TYPE (new_obj));

	if (path) {
		GtkTreeIter iter;
		
		if (gtk_tree_model_get_iter (model, &iter, path)) 
			object_selector_add_object (os, &iter, obj, new_obj, NULL, NULL, TRUE);
		gtk_tree_path_free (path);
	}
	else 
		object_selector_add_object (os, NULL, obj, new_obj, NULL, NULL, TRUE);
}

static void
object_destroyed_cb (GObject *obj, GObject *old_obj, ObjectSelector *os)
{
	remove_object (os, old_obj, FALSE);
}


/* returns an allocated string containing the first part of a path
 * representing a GtkTreePath, depth = 1 for the first ":" 
 */
gchar *
get_cut_path_depth (const gchar *path, guint depth, gboolean with_last)
{
	gchar *str, *ptr;
	guint i = 0;
	
	str = g_strdup (path);
	ptr = str;
	while ((i < depth) && (*ptr != 0)) { 
		if (*ptr == ':')
			i++;

		if (i == depth)
			*ptr = 0;

		ptr++;
	}
	
	if (with_last && (*ptr == 0))
		i++;

	if (i != depth) {
		g_free (str);
		str = NULL;
	}

	return str;
}

void
object_selector_set_selection (ObjectSelector *os, GObject *selection)
{
	GtkTreeModel *model;
	GtkTreePath *path;

	g_return_if_fail (os && IS_OBJECT_SELECTOR (os));
	g_return_if_fail (selection && G_IS_OBJECT (selection));
	
	model = gtk_tree_view_get_model (os->priv->treeview);
	path = object_selector_find_model_path (model, selection, G_TYPE_INVALID);
	if (path) {
		gchar *strpath, *partpath;
		guint i = 1;
		GtkTreePath *ppath;
		GtkTreeSelection *selection;

		strpath = gtk_tree_path_to_string (path);
		partpath = get_cut_path_depth (strpath, i, FALSE);
		while (partpath) {
			ppath = gtk_tree_path_new_from_string (partpath);
			g_free (partpath);
			gtk_tree_view_expand_row (os->priv->treeview, ppath, FALSE);
			gtk_tree_path_free (ppath);

			i++;
			partpath = get_cut_path_depth (strpath, i, FALSE);
		}

		g_free (strpath);
		gtk_tree_view_scroll_to_cell (os->priv->treeview, path, NULL, TRUE,
					      0.5, 0.);
		selection = gtk_tree_view_get_selection (os->priv->treeview);
		gtk_tree_selection_unselect_all (selection);
		gtk_tree_selection_select_path (selection, path);
		gtk_tree_view_set_cursor (os->priv->treeview, path, NULL, FALSE);
		gtk_tree_path_free (path);
	}
}

void
object_selector_set_headers_visible (ObjectSelector *os, gboolean visible)
{
	g_return_if_fail (os && IS_OBJECT_SELECTOR (os));

	gtk_tree_view_set_headers_visible (os->priv->treeview, visible);
}

void
object_selector_set_comments_visible (ObjectSelector *os, gboolean visible)
{
	GtkTreeViewColumn *column;

	g_return_if_fail (os && IS_OBJECT_SELECTOR (os));

	column = gtk_tree_view_get_column (os->priv->treeview, 1);
	gtk_tree_view_column_set_visible (column, visible);
}

void
object_selector_set_column_label (ObjectSelector *os, guint column, const gchar *label)
{
	GtkTreeViewColumn *tcolumn;

	g_return_if_fail (os && IS_OBJECT_SELECTOR (os));
	g_return_if_fail (column <= 1);

	tcolumn = gtk_tree_view_get_column (os->priv->treeview, column);
	gtk_tree_view_column_set_title (tcolumn, label);
}
