/* allfield.c
 *
 * Copyright (C) 2001 - 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


/*
 * This module represents QueryField of type QUERY_FIELD_ALLFIELDS:
 * all the fields of a table or a view or a Query
 */

/* FIXME: in the change to adding support for Query.*:
   - remove the table and query and use a GObject instead
   - rename qv to view
   - ...
*/
   

#include <config.h>
#include "../query.h"
#include "../query-field-private.h"
#include "../database.h"
#include "../object-selector.h"

static void        q_init            (QueryField *qf);
static void        q_finalize         (QueryField *qf);
static void        q_deactivate      (QueryField *qf);
static void        q_activate        (QueryField *qf);
static GtkWidget * q_get_edit_widget (QueryField *qf);
static GtkWidget * q_get_sel_widget  (QueryField *qf, GCallback callback, gpointer data);
static gchar     * q_render_as_sql   (QueryField *qf, GSList * missing_values);
static xmlNodePtr  q_render_as_xml   (QueryField *qf, GSList * missing_values);
static gchar     * q_render_as_string(QueryField *qf, GSList * missing_values);
static void        q_save_to_xml     (QueryField *qf, xmlNodePtr node);
static void        q_load_from_xml   (QueryField *qf, xmlNodePtr node);
static void        q_copy_other_field(QueryField *qf, QueryField *other);
static gboolean    q_is_equal_to     (QueryField *qf, QueryField *other);
static GSList    * q_get_monitored_objects (QueryField *qf);
static void        q_replace_comp    (QueryField *qf, gint ref, GObject   *old, GObject   *new);

/* Weak ref from the refering object */
static void        q_view_obj_destroyed_wr (QueryField *qf, GObject *obj);

static void query_field_allfields_set_queryview (QueryField *qf, QueryView *qv);

typedef struct {
	QueryView   *view;

	gboolean     obj_in_destroy;

	/* XML table name if we did not find it in the first place (like TVxxx:FIyyy) */
	gchar       *obj_name;
} private_data;

#define QF_PRIVATE_DATA(qf) ((private_data *) qf->private_data)

QueryFieldIface * 
query_field_allfields_get_iface()
{
	QueryFieldIface *iface;

	iface = g_new0 (QueryFieldIface, 1);
	iface->field_type = QUERY_FIELD_ALLFIELDS;
	iface->name = "allfields";
	iface->pretty_name = _("All fields of a table, a view or a query");
	iface->init = q_init;
	iface->destroy = q_finalize;
	iface->deactivate = q_deactivate;
	iface->activate = q_activate;
	iface->get_edit_widget = q_get_edit_widget;
	iface->get_sel_widget = q_get_sel_widget;
	iface->render_as_sql = q_render_as_sql;
	iface->render_as_xml = q_render_as_xml;
	iface->render_as_string = q_render_as_string;
	iface->save_to_xml = q_save_to_xml;
	iface->load_from_xml = q_load_from_xml;
	iface->copy_other_field = q_copy_other_field;
	iface->is_equal_to = q_is_equal_to;
	iface->get_monitored_objects = q_get_monitored_objects;
	iface->replace_comp = q_replace_comp;

	return iface;
}

static void        
q_init            (QueryField *qf)
{
	private_data *data;
	data = g_new0 (private_data, 1);
	qf->private_data = (gpointer) data;
	QF_PRIVATE_DATA(qf)->view = NULL;
	QF_PRIVATE_DATA(qf)->obj_name = NULL;
	QF_PRIVATE_DATA(qf)->obj_in_destroy = FALSE;
}

static void        
q_finalize         (QueryField *qf)
{
	query_field_deactivate (qf);
	if (qf->private_data) {
		if (QF_PRIVATE_DATA(qf)->obj_name)
			g_free (QF_PRIVATE_DATA(qf)->obj_name);
		g_free (qf->private_data);
		qf->private_data = NULL;
	}
}

static void
q_deactivate      (QueryField *qf)
{
	if (! qf->activated)
		return;

	/* This function disconnects any event handler from any object
	   this QueryField wants to receive events from.
	   Here we disconnect from the table if we are connected */
	if (QF_PRIVATE_DATA(qf)->view) {
		gchar *str;
		if (! QF_PRIVATE_DATA(qf)->obj_in_destroy)
			g_object_weak_unref (G_OBJECT (QF_PRIVATE_DATA(qf)->view), 
					     (GWeakNotify) q_view_obj_destroyed_wr, qf);
		else
			QF_PRIVATE_DATA(qf)->obj_in_destroy = FALSE;

		if (QF_PRIVATE_DATA(qf)->obj_name) {
			g_free (QF_PRIVATE_DATA(qf)->obj_name);
			QF_PRIVATE_DATA(qf)->obj_name = NULL;
		}

		str = query_view_get_xml_id (QF_PRIVATE_DATA(qf)->view);
		QF_PRIVATE_DATA(qf)->obj_name = str;

		QF_PRIVATE_DATA(qf)->view = NULL;
	}
	
	query_field_set_activated (qf, FALSE);
}

static void
q_activate        (QueryField *qf)
{
	/* this function gets references to any object this QueryField wants to 
	   receive events from.
	   Here we connect to the DbTable or Query of the field we are refering to */
	QueryView *view;

	if (qf->activated)
		return;

	view = QF_PRIVATE_DATA(qf)->view;
	
	if (!view && QF_PRIVATE_DATA(qf)->obj_name) {
		view = query_get_view_by_xmlid (qf->query, QF_PRIVATE_DATA(qf)->obj_name);

		if (view) {
			g_free (QF_PRIVATE_DATA(qf)->obj_name);
			QF_PRIVATE_DATA(qf)->obj_name = NULL;
		}
	}

	if (view) {
		QF_PRIVATE_DATA(qf)->view = view;
		g_object_weak_ref (G_OBJECT (QF_PRIVATE_DATA(qf)->view), 
				   (GWeakNotify) q_view_obj_destroyed_wr, qf);
		query_field_set_activated (qf, TRUE);
	}
}

static void
view_selected_cb (ObjectSelector *os, GObject *sel, QueryField *qf);
static GtkWidget * 
q_get_edit_widget (QueryField *qf)
{
	GtkWidget *vbox, *os;

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD/2.);

	os = object_selector_new (qf->query->conf, OBJECT_SELECTOR_QUERIES | OBJECT_SELECTOR_QVIEWS,
				  OBJECT_SELECTOR_PARAM_TOP_QUERY, qf->query,
				  OBJECT_SELECTOR_PARAM_SUB_QUERIES, FALSE, -1);
	object_selector_set_column_label (OBJECT_SELECTOR (os), 0, _("Source"));
	object_selector_set_comments_visible (OBJECT_SELECTOR (os), FALSE);
	if (QF_PRIVATE_DATA(qf)->view) 
		object_selector_set_selection (OBJECT_SELECTOR (os), G_OBJECT (QF_PRIVATE_DATA(qf)->view));
	gtk_box_pack_start (GTK_BOX (vbox), os, TRUE, TRUE, GNOME_PAD/2.);
	gtk_widget_show (os);

	g_signal_connect (G_OBJECT (os), "selection_changed",
			  G_CALLBACK (view_selected_cb), qf);

	return vbox;
}

static GtkWidget * 
q_get_sel_widget (QueryField *qf, GCallback callback, gpointer data)
{
	GtkWidget *button;
	gchar *str;

	if (qf->activated)
		str = g_strdup_printf("%s.*", query_view_get_name (QF_PRIVATE_DATA(qf)->view));
	else
		str = g_strdup (_("TABLE.*"));
	button = gtk_button_new_with_label (str);
	g_free (str);
	g_signal_connect (G_OBJECT (button), "clicked", callback, data);
	g_object_set_data (G_OBJECT (button), "qf", qf);

	/* Set the "QF_obj_emit_sig" attribute so that we can attach attributes to that button
	   which will be transmitted when the user clicks on it */
	g_object_set_data (G_OBJECT (button), "QF_obj_emit_sig", button);

	return button;
}

/* we consider the selected table is the the one being
   represented in the QueryField */
static void
view_selected_cb (ObjectSelector *os, GObject *sel, QueryField *qf)
{
	if (IS_QUERY_VIEW (sel)) {
		QueryView *view = QUERY_VIEW (sel);
	
		/* test if anything has changed */
		if (QF_PRIVATE_DATA(qf)->view == view) 
			return;

		query_field_deactivate (qf);
		query_field_allfields_set_queryview (qf, view);
		query_field_activate (qf);

#ifdef debug_signal
		g_print (">> 'FIELD_MODIFIED' from widget_field_sel_cb\n");
#endif
		g_signal_emit_by_name (G_OBJECT (qf), "field_modified");
#ifdef debug_signal
		g_print ("<< 'FIELD_MODIFIED' from widget_field_sel_cb\n");
#endif	
	}
}

static void        
q_view_obj_destroyed_wr (QueryField *qf, GObject *obj)
{
	QF_PRIVATE_DATA(qf)->obj_in_destroy = TRUE;
	q_deactivate (qf);

	/* if the field disappears, then destroy is the result */
	g_object_unref (G_OBJECT (qf));
}

static gchar     * 
q_render_as_sql   (QueryField *qf, GSList * missing_values)
{
	gchar *str = NULL;

	if (qf->activated) 
		str = g_strdup_printf ("%s.*", query_view_get_name (QF_PRIVATE_DATA(qf)->view));

	return str;
}

static xmlNodePtr  
q_render_as_xml   (QueryField *qf, GSList * missing_values)
{
	return NULL;
}

static gchar * 
q_render_as_string(QueryField *qf, GSList * missing_values)
{
	gchar *str = NULL;

	if (qf->activated) 
		str = g_strdup_printf ("%s.*", query_view_get_name (QF_PRIVATE_DATA(qf)->view));

	return str;
}

static void  
q_save_to_xml     (QueryField *qf, xmlNodePtr node)
{
	if (qf->activated) {
		gchar *str;

		/* node object ref */
		str = query_view_get_xml_id (QF_PRIVATE_DATA(qf)->view);
		xmlSetProp (node, "object", str);
		g_free (str);

		xmlSetProp (node, "type", "allfields");
	}
	else
		g_warning ("QueryField not activated; can't save\n");
}

static void        
q_load_from_xml   (QueryField *qf, xmlNodePtr node)
{
	query_field_deactivate (qf);

	/* check we have a QueryField */
	if (!strcmp (node->name, "QueryField")) {
		gchar *str;

		str = xmlGetProp (node, "type");
		if (!str || (str && strcmp (str, "allfields"))) {
			if (str) g_free (str);
			return;
		}

		str = xmlGetProp (node, "object");
		QF_PRIVATE_DATA(qf)->obj_name = str;

		query_field_activate (qf);
	}
}

static void        
q_copy_other_field(QueryField *qf, QueryField *other)
{
	/* we can't call q_destroy(qf) because we don't know what the type
	   of QueryField it was before. This is normally done by the
	   QueryField object before the copy */

	if (QF_PRIVATE_DATA(other)->view) {
		QF_PRIVATE_DATA(qf)->view = QF_PRIVATE_DATA(other)->view;
		query_field_activate (qf);
	}
	else {
		if (QF_PRIVATE_DATA(other)->obj_name) {
			QF_PRIVATE_DATA(qf)->obj_name = g_strdup (QF_PRIVATE_DATA(other)->obj_name);
			query_field_activate (qf);
		}
	}
}

static gboolean
q_is_equal_to (QueryField *qf, QueryField *other)
{
	gboolean retval = FALSE;;

	if (qf->activated && other->activated) {
		if (QF_PRIVATE_DATA(qf)->view == QF_PRIVATE_DATA(other)->view)
			retval = TRUE;
	}

	return retval;
}

static GSList *
q_get_monitored_objects (QueryField *qf)
{
	GSList *list = NULL;

	if (qf->activated)
		list = g_slist_prepend (NULL, QF_PRIVATE_DATA(qf)->view);

	return list;
}

static void
q_replace_comp (QueryField *qf, gint ref, GObject   *old, GObject   *new)
{
	/* References can be to QueryViews can change. The obj attribute of the
	 new and the old QueryViews can either:
	 1) be the same and the the field WILL NOT change
	 2) be different and in this case things get complicated...
	*/

	if ((G_OBJECT (QF_PRIVATE_DATA(qf)->view) == old) && 
	    (IS_QUERY_VIEW (old) && IS_QUERY_VIEW (new))) {
		if (QUERY_VIEW (old)->obj == QUERY_VIEW (new)->obj) { /* case 1 */
			/* we can keep it activated because the field does not change */
			g_object_weak_unref (G_OBJECT (QF_PRIVATE_DATA(qf)->view),
					     (GWeakNotify) q_view_obj_destroyed_wr, qf);
			QF_PRIVATE_DATA(qf)->view = QUERY_VIEW (new);
			g_object_weak_ref (G_OBJECT (QF_PRIVATE_DATA(qf)->view),
					   (GWeakNotify) q_view_obj_destroyed_wr, qf);
		}
		else 
			query_field_allfields_set_queryview (qf, QUERY_VIEW (new));	
	}
	

	return;
}




/* 
 * 
 * Real QueryField object's different methods
 * 
 *
 */

QueryField *
query_field_allfields_dbtable_new (Query *q, gchar * name, QueryView *qv, DbTable *table)
{
	QueryField *qf;
	QueryView *realqv;

	g_return_val_if_fail (q && IS_QUERY (q), NULL);
	g_return_val_if_fail (table && IS_DB_TABLE (table), NULL);
	if (qv) {
		g_return_val_if_fail (IS_QUERY_VIEW (qv) && (qv->query == q), NULL);
		realqv = qv;
	}
	else { 
		/* try to find a suitable QV... */
		QueryView *qv2 = NULL;
		GSList *list;

		list = q->views;
		while (list && !qv2) {
			if (IS_DB_TABLE (QUERY_VIEW (list->data)->obj) &&
			    (DB_TABLE (QUERY_VIEW (list->data)->obj) == table))
				qv2 = QUERY_VIEW (list->data);
			list = g_slist_next (list);
		}

		/* ...and otherwise make one */
		if (!qv2) 
			realqv = QUERY_VIEW (query_add_view_with_obj (q, G_OBJECT (table)));
		else
			realqv = qv2;
	}

	qf = QUERY_FIELD (query_field_new (q, name, QUERY_FIELD_ALLFIELDS));
	query_field_allfields_set_queryview (qf, realqv);
	query_field_activate (qf);

	return qf;
}

QueryField *
query_field_allfields_query_new (Query *q, gchar * name, QueryView *qv, Query *query)
{
QueryField *qf;
	QueryView *realqv;

	g_return_val_if_fail (q && IS_QUERY (q), NULL);
	g_return_val_if_fail (query && IS_QUERY (query), NULL);
	if (qv) {
		g_return_val_if_fail (IS_QUERY_VIEW (qv) && (qv->query == q), NULL);
		realqv = qv;
	}
	else { 
		/* try to find a suitable QV... */
		QueryView *qv2 = NULL;
		GSList *list;

		list = q->views;
		while (list && !qv2) {
			if (IS_QUERY (QUERY_VIEW (list->data)->obj) &&
			    (QUERY (QUERY_VIEW (list->data)->obj) == query))
				qv2 = QUERY_VIEW (list->data);
			list = g_slist_next (list);
		}

		/* ...and otherwise make one */
		if (!qv2) 
			realqv = QUERY_VIEW (query_add_view_with_obj (q, G_OBJECT (query)));
		else
			realqv = qv2;
	}

	qf = QUERY_FIELD (query_field_new (q, name, QUERY_FIELD_ALLFIELDS));
	query_field_allfields_set_queryview (qf, realqv);
	query_field_activate (qf);

	return qf;
}


static void
query_field_allfields_set_queryview (QueryField *qf, QueryView *qv)
{
	g_return_if_fail (qf && IS_QUERY_FIELD (qf));
	g_return_if_fail (qf->field_type == QUERY_FIELD_ALLFIELDS);
	g_return_if_fail (qv && IS_QUERY_VIEW (qv));

	if (qf->activated) 
		query_field_deactivate (qf);


	QF_PRIVATE_DATA(qf)->view = qv;
	qf->query = qv->query;

	query_field_activate (qf);
}

gint 
query_field_allfields_get_span (QueryField *qf)
{
	gint retval = -1;

	g_assert (qf);
	g_assert (IS_QUERY_FIELD (qf));
	g_assert (qf->field_type == QUERY_FIELD_ALLFIELDS);

	if (qf->activated) {
		if (IS_DB_TABLE (QF_PRIVATE_DATA(qf)->view->obj)) 
			retval = g_slist_length (DB_TABLE (QF_PRIVATE_DATA(qf)->view->obj)->fields);
		else {
			GSList *list;
			gint i=0;
			list = QUERY (QF_PRIVATE_DATA(qf)->view->obj)->fields;
			while (list) {
				if (QUERY_FIELD (list->data)->is_printed)
					i++;
				list = g_slist_next (list);
			}
			retval = i;
		}
	}
	
	return retval;
}

QueryView *
query_field_allfields_get_queryview (QueryField *qf)
{
	QueryView *view = NULL;

	g_assert (qf);
	g_assert (IS_QUERY_FIELD (qf));
	g_assert (qf->field_type == QUERY_FIELD_ALLFIELDS);

	if (qf->activated) 
		view = QF_PRIVATE_DATA(qf)->view;

	return view;
}
