/* -*- Mode: C; indent-tabs-mode:nil; c-basic-offset: 8-*- */
/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 *Copyright 2001-2002 dodji SEKETELI, Gal CHAMOULAUD.
 *http://www.freespiders.org
 */

#include "mlview-namespace-editor.h"
#include "mlview-app-context.h"

#include "mlview-utils.h"

struct _MlViewNamespaceEditorPrivate {
        GtkCList *namespaces;
        GtkEntry *ns_prefix_edit_entry;
        GtkEntry *ns_uri_edit_entry;
        GtkButton *add_ns_button;
        GtkButton *remove_ns_button;
        gint current_selected_row;
        gboolean editable;
        gint ns_prefix_edit_entry_changed_handler_id;
        gint ns_uri_edit_entry_changed_handler_id;
        xmlNodePtr current_xml_node;
        MlViewAppContext *app_context;
};


typedef struct _MlViewNamespaceDesc MlViewNamespaceDesc;
struct _MlViewNamespaceDesc {
        xmlNs *ns_def;
        xmlNode *xml_node;
};

enum {
        NAMESPACE_ADDED,
        NAMESPACE_PREFIX_CHANGED,
        NAMESPACE_URI_CHANGED,
        NAMESPACE_CHANGED,
        NAMESPACE_DELETED,
        NUMBER_OF_SIGNALS
};

enum {
        NS_PREFIX_COLUMN,
        NS_URI_COLUMN,
};

#define PRIVATE(editor) (editor->private)

static GtkVBoxClass *parent_class = NULL;
static guint mlview_namespace_editor_signals[NUMBER_OF_SIGNALS] =
        { 0 };

static void
 mlview_namespace_editor_class_init (MlViewNamespaceEditorClass *
                                     a_klass);
static void mlview_namespace_editor_init (MlViewNamespaceEditor *
                                          a_editor);
static void mlview_namespace_editor_destroy (GtkObject *
                                             a_object);

static MlViewNamespaceDesc *mlview_namespace_desc_new (xmlNs *
                                                       a_xml_ns,
                                                       xmlNode *
                                                       a_xml_node);
static xmlNs *mlview_namespace_desc_get_ns (MlViewNamespaceDesc *
                                            a_desc);
static xmlNode
        * mlview_namespace_desc_get_xml_node (MlViewNamespaceDesc
                                              * a_desc);
static void mlview_namespace_desc_destroy (MlViewNamespaceDesc *
                                           a_desc);

static gint
mlview_namespace_editor_edit_namespace (MlViewNamespaceEditor *
                                        a_editor, xmlNs * a_ns,
                                        xmlNode * a_xml_node,
                                        gboolean a_selectable);
static MlViewNamespaceDesc
        * mlview_namespace_editor_get_current_selected_ns_desc
        (MlViewNamespaceEditor * a_editor);
static void namespace_desc_destroy_on_ns_editor_destroy (gpointer
                                                         a_data);

static xmlNs
        *
        mlview_namespace_editor_ask_namespace_to_user_and_add_it
        (MlViewNamespaceEditor * a_editor);

/*callbacks function definitions*/
static void row_selected_cb (GtkCList * a_clist, gint a_row,
                             gint a_column,
                             GdkEventButton * a_event,
                             MlViewNamespaceEditor * a_editor);

static void ns_prefix_edit_entry_changed_cb (GtkEntry * a_entry,
                                             MlViewNamespaceEditor
                                             * a_editor);
static void ns_uri_edit_entry_changed_cb (GtkEntry * a_entry,
                                          MlViewNamespaceEditor *
                                          a_editor);
static void add_namespace_button_clicked_cb (GtkButton *
                                             a_add_namespace_button,
                                             MlViewNamespaceEditor
                                             * a_editor);
static void remove_namespace_button_clicked_cb (GtkButton *
                                                a_remove_namespace_button,
                                                MlViewNamespaceEditor
                                                * a_editor);

/*==============================================
 *private gtk object framework methods
 *==============================================*/


/**
 *
 */
static void
mlview_namespace_editor_class_init (MlViewNamespaceEditorClass *
                                    a_klass)
{
        GtkObjectClass *object_class;

        object_class = (GtkObjectClass *) a_klass;
        parent_class = gtk_type_class (gtk_vbox_get_type ());
        g_return_if_fail (a_klass != NULL);

        object_class->destroy = mlview_namespace_editor_destroy; /*overload the destroy method of object */

        mlview_namespace_editor_signals[NAMESPACE_ADDED] =
                g_signal_new ("namespace-added",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewNamespaceEditorClass,
                               namespace_added), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              GTK_TYPE_NONE, 0, NULL);

        mlview_namespace_editor_signals[NAMESPACE_PREFIX_CHANGED]
                =
                g_signal_new ("namespace-prefix-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewNamespaceEditorClass,
                               namespace_prefix_changed), NULL,
                              NULL, gtk_marshal_NONE__NONE,
                              GTK_TYPE_NONE, 0, NULL);

        mlview_namespace_editor_signals[NAMESPACE_URI_CHANGED] =
                g_signal_new ("namespace-uri-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewNamespaceEditorClass,
                               namespace_uri_changed), NULL,
                              NULL, gtk_marshal_NONE__NONE,
                              GTK_TYPE_NONE, 0, NULL);

        mlview_namespace_editor_signals[NAMESPACE_CHANGED] =
                g_signal_new ("namespace-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewNamespaceEditorClass,
                               namespace_changed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              GTK_TYPE_NONE, 0, NULL);

        mlview_namespace_editor_signals[NAMESPACE_DELETED] =
                g_signal_new ("namespace-deleted",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewNamespaceEditorClass,
                               namespace_deleted), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              GTK_TYPE_NONE, 0, NULL);

        a_klass->namespace_added = NULL;
        a_klass->namespace_prefix_changed = NULL;
        a_klass->namespace_uri_changed = NULL;
        a_klass->namespace_changed = NULL;
        a_klass->namespace_deleted = NULL;
}

/**
 *The instance initialyzer. 
 *
 */
static void
mlview_namespace_editor_init (MlViewNamespaceEditor * a_editor)
{
        MlViewNamespaceEditorPrivate *private;
        GtkWidget *table,
        *scrolled_window;
        const gchar *titles[2] =
                { _("namespace prefixes"), _("namespace uris") };
        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (PRIVATE (a_editor) == NULL);


        PRIVATE (a_editor) =
                g_malloc0 (sizeof
                           (MlViewNamespaceEditorPrivate));
        private = PRIVATE (a_editor);

        /*init the clist that shows the list of namespaces */
        private->namespaces =
                GTK_CLIST (gtk_clist_new_with_titles
                           (2, (gchar**)titles));
        gtk_clist_column_titles_passive (private->namespaces);
        g_signal_connect (G_OBJECT (private->namespaces),
                          "select_row",
                          G_CALLBACK (row_selected_cb),
                          a_editor);
        scrolled_window = gtk_scrolled_window_new (NULL, NULL);
        gtk_container_add (GTK_CONTAINER (scrolled_window),
                           GTK_WIDGET (private->namespaces));
        gtk_box_pack_start (GTK_BOX (a_editor), scrolled_window,
                            TRUE, TRUE, 0);

        /*init the prefix/uri edition entries */
        private->ns_prefix_edit_entry =
                GTK_ENTRY (gtk_entry_new ());
        private->ns_prefix_edit_entry_changed_handler_id =
                g_signal_connect (G_OBJECT
                                  (private->
                                   ns_prefix_edit_entry),
                                  "changed",
                                  G_CALLBACK
                                  (ns_prefix_edit_entry_changed_cb),
                                  a_editor);
        private->ns_uri_edit_entry =
                GTK_ENTRY (gtk_entry_new ());
        private->ns_uri_edit_entry_changed_handler_id =
                g_signal_connect (G_OBJECT
                                  (private->ns_uri_edit_entry),
                                  "changed",
                                  G_CALLBACK
                                  (ns_uri_edit_entry_changed_cb),
                                  a_editor);
        table = gtk_table_new (1, 2, TRUE);
        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (private->
                                               ns_prefix_edit_entry),
                                   0, 1, 0, 1);
        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (private->
                                               ns_uri_edit_entry),
                                   1, 2, 0, 1);
        gtk_box_pack_start (GTK_BOX (a_editor), table, FALSE,
                            TRUE, 0);


        /*add/remove/show all/show local button */
        private->add_ns_button =
                GTK_BUTTON (gtk_button_new_with_label
                            (_("add")));
        g_signal_connect (G_OBJECT (private->add_ns_button),
                          "clicked",
                          G_CALLBACK
                          (add_namespace_button_clicked_cb),
                          a_editor);
        private->remove_ns_button =
                GTK_BUTTON (gtk_button_new_with_label
                            (_("remove")));
        g_signal_connect (G_OBJECT (private->remove_ns_button),
                          "clicked",
                          G_CALLBACK
                          (remove_namespace_button_clicked_cb),
                          a_editor);

        table = gtk_table_new (1, 2, TRUE);
        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (private->
                                               add_ns_button), 0,
                                   1, 0, 1);
        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (private->
                                               remove_ns_button),
                                   1, 2, 0, 1);
        gtk_box_pack_start (GTK_BOX (a_editor), table, FALSE,
                            TRUE, 0);

        PRIVATE (a_editor)->current_selected_row = -1;
}


/***
 *mlview_namespace_editor_destroy:
 *@a_object: the object to destroy. This object must be instance of MlViewNamespaceEditor.
 *
 *destroyer of MlViewNamespaceEditor.
 */
static void
mlview_namespace_editor_destroy (GtkObject * a_object)
{
        MlViewNamespaceEditor *editor = NULL;

        g_return_if_fail (MLVIEW_IS_NAMESPACE_EDITOR (a_object));
        editor = MLVIEW_NAMESPACE_EDITOR (a_object);

        if (PRIVATE (editor) == NULL)
                return;

        /*
         *do not free the widgets. 
         *They will be freed when their container (the parent class) is freed
         *
         */

        if (PRIVATE (editor) != NULL) {
                g_free (PRIVATE (editor));
                PRIVATE (editor) = NULL;
        }

        if (GTK_OBJECT_CLASS (parent_class)->destroy)
                GTK_OBJECT_CLASS (parent_class)->
                        destroy (a_object);
}


/*=====================================
 *private helper functions
 *=====================================*/

/**
 *
 */
static MlViewNamespaceDesc *
mlview_namespace_desc_new (xmlNs * a_xml_ns,
                           xmlNode * a_xml_node)
{
        MlViewNamespaceDesc *result;

        result = g_malloc0 (sizeof (MlViewNamespaceDesc));
        result->ns_def = a_xml_ns;
        result->xml_node = a_xml_node;
        return result;
}


/**
 *parameter is NULL. Note that the returned pointer 
 *is a pointer to the actual internal data structure.
 *Do not free it. 
 *
 */
static xmlNs *
mlview_namespace_desc_get_ns (MlViewNamespaceDesc * a_desc)
{
        g_return_val_if_fail (a_desc != NULL, NULL);
        return a_desc->ns_def;
}

/**
 *
 */
static xmlNode *
mlview_namespace_desc_get_xml_node (MlViewNamespaceDesc * a_desc)
{
        g_return_val_if_fail (a_desc != NULL, NULL);

        return a_desc->xml_node;
}

/**
 *Frees the MlViewNamespaceDescriptor. 
 *This structure does not owns the xmlNodePtr and the xmlNs
 *fields. It does not free them. You will have to free them by your self. 
 *
 */
static void
mlview_namespace_desc_destroy (MlViewNamespaceDesc * a_desc)
{
        g_return_if_fail (a_desc != NULL);

        g_free (a_desc);
}


/**
 *Edits the namespace definition given in argument. 
 *A part from the edition, this can method associate
 *an xml node to the namespace edited. 
 *Then, later, one can get the namespace, and that associated namespace. 
 *
 */
static gint
mlview_namespace_editor_edit_namespace (MlViewNamespaceEditor *
                                        a_editor, xmlNs * a_ns,
                                        xmlNode * a_xml_node,
                                        gboolean a_selectable)
{
        gchar *ns_text[2] = { 0 };
        gint row_num;
        MlViewNamespaceDesc *ns_desc = NULL;

        g_return_val_if_fail (a_editor != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_editor) != NULL, -1);
        g_return_val_if_fail (a_ns != NULL, -1);
        g_return_val_if_fail (a_xml_node != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_editor)->namespaces !=
                              NULL, -1);

        ns_desc = mlview_namespace_desc_new (a_ns, a_xml_node);
        g_return_val_if_fail (ns_desc != NULL, -1);

        ns_text[NS_PREFIX_COLUMN] = (gchar *) a_ns->prefix;
        ns_text[NS_URI_COLUMN] = (gchar *) a_ns->href;

        row_num =
                gtk_clist_append (PRIVATE (a_editor)->namespaces,
                                  ns_text);
        gtk_clist_set_row_data_full (PRIVATE (a_editor)->
                                     namespaces, row_num,
                                     ns_desc, (GtkDestroyNotify)
                                     namespace_desc_destroy_on_ns_editor_destroy);

        if (!a_selectable)
                gtk_clist_set_selectable (PRIVATE (a_editor)->
                                          namespaces, row_num,
                                          a_selectable);
        return row_num;
}


static MlViewNamespaceDesc
        * mlview_namespace_editor_get_current_selected_ns_desc
        (MlViewNamespaceEditor * a_editor) {
        MlViewNamespaceDesc *result = NULL;

        g_return_val_if_fail (a_editor != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_editor) != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_editor)->
                              current_selected_row >= 0, NULL);
        g_return_val_if_fail (PRIVATE (a_editor)->namespaces !=
                              NULL, NULL);

        result = (MlViewNamespaceDesc *) gtk_clist_get_row_data
                (PRIVATE (a_editor)->namespaces,
                 PRIVATE (a_editor)->current_selected_row);

        return result;
}


/**
 *
 *
 */
static xmlNs
        *
        mlview_namespace_editor_ask_namespace_to_user_and_add_it
        (MlViewNamespaceEditor * a_editor) {
        static GtkWidget *table = NULL,
                *label = NULL,
                *message = NULL,
                *uri = NULL,
                *prefix = NULL,
                *p_namespace_picker = NULL;

        xmlNs *result = NULL;
        guchar *prefix_str,
        *uri_str;
        gboolean loop = TRUE;
        gint button;

        g_return_val_if_fail (a_editor != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_editor) != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_editor)->
                              current_xml_node != NULL, NULL);

        if (p_namespace_picker == NULL) {
                p_namespace_picker =
                        gtk_dialog_new_with_buttons (_
                                                     ("add namespace:"),
                                                     NULL,
                                                     GTK_DIALOG_MODAL,
                                                     _("OK"),
                                                     GTK_RESPONSE_ACCEPT,
                                                     _("Cancel"),
                                                     GTK_RESPONSE_REJECT,
                                                     NULL);

                gtk_window_set_modal (GTK_WINDOW
                                      (p_namespace_picker),
                                      TRUE);


                message =
                        gtk_label_new (_
                                       ("enter namespace prefix and uri:"));
                gtk_box_pack_start (GTK_BOX
                                    (GTK_DIALOG
                                     (p_namespace_picker)->vbox),
                                    message, TRUE, TRUE, 0);

                label = gtk_label_new (_("prefix"));

                prefix = gtk_entry_new ();

                table = gtk_table_new (1, 2, TRUE);

                gtk_table_attach_defaults (GTK_TABLE (table),
                                           label, 0, 1, 0, 1);

                gtk_table_attach_defaults (GTK_TABLE (table),
                                           prefix, 1, 2, 0, 1);

                gtk_box_pack_start (GTK_BOX (GTK_DIALOG
                                             (p_namespace_picker)->
                                             vbox), table, FALSE,
                                    TRUE, 0);

                label = gtk_label_new (_("uri"));

                uri = gtk_entry_new ();

                table = gtk_table_new (1, 2, TRUE);

                gtk_table_attach_defaults (GTK_TABLE (table),
                                           label, 0, 1, 0, 1);

                gtk_table_attach_defaults (GTK_TABLE (table),
                                           uri, 1, 2, 0, 1);

                gtk_box_pack_start (GTK_BOX (GTK_DIALOG
                                             (p_namespace_picker)->
                                             vbox), table, FALSE,
                                    TRUE, 0);

                gtk_widget_show_all
                        (GTK_WIDGET
                         (GTK_DIALOG (p_namespace_picker)->
                          vbox));
        }

        gtk_widget_grab_focus (prefix);

        uri_str = (guchar *)
                gtk_entry_get_text (GTK_ENTRY (prefix));

        if (!mlview_utils_is_white_string (uri_str))
                gtk_entry_select_region (GTK_ENTRY (uri), 0, -1);

        while (loop) {
                gtk_widget_realize (GTK_WIDGET
                                    (p_namespace_picker));

                mlview_app_context_set_window_icon
                        (PRIVATE (a_editor)->app_context,
                         GTK_WINDOW (p_namespace_picker));

                button = gtk_dialog_run (GTK_DIALOG
                                         (p_namespace_picker));

                switch (button) {
                case GTK_RESPONSE_ACCEPT:
                         /*OK*/ prefix_str = (guchar *)
                                gtk_entry_get_text (GTK_ENTRY
                                                    (prefix));
                        uri_str = (guchar *)
                                gtk_entry_get_text (GTK_ENTRY
                                                    (uri));
                        result = mlview_namespace_editor_add_namespace_def (a_editor, prefix_str, uri_str, PRIVATE (a_editor)->current_xml_node);

                        if (result != NULL) {
                                loop = FALSE;
                                gtk_widget_hide
                                        (GTK_WIDGET
                                         (p_namespace_picker));
                        } else {
                                gtk_label_set_text
                                        (GTK_LABEL (message),
                                         _
                                         ("please enter an other namespace"));
                        }
                        break;

                case GTK_RESPONSE_REJECT: /*cancel */
                        result = NULL;
                        loop = FALSE;
                        gtk_widget_hide
                                (GTK_WIDGET
                                 (p_namespace_picker));
                        break;

                case -1:
                        result = FALSE;
                        loop = FALSE;
                        break;

                default:
                        gtk_label_set_text
                                (GTK_LABEL (message),
                                 _
                                 ("please enter a namespace again"));
                        break;
                }
        }

        return result;
}

/*=====================================
 *MlViewNamespaceEditor public methods.
 *====================================*/

/**
 *
 *@param a_editor the type builder of MlViewNamespaceEditor. 
 *
 *
 */
guint
mlview_namespace_editor_get_type ()
{

        static guint mlview_namespace_editor_type = 0;

        if (!mlview_namespace_editor_type) {

                static const GTypeInfo type_info = {
                        sizeof (MlViewNamespaceEditorClass),
                        NULL,   /* base_init */
                        NULL,   /* base_finalize */
                        (GClassInitFunc)
                                mlview_namespace_editor_class_init,
                        NULL,   /* class_finalize */
                        NULL,   /* class_data */
                        sizeof (MlViewNamespaceEditor),
                        0,
                        (GInstanceInitFunc)
                        mlview_namespace_editor_init
                };

                mlview_namespace_editor_type =
                        g_type_register_static (GTK_TYPE_VBOX,
                                                "MlViewNamespaceEditor",
                                                &type_info, 0);

        }
        return mlview_namespace_editor_type;
}


/**
 *Creates a new instance of MlViewNamespaceEditor. 
 *You can free this instance using gtk_widget_destroy. 
 */
GtkWidget *
mlview_namespace_editor_new (MlViewAppContext * a_app_context)
{
        MlViewNamespaceEditor *result;

        result = gtk_type_new (MLVIEW_TYPE_NAMESPACE_EDITOR);

        PRIVATE (result)->app_context = a_app_context;

        return GTK_WIDGET (result);
}




/**
 *Given an xml node, edits the namespaces definied on that node, 
 *and also, those defined in the parent nodes.
 *Keeps track of the xml node being currently edited. 
 *
 *@param a_xml_node 
 *@param a_editor 
 *
 *
 */
void
 mlview_namespace_editor_edit_node_visible_namespaces
        (MlViewNamespaceEditor * a_editor, xmlNode * a_xml_node)
{
        xmlNode *node = NULL;
        xmlNs *ns_def = NULL;
        gboolean selectable;

        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (PRIVATE (a_editor) != NULL);
        g_return_if_fail (PRIVATE (a_editor)->namespaces !=
                          NULL);
        g_return_if_fail (a_xml_node != NULL);

        PRIVATE (a_editor)->current_xml_node = a_xml_node;

        gtk_clist_freeze (PRIVATE (a_editor)->namespaces);
        node = a_xml_node;
        while (node != NULL) {
                selectable = (node == a_xml_node);
                /*loop over the namespaces defined on the current node */
                for (ns_def = node->nsDef;
                     ns_def != NULL; ns_def = ns_def->next) {
                        mlview_namespace_editor_edit_namespace
                                (a_editor, ns_def, node,
                                 selectable);
                }

                node = node->parent;
        }

        gtk_clist_thaw (PRIVATE (a_editor)->namespaces);
}


/**
 *Clears the content of the current instance of MlViewNamespaceEditor. 
 *This method does not emit any signal. 
 *
 *@param a_editor the current instance of MlViewNamespaceEditor.
 */
void
mlview_namespace_editor_clear (MlViewNamespaceEditor * a_editor)
{
        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (PRIVATE (a_editor) != NULL);
        g_return_if_fail (PRIVATE (a_editor)->namespaces !=
                          NULL);

        /*FIXME: add the clearing of the uri and prefix edition areas */
        gtk_signal_handler_block
                (GTK_OBJECT
                 (PRIVATE (a_editor)->ns_prefix_edit_entry),
                 PRIVATE (a_editor)->
                 ns_prefix_edit_entry_changed_handler_id);

        gtk_signal_handler_block
                (GTK_OBJECT
                 (PRIVATE (a_editor)->ns_uri_edit_entry),
                 PRIVATE (a_editor)->
                 ns_uri_edit_entry_changed_handler_id);

        gtk_entry_set_text (PRIVATE (a_editor)->
                            ns_prefix_edit_entry, "");

        gtk_entry_set_text (PRIVATE (a_editor)->
                            ns_uri_edit_entry, "");

        gtk_clist_clear (PRIVATE (a_editor)->namespaces);

        PRIVATE (a_editor)->current_selected_row = -1;

        gtk_signal_handler_unblock
                (GTK_OBJECT
                 (PRIVATE (a_editor)->ns_prefix_edit_entry),
                 PRIVATE (a_editor)->
                 ns_prefix_edit_entry_changed_handler_id);

        gtk_signal_handler_unblock
                (GTK_OBJECT
                 (PRIVATE (a_editor)->ns_uri_edit_entry),
                 PRIVATE (a_editor)->
                 ns_uri_edit_entry_changed_handler_id);
}


/**
 *mlview_namespace_editor_get_current_selected_ns
 *Gets the last namespace that has been graphicaly selected by the user. 
 *
 *@param a_editor the current instance of MlViewNamespaceEditor.
 *
 *
 */
xmlNs *mlview_namespace_editor_get_current_selected_ns
        (MlViewNamespaceEditor * a_editor) {
        MlViewNamespaceDesc *ns_desc = NULL;
        xmlNs *result = NULL;

        g_return_val_if_fail (a_editor != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_editor) != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_editor)->
                              current_selected_row >= 0, NULL);

        ns_desc = (MlViewNamespaceDesc *)
                gtk_clist_get_row_data (PRIVATE (a_editor)->
                                        namespaces,
                                        PRIVATE (a_editor)->
                                        current_selected_row);
        if (ns_desc != NULL)
                result = mlview_namespace_desc_get_ns (ns_desc);

        return result;
}


/**
 *
 *@param a_editor 
 *
 *
 */
xmlNode *mlview_namespace_editor_get_current_xml_node
        (MlViewNamespaceEditor * a_editor) {
        g_return_val_if_fail (a_editor != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_editor) != NULL, NULL);

        return PRIVATE (a_editor)->current_xml_node;
}


/**
 *
 *@param a_new_ns_prefix the prefix of the namespace to add 
 *to the current selected node.
 *@param a_new_ns_uri the uri of the namespace to add 
 *to the current selected node.
 *@param a_editor the current instance of MlViewNamespaceEditor.
 */
xmlNs *
mlview_namespace_editor_add_namespace_def (MlViewNamespaceEditor
                                           * a_editor,
                                           gchar * a_ns_prefix,
                                           gchar * a_ns_uri,
                                           xmlNode * a_xml_node)
{
        xmlNs *result = NULL;

        g_return_val_if_fail (a_editor != NULL, NULL);
        g_return_val_if_fail (a_ns_uri != NULL, NULL);
        g_return_val_if_fail (a_xml_node != NULL, NULL);

        result = xmlNewNs (a_xml_node, a_ns_uri, a_ns_prefix);

        if (result != NULL)
                mlview_namespace_editor_edit_namespace (a_editor,
                                                        result,
                                                        a_xml_node,
                                                        TRUE);
        return result;
}


/**
 *
 *@param a_ns the ns to remove from it
 *@param a_editor the current instance of MlViewNamespaceEditor
 *
 *
 */
void
 mlview_namespace_editor_remove_namespace_def
        (MlViewNamespaceEditor * a_editor, xmlNs * a_ns,
         xmlNode * a_xml_node) {
        xmlNs *unlinked_ns_def = NULL;
        MlViewNamespaceDesc *ns_desc = NULL;
        int i;

        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (a_ns != NULL);
        g_return_if_fail (a_xml_node);
        g_return_if_fail (PRIVATE (a_editor) != NULL);
        g_return_if_fail (PRIVATE (a_editor)->namespaces !=
                          NULL);

        unlinked_ns_def = xmlUnlinkNsDef (a_xml_node, a_ns);

        if (unlinked_ns_def == NULL) /*the namespace is not on this xml node */
                return;

        if (a_xml_node == PRIVATE (a_editor)->current_xml_node) {
                /*
                 *if the namespace unlinked_ns_def is on the 
                 *xml node that we are currently  visualizing, 
                 *delete the matching entry of the GtkCList
                 */
                for (i = 0;
                     i < PRIVATE (a_editor)->namespaces->rows;
                     i++) {
                        ns_desc =
                                gtk_clist_get_row_data
                                (PRIVATE (a_editor)->namespaces,
                                 i);

                        if (ns_desc == NULL)
                                continue;

                        if (ns_desc->ns_def == unlinked_ns_def) {
                                gtk_clist_remove
                                        (PRIVATE (a_editor)->
                                         namespaces, i);

                                gtk_signal_handler_block
                                        (GTK_OBJECT
                                         (PRIVATE
                                          (a_editor)->
                                          ns_prefix_edit_entry),
                                         PRIVATE (a_editor)->
                                         ns_prefix_edit_entry_changed_handler_id);

                                gtk_signal_handler_block
                                        (GTK_OBJECT
                                         (PRIVATE (a_editor)->
                                          ns_uri_edit_entry),
                                         PRIVATE (a_editor)->
                                         ns_uri_edit_entry_changed_handler_id);

                                gtk_entry_set_text
                                        (PRIVATE
                                         (a_editor)->
                                         ns_prefix_edit_entry,
                                         "");

                                gtk_entry_set_text
                                        (PRIVATE
                                         (a_editor)->
                                         ns_uri_edit_entry, "");

                                gtk_signal_handler_unblock
                                        (GTK_OBJECT
                                         (PRIVATE
                                          (a_editor)->
                                          ns_prefix_edit_entry),
                                         PRIVATE (a_editor)->
                                         ns_prefix_edit_entry_changed_handler_id);

                                gtk_signal_handler_unblock
                                        (GTK_OBJECT
                                         (PRIVATE
                                          (a_editor)->
                                          ns_uri_edit_entry),
                                         PRIVATE (a_editor)->
                                         ns_uri_edit_entry_changed_handler_id);

                                break;
                        }
                }
        }

        if (unlinked_ns_def != NULL) {

                xmlFreeNs (unlinked_ns_def);
                unlinked_ns_def = NULL;
        }
}



/*====================================
 *private callbacks
 *====================================*/

/**
 *
 *@param a_event the event that caused the selection
 *@param a_clist the clist considered.
 *@param a_row the selected row
 *@param a_editor the instance of MlViewNamespaceEditor that caused the edition
 *@param a_column the selected column
 *
 *
 */
static void
row_selected_cb (GtkCList * a_clist, gint a_row, gint a_column,
                 GdkEventButton * a_event,
                 MlViewNamespaceEditor * a_editor)
{
        gchar *ns_prefix = NULL,
                *ns_uri = NULL;
        xmlNode *node = NULL;
        gboolean is_local_ns = FALSE;
        MlViewNamespaceDesc *ns_desc;

        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (MLVIEW_IS_NAMESPACE_EDITOR (a_editor));
        g_return_if_fail (PRIVATE (a_editor) != NULL);

        PRIVATE (a_editor)->current_selected_row = a_row;

        gtk_signal_handler_block (GTK_OBJECT
                                  (PRIVATE (a_editor)->
                                   ns_prefix_edit_entry),
                                  PRIVATE (a_editor)->
                                  ns_prefix_edit_entry_changed_handler_id);
        gtk_signal_handler_block (GTK_OBJECT
                                  (PRIVATE (a_editor)->
                                   ns_uri_edit_entry),
                                  PRIVATE (a_editor)->
                                  ns_uri_edit_entry_changed_handler_id);

        gtk_entry_set_text (PRIVATE (a_editor)->
                            ns_prefix_edit_entry, "");
        gtk_entry_set_text (PRIVATE (a_editor)->
                            ns_uri_edit_entry, "");

        /*see if the namespace has been defined on the current node */
        ns_desc =
                mlview_namespace_editor_get_current_selected_ns_desc
                (a_editor);
        if (ns_desc != NULL) {
                node = mlview_namespace_desc_get_xml_node
                        (ns_desc);
                if (node == PRIVATE (a_editor)->current_xml_node)
                        is_local_ns = TRUE;
                else
                        is_local_ns = FALSE;
        }

        if (is_local_ns) {
                gtk_clist_get_text (PRIVATE (a_editor)->
                                    namespaces, a_row,
                                    NS_PREFIX_COLUMN,
                                    &ns_prefix);
                gtk_clist_get_text (PRIVATE (a_editor)->
                                    namespaces, a_row,
                                    NS_URI_COLUMN, &ns_uri);

                gtk_entry_set_text (PRIVATE (a_editor)->
                                    ns_prefix_edit_entry,
                                    ns_prefix);
                gtk_entry_set_text (PRIVATE (a_editor)->
                                    ns_uri_edit_entry, ns_uri);

        } else {
                PRIVATE (a_editor)->current_selected_row = -1;
        }

        gtk_signal_handler_unblock (GTK_OBJECT
                                    (PRIVATE (a_editor)->
                                     ns_prefix_edit_entry),
                                    PRIVATE (a_editor)->
                                    ns_prefix_edit_entry_changed_handler_id);
        gtk_signal_handler_unblock (GTK_OBJECT
                                    (PRIVATE (a_editor)->
                                     ns_uri_edit_entry),
                                    PRIVATE (a_editor)->
                                    ns_uri_edit_entry_changed_handler_id);
}


/**
 *Update the namespace prefix part to reflect the changes that occured.
 *Emits the signals "namespace-prefix-changed" and "namespace-changed". 
 *
 *@param a_entry 
 *@param a_editor 
 *
 *
 */
static void
ns_prefix_edit_entry_changed_cb (GtkEntry * a_entry,
                                 MlViewNamespaceEditor *
                                 a_editor)
{
        xmlNs *curr_ns;
        gchar *prefix;

        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (MLVIEW_IS_NAMESPACE_EDITOR (a_editor));
        g_return_if_fail (PRIVATE (a_editor) != NULL);
        g_return_if_fail (PRIVATE (a_editor)->namespaces !=
                          NULL);
        g_return_if_fail (PRIVATE (a_editor)->
                          current_selected_row >= 0);

        curr_ns =
                mlview_namespace_editor_get_current_selected_ns
                (a_editor);

        if (curr_ns == NULL)
                return;
        prefix = (guchar *) gtk_entry_get_text (a_entry);
        if (prefix != NULL) {
                if (curr_ns->prefix != NULL)
                        g_free ((gchar *) curr_ns->prefix);
                curr_ns->prefix = g_strdup (prefix);
                gtk_clist_set_text (PRIVATE (a_editor)->
                                    namespaces,
                                    PRIVATE (a_editor)->
                                    current_selected_row,
                                    NS_PREFIX_COLUMN,
                                    curr_ns->prefix);

        }
        g_signal_emit
                (G_OBJECT (a_editor),
                 mlview_namespace_editor_signals
                 [NAMESPACE_PREFIX_CHANGED], 0, NULL);
        g_signal_emit (G_OBJECT (a_editor),
                       mlview_namespace_editor_signals
                       [NAMESPACE_CHANGED], 0, NULL);
}

/**
 *
 *@param a_entry 
 *@param a_editor 
 *
 *
 */
static void
ns_uri_edit_entry_changed_cb (GtkEntry * a_entry,
                              MlViewNamespaceEditor * a_editor)
{
        xmlNs *curr_ns = NULL;
        guchar *uri = NULL;

        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (MLVIEW_IS_NAMESPACE_EDITOR (a_editor));
        g_return_if_fail (PRIVATE (a_editor) != NULL);
        g_return_if_fail (PRIVATE (a_editor)->
                          current_selected_row >= 0);

        curr_ns =
                mlview_namespace_editor_get_current_selected_ns
                (a_editor);

        if (curr_ns == NULL)
                return;
        uri = (guchar *) gtk_entry_get_text (a_entry);
        if (uri != NULL) {
                if (curr_ns->href != NULL)
                        g_free ((gchar *) curr_ns->href);
                curr_ns->href = g_strdup (uri);
                gtk_clist_set_text (PRIVATE (a_editor)->
                                    namespaces,
                                    PRIVATE (a_editor)->
                                    current_selected_row,
                                    NS_URI_COLUMN,
                                    curr_ns->href);
        }
        gtk_signal_emit (GTK_OBJECT (a_editor),
                         mlview_namespace_editor_signals
                         [NAMESPACE_URI_CHANGED]);
        gtk_signal_emit (GTK_OBJECT (a_editor),
                         mlview_namespace_editor_signals
                         [NAMESPACE_CHANGED]);
}

/**
 *
 *@param a_add_namespace_button 
 *@param a_editor 
 *
 *
 */
static void
add_namespace_button_clicked_cb (GtkButton *
                                 a_add_namespace_button,
                                 MlViewNamespaceEditor *
                                 a_editor)
{
        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (MLVIEW_IS_NAMESPACE_EDITOR (a_editor));

        mlview_namespace_editor_ask_namespace_to_user_and_add_it
                (a_editor);
}

/**
 *
 *@param a_remove_namespace_button 
 *@param a_editor 
 *
 *
 */
static void
remove_namespace_button_clicked_cb (GtkButton *
                                    a_remove_namespace_button,
                                    MlViewNamespaceEditor *
                                    a_editor)
{
        MlViewNamespaceDesc *ns_desc;

        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (MLVIEW_IS_NAMESPACE_EDITOR (a_editor));
        g_return_if_fail (PRIVATE (a_editor) != NULL);
        g_return_if_fail (PRIVATE (a_editor)->namespaces !=
                          NULL);
        g_return_if_fail (PRIVATE (a_editor)->
                          current_selected_row >= 0);

        ns_desc =
                gtk_clist_get_row_data (PRIVATE (a_editor)->
                                        namespaces,
                                        PRIVATE (a_editor)->
                                        current_selected_row);
        g_return_if_fail (ns_desc != NULL);
        g_return_if_fail (ns_desc->ns_def != NULL);
        g_return_if_fail (ns_desc->xml_node != NULL);

        mlview_namespace_editor_remove_namespace_def (a_editor,
                                                      ns_desc->
                                                      ns_def,
                                                      ns_desc->
                                                      xml_node);
}

/**
 *
 */
static void
namespace_desc_destroy_on_ns_editor_destroy (gpointer a_data)
{
        MlViewNamespaceDesc *ns_desc;

        ns_desc = (MlViewNamespaceDesc *) a_data;
        mlview_namespace_desc_destroy (ns_desc);
}
