/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 *Copyright 2001-2002 dodji SEKETELI, Gal CHAMOULAUD.
 *http://www.freespiders.org
 */

#include <sys/stat.h>

#include <libxml/tree.h>
#include <libxml/parser.h>

#include "mlview-settings-management.h"
#include "mlview-file-descriptor.h"
#include "mlview-app-context.h"
#include "mlview-parsing-utils.h"

#define MLVIEW_SETTINGS_DIALOG_DEFAULT_WIDTH 500
#define MLVIEW_SETTINGS_DIALOG_DEFAULT_HEIGHT 300
#define MLVIEW_SETTINGS_DIALOG_MAIN_PANED_PROPORTION 30

#define MLVIEW_SETTINGS_PERSONAL_SETTINGS_DIR_NAME ".mlview"
#define MLVIEW_SETTINGS_PERSONAL_SETTINGS_FILE_NAME "settings.xml"

#define PRIVATE(object) ((object)->private)

struct _MlViewSettingsEntryPrivate {
        /*The displayed name of this entry. */
        gchar *name;

        /*
         *A custom structure that holds the data of the settings. 
         *Each apps nows how to deal with it.
         *
         */

        /*Do not delete it !! it belongs to the apps !!! */
        gpointer *settings;

        /*The widget to display in the settings dialog box */
        GtkWidget *settings_widget;

        /*The children of the settings_widget. Can be NULL */
        /*Do not delete it !!! */
        gpointer settings_widgets_structure;

        /*A stucture that contains all the handlers need to 
         *perform load/update/notify tasks on the settings entry
         */
        /*Do not delete it !! there are functions pointers ... */
        MlViewSettingsEntryHandlers *handlers;

        /*is used by the settings manager. 
         *Is the root node of the settings entries tree.
         */
        GtkCTreeNode *parent_visual_node;
};


struct _MlViewSettingsManagerPrivate {
        /*the list of the settings entries */
        GList *settings_entries;

        /*
         *an hash table of the entries type. 
         *To ensure the uniqueness of the entries types
         */
        GHashTable *settings_entries_map;

        /*the root property window */
        GtkDialog *settings_dialog;

        GtkHPaned *paned_widget;

        GtkCTree *settings_entries_tree;

        /*FIXME: consider this field in init and destroy */
        GtkCTreeNode *current_selected_node;

        GtkNotebook *settings_notebook;

        /*the xml dom that holds the settings */
        xmlDoc *settings_xml_doc;

        MlViewAppContext *app_context;
};

/*====================================================================
 *declaration of the private functions used and defined in this file
 *==================================t=================================*/

static void
 tree_selected_row_cb (GtkCTree * a_tree, GList * a_node,
                       gint a_column, gpointer a_data);
static gboolean
        mlview_settings_entry_free_flat_struct
        (GNode * a_node, gpointer * a_entry);

static gboolean
        mlview_settings_manager_call_handlers_for_post_processing
        (GNode * a_node, gpointer a_user_data);

static void
 mlview_settings_manager_do_post_edition_processing
        (MlViewSettingsManager * a_manager);

static void
 mlview_settings_manager_do_pre_edition_processing
        (MlViewSettingsManager * a_manager);

static gboolean
        mlview_settings_manager_call_handlers_for_pre_processing
        (GNode * a_node, gpointer a_user_data);

static gchar
        * mlview_settings_manager_get_user_settings_dir (void);

static gchar
        * mlview_settings_manager_get_user_settings_file_name
        (void);

static gboolean mlview_settings_manager_home_dir_is_valid (void);

static gboolean
mlview_settings_manager_personal_settings_dir_exists (void);

static gint
mlview_settings_manager_create_personal_settings_dir (void);

static xmlDoc
        *
        mlview_settings_manager_create_personal_settings_xml_docs
        (void);

static gboolean
        mlview_settings_manager_call_handlers_to_notify_settings_loading_from_xml_doc
        (GNode * a_node, gpointer a_user_data);
/*========================================================================*/


static gboolean
mlview_settings_entry_free_flat_struct (GNode * a_node,
                                        gpointer * a_entry)
{
        MlViewSettingsEntry *entry = NULL;

        g_return_val_if_fail (a_node != NULL, FALSE);
        g_return_val_if_fail (a_entry != NULL, FALSE);

        entry = a_node->data;

        if (entry && PRIVATE (entry)
            && PRIVATE (entry)->name) {
                g_free (PRIVATE (entry)->name);
        }

        return FALSE;
}

/**
 *Builds a new settings entry. 
 *
 *@param a_settings_widget_structure 
 *a custom structure which fields must 
 *contains the children widgets of the a_settings_widget if any. 
 **MUST NOT* be freed by the caller. Has the fields of this struct
 *contains *ONLY* fields that are children of a_settings_widget, 
 *they will be free when a_settings_widget is free. 
 *
 *@param a_settings_widget the widget that will be shown in the settings dialog, 
 *for this settings entry.Must *NOT* be freed by the caller. 
 *@param a_name the displayed name of the entry. Must be free be the caller.
 *@param a_settings_struct a points to a pointer to the custom struct 
 *that holds the settings. Must be freed by the caller. 
 *Once freed, The pointer
 *to the settings custom struct MUST be set to NULL, 
 *so that the settings manager knows it is freed. 
 */
MlViewSettingsEntry *
mlview_settings_entry_new (gchar * a_name,
                           gpointer a_settings_struct,
                           GtkWidget * a_settings_widget,
                           gpointer a_settings_widget_structure)
{
        MlViewSettingsEntry *result;

        g_return_val_if_fail (a_name != NULL, NULL);
        g_return_val_if_fail (a_settings_widget != NULL, NULL);

        result = g_malloc0 (sizeof (MlViewSettingsEntry));

        result->tree_node = g_node_new (result);

        PRIVATE (result) =
                g_malloc0 (sizeof (MlViewSettingsEntryPrivate));

        PRIVATE (result)->name = g_strdup (a_name);

        PRIVATE (result)->settings = a_settings_struct;

        PRIVATE (result)->settings_widget = a_settings_widget;

        PRIVATE (result)->settings_widgets_structure =
                a_settings_widget_structure;

        PRIVATE (result)->handlers =
                g_malloc0 (sizeof (MlViewSettingsEntryHandlers));

        return result;
}


/**
 *
 *@param a_settings_entry the current instance of MlViewSettings
 *
 *
 *@return the readable name of the settings entry. 
 */
gchar *
mlview_settings_entry_get_name (MlViewSettingsEntry *
                                a_settings_entry)
{
        g_return_val_if_fail (a_settings_entry != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_settings_entry) != NULL,
                              NULL);

        return PRIVATE (a_settings_entry)->name;
}


/**
 *Sets the name of the entry to a_name. 
 *
 *@param a_name the name to set. 
 *Note that the caller can/must free this pointer.
 *@param a_settings_entry the current instance 
 *of MlViewSettingsEntry.
 */
void
 mlview_settings_entry_set_name
        (MlViewSettingsEntry * a_settings_entry, gchar * a_name)
{
        g_return_if_fail (a_settings_entry != NULL);
        g_return_if_fail (PRIVATE (a_settings_entry) != NULL);

        PRIVATE (a_settings_entry)->name = g_strdup (a_name);
}



/**
 *
 *@param a_settings_entry the current instance of MlViewSettingsEntry.
 *
 *
 *@return the settings struct set to this entry. Can return NULL if a NULL settings has been set or 
 */
void *
mlview_settings_entry_get_settings (MlViewSettingsEntry *
                                    a_settings_entry)
{
        g_return_val_if_fail (a_settings_entry != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_settings_entry) != NULL,
                              NULL);

        return PRIVATE (a_settings_entry)->settings;
}


/**
 *Associates a custom settings struct to this settings entry. 
 *
 *@param a_settings the custom data structure the caller wishes to associate to this settings entry. Note that the caller is responsible offreeing this structure but be carefull. 
 *@param a_settings_entry the current intance of MlViewSettingsEntry.
 *
 *
 */
void
mlview_settings_entry_set_settings (MlViewSettingsEntry *
                                    a_settings_entry,
                                    void *a_settings)
{
        g_return_if_fail (a_settings_entry != NULL);
        g_return_if_fail (PRIVATE (a_settings_entry) != NULL);

        PRIVATE (a_settings_entry)->settings = a_settings;
}


/**
 *
 *@param a_settings_entry 
 *
 *
 *@return the widget that will be displayed by the settings manager to edit this settings entry. 
 */
GtkWidget *
mlview_settings_entry_get_settings_widget (MlViewSettingsEntry *
                                           a_settings_entry)
{
        g_return_val_if_fail (a_settings_entry != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_settings_entry) != NULL,
                              NULL);

        return PRIVATE (a_settings_entry)->settings_widget;
}


/**
 *
 *@param a_settings_entry the current instance of MlViewSettingsEntry.
 *
 *
 *@return the structure from wich the caller can access the fields of the settings entry widget. 
 */
void *mlview_settings_entry_get_settings_widgets_struct
        (MlViewSettingsEntry * a_settings_entry) {
        g_return_val_if_fail (a_settings_entry != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_settings_entry) != NULL,
                              NULL);

        return PRIVATE (a_settings_entry)->
                settings_widgets_structure;
}


/**
 *
 *@return a pointer to the internal handler 
 *structure of MlViewSettingsEntry ... 
 *so make sure you know what you are doing.
 */
MlViewSettingsEntryHandlers *
mlview_settings_entry_get_handlers (MlViewSettingsEntry *
                                    a_settings_entry)
{
        g_return_val_if_fail (a_settings_entry != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_settings_entry) != NULL,
                              NULL);

        return PRIVATE (a_settings_entry)->handlers;
}

/**
 *Associates the handlers to the settings entry. 
 *
 *@param a_handlers the new handlers
 *@param a_settings_entry the current instance of MlViewSettingsEntry.
 *
 *
 */
void
mlview_settings_entry_set_handlers (MlViewSettingsEntry *
                                    a_settings_entry,
                                    MlViewSettingsEntryHandlers *
                                    a_handlers)
{
        g_return_if_fail (a_settings_entry != NULL);
        g_return_if_fail (PRIVATE (a_settings_entry) != NULL);

        PRIVATE (a_settings_entry)->handlers = a_handlers;
}




/**
 *
 *@param a_parent_entry 
 *@param a_child_entry 
 *
 *
 */
void
mlview_settings_add_child_entry (MlViewSettingsEntry *
                                 a_parent_entry,
                                 MlViewSettingsEntry *
                                 a_child_entry)
{
        g_return_if_fail (a_parent_entry != NULL);
        g_return_if_fail (a_child_entry != NULL);

        g_node_append (a_parent_entry->tree_node,
                       a_child_entry->tree_node);
}


/**
 *
 *@param a_sibling_entry 
 *@param a_parent_entry 
 *@param a_node 
 *
 *
 */
void
mlview_settings_add_previous_sibling_entry (MlViewSettingsEntry *
                                            a_parent_entry,
                                            MlViewSettingsEntry *
                                            a_sibling_entry,
                                            MlViewSettingsEntry *
                                            a_node)
{
        g_return_if_fail (a_parent_entry != NULL);
        g_return_if_fail (a_sibling_entry != NULL);
        g_return_if_fail (a_node != NULL);

        g_node_insert_before (a_parent_entry->tree_node,
                              a_sibling_entry->tree_node,
                              a_node->tree_node);
}

/**
 *
 *@param a_entry 
 *
 *
 */
MlViewSettingsEntry *
mlview_settings_unlink_entry (MlViewSettingsEntry * a_entry)
{
        g_return_val_if_fail (a_entry != NULL, NULL);

        g_node_unlink (a_entry->tree_node);

        if (a_entry->tree_node != NULL
            && a_entry->tree_node->parent != NULL)
                return a_entry;
        return NULL;
}

/**
 *
 *@param a_settings_entry 
 *
 *
 */
void
mlview_settings_entry_destroy (MlViewSettingsEntry *
                               a_settings_entry)
{
        g_return_if_fail (a_settings_entry != NULL);

        g_node_traverse (a_settings_entry->tree_node,
                         G_PRE_ORDER, G_TRAVERSE_ALL, -1,
                         (GNodeTraverseFunc)
                         mlview_settings_entry_free_flat_struct,
                         a_settings_entry);

        g_node_destroy (a_settings_entry->tree_node);
}


/**
 *
 *@param a_handlers 
 *@param a_settings_entry 
 *
 *
 */
void
 mlview_settings_entry_entry_set_handlers
        (MlViewSettingsEntry * a_settings_entry,
         MlViewSettingsEntryHandlers * a_handlers) {
        g_return_if_fail (a_settings_entry != NULL);
        g_return_if_fail (PRIVATE (a_settings_entry) != NULL);

        PRIVATE (a_settings_entry)->handlers = a_handlers;
}

/*=======================================================
 *MlViewSettingsManager methods and helper functions
 *=======================================================*/

static void
tree_selected_row_cb (GtkCTree * a_tree, GList * a_node,
                      gint a_column, gpointer a_data)
{
        MlViewSettingsEntry *entry = NULL;
        MlViewSettingsManager *manager = a_data;
        gint notebook_page_num = 0;

        g_return_if_fail (a_tree != NULL);
        g_return_if_fail (a_node != NULL);
        g_return_if_fail (a_data != NULL);
        g_return_if_fail (manager != NULL);

        entry = gtk_ctree_node_get_row_data
                (a_tree, GTK_CTREE_NODE (a_node));

        g_return_if_fail (entry != NULL);
        g_return_if_fail (PRIVATE (entry) != NULL);
        g_return_if_fail (PRIVATE (manager) != NULL);

        if (PRIVATE (entry)->settings_widget == NULL
            || PRIVATE (manager)->settings_notebook == NULL)
                return;

        notebook_page_num =
                gtk_notebook_page_num (PRIVATE (manager)->
                                       settings_notebook,
                                       PRIVATE (entry)->
                                       settings_widget);

        if (notebook_page_num == -1)
                return;

        gtk_notebook_set_page (PRIVATE (manager)->
                               settings_notebook,
                               notebook_page_num);
}


MlViewSettingsManager *
mlview_settings_manager_new (MlViewAppContext * a_app_context)
{
        MlViewSettingsManager *result = NULL;

        result = g_malloc0 (sizeof (MlViewSettingsManager));

        PRIVATE (result) = g_malloc0
                (sizeof (MlViewSettingsManagerPrivate));

        PRIVATE (result)->settings_entries_map =
                g_hash_table_new (g_direct_hash, g_direct_equal);

        PRIVATE (result)->app_context = a_app_context;

        /*Create the settings dialog window */
        PRIVATE (result)->settings_dialog =
                GTK_DIALOG (gtk_dialog_new_with_buttons
                            (_("MlView XML Editor Settings"),
                             NULL, GTK_DIALOG_MODAL, _("OK"),
                             GTK_RESPONSE_ACCEPT, _("Cancel"),
                             GTK_RESPONSE_REJECT, NULL));

        gtk_window_set_wmclass (GTK_WINDOW
                                (PRIVATE (result)->
                                 settings_dialog),
                                "editor-settings-dialog",
                                "MlView");

        gtk_window_set_policy (GTK_WINDOW
                               (PRIVATE (result)->
                                settings_dialog), FALSE, TRUE,
                               FALSE);

        gtk_widget_set_usize (GTK_WIDGET
                              (PRIVATE (result)->
                               settings_dialog),
                              MLVIEW_SETTINGS_DIALOG_DEFAULT_WIDTH,
                              MLVIEW_SETTINGS_DIALOG_DEFAULT_HEIGHT);

        /*create the main paned widget of the settiings dialog */
        PRIVATE (result)->paned_widget =
                GTK_HPANED (gtk_hpaned_new ());

        /*
         *the right side notebook that holds 
         *the settings edition views/widgets
         */
        PRIVATE (result)->settings_notebook =
                GTK_NOTEBOOK (gtk_notebook_new ());
        gtk_notebook_set_show_tabs (PRIVATE (result)->
                                    settings_notebook, FALSE);

        /*the left side settings choice tree */
        PRIVATE (result)->settings_entries_tree =
                GTK_CTREE (gtk_ctree_new (1, 0));

        gtk_signal_connect (GTK_OBJECT
                            (PRIVATE (result)->
                             settings_entries_tree),
                            "tree-select-row",
                            GTK_SIGNAL_FUNC
                            (tree_selected_row_cb), result);

        /*Pack the widgets */
        gtk_box_pack_start (GTK_BOX
                            (PRIVATE (result)->settings_dialog->
                             vbox),
                            GTK_WIDGET (PRIVATE (result)->
                                        paned_widget), TRUE,
                            TRUE, 0);

        gtk_paned_add1 (GTK_PANED
                        (PRIVATE (result)->paned_widget),
                        GTK_WIDGET (PRIVATE (result)->
                                    settings_entries_tree));

        gtk_paned_add2 (GTK_PANED
                        (PRIVATE (result)->paned_widget),
                        GTK_WIDGET (PRIVATE (result)->
                                    settings_notebook));

        gtk_widget_show_all (GTK_WIDGET
                             (PRIVATE (result)->paned_widget));

        return result;
}

/**
 *
 *@param a_manager the current settings manager.
 *
 *
 *@return the 
 */
GtkDialog *mlview_settings_manager_get_settings_dialog
        (MlViewSettingsManager * a_manager) {
        g_return_val_if_fail (a_manager != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_manager) != NULL, NULL);

        return PRIVATE (a_manager)->settings_dialog;
}


/**
 *Sets the proportions of the main GtkHPaned widget of the settings dialog. 
 *
 *@param a_manager the settings manager we are talking about.
 *@param a_percentage the percentage of the parent widget taken by the firstpartf of the GtkHPaned. Must be less than or equal to 100. 
 *
 *
 */
void
 mlview_settings_manager_set_settings_dialog_proportions
        (MlViewSettingsManager * a_manager, guint a_percentage) {
        /*the GtkPaned separator position. */
        guint separator_position;
        GtkWidget *top_level_widget = NULL;

        g_return_if_fail (a_manager != NULL);
        g_return_if_fail (a_percentage <= 100);
        g_return_if_fail (PRIVATE (a_manager) != NULL);
        g_return_if_fail (PRIVATE (a_manager)->paned_widget !=
                          NULL);
        g_return_if_fail (PRIVATE (a_manager)->settings_dialog !=
                          NULL);

        top_level_widget = gtk_widget_get_toplevel
                (GTK_WIDGET
                 (PRIVATE (a_manager)->settings_dialog));

        if (top_level_widget == NULL)
                return;

        separator_position =
                top_level_widget->allocation.width *
                a_percentage / 100;

        gtk_paned_set_position (GTK_PANED
                                (PRIVATE (a_manager)->
                                 paned_widget),
                                separator_position);

        gtk_widget_show_all (GTK_WIDGET
                             (PRIVATE (a_manager)->
                              settings_dialog));
}


/**
 *Installs the settings entry into the mlview settings edition 
 *management system. 
 *
 *@param a_entry the entry to install in the settings dialog.
 *@param a_settings_manager the current instance of MlViewSettingsManager.
 *
 *@return 0 if everythings went ok, -1 
 *if that entry exists already, and -2 if at least one 
 */
gint mlview_settings_manager_install_settings
        (MlViewSettingsManager * a_settings_manager,
         MlViewSettingsEntry * a_entry) {
        MlViewSettingsEntry *cur_entry = NULL;
        gchar *text[2] = { 0 };

        g_return_val_if_fail (a_settings_manager != NULL, -2);

        g_return_val_if_fail (a_entry != NULL, -2);

        g_return_val_if_fail (a_entry->tree_node != NULL, -3);

        g_return_val_if_fail (PRIVATE (a_settings_manager) !=
                              NULL, -3);

        g_return_val_if_fail (PRIVATE
                              (a_settings_manager)->
                              settings_entries_map != NULL, -3);

        /*first check if this an entry of the same type does not exist already */
        if (g_hash_table_lookup
            (PRIVATE (a_settings_manager)->settings_entries_map,
             a_entry)
            != NULL)
                return -1;

        cur_entry = a_entry;

        while (cur_entry) {
                GtkCTreeNode *visual_node = NULL;

                text[0] = PRIVATE (cur_entry)->name;

                visual_node =
                        gtk_ctree_insert_node
                        (PRIVATE (a_settings_manager)->
                         settings_entries_tree,
                         PRIVATE (cur_entry)->parent_visual_node,
                         NULL, text, 0, NULL, NULL, NULL, NULL,
                         FALSE, TRUE);

                g_return_val_if_fail (visual_node != NULL, -1);

                gtk_ctree_node_set_row_data
                        (PRIVATE (a_settings_manager)->
                         settings_entries_tree, visual_node,
                         cur_entry);

                if (cur_entry->tree_node->children) {
                        MlViewSettingsEntry *child_entry = NULL;

                        child_entry = cur_entry->tree_node->data;

                        PRIVATE (child_entry)->
                                parent_visual_node = visual_node;

                        mlview_settings_manager_install_settings
                                (a_settings_manager,
                                 child_entry);

                        PRIVATE (child_entry)->
                                parent_visual_node = NULL;
                }

                /*
                 *get the settings edition view and pack 
                 *it into the right side notebook
                 */
                if (PRIVATE (cur_entry)
                    && PRIVATE (cur_entry)->settings_widget
                    && PRIVATE (a_settings_manager)->
                    settings_notebook) {
                        gtk_notebook_append_page (PRIVATE
                                                  (a_settings_manager)->
                                                  settings_notebook,
                                                  PRIVATE
                                                  (cur_entry)->
                                                  settings_widget,
                                                  NULL);
                }

                if (cur_entry->tree_node->next)
                        cur_entry =
                                cur_entry->tree_node->next->data;
                else
                        cur_entry = NULL;
        }

        PRIVATE (a_settings_manager)->settings_entries =
                g_list_prepend (PRIVATE (a_settings_manager)->
                                settings_entries, a_entry);

        g_hash_table_insert (PRIVATE (a_settings_manager)->
                             settings_entries_map, a_entry,
                             PRIVATE (a_settings_manager)->
                             settings_entries);
        return 0;
}

/**
 *
 */
MlViewSettingsEntry *mlview_settings_manager_uninstall_settings
        (MlViewSettingsManager * a_settings_manager,
         MlViewSettingsEntry * a_entry) {
        GList *entries_list_element = NULL;
        GtkCTreeNode *visual_node = NULL;

        g_return_val_if_fail (a_settings_manager != NULL, NULL);
        g_return_val_if_fail (a_entry != NULL, NULL);

        if ((entries_list_element =
             g_hash_table_lookup (PRIVATE
                                  (a_settings_manager)->
                                  settings_entries_map,
                                  a_entry)) == NULL) {
                return NULL;
        }

        /*first, remove the settings entry from the visual tree. */
        visual_node = gtk_ctree_find_by_row_data
                (PRIVATE (a_settings_manager)->
                 settings_entries_tree, NULL,
                 entries_list_element->data);

        gtk_ctree_remove_node
                (PRIVATE (a_settings_manager)->
                 settings_entries_tree, visual_node);
        /*removed the settings from the visual tree */

        /*now, remove the settings from the entries linked list */
        PRIVATE (a_settings_manager)->settings_entries =
                g_list_remove_link
                (PRIVATE (a_settings_manager)->settings_entries,
                 entries_list_element);

        /*now, remove the settings from the hash table */
        g_hash_table_remove
                (PRIVATE (a_settings_manager)->
                 settings_entries_map, a_entry);

        return a_entry;
}


/**
 *Displays the settings dialog box and allow the user to edit the setting 
 *
 *@param a_manager the current settings manager.
 *
 *
 */
void
 mlview_settings_manager_edit_settings_interactive
        (MlViewSettingsManager * a_manager) {
        GtkDialog *dialog;
        gint button;

        dialog = mlview_settings_manager_get_settings_dialog
                (a_manager);

        g_return_if_fail (dialog != NULL);

        gtk_widget_realize (GTK_WIDGET (dialog));
        mlview_settings_manager_set_settings_dialog_proportions
                (a_manager,
                 MLVIEW_SETTINGS_DIALOG_MAIN_PANED_PROPORTION);
        mlview_settings_manager_do_pre_edition_processing
                (a_manager);

        mlview_app_context_set_window_icon (PRIVATE (a_manager)->
                                            app_context,
                                            GTK_WINDOW (dialog));
        button = gtk_dialog_run (dialog);

        switch (button) {
        case GTK_RESPONSE_ACCEPT: /*OK Button */
                mlview_settings_manager_do_post_edition_processing
                        (a_manager);
                gtk_widget_hide (GTK_WIDGET (dialog));
                break;
        case GTK_RESPONSE_REJECT: /*CANCEL Button */
                gtk_widget_hide (GTK_WIDGET (dialog));
                break;
        case GTK_RESPONSE_CLOSE: /*dialog close via Window manager */
                gtk_widget_hide (GTK_WIDGET (dialog));
                break;
        default:               /*Just in case ... */
                gtk_widget_hide (GTK_WIDGET (dialog));
                break;
        }
}


/**
 *Creates the settings.xml file in the personal .mlview directory. 
 *
 *@param a_xml_doc the xml doc created. is Set to NULL, if it could not be created or if the file exists already.
 *@param a_manager the current instance of MlViewSettignsManager
 *
 *
 *@return 0 created if everything is ok and -1 if it could not create the file. 
 */
gint
mlview_settings_manager_create_personal_settings_file (xmlDoc **
                                                       a_xml_doc)
{
        gchar *settings_file;
        xmlDoc *xml_doc;
        MlViewFileDescriptor *file_desc = NULL;

        *a_xml_doc = NULL;
        if (mlview_settings_manager_home_dir_is_valid () ==
            FALSE)
                return -1;

        if (mlview_settings_manager_personal_settings_dir_exists
            () == FALSE)
                mlview_settings_manager_create_personal_settings_dir
                        ();

        if (mlview_settings_manager_personal_settings_file_exists
            ())
                return 0;

        settings_file =
                mlview_settings_manager_get_user_settings_file_name
                ();
        file_desc = mlview_file_descriptor_new (settings_file);

        if (mlview_file_descriptor_create_file
            (file_desc, S_IWUSR | S_IRUSR)) {
                mlview_file_descriptor_destroy (file_desc);
                g_free (settings_file);
                return -1;
        }

        xml_doc =
                mlview_settings_manager_create_personal_settings_xml_docs
                ();
        xmlIndentTreeOutput = 1;
        xmlSaveFormatFile (settings_file, xml_doc, 2);
        *a_xml_doc = xml_doc;

        mlview_file_descriptor_destroy (file_desc);
        g_free (settings_file);

        return 0;
}


/**
 *Loads the xml settings file from the personal .mlview dir, and parses it. 
 *
 *@return the xml doc data structure. 
 */
xmlDoc *mlview_settings_manager_load_settings_from_disk
        (MlViewSettingsManager * a_manager) {
        gchar *settings_file = NULL;
        xmlDoc *result = NULL;

        g_return_val_if_fail (a_manager != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_manager) != NULL, NULL);

        if (!mlview_settings_manager_personal_settings_file_exists ())
                return NULL;
        settings_file =
                mlview_settings_manager_get_user_settings_file_name
                ();
        result = xmlParseFile (settings_file);

        PRIVATE (a_manager)->settings_xml_doc = result;

        xmlCleanupParser ();

        return result;
}

/**
 *Saves the settings contained in the in memory dom 
 *associated to the current Settings manager
 *on the disk, in the current user home dir. 
 *
 *@param a_manager the current instance of Settings manager.
 *
 *
 */
void
 mlview_settings_manager_save_settings_to_disk
        (MlViewSettingsManager * a_manager) {
        gchar *settings_file_name;

        g_return_if_fail (a_manager != NULL);
        g_return_if_fail (PRIVATE (a_manager) != NULL);
        g_return_if_fail (PRIVATE (a_manager) != NULL);

        if (!PRIVATE (a_manager)->settings_xml_doc)
                return;

        settings_file_name =
                mlview_settings_manager_get_user_settings_file_name
                ();

        mlview_parsing_utils_save_xml_doc
                (PRIVATE (a_manager)->settings_xml_doc,
                 settings_file_name,
                 PRIVATE (a_manager)->app_context);

}


/**
 *
 *@return TRUE if the user mlview settings file exists and is writeable. 
 */
gboolean
mlview_settings_manager_personal_settings_file_exists (void)
{
        gboolean is_writeable;
        gint error;
        gchar *settings_file;
        MlViewFileDescriptor *file_desc;

        if (mlview_settings_manager_personal_settings_dir_exists
            () == FALSE)
                return FALSE;
        settings_file =
                mlview_settings_manager_get_user_settings_file_name
                ();
        file_desc = mlview_file_descriptor_new (settings_file);
        g_free (settings_file);

        error = mlview_file_descriptor_is_writeable (file_desc,
                                                     &is_writeable);
        mlview_file_descriptor_destroy (file_desc);

        if (!error && (is_writeable == TRUE))
                return TRUE;
        return FALSE;
}


/**
 *Instructs the settings manager to call settings 
 *entries post settings install handler.
 *The handler called 
 *is: load_settings_from_xml_doc_into_settings_entry() 
 *(See the #MlViewSettingsEntryHandlers struct) 
 */
void
 mlview_settings_manager_post_settings_install_init
        (MlViewSettingsManager * a_manager,
         xmlDoc * a_settings_doc) {
        MlViewSettingsEntry *settings_entry;
        GList *list;

        g_return_if_fail (a_manager != NULL);
        g_return_if_fail (PRIVATE (a_manager) != NULL);
        g_return_if_fail (PRIVATE (a_manager)->
                          settings_entries != NULL);
        g_return_if_fail (a_settings_doc != NULL);

        list = g_list_first (PRIVATE (a_manager)->
                             settings_entries);
        if (list == NULL)
                return;

        PRIVATE (a_manager)->settings_xml_doc = a_settings_doc;

        /*
         *walk thru the list of settings entry and
         *call their appropriate callbacks.
         */
        settings_entry = list->data;
        while (settings_entry) {
                g_node_traverse
                        (settings_entry->tree_node,
                         G_POST_ORDER, G_TRAVERSE_ALL, -1,
                         mlview_settings_manager_call_handlers_to_notify_settings_loading_from_xml_doc,
                         a_settings_doc);

                list = g_list_next (list);

                settings_entry =
                        (list != NULL) ? list->data : NULL;
        }
}




/*
 *private helper functions used by the exported functions of this file.
 */


/**
 *Does the processing necesary after the user has edited the settings i.e calls the different hanlders. 
 *
 *@param a_manager the current instance of MlViewSettingsManager
 *
 *
 */
static void
 mlview_settings_manager_do_post_edition_processing
        (MlViewSettingsManager * a_manager) {
        MlViewSettingsEntry *settings_entry;
        GList *list;

        g_return_if_fail (a_manager != NULL);
        g_return_if_fail (PRIVATE (a_manager) != NULL);
        g_return_if_fail (PRIVATE (a_manager)->
                          settings_entries != NULL);

        list = g_list_first (PRIVATE (a_manager)->
                             settings_entries);

        if (list == NULL)
                return;

        /*
         *walk thru the list of settings entry 
         *and call their appropriate callbacks.
         */
        settings_entry = list->data;

        while (settings_entry) {
                g_node_traverse
                        (settings_entry->tree_node,
                         G_POST_ORDER, G_TRAVERSE_ALL, -1,
                         mlview_settings_manager_call_handlers_for_post_processing,
                         settings_entry);

                list = g_list_next (list);

                settings_entry =
                        (list != NULL) ? list->data : NULL;
        }

        /*Save the new settings on disk. 
         *FIXME: maybe some optims could be done
         *so that we don't save the settings if they have not been modified for example.*/
        mlview_settings_manager_save_settings_to_disk
                (a_manager);
}


/**
 *This helper function is called on each node of a settings entries tree. The sequence of call is determined
 *in. Note that this function must be call on the children nodes bofore on the node itself.
 *Note also that this function is a callback called by the function g_node_traverse of the glib. It therefore
 *has the siganture of GNodeTraverseFunc. 
 *
 *@param a_node the n-ary tree node of the settings entries tree.
 *@param a_user_data 
 *
 *
 */
static gboolean
mlview_settings_manager_call_handlers_for_post_processing (GNode
                                                           *
                                                           a_node,
                                                           gpointer
                                                           a_user_data)
{
        MlViewSettingsEntry *settings_entry;
        MlViewSettingsEntryHandlers *handlers;

        g_return_val_if_fail (a_node != NULL, FALSE);
        settings_entry = a_node->data;
        g_return_val_if_fail (settings_entry != NULL, FALSE);

        handlers =
                mlview_settings_entry_get_handlers
                (settings_entry);
        if (handlers) {
                if (handlers->get_settings_from_settings_widget) {
                        handlers->
                                get_settings_from_settings_widget
                                (settings_entry);
                }
                if (handlers->notify_settings_changed)
                        handlers->
                                notify_settings_changed
                                (settings_entry,
                                 handlers->
                                 settings_changed_notifier_data);
        }
        return FALSE;
}

/**
 *Called by the settings manager to notify the loading of settings. 
 *
 */
static gboolean
        mlview_settings_manager_call_handlers_to_notify_settings_loading_from_xml_doc
        (GNode * a_node, gpointer a_user_data) {

        MlViewSettingsEntry *settings_entry;
        MlViewSettingsEntryHandlers *handlers;
        xmlDoc *xml_doc;

        g_return_val_if_fail (a_node != NULL, FALSE);
        settings_entry = a_node->data;
        g_return_val_if_fail (settings_entry != NULL, FALSE);
        xml_doc = a_user_data;
        g_return_val_if_fail (xml_doc != NULL, FALSE);

        handlers =
                mlview_settings_entry_get_handlers
                (settings_entry);
        if (handlers) {
                if (handlers->
                    load_settings_from_xml_doc_into_settings_entry)
                        handlers->
                                load_settings_from_xml_doc_into_settings_entry
                                (settings_entry, xml_doc);
        }
        return FALSE;
}

/**
 *Called by the settings manager before the settings edition. 
 *
 */
static void
 mlview_settings_manager_do_pre_edition_processing
        (MlViewSettingsManager * a_manager) {
        MlViewSettingsEntry *settings_entry = NULL;
        GList *list;

        g_return_if_fail (a_manager != NULL);
        g_return_if_fail (PRIVATE (a_manager) != NULL);
        g_return_if_fail (PRIVATE (a_manager)->
                          settings_entries != NULL);

        list = g_list_first (PRIVATE (a_manager)->
                             settings_entries);

        if (list == NULL)
                return;

        /*
         *walk thru the list of settings 
         *entry and call their appropriate callbacks.
         */
        settings_entry = list->data;

        while (settings_entry) {
                g_node_traverse
                        (settings_entry->tree_node,
                         G_POST_ORDER, G_TRAVERSE_ALL, -1,
                         mlview_settings_manager_call_handlers_for_pre_processing,
                         settings_entry);

                list = g_list_next (list);

                settings_entry =
                        (list != NULL) ? list->data : NULL;
        }
}

/**
 *This helper function is called on each node of a 
 *settings entries tree. The sequence of call is determined
 *in. Note that this function must be call on the children 
 *nodes bofore on the node itself.
 *Note also that this function is a 
 *callback called by the function g_node_traverse of the glib. It therefore
 *has the siganture of GNodeTraverseFunc. 
 *
 *@param a_node the n-ary tree node of the settings entries tree.
 *@param a_user_data 
 *
 *
 */
static gboolean
        mlview_settings_manager_call_handlers_for_pre_processing
        (GNode * a_node, gpointer a_user_data) {
        MlViewSettingsEntry *settings_entry = NULL;
        MlViewSettingsEntryHandlers *handlers = NULL;

        g_return_val_if_fail (a_node != NULL, FALSE);

        settings_entry = a_node->data;

        g_return_val_if_fail (settings_entry != NULL, FALSE);

        handlers =
                mlview_settings_entry_get_handlers
                (settings_entry);

        if (handlers) {
                if (handlers->set_settings_to_settings_widget) {
                        handlers->set_settings_to_settings_widget
                                (settings_entry);
                }
        }

        return FALSE;
}

/**
 *
 *@return the full path  of the user mlview settings directory. 
 *The caller has to free the returned pointer. 
 */
static gchar *
mlview_settings_manager_get_user_settings_dir (void)
{
        gchar *home_dir,
        *settings_dir;

        home_dir = getenv ("HOME");

        g_return_val_if_fail (home_dir != NULL, NULL);

        settings_dir = g_strconcat (home_dir, "/",
                                    MLVIEW_SETTINGS_PERSONAL_SETTINGS_DIR_NAME,
                                    NULL);

        g_return_val_if_fail (settings_dir != NULL, NULL);


        return settings_dir;
}

/**
 *
 *@return the full path of the user mlview settings file. 
 *The caller has to free the resulting pointer. 
 */
static gchar *
mlview_settings_manager_get_user_settings_file_name (void)
{

        gchar *settings_dir,
        *settings_file;

        settings_dir =
                mlview_settings_manager_get_user_settings_dir ();

        g_return_val_if_fail (settings_dir != NULL, NULL);

        settings_file =
                g_strconcat (settings_dir, "/",
                             MLVIEW_SETTINGS_PERSONAL_SETTINGS_FILE_NAME,
                             NULL);

        g_free (settings_dir);

        g_return_val_if_fail (settings_file != NULL, NULL);

        return settings_file;
}


/**
 *
 *@return TRUE if the user home dir exists and is 
 *writeable by the current process; returns FALSE if not. 
 */
static gboolean
mlview_settings_manager_home_dir_is_valid (void)
{
        gchar *home_dir = NULL;
        gboolean is_dir,
         is_writeable;

        MlViewFileDescriptor *file_desc = NULL;

        home_dir = getenv ("HOME");
        g_return_val_if_fail (home_dir != NULL, FALSE);

        file_desc = mlview_file_descriptor_new (home_dir);
        home_dir = NULL;

        /*test home dir validity */
        mlview_file_descriptor_is_dir (file_desc, &is_dir);
        mlview_file_descriptor_is_writeable (file_desc,
                                             &is_writeable);

        mlview_file_descriptor_destroy (file_desc);
        file_desc = NULL;

        if (is_dir == FALSE || is_writeable == FALSE) {
                return FALSE;
        }

        return TRUE;
}

/**
 *
 *@return TRUE if the user mlview settings dir exists and is writeable. 
 */
static gboolean
mlview_settings_manager_personal_settings_dir_exists (void)
{
        gchar *home_dir = NULL,
                *settings_dir = NULL;
        gboolean is_dir,
         result;

        MlViewFileDescriptor *file_desc = NULL;

        if (mlview_settings_manager_home_dir_is_valid () ==
            FALSE)
                return FALSE;

        home_dir = getenv ("HOME");
        settings_dir = g_strconcat (home_dir, "/",
                                    MLVIEW_SETTINGS_PERSONAL_SETTINGS_DIR_NAME,
                                    NULL);
        home_dir = NULL;
        file_desc = mlview_file_descriptor_new (settings_dir);

        if (file_desc == NULL) {
                if (settings_dir)
                        g_free (settings_dir);
                return FALSE;
        }

        result = (!mlview_file_descriptor_is_dir
                  (file_desc, &is_dir)
                  || is_dir == TRUE) ? TRUE : FALSE;

        if (settings_dir)
                g_free (settings_dir);

        mlview_file_descriptor_destroy (file_desc);

        return result;
}

/**
 *Creates the user personal mlview settings dir. 
 *
 *@return 0 if succeeds,  -1 if not. 
 */
static gint
mlview_settings_manager_create_personal_settings_dir (void)
{
        gchar *home_dir = NULL,
                *settings_dir = NULL;

        MlViewFileDescriptor *file_desc = NULL;

        if (mlview_settings_manager_personal_settings_dir_exists
            () == TRUE)
                return 0;

        home_dir = getenv ("HOME");
        settings_dir = g_strconcat (home_dir, "/",
                                    MLVIEW_SETTINGS_PERSONAL_SETTINGS_DIR_NAME,
                                    NULL);

        file_desc = mlview_file_descriptor_new (settings_dir);
        if (file_desc == NULL) {
                mlview_file_descriptor_destroy (file_desc);
                g_free (settings_dir);
                return -1;
        }

        if (mlview_file_descriptor_create_directory
            (file_desc, S_IRWXU)) {
                mlview_file_descriptor_destroy (file_desc);
                g_free (settings_dir);
                return -1;
        }

        mlview_file_descriptor_destroy (file_desc);
        g_free (settings_dir);
        return 0;
}


/**
 *
 */
static xmlDoc *
mlview_settings_manager_create_personal_settings_xml_docs (void)
{
        xmlDoc *result;
        xmlNs *namespace;
        xmlNode *node,
        *root;

        result = xmlNewDoc ("1.0");

        /*create the root node (mlviewSettings) */
        root = xmlNewNode (NULL, "Settings");

        namespace =
                xmlNewNs (root, "http://www.gnome-mlview.org",
                          "gMlView");

        xmlSetNs (root, namespace);

        xmlDocSetRootElement (result, root);

        /*Set the tree view settings */
        node = xmlNewNode (namespace, "TreeView");
        xmlSetProp (node, "nodeExpansionLevel", "2");
        xmlAddChild (root, node);

        /*set the validation settings */
        node = xmlNewNode (namespace, "Validation");
        xmlSetProp (node, "turnOn", "true");
        xmlAddChild (root, node);
        return result;
}


/*
 *Settings management utils functions:
 */

/**
 *
 */
gint mlview_settings_management_get_settings_xml_nodes
        (xmlXPathObject ** a_xpath_object,
         gchar * a_xpath_expression, xmlDoc * a_xml_doc) {
        xmlXPathContext *xpath_context = NULL;
        xmlXPathObject *xpath_object = NULL;
        xmlNode *node;

        *a_xpath_object = NULL;

        g_return_val_if_fail (a_xpath_object != NULL, -3);
        g_return_val_if_fail (a_xpath_expression != NULL, -3);
        g_return_val_if_fail (a_xml_doc != NULL, -3);


        /*
         *Let the xpath evaluator know about the MlView namespace
         */
        xpath_context = xmlXPathNewContext (a_xml_doc);
        xmlXPathRegisterNs (xpath_context,
                            "gMlView",
                            "http://www.gnome-mlview.org");

        /*set the default node to the doc root node */
        node = xmlDocGetRootElement (a_xml_doc);
        g_return_val_if_fail (node != NULL, -2);
        xpath_context->node = node;

        xpath_object =
                xmlXPathEval (a_xpath_expression, xpath_context);

        if (xpath_context)
                xmlXPathFreeContext (xpath_context);

        if (!xpath_object)
                return -1;

        *a_xpath_object = xpath_object;

        return 0;
}


/**
 *
 */
void
mlview_settings_manager_destroy (MlViewSettingsManager *
                                 a_manager)
{
        g_return_if_fail (a_manager != NULL);
        g_return_if_fail (PRIVATE (a_manager) != NULL);

        if (PRIVATE (a_manager)->settings_entries) {
                g_list_free (PRIVATE (a_manager)->
                             settings_entries);
                PRIVATE (a_manager)->settings_entries = NULL;
        }

        if (PRIVATE (a_manager)->settings_entries_map) {
                g_hash_table_destroy
                        (PRIVATE (a_manager)->
                         settings_entries_map);

                PRIVATE (a_manager)->settings_entries_map = NULL;
        }

        if (PRIVATE (a_manager)->settings_xml_doc) {
                xmlFreeDoc (PRIVATE (a_manager)->
                            settings_xml_doc);
                PRIVATE (a_manager)->settings_xml_doc = NULL;
        }

        if (PRIVATE (a_manager)->settings_dialog) {
                gtk_widget_destroy
                        (GTK_WIDGET
                         (PRIVATE (a_manager)->settings_dialog));
                PRIVATE (a_manager)->settings_dialog = NULL;
        }

        if (PRIVATE (a_manager)) {
                g_free (PRIVATE (a_manager));
                PRIVATE (a_manager) = NULL;
        }

        if (a_manager) {
                g_free (a_manager);
                a_manager = NULL;
        }
}
