/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 8-*- */
/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 *Copyright 2001-2002 dodji SEKETELI, Gal CHAMOULAUD.
 *http://www.freespiders.org
 */

/**
 *@file  
 *Various helper functions used during xml
 *parsing and validation.
 */

#include <string.h>
#include <libxml/parser.h>
#include <libxml/catalog.h>
#include <libxml/parserInternals.h>
#include <libxml/xmlmemory.h>
#include <libxml/SAX.h>
#include <libxml/xmlerror.h>
#include <libxml/uri.h>
#include <libxml/hash.h>
#include <glade/glade.h>
#include <zlib.h>

#include "mlview-utils.h"
#include "config.h"

#include "mlview-parsing-utils.h"
#include "mlview-global-settings.h"
#include "mlview-file-descriptor.h"

static MlViewExternalSubsetDefinition
        * gv_external_subset_def = NULL;

static gboolean gv_store_external_subset_def = FALSE;

static
MlViewAppContext *gv_app_ctxt = NULL;

static const gint
        MAX_NUMBER_OF_ELEMENT_NAMES_IN_CHOICE_LIST = 256;

static GtkWidget *gv_dtd_choice_dialog = NULL;

/*
#ifdef MLVIEW_DATA_DIR
static const gchar 
* gv_mlview_data_dir = MLVIEW_DATA_DIR ;
#else
static const gchar * gv_mlview_data_dir = NULL ;
#endif

#ifdef MLVIEW_XML_CATALOG_DIR_NAME
static const gchar * gv_mlview_xml_catalog_dir_name = 
MLVIEW_XML_CATALOG_DIR_NAME ;
#else
static const gchar * gv_mlview_xml_catalog_dir_name = NULL ;
#endif
*/

typedef struct _MlViewParserCtxt MlViewParserCtxt;

/**
 *A custom xmlParserContext where we can store 
 *the mlview app context also. That way, some lightly
 *modified versions of libxml2 functions are able to
 *get/put information from/into the MlViewAppContext.
 */
struct _MlViewParserCtxt {
        MlViewAppContext *app_ctxt;
        xmlParserCtxt *xml_ctxt;
};

/*helper function declarations*/
static MlViewExternalSubsetDefinition
        * mlview_external_subset_definition_clone
        (MlViewExternalSubsetDefinition * a_ext_subs_def);

static xmlParserInput *mlview_external_entity_loader (const char
                                                      *URL, const char
                                                      *ID,
                                                      xmlParserCtxt
                                                      *
                                                      a_context);

static void
 mlview_external_subset_sax_handler (void *a_ctx,
                                     const xmlChar * a_name,
                                     const xmlChar *
                                     a_external_id,
                                     const xmlChar *
                                     a_system_id);


static xmlParserInput *mlview_sax_resolve_entity (void *a_ctx,
                                                  const xmlChar *
                                                  a_public_id,
                                                  const xmlChar *
                                                  a_system_id);

static enum MlViewStatus
 load_xml_document_from_local_file (gchar * a_xml_file_name,
                                    xmlParserCtxtPtr *
                                    a_parser_context,
                                    gboolean
                                    a_store_external_subset_info,
                                    MlViewAppContext *
                                    a_app_context);

static xmlDtdPtr custom_xmlSAXParseDTD (MlViewAppContext *
                                        a_app_context,
                                        xmlSAXHandlerPtr sax,
                                        const xmlChar *
                                        ExternalID,
                                        const xmlChar *
                                        SystemID);


static gboolean
        mlview_parsing_utils_ask_for_DTD_change_and_validation
        (MlViewAppContext * a_app_context,
         const gchar * a_external_id, const gchar * a_system_id,
         MlViewExternalSubsetDefinition **
         a_external_subset_def);

static xmlChar *mlview_resolve_external_entity (MlViewAppContext
                                                * a_app_ctxt,
                                                const xmlChar *
                                                a_external_id,
                                                const xmlChar *
                                                a_system_id);

static void
 mlview_parsing_utils_build_entities_list (void *a_hash_data,
                                           void *a_output_list,
                                           void *a_entity_name);

static void
 mlview_parsing_utils_scan_and_build_ids_list (void *a_hash_data, void
                                               *a_output_list,
                                               void *a_id_value);

static void
 build_required_element_content (MlViewAppContext * a_app_context,
                                 xmlElementContent * a_content,
                                 xmlNode ** a_node);

static gint
g_list_compare_string_elems (gchar * a_elem1, gchar * a_elem2);

static gboolean
is_an_ancestor_node (xmlNode * a_ancestor, xmlNode * a_cur_node);

/**
 *creates a new MlViewExternalSubsetDefinition. 
 *
 */
MlViewExternalSubsetDefinition *
mlview_external_subset_definition_new (const gchar *
                                       a_root_element_name,
                                       const gchar *
                                       a_external_id,
                                       const gchar * a_system_id)
{
        MlViewExternalSubsetDefinition *result;

        result = g_malloc0 (sizeof
                            (MlViewExternalSubsetDefinition));

        if (a_external_id != NULL)
                result->external_id = g_strdup (a_external_id);

        if (a_system_id != NULL)
                result->system_id = g_strdup (a_system_id);

        if (a_root_element_name != NULL)
                result->root_element_name =
                        g_strdup (a_root_element_name);

        return result;
}


/**
 *
 */
static MlViewExternalSubsetDefinition
        * mlview_external_subset_definition_clone
        (MlViewExternalSubsetDefinition * a_ext_subs_def) {
        g_return_val_if_fail (a_ext_subs_def != NULL, NULL);
        return mlview_external_subset_definition_new
                (a_ext_subs_def->root_element_name,
                 a_ext_subs_def->external_id,
                 a_ext_subs_def->system_id);
}


/**
 *destroys the MlViewExternalSubsetDefinition given in argument. 
 *
 */
void
 mlview_external_subset_definition_destroy
        (MlViewExternalSubsetDefinition * a_def) {
        g_return_if_fail (a_def != NULL);

        if (a_def->external_id != NULL) {
                g_free (a_def->external_id);
                a_def->external_id = NULL;
        }

        if (a_def->system_id != NULL) {
                g_free (a_def->system_id);
                a_def->system_id = NULL;
        }

        if (a_def->root_element_name != NULL) {
                g_free (a_def->root_element_name);
                a_def->root_element_name = NULL;
        }

        g_free (a_def);
}

/**
 *loads the xml document that is in the file which name is
 *a_xml_file_name. 
 *This function creates a new context 
 *and sets this parameter to it. 
 *If the file to open is a gziped, this function knows how
 *to handle it.
 *
 *@param a_parser_context the xml parser 
 *context to refer to for the parsing. 
 *caller of the function must deallocate 
 *the variable a_parser_context. 
 *@param a_xml_file_name the name of the file that contains the xml document.
 *
 *@return 0 if OK, > 0 to report an xmlParserError or < 0 
 *to report  MLVIEW_ERRORS 
 */
static enum MlViewStatus
 load_xml_document_from_local_file
        (gchar * a_xml_file_name,
         xmlParserCtxt ** a_parser_context,
         gboolean a_store_external_subset_info,
         MlViewAppContext * a_app_context) {
        FILE *file_ptr = NULL;
        int compressed = 0,
                c1 = 0,
                c2 = 0;
        gzFile zfile_ptr = NULL;
        int size = 1024,
                num_of_chars_read = 0,
                parse_status = MLVIEW_OK;
        gchar file_buffer[size];
        gchar *validation_is_on = NULL;
        xmlParserCtxt *xml_ctxt = NULL;

        /*initialyze validation flag to "false" 
         *=> no validation.
         *That way, libxml2 won't validate the 
         *document at load time.
         *We can then do post parsing validation.
         */
        xmlDoValidityCheckingDefaultValue = 0;

        if (a_app_context)
                validation_is_on =
                        mlview_app_context_get_settings_value
                        (a_app_context,
                         MLVIEW_STG_K_IS_VALIDATION_ON);

        /*now, open the xml file... */
        g_return_val_if_fail (a_xml_file_name != NULL,
                              MLVIEW_NULL_FILE_NAME_ERROR);

        file_ptr = fopen (a_xml_file_name, "rb");
        c1 = fgetc (file_ptr);
        c2 = fgetc (file_ptr);
        fclose (file_ptr);
        if ((c1 == 0x1F) && (c2 == 0x8B)) {
                compressed = 1;
        } else {
                compressed = 0;
        }

        zfile_ptr = gzopen (a_xml_file_name, "rb");

        g_return_val_if_fail (zfile_ptr != NULL,
                              MLVIEW_BAD_FILE_ERROR);

        num_of_chars_read = gzread (zfile_ptr, file_buffer, 4);
        g_return_val_if_fail (num_of_chars_read > 0,
                              MLVIEW_EMPTY_FILE_ERROR);
        /*
         *Force the parser to ignore ignorable white spaces so that
         *indentation can be made properly at save time.
         */
        xmlKeepBlanksDefault (0);
        gv_app_ctxt = a_app_context;
        xml_ctxt =
                xmlCreatePushParserCtxt (NULL, NULL,
                                         file_buffer,
                                         num_of_chars_read,
                                         a_xml_file_name);
        if (a_store_external_subset_info == TRUE) {
                /*
                 *overload the externalSubset sax handler with our custom one
                 *that stores the info about the external 
                 *subset declared in the xml doc
                 *so that we can do post parsing validation.
                 */

                /*overload the externalSubset sax handler */
                xml_ctxt->sax->externalSubset =
                        mlview_external_subset_sax_handler;
                /*
                 *tell the custom sax handler to save the 
                 *external subset definition ... or not
                 */
                gv_store_external_subset_def =
                        a_store_external_subset_info;
        }
        /*override the error messaging stream of libxml */
        xmlSetGenericErrorFunc (a_app_context,
                                (xmlGenericErrorFunc)
                                mlview_app_context_bufferize_error);
        while ((num_of_chars_read =
                gzread (zfile_ptr, file_buffer, size)) > 0) {
                parse_status =
                        xmlParseChunk
                        (xml_ctxt,
                         file_buffer, num_of_chars_read, 0);

                if (parse_status != XML_ERR_OK)
                        break;
        }
        /*
         *call the xmlParseChunk once more 2 
         *consume all the data stored in 
         *the parser internal cache ...
         */
        if (parse_status == XML_ERR_OK) {
                parse_status =
                        xmlParseChunk (xml_ctxt, file_buffer,
                                       num_of_chars_read, 1);
        }
        *a_parser_context = xml_ctxt;
        /*restore the error display stream of libxml */
        xmlSetGenericErrorFunc (NULL, NULL);
        if (a_app_context
            &&
            !mlview_app_context_error_buffer_is_empty
            (a_app_context)) {
                mlview_app_context_display_buffered_error
                        (a_app_context);
        } else {
                mlview_app_context_set_error_dialog_title
                        (a_app_context, NULL);
        }
        gzclose (zfile_ptr);
        /*
         * If the file is compressed ... we set it inside the doc
         */
        if (compressed) {
                printf ("Compression detected\n");
                xmlSetDocCompressMode ((**a_parser_context).
                                       myDoc, 9);
        }
        return (parse_status == XML_ERR_OK) ? 0 : parse_status;
}


/**
 *A custom entity loader that first fetches 
 *the entities by resolving their url reference
 *through system catalog and then 
 *by performing a lookup on using http or ftp.
 *http and ftp support still need to be added.
 */
static xmlParserInput *
mlview_external_entity_loader (const char *URL,
                               const char *ID,
                               xmlParserCtxt * a_xml_ctxt)
{
        xmlParserInput *result = NULL;
        xmlChar *resource = NULL;

        resource =
                mlview_resolve_external_entity (gv_app_ctxt,
                                                ID, URL);
        if (!resource) {
                MlViewFileDescriptor *fd = NULL;
                int error = 0;
                gboolean is_local = FALSE;

                fd = mlview_file_descriptor_new (URL);
                g_return_val_if_fail (fd, NULL);
                error = mlview_file_descriptor_is_local (fd,
                                                         &is_local);
                if (error || is_local != TRUE) {
                        if (fd) {
                                mlview_file_descriptor_destroy
                                        (fd);
                                fd = NULL;
                        }
                        return NULL;
                }
                /*now, try to load the local dtd file */
                resource = xmlBuildURI (URL, NULL);
        }
        result = xmlNewInputFromFile (a_xml_ctxt, resource);
        if (resource) {
                xmlFree (resource);
                resource = NULL;
        }
        return result;
}


static xmlParserInput *
mlview_sax_resolve_entity (void *a_ctx,
                           const xmlChar * a_public_id,
                           const xmlChar * a_system_id)
{
        xmlParserCtxt *xml_ctxt = (xmlParserCtxt *) a_ctx;
        xmlParserInput *result = NULL;
        xmlChar *URI = NULL;
        guchar *base = NULL;

        if (xml_ctxt == NULL) {
                /*
                 *build a parser context cause we need it to
                 *build a parser input.
                 */
                xml_ctxt = xmlNewParserCtxt ();
        }

        g_return_val_if_fail (xml_ctxt, NULL);

        if (xml_ctxt && (xml_ctxt->input != NULL)) {
                base = (guchar *)
                        xml_ctxt->input->filename;
        }

        if (base == NULL && xml_ctxt) {
                base = xml_ctxt->directory;
        }

        URI = xmlBuildURI (a_system_id, NULL);
        result = mlview_external_entity_loader ((const char *)
                                                URI,
                                                (const char *)
                                                a_public_id,
                                                xml_ctxt);

        if (URI != NULL)
                xmlFree (URI);

        return result;
}


/**
 *Resolves the the external entity and returns an xmlChar *
 *that must be freed with xmlFree().
 *
 *@param a_context the libxml parsing context context.
 *@param a_pub_id the public ID of the external entity
 *@param a_sys_id the system ID of the external entity.
 *@return the URL of the external entity or NULL if not found.
 */
static xmlChar *
mlview_resolve_external_entity (MlViewAppContext * a_app_context,
                                const xmlChar * a_pub_id,
                                const xmlChar * a_sys_id)
{
        xmlChar *result = NULL;

        g_return_val_if_fail (a_app_context != NULL, FALSE);
        g_return_val_if_fail (a_app_context != NULL, FALSE);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_app_context), FALSE);

        result = xmlCatalogResolve (a_pub_id, a_sys_id);
        return result;
}

/**
 *This function is a replacement of the 
 *default "externalSubset" sax handler used by libxml
 *to parse xml documents.
 *If validation is on, it interacts 
 *with the user to know if she wants to use another DTD than the one
 *specified in the doctype. The user can even choose not to validate.
 *
 *@param a_external_id the external id
 *@param a_ctxt the user data (XML parser context)
 *@param a_system_id the SYSTEM ID (e.g filename or URL)
 *@param a_name the root element name
 */
static void
mlview_external_subset_sax_handler (void *a_ctxt,
                                    const xmlChar * a_name,
                                    const xmlChar *
                                    a_external_id,
                                    const xmlChar * a_system_id)
{
        xmlParserCtxt *ctxt = NULL;
        gchar *validation_is_on = NULL;

        g_return_if_fail (a_ctxt);

        ctxt = a_ctxt;

        if (a_external_id == NULL && a_system_id == NULL)
                return;

        if (gv_app_ctxt) {
                validation_is_on =
                        mlview_app_context_get_settings_value
                        (gv_app_ctxt,
                         MLVIEW_STG_K_IS_VALIDATION_ON);
        }

        if (validation_is_on
            && (strcmp (validation_is_on, MLVIEW_STG_V_YES) == 0)
            && ctxt) {
                ctxt->validate = 1;
        } else {
                ctxt->validate = 0;
        }

        /*
         *If validation is on, ask the user if she wants 
         *to choose another DTD but
         *the one referenced by the document in the doctype.
         */
        if (ctxt->validate == 1 && (gv_app_ctxt != NULL)) {
                MlViewExternalSubsetDefinition
                        * external_subset_def = NULL;

                gboolean validate =
                        mlview_parsing_utils_ask_for_DTD_change_and_validation
                        (gv_app_ctxt, a_external_id,
                         a_system_id, &external_subset_def);

                if (validate == FALSE)
                        ctxt->validate = 0;

                if (validate == TRUE && external_subset_def) {
                        a_external_id =
                                external_subset_def->external_id;

                        a_system_id =
                                external_subset_def->system_id;
                }
        }

        if (ctxt->validate == 1) {
                externalSubset (ctxt,
                                a_name, a_external_id,
                                a_system_id);

                if (ctxt->myDoc
                    && (ctxt->myDoc->extSubset == NULL)) {
                        mlview_app_context_warning
                                (gv_app_ctxt,
                                 _
                                 ("The external DTD subset was not found. I couldn't validate the document."));
                }
        }

        /*
         *We will not do validation during the document parsing.
         *We will only do post parsing validation 
         *(if validation is required).
         */
        ctxt->validate = 0;
}


/**
 *This code has been copied and modified from libxml2.
 *GPL code is good for you.
 *
 */
static xmlDtdPtr
custom_xmlSAXParseDTD (MlViewAppContext * a_app_context,
                       xmlSAXHandlerPtr sax,
                       const xmlChar * ExternalID,
                       const xmlChar * SystemID)
{
        xmlDtdPtr ret = NULL;
        xmlParserCtxtPtr ctxt;
        xmlParserInputPtr input = NULL;
        xmlCharEncoding enc;

        if ((ExternalID == NULL) && (SystemID == NULL))
                return (NULL);

        ctxt = xmlNewParserCtxt ();
        if (ctxt == NULL) {
                return (NULL);
        }
        /*
         * Set-up the SAX context
         */
        if (sax != NULL) {
                if (ctxt->sax != NULL)
                        xmlFree (ctxt->sax);
                ctxt->sax = sax;
                ctxt->userData = NULL;
        }
        ctxt->userData = ctxt;  /*usefull for libxml2 */

        /*
         * Ask the Entity resolver to load the damn thing
         */
        if (ctxt->sax != NULL
            && ctxt->sax->resolveEntity != NULL) {
                input = ctxt->sax->resolveEntity (ctxt,
                                                  ExternalID,
                                                  SystemID);

        }

        if (input == NULL) {
                if (sax != NULL)
                        ctxt->sax = NULL;

                xmlFreeParserCtxt (ctxt);
                return (NULL);
        }

        /*
         * plug some encoding conversion routines here.
         */
        xmlPushInput (ctxt, input);

        enc = xmlDetectCharEncoding (ctxt->input->cur, 4);

        xmlSwitchEncoding (ctxt, enc);

        if (input->filename == NULL) {
                input->filename = (char *) xmlStrdup (SystemID);
        }

        input->line = 1;
        input->col = 1;
        input->base = ctxt->input->cur;
        input->cur = ctxt->input->cur;
        input->free = NULL;

        /*
         * let's parse that entity knowing it's an external subset.
         */

        ctxt->inSubset = 2;

        ctxt->myDoc = xmlNewDoc (BAD_CAST "1.0");

        ctxt->myDoc->extSubset =
                xmlNewDtd (ctxt->myDoc, BAD_CAST "none",
                           ExternalID, SystemID);

        xmlParseExternalSubset (ctxt, ExternalID, SystemID);

        if (ctxt->myDoc != NULL) {
                if (ctxt->wellFormed) {
                        ret = ctxt->myDoc->extSubset;

                        ctxt->myDoc->extSubset = NULL;
                } else {
                        ret = NULL;
                }

                xmlFreeDoc (ctxt->myDoc);

                ctxt->myDoc = NULL;
        }

        if (sax != NULL)
                ctxt->sax = NULL;


        xmlFreeParserCtxt (ctxt);

        return (ret);
}

/**
 *
 *@param a_external_subset_def output parameter. 
 *A pointer to the definition of the new 
 *DTD the user choosed,or NULL if user didn't choose another DTD. 
 *@param a_app_context the current instance of MlViewAppContext.
 *
 */
static gboolean
        mlview_parsing_utils_ask_for_DTD_change_and_validation
        (MlViewAppContext * a_app_context,
         const gchar * a_external_id,
         const gchar * a_system_id,
         MlViewExternalSubsetDefinition ** a_external_subset_def)
{
        GladeXML *gxml;
        gchar *gfile;
        gint button;
        gboolean result = TRUE;
        GtkWidget *label_public_id = NULL;
        GtkWidget *label_system_id = NULL;
        GtkWidget *label_resource = NULL;
        const gchar *public_id = NULL;
        const gchar *system_id = NULL;
        xmlChar *resource = NULL;

        g_return_val_if_fail (a_app_context != NULL, TRUE);
        g_return_val_if_fail (a_external_subset_def != NULL,
                              TRUE);

        public_id = ((const gchar *) (a_external_id != NULL)) ?
                a_external_id : "\"\"";

        system_id = ((const gchar *) (a_system_id != NULL)) ?
                a_system_id : "\"\"";

        *a_external_subset_def = NULL; /*sanity initialization */

        resource =
                mlview_resolve_external_entity (a_app_context,
                                                a_external_id,
                                                a_system_id);

        if (resource) {
                gfile = g_strdup_printf ("%s%s",
                                         MLVIEW_GLADE_DIR,
                                         "/mlview-dtd-choice.glade");
                gxml = glade_xml_new (gfile,
                                      "mlview_dtd_choice_with_catalog",
                                      NULL);

                g_return_val_if_fail (gxml != NULL, TRUE);

                label_public_id =
                        glade_xml_get_widget (gxml,
                                              "dtd_public_id_with_catalog");
                label_system_id =
                        glade_xml_get_widget (gxml,
                                              "dtd_system_id_with_catalog");
                label_resource =
                        glade_xml_get_widget (gxml,
                                              "resource_catalog");
                gtk_label_set_text (GTK_LABEL (label_public_id),
                                    public_id);
                gtk_label_set_text (GTK_LABEL (label_system_id),
                                    system_id);
                gtk_label_set_text (GTK_LABEL (label_resource),
                                    resource);

                gv_dtd_choice_dialog =
                        glade_xml_get_widget (gxml,
                                              "mlview_dtd_choice_with_catalog");

        } else {
                gfile = g_strdup_printf ("%s%s",
                                         MLVIEW_GLADE_DIR,
                                         "/mlview-dtd-choice.glade");
                gxml = glade_xml_new (gfile, "mlview_dtd_choice",
                                      NULL);

                g_return_val_if_fail (gxml != NULL, TRUE);

                label_public_id =
                        glade_xml_get_widget (gxml,
                                              "dtd_public_id");
                label_system_id =
                        glade_xml_get_widget (gxml,
                                              "dtd_system_id");
                gtk_label_set_text (GTK_LABEL (label_public_id),
                                    public_id);
                gtk_label_set_text (GTK_LABEL (label_system_id),
                                    system_id);

                gv_dtd_choice_dialog =
                        glade_xml_get_widget (gxml,
                                              "mlview_dtd_choice");
        }

        gv_dtd_choice_dialog =
                glade_xml_get_widget (gxml, "mlview_dtd_choice");





        glade_xml_signal_autoconnect (gxml);

        mlview_app_context_set_window_icon (a_app_context,
                                            GTK_WINDOW
                                            (gv_dtd_choice_dialog));

        mlview_app_context_set_window_icon (a_app_context,
                                            GTK_WINDOW
                                            (gv_dtd_choice_dialog));
        glade_xml_signal_autoconnect (gxml);

        button = gtk_dialog_run (GTK_DIALOG
                                 (gv_dtd_choice_dialog));

        if (resource) {
                switch (button) {
                case 0:
                        /*user clicked the "Use that DTD" button */
                        *a_external_subset_def = NULL;
                        break;
                case 1:
                        /*
                         *user clicked the 
                         *"Choose another DTD" button
                         */
                        *a_external_subset_def =
                                mlview_parsing_utils_let_user_choose_a_dtd
                                (a_app_context,
                                 _("Choose a DTD"));

                        result = (*a_external_subset_def ==
                                  NULL) ? FALSE : TRUE;
                        break;
                case 2:
                        /*
                         *user clicked the 
                         *"Do not validate" button
                         */
                        result = FALSE; /*do not validate */
                        break;
                case -1:

                        /*
                         *user closed the window using 
                         *Window manager=>like case 0
                         */
                default:
                        break;
                }
        } else {
                switch (button) {
                case 0:
                        /*
                         *user clicked the 
                         *"Choose another DTD" button
                         **/
                        *a_external_subset_def =
                                mlview_parsing_utils_let_user_choose_a_dtd
                                (a_app_context,
                                 _("Choose a DTD"));

                        result = (*a_external_subset_def ==
                                  NULL) ? FALSE : TRUE;
                        break;

                case 1:
                        /*user clicked the "Do not validate" button */
                        result = FALSE; /*do not validate */
                        break;

                case -1:

                        /*
                         *user closed the window using 
                         *Window manager=>like case 0
                         **/
                default:
                        break;
                }
        }

        gtk_widget_hide (GTK_WIDGET (gv_dtd_choice_dialog));
        g_object_unref (gxml);
        g_free (gfile);

        return result;
}


/**
 *The callback of the libxml2 hash table scan method xmlHashScan () ;
 *Takes a_hash_data 
 *(the current value of the hash table) and a_entity_name
 *(the key of the current 
 *value of the hash table), this function adds the
 *entity name into a_output_list which is supposed to be a GList **. 
 *
 *@param a_entity_name the key of the 
 *hash table, here, the name of the entity
 *@param a_output_list out parameter. The result of this function.
 *@param a_hash_data the data contained in an entry of the hash table.
 *
 *
 */
static void
mlview_parsing_utils_build_entities_list (void *a_hash_data,
                                          void *a_output_list,
                                          void *a_entity_name)
{

        if (a_entity_name)
                *((GList **) a_output_list) =
                        (void *) g_list_append
                        (*((GList **) a_output_list),
                         a_entity_name);
}


static void
mlview_parsing_utils_scan_and_build_ids_list (void *a_hash_data, void
                                              *a_output_list,
                                              void *a_id_value)
{

        if (a_id_value) {
                *((GList **) a_output_list) =
                        (void *) g_list_append
                        (*((GList **) a_output_list),
                         a_id_value);
        }
}

static gboolean
is_an_ancestor_node (xmlNode * a_ancestor, xmlNode * a_cur_node)
{
        xmlNode *mobile_ptr = a_cur_node;

        g_return_val_if_fail (a_cur_node != NULL, FALSE);
        g_return_val_if_fail (a_ancestor != NULL, FALSE);

        while (mobile_ptr) {
                if (xmlStrEqual (mobile_ptr->name,
                                 a_ancestor->name))
                        return TRUE;
                mobile_ptr = mobile_ptr->parent;
        }
        return FALSE;
}

/**
 *This function builds the required subtree of the 
 *node given in argument.
 *Required means "the minimum required to make 
 *the document validate against de element content
 *declaration passed in parameter in a_content. 
 *
 *@param a_content the element content declaration tree to refer to.
 *@param a_node the output xml node tree.
 *@param a_app_context the current application context.
 *
 *
 */
static void
build_required_element_content (MlViewAppContext * a_app_context,
                                xmlElementContent * a_content,
                                xmlNode ** a_node)
{
        xmlElementContent *current_element_content = a_content;
        xmlNode *child_node = NULL,
                *dummy = NULL;

        g_return_if_fail (a_app_context != NULL);
        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (a_app_context));
        g_return_if_fail (a_node != NULL);
        g_return_if_fail ((*a_node) != NULL);

        if (!current_element_content)
                return;

        dummy = xmlNewNode (NULL, "<!dummy>");
        xmlFree ((xmlChar *) dummy->name);
        dummy->name = NULL;

        switch (current_element_content->type) {

        case XML_ELEMENT_CONTENT_OR:

                /*
                 *this element decl is a 
                 *<!ELEMENT current (child1 | child2)> type.
                 *Note that childi may be an element or a SEQ or whatever.
                 */

                switch (current_element_content->ocur) {
                case XML_ELEMENT_CONTENT_ONCE:
                case XML_ELEMENT_CONTENT_PLUS:
                        /*ex: <!ELEMENT current (child1 | child2)> or
                         * <!ELEMENT current (child1 | child2)+> decl type.
                         *Note that childi may be an 
                         *element or a SEQ or whatever.
                         *=> let's check wether if we will 
                         *insert child1 or child2
                         *(that will depend on child1 and child2 cardinality)
                         */
                        if (current_element_content->type
                            ==
                            XML_ELEMENT_CONTENT_ELEMENT
                            && current_element_content->name) {

                                child_node =
                                        xmlNewChild
                                        ((*a_node),
                                         NULL,
                                         current_element_content->
                                         name, NULL);

                                mlview_parsing_utils_build_required_children_tree
                                        (a_app_context,
                                         &child_node);

                        } else if (current_element_content->c1
                                   &&
                                   ((current_element_content->
                                     c1->ocur ==
                                     XML_ELEMENT_CONTENT_ONCE)
                                    || current_element_content->
                                    c1->ocur ==
                                    XML_ELEMENT_CONTENT_PLUS)) {
                                /*ex: <!ELEMENT current (child1 | child2*)+> 
                                 *decl type or 
                                 *even <!ELEMENT current (child1 | child2+)+>
                                 *Note that childi may be an element or a 
                                 *SEQ or whatever.
                                 *=>we explore child1 and not child2. 
                                 *=> we won't keep child2
                                 */
                                if (current_element_content->c1->
                                    type ==
                                    XML_ELEMENT_CONTENT_ELEMENT)
                                {
                                        dummy->name =
                                                current_element_content->
                                                c1->name;
                                        if (is_an_ancestor_node
                                            (dummy,
                                             *a_node) == FALSE) {
                                                build_required_element_content
                                                        (a_app_context,
                                                         current_element_content->
                                                         c1,
                                                         a_node);
                                        } else {
                                                build_required_element_content
                                                        (a_app_context,
                                                         current_element_content->
                                                         c2,
                                                         a_node);
                                        }
                                } else {
                                        build_required_element_content
                                                (a_app_context,
                                                 current_element_content->
                                                 c1, a_node);
                                }
                        } else {
                                /*ex: <!ELEMENT current (child1* | child2+)+> 
                                 *decl type
                                 *Note that childi may be an element or a 
                                 *SEQ or whatever.
                                 *=> we won't explore child1 => 
                                 *we won't keep it.
                                 */
                                build_required_element_content
                                        (a_app_context,
                                         current_element_content->
                                         c2, a_node);
                        }
                        break;
                default:
                        break;
                }
                break;

        default:
                switch (current_element_content->ocur) {
                case XML_ELEMENT_CONTENT_ONCE:
                case XML_ELEMENT_CONTENT_PLUS:
                        if (current_element_content->name) {
                                child_node =
                                        xmlNewChild
                                        ((*a_node), NULL,
                                         current_element_content->
                                         name, NULL);

                                mlview_parsing_utils_build_required_children_tree
                                        (a_app_context,
                                         &child_node);
                        } else {
                                build_required_element_content
                                        (a_app_context,
                                         current_element_content->
                                         c1, a_node);

                                build_required_element_content
                                        (a_app_context,
                                         current_element_content->
                                         c2, a_node);
                        }

                        break;
                default:
                        break;

                }
                break;
        }
        if (dummy) {
                dummy->name = NULL;
                xmlFreeNode (dummy);
                dummy = NULL;
        }
}


static gint
g_list_compare_string_elems (gchar * a_str1, gchar * a_str2)
{
        g_return_val_if_fail (a_str1 != NULL, -1);
        g_return_val_if_fail (a_str2 != NULL, 1);

        return strcmp (a_str1, a_str2);
}

/******************************************
 *Public methods
 ******************************************/

/**
 *
 *@param a_file_name the name of the file to open and parse.
 *@param a_app_context the application context to use. 
 *If set to NULL, 
 *it is not taken in account. 
 *
 *
 *@return the xmlDoc resulting from 
 *the parsing or NULL if the parsing failed. 
 */
xmlDoc *
mlview_parsing_utils_load_xml_file (gchar * a_file_name,
                                    MlViewAppContext *
                                    a_app_context)
{
        xmlDoc *result = NULL;
        xmlParserCtxtPtr parser_context = NULL;

        g_return_val_if_fail (a_file_name != NULL, NULL);

        if (strcmp (a_file_name, "")) {
                gint load_res;

                /*
                 *fixme: remember to add support of 
                 *error handling while loading the document.
                 */
                load_res =
                        load_xml_document_from_local_file
                        (a_file_name, &parser_context,
                         TRUE, a_app_context);

                if (parser_context == NULL) {
                        mlview_app_context_error
                                (a_app_context,
                                 _
                                 ("could not load xml document %s"),
                                 a_file_name);

                        return NULL;
                }

                if (!load_res) {
                        parser_context->myDoc->name = (xmlChar *)
                                g_strdup (a_file_name);
                        result = parser_context->myDoc;
                }
        }

        if (parser_context != NULL) {
                xmlFreeParserCtxt (parser_context);
        }
        return result;
}


/**
 *Saves an xml file using the application wide settings. 
 *
 *@param a_xml_doc the xml document to save.
 *@param a_app_context the current xml app context.
 *
 *
 *@return the number of bytes written or -1 in case of an error. 
 */
gint
mlview_parsing_utils_save_xml_doc (xmlDoc * a_xml_doc,
                                   gchar * a_file_path,
                                   MlViewAppContext *
                                   a_app_context)
{
        g_return_val_if_fail (a_xml_doc != NULL, -1);
        g_return_val_if_fail (a_file_path != NULL, -1);

        return xmlSaveFormatFile (a_file_path, a_xml_doc, 1);
}


xmlDtd *mlview_parsing_utils_load_a_dtd
        (MlViewExternalSubsetDefinition * a_subset_def,
         MlViewAppContext * a_app_context) {
        xmlDtd *dtd = NULL;
        xmlSAXHandler *sax_handler = NULL;

        g_return_val_if_fail (a_subset_def != NULL, NULL);

        if (a_subset_def->system_id) {
                /*
                 *Replace this function by xmlSAXParseDTD. 
                 *=> that will allow overloading 
                 *of the resolveEntity handler.
                 */
                sax_handler =
                        (xmlSAXHandler *) xmlMalloc
                        (sizeof (xmlSAXHandler));

                g_assert (sax_handler != NULL);

                memset (sax_handler, 0, sizeof (xmlSAXHandler));

                initxmlDefaultSAXHandler (sax_handler, 0);

                g_assert (sax_handler != NULL);

                /*override the libxml error printing routine */
                xmlSetGenericErrorFunc
                        (a_app_context, (xmlGenericErrorFunc)
                         mlview_app_context_bufferize_error);

                mlview_app_context_set_error_dialog_title
                        (a_app_context,
                         _
                         ("Some error(s) occured during the parsing of the dtd.\n\n"));

                /*overload the entity resolver. */
                sax_handler->resolveEntity =
                        mlview_sax_resolve_entity;
                gv_app_ctxt = a_app_context;
                dtd = custom_xmlSAXParseDTD (a_app_context,
                                             sax_handler,
                                             a_subset_def->
                                             external_id,
                                             a_subset_def->
                                             system_id);

                /*restore the libxml error printing routine
                 *FIXME: it would be better re-think the error 
                 *overidding/restoring process.
                 */
                xmlSetGenericErrorFunc (NULL, NULL);

                if (!mlview_app_context_error_buffer_is_empty
                    (a_app_context)) {
                        mlview_app_context_display_buffered_error
                                (a_app_context);
                } else {
                        mlview_app_context_set_error_dialog_title
                                (a_app_context, NULL);
                }
        }

        xmlCleanupParser ();

        return dtd;
}


/**
 *@return 0 if document is valid, 1 if not, 2 if no DTD was given 
 *in parameter, and -1 if one of the parameters are bad. 
 */
gint
mlview_parsing_utils_validate_dtd (xmlDoc * a_doc,
                                   xmlDtd * a_dtd,
                                   MlViewAppContext *
                                   a_app_context)
{
        gint result = 0;
        gint validity_status;
        xmlValidCtxt validation_context;

        g_return_val_if_fail (a_doc != NULL, -1);

        validation_context.userData = a_app_context;
        validation_context.error = (xmlValidityErrorFunc)
                mlview_app_context_bufferize_error;
        validation_context.warning = (xmlValidityWarningFunc)
                mlview_app_context_bufferize_error;

        xmlSetGenericErrorFunc (a_app_context,
                                (xmlGenericErrorFunc)
                                mlview_app_context_bufferize_error);
        mlview_app_context_set_error_dialog_title
                (a_app_context,
                 _
                 ("Some error(s) occured during the validation of the document.\n\n"));

        validity_status =
                xmlValidateDtd (&validation_context, a_doc,
                                a_dtd);

        xmlSetGenericErrorFunc
                (a_app_context, (xmlGenericErrorFunc)
                 mlview_app_context_bufferize_error);

        if (validity_status == 1) {
                /*xmlValidate() returns 1 if doc is valid 
                 *whereas we want to return 0 in that case ...*/
                result = 0;
        } else {
                result = 1;
        }

        if (!mlview_app_context_error_buffer_is_empty
            (a_app_context)) {
                mlview_app_context_display_buffered_error
                        (a_app_context);
        } else
                mlview_app_context_set_error_dialog_title
                        (a_app_context, NULL);

        return result;
}


/**
 *found during the parsing of the last xml document if and only if 
 *the parsing has been triggered by calling the
 *function mlview_parsing_utils_load_xml_file(gchar * a_file_name).
 *Note that after a given xml document parsing, 
 *this function can only be 
 *
 *@return NULL. This function can also return NULL if no external 
 *subset has been declared in the 
 */
MlViewExternalSubsetDefinition *
mlview_utils_get_a_copy_of_last_external_subset_def (void)
{
        MlViewExternalSubsetDefinition *result;

        if (gv_external_subset_def == NULL)
                return NULL;
        result = mlview_external_subset_definition_clone
                (gv_external_subset_def);

        mlview_external_subset_definition_destroy
                (gv_external_subset_def);

        gv_external_subset_def = NULL;

        return result;
}


/*
 *mlview_parsing_utils_let_user_choose_a_dtd:
 *
 *
 */
MlViewExternalSubsetDefinition
        * mlview_parsing_utils_let_user_choose_a_dtd
        (MlViewAppContext * a_app_context, gchar * a_title) {
        MlViewFileSelection *filesel = NULL;
        MlViewExternalSubsetDefinition *subset_def = NULL;

        enum MLVIEW_SELECTED_BUTTON button_clicked;
        gchar *dtd = NULL;

        g_return_val_if_fail (a_app_context != NULL, NULL);

        filesel = mlview_app_context_get_file_selector
                (a_app_context, a_title);
        g_return_val_if_fail (filesel != NULL, NULL);

        gtk_widget_realize (GTK_WIDGET (filesel));
        mlview_app_context_set_window_icon (a_app_context,
                                            GTK_WINDOW
                                            (filesel));
        button_clicked =
                mlview_file_selection_run (filesel, TRUE);

        switch (button_clicked) {
        case OK_BUTTON:
                dtd = g_strdup (gtk_file_selection_get_filename
                                (GTK_FILE_SELECTION (filesel)));
                if (dtd) {
                        subset_def =
                                mlview_external_subset_definition_new
                                (NULL, NULL, dtd);
                }
                if (dtd) {
                        g_free (dtd);
                        dtd = NULL;
                }
                break;
        case CANCEL_BUTTON:
        case WINDOW_CLOSED:
        default:
                break;
        }

        return subset_def;
}

/**
 *Builds the list of the feasible element name that can be 
 *inserted before/after/beneath/ the
 *current node.
 *and -2 if some bad parameter have been passed to this function. 
 *
 *@param a_current_xml_node the reference node.
 *@param a_feasible_names output parameter. 
 *A pointer to the output element completion name list.
 *@param a_insertion_scheme how you want to insert a node 
 *relatively to the @a_current_xml_node.
 *@param a_app_context the current mlview application context.
 */
gint mlview_parsing_utils_build_element_name_completion_list
        (MlViewAppContext * a_app_context,
         enum NODE_INSERTION_SCHEME a_insertion_scheme,
         xmlNode * a_current_xml_node,
         GList ** a_feasible_names_ptr) {
        const xmlChar *feasible_names
                [MAX_NUMBER_OF_ELEMENT_NAMES_IN_CHOICE_LIST];
        gint nb_of_names = 0;
        gchar *validation_is_on = NULL;

        g_return_val_if_fail (a_current_xml_node != NULL, -2);
        g_return_val_if_fail (a_current_xml_node->type
                              == XML_ELEMENT_NODE, -2);
        g_return_val_if_fail (feasible_names != NULL, -2);

        memset (feasible_names, 0,
                MAX_NUMBER_OF_ELEMENT_NAMES_IN_CHOICE_LIST
                * sizeof (xmlChar *));

        if (a_app_context
            && mlview_app_context_settings_exist (a_app_context))
                validation_is_on =
                        mlview_app_context_get_settings_value
                        (a_app_context,
                         MLVIEW_STG_K_IS_VALIDATION_ON);

        if ((validation_is_on == NULL)
            || (strcmp (validation_is_on, MLVIEW_STG_V_YES) !=
                0))
                return -1;      /*validation is not turned on !!! */

        if ((a_insertion_scheme == INSERT_BEFORE)
            && (a_current_xml_node->type == XML_DOCUMENT_NODE)) {
                /*
                 *invocation of xmlValidGetValidElements 
                 *in this case makes libxml2-2.4.16 dump core.*/
                return 0;
        }

        if ((a_insertion_scheme == INSERT_BEFORE)
            && a_current_xml_node->parent
            && a_current_xml_node->parent->type
            == XML_DOCUMENT_NODE) {
                /*
                 *in this case, 
                 *invocation of xmlValidGetValidElements 
                 *with a document that have NULL sigsevs.
                 */
                return 0;
        }

        if ((a_insertion_scheme == INSERT_AFTER)
            && (a_current_xml_node->type == XML_DOCUMENT_NODE)) {
                /*
                 *invocation of xmlValidGetValidElements 
                 *in this case sigsevs libxml2-2.4.16
                 */
                return 0;
        }

        switch (a_insertion_scheme) {
        case INSERT_BEFORE:
                /*
                 *user wants to insert a node just 
                 *before the current selected node
                 */
                nb_of_names =
                        xmlValidGetValidElements
                        (a_current_xml_node->prev,
                         a_current_xml_node,
                         feasible_names,
                         MAX_NUMBER_OF_ELEMENT_NAMES_IN_CHOICE_LIST);
                break;
        case INSERT_AFTER:
                nb_of_names =
                        xmlValidGetValidElements
                        (a_current_xml_node,
                         a_current_xml_node->next,
                         feasible_names,
                         MAX_NUMBER_OF_ELEMENT_NAMES_IN_CHOICE_LIST);
                break;

        case ADD_CHILD:        /*append child */
                if (a_current_xml_node->children) {
                        nb_of_names =
                                xmlValidGetValidElements
                                (a_current_xml_node->last,
                                 NULL, feasible_names,
                                 MAX_NUMBER_OF_ELEMENT_NAMES_IN_CHOICE_LIST);

                } else {
                        nb_of_names =
                                xmlValidGetValidElementsChildren
                                (a_current_xml_node,
                                 feasible_names,
                                 MAX_NUMBER_OF_ELEMENT_NAMES_IN_CHOICE_LIST);
                }

                break;
        default:
                break;
        }                       /*end switch */

        if (nb_of_names > 0) {  /*found some names to propose to the user */
                int i;
                GList *list_ptr = NULL;
                GHashTable *names_index =
                        g_hash_table_new (g_str_hash,
                                          g_str_equal);

                for (list_ptr = *a_feasible_names_ptr;
                     list_ptr; list_ptr = list_ptr->next) {

                        if (list_ptr->data)
                                g_hash_table_insert (names_index,
                                                     list_ptr->
                                                     data,
                                                     list_ptr->
                                                     data);
                }

                for (i = 0; i < nb_of_names; i++) {

                        if (((xmlChar **) feasible_names)[i]
                            && !g_hash_table_lookup
                            (names_index,
                             ((xmlChar **) feasible_names)[i])) {

                                (*a_feasible_names_ptr) =
                                        g_list_append
                                        (*a_feasible_names_ptr,
                                         ((xmlChar **)
                                          feasible_names)[i]);
                        }
                }
                g_hash_table_destroy (names_index);

                *a_feasible_names_ptr =
                        g_list_sort (*a_feasible_names_ptr,
                                     (GCompareFunc)
                                     g_list_compare_string_elems);
        }

        return nb_of_names;
}

/**
 *
 *
 *@param a_current_xml_node the current xml node
 *@param a_attr_names_compl_list out parameter. 
 *The attribute names completion list.
 *@param a_app_context the current context of the mlview application.
 *
 *
 */
gint mlview_parsing_utils_build_attribute_name_completion_list
        (MlViewAppContext * a_app_context,
         xmlNode * a_current_xml_node,
         GList ** a_attr_names_compl_list,
         gboolean a_required_attributes_only) {
        xmlElement *element_desc = NULL;
        gchar *validation_is_on = NULL;
        gint result = 0;

        g_return_val_if_fail (a_app_context != NULL, -2);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_app_context), -2);
        g_return_val_if_fail (a_current_xml_node != NULL, -2);
        g_return_val_if_fail (a_attr_names_compl_list != NULL,
                              -2);

        *a_attr_names_compl_list = NULL;

        if (a_app_context
            && mlview_app_context_settings_exist (a_app_context))
                validation_is_on =
                        mlview_app_context_get_settings_value
                        (a_app_context,
                         MLVIEW_STG_K_IS_VALIDATION_ON);

        if (validation_is_on == NULL
            || (strcmp (validation_is_on, MLVIEW_STG_V_YES) !=
                0))
                return -1;      /*validation is not turned on !!! */

        if (a_current_xml_node->doc->intSubset != NULL)
                element_desc =
                        xmlGetDtdElementDesc
                        (a_current_xml_node->doc->intSubset,
                         a_current_xml_node->name);

        if (element_desc == NULL
            && a_current_xml_node->doc->extSubset) {
                element_desc =
                        xmlGetDtdElementDesc
                        (a_current_xml_node->doc->extSubset,
                         a_current_xml_node->name);
        }

        if (element_desc != NULL
            && element_desc->attributes != NULL) {
                /*build the list of attribute names */
                gboolean add_attribute;
                xmlAttribute *curr_attr =
                        element_desc->attributes;;

                while (curr_attr) {
                        if (a_required_attributes_only == TRUE
                            && curr_attr->def !=
                            XML_ATTRIBUTE_REQUIRED) {
                                add_attribute = FALSE;
                        } else {
                                add_attribute = TRUE;
                        }

                        if (add_attribute == TRUE) {
                                *a_attr_names_compl_list =
                                        g_list_append
                                        (*a_attr_names_compl_list,
                                         (gpointer) curr_attr->
                                         name);
                                result++;
                        }
                        curr_attr = curr_attr->nexth;
                }
        }

        *a_attr_names_compl_list =
                g_list_sort (*a_attr_names_compl_list,
                             (GCompareFunc)
                             g_list_compare_string_elems);
        return result;
}


/**
 *
 *@param a_element_content the element declaration content from the dtd.
 *@param a_children_table out parameter. 
 *The children table. This parameter is valid if and only
 *@param a_app_context the current application context.
 *
 *
 *@return OK if everthing went well, the error code if not. 
 */
enum MLVIEW_PARSING_UTILS_STATUS
 mlview_parsing_utils_get_element_content_table
        (MlViewAppContext * a_app_context,
         xmlElementContent * a_element_content,
         GHashTable ** a_element_content_table) {
        gchar *validation_is_on = NULL;
        enum MLVIEW_PARSING_UTILS_STATUS result = NOK;

        g_return_val_if_fail (a_app_context != NULL,
                              BAD_PARAMETER);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_app_context), BAD_PARAMETER);

        if (a_element_content == NULL)
                return OK;

        validation_is_on =
                mlview_app_context_get_settings_value
                (a_app_context, MLVIEW_STG_K_IS_VALIDATION_ON);

        if (validation_is_on == NULL
            || (strcmp (validation_is_on, MLVIEW_STG_V_YES) !=
                0)) {
                return -1;      /*validation is not turned on !!! */
        }

        if (!*a_element_content_table)
                *a_element_content_table =
                        g_hash_table_new (g_str_hash,
                                          g_str_equal);

        g_return_val_if_fail (*a_element_content_table != NULL,
                              GENERIC_ASSERTION_ERROR);

        switch (a_element_content->type) {
        case XML_ELEMENT_CONTENT_PCDATA:
                result = OK;
                break;
        case XML_ELEMENT_CONTENT_ELEMENT:

                if (a_element_content->name == NULL
                    ||
                    g_hash_table_lookup
                    (*a_element_content_table,
                     a_element_content->name)) {
                        break;
                }

                g_hash_table_insert (*a_element_content_table,
                                     (gpointer)
                                     a_element_content->name,
                                     a_element_content);

                result = OK;
                break;

        case XML_ELEMENT_CONTENT_SEQ:
        case XML_ELEMENT_CONTENT_OR:
                mlview_parsing_utils_get_element_content_table
                        (a_app_context,
                         a_element_content->c1,
                         a_element_content_table);

                mlview_parsing_utils_get_element_content_table
                        (a_app_context,
                         a_element_content->c2,
                         a_element_content_table);

                result = OK;
                break;

        default:
                result = NOK;
                break;

        }

        return result;
}


/**
 *Given an xml node (a_node), builds the required attributes list
 *of that node. 
 *
 *@param a_node the node to add the attributes to.
 *@param a_app_context the current application context.
 *
 *
 */
enum MLVIEW_PARSING_UTILS_STATUS
 mlview_parsing_utils_build_required_attributes_list
        (MlViewAppContext * a_app_context, xmlNode * a_node) {
        GList *attributes_list = NULL,
                *list_ptr = NULL;
        gint nb_of_attributes = 0;
        gchar *validation_is_on = NULL;

        enum MLVIEW_PARSING_UTILS_STATUS result = NOK;

        g_return_val_if_fail (a_app_context != NULL,
                              BAD_PARAMETER);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_app_context), BAD_PARAMETER);
        g_return_val_if_fail (a_node != NULL, BAD_PARAMETER);

        if (mlview_app_context_settings_exist (a_app_context)
            == FALSE) {
                return APP_SETTINGS_NOT_AVAILABLE;
        }

        validation_is_on = mlview_app_context_get_settings_value
                (a_app_context, MLVIEW_STG_K_IS_VALIDATION_ON);

        if (!validation_is_on
            || strcmp (validation_is_on, MLVIEW_STG_V_YES)) {
                return VALIDATION_IS_OFF;
        }

        nb_of_attributes =
                mlview_parsing_utils_build_attribute_name_completion_list
                (a_app_context, a_node, &attributes_list, TRUE);

        if (nb_of_attributes < 0) {
                return NOK;
        }

        else if (nb_of_attributes == 0) {
                return OK;
        }

        result = OK;

        for (list_ptr = attributes_list;
             list_ptr; list_ptr = list_ptr->next) {
                if (!list_ptr->data)
                        continue;
                else {
                        GList *attr_value_set = NULL;
                        xmlAttribute *attr_desc = NULL;
                        gint *last_id_ptr = NULL;
                        guchar *default_value = NULL;
                        xmlAttr *attr = NULL;

                        if (a_node->doc
                            && a_node->doc->intSubset)
                                attr_desc =
                                        xmlGetDtdAttrDesc
                                        (a_node->doc->intSubset,
                                         a_node->name,
                                         (gchar *) list_ptr->
                                         data);

                        if ((attr_desc == NULL && a_node->doc)
                            && a_node->doc->extSubset) {
                                attr_desc =
                                        xmlGetDtdAttrDesc
                                        (a_node->doc->extSubset,
                                         a_node->name,
                                         (gchar *) list_ptr->
                                         data);
                        }

                        if (attr_desc == NULL) {
                                /*attribute unknown by the dtd */
                                continue;
                        }

                        attr_desc->doc = a_node->doc;

                        last_id_ptr =
                                mlview_app_context_get_last_id_ptr
                                (a_app_context);

                        if (last_id_ptr == NULL)
                                continue;

                        attr_value_set =
                                mlview_parsing_utils_build_attribute_value_set
                                (a_app_context,
                                 attr_desc, last_id_ptr);

                        if (attr_value_set
                            && attr_value_set->data)
                                default_value =
                                        attr_value_set->data;
                        else
                                default_value = (guchar *)
                                        "defaultValue";

                        if (!xmlGetProp (a_node,
                                         (gchar *) list_ptr->
                                         data)) {
                                attr = xmlSetProp (a_node,
                                                   (gchar *)
                                                   list_ptr->
                                                   data,
                                                   default_value);
                        }

                        if (attr != NULL
                            && attr_desc->atype ==
                            XML_ATTRIBUTE_ID && a_node->doc) {
                                xmlID *id = NULL;

                                if (a_node->doc->ids == NULL)
                                        a_node->doc->ids =
                                                xmlHashCreate
                                                (0);

                                id = xmlMalloc (sizeof (xmlID));

                                g_assert (id != NULL);

                                id->value =
                                        g_strdup (default_value);

                                id->attr = attr;

                                xmlHashAddEntry
                                        (a_node->doc->ids,
                                         (const xmlChar *)
                                         default_value, id);
                        }

                        g_list_free (attr_value_set);
                        attr_value_set = NULL;
                }
        }

        return result;
}


/**
 *Given an xml node, this method builds the subtree of 
 *this node that is required by the schema (or by the DTD)
 *
 *@param a_node the node to consider.
 *@param a_app_context the current instance of application context.
 */
enum MLVIEW_PARSING_UTILS_STATUS
 mlview_parsing_utils_build_required_children_tree
        (MlViewAppContext * a_app_context, xmlNode ** a_node) {
        gchar *validation_is_on = NULL;
        xmlElement *element_desc = NULL;
        enum MLVIEW_PARSING_UTILS_STATUS status = NOK;

        g_return_val_if_fail (a_app_context != NULL,
                              BAD_PARAMETER);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_app_context), BAD_PARAMETER);
        g_return_val_if_fail (a_node != NULL, BAD_PARAMETER);
        g_return_val_if_fail (*a_node != NULL, BAD_PARAMETER);
        g_return_val_if_fail (((*a_node)->type ==
                               XML_ELEMENT_NODE)
                              || ((*a_node)->type ==
                                  XML_ATTRIBUTE_NODE),
                              BAD_PARAMETER);



        if (mlview_app_context_settings_exist (a_app_context)
            == FALSE) {
                return APP_SETTINGS_NOT_AVAILABLE;
        }

        validation_is_on =
                mlview_app_context_get_settings_value
                (a_app_context, MLVIEW_STG_K_IS_VALIDATION_ON);

        if (!validation_is_on
            || strcmp (validation_is_on, MLVIEW_STG_V_YES)) {
                return VALIDATION_IS_OFF;
        }

        g_return_val_if_fail ((*a_node)->doc != NULL,
                              BAD_PARAMETER);

        if (((*a_node)->doc->intSubset == NULL)
            && ((*a_node)->doc->extSubset == NULL)) {
                return DOCUMENT_HAS_NO_DTD;
        }

        g_return_val_if_fail (((*a_node)->type ==
                               XML_ELEMENT_NODE), BAD_PARAMETER);

        if ((*a_node)->doc->intSubset) {
                element_desc =
                        xmlGetDtdElementDesc
                        ((*a_node)->doc->intSubset,
                         (*a_node)->name);
        }

        if (!element_desc) {
                element_desc =
                        xmlGetDtdElementDesc
                        ((*a_node)->doc->extSubset,
                         (*a_node)->name);
        }

        if (element_desc == NULL) {
                return ELEMENT_DESC_NOT_FOUND;
        }

        if (!strcmp (element_desc->name, "#PCDATA")) {
                xmlNodeSetContent (*a_node, "#PCDATA");
                return OK;
        }

        mlview_parsing_utils_build_required_attributes_list
                (a_app_context, *a_node);

        /*
         *now, walk thru the element 
         *description tree to see what's 
         *going on there
         */
        build_required_element_content (a_app_context,
                                        element_desc->content,
                                        a_node);

        return status;
}

/**
 *
 *@param a_last_id a pointer to the number 
 *part of the last id generated. 
 *This is used only if the current 
 *attribute is an id or an idref. 
 *The function then updates the pointed integer to the
 *new last id value generated. 
 *@param a_attribute_desc the attribute description.
 *@param a_app_context the application context.
 *
 *
 *@return a list of GtkListItem to be put into the GtkList widget. 
 */
GList *mlview_parsing_utils_build_attribute_value_set
        (MlViewAppContext * a_app_context,
         xmlAttribute * a_attribute_desc, gint * a_last_id) {
        GList *result = NULL;
        gchar *id_str = NULL;

        /*xmlNode *a_node ; */

        g_return_val_if_fail (a_app_context != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_app_context), NULL);

        g_return_val_if_fail (a_attribute_desc != NULL, NULL);
        g_return_val_if_fail (a_attribute_desc->name != NULL,
                              NULL);
        g_return_val_if_fail (a_attribute_desc->parent != NULL
                              && a_attribute_desc->parent->doc,
                              NULL);

        g_return_val_if_fail (a_last_id != NULL, NULL);

        switch (a_attribute_desc->atype) {

        case XML_ATTRIBUTE_CDATA:
                break;
        case XML_ATTRIBUTE_ID:

                if (a_attribute_desc->parent->doc->ids == NULL) {
                        a_attribute_desc->parent->doc->ids =
                                xmlHashCreate (0);
                }

                id_str = g_strdup_printf ("_%d", *a_last_id);

                while (xmlHashLookup
                       (a_attribute_desc->parent->doc->ids,
                        id_str)) {

                        (*a_last_id)++;
                        id_str = g_strdup_printf ("_%d",
                                                  *a_last_id);

                }

                result = g_list_append (result,
                                        (gpointer) id_str);
                break;
        case XML_ATTRIBUTE_IDREF:
        case XML_ATTRIBUTE_IDREFS:

                if (a_attribute_desc->parent->doc->ids == NULL) {
                        a_attribute_desc->parent->doc->ids =
                                xmlHashCreate (0);
                }

                xmlHashScan
                        (a_attribute_desc->parent->doc->ids,
                         (xmlHashScanner)
                         mlview_parsing_utils_scan_and_build_ids_list,
                         &result);
                break;

        case XML_ATTRIBUTE_ENTITY:
        case XML_ATTRIBUTE_ENTITIES:

                if (a_attribute_desc->parent->doc->intSubset) {
                        xmlHashScan
                                (a_attribute_desc->parent->doc->
                                 intSubset->entities,
                                 (xmlHashScanner)
                                 mlview_parsing_utils_build_entities_list,
                                 &result);
                }
                if (result == NULL
                    && a_attribute_desc->parent->doc->extSubset)
                {
                        xmlHashScan
                                (a_attribute_desc->parent->
                                 entities, (xmlHashScanner)
                                 mlview_parsing_utils_build_entities_list,
                                 &result);
                }

                break;
        case XML_ATTRIBUTE_NMTOKEN:
        case XML_ATTRIBUTE_NMTOKENS:
                break;
        case XML_ATTRIBUTE_ENUMERATION:
                if (a_attribute_desc->tree
                    && a_attribute_desc->tree->name) {
                        xmlEnumeration *curr =
                                a_attribute_desc->tree;

                        while (curr) {
                                if (curr->name) {
                                        result = g_list_append
                                                (result,
                                                 (gpointer)
                                                 curr->name);
                                }

                                curr = curr->next;
                        }
                }

                break;

        case XML_ATTRIBUTE_NOTATION:
                break;

        default:
                break;
        }

        return result;
}


/**
 *
 *@param a_last_id a pointer to the number part of the last id generated. 
 *This is used only if the current attribute is an id or an idref. 
 *The function then updates the pointed integer to the
 *new last id value generated. 
 *@param a_attribute_desc the attribute description.
 *@param a_app_context the application context.
 *
 *@return a list of GtkListItem to be put into the GtkList widget. 
 */
GList *mlview_parsing_utils_build_graphical_attr_values
        (MlViewAppContext * a_app_context,
         xmlAttribute * a_attribute_desc, gint * a_last_id) {
        GList *result = NULL,
                *value_set;

        g_return_val_if_fail (a_app_context != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_app_context), NULL);
        g_return_val_if_fail (a_attribute_desc != NULL, NULL);
        g_return_val_if_fail (a_attribute_desc->name != NULL,
                              NULL);
        g_return_val_if_fail (a_attribute_desc->doc != NULL,
                              NULL);
        g_return_val_if_fail (a_last_id != NULL, NULL);

        value_set =
                mlview_parsing_utils_build_attribute_value_set
                (a_app_context, a_attribute_desc, a_last_id);

        if (value_set) {
                GList *list_ptr = NULL;

                for (list_ptr = value_set;
                     list_ptr; list_ptr = list_ptr->next) {

                        result = g_list_append
                                (result, (gpointer)
                                 gtk_list_item_new_with_label
                                 ((gchar *) list_ptr->data));
                }
        }

        return result;
}
