/* -*- Mode: C; indent-tabs-mode:nil; c-basic-offset: 8-*- */
/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

/**
 *@file
 *The definition of the #MlViewTreeEditor2 widget. 
 */

#define NEW 1
#ifdef NEW

#include <string.h>
#include <stdlib.h>
#include <glade/glade.h>
#include "mlview-tree-editor2.h"
#include "mlview-node-type-picker.h"
#include "mlview-marshal.h"
#include "mlview-utils.h"
#include "mlview-xml-document.h"
#include "mlview-node-type-picker.h"

#define PRIVATE(mlview_tree_editor2) ((mlview_tree_editor2)->priv)

struct _MlViewTreeEditor2Private {
        /*the xml document being edited. FIXME => handle it lifetime !!! */
        xmlDocPtr xml_doc;

        MlViewXMLDocument *mlview_xml_doc;

        /*the viewable tree that matches xml_doc */
        GtkTreeView *tree_view;

        /*
         *the menu poped up when the user 
         *right clicks on an xml_node
         */
        GtkWidget *xml_node_popup_menu;

        /*a reference to the first selected row */
        GtkTreeRowReference *cur_sel_start;

        /*a reference to the last selected row... 
           not supported atm. */
        /*GtkTreeRowReference * cur_sel_end */

        /*the node type picker used when creating a new node. 
         *MlViewTreeEditor does not have to handle it lifetime
         */
        MlViewNodeTypePicker *node_type_picker;
        
        /*the search dialog*/
        GtkDialog *search_dialog ;

         /*
         *a cache that contains
         *a list of xmlNode/GtkTreeRowReference
         *pair. This is to speed up searches.
         */
        GHashTable *nodes_rows_hash;
        MlViewAppContext *app_context;
        gboolean dispose_has_run ;
        gpointer backup_drag_data_delete ;
        gpointer backup_drag_data_received ;
        gboolean select_issued_by_model ;
};

#define SEARCH_IN_NAMES_CHECK_BUTTON "search-in-names-check-box"
#define SEARCH_IN_ATTRIBUTES_NAMES_CHECK_BUTTON "search-in-attributes-names-check-box"
#define SEARCH_IN_ATTRIBUTES_VALUES_CHECK_BUTTON "search-in-attributes-values-check-box"
#define SEARCH_IN_CONTENT_CHECK_BUTTON "search-in-content-check-box"
#define SEARCHED_TEXT_ENTRY "searched-text-entry"

#define IS_THE_FIND_DIALOG "is-the-find-dialog"

/*===============================================
 *The struct defining the private attributes of 
 *MlViewTreeEditor and it associated macros
 *==============================================*/

enum MlViewTreeEditorColumns  {
        /*hidden column, where the xml node is stored.*/
        XML_NODE_COLUMN = 0,        
        /*
         *contains a boolean that says
         *if the column is editable or not.
         */
        IS_EDITABLE_COLUMN,
        /*the first visible column*/
        START_TAG_COLUMN, 
        /*the second visible column*/
        NODE_TYPE_COLUMN,
        /*
         *This must be the last element
         *of the enum.
         */
        NB_COLUMNS
};

/**
 *The different signals emited
 *by #MlViewTreeEditor2.
 */
enum {
        TREE_CHANGED = 1,
        MARK_SET_TO_NODE,
        MARK_REMOVED_FROM_NODE,
        NODE_CUT,
        NODE_PASTED,
        NODE_ADDED,
        NODE_SELECTED,
        NODE_UNSELECTED,
        NUMBER_OF_SIGNALS
};

enum MlViewTreeEditorColumnsOffsets {
        START_TAG_COLUMN_OFFSET,
        NODE_TYPE_COLUMN_OFFSET
} ;

static guint gv_signals[NUMBER_OF_SIGNALS] = { 0 };
static GtkVBoxClass *gv_parent_class = NULL;
static GtkTargetEntry row_targets[] = {
        {(gchar *)"GTK_TREE_MODEL_ROW", GTK_TARGET_SAME_WIDGET, 0}
        } ;

static void xml_doc_prev_sibling_node_inserted_cb (MlViewXMLDocument *a_this,
                                                   xmlNode *a_sibling_node,
                                                   xmlNode *a_inserted_node,
                                                   MlViewTreeEditor2 *a_editor) ;

static void xml_doc_next_sibling_node_inserted_cb (MlViewXMLDocument *a_this,
                                                   xmlNode *a_sibling_node,
                                                   xmlNode *a_inserted_node,
                                                   MlViewTreeEditor2 *a_editor) ;

static void xml_doc_child_node_added_cb (MlViewXMLDocument *a_this,
                                         xmlNode *a_parent_node,
                                         xmlNode *a_added_node,
                                         MlViewTreeEditor2 *a_editor) ;

static void xml_doc_selected_node_cb (MlViewXMLDocument *a_doc,
                                      xmlNode *a_node,
                                      MlViewTreeEditor2 *a_editor) ;

/****************************************************************
 *The edition popup menu definition array.
 *This menu is poped up on an xml visual node.
 *Note that this pop up menu must be dynamicaly build 
 *when this widget supports
 *Validation.
 *
 *FIXME: do not use this popup menu when validation is on.
 *Maybe build a dynamic menu using GtkMenu widgetware.
 ***************************************************************/


/********************************************
 *declaration of some of the private functions used *
 *and defined by this object.               *
 ********************************************/

static enum MlViewStatus
set_our_dnd_callbacks (MlViewTreeEditor2 *a_this) ;

static void
node_cell_edited_cb (GtkCellRendererText *a_renderer,
                     gchar *a_cell_path,
                     gchar *a_new_text,
                     gpointer a_data) ;

static GtkTreeView *build_tree_view_from_xml_doc (MlViewTreeEditor2 * a_this,
                                                  xmlDoc * a_doc);
static enum MlViewStatus update_visual_node (MlViewTreeEditor2 *a_this, GtkTreeIter *a_iter);
static enum MlViewStatus build_tree_model_from_xml_tree (MlViewTreeEditor2 * a_this,
                                                         const xmlNode * a_node,
                                                         GtkTreeIter * a_ref_iter,
                                                         enum MlViewTreeInsertType a_type,
                                                         GtkTreeModel ** a_model);

/***************************************************
 *private methods required by the GTK typing system
 ***************************************************/

/**
 *The dispose method is suppose to unref all the external
 *reference this object may hold, and free all the members
 *(if applicable) of this object.
 */
static void
mlview_tree_editor2_dispose (GObject *a_this)
{
        MlViewTreeEditor2 *ed=NULL ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)) ;
        ed = MLVIEW_TREE_EDITOR2 (a_this) ;
        g_return_if_fail (ed && PRIVATE (ed)) ;

        if (PRIVATE (ed)->dispose_has_run == TRUE) {
                return ;
        }
        if (PRIVATE (ed)->node_type_picker) {
                gtk_widget_destroy (GTK_WIDGET 
                                    (PRIVATE (ed)->node_type_picker));
                PRIVATE (ed)->node_type_picker = NULL ;
        }
        if (PRIVATE (ed)->search_dialog) {
                gtk_widget_destroy 
                        (GTK_WIDGET (PRIVATE (ed)->search_dialog)) ;
                PRIVATE (ed)->search_dialog = NULL ;
        }
        PRIVATE (ed)->dispose_has_run = TRUE ;
        if (gv_parent_class 
            && G_OBJECT_CLASS (gv_parent_class)->dispose) {
                G_OBJECT_CLASS (gv_parent_class)->dispose (a_this) ;
        }
}

/**
 *Instane finalyzer, responsible of freeing the instance's memory.
 *Is called at the end of the object's destruction process.
 *@param a_this the current instance of #MlViewTreeEditor2
 */
static void
mlview_tree_editor2_finalize (GObject *a_this)
{
        MlViewTreeEditor2 *ed=NULL ;

        g_return_if_fail (a_this
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)) ;
        ed = MLVIEW_TREE_EDITOR2 (a_this) ;
        g_return_if_fail (ed && PRIVATE (ed)) ;
        g_free (PRIVATE (ed)) ;
        PRIVATE (ed) = NULL ;
        if (gv_parent_class) {
                G_OBJECT_CLASS (gv_parent_class)->finalize (a_this) ;
        }
}

/**
 *The Gtk standard class initialyzer of the 
 *MlViewTreeEditor2Class class. 
 *@param a_klass
 */
static void
mlview_tree_editor2_class_init (MlViewTreeEditor2Class * a_klass)
{
        GObjectClass *gobject_class = NULL ;

        g_return_if_fail (a_klass != NULL);

        gv_parent_class = g_type_class_peek_parent (a_klass);
        g_return_if_fail (gv_parent_class) ;
        gobject_class = G_OBJECT_CLASS (a_klass);
        g_return_if_fail (gobject_class);
        gobject_class->dispose = mlview_tree_editor2_dispose  ;
        gobject_class->finalize = mlview_tree_editor2_finalize ;
        /*object_class->destroy = NULL ;mlview_tree_editor2_destroy;*/

        gv_signals[TREE_CHANGED] =
                g_signal_new ("tree-changed",
                              G_TYPE_FROM_CLASS (gobject_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewTreeEditor2Class,
                               tree_changed), NULL, NULL,
                              mlview_marshal_VOID__VOID,
                              G_TYPE_NONE, 0, NULL);

        gv_signals[NODE_CUT] =
                g_signal_new ("node-cut",
                              G_TYPE_FROM_CLASS (gobject_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewTreeEditor2Class, node_cut),
                              NULL, NULL,
                              mlview_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);

        gv_signals[NODE_PASTED] =
                g_signal_new ("node-pasted",
                              G_TYPE_FROM_CLASS (gobject_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewTreeEditor2Class,
                               node_pasted), NULL, NULL,
                              gtk_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);

        gv_signals[NODE_ADDED] =
                g_signal_new ("node-added",
                              G_TYPE_FROM_CLASS (gobject_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewTreeEditor2Class,
                               node_added), NULL, NULL,
                              mlview_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);

        gv_signals[NODE_SELECTED] =
                g_signal_new ("node-selected",
                              G_TYPE_FROM_CLASS (gobject_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewTreeEditor2Class,
                               node_selected), NULL, NULL,
                              mlview_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);

        a_klass->tree_changed = NULL;
        a_klass->node_cut = NULL;
        a_klass->node_added = NULL;
        a_klass->node_pasted = NULL;
        a_klass->node_selected = NULL;

        a_klass->build_tree_view_from_xml_doc = build_tree_view_from_xml_doc;
        a_klass->update_visual_node = update_visual_node;
        a_klass->build_tree_model_from_xml_tree = build_tree_model_from_xml_tree;
}

/**
 *The instance initialyzer of the MlViewTreeEditor2. 
 *
 */
static void
mlview_tree_editor2_init (MlViewTreeEditor2 * a_editor)
{
        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (PRIVATE (a_editor) == NULL);

        PRIVATE (a_editor) =
                g_try_malloc (sizeof (MlViewTreeEditor2Private));
        if (!PRIVATE (a_editor)) {
                mlview_utils_trace_info
                        ("malloc failed, system may be out of memory");
                return;
        }
        memset (PRIVATE (a_editor), 0,
                sizeof (MlViewTreeEditor2Private));
}

/*******************************
 *Private methods and callbacks.
 *******************************/
static void
nodeset_selected_cb (GtkTreeSelection * a_sel, gpointer * a_data)
{
        guint nb_row_selected = 0;
        GtkTreeView *tree_view = NULL;
        MlViewTreeEditor2 *tree_ed = NULL;
        GtkTreeRowReference *row_ref = NULL;
        GtkTreeModel *model = NULL;
        GList *row_sel = NULL;
        GtkTreeIter iter={0} ;
        gboolean is_ok=FALSE ;
        xmlNode *cur_node = NULL;

        g_return_if_fail (a_sel
                          && GTK_IS_TREE_SELECTION (a_sel));
        g_return_if_fail (a_data
                          && MLVIEW_IS_TREE_EDITOR2 (a_data));

        tree_ed = MLVIEW_TREE_EDITOR2 (a_data);
        tree_view = gtk_tree_selection_get_tree_view (a_sel);
        g_return_if_fail (tree_view);
        model = gtk_tree_view_get_model (tree_view);
        g_return_if_fail (model);
        nb_row_selected = gtk_tree_selection_count_selected_rows
                (a_sel);
        /*we just support one row selected at a time now */
        g_return_if_fail (nb_row_selected <= 1);
        if (nb_row_selected == 0){
                /*no node is selected anymore*/
                PRIVATE (tree_ed)->cur_sel_start = NULL ;
                return ;
        }
        /*
         *Now, lets get an iterator on the row selected
         */
        row_sel = gtk_tree_selection_get_selected_rows
                (a_sel, &model);
        g_return_if_fail (row_sel && row_sel->data);
        is_ok = gtk_tree_model_get_iter 
                (model, &iter, row_sel->data) ;
        g_return_if_fail (is_ok == TRUE) ;
        row_ref = mlview_tree_editor2_iter_2_row_ref 
                (tree_ed, &iter) ;
        g_return_if_fail (row_ref) ;
        PRIVATE (tree_ed)->cur_sel_start = row_ref ;
        cur_node = mlview_tree_editor2_get_xml_node 
                (tree_ed, &iter) ;
        g_return_if_fail (cur_node) ;
        mlview_tree_editor2_update_visual_node2 (tree_ed,
                                                 cur_node) ;
        /*
         *Tell every one that this tree editor has
         *selected a given row.
         */
        if (PRIVATE
            (tree_ed)->select_issued_by_model == TRUE) {
                PRIVATE(tree_ed)->select_issued_by_model = FALSE ;
        } else {
                mlview_tree_editor2_select_node (tree_ed, cur_node,
                                                 FALSE, TRUE) ;
        }
        g_signal_emit (G_OBJECT (tree_ed),
                       gv_signals[NODE_SELECTED], 0, row_ref) ;

        g_list_foreach (row_sel, (GFunc)gtk_tree_path_free, NULL) ;
        g_list_free (row_sel) ;
}

static void
xml_doc_node_cut_cb (MlViewXMLDocument *a_this,
                     xmlNode *a_parent_node,

                     xmlNode *a_cut_node,
                     MlViewTreeEditor2 *a_editor)
{
        g_return_if_fail (a_this 
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)) ;

        mlview_tree_editor2_update_node_cut (a_editor, a_parent_node,
                                             a_cut_node) ;
}

static void
xml_doc_prev_sibling_node_inserted_cb (MlViewXMLDocument *a_this,
                                       xmlNode *a_sibling_node,
                                       xmlNode *a_inserted_node,
                                       MlViewTreeEditor2 *a_editor)
{
        g_return_if_fail (a_this
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)) ;

        mlview_tree_editor2_update_sibling_node_inserted (a_editor, a_sibling_node,
                                                          a_inserted_node,
                                                          TRUE, TRUE) ;
}

static void
xml_doc_next_sibling_node_inserted_cb (MlViewXMLDocument *a_this,
                                       xmlNode *a_sibling_node,
                                       xmlNode *a_inserted_node,
                                       MlViewTreeEditor2 *a_editor)
{
        g_return_if_fail (a_this
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)) ;

        mlview_tree_editor2_update_sibling_node_inserted (a_editor, a_sibling_node,
                                                          a_inserted_node,
                                                          FALSE, TRUE) ;
}

static void
xml_doc_child_node_added_cb (MlViewXMLDocument *a_this,
                             xmlNode *a_parent_node,
                             xmlNode *a_added_node,
                             MlViewTreeEditor2 *a_editor)
{
        g_return_if_fail (a_this 
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)) ;
        
        mlview_tree_editor2_update_child_node_added (a_editor, a_parent_node,
                                                     a_added_node, TRUE) ;
}

static void
xml_doc_content_changed_cb (MlViewXMLDocument *a_this,
                            xmlNode *a_node,
                            MlViewTreeEditor2 *a_editor)
{
        g_return_if_fail (a_this 
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && a_editor
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)) ;

        mlview_tree_editor2_update_visual_node2 (a_editor,
                                                 a_node) ;
}

static void
xml_doc_name_changed_cb (MlViewXMLDocument *a_this,
                         xmlNode *a_node,
                         MlViewTreeEditor2 *a_editor)
{
        g_return_if_fail (a_this 
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && a_editor
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)) ;

        mlview_tree_editor2_update_visual_node2 (a_editor, a_node) ;
}

static void
xml_doc_node_attribute_name_changed_cb (MlViewXMLDocument *a_this,
                                        xmlAttr *a_attr,
                                        MlViewTreeEditor2 *a_editor)
{
        g_return_if_fail (a_this 
                          && MLVIEW_XML_DOCUMENT (a_this)
                          && a_editor) ;

        if (!a_attr)
                return ;
        g_return_if_fail (a_attr->parent) ;
        mlview_tree_editor2_update_visual_node2 
                (a_editor, a_attr->parent) ;
}

static void
xml_doc_node_attribute_value_changed_cb (MlViewXMLDocument *a_this,
                                         xmlAttr *a_attr,
                                         MlViewTreeEditor2 *a_editor)
{
        g_return_if_fail (a_this 
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && a_attr
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)) ;

        mlview_tree_editor2_update_visual_node2 (a_editor,
                                                 a_attr->parent) ;
}

static void
xml_doc_node_attribute_removed_cb (MlViewXMLDocument *a_this,
                                   xmlNode *a_node, 
                                   xmlChar *a_name,
                                   MlViewTreeEditor2 *a_editor)
{
        g_return_if_fail (a_this
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)) ;

        mlview_tree_editor2_update_visual_node2 (a_editor, a_node) ;
}

static void
xml_doc_node_namespace_added_cb (MlViewXMLDocument *a_this,
                                 xmlNode *a_node, 
                                 xmlNs *a_ns,
                                 MlViewTreeEditor2 *a_editor)
{
        g_return_if_fail (a_this
                          && a_editor
                          && a_node
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)) ;
        
        mlview_tree_editor2_update_visual_node2 (a_editor, a_node) ;
}

static void
xml_doc_node_namespace_changed_cb (MlViewXMLDocument *a_this,
                                   xmlNode *a_node,
                                   xmlNs *a_ns,
                                   MlViewTreeEditor2 *a_editor)
{
        g_return_if_fail (a_this
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)) ;
        
        mlview_tree_editor2_update_visual_node2 (a_editor, a_node) ;
}

static void
xml_doc_node_namespace_removed_cb (MlViewXMLDocument *a_this,
                                   xmlNode *a_node, 
                                   xmlNs *a_ns,
                                   MlViewTreeEditor2 *a_editor)
{
        g_return_if_fail (a_this
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)) ;
        
        mlview_tree_editor2_update_visual_node2 (a_editor, a_node) ;
}

static void
xml_doc_searched_node_found_cb (MlViewXMLDocument *a_this,
                                xmlNode *a_node_found,
                                MlViewTreeEditor2 *a_editor)
{
        g_return_if_fail (a_this
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)
                          && a_node_found) ;

        mlview_tree_editor2_select_node (a_editor, a_node_found,
                                         TRUE, FALSE) ;
}

static void 
xml_doc_selected_node_cb (MlViewXMLDocument *a_doc,
                          xmlNode *a_node,
                          MlViewTreeEditor2 *a_editor)
{
        g_return_if_fail (a_doc && MLVIEW_XML_DOCUMENT (a_doc)
                          && a_node && a_editor
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)
                          && PRIVATE (a_editor)) ;

        mlview_tree_editor2_select_node (a_editor, a_node, TRUE,
                                         FALSE) ;
}

/**
 *Helper function. Given an attributes node,
 * builds the string "attr1=val1 ... attrN=valN"
 *Note that this function is recursive. 
 *@param a_attr_node the instance of xmlAttr to consider.
 *@param a_result out parameter the resulting string.
 */
static void
xml_attr_to_string (void *a_attr_node, guchar ** a_result)
{
        xmlAttrPtr xml_attr = (xmlAttrPtr) a_attr_node;
        xmlNodePtr xml_node = (xmlNodePtr) a_attr_node;

        static int num_of_use = 0;

        if (num_of_use++ == 0)
                *a_result = NULL;

        if (a_attr_node == NULL)
                return;

        if (xml_attr->type == XML_ATTRIBUTE_NODE) {
                gchar *tmp_str = *a_result,
                        *name;

                if (xml_attr->ns != NULL
                    && xml_attr->ns->prefix != NULL) {
                        name = g_strconcat
                                (xml_attr->ns->prefix, ":",
                                 xml_attr->name, NULL);
                } else {
                        name = g_strdup (xml_attr->name);
                }
                if (tmp_str == NULL)
                        *a_result = g_strdup (name);
                else
                        *a_result = g_strconcat
                                (tmp_str, " ", name, NULL);
                if (tmp_str) {
                        g_free (tmp_str);
                        tmp_str = NULL;
                }
                if (name) {
                        g_free (name);
                        name = NULL;
                }
                if (xml_attr->children)
                        xml_attr_to_string
                                (xml_attr->children, a_result);

                if (xml_attr->next)
                        xml_attr_to_string
                                (xml_attr->next, a_result);
        } else if (xml_node->type == XML_TEXT_NODE) {
                gchar *tmp_str = *a_result;

                if (tmp_str) {
                        *a_result = g_strconcat
                                (tmp_str, "=\"",
                                 xml_node->content, "\"", NULL);
                        g_free (tmp_str);
                        tmp_str = NULL;
                }
        }
}


/**
 *Builds a start tag string 
 *(e.g: <node-name attr0="val0" attr1="val1">)out of
 *an instance of xmlNode *
 *@param a_node the instance of xmlNode * to consider.
 *@return the newly built start tag string.
 */
static guchar *
node_to_string_tag (MlViewAppContext *a_ctxt,
                    xmlNode * a_node)
{
        guchar *result = NULL,
                *content = NULL,
                *escaped_content = NULL;
        gchar *colour_str = NULL ;

        g_return_val_if_fail (a_node != NULL, NULL);        

        colour_str = (gchar*) mlview_tree_editor2_get_colour_string 
                (a_ctxt, a_node->type);
        if (a_node->type == XML_ELEMENT_NODE) {
                guchar *ns_prefix = NULL,
                        *attr_str = NULL,
                        *name = NULL;

                gchar *attr_colour_str = (gchar*) 
                        mlview_tree_editor2_get_colour_string 
                        (a_ctxt, XML_ATTRIBUTE_NODE) ;

                attr_str = mlview_tree_editor2_build_attrs_list_str 
                        (a_ctxt, a_node) ;
                if (a_node->ns != NULL && a_node->ns->prefix) {
                        ns_prefix = g_strconcat (a_node->ns->prefix,
                                                 ":", NULL);
                } else {
                        ns_prefix = NULL;
                }

                if (ns_prefix) {
                        name = g_strconcat (ns_prefix, 
                                            a_node->name, NULL);
                } else {
                        name = g_strdup (a_node->name);
                }
                if (ns_prefix) {
                        g_free (ns_prefix);
                        ns_prefix = NULL;
                }

                if (a_node->children != NULL) {
                        if (attr_str)
                                result = g_strconcat
                                        ("<span foreground=\"",
                                         colour_str, "\">&lt;", 
                                         name, 
                                         "</span> <span foreground=\"",
                                         attr_colour_str, "\">",
                                         attr_str,
                                         "</span><span foreground=\"",
                                         colour_str, "\">&gt;</span>", 
                                         NULL);
                        else
                                result = g_strconcat
                                        ("<span foreground=\"", 
                                         colour_str, "\">&lt;", 
                                         name, "&gt;</span>", NULL);
                } else {        /*empty tag */
                        if (attr_str)
                                result = g_strconcat
                                        ("<span foreground=\"", 
                                         colour_str, "\">&lt;", 
                                         name, 
                                         "</span> <span foreground=\"", 
                                         attr_colour_str,
                                         "\">", attr_str, 
                                         "</span><span foreground=\"", 
                                         colour_str, 
                                         "\"> /&gt;</span>", NULL);
                        else
                                result = g_strconcat
                                        ("<span foreground=\"", 
                                         colour_str, "\">&lt;", name, 
                                         " /&gt;</span>", NULL);
                }

                if (name) {
                        g_free (name);
                        name = NULL;
                }

        } else if (xmlNodeIsText (a_node)) {
                guchar *str = NULL;
                guint esc_content_len = 0 ;
                enum MlViewStatus status = MLVIEW_OK ;

                content = xmlNodeGetContent (a_node);                
                if (content == NULL) {
                        xmlNodeSetContent (a_node, "text");
                        content = xmlNodeGetContent (a_node); }

                status = 
                        mlview_utils_escape_predef_entities_in_str
                        (content, &escaped_content,
                         &esc_content_len) ;
                if (status != MLVIEW_OK) {
                        escaped_content = NULL;
                }
                if (!escaped_content) {
                        str = content ;
                } else {
                        str = escaped_content ;
                }
                result = g_strconcat
                        ("<span foreground=\"", colour_str,
                         "\">", str, "</span>", NULL);
                xmlFree (content);
                if (escaped_content) {
                        g_free (escaped_content) ;
                        escaped_content = NULL ;
                }
        } else if (a_node->type == XML_COMMENT_NODE) {
                content = xmlNodeGetContent (a_node);
                if (content == NULL) {
                        xmlNodeSetContent (a_node, "<!--comment-->");
                        content = xmlNodeGetContent (a_node); 
                        if (!content) {
                                mlview_utils_trace_info 
                                        ("xmlNodeGetContent() failed") ;
                                return NULL ;
                        }
                }
                escaped_content = g_markup_escape_text 
                        (content, 
                         strlen (content));
                result = g_strconcat
                        ("<span foreground=\"",
                         colour_str, "\">&lt;!--", 
                         escaped_content,
                         "--&gt;</span>", NULL);
                if (escaped_content) {
                        g_free (escaped_content);
                        escaped_content = NULL ;
                }
                if (content) {
                        xmlFree (content) ;
                        content = NULL ;
                }
        } else if (a_node->type == XML_PI_NODE) {
                content = xmlNodeGetContent (a_node);
                if (content == NULL) {
                        xmlNodeSetContent
                                (a_node, 
                                 "&lt;?processing instruction node&gt;");
                        content = xmlNodeGetContent (a_node); 
                        if (!content) {
                                mlview_utils_trace_info 
                                        ("xmlNodeGetContent() failed") ;
                                return NULL ;
                        }
                }
                escaped_content = g_markup_escape_text 
                        (content, strlen (content)) ;
                result = g_strconcat
                        ("<span foreground=\"",
                         colour_str, "\">&lt;?", a_node->name, " ", 
                         escaped_content, "?&gt;</span>", NULL);
                if (escaped_content) {
                        g_free (escaped_content) ;
                        escaped_content = NULL ;
                }
                if (content) {
                        xmlFree (content);
                        content = NULL ;
                }
        } else if (a_node->type == XML_DTD_NODE) {
                xmlDtd *node = (xmlDtd*)a_node ;
                guchar *tmp_str = NULL ;
                guchar *dtd_color = (guchar*)
                        mlview_tree_editor2_get_colour_string
                        (a_ctxt, a_node->type) ;

                if (!a_node->name) {
                        mlview_utils_trace_info 
                                ("a node of type XML_DTD_NODE must have a ->name field set!!") ;
                        return NULL ;
                }
                content = g_strconcat ("<span foreground=\"",
                                       dtd_color, "\">"
                                       "&lt;!DOCTYPE ",
                                       a_node->name, NULL) ;
                if (!content) {
                        mlview_utils_trace_info 
                                ("g_strconcat failed") ;
                        return NULL ;
                }
                if (node->ExternalID) {
                        escaped_content = g_markup_escape_text 
                                (node->ExternalID, 
                                 strlen (node->ExternalID)) ;
                        tmp_str = g_strconcat (content, 
                                               " PUBLIC \"", 
                                               escaped_content,
                                               "\"",
                                               NULL) ;
                        if (escaped_content) {
                                g_free (escaped_content) ;
                                escaped_content = NULL ;
                        }
                        if (content) {
                                g_free (content) ;
                        }
                        content = tmp_str ; tmp_str = NULL ;
                        if (node->SystemID) {
                                escaped_content = g_markup_escape_text
                                        (node->SystemID,
                                         strlen (node->SystemID)) ;
                                tmp_str = g_strconcat
                                        (content,
                                         " \"",
                                         escaped_content,
                                         "\"&gt;</span>",
                                         NULL) ;
                                if (escaped_content) {
                                        g_free (escaped_content) ;
                                        escaped_content = NULL ;
                                }
                                if (!tmp_str) {
                                        mlview_utils_trace_info
                                                ("g_strconcat() failed.");
                                        return NULL ;
                                }
                                if (content) {
                                        g_free (content) ;
                                        content = NULL ;
                                }
                                content = tmp_str ; tmp_str = NULL ;
                        }
                } else {
                        if (node->SystemID) {
                                escaped_content = g_markup_escape_text
                                        (node->SystemID, strlen (node->SystemID)) ;
                                tmp_str = g_strconcat
                                        (content,
                                         " SYSTEM \"",
                                         escaped_content,
                                         "\"&gt;</span>",
                                         NULL) ;
                                if (escaped_content) {
                                        g_free (escaped_content) ;
                                        escaped_content = NULL ;
                                }
                                if (!tmp_str) {
                                        mlview_utils_trace_info 
                                                ("g_strconcat failed") ;
                                        return NULL ;
                                }
                                content = tmp_str ; tmp_str = NULL ;
                        }
                }
                result = content ;
                content = NULL ;
        } else if (a_node->type == XML_ENTITY_DECL) {
                xmlEntity *entity = (xmlEntity*) a_node ;
                switch (entity->etype) {
                case XML_INTERNAL_GENERAL_ENTITY:
                        mlview_tree_editor2_internal_general_entity_to_string
                                (a_ctxt, entity, &result) ;
                        break ;
                case XML_EXTERNAL_GENERAL_PARSED_ENTITY:
                        mlview_tree_editor2_external_general_parsed_entity_to_string 
                                (a_ctxt, entity, &result) ;
                        break ;
                case XML_EXTERNAL_GENERAL_UNPARSED_ENTITY:
                        mlview_tree_editor2_external_general_unparsed_entity_to_string
                                (a_ctxt, entity, &result) ;
                        break ;
                case XML_INTERNAL_PARAMETER_ENTITY:
                        mlview_tree_editor2_internal_parameter_entity_to_string
                                (a_ctxt, entity, &result) ;
                        break ;
                case XML_EXTERNAL_PARAMETER_ENTITY:
                        mlview_tree_editor2_external_parameter_entity_to_string
                                (a_ctxt, entity, &result) ;
                        break ;

                case XML_INTERNAL_PREDEFINED_ENTITY:
                        mlview_utils_trace_info
                                ("Oops, dunno how to render "
                                 "XML_INTERNAL_PREDEFINED_ENTITY "
                                 "type of xml entity decl node") ;
                        break ;
                default:
                        mlview_utils_trace_info 
                                ("Unknown entity type") ;
                }
        }

        return result;
}


/**
 *Helper function that builds a GtkTreeModel out of a
 *libxml2 infoset tree.
 *@param a_app_context the application context.
 *@param a_node the xml node to build the gtktreemodel from.
 *@param a_parent_iter an tree iterator that points to the 
 *current parent tree row
 *@param a_model in/out parameter. The built treemodel.
 *@return MLVIEW_OK upon successful completion, an error
 *code otherwise.
 */
enum MlViewStatus 
mlview_tree_editor2_build_tree_model_from_xml_tree (MlViewTreeEditor2 * a_this,
                                                    const xmlNode * a_node,
                                                    GtkTreeIter * a_ref_iter,
                                                    enum MlViewTreeInsertType a_type,
                                                    GtkTreeModel ** a_model)
{
        return MLVIEW_TREE_EDITOR2_CLASS
                (G_OBJECT_GET_CLASS (a_this))->build_tree_model_from_xml_tree
                (a_this, a_node, a_ref_iter, a_type, a_model);
}

static enum MlViewStatus
build_tree_model_from_xml_tree (MlViewTreeEditor2 * a_this,
                                const xmlNode * a_node,
                                GtkTreeIter * a_ref_iter,
                                enum MlViewTreeInsertType a_type,
                                GtkTreeModel ** a_model)
{
        GtkTreeStore *model = NULL;
        GtkTreeIter iter = {0} ;
        GtkTreeIter parent_iter = {0} ;
        GtkTreePath *tree_path = NULL;
        xmlNode *cur_node = NULL,
                *parent_node = NULL;
        gchar *start_tag = NULL;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_node && a_model && *a_model,
                              MLVIEW_BAD_PARAM_ERROR);

        model = GTK_TREE_STORE (*a_model);
        g_return_val_if_fail (model, MLVIEW_BAD_PARAM_ERROR);

        if (!PRIVATE (a_this)->nodes_rows_hash) {
                PRIVATE (a_this)->nodes_rows_hash =
                        g_hash_table_new (g_direct_hash,
                                          g_direct_equal);
                if (!PRIVATE (a_this)->nodes_rows_hash) {
                        mlview_utils_trace_info 
                                ("The system may be out of memory");
                        return MLVIEW_ERROR;
                }
        }

        for (cur_node = (xmlNode *) a_node;
             cur_node; cur_node = cur_node->next) {
                GtkTreeRowReference *row_ref = NULL;

                start_tag = node_to_string_tag 
                        (PRIVATE (a_this)->app_context, cur_node);
                switch (a_type) {
                case INSERT_TYPE_ADD_CHILD:
                        gtk_tree_store_append (model, &iter,
                                               a_ref_iter);
                        break;
                case INSERT_TYPE_INSERT_BEFORE:
                case INSERT_TYPE_INSERT_AFTER:
                        parent_node = cur_node->parent;
                        if (!parent_node) {
                                mlview_utils_trace_info 
                                        ("parent_node failed") ;
                                status = MLVIEW_ERROR ;
                                goto cleanup ;
                                
                        }
                        status = mlview_tree_editor2_get_iter
                                (a_this, parent_node,
                                 &parent_iter);
                        if (status != MLVIEW_OK) {
                                mlview_utils_trace_info 
                                        ("status == MLVIEW_OK failed") ;
                                status = MLVIEW_ERROR ;
                                goto cleanup ;
                        }
                        model = GTK_TREE_STORE
                                (mlview_tree_editor2_get_model
                                 (a_this));
                        if (!model) {
                                mlview_utils_trace_info 
                                        ("model failed") ;
                                status = MLVIEW_ERROR ;
                                goto cleanup ;
                        }
                        if (a_type == INSERT_TYPE_INSERT_BEFORE)
                                gtk_tree_store_insert_before
                                        (model, &iter,
                                         &parent_iter,
                                         a_ref_iter);
                        else
                                gtk_tree_store_insert_after
                                        (model, &iter,
                                         &parent_iter,
                                         a_ref_iter);
                        break;
                default:
                        break;
                }
                tree_path = gtk_tree_model_get_path
                        (GTK_TREE_MODEL (model), &iter);
                if (!tree_path) {
                        mlview_utils_trace_info ("tree_path failed") ;
                        status = MLVIEW_ERROR ;
                        goto cleanup ;
                }
                row_ref = gtk_tree_row_reference_new
                        (GTK_TREE_MODEL (model), tree_path);
                if (!row_ref) {
                        mlview_utils_trace_info ("row_ref failed") ;
                        status = MLVIEW_ERROR ;
                        goto cleanup ;
                }
                g_hash_table_insert
                        (PRIVATE (a_this)->nodes_rows_hash,
                         cur_node, row_ref);
                gtk_tree_store_set (model, &iter,
                                    XML_NODE_COLUMN, cur_node, -1);
                gtk_tree_store_set (model, &iter,
                                    START_TAG_COLUMN, start_tag, -1);
                if (cur_node->type == XML_ELEMENT_NODE) {
                        gtk_tree_store_set (model, &iter,
                                            NODE_TYPE_COLUMN,
                                            "Element Node", 
                                            IS_EDITABLE_COLUMN,
                                            TRUE, -1);
                        if (cur_node->children) {
                                mlview_tree_editor2_build_tree_model_from_xml_tree
                                        (a_this,
                                         cur_node->children,
                                         &iter,
                                         INSERT_TYPE_ADD_CHILD,
                                         a_model);
                        }
                } else if (cur_node->type == XML_TEXT_NODE) {
                        gtk_tree_store_set (model, &iter,
                                            NODE_TYPE_COLUMN,
                                            "Text Node",
                                            IS_EDITABLE_COLUMN,
                                            TRUE, -1);
                } else if (cur_node->type == XML_COMMENT_NODE
                           || cur_node->type == XML_PI_NODE) {
                        gtk_tree_store_set 
                                (model, &iter,
                                 NODE_TYPE_COLUMN, "Comment or PI Node", 
                                 IS_EDITABLE_COLUMN, TRUE,
                                 -1);
                } else if (cur_node->type == XML_DTD_NODE) {
                        gtk_tree_store_set
                                (model, &iter,
                                 NODE_TYPE_COLUMN, "DTD Node",
                                 IS_EDITABLE_COLUMN, FALSE,
                                 -1);
                        if (cur_node->children) {
                                mlview_tree_editor2_build_tree_model_from_xml_tree
                                        (a_this,
                                         cur_node->children,
                                         &iter,
                                         INSERT_TYPE_ADD_CHILD,
                                         a_model);
                        }
                } else if (cur_node->type == XML_ENTITY_DECL) {
                        gtk_tree_store_set
                                (model, &iter,
                                 NODE_TYPE_COLUMN, "ENTITY Declaration Node",
                                 IS_EDITABLE_COLUMN, FALSE,
                                 -1);
                } else {
                        mlview_utils_trace_info ("unknown type of node") ;
                }
                if (start_tag) {
                        g_free (start_tag) ;
                        start_tag = NULL ;
                }
                if (tree_path) {
                        gtk_tree_path_free (tree_path) ;
                        tree_path = NULL ;
                }
                if (a_type == INSERT_TYPE_INSERT_BEFORE
                    || a_type == INSERT_TYPE_INSERT_AFTER) {
                        /*
                         *we are building a tree model which root
                         *node is cur_node so we should not
                         *visit cur_node->next.
                         */
                        break ;
                }
        }
        if (*a_model) {
                g_object_set_data (G_OBJECT (*a_model), 
                                   "MlViewTreeEditor2",
                                   a_this) ;
        }
 cleanup:
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
        if (start_tag) {
                g_free (start_tag) ;
                start_tag = NULL ;
        }
        return status ;
}

/**
 *Builds an instance of GtkTreeModel* out of
 *an instance of xmlDoc*.
 *@param a_context the application context.
 *@param a_doc the instance of xmlDoc* to consider.
 *@param a_model out parameter the resulting model.
 *@return MLVIEW_OK upon successful completion, an error
 *code otherwise.
 */
static enum MlViewStatus
build_tree_model_from_xml_doc (MlViewTreeEditor2 * a_this,
                               const xmlDoc * a_doc,
                               GtkTreeModel ** a_model)
{
        GtkTreeIter iter = { 0 };
        GtkTreeStore *model = NULL;
        GtkTreeRowReference *row_ref = NULL;
        GtkTreePath *tree_path = NULL;
        xmlNode *xml_tree = NULL;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_doc && a_model
                              && *a_model == NULL,
                              MLVIEW_BAD_PARAM_ERROR);

        if (!PRIVATE (a_this)->nodes_rows_hash) {
                PRIVATE (a_this)->nodes_rows_hash =
                        g_hash_table_new (g_direct_hash,
                                          g_direct_equal);
                if (!PRIVATE (a_this)->nodes_rows_hash) {
                        mlview_utils_trace_info 
                                ("The system may be out of memory");
                        return MLVIEW_ERROR;
                }
        }
        model = gtk_tree_store_new (NB_COLUMNS,
                                    G_TYPE_POINTER,
                                    G_TYPE_BOOLEAN,
                                    G_TYPE_STRING,
                                    G_TYPE_STRING);
        g_return_val_if_fail (model, MLVIEW_BAD_PARAM_ERROR);
        *a_model = GTK_TREE_MODEL (model);
        g_return_val_if_fail (model, MLVIEW_BAD_PARAM_ERROR);

        gtk_tree_store_append (model, &iter, NULL);
        tree_path =
                gtk_tree_model_get_path (GTK_TREE_MODEL (model),
                                         &iter);
        g_return_val_if_fail (tree_path, MLVIEW_BAD_PARAM_ERROR);
        row_ref = gtk_tree_row_reference_new
                (GTK_TREE_MODEL (model), tree_path);
        if (!row_ref) {
                mlview_utils_trace_info ("!row_ref failed") ;
                goto cleanup ;
        }
        g_hash_table_insert (PRIVATE (a_this)->nodes_rows_hash,
                             (gpointer) a_doc, row_ref);
        gtk_tree_store_set (model, &iter, XML_NODE_COLUMN, a_doc, -1);
        gtk_tree_store_set (model, &iter, START_TAG_COLUMN,
                            "<span foreground=\"#bbbb00\">XML Document Root</span>", -1);
        gtk_tree_store_set (model, &iter, NODE_TYPE_COLUMN, "", -1);
        xml_tree = a_doc->children;
        status = mlview_tree_editor2_build_tree_model_from_xml_tree
                (a_this, xml_tree,
                 &iter, INSERT_TYPE_ADD_CHILD,
                 (GtkTreeModel **) & model);
 cleanup:
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
        return status;
}

/**
 *Builds an instance of GtkTreeView from an instance
 *of xmlDoc .
 *@param the application context.
 *@param a_doc the instance of xmlDoc to consider.
 *@return the newly built instance of GtkTreeView, or NULL
 *if a probleme arose.
 */

GtkTreeView *
mlview_tree_editor2_build_tree_view_from_xml_doc (MlViewTreeEditor2 * a_this,
                                                  xmlDoc * a_doc)
{
        return MLVIEW_TREE_EDITOR2_CLASS 
                (G_OBJECT_GET_CLASS (a_this))->build_tree_view_from_xml_doc (a_this, a_doc);
}

static GtkTreeView *
build_tree_view_from_xml_doc (MlViewTreeEditor2 * a_this,
                              xmlDoc * a_doc)
{
        GtkTreeView *tree_view = NULL ;
        GtkTreeModel *model = NULL ;
        GtkCellRenderer *renderer = NULL ;
        GtkTreeIter iter = {0} ;
        GtkTreeViewColumn *column = NULL ;
        enum MlViewStatus status = MLVIEW_OK ;
        struct MlViewAppSettings *settings = NULL ;
        gboolean is_ok = TRUE ;
        guint nb_menus = 0 ;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->app_context, 
                              NULL);

        settings = mlview_app_context_get_settings 
                (PRIVATE (a_this)->app_context) ;
        g_return_val_if_fail (settings, NULL) ;

        status = build_tree_model_from_xml_doc
                (a_this, a_doc, &model);
        g_return_val_if_fail (model, NULL);
        is_ok = gtk_tree_model_get_iter_first (model, &iter) ;
        g_return_val_if_fail (is_ok == TRUE, NULL) ;
        tree_view = GTK_TREE_VIEW
                (gtk_tree_view_new_with_model (model));
        g_return_val_if_fail (tree_view, NULL);
        renderer = gtk_cell_renderer_text_new ();
        nb_menus = gtk_tree_view_insert_column_with_attributes
                (tree_view, START_TAG_COLUMN, _("Element start tag"),
                 renderer, "markup", START_TAG_COLUMN, 
                 "editable", IS_EDITABLE_COLUMN, 
                 NULL);
        if (nb_menus)
                column = gtk_tree_view_get_column (tree_view,
                                                   nb_menus -1) ;
        if (column) {
                gtk_tree_view_column_set_resizable 
                        (column, TRUE) ;
        }
        g_signal_connect (G_OBJECT (renderer),
                          "edited", G_CALLBACK (node_cell_edited_cb),
                          a_this) ;
        renderer = gtk_cell_renderer_text_new ();
        gtk_tree_view_insert_column_with_attributes
                (tree_view, NODE_TYPE_COLUMN, _("Element type"),
                 renderer, "text", NODE_TYPE_COLUMN, NULL) ;
        mlview_utils_gtk_tree_view_expand_row_to_depth2
                (tree_view, &iter,
                 settings->tree_editors.default_doc_expansion_depth) ;
        return tree_view;
}


static gboolean
event_cb (GtkWidget * a_widget,
          GdkEvent * a_event, gpointer a_user_data)
{
        MlViewTreeEditor2 *tree_editor = NULL;

        g_return_val_if_fail (a_widget != NULL, FALSE);
        g_return_val_if_fail (GTK_IS_WIDGET (a_widget), FALSE);
        g_return_val_if_fail (a_user_data != NULL, FALSE);
        g_return_val_if_fail (MLVIEW_IS_TREE_EDITOR2
                              (a_user_data), FALSE);
        g_return_val_if_fail (a_event != NULL, FALSE);

        tree_editor = MLVIEW_TREE_EDITOR2 (a_user_data);
        g_return_val_if_fail (tree_editor != NULL, FALSE);
        g_return_val_if_fail (PRIVATE (tree_editor), FALSE);

        switch (a_event->type) {
        case GDK_BUTTON_PRESS:
                if (a_event->button.button == 3) {
                        /*user pressed the right mouse button */
                        GtkWidget *root_widget =
                                gtk_widget_get_toplevel
                                (GTK_WIDGET (tree_editor));

                        g_return_val_if_fail (root_widget !=
                                              NULL, FALSE);

                        gtk_propagate_event (root_widget,
                                             a_event);
                }
                break;
        default:
                break;
        }
        return TRUE;
}

static void
node_cell_edited_cb (GtkCellRendererText *a_renderer,
                     gchar *a_cell_path,
                     gchar *a_new_text,
                     gpointer a_data)
{
        MlViewTreeEditor2 *tree_editor = NULL ;
        GtkTreePath *tree_path = NULL ;
        GtkTreeIter iter = {0} ;
        GtkTreeModel *model = NULL ;
        MlViewXMLDocument *mlview_xml_doc = NULL ;
        enum MlViewStatus status = MLVIEW_OK ;
        GString *element_name = NULL ;
        GList *nv_pair_list = NULL ;
        xmlNode *cur_node = NULL ;
        guchar *start_tag = NULL ;

        g_return_if_fail (a_renderer && a_data && a_cell_path) ;
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR2 (a_data) 
                          && GTK_IS_CELL_RENDERER_TEXT (a_renderer)) ;
        tree_editor = a_data ;
        model = mlview_tree_editor2_get_model (tree_editor) ;
        g_return_if_fail (model) ;
        tree_path = gtk_tree_path_new_from_string (a_cell_path) ;
        g_return_if_fail (tree_path) ;
        status = mlview_tree_editor2_get_cur_sel_start_iter 
                (tree_editor, &iter) ;
        g_return_if_fail (status == MLVIEW_OK) ;
        cur_node = mlview_tree_editor2_get_cur_sel_xml_node (tree_editor) ;
        if (!cur_node) {
                mlview_utils_trace_info ("cur_node failed") ;
                goto cleanup ;
        }
        mlview_xml_doc = mlview_tree_editor2_get_mlview_xml_doc (tree_editor);
        if (!mlview_xml_doc) {
                mlview_utils_trace_info 
                        ("mlview_xml_doc failed") ;
                goto cleanup ;
        }
        start_tag = node_to_string_tag 
                (PRIVATE (tree_editor)->app_context,
                 cur_node) ;
        if (cur_node->type == XML_ELEMENT_NODE) {
                status = mlview_utils_parse_start_tag
                        (a_new_text, &element_name, &nv_pair_list) ;
                if (status != MLVIEW_OK) {
                        g_signal_handlers_block_by_func
                                (a_renderer,
                                 G_CALLBACK (node_cell_edited_cb),
                                 a_data) ;
                        gtk_tree_store_set (GTK_TREE_STORE (model), 
                                            &iter, START_TAG_COLUMN,
                                            start_tag, -1) ;
                        g_signal_handlers_unblock_by_func 
                                (a_renderer,
                                 G_CALLBACK (node_cell_edited_cb),
                                 a_data) ;
                } else {                        
                        mlview_xml_document_set_node_name
                                (mlview_xml_doc, cur_node,
                                 element_name->str, UTF8, TRUE) ;
                        mlview_xml_document_synch_attributes 
                                (mlview_xml_doc, cur_node, nv_pair_list) ;
                }
        } else if (cur_node->type == XML_TEXT_NODE) {
                mlview_xml_document_set_node_content 
                        (mlview_xml_doc, cur_node, a_new_text, UTF8, TRUE) ;
        } else if (cur_node->type == XML_COMMENT_NODE) {
                GString *comment = NULL ;
                status = mlview_utils_parse_comment
                        (a_new_text, &comment) ;
                if (status != MLVIEW_OK) {
                        g_signal_handlers_block_by_func
                                (a_renderer,
                                 G_CALLBACK (node_cell_edited_cb),
                                 a_data) ;
                        gtk_tree_store_set (GTK_TREE_STORE (model),
                                            &iter, START_TAG_COLUMN,
                                            start_tag, -1) ;
                        g_signal_handlers_unblock_by_func
                                (a_renderer,
                                 G_CALLBACK (node_cell_edited_cb),
                                 a_data) ;
                } else {
                        mlview_xml_document_set_node_content
                                (mlview_xml_doc, cur_node,
                                 comment->str, UTF8, TRUE) ;
                }
                if (comment) {
                        g_string_free (comment, TRUE) ;
                        comment = NULL ;
                }
        } else if (cur_node->type == XML_PI_NODE) {
                GString *pi_target = NULL,
                        *pi_param = NULL ;
                status = mlview_utils_parse_pi (a_new_text, 
                                                &pi_target,
                                                &pi_param) ;
                if (pi_target && pi_target->str) {
                        mlview_xml_document_set_node_name
                                (mlview_xml_doc, cur_node,
                                 pi_target->str, UTF8, TRUE) ;
                        if (pi_param && pi_param->str) {
                                mlview_xml_document_set_node_content
                                        (mlview_xml_doc, cur_node,
                                         pi_param->str, UTF8, TRUE) ;
                        }
                } else {
                        g_signal_handlers_block_by_func
                                (a_renderer,
                                 G_CALLBACK (node_cell_edited_cb),
                                 a_data) ;
                        gtk_tree_store_set (GTK_TREE_STORE (model),
                                            &iter, START_TAG_COLUMN,
                                            start_tag, -1) ;
                        g_signal_handlers_unblock_by_func
                                (a_renderer,
                                 G_CALLBACK (node_cell_edited_cb),
                                 a_data) ;
                }
        }

 cleanup:
        if (start_tag) {
                g_free (start_tag) ;
                start_tag = NULL ;
        }
        if (element_name) {
                g_string_free (element_name, TRUE) ;
                element_name = NULL ;
        }
        if (nv_pair_list) {
                GList *cur_item = NULL;
                for (cur_item = nv_pair_list; cur_item ;
                     cur_item = cur_item->next) {
                        if (cur_item->data) {
                                mlview_utils_name_value_pair_free 
                                        (cur_item->data, TRUE) ;
                        }
                }
                g_list_free (nv_pair_list) ;
                nv_pair_list = NULL ;
        }
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
}

/***************************
 *private helper functions
 ***************************/
/**
 *Builds a new xml node which type is given in argument.
 *@param a_node_type the node type.
 *@param the instance of #MlViewXMLDocument that holds the
 *newly created instance of xmlNode.
 *@return the newly created instance of xmlNode, or NULL if something
 *bad happened.
 */
static xmlNode *
new_xml_node (xmlElementType a_node_type,
              MlViewXMLDocument * a_xml_doc)
{
        xmlNodePtr result = NULL;
        xmlDoc *doc = NULL;

        if (a_xml_doc)
                doc = mlview_xml_document_get_xml_document
                        (a_xml_doc);
        switch (a_node_type) {
        case XML_ELEMENT_NODE:
                result = xmlNewNode (NULL, "");
                break;
        case XML_TEXT_NODE:
                result = xmlNewText ("");
                break;
        case XML_CDATA_SECTION_NODE:
                g_return_val_if_fail (doc != NULL, NULL);
                xmlNewCDataBlock (doc, "", 128);
                break;
        case XML_PI_NODE:
                result = xmlNewPI ("", "");
                break;
        case XML_COMMENT_NODE:
                result = xmlNewComment ("");
                break;
        case XML_DOCUMENT_NODE:
        case XML_DOCUMENT_TYPE_NODE:
        case XML_DOCUMENT_FRAG_NODE:
        case XML_NOTATION_NODE:
        case XML_DTD_NODE:
        case XML_ELEMENT_DECL:
        case XML_ATTRIBUTE_DECL:
        case XML_ENTITY_DECL:
        case XML_NAMESPACE_DECL:
        case XML_XINCLUDE_START:
        case XML_XINCLUDE_END:
        default:
                result = xmlNewNode (NULL, "");
                break;
        }
        return result;
}

/**
 *This function is to be called once the user
 *clicks the OK button of the node type picker,
 *which means she wants to add a child node to the
 *currently selected xml node.
 *@param a_this the current instance of #MlViewTreeEditor2 .
 */
static void
handle_nt_picker_ok_button_clicked_to_add_child (MlViewTreeEditor2 *a_this)
{
        guint selected_node_type=0,
		node_addion_status=0;
        MlViewNodeTypePicker *picker;
        xmlNodePtr xml_node = NULL;
        MlViewXMLDocument *xml_doc = NULL;
        xmlNs *ns = NULL;
        guchar *node_name_or_content = NULL,
                *local_name = NULL;
        GtkTreeIter iter={0} ;
        enum MlViewStatus status=MLVIEW_OK ;

        g_return_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)) ;
        picker =
                mlview_tree_editor2_get_node_type_picker
                (a_this);
        g_return_if_fail (picker != NULL);
        node_name_or_content =
                mlview_node_type_picker_get_node_name_or_content
                (picker);
        if (node_name_or_content != NULL
            && !mlview_utils_is_white_string (node_name_or_content)) {
                selected_node_type =
                        mlview_node_type_picker_get_selected_node_type
                        (picker);
                xml_doc =
                        mlview_tree_editor2_get_mlview_xml_doc
                        (a_this);
                xml_node =
                        new_xml_node (selected_node_type, xml_doc);
                switch (selected_node_type) {
                case XML_ELEMENT_NODE:
                case XML_PI_NODE:
                        mlview_utils_parse_full_name
                                (xml_node, node_name_or_content,
                                 &ns, &local_name);
                        if (local_name != NULL) {
                                mlview_xml_document_set_node_name
                                        (xml_doc, xml_node,
                                         local_name, ISO8859_1,
                                         TRUE);
                                g_free (local_name);
                                local_name = NULL;
                        }
                        break;
                default:
                        mlview_xml_document_set_node_content
                                (xml_doc, xml_node,
                                 node_name_or_content, ISO8859_1,
                                 FALSE);
                        break;
                }
                status = mlview_tree_editor2_get_cur_sel_start_iter 
                        (a_this, &iter) ;
                g_return_if_fail (status == MLVIEW_OK) ;
                node_addion_status =
                        mlview_tree_editor2_add_child_node (a_this,
                                                            &iter, 
                                                            xml_node);
                if (!node_addion_status
                    && (selected_node_type == XML_ELEMENT_NODE
                        || selected_node_type == XML_PI_NODE)) {
                        mlview_utils_parse_full_name
                                (xml_node, node_name_or_content,
                                 &ns, &local_name);
                        if (ns) {
                                xmlSetNs (xml_node, ns);
                        } else {
                                xml_node->ns = NULL;
                        }
                        if (local_name) {
                                g_free (local_name);
                                local_name = NULL;
                        }
                }
        }
}

/**
 *This function is to be called once the user
 *clicks the OK button of the node type picker,
 *which means she wants to insert a sibling node to
 *the currently selected xml node.
 *@param a_this the current instance of #MlViewTreeEditor2 .
 */
static void
handle_nt_picker_ok_button_clicked_to_insert_sibling_node (MlViewTreeEditor2 *a_this)
{
        guint selected_node_type=0 ;
        MlViewNodeTypePicker *picker;
        xmlNodePtr xml_node = NULL;
        MlViewXMLDocument *xml_doc = NULL;
        xmlNs *ns = NULL;
        guchar *node_name_or_content = NULL,
                *local_name = NULL;
        GtkTreeIter iter={0} ;
        gboolean *prev_ptr=FALSE ;
        enum MlViewStatus status=MLVIEW_OK ;

        g_return_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)) ;
        picker =
                mlview_tree_editor2_get_node_type_picker (a_this);
        g_return_if_fail (picker != NULL);
        node_name_or_content =
                mlview_node_type_picker_get_node_name_or_content
                (picker);
        if (node_name_or_content != NULL
            && !mlview_utils_is_white_string (node_name_or_content)) {
                selected_node_type =
                        mlview_node_type_picker_get_selected_node_type
                        (picker);
                xml_doc =
                        mlview_tree_editor2_get_mlview_xml_doc
                        (a_this);
                xml_node =
                        new_xml_node (selected_node_type, xml_doc);
                switch (selected_node_type) {
                case XML_ELEMENT_NODE:
                case XML_PI_NODE:
                        mlview_utils_parse_full_name
                                (xml_node, node_name_or_content,
                                 &ns, &local_name);
                        if (local_name != NULL) {
                                mlview_xml_document_set_node_name
                                        (xml_doc, xml_node,
                                         local_name, ISO8859_1,
                                         TRUE);
                                g_free (local_name);
                                local_name = NULL;
                        }
                        break;
                default:
                        mlview_xml_document_set_node_content
                                (xml_doc, xml_node,
                                 node_name_or_content, ISO8859_1,
                                 FALSE);
                        break;
                }
                /*
                 *retrieve a flag set to indicate 
                 *if the sibling has to be inserted 
                 *before or after the current xml node
                 */
                prev_ptr =
                        gtk_object_get_data (GTK_OBJECT
                                             (a_this),
                                             "prev");
                status = mlview_tree_editor2_get_cur_sel_start_iter 
                        (a_this, &iter) ;
                g_return_if_fail (status == MLVIEW_OK) ;
                status = mlview_tree_editor2_insert_sibling_node
                        (a_this, &iter, xml_node, 
                         GPOINTER_TO_INT (prev_ptr)) ;

                if (status == MLVIEW_OK
                    && (selected_node_type == XML_ELEMENT_NODE
                        || selected_node_type == XML_PI_NODE)) {
                        mlview_utils_parse_full_name
                                (xml_node, node_name_or_content,
                                 &ns, &local_name);
                        if (ns) {
                                xmlSetNs (xml_node, ns);
                        } else {
                                xml_node->ns = NULL;
                        }
                        if (local_name) {
                                g_free (local_name);
                                local_name = NULL;
                        }
                        mlview_tree_editor2_update_visual_node 
                                (a_this, &iter) ;
                }
        }
}

/**
 *Returns a copy of the search string typed by the user.
 *@param a GtkDialog that represents the "search" dialog. Must
 *have been constructed by a call to 
 *get_search_dialog().
 *@return a pointer to the search string typed by the user.
 *The return string must _NOT_ be freed by the user.
 */
static const guchar *
get_search_string (GtkDialog *a_search_dialog)
{
        GtkWidget *text_entry = NULL ;

        g_return_val_if_fail (a_search_dialog 
                              && GTK_IS_DIALOG (a_search_dialog),
                              NULL) ;
        
        text_entry = g_object_get_data (G_OBJECT (a_search_dialog),
                                        "SearchEntry") ;
        if (!text_entry || !GTK_IS_ENTRY (text_entry)) {
                mlview_utils_trace_info 
                        ("Retrieving data associated to "
                         "SearchEntry from the Search Dialog failed. "
                         "The Search dialog may not be a valid one.");
                return NULL ;
        }
        return gtk_entry_get_text (GTK_ENTRY (text_entry)) ;
}

/**
 *Gets the search configuration from what the
 *user has selected through the search dialog.
 *@param a_search_dialog the search dialog. Must have
 *been constructed by a call to get_search_dialog() .
 *@param a_config the search configuration (i.e: whether to
 *look into node names, content, attribute names etc ...)
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
static enum MlViewStatus
get_search_config (GtkDialog *a_search_dialog,
                   struct SearchConfig *a_config)
{       
        GtkWidget *widget = NULL ;

        g_return_val_if_fail (a_search_dialog
                          && GTK_IS_DIALOG (a_search_dialog)
                          && a_config, MLVIEW_BAD_PARAM_ERROR) ;

        widget = g_object_get_data (G_OBJECT (a_search_dialog),
                                    "MatchCaseButton") ;        
        g_return_val_if_fail (widget 
                              && GTK_IS_CHECK_BUTTON (widget), 
                              MLVIEW_ERROR) ;
        a_config->ignore_case = 
                gtk_toggle_button_get_active
                (GTK_TOGGLE_BUTTON (widget)) ;
        if (a_config->ignore_case == TRUE) {
                a_config->ignore_case = FALSE ;
        } else {
                a_config->ignore_case = TRUE ;
        }        
        widget = g_object_get_data (G_OBJECT (a_search_dialog),
                                    "SearchInNodeNamesButton") ;
        g_return_val_if_fail (widget 
                              && GTK_IS_CHECK_BUTTON (widget), 
                              MLVIEW_ERROR) ;
        if (gtk_toggle_button_get_active 
            (GTK_TOGGLE_BUTTON (widget)) == TRUE) {
                a_config->where |=  NODE_NAME;
        }
        widget = g_object_get_data (G_OBJECT (a_search_dialog),
                                    "SearchInAttrNamesButton") ;
        g_return_val_if_fail (widget
                              && GTK_IS_CHECK_BUTTON (widget), 
                              MLVIEW_ERROR) ;
        if (gtk_toggle_button_get_active 
            (GTK_TOGGLE_BUTTON (widget)) == TRUE) {
                a_config->where |= NODE_ATTRIBUTE_NAME ;
        }
        widget = g_object_get_data (G_OBJECT (a_search_dialog),
                                    "SearchInAttrValuesButton") ;
        g_return_val_if_fail (widget
                              && GTK_IS_CHECK_BUTTON (widget), 
                              MLVIEW_ERROR) ;
        if (gtk_toggle_button_get_active 
            (GTK_TOGGLE_BUTTON (widget)) == TRUE) {
                a_config->where |=  NODE_ATTRIBUTE_VALUE ;
        }        
        widget = g_object_get_data (G_OBJECT (a_search_dialog),
                                    "SearchInNodeContentButton") ;
        g_return_val_if_fail (widget
                              && GTK_IS_CHECK_BUTTON (widget), 
                              MLVIEW_ERROR) ;
        if (gtk_toggle_button_get_active 
            (GTK_TOGGLE_BUTTON (widget)) == TRUE) {
                a_config->where |=  NODE_CONTENT ;
        }
        a_config->search_string = (guchar*)
                get_search_string (a_search_dialog) ;
        return MLVIEW_OK ;
}

/**
 *Builds and returns the "search" dialog.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@return the dialog.
 */
static GtkWidget *
get_search_dialog (MlViewTreeEditor2 *a_this)
{
        GladeXML *glade_xml = NULL ;
        GtkWidget *widget = NULL, *dialog_widget = NULL ;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              NULL) ;
        if (!PRIVATE (a_this)->search_dialog) {
                guchar *glade_file_path=NULL ;
                glade_file_path =
                gnome_program_locate_file 
                        (NULL, 
                         GNOME_FILE_DOMAIN_APP_DATADIR,
                         PACKAGE "/tree-view-find-dialog.glade", TRUE,
                         NULL) ;
                g_return_val_if_fail (glade_file_path, NULL) ;
                glade_xml = glade_xml_new
                        (glade_file_path,
                         "MlViewTreeViewSearchDialog", NULL) ;
                if (!glade_xml) {
                        mlview_utils_trace_info 
                                ("glade xml file loading failed") ;
                        return NULL ;                        
                }
                dialog_widget =
                        glade_xml_get_widget
                                    (glade_xml, 
                                     "MlViewTreeViewSearchDialog") ;
                if (!dialog_widget) {
                        mlview_utils_trace_info 
                                ("getting widget from glade failed") ;
                        goto cleanup ;
                }
                widget = glade_xml_get_widget (glade_xml,
                                               "SearchEntry") ;
                if (!widget) {
                        mlview_utils_trace_info 
                                ("getting SearchEntry " 
                                 "from glade file failed") ;
                        goto cleanup ;
                }
                g_object_set_data (G_OBJECT (dialog_widget), 
                                   "SearchEntry", widget) ;
                widget = glade_xml_get_widget 
                        (glade_xml,
                         "MatchCaseButton") ;
                if (!widget) {
                        mlview_utils_trace_info 
                                ("getting MatchCaseButton from "
                                 "glade file failed") ;
                        goto cleanup ;
                }
                g_object_set_data (G_OBJECT (dialog_widget),
                                   "MatchCaseButton",
                                   widget) ;
                widget = glade_xml_get_widget 
                        (glade_xml, 
                         "SearchInNodeNamesButton") ;
                if (!widget) {
                        mlview_utils_trace_info 
                                ("getting from SearchInNodeNamesButton "
                                 "glade file failed") ;
                        goto cleanup ;
                }
                g_object_set_data (G_OBJECT (dialog_widget),
                                   "SearchInNodeNamesButton",
                                   widget) ;
                widget = glade_xml_get_widget 
                        (glade_xml, 
                         "SearchInAttrNamesButton") ;
                if (!widget) {
                        mlview_utils_trace_info 
                                ("getting from SearchInAttrNamesButton "
                                 "glade file failed") ;
                        goto cleanup ;
                }
                g_object_set_data (G_OBJECT (dialog_widget),
                                   "SearchInAttrNamesButton",
                                   widget) ;
                widget = glade_xml_get_widget 
                        (glade_xml, 
                         "SearchInAttrValuesButton") ;
                if (!widget) {
                        mlview_utils_trace_info 
                                ("getting from SearchInAttrValuesButton "
                                 "glade file failed") ;
                        goto cleanup ;
                }
                g_object_set_data (G_OBJECT (dialog_widget),
                                   "SearchInAttrValuesButton", 
                                   widget) ;
                widget = glade_xml_get_widget 
                        (glade_xml, 
                         "SearchInNodeContentButton") ;
                if (!widget) {
                        mlview_utils_trace_info 
                                ("getting from SearchInNodeContentButton "
                                 "glade file failed") ;
                        goto cleanup ;
                }
                g_object_set_data (G_OBJECT (dialog_widget),
                                   "SearchInNodeContentButton", 
                                   widget) ;

                PRIVATE (a_this)->search_dialog = GTK_DIALOG (dialog_widget);
                dialog_widget = NULL ;                
        }

 cleanup:
        if (dialog_widget) {
                gtk_widget_destroy (dialog_widget) ;
                dialog_widget = NULL ;
        }
        if (glade_xml) {
                g_object_unref (glade_xml) ;
                glade_xml = NULL ;
        }
        return  GTK_WIDGET (PRIVATE (a_this)->search_dialog) ;
}
typedef gboolean (*DragDataReceivedFunc) (GtkTreeDragDest *,
                                           GtkTreePath *,
                                           GtkSelectionData *) ;
static gboolean
drag_data_received (GtkTreeDragDest *a_drag_dest,
                    GtkTreePath *a_dest_path,
                    GtkSelectionData *a_sel_data)
{
        GtkTreeModel *src_model, *dest_model = NULL ;
        GtkTreePath *src_path = NULL ;
        MlViewTreeEditor2 *editor = NULL ;
        gboolean is_ok = TRUE, result = FALSE ;        
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_drag_dest && a_dest_path
                              && a_sel_data, FALSE) ;
        dest_model = GTK_TREE_MODEL (a_drag_dest) ;
        g_return_val_if_fail (dest_model, FALSE) ;
        is_ok = gtk_tree_get_row_drag_data 
                (a_sel_data, &src_model, &src_path) ;
        g_return_val_if_fail (is_ok == TRUE
                              && src_model == dest_model, 
                              FALSE);
        editor = g_object_get_data (G_OBJECT (a_drag_dest),
                                    "MlViewTreeEditor2") ;
        if (!editor) {
                mlview_utils_trace_info ("editor != NULL failed.") ;
                goto cleanup ;
        }
        /*copy the xml node pointed to by src_path to the clipboard.*/
        status = mlview_tree_editor2_copy_node2 (editor, src_path) ;
        if (status != MLVIEW_OK) {
                mlview_utils_trace_info 
                        ("status == MLVIEW_OK failed.") ;
                goto cleanup ;
        }
        /*
         *paste the newly copied node from the clipboard
         *to the new location
         */
        status = mlview_tree_editor2_paste_node_as_sibling2
                (editor, a_dest_path,
                 TRUE/*paste as previous*/);
        if (status != MLVIEW_OK) {
                mlview_utils_trace_info 
                        ("status == MLVIEW_OK failed") ;
                goto cleanup ;
        }
        result = TRUE ;
 cleanup:
        if (src_path) {
                gtk_tree_path_free (src_path) ;
                src_path = NULL ;
        }
        return result ;
}

typedef gboolean (*DragDataDeleteFunc) (GtkTreeDragSource *, 
                                        GtkTreePath *) ;

static gboolean
drag_data_delete (GtkTreeDragSource *a_drag_src,
                  GtkTreePath *a_path)
{
        GtkTreeModel *model = NULL ;
        MlViewTreeEditor2 *editor = NULL ;
        enum MlViewStatus status =  MLVIEW_OK ;

        editor = g_object_get_data (G_OBJECT (a_drag_src),
                                    "MlViewTreeEditor2") ;
        g_return_val_if_fail (editor, FALSE) ;
        model = GTK_TREE_MODEL (a_drag_src) ;
        g_return_val_if_fail (model, FALSE) ;
        status = mlview_tree_editor2_cut_node2 (editor, a_path) ;
        if (status == MLVIEW_OK) {
                return TRUE ;
        }
        return FALSE ;
}
static enum MlViewStatus
backup_original_dnd_callbacks (MlViewTreeEditor2 *a_this)
{
        GtkTreeDragSourceIface *drag_source_iface = NULL;
        GtkTreeDragDestIface *drag_dest_iface = NULL;
        GtkTreeModel *model = NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;
        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model && GTK_IS_TREE_STORE (model),
                              MLVIEW_ERROR) ;
        drag_source_iface = GTK_TREE_DRAG_SOURCE_GET_IFACE (model) ;
        g_return_val_if_fail (drag_source_iface, MLVIEW_ERROR) ;
        drag_dest_iface = GTK_TREE_DRAG_DEST_GET_IFACE (model) ;

        if (!PRIVATE (a_this)->backup_drag_data_delete) {
                PRIVATE (a_this)->backup_drag_data_delete = 
                        drag_source_iface->drag_data_delete ;
        }
        if (!PRIVATE (a_this)->backup_drag_data_received) {
                PRIVATE (a_this)->backup_drag_data_received =
                        drag_dest_iface->drag_data_received ;
        }
        return MLVIEW_OK ;
}

static enum MlViewStatus
restore_original_dnd_callbacks (MlViewTreeEditor2 *a_this)
{
        GtkTreeDragSourceIface *drag_source_iface ;
        GtkTreeDragDestIface *drag_dest_iface ;
        GtkTreeModel *model = NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;
        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model && GTK_IS_TREE_STORE (model),
                              MLVIEW_ERROR) ;
        drag_source_iface = GTK_TREE_DRAG_SOURCE_GET_IFACE (model) ;
        g_return_val_if_fail (drag_source_iface, MLVIEW_ERROR) ;
        drag_dest_iface = GTK_TREE_DRAG_DEST_GET_IFACE (model) ;

        drag_source_iface->drag_data_delete  =
                PRIVATE (a_this)->backup_drag_data_delete ;
        PRIVATE (a_this)->backup_drag_data_delete = NULL ;
        drag_dest_iface->drag_data_received =
                PRIVATE (a_this)->backup_drag_data_received ;
        PRIVATE (a_this)->backup_drag_data_received = NULL ;

        return MLVIEW_OK ;
}

/**
 *Enables the Drad and drop functionality
 *on the PRIVATE (a_this)->tree_view.
 *This function must be called once a
 *PRIVATE (a_this)->tree_view has been allocated.
 */
static enum MlViewStatus
set_our_dnd_callbacks (MlViewTreeEditor2 *a_this)
{
        GtkTreeDragSourceIface *drag_source_iface ;
        GtkTreeDragDestIface *drag_dest_iface ;
        GtkTreeModel *model = NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;
        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model && GTK_IS_TREE_STORE (model),
                              MLVIEW_ERROR) ;
        drag_source_iface = GTK_TREE_DRAG_SOURCE_GET_IFACE (model) ;
        g_return_val_if_fail (drag_source_iface, MLVIEW_ERROR) ;
        drag_dest_iface = GTK_TREE_DRAG_DEST_GET_IFACE (model) ;
        g_return_val_if_fail (drag_dest_iface, MLVIEW_ERROR) ;
        backup_original_dnd_callbacks (a_this) ;
        drag_source_iface->drag_data_delete  = drag_data_delete ;
        drag_dest_iface->drag_data_received = drag_data_received ;
        return MLVIEW_OK ;
}

/*********************
 *Public methods
 ********************/

/**
 *the standard type id builder of the MlViewTreeEditor2 object. 
 *@return the type id of the MlViewTreeEditor2 object. 
 */
guint
mlview_tree_editor2_get_type (void)
{
        static guint type = 0;

        if (!type) {
                static const GTypeInfo type_info = {
                        sizeof (MlViewTreeEditor2Class),
                        NULL, NULL,
                        (GClassInitFunc)
                                mlview_tree_editor2_class_init,
                        NULL, NULL,
                        sizeof (MlViewTreeEditor2),
                        0,
                        (GInstanceInitFunc)
                        mlview_tree_editor2_init
                };
                type = g_type_register_static (GTK_TYPE_VBOX,
                                               "MlViewTreeEditor2",
                                               &type_info, 0);
        }
        return type;
}


/**
 *Walks through the list of attributes of
 *the instance of xmlNode and
 *construct a string which looks like
 *'attrname0="attrval0" attrname1=attrvall'
 *where each attrnamei is
 *@param a_node the  node to build the attribute list from.
 *@return the newly built attribute list string, or NULL if something
 *bad happened.
 */
guchar *
mlview_tree_editor2_build_attrs_list_str (MlViewAppContext *a_ctxt,
                                          xmlNode * a_node)
{        
        enum MlViewStatus status = MLVIEW_OK ;
        xmlAttrPtr attr_iter = NULL ;
        const gchar *attval_col = "#00FF00",
                *attname_col = mlview_tree_editor2_get_colour_string 
                (a_ctxt, XML_ATTRIBUTE_NODE);
        gchar *attribute, *result = NULL, *tmp = NULL ;
        guchar *escaped_content = NULL;
        guint esc_content_len = 0 ;

        g_return_val_if_fail (a_node 
                              && a_node->type == XML_ELEMENT_NODE, 
                              NULL);

        for (attr_iter = a_node->properties; 
             attr_iter;
             attr_iter = attr_iter->next) {
                xmlChar *content = NULL;
                if (!attr_iter->name)
                        continue ;
                content = xmlGetProp (a_node, attr_iter->name) ;
                if (content) {
                        status = mlview_utils_escape_predef_entities_in_str
                                (content, &escaped_content, 
                                 &esc_content_len) ;
                        if (status != MLVIEW_OK)
                                goto out ;
                        if (!escaped_content) {
                                escaped_content = g_strdup (content) ;
                        }
                        attribute = g_strdup_printf
                                ("<span foreground=\"%s\">%s=<span foreground=\"%s\">\"%s\"</span></span>",
                                 attname_col, attr_iter->name, attval_col, escaped_content);
                } else {
                        attribute = g_strdup_printf
                                ("<span foreground=\"%s\">%s</span>",
                                 attname_col, attr_iter->name);
                }

        out:
                if (content) {
                        xmlFree (content) ;
                        content = NULL ;
                } 
                if (escaped_content) {
                        g_free (escaped_content) ;
                        escaped_content = NULL ;
                }

                if (result) {
                        tmp = g_strdup_printf ("%s %s", result, attribute);
                        g_free (result);
                        result = tmp;
                } else {
                        result = attribute; 
                }                
        }

        return result;
}

/**
 *MlViewTreeEditor2 instance builder.
 *@param a_context the application context.
 *@return the newly built app context or NULL
 *if an error arises.
 */
GtkWidget *
mlview_tree_editor2_new (MlViewAppContext * a_context)
{
        MlViewTreeEditor2 *editor = NULL;

        editor = g_object_new (MLVIEW_TYPE_TREE_EDITOR2, NULL);
       
        mlview_tree_editor2_construct (editor, a_context);

        return GTK_WIDGET (editor);
}

/**
 *The instance initialyzer of #MlViewTreeEditor2.
 *Should be called by the constructor.
 *@param a_editor the current instance of #MlViewTreeEditor2.
 *@param a_context the mlview application context.
 */
void
mlview_tree_editor2_construct (MlViewTreeEditor2 *a_editor, 
                               MlViewAppContext *a_context)
{
        g_return_if_fail (a_editor && PRIVATE (a_editor)) ;

        PRIVATE (a_editor)->app_context = a_context;
        g_signal_connect (G_OBJECT (a_editor),
                          "button_press_event",
                          G_CALLBACK (event_cb), a_editor);    
}

/**
 *Setter of the application context
 *@param a_this the current instance
 *of MlViewTreeEditor2.
 *@param a_app_context the new application context.
 */
void
mlview_tree_editor2_set_application_context (MlViewTreeEditor2 * a_this,
                                              MlViewAppContext * a_app_context) 
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR2
                          (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        PRIVATE (a_this)->app_context = a_app_context;
}

/**
 *Returns the color code of a given type of xml node.
 *@param a_type the type of xml node to consider.  
 *@return string representing the color code of the node
 *in the form "#006FFF" for example.
 */
const gchar*
mlview_tree_editor2_get_colour_string (MlViewAppContext *a_ctxt,
                                       xmlElementType a_type)
{
        struct MlViewAppSettings *settings = NULL ;
        gchar *result = NULL ;

        g_return_val_if_fail (a_ctxt 
                              && MLVIEW_IS_APP_CONTEXT (a_ctxt),
                              NULL) ;
        settings = mlview_app_context_get_settings (a_ctxt) ;
        g_return_val_if_fail (settings, NULL) ;
	switch (a_type) {
        case XML_ELEMENT_NODE:
                result = settings->tree_editors.nodes_colours[MLVIEW_XML_ELEMENT_NODE_COLOUR];
                break ;
	case XML_ATTRIBUTE_NODE:
                result = settings->tree_editors.nodes_colours[MLVIEW_XML_ATTR_VAL_COLOUR] ;
                break ;
	case XML_TEXT_NODE:
                result = settings->tree_editors.nodes_colours[MLVIEW_XML_TEXT_NODE_COLOUR] ;
                break ;
	case XML_CDATA_SECTION_NODE:
                result = (gchar*)"#000000" ;
                break ;
	case XML_ENTITY_REF_NODE:
                result = (gchar*)"#000000" ;
                break ;
	case XML_ENTITY_NODE:
		result = (gchar*)"#000000";
                break ;
	case XML_PI_NODE:
                result = settings->tree_editors.nodes_colours[MLVIEW_XML_PI_NODE_COLOUR] ;
                break ;
	case XML_COMMENT_NODE:
		result = settings->tree_editors.nodes_colours[MLVIEW_XML_COMMENT_NODE_COLOUR] ;
                break ;
	case XML_DOCUMENT_NODE:
                result = settings->tree_editors.nodes_colours[MLVIEW_XML_DOCUMENT_NODE_COLOUR] ;
                break ;
	case XML_DOCUMENT_TYPE_NODE:
                result = (gchar*)"#000000" ;
                break ;
	case XML_DOCUMENT_FRAG_NODE:
                result = (gchar*)"#000000" ;
                break ;
	case XML_NOTATION_NODE:
		result = (gchar*)"#000000";
                break ;
	case XML_HTML_DOCUMENT_NODE:
		result = (gchar*)"#000000";
                break ;
	case XML_DTD_NODE:
                result = settings->tree_editors.nodes_colours[MLVIEW_XML_DTD_NODE_COLOUR] ;
                break ;
	case XML_ELEMENT_DECL:
		result = (gchar*) "#000000";
                break ;
	case XML_ATTRIBUTE_DECL:
		result = (gchar*)"#000000";
                break ;
	case XML_ENTITY_DECL:
                result = settings->tree_editors.nodes_colours[MLVIEW_XML_ENTITY_DECL_NODE_COLOUR] ;
                break ;
	case XML_NAMESPACE_DECL:
		result = (gchar*)"#000000";
                break ;
	case XML_XINCLUDE_START:
		result = (gchar*)"#000000";
                break ;
	case XML_XINCLUDE_END:
		result = (gchar*) "#000000";
                break ;
	default: 
                result = (gchar*)"#000000" ;
	}

        if (!result)
                result = (gchar*)"#000000" ;
        return result ;
}

/**
 *Builds a string of the form 
 *<!ENTITY  internal-general-entity "abracadabra">
 *from a given internal general entity data structure.
 *@param a_entity the input internal general entity.
 *Its ->etype field must be of type XML_INTERNAL_GENERAL_ENTITY, 
 *otherwise the MLVIEW_BAD_PARAM_ERROR is returned.
 *@param a_ctxt the application context.
 *@param a_string out parameter. A pointer to the returned string.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_internal_general_entity_to_string ( MlViewAppContext *a_ctxt,
                                                        xmlEntity *a_entity,
                                                        guchar **a_string)
{
        guchar *sep = NULL,*tmp_str = NULL ;
        gchar *colour = NULL, *esc_name = NULL, *esc_content = NULL ;

        g_return_val_if_fail (a_entity 
                              && a_entity->etype == XML_INTERNAL_GENERAL_ENTITY
                              && a_entity->name
                              && a_entity->content
                              && a_string,
                              MLVIEW_BAD_PARAM_ERROR) ;

        colour = (gchar*) mlview_tree_editor2_get_colour_string 
                (a_ctxt, XML_ENTITY_DECL) ;
        g_return_val_if_fail (colour, MLVIEW_ERROR) ;
        if (strchr (a_entity->content, '"'))
                sep = (guchar*) "'" ;
        else
                sep = (guchar*) "\"" ;

        esc_name = g_markup_escape_text (a_entity->name, 
                                         strlen (a_entity->name)) ;
        esc_content = g_markup_escape_text 
                (a_entity->content,
                 strlen (a_entity->content)) ;
        tmp_str = g_strconcat ("<span foreground=\"",
                               colour,"\">"
                               "&lt;!ENTITY ",
                               esc_name,
                               " ",
                               sep,
                               esc_content,
                               sep,
                               "&gt;",
                               "</span>",
                               NULL) ;
        if (esc_content) {
                g_free (esc_content) ;
                esc_content = NULL ;
        }
        if (esc_name) {
                g_free (esc_name) ;
                esc_name = NULL ;
        }
        if (tmp_str) {
                *a_string = tmp_str ;
                tmp_str = NULL ;
        } else {
                return MLVIEW_OUT_OF_MEMORY_ERROR ;
        }

        return MLVIEW_OK ;
}


/**
 *Builds a string of the form
 *<!ENTITY  external-general-parsed-entity SYSTEM "entity-text.xml">
 *from an external general parsed entity data structure.
 *
 *@param a_entity the entity to consider
 *@param a_string out parameter the result string
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_external_general_parsed_entity_to_string (MlViewAppContext *a_ctxt,
                                                              xmlEntity *a_entity,
                                                              guchar **a_string)
{
        guchar * tmp_str = NULL, *sep = NULL ;
        gchar *colour = NULL, *esc_name = NULL, *esc_content = NULL ;

        g_return_val_if_fail (a_entity 
                              && a_entity->etype == XML_EXTERNAL_GENERAL_PARSED_ENTITY
                              && a_entity->name
                              && a_entity
                              && a_entity->SystemID
                              && a_string,
                              MLVIEW_BAD_PARAM_ERROR) ;

        colour = (gchar*) mlview_tree_editor2_get_colour_string 
                (a_ctxt, XML_ENTITY_DECL) ;
        g_return_val_if_fail (colour, MLVIEW_ERROR) ;

        if (strchr (a_entity->SystemID, '"'))
                sep = (guchar*) "'";
        else
                sep = (guchar*) "\"";

        esc_name = g_markup_escape_text (a_entity->name, 
                                         strlen (a_entity->name)) ;
        esc_content = g_markup_escape_text 
                (a_entity->SystemID,
                 strlen (a_entity->SystemID)) ;

        tmp_str = g_strconcat ("<span foreground=\"",
                               colour,
                               "\">",
                               "&lt;!ENTITY ",
                               esc_name,
                               " SYSTEM ",
                               sep,
                               esc_content,
                               sep,
                               "&gt;",
                               "</span>",
                               NULL) ;

        if (tmp_str) {
                *a_string = tmp_str ;
                tmp_str = NULL ;
        }
        if (esc_name) {
                g_free (esc_name) ;
                esc_name = NULL ;
        }
        if (esc_content) {
                g_free (esc_content) ;
                esc_content = NULL ;
        }
        if (!*a_string) {
                return MLVIEW_OUT_OF_MEMORY_ERROR ;
        }

        return MLVIEW_OK ;
}

/**
 *Builds a string of the form 
 *<!ENTITY  external-general-unparsed-entity SYSTEM "/a/path" NDATA "a-tag">
 *from a given internal general entity data structure.
 *@param a_entity the input internal general entity.
 *Its ->etype field must be of type XML_EXTERNAL_GENERAL_UNPARSED_ENTITY,
 *otherwise the MLVIEW_BAD_PARAM_ERROR is returned.
 *@param a_ctxt the application context.
 *@param a_string out parameter. A pointer to the returned string.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_external_general_unparsed_entity_to_string (MlViewAppContext *a_ctxt,
                                                                xmlEntity *a_entity,
                                                                guchar **a_string)
{
        guchar * tmp_str = NULL, *sep = NULL ;
        gchar *colour = NULL, *esc_name = NULL, 
                *esc_content = NULL, *esc_sysid = NULL ;

        g_return_val_if_fail (a_entity 
                              && a_entity->etype == XML_EXTERNAL_GENERAL_UNPARSED_ENTITY
                              && a_entity->name
                              && a_entity->SystemID
                              && a_string,
                              MLVIEW_BAD_PARAM_ERROR) ;

        colour = (gchar*) mlview_tree_editor2_get_colour_string
                (a_ctxt, XML_ENTITY_DECL) ;
        g_return_val_if_fail (colour, MLVIEW_ERROR) ;

        esc_name = g_markup_escape_text (a_entity->name,
                                         strlen (a_entity->name)) ;
        g_return_val_if_fail (esc_name, MLVIEW_ERROR) ;
        esc_sysid = g_markup_escape_text 
                (a_entity->SystemID,
                 strlen (a_entity->SystemID)) ;
        g_return_val_if_fail (esc_sysid, MLVIEW_ERROR) ;
        if (a_entity->content) {
                esc_content = g_markup_escape_text 
                        (a_entity->content,
                         strlen (a_entity->content) ) ;
                g_return_val_if_fail (esc_content, MLVIEW_ERROR) ;
        }
        if (strchr (a_entity->SystemID, '"'))
                sep = (guchar*) "'";
        else
                sep = (guchar*) "\"";

        if (a_entity->content)
                tmp_str = g_strconcat ("<span foreground=\"",
                                       colour,
                                       "\">"
                                       "&lt;!ENTITY ",
                                       esc_name,
                                       " SYSTEM ",
                                       sep,
                                       esc_sysid,
                                       sep,
                                       " NDATA ",
                                       esc_content,
                                       "&gt;",
                                       "</span>",
                                       NULL) ;
        else
                tmp_str = g_strconcat ("<span foreground=\"",
                                       colour,
                                       "\">",
                                       "&lt;!ENTITY ",
                                       esc_name,
                                       " SYSTEM ",
                                       sep,
                                       esc_sysid,
                                       sep,
                                       "&gt;",
                                       "</span>",
                                       NULL) ;

        if (tmp_str) {
                *a_string = tmp_str ;
                tmp_str = NULL ;
        } 
        if (esc_name) {
                g_free (esc_name) ;
                esc_name = NULL ;
        }
        if (esc_content) {
                g_free (esc_content) ;
                esc_content = NULL ;                
        }
        if (esc_sysid) {
                g_free (esc_sysid) ;
                esc_sysid = NULL ;
        }
        if (!*a_string) {
                return MLVIEW_ERROR ;
        }
        return MLVIEW_OK ;
}

/**
 *Builds a string of the form
 *<!ENTITY  % internal-parameter-entity "the-entity-content">
 *from a given internal general entity data structure.
 *@param a_entity the input internal parameter entity.
 *Its ->etype field must be of type XML_INTERNAL_PARAMETER_ENTITY,
 *otherwise the MLVIEW_BAD_PARAM_ERROR is returned.
 *@param a_ctxt the application context.
 *@param a_string out parameter. A pointer to the returned string.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_internal_parameter_entity_to_string (MlViewAppContext *a_ctxt,
                                                         xmlEntity *a_entity,
                                                         guchar **a_string)
{
        guchar * tmp_str = NULL, *sep = NULL ;
        gchar *colour = NULL, *esc_name = NULL, *esc_content = NULL ;

        g_return_val_if_fail (a_entity 
                              && a_entity->etype == XML_INTERNAL_PARAMETER_ENTITY
                              && a_entity->name
                              && a_entity->content
                              && a_string,
                              MLVIEW_BAD_PARAM_ERROR) ;

        if (strchr (a_entity->content, '"'))
                sep = (guchar*) "'";
        else
                sep = (guchar*) "\"";

        colour = (gchar*) mlview_tree_editor2_get_colour_string
                (a_ctxt, XML_ENTITY_DECL) ;
        g_return_val_if_fail (colour, MLVIEW_ERROR) ;

        esc_name = g_markup_escape_text (a_entity->name,
                                         strlen (a_entity->name)) ;
        esc_content = g_markup_escape_text 
                (a_entity->content,
                 strlen (a_entity->content)) ;
        tmp_str = g_strconcat ("<span foreground=\"",
                               colour,
                               "\">",
                               "&lt;!ENTITY % ",
                               esc_name,
                               " ",
                               sep,
                               esc_content,
                               sep,
                               "&gt;",
                               "</span>",
                               NULL) ;

        if (tmp_str) {
                *a_string = tmp_str ;
        }
        if (esc_name) {
                g_free (esc_name) ;
                esc_name = NULL ;
        }
        if (esc_content) {
                g_free (esc_content) ;
                esc_content = NULL ;
        }
        if (!*a_string) {
                return MLVIEW_ERROR ;
        }
        
        return MLVIEW_OK ;
}

/**
 *Builds a string of the form 
 *<!ENTITY  % external-parameter-entity SYSTEM "/a/path">
 *from a given external parameter entity data structure.
 *@param a_entity the input external parameter entity.
 *Its ->etype field must be of type XML_EXTERNAL_PARAMETER_ENTITY,
 *otherwise the MLVIEW_BAD_PARAM_ERROR is returned.
 *@param a_ctxt the application context.
 *@param a_string out parameter. A pointer to the returned string.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_external_parameter_entity_to_string (MlViewAppContext *a_ctxt,
                                                         xmlEntity *a_entity,
                                                         guchar **a_string)
{
        guchar * tmp_str = NULL, *sep = NULL ;
        gchar *colour = NULL, *esc_name = NULL, *esc_sysid = NULL ;

        g_return_val_if_fail (a_entity
                              && a_entity->etype == XML_EXTERNAL_PARAMETER_ENTITY
                              && a_entity->name
                              && a_entity->SystemID
                              && a_string,
                              MLVIEW_BAD_PARAM_ERROR) ;

        if (strchr (a_entity->SystemID, '"'))
                sep = (guchar*) "'";
        else
                sep = (guchar*) "\"";

        colour = (gchar*) mlview_tree_editor2_get_colour_string
                (a_ctxt, XML_ENTITY_DECL) ;
        g_return_val_if_fail (colour, MLVIEW_ERROR) ;

        esc_name = g_markup_escape_text (a_entity->name,
                                         strlen (a_entity->name)) ;
        g_return_val_if_fail (esc_name, MLVIEW_ERROR) ;
        esc_sysid = g_markup_escape_text 
                (a_entity->SystemID,
                 strlen (a_entity->SystemID)) ;
        g_return_val_if_fail (esc_sysid, MLVIEW_ERROR) ;

        tmp_str = g_strconcat ("<span foreground=\"",
                               colour,
                               "\">",
                               "&lt;!ENTITY % ",
                               esc_name,
                               " SYSTEM ",
                               sep,
                               esc_sysid,
                               sep,
                               "&gt;",
                               "</span>",
                               NULL) ;

        if (tmp_str) {
                *a_string = tmp_str ;
                tmp_str = NULL ;
        } 

        if (esc_name) {
                g_free (esc_name) ;
                esc_name = NULL ;
        }
        if (esc_sysid) {
                g_free (esc_sysid) ;
                esc_sysid = NULL ;
        }
        if (!*a_string) {
                return MLVIEW_ERROR ;
        }
        return MLVIEW_OK ;
}

/**
 *Gets the instance of GtkTreeModel (namely an instance of
 *GtkTreeStore) of the current xml document.
 *@param a_this the current instance of #MlViewEditor.
 *@return the instance of GtkTreeModel, or NULL.
 */
GtkTreeModel *
mlview_tree_editor2_get_model (MlViewTreeEditor2 * a_this)
{
        GtkTreeView *tree_view = NULL;
        GtkTreeModel *model = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this),
                              NULL);
        tree_view = mlview_tree_editor2_get_tree_view (a_this);
        g_return_val_if_fail (tree_view, NULL);
        model = gtk_tree_view_get_model (tree_view);
        g_return_val_if_fail (model, NULL);
        return model;
}

/**
 *Gets the instance of GtkTreeView that
 *features the xml document graphically.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@return the GtkTreeView graphical tree.
 */
GtkTreeView *
mlview_tree_editor2_get_tree_view (MlViewTreeEditor2 * a_this)
{
        g_return_val_if_fail (a_this && PRIVATE (a_this), NULL);
        g_return_val_if_fail
                (MLVIEW_IS_TREE_EDITOR2 (a_this), NULL);

        return PRIVATE (a_this)->tree_view;
}

/**
 *Getter of the application context.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@return the app context contained in this instance of MlViewTreeEditor2.
 */
MlViewAppContext *
mlview_tree_editor2_get_application_context (MlViewTreeEditor2 * a_this) 
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_TREE_EDITOR2 (a_this),
                              NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);

        return PRIVATE (a_this)->app_context;
}

/**
 *Getter of the private nodes_row_hash hashtable
 *@param a_this the current instance of #MlViewTreeEditor2
 *@return return the private nodes_row_hash hashtable
 */
GHashTable *
mlview_tree_editor2_get_nodes_rows_hash (MlViewTreeEditor2 *a_this)
{
        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this),
                              NULL) ;
        return PRIVATE (a_this)->nodes_rows_hash ;
}

/**
 *Setter of the private nodes_row_hash hastable
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_nodes_rows_hash the new instance of the hashtable to set.
 */
void
mlview_tree_editor2_set_nodes_rows_hash (MlViewTreeEditor2 *a_this,
                                         GHashTable *a_nodes_rows_hash)
{
        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)
                          && PRIVATE (a_this)) ;

        PRIVATE (a_this)->nodes_rows_hash = a_nodes_rows_hash ;
}

/**
 *Sets the title of the xml DOM to @a_file_path.
 *Updates this information in the tree editor and in the
 *XML DOM. 
 *@param a_file_path the new file path of xml document.
 *@param a_this the current tree editor.
 */
void
mlview_tree_editor2_set_xml_document_path (MlViewTreeEditor2 * a_this, 
                                           gchar * a_file_path) 
{
        GtkTreeViewColumn *tree_column = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (a_file_path != NULL);

        if (!PRIVATE (a_this)->xml_doc)
                return;

        if (PRIVATE (a_this)->xml_doc->name) {
                g_free (PRIVATE (a_this)->xml_doc->name);
                PRIVATE (a_this)->xml_doc->name = NULL ;
        }
        PRIVATE (a_this)->xml_doc->name = g_strdup (a_file_path);
        tree_column = gtk_tree_view_get_column
                (PRIVATE (a_this)->tree_view,
                 START_TAG_COLUMN_OFFSET) ;
        g_return_if_fail (tree_column);

        gtk_tree_view_column_set_title
                (tree_column, PRIVATE (a_this)->xml_doc->name);
}

/**
 *Return the first row of the currently selected row.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@return the first row of the set of selected rows.
 */
GtkTreeRowReference *
mlview_tree_editor2_get_sel_start (MlViewTreeEditor2 * a_this)
{
        GtkTreeRowReference *row_ref = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this), NULL);
        return row_ref = PRIVATE (a_this)->cur_sel_start;
}

/**
 *Get the instance of #MlViewXMLDocument associated to this
 *instance of #MlViewTreeEditor2
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@return the instance of #MlViewXMLDocument associated to this
 *the tree editor or NULL.
 */
MlViewXMLDocument *
mlview_tree_editor2_get_mlview_xml_doc (MlViewTreeEditor2 *a_this)
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              NULL) ;
        return PRIVATE (a_this)->mlview_xml_doc ;
}

/**
 *Gets the node type picker associated to this
 *instance of #MlViewTreeEditor2. If the picker doesn't
 *exists, this function creates it.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@return the node type picker, or NULL if something bad happened.
 */
MlViewNodeTypePicker *
mlview_tree_editor2_get_node_type_picker (MlViewTreeEditor2 *a_this)
{
        GtkWidget *res=NULL ;
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this), NULL) ;

        if (!PRIVATE (a_this)->node_type_picker) {
                res = mlview_node_type_picker_new 
                        (PRIVATE (a_this)->app_context) ;
                g_return_val_if_fail (res, NULL) ;
                if (! MLVIEW_IS_NODE_TYPE_PICKER (res))
                {
                        mlview_utils_trace_info 
                                ("Expected a Node type picker, found "
                                 "an unknown type" ) ;
                        return NULL ;
                }
                gtk_window_set_modal (GTK_WINDOW (res), TRUE) ;
                mlview_tree_editor2_set_node_type_picker
                        (a_this, MLVIEW_NODE_TYPE_PICKER (res)) ;
        }
        res = GTK_WIDGET (PRIVATE (a_this)->node_type_picker) ;
        return MLVIEW_NODE_TYPE_PICKER (res) ;
}

/**
 *Sets the node type picker associated to
 *the current instance of #MlViewTreeEditor2.
 *If a node type picker was already associated to
 *the current instance of #MlViewTreeEditor2, unref it
 *before associating the new one.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_set_node_type_picker (MlViewTreeEditor2 *a_this,
                                          MlViewNodeTypePicker *a_picker)

{
        g_return_val_if_fail (a_this 
                              && MLVIEW_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_picker 
                              && MLVIEW_IS_NODE_TYPE_PICKER (a_picker),
                              MLVIEW_BAD_PARAM_ERROR) ;

        if (PRIVATE (a_this)->node_type_picker) {
                g_object_unref 
                        (G_OBJECT (PRIVATE (a_this)->node_type_picker)) ;
        }
        PRIVATE (a_this)->node_type_picker = a_picker ;
        return MLVIEW_OK ;
}

/**
 *Gets the GtkTreeRowReference associated
 *to the node pointed to by a_iter. User doesn't
 *need to free the returned GtkTreeRowReference because
 *MlViewTreeEditor2 takes care of it.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_iter the node iterator to consider.
 *@return the relevant GtkTreeRowReference, or NULL in case
 *of an error.
 */
GtkTreeRowReference *
mlview_tree_editor2_iter_2_row_ref (MlViewTreeEditor2 *a_this,
                                    GtkTreeIter *a_iter)
{
        GtkTreeRowReference *result=NULL ;
        xmlNode *xml_node=NULL ;
        GtkTreeModel *model=NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->nodes_rows_hash
                              && a_iter, NULL) ;

        model = mlview_tree_editor2_get_model (a_this) ;
        gtk_tree_model_get (model, a_iter, XML_NODE_COLUMN,
                            &xml_node, -1) ;
        g_return_val_if_fail (xml_node, NULL) ;
        result = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash,
                 xml_node) ;
        return result ;
}

/**
 *A very important method in this class 
 *(and even in the whole MlView software).
 *It takes an xml document in parameter, 
 *builds the graphical view that matches it and displays it. 
 *
 *@param a_xmldoc the xml document to edit.
 *@param a_editor the current instance of MlViewTreeEditor.
 *@return MLVIEW_OK upon successful completion,
 *an error code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_edit_xml_doc (MlViewTreeEditor2 * a_this,
                                  MlViewXMLDocument * a_doc,
                                  guchar * a_doc_name)
{
        GtkTreeView *tree_view = NULL;
        GtkTreeSelection *selection = NULL;
        GtkWidget *scr_win = NULL;
        struct MlViewAppSettings *settings = NULL ;
        xmlDoc *xml_doc = NULL;

        g_return_val_if_fail (a_this != NULL, MLVIEW_ERROR);
        g_return_val_if_fail (MLVIEW_IS_TREE_EDITOR2 (a_this),
                              MLVIEW_ERROR);
        g_return_val_if_fail (PRIVATE (a_this)
                              && PRIVATE (a_this)->app_context,
                              MLVIEW_ERROR);
        g_return_val_if_fail (a_doc != NULL, MLVIEW_ERROR);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT
                              (a_doc), MLVIEW_ERROR);

        settings = mlview_app_context_get_settings
                (PRIVATE (a_this)->app_context) ;
        g_return_val_if_fail (MLVIEW_ERROR, MLVIEW_ERROR) ;
        xml_doc =
                mlview_xml_document_get_xml_document
                ((MlViewXMLDocument *) a_doc);
        g_return_val_if_fail (xml_doc != NULL, -1);
        PRIVATE (a_this)->mlview_xml_doc = a_doc;

        tree_view = mlview_tree_editor2_build_tree_view_from_xml_doc
                (a_this, xml_doc);
        g_assert (tree_view != NULL);
        
        if (PRIVATE (a_this)->tree_view) {
                gtk_widget_destroy
                        (GTK_WIDGET (PRIVATE
                                   (a_this)->tree_view));
        }
        PRIVATE (a_this)->tree_view = tree_view;
        selection = gtk_tree_view_get_selection (tree_view);
        g_return_val_if_fail (selection, MLVIEW_ERROR);
        gtk_tree_selection_set_mode
                (selection, GTK_SELECTION_SINGLE);
        g_signal_connect (G_OBJECT (selection),
                          "changed",
                          G_CALLBACK (nodeset_selected_cb),
                          a_this);        
        scr_win = gtk_scrolled_window_new (NULL, NULL);
        gtk_container_add (GTK_CONTAINER (scr_win),
                           GTK_WIDGET (tree_view));
        gtk_box_pack_start (GTK_BOX (a_this), scr_win,
                            TRUE, TRUE, 0);
        gtk_widget_show_all (GTK_WIDGET (a_this));
        PRIVATE (a_this)->xml_doc = xml_doc;
        set_our_dnd_callbacks (a_this) ;
        gtk_tree_view_enable_model_drag_source 
                (tree_view, GDK_BUTTON1_MASK|GDK_BUTTON2_MASK,
                 row_targets, G_N_ELEMENTS (row_targets),
                 GDK_ACTION_MOVE | GDK_ACTION_COPY) ;
        gtk_tree_view_enable_model_drag_dest
                (tree_view, row_targets, 
                 G_N_ELEMENTS (row_targets),
                 GDK_ACTION_MOVE|GDK_ACTION_COPY) ;
        
        return 0;
}

/**
 *Creates a new xml document inside the tree editor.
 *@param a_this the current instance of #MlViewTreeEditor.
 *@param a_doc the mlview xml document to edit.
 */
void
mlview_tree_editor2_create_new_xml_doc (MlViewTreeEditor2 *a_this,
                                        MlViewXMLDocument *a_doc)
{
        g_return_if_fail (a_this && a_doc);
        mlview_tree_editor2_edit_xml_doc (a_this, a_doc, NULL);
}

/**
 *Set the root element of the visual xml doc.
 *Note that the xml doc must be empty ,that is, it must
 *not have any root element.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_node the new document root.
 *@param a_emit_signals if is TRUE, this function emits the
 *"node-added" and the "tree-changed" signals in case of successful
 *completion.
 */
void
mlview_tree_editor2_set_root_element (MlViewTreeEditor2 *a_this,
                                      xmlNode *a_node, 
                                      gboolean a_emit_signals)
{
        GtkTreeIter iter={0} ;
        xmlNode *node=NULL ;
        GtkTreeModel *model=NULL ;
        gboolean is_ok=TRUE ;
        GtkTreeRowReference *row_ref=NULL ;
        enum MlViewStatus status=MLVIEW_OK ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)
                          && PRIVATE (a_this)
                          && PRIVATE (a_this)->xml_doc
                          && PRIVATE (a_this)->tree_view
                          && a_node) ;
        node = xmlDocGetRootElement (PRIVATE (a_this)->xml_doc) ;
        g_return_if_fail (node == NULL) ;
        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_if_fail (model) ;
        xmlDocSetRootElement (PRIVATE (a_this)->xml_doc,
                              a_node) ;        
        is_ok = gtk_tree_model_get_iter_first (model, &iter) ;
        g_return_if_fail (is_ok == TRUE) ;
        status = mlview_tree_editor2_build_tree_model_from_xml_tree (a_this, a_node, &iter, 
                                                                     INSERT_TYPE_ADD_CHILD,
                                                                     &model) ;        
        g_return_if_fail (status == MLVIEW_OK) ;        
        if (a_emit_signals == TRUE) {
                row_ref = g_hash_table_lookup 
                        (PRIVATE (a_this)->nodes_rows_hash, a_node) ;
                g_return_if_fail (row_ref) ;
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_ADDED],0,
                               row_ref) ;
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[TREE_CHANGED], 0) ;
        }
}

/**
 *Gets a tree iterator that points to the tree row associated
 *to a given xml node.
 *An example of use of this function is:
 *GtkTreeIter iter ;
 *enum MlViewStatus status ;
 *status = mlview_tree_editor2_get_iter (a_this,
 *                                       an_xml_node,
 *                                       &iter) ;
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_node the xml node to consider.
 *@param a_iter out parameter the place where to copy the
 *iterator. Must have been allocated by the caller.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_get_iter (MlViewTreeEditor2 * a_this,
                              xmlNode * a_node,
                              GtkTreeIter * a_iter)
{
        GtkTreeModel *model = NULL;
        GtkTreeRowReference *row_ref = NULL;
        GtkTreePath *tree_path = NULL;
        gboolean is_ok = FALSE;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->nodes_rows_hash
                              && a_iter, MLVIEW_BAD_PARAM_ERROR);

        model = mlview_tree_editor2_get_model (a_this);
        g_return_val_if_fail (model, MLVIEW_ERROR);

        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash, a_node);
        if (!row_ref)
                return MLVIEW_NODE_NOT_FOUND_ERROR;
        tree_path = gtk_tree_row_reference_get_path (row_ref);
        g_return_val_if_fail (tree_path, MLVIEW_ERROR);
        is_ok = gtk_tree_model_get_iter (model, a_iter,
                                         tree_path);
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
        if (is_ok == TRUE)
                return MLVIEW_OK;
        return MLVIEW_ERROR;
}

/**
 *Gets a row reference pointer to the first node
 *a user selection.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@return the row reference pointer.
 */
GtkTreeRowReference* 
mlview_tree_editor2_get_cur_sel_start (MlViewTreeEditor2 *a_this)
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              NULL) ;

        return PRIVATE (a_this)->cur_sel_start ;
}

/**
 *Gets an iterator on the first element of a selection.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_iter out parameter, the iterator to fill. Must be allocated
 *by the caller.
 *@return MLVIEW_OK upon successful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_get_cur_sel_start_iter (MlViewTreeEditor2 *a_this,
                                            GtkTreeIter *a_iter)
{       
        GtkTreeModel *model=NULL ;
        GtkTreePath *tree_path=NULL ;
        gboolean is_ok=TRUE ;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_iter,
                              MLVIEW_BAD_PARAM_ERROR) ;

        if (!PRIVATE (a_this)->cur_sel_start) {
                return MLVIEW_NODE_NOT_FOUND_ERROR ;
        }
        tree_path = gtk_tree_row_reference_get_path 
                (PRIVATE (a_this)->cur_sel_start) ;
        g_return_val_if_fail (tree_path, MLVIEW_ERROR) ;
        model = mlview_tree_editor2_get_model (a_this) ;
        if (!model) {
                mlview_utils_trace_info ("model failed") ;
                status = MLVIEW_OK ;
                goto cleanup ;
        }
        is_ok = gtk_tree_model_get_iter (model, a_iter, tree_path) ;
        if (is_ok != TRUE) {
                mlview_utils_trace_info ("is_ok == TRUE failed") ;
                status = MLVIEW_OK ;
                goto cleanup ;
        }

 cleanup:
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
        return status ;
}

/**
 *Gets the xml node associated to the current selected
 *Row.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@return the xml node or NULL if something bad happened.
 */
xmlNode * 
mlview_tree_editor2_get_cur_sel_xml_node (MlViewTreeEditor2 *a_this)
{
        GtkTreeIter iter = {0} ;
        enum MlViewStatus status = MLVIEW_OK ;
        xmlNode *result ;
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this), NULL) ;
        status = mlview_tree_editor2_get_cur_sel_start_iter (a_this, &iter) ;
        g_return_val_if_fail (status == MLVIEW_OK, NULL) ;
        result = mlview_tree_editor2_get_xml_node (a_this, &iter) ;
        return result ;
}

/**
 *Gets the instance of xmlNode* associated to
 *an instance of GtkTreeIter*
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_iter the iterator to consider.
 *@return the xmlNode associated to the iterator or NULL.
 */
xmlNode *
mlview_tree_editor2_get_xml_node (MlViewTreeEditor2 * a_this,
                                  GtkTreeIter * a_iter)
{
        GtkTreeModel *model = NULL;
        xmlNode *result = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_iter, NULL);

        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model, NULL);
        gtk_tree_model_get (model, a_iter,
                            XML_NODE_COLUMN, &result, -1);
        return result;
}

/**
 *Gets the instance of xmlNode* associated to
 *an instance of GtkTreeRowReference.
 *@param a_this the current instance of #MlViewTreeEditor.
 *@param a_iter the instance of GtkTreeRowReference to consider.
 *@return the xmlNode associated to the iterator or NULL.
 */
xmlNode *
mlview_tree_editor2_get_xml_node2 (MlViewTreeEditor2 * a_this,
                                   GtkTreeRowReference * a_row_ref)
{
        GtkTreeModel *model=NULL;
        GtkTreePath *tree_path=NULL ;
        xmlNode *result=NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_row_ref, NULL);

        model = mlview_tree_editor2_get_model (a_this);
        g_return_val_if_fail (model, NULL);
        tree_path = gtk_tree_row_reference_get_path (a_row_ref) ;
        g_return_val_if_fail (tree_path, NULL) ;
        result = mlview_tree_editor2_get_xml_node3 (a_this, tree_path) ;
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
        return result;
}

/**
 *Gets the instance of xmlNode associated to an instance
 *of GtkTreePath.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_path the path to a graphical GtkTreeModel node.
 *@return the xml node associated to a_path or NULL.
 */
xmlNode *
mlview_tree_editor2_get_xml_node3 (MlViewTreeEditor2 *a_this,
                                   GtkTreePath *a_path)
{
        xmlNode *result = NULL ;
        GtkTreeModel *model = NULL ;
        gboolean is_ok = TRUE ;
        GtkTreeIter iter = {0} ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_path,
                              NULL) ;
        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model, NULL) ;
        is_ok=gtk_tree_model_get_iter (model, &iter, a_path) ;
        if (is_ok != TRUE) {
                mlview_utils_trace_info ("is_ok == TRUE failed") ;
                return NULL ;
        }
        gtk_tree_model_get (model, &iter,
                            XML_NODE_COLUMN, &result, -1);
        return result ;
}

/**
 *Adds a child node to the node pointed to by a_parent_iter.
 *@param a_this the current instance of #MlViewTreeEditor
 *@param a_parent_iter an iterator to the parent of the node
 *to be added.
 *@param a_node the xml node to add.
 *@return MVLIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_add_child_node (MlViewTreeEditor2 * a_this,
                                    GtkTreeIter * a_parent_iter,
                                    xmlNode * a_node)
{
        xmlNode *parent_xml_node = NULL,
                *added_node = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_parent_iter
                              && a_node, MLVIEW_BAD_PARAM_ERROR);

        parent_xml_node = mlview_tree_editor2_get_xml_node
                (a_this, a_parent_iter);
        g_return_val_if_fail (parent_xml_node, MLVIEW_ERROR);
        added_node = mlview_xml_document_add_child_node
                (PRIVATE (a_this)->mlview_xml_doc,
                 parent_xml_node, a_node, TRUE,
                 TRUE);
        if (added_node)
                return MLVIEW_OK;
        return MLVIEW_ERROR;
}

/**
 *Asks the user for the type of node he wants to add and adds it.
 *@param a_this the current instance of #MlViewTreeEditor2
 */
void
mlview_tree_editor2_add_child_node_interactive (MlViewTreeEditor2 *a_this)
{
        MlViewNodeTypePicker *picker=NULL ;
        gint button=0 ;
        xmlNode *cur_node=NULL ;
        
        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)
                          && PRIVATE (a_this)->cur_sel_start) ;

        cur_node = mlview_tree_editor2_get_xml_node2 
                (a_this, PRIVATE (a_this)->cur_sel_start) ;
        g_return_if_fail (cur_node) ;
        /*only an element node should have children*/
        if (cur_node->type != XML_ELEMENT_NODE)
                return ;
        picker = mlview_tree_editor2_get_node_type_picker (a_this) ;
        g_return_if_fail (picker) ;
        mlview_node_type_picker_set_title (picker, 
                                           _("add a child node")) ;
        mlview_node_type_picker_build_element_name_choice_list
                (picker, ADD_CHILD, cur_node) ;
        mlview_node_type_picker_select_node_name_or_content_entry_text 
                (picker) ;
        button = gtk_dialog_run (GTK_DIALOG (picker)) ;
        switch (button) {
        case GTK_RESPONSE_ACCEPT:
                handle_nt_picker_ok_button_clicked_to_add_child 
                        (a_this) ;
                break ;
        default:
                break ;
        }
        gtk_widget_hide 
                (GTK_WIDGET (PRIVATE (a_this)->node_type_picker)) ;
}

/**
 *Asks the user for the type of nodes she wants
 *to insert an inserts it.
 *@param a_this the current instance of #MlViewTreeEditor2
 */
void
mlview_tree_editor2_insert_prev_sibling_node_interactive (MlViewTreeEditor2 *a_this)
{
        MlViewNodeTypePicker *picker=NULL ;
        gint button=0 ;
        xmlNode *cur_node=NULL ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)
                          && PRIVATE (a_this)
                          && PRIVATE (a_this)->cur_sel_start) ;
        
        picker = mlview_tree_editor2_get_node_type_picker (a_this) ;
        g_return_if_fail (picker) ;
        mlview_node_type_picker_set_title
                (picker, _("insert a previous sibling node")) ;
        mlview_node_type_picker_select_node_name_or_content_entry_text
                (picker) ;
        /*
         *insert a flag to indicate that the picker is used
         *to insert node as previous node.
         *this is used by function
         *mlview_tree_editor_insert_prev_sibling_node_interactive().
         */
        g_object_set_data (G_OBJECT (a_this), "prev",
                           GINT_TO_POINTER (TRUE)) ;
        cur_node = mlview_tree_editor2_get_xml_node2 
                (a_this, PRIVATE (a_this)->cur_sel_start) ;
        g_return_if_fail (cur_node) ;
        mlview_node_type_picker_build_element_name_choice_list 
                (picker, INSERT_BEFORE, cur_node) ;

        button = gtk_dialog_run (GTK_DIALOG (picker)) ;
        switch (button) {
        case GTK_RESPONSE_ACCEPT:/*OK button*/
                handle_nt_picker_ok_button_clicked_to_insert_sibling_node
                        (a_this) ;                
                break ;
        default:
                break ;
        }
        gtk_widget_hide 
                (GTK_WIDGET (PRIVATE (a_this)->node_type_picker)) ;
}

/**
 *Asks the user for the type of node she wants
 *to insert and inserts it as a next sibling of the
 *currently selected node.
 *@param a_this the current instance of #MlViewTreeEditor.
 */
void
mlview_tree_editor2_insert_next_sibling_node_interactive (MlViewTreeEditor2 *a_this)
{
        MlViewNodeTypePicker *picker=NULL ;
        gint button=0 ;
        xmlNode *cur_node=NULL ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)
                          && PRIVATE (a_this)
                          && PRIVATE (a_this)->cur_sel_start) ;
        
        picker = mlview_tree_editor2_get_node_type_picker (a_this) ;
        g_return_if_fail (picker) ;
        mlview_node_type_picker_set_title 
                (picker, _("insert a next sibling node")) ;
        mlview_node_type_picker_select_node_name_or_content_entry_text
                (picker) ;
        /*
         *insert a flag to indicate that the picker is used
         *to insert node as next sibling node.
         *this is used by function
         *mlview_tree_editor_insert_next_sibling_node_interactive().
         */
        g_object_set_data (G_OBJECT (a_this), "prev",
                           GINT_TO_POINTER (FALSE)) ;
        cur_node = mlview_tree_editor2_get_xml_node2 
                (a_this, PRIVATE (a_this)->cur_sel_start) ;
        g_return_if_fail (cur_node) ;
        mlview_node_type_picker_build_element_name_choice_list 
                (picker, INSERT_BEFORE, cur_node) ;

        button = gtk_dialog_run (GTK_DIALOG (picker)) ;
        switch (button) {
        case GTK_RESPONSE_ACCEPT:/*OK button*/
                handle_nt_picker_ok_button_clicked_to_insert_sibling_node
                        (a_this) ;                
                break ;
        default:
                break ;
        }
        gtk_widget_hide 
                (GTK_WIDGET (PRIVATE (a_this)->node_type_picker)) ;
}

/**
 *Inserts a sibling node to the xml document.
 *@param a_this the current instance of #MlViewTreeEditor
 *@param a_ref_iter an iterator to the reference row of the visual
 *tree.
 *@param a_node the node to add before/previous the node
 *pointed to by a_ref_iter.
 *@param a_previous if set to TRUE, a_node is inserted before
 *the node pointed to by a_ref_iter, otherwise a_node is inserted
 *after the node pointed to by a_ref_iter.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_insert_sibling_node (MlViewTreeEditor2 *a_this,
                                         GtkTreeIter * a_ref_iter,
                                         xmlNode * a_node,
                                         gboolean a_previous)
{
        GtkTreeRowReference *row_ref = NULL;
        xmlNode *ref_node = NULL,
                *tmp_node = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->nodes_rows_hash 
                              && a_node
                              && a_ref_iter,
                              MLVIEW_BAD_PARAM_ERROR);

        /*make sure a_node hasn't be drawn already */
        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash, a_node);
        g_return_val_if_fail (row_ref == NULL,
                              MLVIEW_BAD_PARAM_ERROR);

        ref_node = mlview_tree_editor2_get_xml_node (a_this,
                                                     a_ref_iter);
        g_return_val_if_fail (ref_node, MLVIEW_BAD_PARAM_ERROR);
        /*
         *call mlview_xml_document_insert_x_sibling_node()
         *to actualy insert the node.
         */
        if (a_previous == TRUE) {
                tmp_node =
                        mlview_xml_document_insert_prev_sibling_node
                        (PRIVATE (a_this)->mlview_xml_doc,
                         ref_node, a_node, TRUE, TRUE);
        } else {
                tmp_node =
                        mlview_xml_document_insert_next_sibling_node
                        (PRIVATE (a_this)->mlview_xml_doc,
                         ref_node, a_node, TRUE, TRUE);
        }
        g_return_val_if_fail (tmp_node == a_node, MLVIEW_ERROR);
        return MLVIEW_OK;
}

/**
 *Cuts the xml node associated to a given tree iterator.
 *Basically, this just cut the underlying xml node.
 *@param a_this the current instance of #MlViewTreeEditor.
 *@param a_iter an iterator to the node to cut.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_cut_node (MlViewTreeEditor2 * a_this,
                              GtkTreeIter * a_iter)
{
        xmlNode *node = NULL,
                *tmp_node = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && a_iter, MLVIEW_BAD_PARAM_ERROR);

        node = mlview_tree_editor2_get_xml_node (a_this, a_iter);
        g_return_val_if_fail (node, MLVIEW_ERROR);
        tmp_node = mlview_xml_document_cut_node
                (PRIVATE (a_this)->mlview_xml_doc, node, TRUE);
        g_return_val_if_fail (tmp_node == node, MLVIEW_ERROR);
        return MLVIEW_OK;
}

/**
 *Cuts the xml node associated to a given tree iterator.
 *Basically, this just cut the underlying xml node.
 *@param a_this the current instance of #MlViewTreeEditor.
 *@param a_pat a path to the node to cut.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_cut_node2 (MlViewTreeEditor2 * a_this,
                              GtkTreePath * a_path)
{
        GtkTreeIter iter = {0} ;
        GtkTreeModel *model = NULL ;
        gboolean is_ok = TRUE ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;
        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model, MLVIEW_ERROR) ;
        is_ok = gtk_tree_model_get_iter (model, &iter, a_path) ;
        g_return_val_if_fail (is_ok == TRUE, MLVIEW_ERROR) ;
        return mlview_tree_editor2_cut_node (a_this, &iter) ;        
}

/**
 *Copy the node pointed by an iterator to the clipboard.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_iter an iterator to the node to copy.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_copy_node (MlViewTreeEditor2 * a_this,
                               GtkTreeIter * a_iter)
{
        xmlNode *xml_node = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR);

        xml_node = mlview_tree_editor2_get_xml_node (a_this,
                                                     a_iter);
        g_return_val_if_fail (xml_node, MLVIEW_ERROR);
        mlview_xml_document_copy_node_to_clipboard
                (xml_node, PRIVATE (a_this)->xml_doc);
        return MLVIEW_OK;
}

/**
 *Copy the node pointed by a_path to the clipboard.
 *@param a_this the current instance of #MlViewTreeEditor.
 *@param a_path the path to the xml node to copy to the clipboard.
 *@return MLVIEW_OK upon successful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_copy_node2 (MlViewTreeEditor2 *a_this,
                                GtkTreePath *a_path)
{
        GtkTreeModel *model = NULL ;
        GtkTreeIter iter = {0} ;
        gboolean is_ok = TRUE ;
        
        g_return_val_if_fail (a_this 
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)
                          && a_path,
                          MLVIEW_BAD_PARAM_ERROR) ;

        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model, MLVIEW_ERROR) ;
        is_ok = gtk_tree_model_get_iter (model, &iter, a_path) ;
        g_return_val_if_fail (is_ok == TRUE, MLVIEW_ERROR) ;
        return mlview_tree_editor2_copy_node (a_this, &iter) ;
}

/**
 *Gets the last node put into the clipboard and inserts it into
 *the visual tree.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_parent_iter an iterator to the parent node.
 *@return MLVIEW_OK upon successful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_paste_node_as_child (MlViewTreeEditor2 *a_this,
                                         GtkTreeIter *a_parent_iter)
{
        xmlNode *parent_node=NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->mlview_xml_doc
                              && a_parent_iter,
                              MLVIEW_BAD_PARAM_ERROR) ;
        
        parent_node = mlview_tree_editor2_get_xml_node (a_this,
                                                        a_parent_iter) ;
        g_return_val_if_fail (parent_node, MLVIEW_NODE_NOT_FOUND_ERROR) ;
        mlview_xml_document_paste_node_as_child 
                (PRIVATE (a_this)->mlview_xml_doc, parent_node, TRUE) ;
        return MLVIEW_OK ;
}

/**
 *Gets the last node put in the clipboard and pastes
 *it as a sibling of the node pointed to by a_ref_iter.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_ref_iter an iterator that points to the
 *reference node.
 *@param a_previous if set to TRUE, the node must be pasted before
 *the reference node otherwise it is to be pasted after the reference 
 *node.
 *@return MLVIEW_OK upon sucessful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_paste_node_as_sibling (MlViewTreeEditor2 *a_this,
                                           GtkTreeIter *a_ref_iter,
                                           gboolean a_previous)
{
        xmlNode *sibling_node=NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->mlview_xml_doc
                              && a_ref_iter,
                              MLVIEW_BAD_PARAM_ERROR) ;

        sibling_node = mlview_tree_editor2_get_xml_node 
                (a_this, a_ref_iter) ;
        g_return_val_if_fail (sibling_node
                              && sibling_node->parent, 
                              MLVIEW_ERROR) ;
        mlview_xml_document_paste_node_as_sibling 
                (PRIVATE (a_this)->mlview_xml_doc,
                 sibling_node->parent, sibling_node, 
                 a_previous, TRUE) ;
        return MLVIEW_OK ;
}

/**
 *Gets the last node put in the clipboard and pastes
 *it as a sibling of the node pointed to by a_ref_iter.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_ref_path the path that points to the reference node.
 *@param a_previous if set to TRUE, the node must be pasted before
 *the reference node otherwise it is to be pasted after the reference 
 *node.
 *@return MLVIEW_OK upon sucessful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_paste_node_as_sibling2 (MlViewTreeEditor2 *a_this,
                                            GtkTreePath *a_ref_path,
                                            gboolean a_previous)
{
        GtkTreeModel *model = NULL ;
        GtkTreeIter iter = {0} ;
        gboolean is_ok = TRUE ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_ref_path,
                              MLVIEW_BAD_PARAM_ERROR) ;
        
        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model, MLVIEW_ERROR) ;
        is_ok = gtk_tree_model_get_iter (model, &iter, a_ref_path) ;
        g_return_val_if_fail (is_ok == TRUE, MLVIEW_ERROR) ;
        return mlview_tree_editor2_paste_node_as_sibling 
                (a_this, &iter, a_previous) ;        
}

/**
 *Visualy Updates the tree editor
 *to make the addition of child node visible.
 *If the node addition has been already "updated",
 *this method does nothing.
 *@param a_this the current instance of #MlViewEditor.
 *@param a_parent the parent node of the newly added
 *instance of xmlNode.
 *@param a_node the newly added instance of xmlNode.
 *@return MLVIEW_OK upon successful completion, an error
 *code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_update_child_node_added (MlViewTreeEditor2 * a_this,
                                             xmlNode * a_parent,
                                             xmlNode * a_node, 
                                             gboolean a_emit_signals) 
{
        GtkTreeView *tree_view = NULL ;
        GtkTreeRowReference *parent_row_ref = NULL,
                *row_ref = NULL;
        GtkTreeIter iter = { 0 };
        GtkTreeModel *model = NULL;
        GtkTreePath *tree_path = NULL;
        gboolean is_ok = FALSE;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_val_if_fail
                (a_this && MLVIEW_IS_TREE_EDITOR2 (a_this)
                 && PRIVATE (a_this), MLVIEW_BAD_PARAM_ERROR);
        /*
         *make sure the a_node hasn't been added to
         *the visual tree yet. row_ref must be NULL.
         */
        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash, a_node);
        if (row_ref) {
                mlview_tree_editor2_select_node (a_this, a_node,
                                                 TRUE, TRUE) ;
                return MLVIEW_OK ;
        }
        /*only an element node should have children */
        g_return_val_if_fail
                (a_parent->type == XML_ELEMENT_NODE,
                 MLVIEW_BAD_PARAM_ERROR);
        tree_view = mlview_tree_editor2_get_tree_view (a_this);
        g_return_val_if_fail (tree_view != NULL, MLVIEW_ERROR);
        model = gtk_tree_view_get_model (tree_view);
        g_return_val_if_fail (model, MLVIEW_ERROR);
        /*
         *get the visual node that matches the a_parent_node
         *It must have been drawn already.
         */
        parent_row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash, a_parent);
        g_return_val_if_fail (parent_row_ref,
                              MLVIEW_NODE_NOT_FOUND_ERROR);
        /*get an iterator on the parent node */
        tree_path = gtk_tree_row_reference_get_path
                (parent_row_ref);
        g_return_val_if_fail (tree_path, MLVIEW_ERROR);
        is_ok = gtk_tree_model_get_iter (model, &iter,
                                         tree_path);
        if (is_ok != TRUE) {
                mlview_utils_trace_info ("is_ok == TRUE failed") ;
                status = MLVIEW_OK ;
                goto cleanup ;
        }
        /*
         *build a GtkTreeModel subtree that matches the a_node
         *subtree.
         */
        status = mlview_tree_editor2_build_tree_model_from_xml_tree
                (a_this, a_node,
                 &iter, INSERT_TYPE_ADD_CHILD, &model);
        /*
         *Build the visual representation of the newly added
         *row.
         */
        status = mlview_tree_editor2_update_visual_node
                (a_this, &iter);
        /*
         *expand the current selected node to it leaves
         *and select the newly added node.
         */
        mlview_utils_gtk_tree_view_expand_row_to_depth 
                (tree_view, tree_path, -1) ;

        mlview_tree_editor2_select_node (a_this, a_node, FALSE,
                                         TRUE) ;

        if (status != MLVIEW_OK) {
                mlview_utils_trace_info 
                        ("status ==  MVIEW_OK failed") ;
                goto cleanup ;                
        }
        
        /*emit the appropriate signals */
        if (a_emit_signals == TRUE) {
                row_ref = g_hash_table_lookup
                        (PRIVATE (a_this)->nodes_rows_hash,
                         a_node);
                if (!row_ref) {
                        mlview_utils_trace_info ("row_ref failed") ;
                        status = MLVIEW_ERROR ;
                        goto cleanup ;
                }
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_ADDED], 0,
                               row_ref);
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[TREE_CHANGED], 0);
        }
 cleanup:
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
        return status;
}

/**
 *Updates the graphical tree to reflect a
 *"node pasted" event.
 *@param a_this the current instance of #MlViewTreeEditor
 *@param a_parent_node the parent node of the pasted node.
 *@param a_node the pasted node.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_update_node_pasted (MlViewTreeEditor2 *a_this,
                                        xmlNode * a_parent_node,
                                        xmlNode * a_node,
                                        gboolean a_emit_signals)
{
        GtkTreeRowReference *row_ref = NULL;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_parent_node
                              && a_node, MLVIEW_BAD_PARAM_ERROR);
        /*make sure a_parent_node is drawn */
        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash,
                 a_parent_node);
        g_return_val_if_fail (row_ref, MLVIEW_BAD_PARAM_ERROR);
        /*make sure a_node is not drawn */
        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash, a_node);
        if (row_ref) {
                mlview_tree_editor2_select_node (a_this, a_node,
                                                 TRUE, TRUE) ;
                return MLVIEW_OK ;
        }
        g_return_val_if_fail (row_ref == NULL,
                              MLVIEW_BAD_PARAM_ERROR);
        status = mlview_tree_editor2_update_child_node_added
                (a_this, a_parent_node, a_node, FALSE);
        g_return_val_if_fail (status == MLVIEW_OK, status);
        if (a_emit_signals == TRUE) {
                row_ref = g_hash_table_lookup
                        (PRIVATE (a_this)->nodes_rows_hash,
                         a_node);
                g_return_val_if_fail (row_ref, MLVIEW_ERROR);
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_PASTED], 0,
                               row_ref);
        }
        return MLVIEW_OK;
}

/**
 *updates the visual tree node to reflect the
 *"sibling node inserted event."
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_ref_node the reference node.
 *@param a_inserted the node that has been inserted after
 *or before ref_node.
 *@param a_previous if set to TRUE, a_inserted_node
 *is to be inserted before a_ref_node, otherwise a_inserted_node
 *is to be inserted after a_ref_node.
 *@return MLVIEW_OK upon sucessfull completion, an error
 *code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_update_sibling_node_inserted (MlViewTreeEditor2 * a_this, 
                                                  xmlNode * a_ref_node, 
                                                  xmlNode * a_inserted_node, 
                                                  gboolean a_previous,
                                                  gboolean a_emit_signals)
{
        GtkTreeRowReference *row_ref = NULL;
        GtkTreeModel *model = NULL;
        GtkTreeIter iter = { 0 };
        GtkTreeView *tree_view = NULL ;
        enum MlViewStatus status = MLVIEW_OK;
        struct MlViewAppSettings *settings = NULL ;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->nodes_rows_hash 
                              && a_ref_node
                              && a_inserted_node,
                              MLVIEW_BAD_PARAM_ERROR);

        /*make sure ref_node has been drawn already */
        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash, a_ref_node);
        g_return_val_if_fail (row_ref, MLVIEW_BAD_PARAM_ERROR);
        /*make sure a_inserted_node hasn't been drawn already */
        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash,
                 a_inserted_node);
        if (row_ref) {
                mlview_tree_editor2_select_node (a_this, a_inserted_node,
                                                 TRUE, TRUE) ;
                return MLVIEW_OK ;
        }
        status = mlview_tree_editor2_get_iter
                (a_this, a_ref_node, &iter);
        g_return_val_if_fail (status == MLVIEW_OK, status);
        model = mlview_tree_editor2_get_model (a_this);
        g_return_val_if_fail (model, MLVIEW_ERROR);
        if (a_previous == TRUE) {
                status = mlview_tree_editor2_build_tree_model_from_xml_tree
                        (a_this, a_inserted_node,
                         &iter, INSERT_TYPE_INSERT_BEFORE,
                         &model);                
        } else {
                status = mlview_tree_editor2_build_tree_model_from_xml_tree
                        (a_this, a_inserted_node,
                         &iter, INSERT_TYPE_INSERT_AFTER,
                         &model);
        }
        g_return_val_if_fail (status == MLVIEW_OK, status);
        /*
         *expand the current node and select a_inserted_node
         */
        tree_view = mlview_tree_editor2_get_tree_view (a_this) ;
        mlview_tree_editor2_get_iter (a_this, a_inserted_node, 
                                      &iter) ;

        settings = mlview_app_context_get_settings 
                (PRIVATE (a_this)->app_context) ;
        mlview_utils_gtk_tree_view_expand_row_to_depth2
                (tree_view, &iter, 
                 settings->tree_editors.default_doc_expansion_depth);

        mlview_tree_editor2_select_node (a_this, a_inserted_node,
                                         FALSE, TRUE) ;
        status = mlview_tree_editor2_update_visual_node
                (a_this, &iter);
        if (status == MLVIEW_OK && a_emit_signals == TRUE) {
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_ADDED], 0,
                               row_ref);
        }
        return status;
}

/**
 *Updates the visual tree so that it reflects
 *a node cut event.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_parent_node the parent node of the xml node that
 *has been cut.
 *@param a_node_cut the xml node that has been cut.
 *@return MLVIEW_OK upon successful completion, an error
 *code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_update_node_cut (MlViewTreeEditor2 * a_this,
                                     xmlNode * a_parent_node,
                                     xmlNode * a_node_cut)
{
        GtkTreeIter iter = { 0 };
        GtkTreeModel *model = NULL;
        GtkTreeRowReference *row_ref=NULL ;
        gboolean is_ok = TRUE;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_node_cut
                              && a_parent_node,
                              MLVIEW_BAD_PARAM_ERROR);
        g_return_val_if_fail (a_node_cut->parent == NULL
                              && a_parent_node,
                              MLVIEW_BAD_PARAM_ERROR);

        g_return_val_if_fail (a_node_cut, MLVIEW_ERROR);
        /*
         *make sure the a_parent_node and a_cut_node are still
         *referenced.
         */
        status = mlview_tree_editor2_get_iter
                (a_this, a_parent_node, &iter);
        if (status != MLVIEW_OK)
                return status;
        row_ref = g_hash_table_lookup 
                (PRIVATE (a_this)->nodes_rows_hash, a_node_cut);
        g_return_val_if_fail (row_ref, MLVIEW_ERROR) ;        
        status = mlview_tree_editor2_get_iter
                (a_this, a_node_cut, &iter);
        if (status != MLVIEW_OK)
                return status;
        model = mlview_tree_editor2_get_model (a_this);
        g_return_val_if_fail (model, MLVIEW_ERROR);
        is_ok = gtk_tree_store_remove
                (GTK_TREE_STORE (model), &iter);
        g_hash_table_remove
                (PRIVATE (a_this)->nodes_rows_hash, a_node_cut);
        gtk_tree_row_reference_free (row_ref) ;
        row_ref = NULL ;
        g_signal_emit (G_OBJECT (a_this),
                       gv_signals[NODE_CUT], 0, a_node_cut);
        g_signal_emit (G_OBJECT (a_this),
                       gv_signals[TREE_CHANGED], 0);
        return MLVIEW_OK;
}

/**
 *Update the tag string of a given visual node.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_iter an iterator that points to the node to be updated.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
static enum MlViewStatus
update_visual_node (MlViewTreeEditor2 *a_this, GtkTreeIter *a_iter)
{
        guchar *start_tag_str = NULL;
        GtkTreeModel *model = NULL;
        xmlNode *xml_node = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && a_iter, MLVIEW_BAD_PARAM_ERROR);
        
        model = mlview_tree_editor2_get_model (a_this);
        g_return_val_if_fail (model, MLVIEW_ERROR);
        gtk_tree_model_get (model, a_iter, XML_NODE_COLUMN,
                            &xml_node, -1);
        start_tag_str = node_to_string_tag 
                (PRIVATE (a_this)->app_context, xml_node);
        if (!start_tag_str) {
                return MLVIEW_OK;
        }
        gtk_tree_store_set (GTK_TREE_STORE (model),
                            a_iter, START_TAG_COLUMN,
                            start_tag_str, -1);
        if (start_tag_str) {
                g_free (start_tag_str) ;
                start_tag_str = NULL ;
        }

        return MLVIEW_OK;
}

/**
 *Updates the representation of a visual node denoted
 *by an instance of GtkTreeIter (a visual node iterator).
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_iter the iterator to consider.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_update_visual_node (MlViewTreeEditor2 *a_this,
                                        GtkTreeIter * a_iter)
{
        return MLVIEW_TREE_EDITOR2_CLASS 
                (G_OBJECT_GET_CLASS 
                 (a_this))->update_visual_node (a_this, a_iter);
}

/**
 *Update the tag string of a given visual node.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_iter an iterator that points to the node to be updated.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_update_visual_node2 (MlViewTreeEditor2 *a_this,
                                         xmlNode * a_node)
{
        GtkTreeRowReference *row_ref = NULL;
        GtkTreePath *tree_path = NULL;
        GtkTreeIter iter = {0};
        GtkTreeModel *model = NULL;
        gboolean is_ok = FALSE;
        enum MlViewStatus status = MLVIEW_ERROR;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_node, MLVIEW_BAD_PARAM_ERROR);

        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash, a_node);
        if (!row_ref) {
                return MLVIEW_NODE_NOT_FOUND_ERROR;
        }
        tree_path = gtk_tree_row_reference_get_path (row_ref);
        g_return_val_if_fail (tree_path, MLVIEW_ERROR);
        model = mlview_tree_editor2_get_model (a_this);
        if (!model) {
                mlview_utils_trace_info ("model failed") ;
                status = MLVIEW_ERROR ;
                goto cleanup ;
        }
        is_ok = gtk_tree_model_get_iter (model, &iter,
                                         tree_path);
        if (is_ok != TRUE) {
                mlview_utils_trace_info ("is_ok == TRUE failed") ;
                status = MLVIEW_ERROR ;
                goto cleanup ;
        }
        status = mlview_tree_editor2_update_visual_node
                (a_this, &iter);
 cleanup:
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
        return status;
}

/**
 *Displays a dialog box to let the user
 *type in search info and triggers the
 *search.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_search_interactive (MlViewTreeEditor2 *a_this)
{
        GtkWidget *find_dialog = NULL ;
        gint button = 0, loop = 1 ;

        struct SearchConfig search_config = {0} ;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_this, MLVIEW_BAD_PARAM_ERROR) ;

        find_dialog = get_search_dialog (a_this) ;
        g_return_val_if_fail (find_dialog
                              && GTK_IS_DIALOG (find_dialog),
                              MLVIEW_ERROR) ;
        while (loop) {
                button = gtk_dialog_run (GTK_DIALOG (find_dialog)) ;
                switch (button) {
                case 0:/*next button*/
                case 1:/*prev button*/
                        status = get_search_config
                                (GTK_DIALOG (find_dialog),
                                 &search_config) ;
                        g_return_val_if_fail (status == MLVIEW_OK,
                                              status) ;
                        if (button == 0) {
                                search_config.downward = TRUE ;
                        } else {
                                search_config.downward = FALSE ;
                        }
                        mlview_tree_editor2_search
                                (a_this, 
                                 PRIVATE (a_this)->cur_sel_start,
                                 &search_config) ;
                        break ;
                        
                case 2:/*cancel button*/
                default:
                        loop = 0 ;/*get out of the loop*/
                        break ;
                }
        }
        gtk_widget_hide (GTK_WIDGET (find_dialog)) ;
        return MLVIEW_OK ;
}

/**
 *Calls the search backend on the current instance
 *of #MlViewXMLDocument.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_from the row to start the search from.
 *@param a_config the search configuration .
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_search (MlViewTreeEditor2 *a_this,
                            GtkTreeRowReference *a_from,
                            struct SearchConfig *a_config)
{
        xmlNode *xml_node = NULL, *node_found = NULL ;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->mlview_xml_doc,
                              MLVIEW_OK) ;

        if (PRIVATE (a_this)->cur_sel_start) {
                xml_node = mlview_tree_editor2_get_xml_node2 
                        (a_this, a_from) ;
                g_return_val_if_fail (xml_node, MLVIEW_ERROR) ;
        }
        status = mlview_xml_document_search
                (PRIVATE (a_this)->mlview_xml_doc, a_config,
                 xml_node, &node_found, TRUE) ;
        return status ;
}

/**
 *Selects the visual node that matches the xml node.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_node the node to select.
 */
void
mlview_tree_editor2_select_node (MlViewTreeEditor2 *a_this,
                                 xmlNode *a_node,
                                 gboolean a_issued_by_model,
                                 gboolean a_signal_model)
{
        enum MlViewStatus status = MLVIEW_OK ;
        GtkTreePath *tree_path = NULL, *parent_path = NULL ;
        GtkTreeIter iter = {0} ;
        GtkTreeModel * model = NULL ;
        GtkTreeView *tree_view = NULL ;
        GtkTreeSelection *tree_sel = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_EDITOR2 (a_this)
                          && PRIVATE (a_this)) ;

        status = mlview_tree_editor2_get_iter (a_this, a_node,
                                               &iter) ;
        if (status != MLVIEW_OK) {
                return ;
        }
        model = mlview_tree_editor2_get_model (a_this) ;
        if (!model) {
                mlview_utils_trace_info ("model failed") ;
                goto cleanup ;
        }
        tree_view = mlview_tree_editor2_get_tree_view (a_this) ;
        if (!tree_view) {
                mlview_utils_trace_info ("tree_view failed") ;
                goto cleanup ;
        }
        /*expand the tree up to the current node level*/
        tree_path = gtk_tree_model_get_path (model, &iter) ;
        parent_path = gtk_tree_path_copy (tree_path);
        if (!tree_path) {
                mlview_utils_trace_info ("tree_path failed") ;
                goto cleanup ;
        }
        gtk_tree_path_up (parent_path);
        tree_sel = gtk_tree_view_get_selection (tree_view) ;
        if (!tree_sel) {
                mlview_utils_trace_info ("tree_sel failed") ;
                goto cleanup ;
        }
        gtk_tree_view_expand_to_path 
                (tree_view, parent_path) ;

        /*now, select the current node*/
        if (a_signal_model == TRUE
            && a_issued_by_model == FALSE) {
                /*
                 *The signal has been generated
                 *by the current view ->
                 *tell everyone that node has been emited
                 */
                mlview_xml_document_select_node
                        (PRIVATE (a_this)->mlview_xml_doc, a_node) ;
        } else if (a_issued_by_model == TRUE){
                /*
                 *the signal has been generated by
                 *another view and we get notified
                 *via the model.
                 */
                if (PRIVATE
                    (a_this)->select_issued_by_model == TRUE) {
                        /*we are being called twice in
                        *the same selection process.
                        *stop the process here.
                        */
                        PRIVATE (a_this)->select_issued_by_model
                                = FALSE ;
                } else {
                        /*
                         *first time we are being called
                         *as the result of the model issuing
                         *the "node-selected" signal.
                         */
                        PRIVATE (a_this)->select_issued_by_model 
                                = TRUE ;
                        gtk_tree_selection_select_iter (tree_sel,
                                                        &iter) ;
                        PRIVATE (a_this)->select_issued_by_model 
                                = FALSE ;
                }
        }

 cleanup:
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
        if (parent_path) {
                gtk_tree_path_free (parent_path) ;
                parent_path = NULL ;
        }
}

/**
 *Expands the current selected visual row to the depth
 *a_depth. If a_depth is set to -1 the tree is expanded to the
 *leaves.
 *@param a_this the current instance of #MlViewTreeEditor.
 *@param a_depth the expansion depth.
 */
void
mlview_tree_editor2_expand_tree_to_depth (MlViewTreeEditor2 * a_this,
                                         gint a_depth)
{
        GtkTreeRowReference *cur_row_ref=NULL ;
        GtkTreePath *cur_path=NULL ;
        GtkTreeView *tree_view=NULL ;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)) ;
        cur_row_ref = PRIVATE (a_this)->cur_sel_start ;
        g_return_if_fail (cur_row_ref) ;
        cur_path = gtk_tree_row_reference_get_path (cur_row_ref) ;
        g_return_if_fail (cur_path) ;
        tree_view = mlview_tree_editor2_get_tree_view (a_this) ;
        if (!tree_view) {
                mlview_utils_trace_info ("tree_view failed") ;
                goto cleanup ;
        }
        status = mlview_utils_gtk_tree_view_expand_row_to_depth 
                        (tree_view, cur_path, a_depth) ;
        if (status != MLVIEW_OK) {
                mlview_utils_trace_info 
                        ("status == MLVIEW_OK failed.") ;
        }
 cleanup:
        if (cur_path) {
                gtk_tree_path_free (cur_path) ;
                  cur_path = NULL ;
        }
}

/**
 *Connects to the relevant signals emited by the document object
 *model and register the relevant callbacks
 *@param a_this the current instance of #MlViewTreeEditor
 *@param a_doc the document object model to connect to.
 *@return MLVIEW_OK upon successful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_connect_to_doc (MlViewTreeEditor2 *a_this,
                                    MlViewXMLDocument *a_doc)
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && MLVIEW_IS_XML_DOCUMENT (a_doc),
                              MLVIEW_BAD_PARAM_ERROR) ;

        
        g_signal_connect (G_OBJECT (a_doc),
                          "node-selected",
                          G_CALLBACK (xml_doc_selected_node_cb),
                          a_this) ;

        g_signal_connect (G_OBJECT (a_doc),
                          "node-cut",
                          G_CALLBACK (xml_doc_node_cut_cb),
                          a_this) ;

        g_signal_connect (G_OBJECT (a_doc),
                          "prev-sibling-node-inserted",
                          G_CALLBACK (xml_doc_prev_sibling_node_inserted_cb),
                          a_this) ;

        g_signal_connect (G_OBJECT (a_doc),
                          "next-sibling-node-inserted",
                          G_CALLBACK (xml_doc_next_sibling_node_inserted_cb),
                          a_this) ;
        
        g_signal_connect (G_OBJECT (a_doc),
                          "child-node-added",
                          G_CALLBACK (xml_doc_child_node_added_cb),
                          a_this) ;
        
        g_signal_connect (G_OBJECT (a_doc),
                          "content-changed",
                          G_CALLBACK (xml_doc_content_changed_cb),
                          a_this) ;

        g_signal_connect (G_OBJECT (a_doc),
                          "name-changed",
                          G_CALLBACK (xml_doc_name_changed_cb),
                          a_this) ;

        g_signal_connect (G_OBJECT (a_doc),
                          "node-attribute-name-changed",
                          G_CALLBACK (xml_doc_node_attribute_name_changed_cb),
                          a_this) ;

        g_signal_connect (G_OBJECT (a_doc),
                          "node-attribute-value-changed",
                          G_CALLBACK (xml_doc_node_attribute_value_changed_cb),
                          a_this) ;

        g_signal_connect (G_OBJECT (a_doc),
                          "node-attribute-removed",
                          G_CALLBACK (xml_doc_node_attribute_removed_cb),
                          a_this) ;

        g_signal_connect (G_OBJECT (a_doc),
                          "node-namespace-added",
                          G_CALLBACK (xml_doc_node_namespace_added_cb),
                          a_this) ;

        g_signal_connect (G_OBJECT (a_doc),
                          "node-namespace-changed",
                          G_CALLBACK (xml_doc_node_namespace_changed_cb),
                          a_this) ;

        g_signal_connect (G_OBJECT (a_doc),
                          "node-namespace-removed",
                          G_CALLBACK (xml_doc_node_namespace_removed_cb),
                          a_this) ;

        g_signal_connect (G_OBJECT (a_doc),
                          "searched-node-found",
                          G_CALLBACK (xml_doc_searched_node_found_cb),
                          a_this) ;

        return MLVIEW_OK ;
}

/**
 *Disconnects from the signal emited by the document object model
 *that is,  unregister all the callback registered by
 *mlview_tree_editor2_connect_to_doc() .
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_doc the document object model to disconnect from.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_disconnect_from_doc (MlViewTreeEditor2 *a_this,
                                         MlViewXMLDocument *a_doc)
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && MLVIEW_IS_XML_DOCUMENT (a_doc),
                              MLVIEW_BAD_PARAM_ERROR) ;

        g_signal_handlers_disconnect_by_func 
                (G_OBJECT (a_doc),
                 G_CALLBACK (xml_doc_node_cut_cb),
                 a_this) ;
        g_signal_handlers_disconnect_by_func 
                (G_OBJECT (a_doc),
                 G_CALLBACK (xml_doc_prev_sibling_node_inserted_cb),
                 a_this) ;
        g_signal_handlers_disconnect_by_func
                (G_OBJECT (a_doc),
                 G_CALLBACK (xml_doc_next_sibling_node_inserted_cb),
                 a_this) ;
        g_signal_handlers_disconnect_by_func
                (G_OBJECT (a_doc),
                 G_CALLBACK (xml_doc_child_node_added_cb),
                 a_this) ;
        g_signal_handlers_disconnect_by_func
                (G_OBJECT (a_doc),
                 G_CALLBACK (xml_doc_name_changed_cb),
                 a_this) ;
        g_signal_handlers_disconnect_by_func
                (G_OBJECT (a_doc),
                 G_CALLBACK (xml_doc_node_attribute_name_changed_cb),
                 a_this) ;
        g_signal_handlers_disconnect_by_func
                (G_OBJECT (a_doc),
                 G_CALLBACK (xml_doc_node_attribute_value_changed_cb),
                 a_this) ;
        g_signal_handlers_disconnect_by_func
                (G_OBJECT (a_doc),
                 G_CALLBACK (xml_doc_node_attribute_removed_cb),
                 a_this) ;
        g_signal_handlers_disconnect_by_func
                (G_OBJECT (a_doc),
                 G_CALLBACK (xml_doc_node_namespace_changed_cb),
                 a_this) ;
        g_signal_handlers_disconnect_by_func
                (G_OBJECT (a_doc),
                 G_CALLBACK (xml_doc_node_namespace_removed_cb),
                 a_this) ;
        g_signal_handlers_disconnect_by_func
                (G_OBJECT (a_doc),
                 G_CALLBACK (xml_doc_searched_node_found_cb),
                 a_this) ;
        g_signal_handlers_disconnect_by_func
                (G_OBJECT (a_doc),
                 G_CALLBACK (xml_doc_selected_node_cb),
                 a_this) ;
        return MLVIEW_OK ;
}

#endif
