/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 8-*- */

/*
 *This file is part of MlView
 *
 *MlView is free software; you can redistribute it and/or 
 *modify it under the terms of 
 *the GNU General Public License as published 
 *by the Free Software Foundation; either version 2
 *or (at your option) any later version.
 *
 *MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView.
 *see the file COPYING. 
 *If not, write to the 
 *Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include <stdio.h>
#include <string.h>
#include <libxslt/xslt.h>
#include <libxslt/xsltInternals.h>
#include <libxslt/transform.h>
#include "mlview-editor.h"
#include "mlview-utils.h"
#include "mlview-xml-document.h"
#include "mlview-xslt-utils.h"

/**
 *@file 
 *The mlview xslt transformer
 * 
 *This is a wrapper of the libxslt from Daniel Veillard.
 *It provides an interface to apply a xslt stylesheet to 
 *xml document opened in mlview. 
 *
 */

#define PRIVATE(object) (object)->priv

/*======================================================
 *private helper functions
 *=====================================================*/

/**
 * Checks whether a mlview document is an xslt stylesheet
 * The test is performed on the namespaces declared on
 * the root node of the document 
 *@param mlv_xml_doc : the document to test
 *@return TRUE if the document is a xslt stylesheet,
 *FALSE otherwise.
 */ 
gboolean
mlview_xslt_utils_is_xslt_doc(MlViewXMLDocument* mlv_xml_doc)
{
        xmlDocPtr xml_doc;
        xmlNodePtr root_node;

        xmlNs *cur_ns = NULL ;
        gint found=FALSE;
        
        xml_doc = mlview_xml_document_get_xml_document(mlv_xml_doc);
        root_node = xmlDocGetRootElement (xml_doc);
        for (cur_ns = root_node->nsDef ;
             !found && cur_ns; 
             cur_ns = cur_ns->next) {
                found = (gboolean)(!xmlStrcmp(cur_ns->href, 
                                              XSLT_NAMESPACE_URI));
        }
        return found;
}


/**
 *Applies a xslt stylesheet to a xml document
 *at libxml level
 *@param src_doc : the libxml2 xmlDocPtr to the source doc
 *@param xsl_doc : the libxml2 xmlDocPtr to the xslt stylesheet
 *@return the xmlDocPtr to the result of the transformation
 *
 */ 
static xmlDocPtr
mlview_xslt_utils_do_transform (xmlDocPtr src_doc,
                          xmlDocPtr xsl_doc)
{
        xmlDocPtr res_doc;
        xmlDocPtr xsl_copy_doc;        
        xsltStylesheetPtr xsl_stylesheet=NULL; 
	const char *params[16+1];
        params[0] = NULL;

        /* makes a copy of the stylesheet */
        xsl_copy_doc = xmlCopyDoc (xsl_doc, 1);
        xsl_stylesheet = xsltParseStylesheetDoc (xsl_copy_doc);
        res_doc = xsltApplyStylesheet(xsl_stylesheet, src_doc, params);
        xsltFreeStylesheet(xsl_stylesheet);
        return res_doc;
}


/**
 *Applies a xslt stylesheet to a xml document
 *at mlview level
 *@param src_doc : the libxml2 xmlDocPtr to the source doc
 *@param xsl_doc : the libxml2 xmlDocPtr to the xslt stylesheet
 *@return the xmlDocPtr to the result of the transformation
 *
 */ 
static MlViewXMLDocument*
mlview_xslt_utils_do_mlview_transform (MlViewXMLDocument* mlv_src_doc,
                                 MlViewXMLDocument* mlv_xsl_doc)
{
        MlViewXMLDocument* mlv_res_doc = NULL;
        MlViewAppContext *app_context;
        xmlDocPtr src_doc, xsl_doc, res_doc;

        app_context = mlview_xml_document_get_app_context(mlv_src_doc); 
        src_doc = mlview_xml_document_get_xml_document(mlv_src_doc);
        xsl_doc = mlview_xml_document_get_xml_document(mlv_xsl_doc);
        res_doc = mlview_xslt_utils_do_transform (src_doc, xsl_doc);
        if (res_doc != NULL) {
                mlv_res_doc = mlview_xml_document_new (res_doc, app_context);
        } else {
                mlview_utils_display_error_dialog (app_context, 
                                                   "%s",  
                                                   _("XSLT transformation failed"));
        }
                
        return mlv_res_doc;
}


/** 
 *
 *Shows a file chooser dialog to select an external stylesheet
 *Loads the file selected, and performs a xslt namespace check
 *@param a_this : mlview editor context
 *@return the document selected or NULL if user cancels or 
 *document is not an xslt stylesheet
 * 
 */
static MlViewXMLDocument*
mlview_xslt_utils_stylesheet_choose_and_open (MlViewEditor *a_this)
{
        GtkWidget *file_sel;
        enum MLVIEW_SELECTED_BUTTON result_file_sel;
        gchar *file_name = NULL;
        MlViewXMLDocument *mlv_xsl_doc = NULL;
        MlViewAppContext *app_context = NULL;

        file_sel = mlview_file_selection_new();
        result_file_sel=mlview_file_selection_run(MLVIEW_FILE_SELECTION(file_sel), TRUE);
        switch (result_file_sel) {
        case OK_BUTTON:
                file_name = g_strdup (gtk_file_selection_get_filename
                                      (GTK_FILE_SELECTION
                                       (file_sel)));
                
                if (file_name && strcmp (file_name, "")) {
                        /* mlview_editor_load_xml_file (a_this, file_name); */
                        app_context = mlview_editor_get_app_context(a_this);
                        mlv_xsl_doc = mlview_xml_document_open(file_name, app_context);
                        mlview_xml_document_ref(mlv_xsl_doc);
                        if (!mlview_xslt_utils_is_xslt_doc(mlv_xsl_doc)) {
                                /*display error box and close doc*/
                                mlview_utils_display_error_dialog (app_context, 
                                                                   "%s",  
                                                                   _("document is not an XSLT Stylesheet"));
                                mlview_xml_document_unref(mlv_xsl_doc);
                                mlv_xsl_doc=NULL;
                        }
                }
                if (file_name) {
                        g_free (file_name);
                        file_name = NULL;
                }                     
                break;
        case CANCEL_BUTTON:
        case WINDOW_CLOSED:
        default:
                break;
        }
        return mlv_xsl_doc;
}


/*=================================================
 *public methods
 *=================================================*/
 
/** 
 *
 *Shows the xslt chooser dialog
 *@param a_this : mlview editor context
 *@return the document selected or NULL if user cancels or 
 *document is not an xslt stylesheet
 * 
 */
MlViewXMLDocument*
mlview_xslt_utils_select_xsl_doc  (MlViewEditor *a_this) 
{
        GtkWidget *dialog1;
        GtkWidget *dialog_vbox1;
        GtkWidget *hbox1;
        GtkWidget *label1;
        GtkWidget *menu;
        GtkWidget *option_menu;
        GtkWidget *menu_item;
        GtkWidget *sel_menu_item;
        GtkWidget *dialog_action_area1;
        GtkWidget *button4;
        GtkWidget *button5;
        GtkWidget *button6;
        
        GList *docs, *mobile_doc_ptr;
        GList *xslt_docs = NULL;
        MlViewXMLDocument *mobile_mlv_doc;
        MlViewXMLDocument *mlv_xsl_doc=NULL;
        gchar *file_path, *base_name;
        gint result;

        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_EDITOR (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);

        /* builds a GList of opened XSLT doc */

        docs = mlview_editor_get_list_open_doc (a_this);
        if (docs != NULL) {
                for (mobile_doc_ptr = g_list_first(docs);
                     mobile_doc_ptr;
                     mobile_doc_ptr = g_list_next(mobile_doc_ptr)) {
                        mobile_mlv_doc = MLVIEW_XML_DOCUMENT(mobile_doc_ptr->data);
                        if (mlview_xslt_utils_is_xslt_doc(mobile_mlv_doc)) {
                                xslt_docs = g_list_append (xslt_docs, (gpointer)mobile_mlv_doc);
                        }
                }
        }
        
        /* build dialog window */

        dialog1 = gtk_dialog_new ();
        gtk_window_set_title (GTK_WINDOW (dialog1), _("Select XSLT"));
        
        dialog_vbox1 = GTK_DIALOG (dialog1)->vbox;
        gtk_widget_show (dialog_vbox1);
        
        hbox1 = gtk_hbox_new (FALSE, 0);
        gtk_widget_show (hbox1);
        gtk_box_pack_start (GTK_BOX (dialog_vbox1), hbox1, TRUE, TRUE, 0);
        
        if (xslt_docs != NULL) {
                label1 = gtk_label_new (_("Select xslt stylesheet"));
        } else {
                label1 = gtk_label_new (_("No xslt stylesheet is open"));
        }
        gtk_widget_show (label1);
        gtk_box_pack_start (GTK_BOX (hbox1), label1, FALSE, FALSE, 10);
        
        /* build select xslt menu */
        
        if (xslt_docs != NULL) {
                option_menu = gtk_option_menu_new();
                menu = gtk_menu_new();
                gtk_option_menu_set_menu (GTK_OPTION_MENU(option_menu), menu);
                
                gtk_widget_show (menu);
                gtk_widget_show (option_menu);
                
                gtk_box_pack_start (GTK_BOX (hbox1), option_menu, TRUE, TRUE, 0);
  
                for (mobile_doc_ptr = g_list_first(xslt_docs);
                     mobile_doc_ptr;
                     mobile_doc_ptr = g_list_next(mobile_doc_ptr)) {
                                mobile_mlv_doc = MLVIEW_XML_DOCUMENT(mobile_doc_ptr->data);
                                file_path = mlview_xml_document_get_file_path (mobile_mlv_doc);
                                base_name = (gchar *) g_basename (file_path);
                                menu_item = gtk_menu_item_new_with_label(base_name);
                                gtk_menu_shell_append (GTK_MENU_SHELL(menu),  menu_item);
                                gtk_widget_show(menu_item);
                                g_object_set_data (G_OBJECT(menu_item), "mlview_doc", mobile_mlv_doc);
                }
                gtk_option_menu_set_history(GTK_OPTION_MENU(option_menu), 0);
        }
        
        /** end menu **/
       
        dialog_action_area1 = GTK_DIALOG (dialog1)->action_area;
        gtk_widget_show (dialog_action_area1);
        gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area1), GTK_BUTTONBOX_END);
 
        button4 = gtk_button_new_with_mnemonic (_("Browse..."));
        gtk_widget_show (button4);
        gtk_dialog_add_action_widget (GTK_DIALOG (dialog1), button4, MLVIEW_XSLT_UTILS_RESPONSE_BROWSE);
        GTK_WIDGET_SET_FLAGS (button4, GTK_CAN_DEFAULT);
        
        button5 = gtk_button_new_from_stock ("gtk-cancel");
        gtk_widget_show (button5);
        gtk_dialog_add_action_widget (GTK_DIALOG (dialog1), button5, GTK_RESPONSE_CANCEL);
        GTK_WIDGET_SET_FLAGS (button5, GTK_CAN_DEFAULT);
        
        if (xslt_docs != NULL) {
                button6 = gtk_button_new_from_stock ("gtk-ok");
                gtk_widget_show (button6);
                gtk_dialog_add_action_widget (GTK_DIALOG (dialog1), button6, GTK_RESPONSE_OK);  
                GTK_WIDGET_SET_FLAGS (button6, GTK_CAN_DEFAULT);
        }

        /* runs the dialog box */
        result = gtk_dialog_run (GTK_DIALOG (dialog1));
        switch (result) {
        case GTK_RESPONSE_OK:
                sel_menu_item = gtk_menu_get_active(GTK_MENU(menu));                
                mlv_xsl_doc = g_object_get_data (G_OBJECT(sel_menu_item), "mlview_doc");
                break;
        case GTK_RESPONSE_CANCEL:
                break;
        case MLVIEW_XSLT_UTILS_RESPONSE_BROWSE:
                mlv_xsl_doc = mlview_xslt_utils_stylesheet_choose_and_open(a_this);
                break;
        }

        g_list_free(docs);
        g_list_free(xslt_docs);
        gtk_widget_destroy (dialog1);

        return mlv_xsl_doc;
}


/**
 *Applies a XSLT stylesheet to an mlview document
 *
 *@param src_doc the source mlview xml document
 *@param xsl_doc the xslt mlview xml document
 *@result the document resuting of the transformation
 * or NULL if it fails
 *
 */
MlViewXMLDocument *
mlview_xslt_utils_transform_document (MlViewXMLDocument *src_doc, 
                                      MlViewXMLDocument *xsl_doc)
{
        MlViewXMLDocument *res_doc = NULL;
        g_return_val_if_fail (src_doc != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT (src_doc),
                              NULL);
        g_return_val_if_fail (xsl_doc != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT (xsl_doc),
                              NULL);        
        res_doc = mlview_xslt_utils_do_mlview_transform(src_doc, xsl_doc);
        return res_doc;
}
