/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 8-*- */

/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute it 
 *and/or modify it under the terms of 
 *the GNU General Public License as published 
 *by the Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *MlView is distributed in the hope that it will be useful, 
 *but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. If not, write to the Free Software Foundation, Inc., 
 *59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include <string.h>
#include <libxml/uri.h>
#include <libxml/tree.h>
#include "mlview-editor.h"
#include "mlview-tree-view.h"
#ifdef MLVIEW_WITH_STYLE
#include "mlview-styled-view.h"
#endif
#include "mlview-file-descriptor.h"
#include "mlview-xml-document.h"
#include "mlview-utils.h"
#include "mlview-xslt-utils.h"


/**
 *@file
 *The definition of the methods of #MlViewEditor class.
 */

enum {
        DOCUMENT_CHANGED,
        LAST_VIEW_REMOVED,
        FIRST_VIEW_ADDED,
        SIGNAL_NUM
};

struct MlViewViewDesc gv_view_types[] = {
        {
                (gchar*)"tree-view", 
                (gchar*)N_("An editing view well suited for tree the oriented editing paradigm"),
                mlview_tree_view_new
        },
#ifdef MLVIEW_WITH_STYLE
        {
                (gchar*)"styled-view",
                (gchar*)N_("A highly experimental view to render/edit XML using a CSS"),
                mlview_styled_view_new
        },
#endif        
        {0}
} ;

/*======================================================
 *Some private (static) data structures used by
 *MlViewEditor widget.
 *=====================================================*/

/**
 *The #MlViewEditor private data member structure.
 *The fields of this structure must be accessed only by 
 *the mean of the available accessors.
 */
struct _MlViewEditorPrivate {
        /**
	 *The set of instances of #MlViewXMLDocument
	 *opened in the editor. The key is the pointer to 
	 *the instance of #MlViewXMLDocument opened in the editor
	 *The value is a an hash table that contains the 
	 *instances of #MlViewXMLDocumentView that holds 
	 *the key #MlViewXMLDocument.
	 */
        GHashTable *mlview_xml_docs;

        /**
	 *A hash table where the keys are the instances of
	 *MlViewXMLDocumentViews and the values are their
	 *associated document.
	 */
        GHashTable *mlview_xml_doc_views;

        /**
	 *The notebook that holds all the notebook pages (document views)
	 */
        GtkNotebook *notebook;

        /**
	 *The current document being viewed by the user
	 */
        MlViewIView *cur_view ;

        /**
	 *a hash table which keys are the base name of 
	 *the files already opened. 
	 *The associated data is the number 
	 *of times the file name
	 *has been opened. When destroying this hash 
	 *table, do not destroy the referenced data (base names) because
	 *the base names are substring of file_paths. 
	 *And file names are hold by the instances of MlViewXMLDocument.
	 */
        GHashTable *opened_file_base_names;

        /**
	 *a hash table wich keys are the file paths of 
	 *the file files already opened.
	 *the associated MlViewXMLDocument. 
	 *When destroying this hashtable do not 
	 *destroy the referenced data (file path)
	 *because file paths are hold by instances 
	 *of MlViewXMLDocument and destroyed by them.
	 */
        GHashTable *opened_file_paths;

        /**
	 *An hash table that associates the opened document label names to
	 *the matching opened view.
	 */
        GHashTable *opened_document_label_names;

        /**
         *Number of untitled document opened.
         */
        guint untitled_docs_num;

        /**
         *total number of docs opened
         */
        guint opened_docs_num;

        /**
         *The context of the application
         */
        MlViewAppContext *app_context;

        /*
         *the editor contextual menu 
         */
        GtkWidget *contextual_menu;

        gboolean dispose_has_run ;
};

#define PRIVATE(editor) (editor->priv)

static void mlview_editor_class_init (MlViewEditorClass * a_klass);

static void mlview_editor_init (MlViewEditor * a_this);

/*signal callbacks and default handlers*/
static void mlview_editor_switch_notebook_page_cb (GtkNotebook * a_notebook,
                                                   GtkNotebookPage * a_page,
                                                   gint a_page_num,
                                                   MlViewEditor * a_this);

static void view_name_changed_cb (MlViewIView * a_view,
                                  gpointer a_this);

static GList *build_view_list_from_hashtable (GHashTable *a_views);

static GList *build_doc_list_from_hashtable (GHashTable * a_docs);

static void add_hash_key_to_list (gpointer a_key,
                                  gpointer a_value, GList ** a_list);

static void mlview_editor_finalize (GObject *a_this) ;

static void mlview_editor_dispose (GObject *a_this) ;

static GtkVBoxClass *gv_parent_class = NULL;
static guint gv_signals[SIGNAL_NUM] = { 0 };



/*============================================================
 *private method required by the GTK typing system
 *=============================================================*/


/**
 *The vtable initialyzer.
 *@param a_klass the vtable structure.
 *
 */
static void
mlview_editor_class_init (MlViewEditorClass * a_klass)
{
        GObjectClass *object_class = NULL ;

        gv_parent_class = g_type_class_peek_parent (a_klass);

        object_class = G_OBJECT_CLASS (a_klass) ;
        g_return_if_fail (object_class) ;

        /*overload the destroy method of GtkObject */
        object_class->dispose = mlview_editor_dispose ;
        object_class->finalize = mlview_editor_finalize ;

        /*define the signals */
        gv_signals[DOCUMENT_CHANGED] =
                g_signal_new ("document-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewEditorClass,
                               document_changed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              GTK_TYPE_NONE, 0, NULL);

        gv_signals[LAST_VIEW_REMOVED] =
                g_signal_new ("last_view_removed",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewEditorClass,
                               last_view_removed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              GTK_TYPE_NONE, 0, NULL) ;

        gv_signals[FIRST_VIEW_ADDED] =
                g_signal_new ("first_view_added",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewEditorClass,
                               first_view_added), NULL, NULL,
                              gtk_marshal_NONE__POINTER,
                              GTK_TYPE_NONE, 1, G_TYPE_POINTER) ;

        /*gtk_object_class_add_signals (object_class, mlview_editor_signals, 
           SIGNAL_NUM) ; */
        a_klass->document_changed = NULL;
}


/**
 *The #MlViewEditor instance allocator.
 *Allocates the space necessary to the members of the #MlViewEditor structure
 *and does very basic initialyzation.
 *
 *@param a_this the instance of #MlViewEditor to allocate.
 */
static void
mlview_editor_init (MlViewEditor * a_this)
{
        g_assert (a_this != NULL);

        PRIVATE (a_this) =
                g_malloc0 (sizeof (MlViewEditorPrivate));

        PRIVATE (a_this)->notebook =
                GTK_NOTEBOOK (gtk_notebook_new ());

        g_signal_connect (G_OBJECT
                          (PRIVATE (a_this)->notebook),
                          "switch-page",
                          G_CALLBACK
                          (mlview_editor_switch_notebook_page_cb),
                          a_this);

        gtk_box_pack_start (GTK_BOX (a_this),
                            GTK_WIDGET (PRIVATE (a_this)->
                                        notebook), TRUE, TRUE,
                            0);

        PRIVATE (a_this)->opened_file_base_names =
                g_hash_table_new (g_str_hash, g_str_equal);

        PRIVATE (a_this)->opened_file_paths =
                g_hash_table_new (g_str_hash, g_str_equal);

        PRIVATE (a_this)->opened_document_label_names =
                g_hash_table_new (g_str_hash, g_str_equal);

        PRIVATE (a_this)->mlview_xml_docs =
                g_hash_table_new (g_direct_hash, g_direct_equal);

        PRIVATE (a_this)->mlview_xml_doc_views =
                g_hash_table_new (g_direct_hash, g_direct_equal);

        PRIVATE (a_this)->untitled_docs_num = 0;

        PRIVATE (a_this)->opened_docs_num = 0;
}


static void
mlview_editor_dispose (GObject *a_this)
{
        MlViewEditor *editor = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)) ;
        editor = MLVIEW_EDITOR (a_this) ;
        g_return_if_fail (PRIVATE (editor))  ;

        if (PRIVATE (editor)->dispose_has_run == TRUE)
                return ;
        /*
         *destroy each remaining view.
         **/
        if (PRIVATE (editor)->mlview_xml_doc_views) {
                GList *list_item = NULL ;
                GList *view_list = build_view_list_from_hashtable 
                        (PRIVATE (editor)->mlview_xml_doc_views) ;

                for (list_item = view_list ;
                     list_item ;list_item = list_item->next) {
                        if (list_item->data 
                            && MLVIEW_IS_IVIEW (list_item->data)) {
                                mlview_editor_remove_xml_document_view 
                                        (editor, MLVIEW_IVIEW 
                                         (list_item->data)) ;
                        }
                }
                g_hash_table_destroy
                        (PRIVATE (editor)->mlview_xml_doc_views);
                PRIVATE (editor)->mlview_xml_doc_views = NULL;
        }
        /*
         *do not free this element because 
         *it has been freed as an element of the link list above
         */
        PRIVATE (editor)->cur_view = NULL;

        if (PRIVATE (editor)->mlview_xml_docs) {
                g_hash_table_destroy (PRIVATE (editor)->mlview_xml_docs) ;
                PRIVATE (editor)->mlview_xml_docs = NULL ;
        }
        if (PRIVATE (editor)->opened_file_base_names) {
                g_hash_table_destroy
                        (PRIVATE (editor)->
                         opened_file_base_names);

                PRIVATE (editor)->opened_file_base_names = NULL;
        }
        if (PRIVATE (editor)->opened_file_paths) {
                g_hash_table_destroy
                        (PRIVATE (editor)->opened_file_paths);

                PRIVATE (editor)->opened_file_paths = NULL;
        }

        if (PRIVATE (editor)->opened_document_label_names) {
                g_hash_table_destroy
                        (PRIVATE (editor)->
                         opened_document_label_names);

                PRIVATE (editor)->opened_document_label_names = NULL;
        }

        if (G_OBJECT_CLASS (gv_parent_class)->dispose) {
                G_OBJECT_CLASS (gv_parent_class)->dispose (a_this) ;
        }
}


static void
mlview_editor_finalize (GObject *a_this)
{
        MlViewEditor *editor = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)) ;
        
        editor = MLVIEW_EDITOR (a_this) ;

        if (PRIVATE (editor)) {
                g_free (PRIVATE (editor)) ;
                PRIVATE (editor) = NULL ;
        }
}

/*================================================
 *signal callbacks and default handlers
 *
 *=================================================*/

/**
 *this callback is called when the editor
 *switch from one view to another.
 *
 *@param a_notebook the notebook editor that holds the view notebook pages.
 *@param a_page the notebook page that holds the view 
 *(the view is an instance of #MlViewXMLDocumentView)
 *@param a_page_num the number of the new notebook page.
 *@param a_this the current instance of #MlViewEditor.
 */
static void
mlview_editor_switch_notebook_page_cb (GtkNotebook * a_notebook,
                                       GtkNotebookPage * a_page,
                                       gint a_page_num,
                                       MlViewEditor * a_this)
{
        MlViewIView *doc_view = NULL, *prev_view = NULL ;
        GtkWidget *cur_child_widget = NULL, *main_menubar = NULL;
        MlViewAppContext *ctxt = NULL ;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        cur_child_widget = gtk_notebook_get_nth_page (a_notebook,
                                                      a_page_num);
        g_return_if_fail (cur_child_widget);
        doc_view = MLVIEW_IVIEW (cur_child_widget) ;
        g_return_if_fail (doc_view != NULL);

        prev_view = PRIVATE (a_this)->cur_view ;
        PRIVATE (a_this)->cur_view = doc_view;

        /*
         *Detach the content of the edit menu 
         *(when we can do menu merging)so that
         *the newly selected view can attach it's "edit"
         *menu content there.
         */
        ctxt = mlview_editor_get_app_context (a_this) ;
        g_return_if_fail (ctxt) ;

        main_menubar =
                mlview_app_context_get_element
                (ctxt, "MlViewAppMainMenuBar") ;
        if (main_menubar) {
                /*
                 *put here the code of menu merging when we
                 *have it.
                 */
        }
        /*
         *notify the application that a new view became the
         *current view.
         */
        mlview_app_context_notify_view_swapped (ctxt,
                                                prev_view, 
                                                doc_view) ;
}

/*
 *TODO: replace all the MlViewXMLDocumentView references 
 *by MlViewIView.
 *do not forget to call mlview_iview_connect_to_doc() during
 *the instanciation of a view.
 */


/**
 *This callback is called when the name of the current
 *instance of #MlViewXMLDocumentView changes.
 *
 *@param a_view the current instance of #MlViewXMLDocumentView
 *which name has changed.
 *@param a_this the current instance of #MlViewEditor.
 */
static void
view_name_changed_cb (MlViewIView * a_view,
                      gpointer a_this)
{
        MlViewEditor *editor = NULL;
        guchar *new_view_name = NULL;

        g_return_if_fail (a_view 
                          && MLVIEW_IS_IVIEW (a_view)
                          && a_this
                          && MLVIEW_IS_EDITOR (a_this));

        editor = MLVIEW_EDITOR (a_this);
        g_return_if_fail (PRIVATE (editor) != NULL);
        g_return_if_fail (PRIVATE (editor)->notebook != NULL);

        mlview_iview_get_name (a_view, &new_view_name) ;
        g_return_if_fail (new_view_name) ;
        gtk_notebook_set_tab_label_text 
                (PRIVATE (editor)->notebook,
                 GTK_WIDGET (a_view), new_view_name);

}

static GList *
build_view_list_from_hashtable (GHashTable * a_views)
{
        GList *result = NULL;

        g_hash_table_foreach (a_views,
                              (GHFunc) add_hash_key_to_list,
                              &result);
        return result;
}

static GList *
build_doc_list_from_hashtable (GHashTable * a_docs)
{
        GList *result = NULL;

        g_hash_table_foreach (a_docs,
                              (GHFunc) add_hash_key_to_list,
                              &result);
        return result;
}

static void
add_hash_key_to_list (gpointer a_key,
                      gpointer a_value, GList ** a_list)
{

        g_return_if_fail (a_list != NULL);
        *a_list = g_list_append (*a_list, a_key);
}

static struct MlViewViewDesc *
mlview_editor_select_view_to_open (void)
{
        GtkWidget *dialog;
        GtkWidget *dialog_vbox;
        GtkWidget *hbox1;
        GtkWidget *label1;
        GtkWidget *option_menu;
        GtkWidget *menu;
        GtkWidget *menu_item;
        GtkWidget *dialog_action_area1;
        GtkWidget *button5;
        GtkWidget *button6;
        GtkWidget *sel_menu_item;

        enum MLVIEW_SELECTED_BUTTON button;
        struct MlViewViewDesc *result = NULL;
        struct MlViewViewDesc *view_desc_ptr = NULL ;
        gchar *base_name;
        guint nr_view_desc = 0 ;

        nr_view_desc = mlview_editor_get_number_of_view_desc () ;
        g_return_val_if_fail (nr_view_desc, NULL) ;        
        if (nr_view_desc == 1) {
                result = mlview_editor_get_view_descriptor_at (0) ;
                g_return_val_if_fail (result, NULL) ;
                return result ;
        }

        dialog = gtk_dialog_new ();
        gtk_window_set_title (GTK_WINDOW (dialog), _("Select View"));
        
        dialog_vbox = GTK_DIALOG (dialog)->vbox;
        gtk_widget_show (dialog_vbox);
        
        hbox1 = gtk_hbox_new (FALSE, 0);
        gtk_widget_show (hbox1);
        gtk_box_pack_start (GTK_BOX (dialog_vbox), hbox1, TRUE, TRUE, 0);
        label1 = gtk_label_new (_("Select view to open"));
        gtk_widget_show (label1);
        gtk_box_pack_start (GTK_BOX (hbox1), label1, FALSE, FALSE, 10);
        
        /* build select view menu */
        option_menu = gtk_option_menu_new();
        menu = gtk_menu_new();
        gtk_option_menu_set_menu (GTK_OPTION_MENU(option_menu), menu);
        
        gtk_widget_show (menu);
        gtk_widget_show (option_menu);
        
        gtk_box_pack_start (GTK_BOX (hbox1), option_menu, TRUE, TRUE, 0);
        for (view_desc_ptr = gv_view_types; 
             view_desc_ptr && view_desc_ptr->view_type_name;
             view_desc_ptr ++) {
                base_name = view_desc_ptr->view_type_name; 
                menu_item = gtk_menu_item_new_with_label(base_name);
                gtk_menu_shell_append (GTK_MENU_SHELL(menu),  menu_item);
                gtk_widget_show(menu_item);
                g_object_set_data (G_OBJECT(menu_item), "mlview_view_desc", (gpointer)view_desc_ptr);
        }
        gtk_option_menu_set_history(GTK_OPTION_MENU(option_menu), 0);

        /* dialog box buttons */
        dialog_action_area1 = GTK_DIALOG (dialog)->action_area;
        gtk_widget_show (dialog_action_area1);
        gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area1), GTK_BUTTONBOX_END);

        button5 = gtk_button_new_from_stock ("gtk-cancel");
        gtk_widget_show (button5);
        gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button5, GTK_RESPONSE_CANCEL);
        GTK_WIDGET_SET_FLAGS (button5, GTK_CAN_DEFAULT);

        button6 = gtk_button_new_from_stock ("gtk-ok");
        gtk_widget_show (button6);
        gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button6, GTK_RESPONSE_OK);  
        GTK_WIDGET_SET_FLAGS (button6, GTK_CAN_DEFAULT);

        /* show dialog */
        button = gtk_dialog_run (GTK_DIALOG (dialog));


        switch (button) {
        case GTK_RESPONSE_OK:
                sel_menu_item = gtk_menu_get_active(GTK_MENU(menu));  
             
                result = (struct MlViewViewDesc *)g_object_get_data (G_OBJECT(sel_menu_item), "mlview_view_desc");
                break;
                
        default:
                result = NULL ;
                break;
        }
        gtk_widget_destroy (dialog);
        return result; 
}

/*=================================================
 *public methods
 *=================================================*/

/**
 *the standard type builder of #MlViewEditor 
 *@return the type id of #MlViewEditor.
 */
GType
mlview_editor_get_type (void)
{
        static guint type = 0;

        if (!type) {
                static const GTypeInfo type_info = {
                        sizeof (MlViewEditorClass),
                        NULL, NULL,
                        (GClassInitFunc)
                                mlview_editor_class_init,
                        NULL, NULL,
                        sizeof (MlViewEditor),
                        0,
                        (GInstanceInitFunc) mlview_editor_init
                };

                type = g_type_register_static (GTK_TYPE_VBOX,
                                               "MlViewEditor",
                                               &type_info, 0);
        }
        return type;
}


/**
 *Creates and returns a new instance of #MlViewEditor.
 *
 *@param a_title the title of the editor (not used yet).
 *@param a_context the current instance of MlViewApplicationContext, or NULL
 *if you want this function to create a new context.
 */
GtkWidget *
mlview_editor_new (const guchar * a_title,
                   MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;
        MlViewAppContext *context = a_context ;

        if (!context) {
               context =  MLVIEW_APP_CONTEXT (mlview_app_context_get_instance ()) ;
               g_return_val_if_fail (context, NULL) ;
        }

        /*
         *if this app context is brand new 
         *(i.e: this instance of #MlViewEditor created it), make sure
         *the current instance of #MlViewEditor is "set" in the 
         *app context
         *If the app context is not "new", don't set the current instance
         *of #MlViewEditor in it. It would be the duty of the caller to
         *set the current instance of #MlViewEditor in the app context,
         *under a name she chooses.
         */
        if (!a_context) {
                mlview_app_context_set_element (context,
                                                "MlViewEditor", 
                                                editor);
        }

        editor = g_object_new (MLVIEW_TYPE_EDITOR, NULL);

        PRIVATE (editor)->app_context = context ;

        /* 
         *FIXME: work out how to handle the mouse event etc ...
         * The way I used to do it is broken.
        tmp = mlview_app_context_get_element (a_context,
                                              "GnomeApp");
        if (tmp && GTK_IS_WIDGET (tmp)) {
                GtkWidget *parent_app = NULL;

                parent_app = GTK_WIDGET (tmp);
                gtk_widget_add_events (parent_app,
                                       GDK_BUTTON3_MOTION_MASK);
                gtk_signal_connect (GTK_OBJECT (parent_app),
                                    "button_press_event",
                                    GTK_SIGNAL_FUNC
                                    (mlview_editor_event_cb),
                                    editor);
        }
        */

        return GTK_WIDGET (editor);
}


/**
 *The setter of the application context associated to the 
 *current mlview editor.
 *
 *@param a_this the current instance of #MlViewEditor.
 *@param a_context the new application context.
 */
void
mlview_editor_set_app_context (MlViewEditor * a_this,
                               MlViewAppContext * a_context)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (a_context != NULL);
        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (a_context));

        PRIVATE (a_this)->app_context = a_context;
}


/**
 *The getter of the application context associated
 *to the current mlview editor.
 *
 *@param a_this the current instance of #MlViewEditor.
 *@return the application context associated to the editor.
 */
MlViewAppContext *
mlview_editor_get_app_context (MlViewEditor * a_this)
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_EDITOR (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);

        return PRIVATE (a_this)->app_context;
}

/**
 *Opens the xml file a_file_path, loads it, build a view to edit it and
 *adds this new view to the editor.
 *
 *@param a_this the current instance of #MlViewEditor.
 *@param a_file_path the path of the file to open.
 */
void
mlview_editor_load_xml_file (MlViewEditor * a_this,
                             gchar * a_file_path)
{
        enum MlViewStatus status = MLVIEW_OK ;
        gboolean is_relative = FALSE ;
        gchar *absolute_path = NULL, *cur_dir = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)
                          && PRIVATE (a_this)) ;

        status = mlview_utils_uri_is_relative (a_file_path, &is_relative) ;
        if (status != MLVIEW_OK) {
                mlview_app_context_error (PRIVATE (a_this)->app_context, 
                                          _("The following URI is not well formed: %s"),
                                          a_file_path) ;
                return ;
        }
        if (is_relative == TRUE) {
                cur_dir = g_get_current_dir () ;
                g_return_if_fail (cur_dir) ;
                mlview_utils_relative_uri_to_absolute_uri (a_file_path,
                                                           cur_dir,
                                                           &absolute_path) ;
                g_return_if_fail (absolute_path) ;                
        } else {
                absolute_path = g_strdup (a_file_path) ;
        }
        mlview_editor_load_xml_file_with_dtd (a_this, absolute_path, NULL);
        g_free (absolute_path) ;
}

/**
 *Opens the xml file a_file_path, loads it, build a view to edit it and
 *adds this new view to the editor.
 *The file is validated with the DTD if provided.
 *
 *@param a_this the current instance of #MlViewEditor.
 *@param a_file_path the path of the file to open.
 *@param a_dtd_path the path of the dtd to validate against.
 */
void
mlview_editor_load_xml_file_with_dtd (MlViewEditor * a_this,
                                      gchar * a_file_path,
                                      gchar * a_dtd_path)
{
        MlViewXMLDocument *mlview_xml_document = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this));

        if (a_file_path == NULL)
                return;

        mlview_app_context_sbar_push_message (PRIVATE (a_this)->app_context,
                                              _("Opening file %s..."),
                                              a_file_path);
        mlview_xml_document =
                mlview_xml_document_open_with_dtd (a_file_path,
                                                   a_dtd_path,
                                                   PRIVATE (a_this)->app_context);

        if (mlview_xml_document) {
                GtkWidget *parent_window = NULL;
                MlViewIView *new_view = NULL ;

                new_view = MLVIEW_IVIEW 
                        (mlview_tree_view_new 
                         (mlview_xml_document,
                          a_file_path,
                          PRIVATE (a_this)->app_context)) ;

                g_return_if_fail (new_view != NULL);

                parent_window = gtk_widget_get_toplevel
                        (GTK_WIDGET (a_this));

                mlview_editor_add_xml_document_view
                        (a_this, new_view) ;
        }

        /*
         *FIXME: add support for exception thrown by 
         *mlview_xml_document_open in case 
         *of error and display that error => BIG DEAL
         */

        mlview_app_context_sbar_pop_message (PRIVATE (a_this)->
                                             app_context);
}


/**
 *interactively open/edit a local xml file name.
 *@param a_this the current mlview editor.
 */
void
mlview_editor_open_local_xml_document_interactive (MlViewEditor *a_this)
{
        MlViewFileSelection *file_selector;
        enum MLVIEW_SELECTED_BUTTON button;
        gchar *file_name = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (PRIVATE (a_this)->app_context !=
                          NULL);

        file_selector =
                mlview_app_context_get_file_selector
                (PRIVATE (a_this)->app_context,
                 _("Open xml document"));

        g_return_if_fail (file_selector != NULL);

        mlview_app_context_sbar_push_message
                (PRIVATE (a_this)->app_context,
                 _("Choose the xml file to open"));

        button = mlview_file_selection_run
                (MLVIEW_FILE_SELECTION (file_selector), TRUE);

        switch (button) {
        case OK_BUTTON:
                file_name =
                        g_strdup (gtk_file_selection_get_filename
                                  (GTK_FILE_SELECTION
                                   (file_selector)));

                if (file_name && strcmp (file_name, ""))
                        mlview_editor_load_xml_file (a_this,
                                                     file_name);

                if (file_name) {
                        g_free (file_name);
                        file_name = NULL;
                }
        case CANCEL_BUTTON:
        case WINDOW_CLOSED:
        default:
                break;
        }

        mlview_app_context_sbar_pop_message
                (PRIVATE (a_this)->app_context);
}


/**
 *Creates a new view on an existing document.
 *
 *@param a_this the current mlview editor.
 *@param a_xml_doc the xml document on which the new
 *view must be created.
 *@param a_view_type the type of view.
 *@return the newly created view.
 *TODO:-> plug here the dynamic view choosing stuff.
 */
MlViewIView *
mlview_editor_create_new_view_on_document (MlViewEditor *a_this,
                                           MlViewXMLDocument *a_xml_doc)
{
        MlViewIView *result = NULL;
        struct MlViewAppSettings *settings = NULL ;
        struct MlViewViewDesc *view_descriptor = NULL ;
        
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_EDITOR (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);
        g_return_val_if_fail (a_xml_doc != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc),
                              NULL);
        g_return_val_if_fail (PRIVATE (a_this)->app_context,
                              NULL) ;

        settings = mlview_app_context_get_settings 
                (PRIVATE (a_this)->app_context) ;
        g_return_val_if_fail (settings, NULL) ;
              
        view_descriptor = mlview_editor_select_view_to_open ();

        if (!view_descriptor
            || !view_descriptor->view_constructor) {
                mlview_utils_trace_info
                        ("Unknown view type name: ") ;
                mlview_utils_trace_info (settings->general.default_editing_view_type) ;
                return NULL ;
        }
        result = MLVIEW_IVIEW (view_descriptor->view_constructor
                               (a_xml_doc, NULL,
                                PRIVATE (a_this)->app_context)) ;
        mlview_editor_add_xml_document_view (a_this, result) ;
          return result;
 
}

/**
 *Interactively creates a new view on an existing
 *xml document.
 *The view will be created if and only if an non empty document is
 *currently selected in the editor.
 *
 *@param a_this the current mlview editor.
 *@return the newly created view or NULL if it could not be created.
 */
MlViewIView*
mlview_editor_create_new_view_on_current_document_interactive (MlViewEditor * a_this) 
{
        MlViewXMLDocument *xml_document = NULL;

        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_EDITOR (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);


        if (PRIVATE (a_this)->cur_view == NULL)
                return NULL;

        mlview_iview_get_document
                (PRIVATE (a_this)->cur_view, &xml_document);

        return mlview_editor_create_new_view_on_document
                (a_this, xml_document);

}

/**
 *Getter of the current selected edition
 *view.
 *@param a_this the current instance of #MlViewEditor.
 *@return the current MlViewXMLDocumentView or NULL.
 */

MlViewIView *
mlview_editor_get_cur_doc_view (MlViewEditor * a_this)
{
        g_return_val_if_fail (a_this && PRIVATE (a_this), NULL);

        return PRIVATE (a_this)->cur_view;
}


/**
 *Getter of the current selected instance of
 *#MlViewXMLDocument *
 *
 *@param a_this the current instance of #MlViewEditor.
 *@return the current selected instance of #MlViewXMLDocument,
 *or NULL.
 */
MlViewXMLDocument *
mlview_editor_get_cur_doc (MlViewEditor * a_this)
{
        MlViewXMLDocument *doc = NULL ;

        g_return_val_if_fail (a_this && PRIVATE (a_this), NULL);

        if (PRIVATE (a_this)->cur_view) {
                mlview_iview_get_document
                        (PRIVATE (a_this)->cur_view, &doc);
        }

        return doc ;
}

/**
 *Getter of the glist of opened docs
 *#MlViewXMLDocument *
 *
 *@param a_this the current instance of #MlViewEditor.
 *@return a GList* of #MlViewXMLDocument, taht the caller 
 * is responsble to free
 */
GList *
mlview_editor_get_list_open_doc (MlViewEditor * a_this)
{
   GList *docs = NULL;

   docs = build_doc_list_from_hashtable
                (PRIVATE (a_this)->mlview_xml_docs);

   return docs;
}

/**
 *Setter of the name of the currently selected view.
 *
 *@param a_this the current mlview editor.
 *@param a_name the new name of the view.
 */
void
mlview_editor_set_current_view_name (MlViewEditor * a_this, gchar * a_name)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        if (PRIVATE (a_this)->cur_view == NULL)
                return;
        mlview_iview_set_name
                (PRIVATE (a_this)->cur_view, a_name);
}


/**
 *Interactively sets the name of the view currently
 *selected in the mlview editor.
 *
 *@param a_this the current mlview editor.
 */
void
mlview_editor_set_current_view_name_interactive (MlViewEditor *a_this)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        if (PRIVATE (a_this)->cur_view == NULL)
                return;

        mlview_iview_set_name_interactive
                (PRIVATE (a_this)->cur_view);
}


/**
 *Adds a document view to the editor. If a_xml_doc_view is a new 
 *document view, adds a new notebook page to the #MlViewEditor notebook.
 *The base name of the document is shown in the notebook page tab. 
 *If the underlying document of a_xml_doc_view has the 
 *same base name as a document already opened,
 *the notebook page tab string will be "basename<nb>" where 'nb' is 
 *the number of docs that have the same name in the editor (like in emacs).
 *If a_xml_doc_view is already loaded in the current instance of #MlViewEditor, 
 *the old instance of a_xml_doc is destroyed 
 *(as well as the notebook page that contains it) and the new 
 *instance of a_xml_doc is added to the editor. 
 *
 *@param a_this the current mlview editor.
 *@param a_xml_doc_view the document view to be added to the editor.
 */
void
mlview_editor_add_xml_document_view (MlViewEditor * a_this,
                                     MlViewIView *a_view)
{
        MlViewFileDescriptor *file_desc = NULL;
        guchar *file_path = NULL,
                *base_name = NULL,
                *label_str = NULL;
        MlViewXMLDocument *mlview_xml_document = NULL;
        GHashTable *views_associated_to_document = NULL;
        MlViewIView *iview = NULL;
        gpointer ptr = NULL ;
        gboolean is_new_doc_tree = TRUE;
        GtkWidget *label = NULL, *view_impl = NULL,
                *view_impl_tmp = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (PRIVATE (a_this)->notebook != NULL);
        g_return_if_fail (a_view != NULL);
        g_return_if_fail (MLVIEW_IS_IVIEW (a_view)) ;
        mlview_iview_get_impl (a_view, &view_impl) ;
        g_return_if_fail (view_impl) ;
        g_return_if_fail (PRIVATE (a_this)->mlview_xml_docs);
        g_return_if_fail (PRIVATE (a_this)->mlview_xml_doc_views);

        mlview_iview_get_document (a_view, &mlview_xml_document) ;
        g_return_if_fail (mlview_xml_document != NULL);

        file_desc = mlview_xml_document_get_file_descriptor 
                (mlview_xml_document) ;
        if (file_desc)
                file_path =
                        mlview_file_descriptor_get_file_path
                        (file_desc) ;
        /*check if a view on the same document has been added to
         *the editor already.
         */
        views_associated_to_document =
                g_hash_table_lookup
                (PRIVATE (a_this)->mlview_xml_docs,
                 mlview_xml_document);

        if (views_associated_to_document) {
                is_new_doc_tree = FALSE;
        }

        if (file_path == NULL) {
                gchar *tmp_str = NULL,
                        *label_str = NULL;

                if (is_new_doc_tree == TRUE)
                        PRIVATE (a_this)->untitled_docs_num++;
                tmp_str =
                        g_strdup_printf
                        ("%d",
                         PRIVATE (a_this)->untitled_docs_num);

                label_str =
                        g_strconcat ("untitled", tmp_str, NULL);
                label = gtk_label_new (label_str);
                g_free (label_str);
                g_free (tmp_str);

        } else {

                gint base_name_nb = 0;

                gboolean file_is_already_opened = FALSE;

                base_name = (guchar *) g_basename (file_path);

                if (is_new_doc_tree
                    && (iview = g_hash_table_lookup
                        (PRIVATE (a_this)->opened_file_paths,
                         file_path)) != NULL) {
                        /*
                         *There is an xml document coming from the
                         *same url that is opened already. 
                         *So, reopen it.
                         *That is, remove it previous 
                         *instance and add
                         *the new one.
                         */
                        GtkWidget *old_label = NULL ;
                        gchar *old_label_str_tmp = NULL,
                                *old_label_str = NULL;
                        
                        /*
                         *get the old label string because 
                         *this document
                         *must have the same label as 
                         *the one alreay opened
                         */
                        mlview_iview_get_impl (iview, 
                                               &view_impl_tmp) ;
                        old_label = gtk_notebook_get_tab_label
                                (PRIVATE (a_this)->notebook,
                                 view_impl_tmp);

                        g_assert (old_label != NULL);

                        gtk_label_get (GTK_LABEL (old_label),
                                       &old_label_str_tmp);

                        /*
                         *make a copy of the label string because
                         *mlview_editor_remove_xml_document ()
                         *will destroy this label.
                         */

                        /*old_label_str_tmp belongs to label,
                         *and will be freed by him
                         */
                        old_label_str =
                                g_strdup (old_label_str_tmp);

                        mlview_editor_remove_xml_document_view
                                (a_this, iview);

                        /*create the label of this document notebook page */
                        label = gtk_label_new (old_label_str);
                        /*old_label_str has been strduped */
                        g_free (old_label_str);
                        gtk_label_get (GTK_LABEL (label),
                                       &old_label_str);
                        /*
                         *old_label_str belongs to 
                         *label and will be freed by him
                         *so we can use it as a key in a hash 
                         *table without any need of
                         *freeing it.
                         */
                        g_hash_table_insert
                                (PRIVATE
                                 (a_this)->
                                 opened_document_label_names,
                                 g_strdup (old_label_str),
                                 a_view);

                        file_is_already_opened = TRUE;

                } else if ((ptr = g_hash_table_lookup
                            (PRIVATE (a_this)->
                             opened_file_base_names,
                             base_name)) == NULL) {
                        /*
                         *It is the first time a 
                         *document with this basename is opened
                         */
                        base_name_nb = 1;

                } else if (ptr != NULL) {
                        /*
                         *some documents with the this basename 
                         *are already opened
                         */
                        base_name_nb = GPOINTER_TO_INT (ptr);
                        if (!is_new_doc_tree)
                                base_name_nb++;
                }

                g_hash_table_insert
                        (PRIVATE (a_this)->opened_file_base_names, 
                         base_name, GINT_TO_POINTER (base_name_nb));
                g_hash_table_insert
                        (PRIVATE (a_this)->opened_file_paths,
                         file_path, a_view);
                if (base_name_nb > 1) {
                        gchar *tmp_str = NULL ;
                        while (1) {
                                tmp_str =
                                        g_strdup_printf 
                                        ("%d", base_name_nb);
                                label_str =
                                        g_strconcat (base_name,
                                                     "<", tmp_str,
                                                     ">", NULL);

                                if (g_hash_table_lookup
                                    (PRIVATE (a_this)->
                                     opened_document_label_names,
                                     label_str)) {
                                        base_name_nb++;
                                        g_free (tmp_str);
                                        continue;
                                }
                                break;
                        }                        
                        label = gtk_label_new (label_str);
                        g_hash_table_insert
                                (PRIVATE
                                 (a_this)->
                                 opened_document_label_names,
                                 g_strdup (label_str),
                                 a_view) ;
                        g_free (tmp_str);
                        g_free (label_str);
                } else if (file_is_already_opened == FALSE) {
                        label = gtk_label_new (base_name);
                        g_hash_table_insert
                                (PRIVATE
                                 (a_this)->
                                 opened_document_label_names,
                                 g_strdup (base_name),
                                 a_view);
                }
        }

        /*update the view->document index */
        g_hash_table_insert 
                (PRIVATE (a_this)->mlview_xml_doc_views,
                 a_view, mlview_xml_document) ;

        /*update the document->views index */
        views_associated_to_document =
                g_hash_table_lookup 
                (PRIVATE (a_this)->mlview_xml_docs,
                 mlview_xml_document);
        if (!views_associated_to_document) {
                views_associated_to_document =
                        g_hash_table_new (g_direct_hash,
                                          g_direct_equal);
                g_assert (views_associated_to_document != NULL);
                g_hash_table_insert 
                        (PRIVATE (a_this)->mlview_xml_docs,
                         mlview_xml_document,
                         views_associated_to_document);
        }
        g_hash_table_insert (views_associated_to_document,
                             a_view,
                             mlview_xml_document);
        if (is_new_doc_tree == TRUE)
                PRIVATE (a_this)->opened_docs_num++;

        /*now, visually add the view */
        gtk_notebook_append_page (PRIVATE (a_this)->notebook,
                                  view_impl, label);
        g_signal_connect (G_OBJECT (a_view),
                          "name-changed",
                          G_CALLBACK (view_name_changed_cb),
                          a_this);
        label_str = (guchar*)gtk_label_get_text (GTK_LABEL (label)) ;
        if (label_str) {
                mlview_iview_set_name (a_view, label_str) ;
                label_str = NULL ;
        }
        /*make sure the newly added view comes on "front"*/
        gtk_widget_show (view_impl) ;
        gtk_notebook_set_current_page
                (PRIVATE (a_this)->notebook, -1) ;

        if (g_hash_table_size 
            (PRIVATE (a_this)->mlview_xml_doc_views) == 1) {
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[FIRST_VIEW_ADDED], 0,
                               a_view) ;
        }
        gtk_widget_show_all (GTK_WIDGET (a_this));

}

/**
 *removes the document view a_view from the editor.
 *This method does not save the document.
 *
 *@param a_this the current mlview editor.
 *@param a_view the document view to remove from the editor.
 */
void
mlview_editor_remove_xml_document_view (MlViewEditor * a_this,
                                        MlViewIView *a_view)
{
        GtkWidget *label = NULL, *view_impl = NULL ;
        gpointer *ptr = NULL ;
        gint notebook_page_num ;
        MlViewFileDescriptor *file_desc = NULL;
        MlViewXMLDocument *mlview_xml_doc = NULL;
        GHashTable *views_related_to_document = NULL;
        gboolean doc_to_be_closed = FALSE;
        guchar *file_path = NULL,
                *base_name = NULL,
                *label_str = NULL ;
        
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail 
                (PRIVATE (a_this)->mlview_xml_doc_views != NULL);
        g_return_if_fail (a_view != NULL);
        g_return_if_fail (MLVIEW_IS_IVIEW (a_view)) ;
        mlview_iview_get_impl (a_view, &view_impl) ;
        g_return_if_fail (view_impl) ;

        mlview_iview_get_document (a_view, &mlview_xml_doc) ;
        g_return_if_fail (mlview_xml_doc);

        file_desc = mlview_xml_document_get_file_descriptor
                (mlview_xml_doc) ;
        if (file_desc)
                file_path =
                        mlview_file_descriptor_get_file_path
                        (file_desc);
        if (file_path != NULL)
                base_name = (guchar *) g_basename (file_path);       
        label = gtk_notebook_get_tab_label 
                (PRIVATE (a_this)->notebook,
                 GTK_WIDGET (a_view)) ;
        gtk_label_get (GTK_LABEL (label), (char **) &label_str);
        label_str = g_strdup (label_str);
        g_return_if_fail (label != NULL);
        /*
         *check if the document view a_view
         *is opened in this instance of MlViewEditor.
         */
        ptr = g_hash_table_lookup 
                (PRIVATE (a_this)->mlview_xml_doc_views,
                 a_view);
        g_return_if_fail (ptr != NULL);
        /*
         *removes a_view from the hashtable of 
         *the opened document views.
         *and from the opened_file_paths hash_table.
         */
        g_hash_table_remove (PRIVATE (a_this)->mlview_xml_doc_views,
                             a_view);
        views_related_to_document =
                g_hash_table_lookup 
                (PRIVATE (a_this)->mlview_xml_docs,
                 mlview_xml_doc);
        g_return_if_fail (views_related_to_document != NULL);

        ptr = g_hash_table_lookup (views_related_to_document, 
                                   a_view) ;
        g_return_if_fail (ptr != NULL) ;
        g_hash_table_remove (views_related_to_document,
                             a_view) ;
        /*removes the notebook page that contains the view */
        notebook_page_num = gtk_notebook_page_num
                (PRIVATE (a_this)->notebook, view_impl);

        g_return_if_fail (notebook_page_num != -1);

        gtk_notebook_remove_page (PRIVATE (a_this)->notebook,
                                  notebook_page_num);

        if (g_hash_table_size (views_related_to_document) == 0) {
                /*
                 *no views are opened on the current 
                 *doc anymore=>close the doc
                 */
                g_hash_table_remove (PRIVATE (a_this)->
                                     mlview_xml_docs,
                                     mlview_xml_doc);

                if (file_path != NULL) {
                        g_hash_table_remove
                                (PRIVATE (a_this)->
                                 opened_file_paths, file_path);
                }

                doc_to_be_closed = TRUE;

                PRIVATE (a_this)->opened_docs_num--;
        }

        if (doc_to_be_closed && label_str) {
                /*
                 *remove the entry in the 
                 *opened_document_label_names hash table
                 */
                g_hash_table_remove
                        (PRIVATE (a_this)->
                         opened_document_label_names, label_str);

                g_free (label_str);
                label_str = NULL;
        }

        /*if there are several docs that have the save base name as this one,
         *decrement their number, and if the number reaches 0, 
         *remove the entry matching this base name
         *from the hash table.
         */
        if (doc_to_be_closed == TRUE && file_path != NULL) {
                gint tmp_int;

                ptr = g_hash_table_lookup
                        (PRIVATE (a_this)->
                         opened_file_base_names, base_name);

                tmp_int = GPOINTER_TO_INT (ptr);
                tmp_int--;

                if (tmp_int == 0) {
                        g_hash_table_remove
                                (PRIVATE (a_this)->
                                 opened_file_base_names,
                                 base_name);
                } else {
                        ptr = GINT_TO_POINTER (tmp_int);
                        g_hash_table_insert
                                (PRIVATE (a_this)->
                                 opened_file_base_names,
                                 base_name, ptr);
                }
        } else if (doc_to_be_closed == TRUE && !file_path) {
                PRIVATE (a_this)->untitled_docs_num--;
        }

        if (doc_to_be_closed && mlview_xml_doc) {
                mlview_xml_document_unref (mlview_xml_doc);
                mlview_xml_doc = NULL;
        }
        if (g_hash_table_size
            (PRIVATE (a_this)->mlview_xml_doc_views) == 0) {
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[LAST_VIEW_REMOVED], 0) ;
        }
}

/**
 *Displays a dialog with an entry for the root element name.
 *It's up to the caller check the validity of the name.
 *
 *@param elname the location to return the name string.
 *@return TRUE if the user pressed OK, FALSE otherwise
 */
static gboolean
mlview_editor_ask_root_element_name (gchar **elname)
{
        GtkWidget *dialog;
        GtkWidget *hbox;
        GtkWidget *image;
        GtkWidget *label;
        GtkWidget *entry;
        gint res;
        gboolean ret = FALSE;

        g_return_val_if_fail (elname, FALSE);

        *elname = NULL;

        dialog = gtk_dialog_new_with_buttons 
                (_("Name of the root element"),
                 NULL, GTK_DIALOG_MODAL,
                 GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
                 GTK_STOCK_OK, GTK_RESPONSE_OK,
                 NULL);

        gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
        gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);

        hbox = gtk_hbox_new (FALSE, 6);
        image = gtk_image_new_from_stock (GTK_STOCK_DIALOG_QUESTION,
                                          GTK_ICON_SIZE_DIALOG);
        gtk_misc_set_alignment (GTK_MISC (image), 0.5, 0.0);
        label = gtk_label_new (_("Root element name:"));
        entry = gtk_entry_new ();
        gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);

        gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
                            hbox, FALSE, FALSE, 0);

        gtk_widget_show_all (dialog);

        res = gtk_dialog_run (GTK_DIALOG (dialog));
        switch (res) {
        case GTK_RESPONSE_OK:
                *elname = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));
                ret = TRUE;
                break;
        case GTK_RESPONSE_CANCEL:
        case GTK_RESPONSE_DELETE_EVENT:
		break;
        default:
                g_assert_not_reached ();
	}

        gtk_widget_destroy (dialog);
        return ret;
}

/**
 *Interactively create a new document.
 *Asks the user for info like 
 *root element, dtd (if validation is switched on) ...etc.
 *
 *@param a_this the current mlview editor.
 */
void
mlview_editor_create_new_xml_document (MlViewEditor * a_this)
{
        struct MlViewAppSettings *settings = NULL;
        MlViewXMLDocument *mlview_doc = NULL;
        xmlDocPtr xml_doc = NULL;
        xmlNodePtr xml_node = NULL;
        MlViewIView *view = NULL ;
        struct MlViewViewDesc *view_desc_ptr = NULL ;

        gboolean loop = TRUE;
        gchar *utf8_elname = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (PRIVATE (a_this)->app_context != NULL);

        settings = mlview_app_context_get_settings 
                (PRIVATE (a_this)->app_context);
        g_return_if_fail (settings);

        view_desc_ptr = 
                mlview_editor_peek_editing_view_descriptor 
                (settings->general.default_editing_view_type);
        g_return_if_fail (view_desc_ptr) ;

        while (loop) {
                gboolean res = FALSE;
                enum MlViewStatus status = MLVIEW_OK;
                guchar *name_end = NULL;

                res = mlview_editor_ask_root_element_name 
                        (&utf8_elname);

                /*the user hit cancel */
                if (!res)
                        break;
                /*empty root name: show the dialog again */
                if (!utf8_elname)
                        continue;

                if (mlview_utils_is_white_string (utf8_elname) == TRUE)
                        continue;

                /*check if is a valid element name */
                status = mlview_utils_parse_element_name (utf8_elname,
                                                          &name_end);
                if (status != MLVIEW_OK || !name_end) {
                        mlview_app_context_error (PRIVATE (a_this)->app_context,
                                _("The string entered is not a well formed element name!"));
                        continue;
                }

                xml_node = xmlNewNode (NULL, utf8_elname);
                xml_doc = xmlNewDoc ("1.0");
                xml_doc->name = g_strdup ("untitled");
                xmlDocSetRootElement (xml_doc, xml_node);

                mlview_doc = mlview_xml_document_new (xml_doc,
                                                      PRIVATE (a_this)->app_context);
                g_return_if_fail (mlview_doc != NULL);

                /*validation */
                if (settings->general.validation_is_on == TRUE) {
                        mlview_xml_document_associate_dtd_interactive
                                       (mlview_doc);
                        if (xml_node->type == XML_ELEMENT_NODE) {
                                mlview_parsing_utils_build_required_attributes_list
                                                (PRIVATE (a_this)->app_context,
                                                 xml_node);

                                mlview_parsing_utils_build_required_children_tree
                                                (PRIVATE
                                                 (a_this)->app_context,
                                                 &xml_node);
                        }
                }
                view = MLVIEW_IVIEW 
                        (view_desc_ptr->view_constructor
                         (mlview_doc, (gchar*)"",
                          PRIVATE (a_this)->app_context)) ;
                if (!view)
                {
                        mlview_utils_trace_info ("view instanciation failed") ;
                        return ;
                }
                mlview_editor_add_xml_document_view
                        (a_this, view);
                /*everything went fine */
                break;
        }
        if (utf8_elname) {
                g_free (utf8_elname);
                utf8_elname = NULL ;
        }
}


/**
 *Edits the xml document given in argument.
 *Actually, this method creates a view on this document and
 *adds the view to the editor.
 *@param a_this the current mlview editor.
 *@param a_doc the xml document to edit. This is an
 *object of the libxml2
 *@param a_doc_name the name of the xml document to edit.
 */
void
mlview_editor_edit_xml_document (MlViewEditor * a_this,
                                 xmlDocPtr a_doc,
                                 gchar * a_doc_name)
{
        MlViewIView *doc_view = NULL ;
        MlViewXMLDocument *mlview_xml_doc = NULL;
        struct MlViewViewDesc *view_desc_ptr = NULL ;
        struct MlViewAppSettings *settings = NULL ;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (a_doc != NULL);

        settings = mlview_app_context_get_settings 
                (PRIVATE (a_this)->app_context) ;
        g_return_if_fail (settings) ;
        view_desc_ptr = 
                mlview_editor_peek_editing_view_descriptor 
                (settings->general.default_editing_view_type);
        g_return_if_fail (view_desc_ptr) ;

        mlview_xml_doc =
                mlview_xml_document_new
                (a_doc, PRIVATE (a_this)->app_context) ;

        doc_view = MLVIEW_IVIEW 
                        (view_desc_ptr->view_constructor
                        (mlview_xml_doc, a_doc_name,
                         PRIVATE (a_this)->app_context)) ;
        mlview_editor_add_xml_document_view (a_this, doc_view) ;
}

/**
 *Execute an editing editing action.
 *@param a_this the current instance of #MlViewEditor
 *@param a_action the action to execute.
 *@return MLVIEW_OK upon successful completion an error code
 *otherwise.
 */
enum MlViewStatus
mlview_editor_execute_action (MlViewEditor *a_this,
                              MlViewAction *a_action)
{
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_EDITOR (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        if (PRIVATE (a_this)->cur_view) {
                status = mlview_iview_execute_action
                        (PRIVATE (a_this)->cur_view,
                         a_action) ;
        }
        return status ;
}


/**
 *Getter of the file path of the document 
 *associated to the currently selected view.
 *@param a_this the current mlview editor.
 *@return the file path.
 */
gchar *
mlview_editor_get_current_xml_doc_file_path (MlViewEditor *a_this)
{
        MlViewXMLDocument *doc = NULL;

        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_EDITOR (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);

        mlview_iview_get_document
                (PRIVATE (a_this)->cur_view, &doc);
        g_return_val_if_fail (doc != NULL, NULL);

        return mlview_xml_document_get_file_path (doc);
}


/**
 *Interactively saves the underlying xml document of the 
 *currently selected document view.
 *If the document has an associated file name, 
 *save it in that file or else, asks the user where to
 *save it.
 *@param a_this the current mlview editor.
 */
void
mlview_editor_save_xml_document (MlViewEditor * a_this)
{
        MlViewXMLDocument *xml_doc = NULL;
        gchar *file_path = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        if (!PRIVATE (a_this)->cur_view)
                return;

        mlview_iview_get_document (PRIVATE (a_this)->cur_view, &xml_doc) ;
        if (xml_doc == NULL)
                return;

        file_path =
                mlview_editor_get_current_xml_doc_file_path
                (a_this);

        if (file_path == NULL)
                mlview_editor_save_xml_document_as_interactive
                        (a_this);
        else
                mlview_editor_save_xml_document_as (a_this,
                                                    file_path);

}


/**
 *Classical "save as" functionnality.
 *Saves the underlying document of the currently selected view
 *into the file denoted by the file path a_file_path.
 *
 *@param a_this the current mlview editor.
 *@param a_file_path file path where to save the document.
 */
void
mlview_editor_save_xml_document_as (MlViewEditor * a_this,
                                    gchar * a_file_path)
{
        MlViewXMLDocument *mlview_xml_document = NULL;
        gboolean file_was_untitled = FALSE;
        guchar *prev_file_path = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        if (!PRIVATE (a_this)->cur_view)
                return;
        g_return_if_fail (PRIVATE (a_this)->opened_file_paths);
        g_return_if_fail (a_file_path != NULL);

        mlview_iview_get_document (PRIVATE (a_this)->cur_view,
                                   &mlview_xml_document) ;
        mlview_app_context_sbar_push_message
                (PRIVATE (a_this)->app_context,
                 _("Saving xml document as file %s..."),
                 a_file_path);

        file_was_untitled =
                (mlview_xml_document_get_file_descriptor (mlview_xml_document)
                 == NULL);

        prev_file_path =
                mlview_xml_document_get_file_path
                (mlview_xml_document);

        /*really save the document now */
        if (mlview_xml_document_save (mlview_xml_document,
                                      a_file_path, TRUE) > 0
            && (!prev_file_path
                || strcmp (prev_file_path, a_file_path))) {
                guchar *new_file_path = NULL;

                /*
                 *The save was OK and the new file path of this doc
                 * is different from the previous file path of this doc.
                 */

                /*
                 *remove the reference to the previous file path 
                 *of this document.
                 */
                if (prev_file_path) {
                        g_hash_table_remove
                                (PRIVATE (a_this)->
                                 opened_file_paths,
                                 prev_file_path);
                }

                /*
                 *Now, reference the new path of this doc.
                 *We must make sure that the string we put in the
                 *hash table belongs to mlview_xml_document.
                 *That way, one can destroy the hash table without
                 *any fear to leak the memory hold by the string.
                 */
                new_file_path = mlview_xml_document_get_file_path
                        (mlview_xml_document);

                if (new_file_path)
                        g_hash_table_insert (PRIVATE (a_this)->
                                             opened_file_paths,
                                             new_file_path,
                                             PRIVATE (a_this)->
                                             cur_view);
        }

        mlview_app_context_sbar_pop_message (PRIVATE (a_this)->
                                             app_context);
}


/**
 *Interactively saves the underlying document of the currently
 *selected view. Graphically asks the user where to save the
 *document and saves it.
 *
 *@param a_this the current mlview editor.
 */
void
mlview_editor_save_xml_document_as_interactive (MlViewEditor *a_this)
{
        MlViewFileSelection *file_sel;
        enum MLVIEW_SELECTED_BUTTON button;
        gchar *file_name = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (PRIVATE (a_this)->app_context !=
                          NULL);

        file_sel =
                mlview_app_context_get_file_selector
                (PRIVATE (a_this)->app_context,
                 _("Choose a xml document"));

        g_return_if_fail (file_sel);

        button = mlview_file_selection_run (file_sel, TRUE);

        switch (button) {
        case OK_BUTTON:
                file_name =
                        g_strdup (gtk_file_selection_get_filename
                                  (GTK_FILE_SELECTION
                                   (file_sel)));

                if (file_name && strcmp (file_name, "")) {
                        mlview_editor_save_xml_document_as
                                (a_this, file_name);
                }

                if (file_name) {
                        g_free (file_name);
                        file_name = NULL;
                }
                break;
        case CANCEL_BUTTON:
        case WINDOW_CLOSED:
        default:
                break;
        }
}

/**
 *closes the current view without saving the underlying document.
 *@param a_this the current mlview editor.
 */
void
mlview_editor_close_xml_document_without_saving (MlViewEditor *a_this)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        mlview_editor_remove_xml_document_view
                (a_this,
                 PRIVATE (a_this)->cur_view);

        if (g_hash_table_size
            (PRIVATE (a_this)->mlview_xml_docs) == 0)
                PRIVATE (a_this)->cur_view = NULL;
}


/**
 *Saves the underlying document of the currently selected view
 *and closes the view.
 *
 *@param a_this the current mlview editor.
 */
void
mlview_editor_save_and_close_xml_document (MlViewEditor *a_this)
{
        MlViewFileDescriptor *file_desc = NULL ;
        MlViewFileSelection *file_sel = NULL;
        MlViewXMLDocument *mlview_xml_document = NULL;
        enum MLVIEW_SELECTED_BUTTON button;
        gchar *file_name = NULL, *tmp_str = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        if (!PRIVATE (a_this)->cur_view)
                return;
        g_return_if_fail (PRIVATE (a_this)->app_context !=
                          NULL);
        
        mlview_iview_get_document (PRIVATE (a_this)->cur_view,
                                   &mlview_xml_document) ;
        g_return_if_fail (mlview_xml_document != NULL);

        file_desc = mlview_xml_document_get_file_descriptor 
                (mlview_xml_document) ;
        if (!file_desc) {
                /*
                 *No file is associated to the document.
                 *=>Ask the user which file she wants to save
                 *the document in.
                 */
                file_sel = mlview_app_context_get_file_selector
                        (PRIVATE (a_this)->app_context,
                         _("Choose a xml document"));
                g_return_if_fail (file_sel);

                button = mlview_file_selection_run (file_sel, TRUE);

                switch (button) {
                case OK_BUTTON:
                        file_name =
                                g_strdup (gtk_file_selection_get_filename
                                          (GTK_FILE_SELECTION
                                           (file_sel))) ;
                        break;
                case CANCEL_BUTTON:
                case WINDOW_CLOSED:
                        break;
                }
        } else {
                /*The document already has*/
                tmp_str = mlview_file_descriptor_get_file_path (file_desc);
                g_return_if_fail (tmp_str);
                file_name = g_strdup (tmp_str);
        }
        if (file_name && strcmp (file_name, "")) {
                mlview_xml_document_save
                        (mlview_xml_document, file_name,
                         TRUE);
                mlview_editor_close_xml_document_without_saving
                        (a_this);
        }
        if (file_name) {
                g_free (file_name);
                file_name = NULL;
        }
}


/**
 *Closes all the views (and their underlying documents).
 *For document that needs to be saved, this function 
 *popups a dialog asking the user if she wants
 *to save the document before closing, closing without saving
 *or cancel. If she cancels, then the function just returns FALSE
 *and the processing is stopped.
 *
 *@param a_this the current mlview editor.
 *@return TRUE if all the document views have been closed, FALSE otherwise.
 */
gboolean
mlview_editor_close_all_xml_documents_interactive (MlViewEditor *a_this)
{
        GList *views = NULL,
                *mobile_view_ptr = NULL;

        g_return_val_if_fail (a_this != NULL, FALSE);
        g_return_val_if_fail (MLVIEW_EDITOR (a_this), FALSE);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, FALSE);
        g_return_val_if_fail (PRIVATE (a_this)->
                          mlview_xml_doc_views != NULL, FALSE);

        views = build_view_list_from_hashtable
                (PRIVATE (a_this)->mlview_xml_doc_views);

        if (views == NULL)
                return TRUE ;

        for (mobile_view_ptr = views;
             mobile_view_ptr;
             mobile_view_ptr = mobile_view_ptr->next) {
                PRIVATE (a_this)->cur_view =
                        (MlViewIView *) mobile_view_ptr->data;
                mlview_editor_close_xml_document_interactive (a_this);
        }

        if (g_list_length (mlview_editor_get_list_open_doc (a_this)) == 0)
                return TRUE ;
        else
                return FALSE ;
}


/**
 *Shows a confirmation dialog before closing a modified document.
 *
 *@param a_this the current mlview editor.
 */
static void
mlview_editor_confirm_close (MlViewEditor *a_this)
{
        enum MlViewStatus status = MLVIEW_OK ;
        GtkWidget *dialog;
        gint ret;
        guchar *a_name = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this));

        status = mlview_iview_get_name(PRIVATE (a_this)->cur_view, &a_name) ;

        g_return_if_fail (status == MLVIEW_OK && a_name) ;

        dialog = gtk_message_dialog_new (NULL,
                                         GTK_DIALOG_MODAL,
                                         GTK_MESSAGE_QUESTION,
                                         GTK_BUTTONS_NONE,
                                         _("The document \"%s\" has been modifed.\n"
					   "Should I save it before closing it?"), 
                                         a_name);

        gtk_dialog_add_buttons (GTK_DIALOG (dialog),
                                _("_Close without Saving"), GTK_RESPONSE_NO,
                                GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                GTK_STOCK_SAVE, GTK_RESPONSE_YES, NULL);

        gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_YES);

        ret = gtk_dialog_run (GTK_DIALOG (dialog));
        switch (ret) {
        case GTK_RESPONSE_YES:
                mlview_editor_save_and_close_xml_document (a_this);
                break;
        case GTK_RESPONSE_NO:
                mlview_editor_close_xml_document_without_saving (a_this);
                break;
        case GTK_RESPONSE_CANCEL:
        case GTK_RESPONSE_DELETE_EVENT:
                break ;
        default:
                g_assert_not_reached ();
        }

        gtk_widget_destroy (dialog);
}


/**
 *Interactively closes the current document view.
 *Saves the underlying document if needed.
 *
 *@param a_this the current mlview editor.
 */
void
mlview_editor_close_xml_document_interactive (MlViewEditor *a_this)
{
        MlViewXMLDocument *doc = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this));

        if (!PRIVATE (a_this)->cur_view)
                return ;
        mlview_iview_get_document (PRIVATE (a_this)->cur_view,
                                   &doc) ;
        if (!doc) {
                mlview_utils_trace_info
                        ("The current view has no associated document. "
                         "This is truly weird, something bad is happening.") ;
                return ;
        }

        if (mlview_xml_document_needs_saving (doc) == FALSE)
                mlview_editor_close_xml_document_without_saving (a_this);
        else
                mlview_editor_confirm_close (a_this);
}

/**
 *Interactively edits the editor settings.
 *
 *@param a_this the current mlview editor.
 */
void
mlview_editor_edit_settings_interactive (MlViewEditor * a_this)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

}


/**
 *Associates a dtd to the current mlview xml document. 
 *
 *@param a_this the current mlview editor.
 *@return 0 if association worked, 1 if not xml document is opened, 
 */
gint
mlview_editor_associate_dtd_interactive (MlViewEditor * a_this)
{

        MlViewXMLDocument *mlview_xml_doc = NULL;

        g_return_val_if_fail (a_this != NULL, -1);
        g_return_val_if_fail (MLVIEW_IS_EDITOR (a_this), -1);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, -1);

        if (!PRIVATE (a_this)->cur_view)
                return 1;

        mlview_iview_get_document (PRIVATE (a_this)->cur_view,
                                   &mlview_xml_doc) ;
        if (mlview_xml_doc == NULL)
                return 1;

        if (mlview_xml_document_associate_dtd_interactive
            (mlview_xml_doc))
                return 0;

        return 2;
}

/**
 *validate the current selected document against the dtd it is associated to.
 *If there is no dtd associated to the document, the functions asks the user
 *to associate a dtd to it and then, does the validation 
 *
 *@param a_this the current mlview editor.
 */
gint
mlview_editor_validate (MlViewEditor * a_this)
{

        MlViewXMLDocument *doc = NULL;

        g_return_val_if_fail (a_this != NULL, -1);
        g_return_val_if_fail (MLVIEW_IS_EDITOR (a_this), -1);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, -1);

        if (!PRIVATE (a_this)->cur_view)
                return 1;

        mlview_iview_get_document (PRIVATE (a_this)->cur_view,
                                   &doc) ;

        if (doc == NULL)
                return 1;

        return mlview_xml_document_validate (doc);
}


/**
 *Applies a XSLT stylesheet to the current document
 *displaying a dialog for xslt selection and opens
 *the transformation resulting document in the editor
 *@param a_this the current instance of #MlViewEditor.
 *
 */
void
mlview_editor_xslt_transform_document_interactive (MlViewEditor *a_this)
{
        MlViewXMLDocument *src_doc = NULL;
        MlViewXMLDocument *xsl_doc = NULL;
        MlViewXMLDocument *res_doc = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        
        src_doc = mlview_editor_get_cur_doc (a_this);
        xsl_doc = mlview_xslt_utils_select_xsl_doc (a_this);
        if (xsl_doc != NULL) {
                res_doc = mlview_xslt_utils_transform_document
                        (src_doc, xsl_doc);
                mlview_xml_document_unref(xsl_doc);
                if (res_doc != NULL) 
                        mlview_editor_create_new_view_on_document 
                                (a_this, res_doc);
        }
}

/**
 *Returns the descriptor of an editing view
 *denoted by it's view type name.
 *@param a_view_name the view name to test
 *@return a pointer to the view descriptor.
 *This pointer must *NOT* be freed by the caller.
 */
struct MlViewViewDesc *
mlview_editor_peek_editing_view_descriptor (gchar *a_view_type_name)
{
        struct MlViewViewDesc *view_desc_ptr = NULL ;

        for (view_desc_ptr = gv_view_types; 
             view_desc_ptr && view_desc_ptr->view_type_name;
             view_desc_ptr ++) {
                if (view_desc_ptr->view_type_name
                    && a_view_type_name
                    && !strcmp (view_desc_ptr->view_type_name,
                                a_view_type_name)) {
                        return view_desc_ptr ;
                }
        }
        return NULL ;
}

/**
 *Getter of the number of view descriptors available
 *in the system.
 *@return the number of view descriptor available in the system.
 */
guint
mlview_editor_get_number_of_view_desc (void)
{
        gint result = 0 ;
        struct MlViewViewDesc *cur_view_desc = NULL ;

        for (cur_view_desc = gv_view_types ; 
             cur_view_desc && cur_view_desc->view_type_name;
             cur_view_desc ++)
        {
                result ++ ;
        }
        return result ;
}

/**
 *Get a view descriptor from the table of view descriptors.
 *@param a_offset the offset of the view descriptor to get.
 *The first view descriptor in the table has an offset of zero.
 *@param the offset of the view descriptor to get
 *@return the found view descriptor, or NULL if no view descriptor is
 *found.
 */
struct MlViewViewDesc *
mlview_editor_get_view_descriptor_at (guint a_offset)
{
        guint nr_views = 0 ;

        nr_views = mlview_editor_get_number_of_view_desc () ;
        if (a_offset >= nr_views)
                return NULL ;
        return &gv_view_types[a_offset] ;
}
