/* -*- Mode: C; indent-tabs-mode:nil; c-basic-offset: 8-*- */

/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include <string.h>
#include "mlview-completion-table.h"

/**
 *@file
 *The definition of the #MlViewCompletionTable class.
 */

#define PRIVATE(completion_table) ((completion_table)->priv)

struct _MlViewCompletionTablePrivate {
        struct {
                GtkListStore *feasible_children;
                GtkListStore *feasible_prev_siblings;
                GtkListStore *feasible_next_siblings;
                GtkListStore *feasible_attributes;
        } models;
        
        struct {
                GtkTreeSelection *feasible_children;
                GtkTreeSelection *feasible_prev_siblings;
                GtkTreeSelection *feasible_next_siblings;
                GtkTreeSelection *feasible_attributes;
        } selections;
        
        struct {
                GtkWidget *feasible_children;
                GtkWidget *feasible_prev_siblings;
                GtkWidget *feasible_next_siblings;
                GtkWidget *feasible_attributes;
        } widgets;

        MlViewXMLDocument *xml_doc;
        xmlNode *cur_node;

        gboolean dispose_has_run;
};

static GtkTable *gv_parent_class;

static void
mlview_completion_table_finalize (GObject *a_this)
{
        MlViewCompletionTable *table;
        
        g_return_if_fail (a_this && MLVIEW_COMPLETION_TABLE (a_this));
        table = MLVIEW_COMPLETION_TABLE (a_this);
        g_return_if_fail (PRIVATE (table));
        g_free (PRIVATE (table));
        PRIVATE (table) = NULL;
        
        if (gv_parent_class 
            && G_OBJECT_CLASS (gv_parent_class)->finalize) {
                G_OBJECT_CLASS (gv_parent_class)->finalize (a_this) ;
        }
}

static void
mlview_completion_table_dispose (GObject *a_this)
{
        MlViewCompletionTable *table = NULL;
        
        g_return_if_fail (a_this && MLVIEW_COMPLETION_TABLE (a_this));
        table = MLVIEW_COMPLETION_TABLE (a_this);
        g_return_if_fail (PRIVATE (table));
        
        if (PRIVATE (table)->dispose_has_run == TRUE) 
                return;
        
        PRIVATE (table)->dispose_has_run = TRUE;
        
        if (gv_parent_class 
            && G_OBJECT_CLASS (gv_parent_class)->dispose)
                G_OBJECT_CLASS (gv_parent_class)->dispose (a_this);
}

static void
mlview_completion_table_class_init (MlViewCompletionTableClass *a_klass)
{
        GObjectClass *gobject_class = NULL;
        
        gobject_class = G_OBJECT_CLASS (a_klass);
        g_return_if_fail (gobject_class);
        
        gobject_class->dispose = mlview_completion_table_dispose;
        gobject_class->finalize = mlview_completion_table_finalize;
        
        gv_parent_class =
                gtk_type_class (GTK_TYPE_TABLE);
}

static void
update_list_store (GtkListStore *a_store, GList *a_list)
{
        GtkTreeIter iter = {0} ;

        gtk_list_store_clear (a_store);
        while (a_list) {
                gtk_list_store_append (a_store, &iter);
                gtk_list_store_set (a_store, &iter, 0, a_list->data, -1);
                a_list = g_list_next (a_list);
        }
}


static void
feasible_attribute_selected_cb (GtkTreeSelection *a_tree_selection,
                                gpointer a_user_data)
{
        MlViewCompletionTable *widget;
        GtkTreeIter iter;
        gchar *str = NULL;
        xmlAttr *attr;

        g_return_if_fail (a_tree_selection && GTK_IS_TREE_SELECTION (a_tree_selection));
        g_return_if_fail (a_user_data && MLVIEW_IS_COMPLETION_TABLE (a_user_data));

        widget = MLVIEW_COMPLETION_TABLE (a_user_data);
        
        g_return_if_fail (PRIVATE (widget));
        g_return_if_fail (PRIVATE (widget)->cur_node);

        if (gtk_tree_selection_get_selected (a_tree_selection, NULL, &iter)) {
                gtk_tree_selection_unselect_all (PRIVATE (widget)->selections.feasible_children);
                gtk_tree_selection_unselect_all (PRIVATE (widget)->selections.feasible_prev_siblings);
                gtk_tree_selection_unselect_all (PRIVATE (widget)->selections.feasible_next_siblings);

                gtk_tree_model_get 
                        (GTK_TREE_MODEL (PRIVATE (widget)->models.feasible_attributes), 
                         &iter, 0, &str, -1);
                
                attr = mlview_xml_document_set_attribute (PRIVATE (widget)->xml_doc,
                                                          PRIVATE (widget)->cur_node,
                                                          str, "value", TRUE);

                g_return_if_fail (attr);
        }
}

static void
feasible_next_sibling_selected_cb (GtkTreeSelection *a_tree_selection,
                                   gpointer a_user_data)
{
        MlViewCompletionTable *widget;
        GtkTreeIter iter;
        gchar *str = NULL;
        xmlNode *new_node = NULL,
                *added_node = NULL;

        g_return_if_fail (a_tree_selection && GTK_IS_TREE_SELECTION (a_tree_selection));
        g_return_if_fail (a_user_data && MLVIEW_IS_COMPLETION_TABLE (a_user_data));

        widget = MLVIEW_COMPLETION_TABLE (a_user_data);
        
        g_return_if_fail (PRIVATE (widget));
        g_return_if_fail (PRIVATE (widget)->cur_node);

        if (gtk_tree_selection_get_selected (a_tree_selection, NULL, &iter)) {
                gtk_tree_selection_unselect_all (PRIVATE (widget)->selections.feasible_children);
                gtk_tree_selection_unselect_all (PRIVATE (widget)->selections.feasible_prev_siblings);
                gtk_tree_selection_unselect_all (PRIVATE (widget)->selections.feasible_attributes);

                gtk_tree_model_get 
                        (GTK_TREE_MODEL (PRIVATE (widget)->models.feasible_next_siblings), 
                         &iter, 0, &str, -1);
                
                if (strcmp (str, "#PCDATA") == 0) {
                        new_node = xmlNewNode (NULL, "text");
                        new_node->type = XML_TEXT_NODE;
                } else 
                        new_node = xmlNewNode (NULL, str);
                
                added_node = mlview_xml_document_insert_next_sibling_node
                        (PRIVATE (widget)->xml_doc, PRIVATE (widget)->cur_node,
                         new_node, TRUE, TRUE);
                
                g_return_if_fail (added_node);
        }
}

static void
feasible_prev_sibling_selected_cb (GtkTreeSelection *a_tree_selection,
                                   gpointer a_user_data)
{
        MlViewCompletionTable *widget;
        GtkTreeIter iter;
        gchar *str = NULL;
        xmlNode *new_node = NULL,
                *added_node = NULL;

        g_return_if_fail (a_tree_selection && GTK_IS_TREE_SELECTION (a_tree_selection));
        g_return_if_fail (a_user_data && MLVIEW_IS_COMPLETION_TABLE (a_user_data));

        widget = MLVIEW_COMPLETION_TABLE (a_user_data);
        
        g_return_if_fail (PRIVATE (widget));
        g_return_if_fail (PRIVATE (widget)->cur_node);

        if (gtk_tree_selection_get_selected (a_tree_selection, NULL, &iter)) {
                gtk_tree_selection_unselect_all (PRIVATE (widget)->selections.feasible_children);
                gtk_tree_selection_unselect_all (PRIVATE (widget)->selections.feasible_next_siblings);
                gtk_tree_selection_unselect_all (PRIVATE (widget)->selections.feasible_attributes);

                gtk_tree_model_get 
                        (GTK_TREE_MODEL (PRIVATE (widget)->models.feasible_prev_siblings), 
                         &iter, 0, &str, -1);
                
                if (strcmp (str, "#PCDATA") == 0) {
                        new_node = xmlNewNode (NULL, "text");
                        new_node->type = XML_TEXT_NODE;
                } else 
                        new_node = xmlNewNode (NULL, str);
                
                added_node = mlview_xml_document_insert_prev_sibling_node
                        (PRIVATE (widget)->xml_doc, PRIVATE (widget)->cur_node,
                         new_node, TRUE, TRUE);
                
                g_return_if_fail (added_node);
        }
}

static void        
feasible_child_selected_cb (GtkTreeSelection *a_tree_selection,
                            gpointer a_user_data)
{
        MlViewCompletionTable *widget;
        GtkTreeIter iter;
        gchar *str = NULL;
        xmlNode *new_node = NULL,
                *added_node = NULL;

        g_return_if_fail (a_tree_selection && GTK_IS_TREE_SELECTION (a_tree_selection));
        g_return_if_fail (a_user_data && MLVIEW_IS_COMPLETION_TABLE (a_user_data));

        widget = MLVIEW_COMPLETION_TABLE (a_user_data);
        
        g_return_if_fail (PRIVATE (widget));
        g_return_if_fail (PRIVATE (widget)->cur_node);

        if (gtk_tree_selection_get_selected (a_tree_selection, NULL, &iter)) {
                gtk_tree_selection_unselect_all (PRIVATE (widget)->selections.feasible_prev_siblings);
                gtk_tree_selection_unselect_all (PRIVATE (widget)->selections.feasible_next_siblings);
                gtk_tree_selection_unselect_all (PRIVATE (widget)->selections.feasible_attributes);

                gtk_tree_model_get 
                        (GTK_TREE_MODEL (PRIVATE (widget)->models.feasible_children), 
                         &iter, 0, &str, -1);

                if (strcmp (str, "#PCDATA") == 0) {
                        new_node = xmlNewNode (NULL, "text");
                        new_node->type = XML_TEXT_NODE;
                } else 
                        new_node = xmlNewNode (NULL, str);
                
                added_node = mlview_xml_document_add_child_node
                        (PRIVATE (widget)->xml_doc, PRIVATE (widget)->cur_node,
                         new_node, TRUE, TRUE);
                
                g_return_if_fail (added_node);
        }
}

static void
mlview_completion_table_init (MlViewCompletionTable *a_this)
{
        GtkWidget *scrolled = NULL;
        GtkCellRenderer *renderer = NULL;
        GtkTreeSelection *selection = NULL;

        gtk_table_set_col_spacings (GTK_TABLE (a_this), 3);
        gtk_table_set_row_spacings (GTK_TABLE (a_this), 3);
        
        if (PRIVATE (a_this) == NULL) {
                PRIVATE (a_this) = g_try_malloc (sizeof (MlViewCompletionTablePrivate));
                g_return_if_fail (PRIVATE (a_this));
        }

        memset (PRIVATE (a_this), 0, sizeof (MlViewCompletionTablePrivate));
        
        PRIVATE (a_this)->models.feasible_children = 
                gtk_list_store_new (1, G_TYPE_STRING);
        PRIVATE (a_this)->widgets.feasible_children = 
                gtk_tree_view_new_with_model 
                (GTK_TREE_MODEL (PRIVATE (a_this)->models.feasible_children));
        renderer = gtk_cell_renderer_text_new ();
        gtk_tree_view_insert_column_with_attributes 
                (GTK_TREE_VIEW
                 (PRIVATE (a_this)->widgets.feasible_children), 1,
                 _("Possible children"), renderer, "text", 0, NULL);
        selection = gtk_tree_view_get_selection 
                (GTK_TREE_VIEW 
                 (PRIVATE (a_this)->widgets.feasible_children));
        g_signal_connect (G_OBJECT (selection), "changed",
                          G_CALLBACK (feasible_child_selected_cb),
                          a_this);
        PRIVATE (a_this)->selections.feasible_children = selection;

        PRIVATE (a_this)->models.feasible_prev_siblings = 
                gtk_list_store_new (1, G_TYPE_STRING);
        PRIVATE (a_this)->widgets.feasible_prev_siblings = 
                gtk_tree_view_new_with_model 
                (GTK_TREE_MODEL (PRIVATE (a_this)->models.feasible_prev_siblings));
        renderer = gtk_cell_renderer_text_new ();
        gtk_tree_view_insert_column_with_attributes 
                (GTK_TREE_VIEW 
                 (PRIVATE (a_this)->widgets.feasible_prev_siblings), 
                 1,
                 _("Possible previous siblings"), 
                 renderer, "text", 0, 
                 NULL);
        selection = gtk_tree_view_get_selection 
                (GTK_TREE_VIEW 
                 (PRIVATE (a_this)->widgets.feasible_prev_siblings));
        g_signal_connect 
                (G_OBJECT (selection), "changed",
                 G_CALLBACK (feasible_prev_sibling_selected_cb),
                 a_this);
        PRIVATE (a_this)->selections.feasible_prev_siblings = selection;
        
        PRIVATE (a_this)->models.feasible_next_siblings = gtk_list_store_new (1, G_TYPE_STRING);
        PRIVATE (a_this)->widgets.feasible_next_siblings = gtk_tree_view_new_with_model 
                (GTK_TREE_MODEL (PRIVATE (a_this)->models.feasible_next_siblings));
        renderer = gtk_cell_renderer_text_new ();
        gtk_tree_view_insert_column_with_attributes 
                (GTK_TREE_VIEW 
                 (PRIVATE (a_this)->widgets.feasible_next_siblings), 
                 1,
                 _("Possible next siblings"), 
                 renderer, 
                 "text", 0, NULL);
        selection = gtk_tree_view_get_selection 
                (GTK_TREE_VIEW 
                 (PRIVATE (a_this)->widgets.feasible_next_siblings));
        g_signal_connect (G_OBJECT (selection), "changed",
                          G_CALLBACK (feasible_next_sibling_selected_cb),
                          a_this);
        PRIVATE (a_this)->selections.feasible_next_siblings = 
                selection;
        
        PRIVATE (a_this)->models.feasible_attributes = 
                gtk_list_store_new (1, G_TYPE_STRING);
        PRIVATE (a_this)->widgets.feasible_attributes = 
                gtk_tree_view_new_with_model 
                (GTK_TREE_MODEL (PRIVATE (a_this)->models.feasible_attributes));
        renderer = gtk_cell_renderer_text_new ();
        gtk_tree_view_insert_column_with_attributes 
                (GTK_TREE_VIEW 
                 (PRIVATE (a_this)->widgets.feasible_attributes), 1,
                 _("Possible attributes"), renderer, "text", 0, NULL);
        selection = gtk_tree_view_get_selection 
                (GTK_TREE_VIEW (PRIVATE (a_this)->widgets.feasible_attributes));
        g_signal_connect (G_OBJECT (selection), "changed",
                          G_CALLBACK (feasible_attribute_selected_cb),
                          a_this);
        PRIVATE (a_this)->selections.feasible_attributes = selection;

        gtk_table_resize (GTK_TABLE (a_this), 2, 2);
        scrolled = gtk_scrolled_window_new (NULL, NULL);
        gtk_scrolled_window_set_policy 
                (GTK_SCROLLED_WINDOW (scrolled),
                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
        gtk_container_add (GTK_CONTAINER (scrolled), 
                           PRIVATE (a_this)->widgets.feasible_children);
        gtk_table_attach_defaults (GTK_TABLE (a_this), 
                                   scrolled, 0, 1, 0, 1);
        
        scrolled = gtk_scrolled_window_new (NULL, NULL);
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW 
                                        (scrolled),
                                        GTK_POLICY_AUTOMATIC, 
                                        GTK_POLICY_AUTOMATIC);
        gtk_container_add (GTK_CONTAINER (scrolled), 
                           PRIVATE (a_this)->widgets.feasible_prev_siblings);
        gtk_table_attach_defaults (GTK_TABLE (a_this), 
                                   scrolled, 1, 2, 0, 1);
        
        scrolled = gtk_scrolled_window_new (NULL, NULL);
        gtk_scrolled_window_set_policy 
                (GTK_SCROLLED_WINDOW (scrolled),
                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
        gtk_container_add (GTK_CONTAINER (scrolled), 
                           PRIVATE (a_this)->widgets.feasible_next_siblings);
        gtk_table_attach_defaults (GTK_TABLE (a_this), 
                                   scrolled, 1, 2, 1, 2);
        
        scrolled = gtk_scrolled_window_new (NULL, NULL);
        gtk_scrolled_window_set_policy 
                (GTK_SCROLLED_WINDOW (scrolled),
                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
        gtk_container_add (GTK_CONTAINER (scrolled), 
                           PRIVATE (a_this)->widgets.feasible_attributes);
        gtk_table_attach_defaults (GTK_TABLE (a_this), 
                                   scrolled, 0, 1, 1, 2);
}

GType
mlview_completion_table_get_type (void)
{
        static GType mlview_completion_table_type = 0;
        
        if (!mlview_completion_table_type) {
                static const GTypeInfo
                        mlview_completion_table_type_info = {
                                sizeof (MlViewCompletionTableClass),
                                NULL,   /* base_init */
                                NULL,   /* base_finalize */
                                (GClassInitFunc)
                                mlview_completion_table_class_init,
                                NULL,   /* class_finalize */
                                NULL,   /* class_data */
                                sizeof (MlViewCompletionTable),
                                0,
                                (GInstanceInitFunc)
                                mlview_completion_table_init
                        };
                mlview_completion_table_type =
                        g_type_register_static (GTK_TYPE_TABLE,
                                                "MlViewCompletionTable",
                                                &mlview_completion_table_type_info,
                                                0);
        }
        return mlview_completion_table_type;
}

/**
 *Changes the currently selected node for the completion table.
 *@param a_widget the current #MlViewCompletionTable instance. 
 *@param a_node_found the node to be selected.
 */
void
mlview_completion_table_select_node (MlViewCompletionTable *a_widget,
                                     xmlNode *a_node_found)
{
        MlViewAppContext *app_context = NULL;
        GList *list = NULL;

        g_return_if_fail (a_widget 
                          && MLVIEW_IS_COMPLETION_TABLE (a_widget));
        g_return_if_fail (PRIVATE (a_widget) 
                          && PRIVATE (a_widget)->xml_doc &&
                          MLVIEW_IS_XML_DOCUMENT 
                          (PRIVATE (a_widget)->xml_doc));
        g_return_if_fail (a_node_found);

        app_context = mlview_xml_document_get_app_context 
                (PRIVATE (a_widget)->xml_doc);

        if (a_node_found->type == XML_ELEMENT_NODE &&
            mlview_xml_document_is_node_valid 
            (PRIVATE (a_widget)->xml_doc,
             a_node_found)) { 
                mlview_parsing_utils_build_element_name_completion_list
                        (app_context,
                         ADD_CHILD,
                         a_node_found, &list);   
                update_list_store (PRIVATE (a_widget)->models.feasible_children, list);
                g_list_free (list);
                list = NULL;
                mlview_parsing_utils_build_element_name_completion_list
                        (app_context,
                         INSERT_BEFORE,
                         a_node_found, &list);   
                update_list_store (PRIVATE (a_widget)->models.feasible_prev_siblings, list);
                g_list_free (list);
                list = NULL;
                mlview_parsing_utils_build_element_name_completion_list
                        (app_context,
                         INSERT_AFTER,
                         a_node_found, &list);   
                update_list_store (PRIVATE (a_widget)->models.feasible_next_siblings, list);
                g_list_free (list);
                list = NULL;
                mlview_parsing_utils_build_attribute_name_completion_list
                        (app_context, a_node_found,
                         &list, FALSE);
                update_list_store (PRIVATE (a_widget)->models.feasible_attributes, list);
                g_list_free (list);
                list = NULL;

                PRIVATE (a_widget)->cur_node = a_node_found;

                gtk_widget_set_sensitive (GTK_WIDGET (a_widget), TRUE);
        }
        else {
                update_list_store (PRIVATE (a_widget)->models.feasible_children, NULL);
                update_list_store (PRIVATE (a_widget)->models.feasible_prev_siblings, NULL);
                update_list_store (PRIVATE (a_widget)->models.feasible_next_siblings, NULL);
                update_list_store (PRIVATE (a_widget)->models.feasible_attributes, NULL);

                PRIVATE (a_widget)->cur_node = NULL;

                gtk_widget_set_sensitive (GTK_WIDGET (a_widget), FALSE);
        }
}

/**
 *A constructor of the #MlViewCompletionTable class.
 *@param a_xml_doc the xml document object model 
 *the #MlViewCompletionTable must work on.
 *@return the newly created #MlViewCompletionTable, or NULL in case of an error.
 */
GtkWidget *
mlview_completion_table_new (MlViewXMLDocument *a_xml_doc)
{
        MlViewCompletionTable *table = NULL;

        g_return_val_if_fail (a_xml_doc, NULL);

        table = g_object_new (MLVIEW_TYPE_COMPLETION_TABLE, NULL);
        PRIVATE (table)->xml_doc = a_xml_doc;

        return GTK_WIDGET (table);
}
