/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 8-*- */

/*This file is part of GNU MlView
 *
 *GNU MlView is free software; 
 *you can redistribute it and/or modify it under the terms of 
 *the GNU General Public License as 
 *published by the Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope 
 *that it will be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include <string.h>

#include "mlview-schema.h"
#include "mlview-parsing-utils.h"
#include "mlview-utils.h"

struct _MlViewSchemaPrivate {
        gchar *url;
        guint ref;
        enum MlViewSchemaType type;

        union {
                xmlDtdPtr dtd;
                xmlRelaxNGPtr rng;
                xmlSchemaPtr xsd;
        } schema;
};

MlViewSchema *
mlview_schema_new_from_dtd (xmlDtdPtr a_dtd,
                            const gchar *a_url,
                            MlViewAppContext *a_ctxt)
{
        MlViewSchema *schema = NULL;

        g_return_val_if_fail (a_dtd && a_url, NULL);
        g_return_val_if_fail (a_ctxt && MLVIEW_IS_APP_CONTEXT (a_ctxt),
                              NULL);
        
        schema = g_try_malloc (sizeof (MlViewSchema));

        if (!schema)
                goto cleanup;

        memset (schema, 0, sizeof (MlViewSchema));

        schema->priv = g_try_malloc (sizeof (MlViewSchemaPrivate));

        if (!schema->priv)
                goto cleanup;

        memset (schema->priv, 0, sizeof (MlViewSchemaPrivate));

        schema->priv->url = g_strdup (a_url);

        if (!schema->priv->url)
                goto cleanup;
        
        schema->priv->type = SCHEMA_TYPE_DTD;

        schema->priv->schema.dtd = a_dtd;

        mlview_schema_ref (schema);

        return schema;

 cleanup:
        if (schema) {
                if (schema->priv) {
                        if (schema->priv->url) {
                                g_free (schema->priv->url);
                                schema->priv->url = NULL;
                        }
                        
                        g_free (schema->priv);
                        schema->priv = NULL;
                }
                
                g_free (schema);
                schema = NULL;
        }

        return NULL;
}

MlViewSchema *
mlview_schema_load_from_file (const gchar *a_url,
                              enum MlViewSchemaType a_type,
                              MlViewAppContext *a_ctxt)
{
        MlViewSchema *schema = NULL;
        
        g_return_val_if_fail (a_url, NULL);
        g_return_val_if_fail (a_ctxt && MLVIEW_IS_APP_CONTEXT (a_ctxt),
                              NULL);

        schema = g_try_malloc (sizeof (MlViewSchema));
        
        if (!schema)
                goto cleanup;

        memset (schema, 0, sizeof (MlViewSchema));

        schema->priv = g_try_malloc (sizeof (MlViewSchemaPrivate));

        if (!schema->priv)
                goto cleanup;

        memset (schema->priv, 0, sizeof (MlViewSchemaPrivate));

        schema->priv->type = a_type;

        schema->priv->url = g_strdup (a_url);
        
        if (!schema->priv->url)
                goto cleanup;

        switch (a_type) {
        case SCHEMA_TYPE_DTD:
                schema->priv->schema.dtd = 
                        mlview_parsing_utils_load_dtd (a_url,
                                                       a_ctxt);
                
                if (!schema->priv->schema.dtd)
                        goto cleanup;
                
                break;
        case SCHEMA_TYPE_RNG:
                schema->priv->schema.rng = 
                        mlview_parsing_utils_load_rng (a_url,
                                                       a_ctxt);
                
                if (!schema->priv->schema.rng)
                        goto cleanup;
                
                break;
        case SCHEMA_TYPE_XSD:
                schema->priv->schema.xsd = 
                        mlview_parsing_utils_load_xsd (a_url,
                                                       a_ctxt);

                if (!schema->priv->schema.xsd)
                        goto cleanup;

                break;
        default:
                g_assert_not_reached () ;
                break ;
        }

        mlview_schema_ref (schema);

        return schema;

 cleanup:
        if (schema) {
                if (schema->priv) {
                        if (schema->priv->url) {
                                g_free (schema->priv->url);
                                schema->priv->url = NULL;
                        }
                        
                        g_free (schema->priv);
                        schema->priv = NULL;
                }
                
                g_free (schema);
                schema = NULL;
        }

        return NULL;
}

void
mlview_schema_ref (MlViewSchema *a_schema)
{
        g_return_if_fail (a_schema && a_schema->priv);

        a_schema->priv->ref++;
}

void
mlview_schema_unref (MlViewSchema *a_schema)
{
        g_return_if_fail (a_schema && a_schema->priv);

        a_schema->priv->ref--;

        if (!a_schema->priv->ref) {
                switch (a_schema->priv->type) {
                case SCHEMA_TYPE_DTD:
                        if (a_schema->priv->schema.dtd) {
                                xmlFreeDtd (a_schema->priv->schema.dtd);
                                a_schema->priv->schema.dtd = NULL;
                        }
                        break;
                case SCHEMA_TYPE_RNG:
                        if (a_schema->priv->schema.rng) {
                                xmlRelaxNGFree (a_schema->priv->schema.rng);
                                a_schema->priv->schema.rng = NULL;
                        }
                        break;
                case SCHEMA_TYPE_XSD:
                        if (a_schema->priv->schema.xsd) {
                                xmlSchemaFree (a_schema->priv->schema.xsd);
                                a_schema->priv->schema.xsd = NULL;
                        }
                        break;

                default:
                        g_assert_not_reached () ;
                        break ;
                }

                if (a_schema->priv) {
                        if (a_schema->priv->url) {
                                g_free (a_schema->priv->url);
                                a_schema->priv->url = NULL;
                        }
                        
                        g_free (a_schema->priv);
                        a_schema->priv = NULL;
                }

                g_free (a_schema);
                a_schema = NULL;
        }
}

gchar *
mlview_schema_get_url (MlViewSchema *a_schema)
{
        g_return_val_if_fail (a_schema && a_schema->priv, NULL);
        g_return_val_if_fail (a_schema->priv->url, NULL);

        return a_schema->priv->url;
}


MlViewSchema *
mlview_schema_load_interactive (enum MlViewSchemaType a_type,
                                MlViewAppContext *a_ctxt)
{
        GtkWidget *fs = NULL;
        enum MLVIEW_SELECTED_BUTTON res;
        gchar *file = NULL;
        MlViewSchema *schema = NULL;

        g_return_val_if_fail (a_ctxt && MLVIEW_IS_APP_CONTEXT (a_ctxt),
                              NULL);

        fs = GTK_WIDGET (mlview_app_context_get_file_chooser (a_ctxt,
							      _("Open a DTD"))) ; 
        g_return_val_if_fail (fs, NULL);
        
	res = gtk_dialog_run (GTK_DIALOG (fs)) ;
	gtk_widget_hide (fs) ;
        
        switch (res) {
        case GTK_RESPONSE_OK:
                file = (gchar *)gtk_file_chooser_get_filename 
                        (GTK_FILE_CHOOSER (fs));
                if (file && strncmp (file, "", 1)) {
                        switch (a_type) {
                        case SCHEMA_TYPE_DTD:
                                schema = mlview_schema_load_from_file
                                        (file, SCHEMA_TYPE_DTD, a_ctxt);
                                break;
                        case SCHEMA_TYPE_RNG:
                                schema = mlview_schema_load_from_file
                                        (file, SCHEMA_TYPE_RNG, a_ctxt);
                                break;
                        case SCHEMA_TYPE_XSD:
                                schema = mlview_schema_load_from_file
                                        (file, SCHEMA_TYPE_XSD, a_ctxt);
                                break;
                        default:
                                g_assert_not_reached () ;
                                break ;
                        }
                }
                
                break;
        case CANCEL_BUTTON:
        case WINDOW_CLOSED:
        default:
                goto cleanup;
        }
        
        if (!schema)
                mlview_app_context_warning (a_ctxt, 
                                            (const gchar*)
                                            _("Unable to open the selected schema."));
        
 cleanup:
        return schema;
}

enum MlViewStatus 
mlview_schema_get_type (MlViewSchema *a_this,
                        enum MlViewSchemaType *a_type)
{
        g_return_val_if_fail (a_this 
                              && a_this->priv
                              && a_type,
                              MLVIEW_OK) ;

        *a_type = a_this->priv->type ;

        return MLVIEW_OK ;
}

enum MlViewStatus 
mlview_schema_get_native_schema (MlViewSchema *a_this,
                                 gpointer *a_nativeSchema)
{
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_this && a_this->priv
                              && a_nativeSchema,
                              MLVIEW_BAD_PARAM_ERROR) ;

        switch (a_this->priv->type)
        {
        case SCHEMA_TYPE_DTD:
                *a_nativeSchema = a_this->priv->schema.dtd ;
                break ;
        case SCHEMA_TYPE_RNG:
                *a_nativeSchema = a_this->priv->schema.rng ;
                break ;
        case SCHEMA_TYPE_XSD:
                *a_nativeSchema = a_this->priv->schema.xsd ;
                break ;
        default:
                status = MLVIEW_ERROR ;
                break ;
        }
        return status ;
}
